import { PluggyAcquirerClient } from './acquirerClient';
import { BaseApi, ClientParams } from './baseApi';
import { PluggyPaymentsClient } from './paymentsClient';
import { Account, AccountType, Benefit, Category, Connector, ConnectorFilters, ConnectTokenOptions, CreateItemOptions, CreateWebhook, IdentityResponse, Investment, InvestmentTransaction, InvestmentType, Item, Opportunity, OpportunityFilters, PageResponse, Parameters, Transaction, TransactionFilters, UpdateWebhook, Webhook, IncomeReport, Loan, PageFilters, InvestmentsFilters } from './types';
import { CreditCardBills } from './types/creditCardBills';
import { ValidationResult } from './types/validation';
/**
 * Creates a new client instance for interacting with Pluggy API
 * @constructor
 * @param API_KEY for authenticating to the API
 * @returns {PluggyClient} a client for making requests
 */
export declare class PluggyClient extends BaseApi {
    payments: PluggyPaymentsClient;
    acquirer: PluggyAcquirerClient;
    constructor(params: ClientParams);
    /**
     * Fetch all available connectors
     * @returns {PageResponse<Connector>} paged response of connectors
     */
    fetchConnectors(options?: ConnectorFilters): Promise<PageResponse<Connector>>;
    /**
     * Fetch a single Connector
     * @param id The Connector ID
     * @returns {Connector} a connector object
     */
    fetchConnector(id: number): Promise<Connector>;
    /**
     * Fetch a single item
     * @param id The Item ID
     * @returns {Item} a item object
     */
    fetchItem(id: string): Promise<Item>;
    /**
     * Check that connector parameters are valid
     * @param id The Connector ID
     * @param parameters A map of name and value for the credentials to be validated
     * @returns {ValidationResult} an object with the info of which parameters are wrong
     */
    validateParameters(id: number, parameters: Parameters): Promise<ValidationResult>;
    /**
     * Creates an item
     * @param connectorId The Connector's id
     * @param parameters A map of name and value for the needed credentials
     * @param options Options available to set to the item
     * @returns {Item} a item object
     */
    createItem(connectorId: number, parameters: Record<string, string>, options?: CreateItemOptions): Promise<Item>;
    /**
     * Updates an item
     * @param id The Item ID
     * @param parameters A map of name and value for the credentials to be updated.
     *                   Optional; if none submitted, an Item update will be attempted with the latest used credentials.
     * @returns {Item} a item object
     */
    updateItem(id: string, parameters?: Parameters, options?: CreateItemOptions): Promise<Item>;
    /**
     * Send MFA for item execution
     * @param id The Item ID
     * @param parameters A map of name and value for the mfa requested
     * @returns {Item} a item object
     */
    updateItemMFA(id: string, parameters?: Parameters): Promise<Item>;
    /**
     * Deletes an item
     */
    deleteItem(id: string): Promise<void>;
    /**
     * Fetch accounts from an Item
     * @param itemId The Item id
     * @returns {PageResponse<Account>} paged response of accounts
     */
    fetchAccounts(itemId: string, type?: AccountType): Promise<PageResponse<Account>>;
    /**
     * Fetch a single account
     * @returns {Account} an account object
     */
    fetchAccount(id: string): Promise<Account>;
    /**
     * Fetch transactions from an account
     * @param accountId The account id
     * @param {TransactionFilters} options Transaction options to filter
     * @returns {PageResponse<Transaction[]>} object which contains the transactions list and related paging data
     */
    fetchTransactions(accountId: string, options?: TransactionFilters): Promise<PageResponse<Transaction>>;
    /**
     * Fetch all transactions from an account
     * @param accountId The account id
     * @returns {Transaction[]} an array of transactions
     */
    fetchAllTransactions(accountId: string): Promise<Transaction[]>;
    /**
     * Post transaction user category for transactin
     * @param id The Transaction id
     *
     * @returns {Transaction} updated transaction object
     */
    updateTransactionCategory(id: string, categoryId: string): Promise<Transaction>;
    /**
     * Fetch a single transaction
     *
     * @returns {Transaction} an transaction object
     */
    fetchTransaction(id: string): Promise<Transaction>;
    /**
     * Fetch investments from an Item
     *
     * @param itemId The Item id
     * @returns {PageResponse<Investment>} paged response of investments
     */
    fetchInvestments(itemId: string, type?: InvestmentType, options?: InvestmentsFilters): Promise<PageResponse<Investment>>;
    /**
     * Fetch a single investment
     *
     * @returns {Investment} an investment object
     */
    fetchInvestment(id: string): Promise<Investment>;
    /**
     * Fetch transactions from an investment
     *
     * @param investmentId The investment id
     * @param {TransactionFilters} options Transaction options to filter
     * @returns {PageResponse<InvestmentTransaction[]>} object which contains the transactions list and related paging data
     */
    fetchInvestmentTransactions(investmentId: string, options?: TransactionFilters): Promise<PageResponse<InvestmentTransaction>>;
    /**
     * Fetch all investment transactions from an investment
     * @param investmentId The investment id
     * @returns {InvestmentTransaction[]} an array of investment transactions
     */
    fetchAllInvestmentTransactions(investmentId: string): Promise<InvestmentTransaction[]>;
    /**
     * Fetch opportunities from an Item
     *
     * @param itemId the Item id
     * @param options - request search filters
     * @returns {PageResponse<Opportunity>} paged response of opportunities
     */
    fetchOpportunities(itemId: string, options?: OpportunityFilters): Promise<PageResponse<Opportunity>>;
    /**
     * Fetch loans from an Item
     *
     * @param {string} itemId
     * @param {PageFilters} options - request search filters
     * @returns {Promise<PageResponse<Loan>>} - paged response of loans
     */
    fetchLoans(itemId: string, options?: PageFilters): Promise<PageResponse<Loan>>;
    /**
     * Fetch loan by id
     *
     * @param {string} id - the loan id
     * @returns {Promise<Loan>} - loan object, if found
     */
    fetchLoan(id: string): Promise<Loan>;
    /**
     * Fetch benefits from an Item
     *
     * @param {string} itemId
     * @param {PageFilters} options - request search filters
     * @returns {Promise<PageResponse<Benefit>>} - paged response of benefits
     */
    fetchBenefits(itemId: string, options?: PageFilters): Promise<PageResponse<Benefit>>;
    /**
     * Fetch benefit by id
     *
     * @param {string} id - the benefit id
     * @returns {Promise<Benefit>} - benefit object, if found
     */
    fetchBenefit(id: string): Promise<Benefit>;
    /**
     * Fetch the identity resource
     * @returns {IdentityResponse} an identity object
     */
    fetchIdentity(id: string): Promise<IdentityResponse>;
    /**
     * Fetch the identity resource by it's Item ID
     * @returns {IdentityResponse} an identity object
     */
    fetchIdentityByItemId(itemId: string): Promise<IdentityResponse>;
    /**
     * Fetch credit card bills from an accountId
     * @returns {PageResponse<CreditCardBills>} an credit card bills object
     */
    fetchCreditCardBills(accountId: string, options?: PageFilters): Promise<PageResponse<CreditCardBills>>;
    /**
     * Fetch a single credit card bill by its id
     * @param {string} id - the credit card bill id
     * @returns {Promise<CreditCardBills>} - credit card bill object, if found
     */
    fetchCreditCardBill(id: string): Promise<CreditCardBills>;
    /**
     * Fetch all available categories
     * @returns {Categories[]} an paging response of categories
     */
    fetchCategories(): Promise<PageResponse<Category>>;
    /**
     * Fetch a single category
     * @returns {Category} a category object
     */
    fetchCategory(id: string): Promise<Category>;
    /**
     * Fetch a single webhook
     * @returns {Webhook} a webhook object
     */
    fetchWebhook(id: string): Promise<Webhook>;
    /**
     * Fetch all available webhooks
     * @returns {Webhook[]} a paging response of webhooks
     */
    fetchWebhooks(): Promise<PageResponse<Webhook>>;
    /**
     * Creates a Webhook
     * @param webhookParams - The webhook params to create, this includes:
     * - url: The url where will receive notifications
     * - event: The event to listen for
     * - headers (optional): The headers to send with the webhook
     * @returns {Webhook} the created webhook object
     */
    createWebhook(event: CreateWebhook['event'], url: CreateWebhook['url'], headers?: CreateWebhook['headers']): Promise<Webhook>;
    /**
     * Updates a Webhook
     * @param id - The Webhook ID
     * @param updatedWebhookParams - The webhook params to update
     * @returns {Webhook} The webhook updated
     */
    updateWebhook(id: string, updatedWebhookParams: UpdateWebhook): Promise<Webhook>;
    /**
     * Deletes a Webhook
     */
    deleteWebhook(id: string): Promise<void>;
    /**
     * Fetch all income reports for the last years that the Financial Institution provides
     * @param {string} itemId - The Item ID to fetch income reports for
     * @returns {PageResponse<IncomeReport>} paged response of income reports
     */
    fetchIncomeReports(itemId: string): Promise<PageResponse<IncomeReport>>;
    /**
     * Creates a connect token that can be used as API KEY to connect items from the Frontend
     * @returns {string} Access token to connect items with restrict access
     */
    createConnectToken(itemId?: string, options?: ConnectTokenOptions): Promise<{
        accessToken: string;
    }>;
}
