"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PluggyPaymentsClient = void 0;
const baseApi_1 = require("./baseApi");
/**
 * Creates a new client instance for interacting with Pluggy API for the Payments API
 * @constructor
 * @param API_KEY for authenticating to the API
 * @returns {PluggyPaymentsClient} a client for making requests
 */
class PluggyPaymentsClient extends baseApi_1.BaseApi {
    /**
     * Creates a payment request
     * @returns {PaymentRequest} PaymentRequest object
     */
    createPaymentRequest(paymentRequest) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createPostRequest(`payments/requests`, null, paymentRequest);
        });
    }
    /**
     * Fetch a single payment request
     * @returns {PaymentRequest} PaymentRequest object
     */
    fetchPaymentRequest(id) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest(`payments/requests/${id}`);
        });
    }
    /**
     * Fetch all payment requests
     * @returns {PageResponse<PaymentRequest>} paged response of payment requests
     */
    fetchPaymentRequests(options = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest('payments/requests', Object.assign({}, options));
        });
    }
    /**
     * Delete a payment request
     */
    deletePaymentRequest(id) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.createDeleteRequest(`payments/requests/${id}`);
        });
    }
    /**
     * Creates a payment intent
     * @returns {PaymentIntent} PaymentIntent object
     */
    createPaymentIntent(params) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createPostRequest(`payments/intents`, null, params);
        });
    }
    /**
     * Fetch a single payment intent
     * @returns {PaymentIntent} PaymentIntent object
     */
    fetchPaymentIntent(id) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest(`payments/intents/${id}`);
        });
    }
    /**
     * Fetch all payment intents
     * @returns {PageResponse<PaymentIntent>} paged response of payment intents
     */
    fetchPaymentIntents(options = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest('payments/intents', Object.assign({}, options));
        });
    }
    /**
     * Creates a payment customer
     * @returns {PaymentCustomer} PaymentCustomer object
     */
    createPaymentCustomer(payload) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createPostRequest(`payments/customers`, null, payload);
        });
    }
    /**
     * Fetch a single payment customer
     * @returns {PaymentCustomer} PaymentCustomer object
     */
    fetchPaymentCustomer(id) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest(`payments/customers/${id}`);
        });
    }
    /**
     * Fetch all payment customers
     * @returns {PageResponse<PaymentCustomer>} paged response of payment customers
     */
    fetchPaymentCustomers(options = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest('payments/customers', Object.assign({}, options));
        });
    }
    /**
     * Delete a payment customer
     */
    deletePaymentCustomer(id) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.createDeleteRequest(`payments/customers/${id}`);
        });
    }
    updatePaymentCustomer(id, payload) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createPatchRequest(`payments/customers/${id}`, null, payload);
        });
    }
    /**
     * Creates a payment recipient (bank account)
     * @returns {PaymentRecipient} PaymentRecipient object
     */
    createPaymentRecipient(payload) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createPostRequest(`payments/recipients`, null, payload);
        });
    }
    /**
     * Fetch a single payment recipient
     * @returns {PaymentRecipient} PaymentRecipient object
     */
    fetchPaymentRecipient(id) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest(`payments/recipients/${id}`);
        });
    }
    /**
     * Fetch all payment recipients
     * @returns {PageResponse<PaymentRecipient>} paged response of payment recipients
     */
    fetchPaymentRecipients(options = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest('payments/recipients', Object.assign({}, options));
        });
    }
    /**
     * Delete a payment recipient
     */
    deletePaymentRecipient(id) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.createDeleteRequest(`payments/recipients/${id}`);
        });
    }
    /**
     * Update a payment recipient
     * @param id ID of the payment recipient
     * @param payload Fields to update
     * @returns {PaymentRecipient} PaymentRecipient object
     */
    updatePaymentRecipient(id, payload) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createPatchRequest(`payments/recipients/${id}`, null, payload);
        });
    }
    /**
     * Fetch a single payment recipient institution
     * @returns {PaymentInstitution} PaymentInstitution object
     */
    fetchPaymentInstitution(id) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest(`payments/recipients/institutions/${id}`);
        });
    }
    /**
     * Fetch all payment recipient institutions
     * @returns {PageResponse<PaymentInstitution>} paged response of payment recipient institutions
     */
    fetchPaymentInstitutions(options) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest('payments/recipients/institutions', Object.assign({}, options));
        });
    }
    /**
     * Create a bulk payment
     * @returns {BulkPayment} BulkPayment object
     */
    createBulkPayment(payload) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createPostRequest(`payments/bulk`, null, payload);
        });
    }
    /**
     * Fetch a single bulk payment
     * @returns {BulkPayment} BulkPayment object
     */
    fetchBulkPayment(id) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest(`payments/bulk/${id}`);
        });
    }
    /**
     * Fetch all bulk payments
     * @returns {PageResponse<BulkPayment>} paged response of bulk payments
     */
    fetchBulkPayments(options = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest('payments/bulk', options);
        });
    }
    /** Deletes the bulk payment */
    deleteBulkPayment(id) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.createDeleteRequest(`payments/bulk/${id}`);
        });
    }
    /**
     * Creates a smart account
     * @returns {SmartAccount} SmartAccount object
     */
    createSmartAccount(payload) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createPostRequest(`payments/smart-accounts`, null, payload);
        });
    }
    /**
     * Fetch a single smart account
     * @returns {SmartAccount} SmartAccount object
     */
    fetchSmartAccount(id) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest(`payments/smart-accounts/${id}`);
        });
    }
    /**
     * Fetch a smart account current balance
     * @returns {SmartAccountBalance} SmartAccountBalance object
     */
    fetchSmartAccountBalance(id) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest(`payments/smart-accounts/${id}/balance`);
        });
    }
    /**
     * Fetch all smart accounts
     * @returns {PageResponse<SmartAccount>} paged response of smart accounts
     */
    fetchSmartAccounts(options = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest('payments/smart-accounts', options);
        });
    }
    /**
     * Creates a payment request receipt
     * @param id ID of the payment request
     */
    createPaymentRequestReceipt(id) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createPostRequest(`payments/requests/${id}/receipt`);
        });
    }
    /**
     * Fetch all payment request receipts
     * @param id ID of the payment request
     */
    fetchPaymentRequestReceipts(id) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest(`payments/requests/${id}/receipts`);
        });
    }
    /**
     * Fetch a single payment request receipt
     * @param requestId ID of the payment request
     * @param receiptId ID of the payment request receipt
     */
    fetchPaymentRequestReceipt(requestId, receiptId) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest(`payments/requests/${requestId}/receipts/${receiptId}`);
        });
    }
    /**
     * Fetch all scheduled payments from a payment request
     * @param paymentRequest ID of the payment request
     */
    fetchScheduledPayments(paymentRequest) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest(`payments/requests/${paymentRequest}/schedules`);
        });
    }
    /**
     * Fetch a single scheduled payment from a payment request
     * @param paymentRequest ID of the payment request
     * @param scheduleId ID of the scheduled payment
     */
    fetchScheduledPayment(paymentRequest, scheduleId) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.createGetRequest(`payments/requests/${paymentRequest}/schedules/${scheduleId}`);
        });
    }
    /**
     * Cancel a scheduled payment from a payment request
     * @param paymentRequest ID of the payment request
     * @param scheduleId ID of the scheduled payment
     */
    cancelScheduledPayment(paymentRequest, scheduleId) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.createPostRequest(`payments/requests/${paymentRequest}/schedules/${scheduleId}/cancel`);
        });
    }
    /**
     * Cancel the payment request authorization of a scheduled payment (cancel all pending payments)
     * @param paymentRequest ID of the payment request
     */
    cancelScheduledPayments(paymentRequest) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.createPostRequest(`payments/requests/${paymentRequest}/schedules/cancel`);
        });
    }
}
exports.PluggyPaymentsClient = PluggyPaymentsClient;
