import { CurrencyCode, PageFilters } from './common';
export declare const INVESTMENT_TYPES: readonly ["MUTUAL_FUND", "SECURITY", "EQUITY", "COE", "FIXED_INCOME", "ETF", "OTHER"];
/**
 * @typedef InvestmentType
 * Type of investment
 */
export declare type InvestmentType = typeof INVESTMENT_TYPES[number];
export declare const INVESTMENT_STATUS: readonly ["ACTIVE", "PENDING", "TOTAL_WITHDRAWAL"];
/**
 * @typedef InvestmentStatus
 * Status of investment
 */
export declare type InvestmentStatus = typeof INVESTMENT_STATUS[number];
export declare const COE_INVESTMENT_SUBTYPES: readonly ["STRUCTURED_NOTE"];
export declare type CoeInvestmentSubtype = typeof COE_INVESTMENT_SUBTYPES[number];
export declare const MUTUAL_FUND_INVESTMENT_SUBTYPES: readonly ["INVESTMENT_FUND", "MULTIMARKET_FUND", "FIXED_INCOME_FUND", "STOCK_FUND", "ETF_FUND", "OFFSHORE_FUND", "FIP_FUND", "EXCHANGE_FUND"];
export declare type MutualFundInvestmentSubtype = typeof MUTUAL_FUND_INVESTMENT_SUBTYPES[number];
export declare const SECURITY_INVESTMENT_SUBTYPES: readonly ["RETIREMENT"];
export declare type SecurityInvestmentSubtype = typeof SECURITY_INVESTMENT_SUBTYPES[number];
export declare const EQUITY_INVESTMENT_SUBTYPES: readonly ["STOCK", "ETF", "REAL_ESTATE_FUND", "BDR", "DERIVATIVES", "OPTION"];
export declare type EquityInvestmentSubtype = typeof EQUITY_INVESTMENT_SUBTYPES[number];
export declare const FIXED_INCOME_INVESTMENT_SUBTYPES: readonly ["TREASURY", "LCI", "LCA", "CDB", "CRI", "CRA", "CORPORATE_DEBT", "LC", "DEBENTURES"];
export declare type FixedIncomeInvestmentSubtype = typeof FIXED_INCOME_INVESTMENT_SUBTYPES[number];
export declare const INVESTMENT_SUBTYPES: readonly ["INVESTMENT_FUND", "MULTIMARKET_FUND", "FIXED_INCOME_FUND", "STOCK_FUND", "ETF_FUND", "OFFSHORE_FUND", "FIP_FUND", "EXCHANGE_FUND", "RETIREMENT", "STOCK", "ETF", "REAL_ESTATE_FUND", "BDR", "DERIVATIVES", "OPTION", "TREASURY", "LCI", "LCA", "CDB", "CRI", "CRA", "CORPORATE_DEBT", "LC", "DEBENTURES", "STRUCTURED_NOTE", "OTHER"];
export declare type InvestmentSubtype = typeof INVESTMENT_SUBTYPES[number];
export declare const INVESTMENT_TRANSACTION_TYPE: readonly ["BUY", "SELL", "TAX", "TRANSFER"];
export declare type InvestmentTransactionType = typeof INVESTMENT_TRANSACTION_TYPE[number];
/*!
  For extra details visit: https://docs.pluggy.ai/docs/investment-1
  RateTypes represent the index from where the rate is based.
*/
export declare const INVESTMENT_RATE_TYPES: string[];
export declare type InvestmentRateType = typeof INVESTMENT_RATE_TYPES[number];
/**
 * @typedef MovementType
 * The direction of the transaction.
 * If DEBIT, balance decreasing from the investment.
 * If CREDIT, balance increasing on the investment.
 */
export declare const MOVEMENT_TYPES: readonly ["DEBIT", "CREDIT"];
export declare type MovementType = typeof MOVEMENT_TYPES[number];
export declare type Expenses = {
    /** Service tax that varies according to state */
    serviceTax: number | null;
    /** Commission charged by the brokerage for carrying out transactions on the stock market */
    brokerageFee: number | null;
    /** Income Tax Withholding, amount paid to the Internal Revenue Service */
    incomeTax: number | null;
    /** Sum of other not defined expenses */
    other: number | null;
    /** Fee of Notice of Trading in Assets */
    tradingAssetsNoticeFee: number | null;
    /** Fees charged by BM&F Bovespa in negotiations */
    maintenanceFee: number | null;
    /** Liquidation fee for the settlement of a position on the expiration date or the financial settlement of physical delivery */
    settlementFee: number | null;
    /** Registration fee */
    clearingFee: number | null;
    /** Fees charged by BM&F Bovespa as a source of operating income  */
    stockExchangeFee: number | null;
    /** Fee by brokers to keep recordsin their home broker systems or on the trading desk */
    custodyFee: number | null;
    /** Amount paid to the Operator for the intermediation service  */
    operatingFee: number | null;
};
export declare type InvestmentMetadata = {
    /** Regime of the tax used for the asset */
    taxRegime: string | null;
    /** Asset proposal number identification */
    proposalNumber: string | null;
    /** Process identification number from the institution (susep) */
    processNumber: string | null;
};
export declare type InvestmentTransaction = {
    /** Primary identifier of the transacion */
    id: string;
    /** Type of the transaction */
    type: InvestmentTransactionType | null;
    /** Description of the transaction */
    description: string | null;
    /** Investment identifier related to the transaction */
    investmentId: string | null;
    /** Quantity of quotas purchased */
    quantity: number | null;
    /** Value of the purchased quotas */
    value: number | null;
    /** Amount spent or withrawaled from the investment. */
    amount: number | null;
    /** Date the transaction was placed. */
    date: Date;
    /** Date the transaction was confirmed */
    tradeDate: Date | null;
    /** Number of the corresponding brokerage note */
    brokerageNumber: string | null;
    /** Value including expenses */
    netAmount: number | null;
    /** Taxes and fees that apply */
    expenses: Expenses | null;
    /** Type of movement */
    movementType: MovementType;
};
/** institution holding the investment */
export declare type InvestmentInstitution = {
    /** Full name of the institution */
    name: string | null;
    /** Number identifier for the institution CNPJ / Other */
    number: string | null;
};
export declare type Investment = {
    id: string;
    /** Unique primary identifier for the investment available for the hole country. In brazil is CNPJ. */
    code: string | null;
    /** CNPJ of the issuer behind the investment. */
    issuerCNPJ: string | null;
    /** Unique FI provider identifier that attach's the owner to an investment and its available as a reference. */
    number: string | null;
    /** 12-character ISIN, a globally unique identifier */
    isin: string | null;
    /** Item identifier asscoiated with the investment */
    itemId: string;
    /** Type of investment associated. */
    type: InvestmentType;
    /** Subtype of investment */
    subtype: InvestmentSubtype | null;
    /** Primary name for the investment */
    name: string;
    /** Currency ISO code where amounts are shown */
    currencyCode: CurrencyCode;
    /** Quota's date | Value's Date. (Quota's are related to MUTUAL_FUNDS or ETF, others use the investment amount reference date) */
    date: Date | null;
    /** Value of the adquired quantity. (Quota's are related to MUTUAL_FUNDS or ETF, others usually default to the amount) */
    value: number | null;
    /** Quota's quantity adquired. (Quota's are related to MUTUAL_FUNDS or ETF, others usually default to 1) */
    quantity: number | null;
    /** Rent type taxes associated (I.R , Ingresos Brutos) */
    taxes: number | null;
    /** Financial type taxes associated (Impuesto Operaciones Financieras) */
    taxes2: number | null;
    /** Net worth balance / amount of the investment. Is the real current value. */
    balance: number;
    /** Current gross amount of the investment pre-taxes. (As a general rule, `Value` * `Quantity` = `Amount`) */
    amount: number | null;
    /** Available for withdraw balance. */
    amountWithdrawal: number | null;
    /** Amount that was gained / loss from the investment */
    amountProfit: number | null;
    /** Original amount deposited in the investment */
    amountOriginal: number | null;
    /** Date when the investment is due. (Normally FIXED_INCOME investments have a dueDate) */
    dueDate: Date | null;
    /** Entity name that issued the investment. (Normally FIXED_INCOME investments are issued by an entity) */
    issuer: string | null;
    /** Date when the investment was issued. (Normally FIXED_INCOME investments are issued by an entity) */
    issueDate: Date | null;
    /** Fixed rate for the investment. (Normally only available in FIXED_INCOME types) */
    rate: number | null;
    /** Fixed rate type for the investment, ie. CDI. (Normally only available in FIXED_INCOME types) */
    rateType: InvestmentRateType | null;
    /** Fixed annual rate for the investment, ie. 10.5. (Normally only available in FIXED_INCOME types) */
    fixedAnnualRate: number | null;
    /** Previous months rate value of the investment */
    lastMonthRate: number | null;
    /** Calendar annual rate, is a percentage of how it performed. (Normally only available in MUTUAL_FUNDS or ETF types) */
    annualRate: number | null;
    /**  Last 12 month rate, is a percentage of how it performed. (Normally only available in MUTUAL_FUNDS or ETF types) */
    lastTwelveMonthsRate: number | null;
    /** Current status of the investment */
    status: InvestmentStatus | null;
    /**
     * Transactions made related to the investment, like adquisitions (BUY) or withdrawals (SELL).
     * @deprecated use `client.fetchInvestmentTransactions(investmentId, searchFilters)` instead
     * this field is null unless the application was created before 2023-03-20
     *  */
    transactions: InvestmentTransaction[] | null;
    /** Investment tax information */
    metadata: InvestmentMetadata | null;
    /** Name of the owner */
    owner: string | null;
    /** Financial institution holder  of the investment */
    institution: InvestmentInstitution | null;
};
export declare type InvestmentsFilters = PageFilters;
