import fs from 'node:fs';
import path from 'node:path';
import { fileURLToPath, pathToFileURL } from 'node:url';
import IntegrationBank from './banks/integration-bank.js';
const dirname = path.resolve(fileURLToPath(import.meta.url), '..');
const banksDir = path.resolve(dirname, 'banks');
async function loadBanks() {
    const bankHandlers = fs
        .readdirSync(banksDir)
        .filter(filename => filename.includes('_') && filename.endsWith('.js'));
    const imports = await Promise.all(bankHandlers.map(file => {
        const fileUrlToBank = pathToFileURL(path.resolve(banksDir, file)); // pathToFileURL for ESM compatibility
        return import(fileUrlToBank.toString()).then(handler => handler.default);
    }));
    return imports;
}
export const banks = await loadBanks();
export function BankFactory(institutionId) {
    return (banks.find(b => b.institutionIds.includes(institutionId)) || IntegrationBank);
}
// this list contains banks that are known by GoCardless to only provide
// continuous access if the EUA has max_historical_days <= 90
// please do not add to this list unless the institution ID appears in
// https://bankaccountdata.zendesk.com/hc/en-gb/articles/11529718632476-Extended-history-and-continuous-access-edge-cases
const SPECIAL_CONTINUOUS_ACCESS_BANKS = [
    'BRED_BREDFRPPXXX',
    'INDUSTRA_MULTLV2X',
    'MEDICINOSBANK_MDBALT22XXX',
    'CESKA_SPORITELNA_GIBACZPX',
    'LHV_LHVBEE22',
    'LUMINOR_*',
    'SWEDBANK_*',
    'SEB_*',
    'LABORALKUTXA_CLPEES2M*',
    'BANKINTER_BKBKESMM*',
    'CAIXABANK_CAIXESBB',
    'SANTANDER_DE_SCFBDE33',
    'BBVA_BBVAESMM',
    'COOP_EKRDEE22*',
    'BANCA_AIDEXA_AIDXITMM',
    'BANCA_PATRIMONI_SENVITT1',
    'BANCA_SELLA_SELBIT2B',
    'CARTALIS_CIMTITR1',
    'DOTS_HYEEIT22',
    'HYPE_BUSINESS_HYEEIT22',
    'HYPE_HYEEIT2',
    'ILLIMITY_ITTPIT2M',
    'SMARTIKA_SELBIT22',
    'TIM_HYEEIT22',
    'TOT_SELBIT2B',
    'OPYN_BITAITRRB2B',
    'PAYTIPPER_PAYTITM1',
    'SELLA_PERSONAL_CREDIT_SELBIT22',
    'SIAULIU_BANKAS_CBSBLT26XXX',
    'INTESA_SANPAOLO_CORPORATE_BCITITMMXXX',
    'AGRICOLE_HBW_CRPPIT2P',
    'AGRICOLE_NET2_CRPPIT2P',
    'AGRICOLE_SBW_CRPPIT2P',
    'ALTO_ADIGE_BANCA_SUDTIROL_BANK_ALOAIT21',
    'BANCATER_CREDITO_COOPERATIVO_FVG_CCRTIT2TTER',
    'BANCA_ADRIA_COLLI_EUGANEI_CCRTIT2T97A',
    'BANCA_AKROS_AKROITMM',
    'BANCA_ALETTI_ALETITMM',
    'BANCA_ALTO_VICENTINO_CREDITO_COOPERATIVO_DI_SCHIO_PEDEMONTE_E_ROANA_CCRTIT2TF01',
    'BANCA_CENTRO_EMILIA_CCRTIT2TBCE',
    'BANCA_CENTRO_LAZIO_CCRTIT2TBCL',
    'BANCA_CESARE_BCEPITMM',
    'BANCA_DEL_TERRITORIO_LOMBARDO_CCRTIT2TBTL',
    'BANCA_DI_BOLOGNA_BDBOIT21',
    'BANCA_DI_CARAGLIO_DEL_CUNEESE_E_DELLA_RIVIERA_DEI_FIORI_CCRTIT2TCAR',
    'BANCA_DI_MACERATA_PRACIT31',
    'BANCA_DI_RISPARMIO_DI_SAVIGLIANO_SARCIT2SXXX',
    'BANCA_EUROMOBILIARE_EUBAITM1',
    'BANCA_FIDEURAM_FIBKITMM',
    'BANCA_GALILEO_GAEOITM1',
    'BANCA_LAZIO_NORD_CREDITO_COOPERATIVO_SOC_COOP_PER_AZIONI_CCRTIT2TVIT',
    'BANCA_MALATESTIANA_CCRTIT2TMAL',
    'BANCA_MONTE_CORPORATE_PASCITMM',
    'BANCA_MONTE_RETAIL_PASCITMM',
    'BANCA_NAZIONALE_DEL_LAVORO_BNLIITRRXXX',
    'BANCA_NAZIONALE_DEL_LAVORO_CORPORATE_BNLIITRRXXX',
    'BANCA_PASSADORE_PASBITGGXXX',
    'BANCA_POPOLARE_DELLE_PROVINCIE_MOLISANE_PPMSIT31',
    'BANCA_POPOLARE_DEL_CASSINATE_PER_AZIONI_POCAIT3C',
    'BANCA_POPOLARE_DI_CORTONA_PER_AZIONI_POCSIT31019',
    'BANCA_POPOLARE_DI_LAJATICO_BLJAIT3LXXX',
    'BANCA_POPOLARE_DI_SONDRIO_POSOIT22XXX',
    'BANCA_POPOLARE_VESUVIANA_POVEIT33',
    'BANCA_PROMOS_PROSITNN',
    'BANCA_SANTA_GIULIA_SGIUIT21',
    'BANCA_SICANA_DI_SOMMATINO_SERRADIFALCO_E_SAMBUCA_DI_SICILIA_CCRTIT2TNIS',
    'BANCO_BPM_CORP_BAPPIT22',
    'BANCO_BPM_WEBANK_BAPPIT22',
    'BANCO_BPM_WEB_BAPPIT22',
    'BANCO_MARCHIGIANO_CREDITO_COOPERATIVO_CCRTIT2TCIV',
    'BCC_CASSA_RURALE_DI_VAL_DI_FIEMME_CCRTIT2T50A',
    'BCC_DEI_CASTELLI_E_DEGLI_IBLEI_CCRTIT2TIBL',
    'BCC_DEI_CASTELLI_ROMANI_CCRTIT2TCRT',
    'BCC_DELLALTA_MURGIA_CCRTIT2TMUR',
    'BCC_DELLALTO_TIRRENO_DELLA_CALABRIA_VERBICARO_CCRTIT2TATC',
    'BCC_DELLA_ROMAGNA_OCCIDENTALE_CCRTIT2TROC',
    'BCC_DELLE_PREALPI_CCRTIT2TPRE',
    'BCC_DEL_CIRCEO_E_PRIVERNATE_CCRTIT2TCIR',
    'BCC_DEL_FRIULI_CENTRALE_ICRAITRRU50',
    'BCC_DI_ALBEROBELLO_E_SAMMICHELE_DI_BARI_CCRTIT2TALB',
    'BCC_DI_ANAGNI_CCRTIT2TN00',
    'BCC_DI_AQUARA_CCRTIT2TAQU',
    'BCC_DI_BARLASSINA_MILANO_ICRAITRRAE0',
    'BCC_DI_BRESCIA_CREDITO_COOPERATIVO_DI_BRESCIA_CCRTIT2TH00',
    'BCC_DI_CASALGRASSO_E_SANTALBANO_STURA_CCRTIT2TCSS',
    'BCC_DI_CASSANO_DELLE_MURGE_E_TOLVE_CCRTIT2TCSM',
    'BCC_DI_CASTAGNETO_CARDUCCI_PER_AZIONI_BCCDIT3C',
    'BCC_DI_CHERASCO_CCRTIT2TCHE',
    'BCC_DI_CONVERSANO_CCRTIT2TT00',
    'BCC_DI_FLUMERI_ICRAITRRFD0',
    'BCC_DI_LOCOROTONDO_CASSA_RURALE_ED_ARTIGIANA_CCRTIT2TLND',
    'BCC_DI_PIANFEI_E_ROCCA_DE_BALDI_ICRAITRRKX0',
    'BCC_DI_SAN_GIOVANNI_ROTONDO_CCRTIT2TSGR',
    'BCC_DI_SAN_MARZANO_DI_SAN_GIUSEPPE_TARANTO_CCRTIT2TSMZ',
    'BCC_DI_SARSINA_CCRTIT2TSAR',
    'BCC_DI_SPELLO_E_BETTONA_CCRTIT2TSBE',
    'BCC_FELSINEA_CCRTIT2TFEL',
    'BCC_LAUDENSE_LODI_MILANO_CCRTIT2TLAU',
    'BCC_LA_RISCOSSA_DI_REGALBUTO_CCRTIT2TO00',
    'BCC_MONTE_PRUNO_DI_ROSCIGNO_E_DI_LAURINO_CCRTIT2TMPR',
    'BCC_SANGRO_TAETINA_DI_ATESSA_CASTIGLIONE_GIULIANO_CCRTIT2TTEA',
    'BCC_VALDOSTANA_CCRTIT2TVAL',
    'BENE_BANCA_CREDITO_COOPERATIVO_DI_BENE_VAGIENNA_CUNEO_CCRTIT2TBENBPER_CORP_BPMOIT22',
    'BPER_RETAIL_BPMOIT22',
    'CASSA_CENTRALE_BANCA_CCRTIT2T',
    'CASSA_PADANA_BCC_CCRTIT2TPAD',
    'CASSA_RAIFFEISEN_DI_SAN_MARTINO_IN_PASSIRIA_ICRAITRR6A0',
    'CASSA_RURALE_ALTA_VALLAGARINA_DI_BESENELLO_CALLIANO_NOMI_VOLANO_BANCA_DI_CEDITO_COOPERATIVO_CCRTIT2T77A',
    'CASSA_RURALE_ALTA_VALSUGANA_BCC_CCRTIT2T47A',
    'CASSA_RURALE_ALTO_GARDA_BCC_CCRTIT2T04A',
    'CASSA_RURALE_DELLA_BASSA_VALSUGANA_BCC_CCRTIT2T27A',
    'CASSA_RURALE_DI_LEDRO_BCC_CCRTIT2T06A',
    'CASSA_RURALE_DI_TRENTO_BCC_CCRTIT2T76A',
    'CASSA_RURALE_DOLOMITI_DI_FASSA_PRIMIERO_E_BELLUNO_BCC_CCRTIT2T38A',
    'CASSA_RURALE_ED_ARTIGIANA_DI_BOVES_BCC_ICRAITRRB10',
    'CASSA_RURALE_ED_ARTIGIANA_DI_CORTINA_DAMPEZZO_E_DELLE_DOLOMITI_CCRTIT2T95A',
    'CASSA_RURALE_GIUDICARIE_VALSABBIA_PAGANELLA_BCC_CCRTIT2T20A',
    'CASSA_RURALE_NOVELLA_E_ALTA_ANAUNIA_BCC_CCRTIT2T54A',
    'CASSA_RURALE_RENON_CCRTIT2TRIT',
    'CASSA_RURALE_ROTALIANA_E_GIOVO_BCC_CCRTIT2T37A',
    'CASSA_RURALE_VALLAGARINA_BCC_CCRTIT2T01A',
    'CASSA_RURALE_VAL_DI_NON_BCC_CCRTIT2T73A',
    'CASSA_RURALE_VAL_DI_SOLE_BCC_CCRTIT2T43A',
    'CENTROVENETO_BASSANO_BANCA_ICRAITRRGE0',
    'CHEBANCA_CORP_MICSITM1',
    'CREDEM_CORP_BACRIT22',
    'CREDEM_RETAIL_BACRIT22',
    'CREDITO_COOPERATIVO_CASSA_RURALE_ED_ARTIGIANA_DEL_FRIULI_VENEZIA_GIULIA_CCRTI2T96A',
    'CREDITO_COOPERATIVO_CENTRO_CALABRIA_CCRTIT2TCA',
    'CREDITO_ETNEO_BCC_CCRTIT2TETN',
    'CREDITO_LOMBARDO_VENETO_CRLVIT21',
    'FINDOMESTIC_FIDMIT31',
    'FRIULOVEST_BANCA_CCRTIT2TFOB',
    'GUBER_BANCA_GUBEIT21',
    'HELLO_BANK_BNLIITRRXXX',
    'HYPO_TIROL_BANK_HYPTIT2B',
    'INTESA_SANPAOLO_BCITITMMXXX',
    'INTESA_SANPAOLO_CORPORATE_BCITITMMXXX',
    'INTESA_SANPAOLO_PRIVATE_BCITITMMXXX',
    'ISYBANK_ITBBITMM',
    'MEDIOCREDITO_TRENTINO_ALTO_ADIGE_CCRTIT2TM00',
    'POSTE_ITALIANE_BANCO_POSTA_BPPIITRRXXX',
    'POSTE_ITALIANE_BANCO_POSTA_CORPORATE_BPPIITRRXXX',
    'POSTE_PAY_BPPIITRRXXX',
    'PRADER_BANK_PRRDIT21XXX',
    'ROMAGNABANCA_CCRTIT2TRMB',
    'ROVIGO_BANCA_ICRAITRRCK0',
    'SARDEGNA_CORP_SARDIT31',
    'SARDEGNA_RETAIL_SARDIT31',
    'SOLUTION_BANK_CRDRIT2FXXX',
    'WIDIBA_WIDIITMM',
    'ZADRUZNA_KRASKA_BANKA_TRST_GORICA_ZADRUGA_CCRTIT2TV00',
].map(i => i.replace('*', ''));
export function isSpecialContinuousAccessBank(institutionId) {
    return SPECIAL_CONTINUOUS_ACCESS_BANKS.some(i => institutionId.startsWith(i));
}
