/// <reference types="node" />
import { TrieNode } from './merkle';
/**
 * Hybrid Unique Logical Clock (HULC) timestamp generator
 *
 * Globally-unique, monotonic timestamps are generated from the
 *    combination of the unreliable system time, a counter, and an
 *    identifier for the current node (instance, machine, process, etc.).
 *    These timestamps can accommodate clock stuttering (duplicate values),
 *    regression, and node differences within the configured maximum drift.
 *
 * In order to generate timestamps locally or for transmission to another
 *    node, use the send() method. For global causality, timestamps must
 *    be included in each message processed by the system. Whenever a
 *    message is received, its timestamp must be passed to the recv()
 *    method.
 *
 * Timestamps serialize into a 46-character collatable string
 *    example: 2015-04-24T22:23:42.123Z-1000-0123456789ABCDEF
 *    example: 2015-04-24T22:23:42.123Z-1000-A219E7A71CC18912
 *
 * The 64-bit hybrid clock is based on the HLC specification,
 * http://www.cse.buffalo.edu/tech-reports/2014-04.pdf
 */
export type Clock = {
    timestamp: MutableTimestamp;
    merkle: TrieNode;
};
export declare function setClock(clock_: Clock): void;
export declare function getClock(): Clock;
export declare function makeClock(timestamp: Timestamp, merkle?: TrieNode): {
    timestamp: MutableTimestamp;
    merkle: TrieNode;
};
export declare function serializeClock(clock: Clock): string;
export declare function deserializeClock(clock: string): Clock;
export declare function makeClientId(): string;
/**
 * timestamp instance class
 */
export declare class Timestamp {
    _state: {
        millis: number;
        counter: number;
        node: string;
    };
    constructor(millis: number, counter: number, node: string);
    valueOf(): string;
    toString(): string;
    millis(): number;
    counter(): number;
    node(): string;
    hash(): number;
    static init(options?: {
        maxDrift?: number;
        node?: string;
    }): void;
    /**
     * maximum timestamp
     */
    static max: Timestamp;
    /**
     * timestamp parsing
     * converts a fixed-length string timestamp to the structured value
     */
    static parse(timestamp: string | Timestamp): Timestamp | null;
    /**
     * Timestamp send. Generates a unique, monotonic timestamp suitable
     * for transmission to another system in string format
     */
    static send(): Timestamp | null;
    static recv(msg: Timestamp): Timestamp | null;
    /**
     * zero/minimum timestamp
     */
    static zero: Timestamp;
    static since: (isoString: string) => string;
    /**
     * error classes
     */
    static DuplicateNodeError: {
        new (node: string): {
            name: string;
            message: string;
            stack?: string | undefined;
        };
        captureStackTrace(targetObject: object, constructorOpt?: Function | undefined): void;
        prepareStackTrace?: ((err: Error, stackTraces: NodeJS.CallSite[]) => any) | undefined;
        stackTraceLimit: number;
    };
    static ClockDriftError: {
        new (...args: unknown[]): {
            name: string;
            message: string;
            stack?: string | undefined;
        };
        captureStackTrace(targetObject: object, constructorOpt?: Function | undefined): void;
        prepareStackTrace?: ((err: Error, stackTraces: NodeJS.CallSite[]) => any) | undefined;
        stackTraceLimit: number;
    };
    static OverflowError: {
        new (): {
            name: string;
            message: string;
            stack?: string | undefined;
        };
        captureStackTrace(targetObject: object, constructorOpt?: Function | undefined): void;
        prepareStackTrace?: ((err: Error, stackTraces: NodeJS.CallSite[]) => any) | undefined;
        stackTraceLimit: number;
    };
    static InvalidError: {
        new (...args: unknown[]): {
            name: string;
            message: string;
            stack?: string | undefined;
        };
        captureStackTrace(targetObject: object, constructorOpt?: Function | undefined): void;
        prepareStackTrace?: ((err: Error, stackTraces: NodeJS.CallSite[]) => any) | undefined;
        stackTraceLimit: number;
    };
}
declare class MutableTimestamp extends Timestamp {
    static from(timestamp: Timestamp): MutableTimestamp;
    setMillis(n: number): void;
    setCounter(n: number): void;
    setNode(n: string): void;
}
export {};
