import type { PaymentCustomer } from './paymentCustomer';
import { PaymentRecipient } from './paymentRecipient';
export declare const DAYS_OF_WEEK: readonly [
    'MONDAY',
    'TUESDAY',
    'WEDNESDAY',
    'THURSDAY',
    'FRIDAY',
    'SATURDAY',
    'SUNDAY'
];
export declare type DaysOfWeek = typeof DAYS_OF_WEEK[number];
export declare const PAYMENT_REQUEST_STATUS: readonly ["CREATED", "IN_PROGRESS", "COMPLETED", "SCHEDULED", "WAITING_PAYER_AUTHORIZATION", "ERROR", "REFUND_IN_PROGRESS", "REFUNDED", "REFUND_ERROR", "EXPIRED", "AUTHORIZED", "CANCELED"];
/**
 * @typedef PaymentRequestStatus
 * Status of payment request where:
 * CREATED - Payment request was created
 * IN_PROGRESS - User started the payment process
 * WAITING_PAYER_AUTHORIZATION - User needs to authorize the payment in the payment institution
 * COMPLETED - Payment request was completed
 * ERROR - Payment request has an error
 * AUTHORIZED - Payment request authorized and active (for automatic PIX)
 * CANCELED - Payment request canceled or authorization revoked
 */
export declare type PaymentRequestStatus = typeof PAYMENT_REQUEST_STATUS[number];
export declare type CreatePaymentRequest = {
    amount: number;
    callbackUrls?: CallbackUrls;
    description?: string;
    recipientId?: string;
    customerId?: string;
    schedule?: PaymentRequestSchedule;
    smartAccountId?: string;
    clientPaymentId?: string;
    isSandbox: boolean;
};
export declare type PaymentRequestSchedule = SingleSchedule | DailySchedule | WeeklySchedule | MonthlySchedule | CustomSchedule;
export declare type SingleSchedule = {
    type: 'SINGLE';
    date: string;
};
export declare type DailySchedule = {
    type: 'DAILY';
    startDate: string;
    occurrences: number;
};
export declare type WeeklySchedule = {
    type: 'WEEKLY';
    startDate: string;
    occurrences: number;
    dayOfWeek: DaysOfWeek;
};
export declare type MonthlySchedule = {
    type: 'MONTHLY';
    startDate: string;
    occurrences: number;
    dayOfMonth: number;
};
export declare type CustomSchedule = {
    type: 'CUSTOM';
    dates: string[];
    additionalInformation?: string;
};
export declare const AUTOMATIC_PIX_INTERVALS: readonly ["WEEKLY", "MONTHLY", "QUARTERLY", "SEMESTER", "YEARLY"];
export declare type AutomaticPixInterval = typeof AUTOMATIC_PIX_INTERVALS[number];
export declare type AutomaticPixFirstPayment = {
    /**! date of the first payment, default to instant */
    date?: string;
    /**! amount of the first payment */
    amount: number;
    /**! description of the first payment */
    description?: string;
};
export declare type PaymentRequestAutomaticPixDetails = {
    /**! interval of the authorization */
    interval: AutomaticPixInterval;
    /**! start date of the authorization */
    startDate: string;
    /**! fixed amount of the payment, if provided can't be used with minimumVariableAmount or maximumVariableAmount */
    fixedAmount?: number;
    /**! minimum variable amount of the payment, if provided can't be used with fixedAmount */
    minimumVariableAmount?: number;
    /**! maximum variable amount of the payment, if provided can't be used with fixedAmount */
    maximumVariableAmount?: number;
    /**! expiration date of the authorization */
    expiresAt?: string;
    /**! if the payments done for this authorization can be retried */
    isRetryAccepted?: boolean;
    /**! if provided, will execute a first payment */
    firstPayment?: AutomaticPixFirstPayment;
};
export declare type PaymentRequest = {
    /**! primary identifier */
    id: string;
    /**! payment to send the user to pay */
    paymentUrl: string;
    /**! amount to be paid */
    amount: number;
    /**! description of the payment */
    description: string | null;
    /**! status of the payment request */
    status: PaymentRequestStatus;
    /**! where the user will be redirected when the payment finishes */
    callbackUrls: CallbackUrls | null;
    /** if the payment request is inside a bulk payment, primary identifier of the bulk */
    bulkPaymentId: string | null;
    /**! recipient of the payment */
    recipient: PaymentRecipient | null;
    /**! customer that will pay the payment */
    customer: PaymentCustomer | null;
    /**! fees that will be charged to the customer (related to bulk) */
    fees: number | null;
    /**! schedule of the payment */
    schedule: PaymentRequestSchedule | null;
    /**! automaticPix of the payment */
    automaticPix: PaymentRequestAutomaticPixDetails | null;
    /**! createdAt date */
    createdAt: Date;
    /**! updatedAt date */
    updatedAt: Date;
    /**! Identification from client that can be used to track payment */
    clientPaymentId: string | null;
    /**! Indicates if the paymentRequest is sandbox */
    isSandbox: boolean;
};
export declare type PaymentRequestAutomaticPix = Omit<PaymentRequest, 'amount'>;
export declare type CallbackUrls = {
    success?: string;
    error?: string;
};
export declare type CreatePaymentRequestAutomaticPix = Pick<CreatePaymentRequest, 'description' | 'recipientId' | 'customerId' | 'callbackUrls' | 'clientPaymentId'> & PaymentRequestAutomaticPixDetails & {
    isSandbox: boolean;
};
