/**
 * Requestarr Core - Main class, initialization, and view management
 */

import { RequestarrContent } from './requestarr-content.js';
import { RequestarrSearch } from './requestarr-search.js';
import { RequestarrModal } from './requestarr-modal.js';
import { RequestarrSettings } from './requestarr-settings.js';
import { RequestarrFilters } from './requestarr-filters.js';
import { RequestarrTVFilters } from './requestarr-tv-filters.js';

/**
 * Encode a compound instance value: "appType:instanceName"
 */
export function encodeInstanceValue(appType, name) {
    return `${appType}:${name}`;
}

/**
 * Decode a compound instance value back to { appType, name }.
 * Backward compat: values without ':' use defaultAppType (radarr for movies, sonarr for TV).
 */
export function decodeInstanceValue(value, defaultAppType = 'radarr') {
    if (!value) return { appType: defaultAppType, name: '' };
    const idx = value.indexOf(':');
    if (idx === -1) return { appType: defaultAppType, name: value };
    return { appType: value.substring(0, idx), name: value.substring(idx + 1) };
}

export class RequestarrDiscover {
    constructor() {
        this.currentView = 'discover';
        this.instances = { sonarr: [], radarr: [], movie_hunt: [], tv_hunt: [] };
        this.qualityProfiles = {};
        this.searchTimeouts = {};
        this.currentModal = null;
        this.currentModalData = null;
        
        // Initialize modules
        this.content = new RequestarrContent(this);
        this.search = new RequestarrSearch(this);
        this.modal = new RequestarrModal(this);
        this.settings = new RequestarrSettings(this);
        this.filters = new RequestarrFilters(this);
        this.tvFilters = new RequestarrTVFilters(this);
        
        this.init();
    }

    // ========================================
    // INITIALIZATION
    // ========================================

    init() {
        this.loadInstances();
        this.setupCarouselArrows();
        this.search.setupGlobalSearch();
        this.content.loadDiscoverContent();
    }

    async loadInstances() {
        try {
            const _ts = Date.now();
            const response = await fetch(`./api/requestarr/instances?t=${_ts}`, { cache: 'no-store' });
            const data = await response.json();
            
            if (data.sonarr || data.radarr || data.movie_hunt || data.tv_hunt) {
                this.instances = {
                    sonarr: data.sonarr || [],
                    radarr: data.radarr || [],
                    movie_hunt: data.movie_hunt || [],
                    tv_hunt: data.tv_hunt || []
                };
                await this.loadAllQualityProfiles();
            }
        } catch (error) {
            console.error('[RequestarrDiscover] Error loading instances:', error);
        }
    }
    
    async loadAllQualityProfiles() {
        // Load Radarr quality profiles
        for (const instance of this.instances.radarr) {
            await this.loadQualityProfilesForInstance('radarr', instance.name);
        }
        
        // Load Sonarr quality profiles
        for (const instance of this.instances.sonarr) {
            await this.loadQualityProfilesForInstance('sonarr', instance.name);
        }
        
        // Load Movie Hunt quality profiles
        for (const instance of this.instances.movie_hunt) {
            await this.loadQualityProfilesForInstance('movie_hunt', instance.name);
        }
        
        // Load TV Hunt quality profiles
        for (const instance of this.instances.tv_hunt) {
            await this.loadQualityProfilesForInstance('tv_hunt', instance.name);
        }
    }

    async loadQualityProfilesForInstance(appType, instanceName) {
        try {
            const response = await fetch(`./api/requestarr/quality-profiles/${appType}/${encodeURIComponent(instanceName)}`);
            const data = await response.json();
            if (data.success) {
                this.qualityProfiles[`${appType}-${instanceName}`] = data.profiles;
                return data.profiles;
            }
        } catch (error) {
            console.error(`[RequestarrDiscover] Error loading quality profiles for ${appType}/${instanceName}:`, error);
        }
        return [];
    }

    // ========================================
    // VIEW MANAGEMENT
    // ========================================

    switchView(view) {
        console.log(`[RequestarrDiscover] switchView called with: ${view}`);
        
        // Clear global search
        const globalSearch = document.getElementById('global-search-input');
        if (globalSearch) {
            globalSearch.value = '';
        }
        
        // Hide/show global search bar based on view
        // Use ID to find input, then get parent to ensure we have the right element
        let globalSearchBar = null;
        if (globalSearch) {
            globalSearchBar = globalSearch.closest('.global-search-bar');
        } else {
            // Fallback
            globalSearchBar = document.querySelector('#requestarr-section .global-search-bar');
        }

        if (globalSearchBar) {
            console.log(`[RequestarrDiscover] Found global search bar, applying visibility for ${view}`);
            if (view === 'hidden' || view === 'settings' || view === 'smarthunt-settings') {
                globalSearchBar.style.setProperty('display', 'none', 'important');
                console.log('[RequestarrDiscover] Hiding global search bar');
            } else {
                globalSearchBar.style.setProperty('display', 'flex', 'important');
                console.log('[RequestarrDiscover] Showing global search bar');
            }
        } else {
            console.error('[RequestarrDiscover] Global search bar not found!');
        }
        
        // Hide search results view
        const searchResultsView = document.getElementById('search-results-view');
        if (searchResultsView) {
            searchResultsView.style.display = 'none';
        }
        
        // Hide all view headers, show the one for current view (settings/smarthunt have their own toolbar)
        document.querySelectorAll('.requestarr-view-header').forEach(el => {
            el.style.display = 'none';
        });
        // Hide the entire header bar when settings/smarthunt-settings have their own toolbar
        const headerBar = document.querySelector('.requestarr-header-bar');
        const contentEl = document.querySelector('.requestarr-content');
        if (view === 'settings' || view === 'smarthunt-settings') {
            if (headerBar) headerBar.style.display = 'none';
            // Allow dropdowns to overflow outside cards in settings view
            if (contentEl) contentEl.classList.add('settings-active');
        } else {
            if (headerBar) headerBar.style.display = '';
            if (contentEl) contentEl.classList.remove('settings-active');
            const headerEl = document.getElementById(`requestarr-header-${view}`);
            if (headerEl) {
                headerEl.style.display = '';
            }
        }

        // Hide all views
        document.querySelectorAll('.requestarr-view').forEach(container => {
            container.classList.remove('active');
            container.style.display = 'none';
        });
        
        // Show target view
        const targetView = document.getElementById(`requestarr-${view}-view`);
        if (targetView) {
            targetView.classList.add('active');
            targetView.style.display = 'block';
        }

        this.currentView = view;

        // Load content for view — always refresh on navigate to avoid stale cache
        switch (view) {
            case 'discover':
                this.content.loadDiscoverContent();
                break;
            case 'movies':
                // Always ensure instance selectors are populated before loading.
                // setupInstanceSelectors → loadMovieInstances short-circuits if already populated.
                this.content.setupInstanceSelectors().then(() => {
                    this.content.moviesPage = 1;
                    this.content.moviesHasMore = true;
                    this.content.loadMovies();
                    this.content.setupMoviesInfiniteScroll();
                });
                break;
            case 'tv':
                // Always ensure instance selectors are populated before loading.
                this.content.setupInstanceSelectors().then(() => {
                    this.content.tvPage = 1;
                    this.content.tvHasMore = true;
                    this.content.loadTV();
                    this.content.setupTVInfiniteScroll();
                });
                break;
            case 'hidden':
                this.settings.loadHiddenMedia();
                break;
            case 'settings':
                this.settings.loadSettings();
                break;
            case 'smarthunt-settings':
                this.settings.loadSmartHuntSettings();
                break;
        }
    }

    setupCarouselArrows() {
        const arrows = document.querySelectorAll('.carousel-arrow');
        const carousels = new Set();
        /** Per-carousel: once user has scrolled right, left arrow stays visible (so they know they can scroll back). */
        const hasScrolledRight = {};
        
        // Collect all unique carousels
        arrows.forEach(arrow => {
            const targetId = arrow.dataset.target;
            const carousel = document.getElementById(targetId);
            if (carousel) {
                carousels.add(carousel);
            }
        });
        
        // Setup scroll listeners for each carousel
        carousels.forEach(carousel => {
            const updateArrowVisibility = () => {
                const carouselId = carousel.id;
                const leftArrow = document.querySelector(`.carousel-arrow.left[data-target="${carouselId}"]`);
                const rightArrow = document.querySelector(`.carousel-arrow.right[data-target="${carouselId}"]`);
                
                if (!leftArrow || !rightArrow) return;
                
                const scrollLeft = carousel.scrollLeft;
                const maxScroll = carousel.scrollWidth - carousel.clientWidth;
                const atStart = scrollLeft <= 5;
                const atEnd = maxScroll > 5 && scrollLeft >= maxScroll - 5;
                
                // Once user scrolls right, left arrow stays visible so they know they can scroll back
                if (!atStart) {
                    hasScrolledRight[carouselId] = true;
                }
                
                // Left arrow: hidden at start until user scrolls right; then always visible
                if (atStart && !hasScrolledRight[carouselId]) {
                    leftArrow.style.opacity = '0';
                    leftArrow.style.pointerEvents = 'none';
                } else {
                    leftArrow.style.opacity = '0.8';
                    leftArrow.style.pointerEvents = 'auto';
                }
                
                // Right arrow: always visible when there's more content (or content still loading); hide only at end
                if (atEnd) {
                    rightArrow.style.opacity = '0';
                    rightArrow.style.pointerEvents = 'none';
                } else {
                    rightArrow.style.opacity = '0.8';
                    rightArrow.style.pointerEvents = 'auto';
                }
            };
            
            carousel.addEventListener('scroll', updateArrowVisibility);
            setTimeout(() => updateArrowVisibility(), 100);
            window.addEventListener('resize', updateArrowVisibility);
            // When carousel content loads (e.g. async), update arrows so right arrow becomes visible
            const observer = new MutationObserver(() => {
                updateArrowVisibility();
            });
            observer.observe(carousel, { childList: true, subtree: true });
        });
        
        // Click handlers
        arrows.forEach(arrow => {
            arrow.addEventListener('click', (e) => {
                const targetId = arrow.dataset.target;
                const carousel = document.getElementById(targetId);
                
                const carouselWidth = carousel.offsetWidth;
                const cardWidth = 150;
                const gap = 20;
                const itemWidth = cardWidth + gap;
                const visibleItems = Math.floor(carouselWidth / itemWidth);
                const scrollAmount = visibleItems * itemWidth;
                
                if (arrow.classList.contains('left')) {
                    carousel.scrollBy({ left: -scrollAmount, behavior: 'smooth' });
                } else {
                    carousel.scrollBy({ left: scrollAmount, behavior: 'smooth' });
                }
            });
        });
    }

    // ========================================
    // UTILITIES
    // ========================================

    closeFiltersModal() {
        if (this.filters) {
            this.filters.closeFiltersModal();
        }
    }
    
    closeTVFiltersModal() {
        if (this.tvFilters) {
            this.tvFilters.closeFiltersModal();
        }
    }

    applyFilters() {
        if (this.filters) {
            this.filters.applyFilters();
        }
    }

    clearFilters() {
        if (this.filters) {
            this.filters.clearFilters();
        }
    }

    showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `requestarr-notification ${type}`;
        notification.innerHTML = `
            <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
            <span>${message}</span>
        `;
        
        document.body.appendChild(notification);
        
        setTimeout(() => notification.classList.add('show'), 10);
        
        setTimeout(() => {
            notification.classList.remove('show');
            notification.classList.add('slideOut');
            setTimeout(() => notification.remove(), 300);
        }, 4000);
    }
}
