import { MachineLearningConfig } from 'src/config';
import { CLIPConfig } from 'src/dtos/model-config.dto';
import { LoggingRepository } from 'src/repositories/logging.repository';
export interface BoundingBox {
    x1: number;
    y1: number;
    x2: number;
    y2: number;
}
export declare enum ModelTask {
    FACIAL_RECOGNITION = "facial-recognition",
    SEARCH = "clip",
    OCR = "ocr"
}
export declare enum ModelType {
    DETECTION = "detection",
    PIPELINE = "pipeline",
    RECOGNITION = "recognition",
    TEXTUAL = "textual",
    VISUAL = "visual",
    OCR = "ocr"
}
export type ModelPayload = {
    imagePath: string;
} | {
    text: string;
};
type ModelOptions = {
    modelName: string;
};
export type FaceDetectionOptions = ModelOptions & {
    minScore: number;
};
export type OcrOptions = ModelOptions & {
    minDetectionScore: number;
    minRecognitionScore: number;
    maxResolution: number;
};
type VisualResponse = {
    imageHeight: number;
    imageWidth: number;
};
export type ClipVisualRequest = {
    [ModelTask.SEARCH]: {
        [ModelType.VISUAL]: ModelOptions;
    };
};
export type ClipVisualResponse = {
    [ModelTask.SEARCH]: string;
} & VisualResponse;
export type ClipTextualRequest = {
    [ModelTask.SEARCH]: {
        [ModelType.TEXTUAL]: ModelOptions;
    };
};
export type ClipTextualResponse = {
    [ModelTask.SEARCH]: string;
};
export type OCR = {
    text: string[];
    box: number[];
    boxScore: number[];
    textScore: number[];
};
export type OcrRequest = {
    [ModelTask.OCR]: {
        [ModelType.DETECTION]: ModelOptions & {
            options: {
                minScore: number;
                maxResolution: number;
            };
        };
        [ModelType.RECOGNITION]: ModelOptions & {
            options: {
                minScore: number;
            };
        };
    };
};
export type OcrResponse = {
    [ModelTask.OCR]: OCR;
} & VisualResponse;
export type FacialRecognitionRequest = {
    [ModelTask.FACIAL_RECOGNITION]: {
        [ModelType.DETECTION]: ModelOptions & {
            options: {
                minScore: number;
            };
        };
        [ModelType.RECOGNITION]: ModelOptions;
    };
};
export interface Face {
    boundingBox: BoundingBox;
    embedding: string;
    score: number;
}
export type FacialRecognitionResponse = {
    [ModelTask.FACIAL_RECOGNITION]: Face[];
} & VisualResponse;
export type DetectedFaces = {
    faces: Face[];
} & VisualResponse;
export type MachineLearningRequest = ClipVisualRequest | ClipTextualRequest | FacialRecognitionRequest | OcrRequest;
export type TextEncodingOptions = ModelOptions & {
    language?: string;
};
export declare class MachineLearningRepository {
    private logger;
    private healthyMap;
    private interval?;
    private _config?;
    private get config();
    constructor(logger: LoggingRepository);
    setup(config: MachineLearningConfig): void;
    teardown(): void;
    private tick;
    private check;
    private setHealthy;
    private isHealthy;
    private predict;
    detectFaces(imagePath: string, { modelName, minScore }: FaceDetectionOptions): Promise<{
        imageHeight: number;
        imageWidth: number;
        faces: Face[];
    }>;
    encodeImage(imagePath: string, { modelName }: CLIPConfig): Promise<string>;
    encodeText(text: string, { language, modelName }: TextEncodingOptions): Promise<string>;
    ocr(imagePath: string, { modelName, minDetectionScore, minRecognitionScore, maxResolution }: OcrOptions): Promise<OCR>;
    private getFormData;
}
export {};
