"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseBackupService = void 0;
const common_1 = require("@nestjs/common");
const lodash_1 = require("lodash");
const luxon_1 = require("luxon");
const node_path_1 = __importStar(require("node:path"));
const node_stream_1 = require("node:stream");
const promises_1 = require("node:stream/promises");
const semver_1 = __importDefault(require("semver"));
const constants_1 = require("../constants");
const storage_core_1 = require("../cores/storage.core");
const decorators_1 = require("../decorators");
const enum_1 = require("../enum");
const maintenance_health_repository_1 = require("../maintenance/maintenance-health.repository");
const config_repository_1 = require("../repositories/config.repository");
const cron_repository_1 = require("../repositories/cron.repository");
const database_repository_1 = require("../repositories/database.repository");
const job_repository_1 = require("../repositories/job.repository");
const logging_repository_1 = require("../repositories/logging.repository");
const process_repository_1 = require("../repositories/process.repository");
const storage_repository_1 = require("../repositories/storage.repository");
const system_metadata_repository_1 = require("../repositories/system-metadata.repository");
const config_1 = require("../utils/config");
const database_backups_1 = require("../utils/database-backups");
const misc_1 = require("../utils/misc");
let DatabaseBackupService = class DatabaseBackupService {
    logger;
    storageRepository;
    configRepository;
    systemMetadataRepository;
    processRepository;
    databaseRepository;
    cronRepository;
    jobRepository;
    maintenanceHealthRepository;
    constructor(logger, storageRepository, configRepository, systemMetadataRepository, processRepository, databaseRepository, cronRepository, jobRepository, maintenanceHealthRepository) {
        this.logger = logger;
        this.storageRepository = storageRepository;
        this.configRepository = configRepository;
        this.systemMetadataRepository = systemMetadataRepository;
        this.processRepository = processRepository;
        this.databaseRepository = databaseRepository;
        this.cronRepository = cronRepository;
        this.jobRepository = jobRepository;
        this.maintenanceHealthRepository = maintenanceHealthRepository;
        this.logger.setContext(this.constructor.name);
    }
    backupLock = false;
    async onConfigInit({ newConfig: { backup: { database }, }, }) {
        if (!this.cronRepository || !this.jobRepository) {
            return;
        }
        this.backupLock = await this.databaseRepository.tryLock(enum_1.DatabaseLock.BackupDatabase);
        if (this.backupLock) {
            this.cronRepository.create({
                name: 'backupDatabase',
                expression: database.cronExpression,
                onTick: () => (0, misc_1.handlePromiseError)(this.jobRepository.queue({ name: enum_1.JobName.DatabaseBackup }), this.logger),
                start: database.enabled,
            });
        }
    }
    onConfigUpdate({ newConfig: { backup } }) {
        if (!this.cronRepository || !this.jobRepository || !this.backupLock) {
            return;
        }
        this.cronRepository.update({
            name: 'backupDatabase',
            expression: backup.database.cronExpression,
            start: backup.database.enabled,
        });
    }
    async handleBackupDatabase() {
        try {
            await this.createDatabaseBackup();
        }
        catch (error) {
            if (error instanceof database_backups_1.UnsupportedPostgresError) {
                return enum_1.JobStatus.Failed;
            }
            throw error;
        }
        await this.cleanupDatabaseBackups();
        return enum_1.JobStatus.Success;
    }
    async buildPostgresLaunchArguments(bin, options = {}) {
        const { database: { config: databaseConfig }, } = this.configRepository.getEnv();
        const isUrlConnection = databaseConfig.connectionType === 'url';
        const databaseVersion = await this.databaseRepository.getPostgresVersion();
        const databaseSemver = semver_1.default.coerce(databaseVersion);
        const databaseMajorVersion = databaseSemver?.major;
        const args = [];
        let databaseUsername;
        if (isUrlConnection) {
            if (bin !== 'pg_dump') {
                args.push('--dbname');
            }
            let url = databaseConfig.url;
            if (URL.canParse(databaseConfig.url)) {
                const parsedUrl = new URL(databaseConfig.url);
                parsedUrl.searchParams.delete('uselibpqcompat');
                databaseUsername = parsedUrl.username;
                url = parsedUrl.toString();
            }
            databaseUsername ??= 'postgres';
            args.push(url);
        }
        else {
            databaseUsername = databaseConfig.username;
            args.push('--username', databaseUsername, '--host', databaseConfig.host, '--port', databaseConfig.port.toString());
            switch (bin) {
                case 'pg_dumpall': {
                    args.push('--database');
                    break;
                }
                case 'psql': {
                    args.push('--dbname');
                    break;
                }
            }
            args.push(databaseConfig.database);
        }
        switch (bin) {
            case 'pg_dump':
            case 'pg_dumpall': {
                args.push('--clean', '--if-exists');
                break;
            }
            case 'psql': {
                if (options.singleTransaction) {
                    args.push('--single-transaction', '--set', 'ON_ERROR_STOP=on');
                }
                args.push('--echo-all', '--output=/dev/null');
                break;
            }
        }
        if (!databaseMajorVersion || !databaseSemver || !semver_1.default.satisfies(databaseSemver, '>=14.0.0 <19.0.0')) {
            this.logger.error(`Database Restore Failure: Unsupported PostgreSQL version: ${databaseVersion}`);
            throw new database_backups_1.UnsupportedPostgresError(databaseVersion);
        }
        return {
            bin: `/usr/bin/${bin}`,
            args,
            databaseUsername,
            databasePassword: isUrlConnection ? new URL(databaseConfig.url).password : databaseConfig.password,
            databaseVersion,
            databaseMajorVersion,
        };
    }
    async createDatabaseBackup(filenamePrefix = '') {
        this.logger.debug(`Database Backup Started`);
        const { bin, args, databasePassword, databaseVersion, databaseMajorVersion } = await this.buildPostgresLaunchArguments('pg_dump');
        this.logger.log(`Database Backup Starting. Database Version: ${databaseMajorVersion}`);
        const filename = `${filenamePrefix}immich-db-backup-${luxon_1.DateTime.now().toFormat("yyyyLLdd'T'HHmmss")}-v${constants_1.serverVersion.toString()}-pg${databaseVersion.split(' ')[0]}.sql.gz`;
        const backupFilePath = node_path_1.default.join(storage_core_1.StorageCore.getBaseFolder(enum_1.StorageFolder.Backups), filename);
        const temporaryFilePath = `${backupFilePath}.tmp`;
        try {
            const pgdump = this.processRepository.spawnDuplexStream(bin, args, {
                env: {
                    PATH: process.env.PATH,
                    PGPASSWORD: databasePassword,
                },
            });
            const gzip = this.processRepository.spawnDuplexStream('gzip', ['--rsyncable']);
            const fileStream = this.storageRepository.createWriteStream(temporaryFilePath);
            await (0, promises_1.pipeline)(pgdump, gzip, fileStream);
            await this.storageRepository.rename(temporaryFilePath, backupFilePath);
        }
        catch (error) {
            this.logger.error(`Database Backup Failure: ${error}`);
            await this.storageRepository
                .unlink(temporaryFilePath)
                .catch((error) => this.logger.error(`Failed to delete failed backup file: ${error}`));
            throw error;
        }
        this.logger.log(`Database Backup Success`);
        return backupFilePath;
    }
    async uploadBackup(file) {
        const backupsFolder = storage_core_1.StorageCore.getBaseFolder(enum_1.StorageFolder.Backups);
        const fn = (0, node_path_1.basename)(file.originalname);
        if (!(0, database_backups_1.isValidDatabaseBackupName)(fn)) {
            throw new common_1.BadRequestException('Invalid backup name!');
        }
        const filePath = node_path_1.default.join(backupsFolder, `uploaded-${fn}`);
        await this.storageRepository.createOrOverwriteFile(filePath, file.buffer);
    }
    downloadBackup(fileName) {
        if (!(0, database_backups_1.isValidDatabaseBackupName)(fileName)) {
            throw new common_1.BadRequestException('Invalid backup name!');
        }
        const filePath = node_path_1.default.join(storage_core_1.StorageCore.getBaseFolder(enum_1.StorageFolder.Backups), fileName);
        return {
            path: filePath,
            fileName,
            cacheControl: enum_1.CacheControl.PrivateWithoutCache,
            contentType: fileName.endsWith('.gz') ? 'application/gzip' : 'application/sql',
        };
    }
    async listBackups() {
        const backupsFolder = storage_core_1.StorageCore.getBaseFolder(enum_1.StorageFolder.Backups);
        const files = await this.storageRepository.readdir(backupsFolder);
        const validFiles = files
            .filter((fn) => (0, database_backups_1.isValidDatabaseBackupName)(fn))
            .toSorted((a, b) => (a.startsWith('uploaded-') === b.startsWith('uploaded-') ? a.localeCompare(b) : 1))
            .toReversed();
        const backups = await Promise.all(validFiles.map(async (filename) => {
            const stats = await this.storageRepository.stat(node_path_1.default.join(backupsFolder, filename));
            return { filename, filesize: stats.size };
        }));
        return {
            backups,
        };
    }
    async deleteBackup(files) {
        const backupsFolder = storage_core_1.StorageCore.getBaseFolder(enum_1.StorageFolder.Backups);
        if (files.some((filename) => !(0, database_backups_1.isValidDatabaseBackupName)(filename))) {
            throw new common_1.BadRequestException('Invalid backup name!');
        }
        await Promise.all(files.map((filename) => this.storageRepository.unlink(node_path_1.default.join(backupsFolder, filename))));
    }
    async cleanupDatabaseBackups() {
        this.logger.debug(`Database Backup Cleanup Started`);
        const { backup: { database: config }, } = await (0, config_1.getConfig)({
            configRepo: this.configRepository,
            metadataRepo: this.systemMetadataRepository,
            logger: this.logger,
        }, {
            withCache: false,
        });
        const backupsFolder = storage_core_1.StorageCore.getBaseFolder(enum_1.StorageFolder.Backups);
        const files = await this.storageRepository.readdir(backupsFolder);
        const backups = files
            .filter((filename) => (0, database_backups_1.isValidDatabaseRoutineBackupName)(filename))
            .toSorted()
            .toReversed();
        const failedBackups = files.filter((filename) => (0, database_backups_1.isFailedDatabaseBackupName)(filename));
        const toDelete = backups.slice(config.keepLastAmount);
        toDelete.push(...failedBackups);
        for (const file of toDelete) {
            await this.storageRepository.unlink(node_path_1.default.join(backupsFolder, file));
        }
        this.logger.debug(`Database Backup Cleanup Finished, deleted ${toDelete.length} backups`);
    }
    async restoreDatabaseBackup(filename, progressCb) {
        this.logger.debug(`Database Restore Started`);
        let complete = false;
        try {
            if (!(0, database_backups_1.isValidDatabaseBackupName)(filename)) {
                throw new Error('Invalid backup file format!');
            }
            const backupFilePath = node_path_1.default.join(storage_core_1.StorageCore.getBaseFolder(enum_1.StorageFolder.Backups), filename);
            await this.storageRepository.stat(backupFilePath);
            let isPgClusterDump = false;
            const version = (0, database_backups_1.findDatabaseBackupVersion)(filename);
            if (version && semver_1.default.satisfies(version, '<= 2.4')) {
                isPgClusterDump = true;
            }
            const { bin, args, databaseUsername, databasePassword, databaseMajorVersion } = await this.buildPostgresLaunchArguments('psql', {
                singleTransaction: !isPgClusterDump,
            });
            progressCb?.('backup', 0.05);
            const restorePointFilePath = await this.createDatabaseBackup('restore-point-');
            this.logger.log(`Database Restore Starting. Database Version: ${databaseMajorVersion}`);
            let inputStream;
            if (backupFilePath.endsWith('.gz')) {
                const fileStream = this.storageRepository.createPlainReadStream(backupFilePath);
                const gunzip = this.storageRepository.createGunzip();
                fileStream.pipe(gunzip);
                inputStream = gunzip;
            }
            else {
                inputStream = this.storageRepository.createPlainReadStream(backupFilePath);
            }
            const sqlStream = node_stream_1.Readable.from(sql(inputStream, databaseUsername, isPgClusterDump));
            const psql = this.processRepository.spawnDuplexStream(bin, args, {
                env: {
                    PATH: process.env.PATH,
                    PGPASSWORD: databasePassword,
                },
            });
            const [progressSource, progressSink] = createSqlProgressStreams((progress) => {
                if (complete) {
                    return;
                }
                this.logger.log(`Restore progress ~ ${(progress * 100).toFixed(2)}%`);
                progressCb?.('restore', progress);
            });
            await (0, promises_1.pipeline)(sqlStream, progressSource, psql, progressSink);
            try {
                progressCb?.('migrations', 0.9);
                await this.databaseRepository.runMigrations();
                await this.maintenanceHealthRepository.checkApiHealth();
            }
            catch (error) {
                progressCb?.('rollback', 0);
                const fileStream = this.storageRepository.createPlainReadStream(restorePointFilePath);
                const gunzip = this.storageRepository.createGunzip();
                fileStream.pipe(gunzip);
                inputStream = gunzip;
                const sqlStream = node_stream_1.Readable.from(sqlRollback(inputStream, databaseUsername));
                const psql = this.processRepository.spawnDuplexStream(bin, args, {
                    env: {
                        PATH: process.env.PATH,
                        PGPASSWORD: databasePassword,
                    },
                });
                const [progressSource, progressSink] = createSqlProgressStreams((progress) => {
                    if (complete) {
                        return;
                    }
                    this.logger.log(`Rollback progress ~ ${(progress * 100).toFixed(2)}%`);
                    progressCb?.('rollback', progress);
                });
                await (0, promises_1.pipeline)(sqlStream, progressSource, psql, progressSink);
                throw error;
            }
        }
        catch (error) {
            this.logger.error(`Database Restore Failure: ${error}`);
            throw error;
        }
        finally {
            complete = true;
        }
        this.logger.log(`Database Restore Success`);
    }
};
exports.DatabaseBackupService = DatabaseBackupService;
__decorate([
    (0, decorators_1.OnEvent)({ name: 'ConfigInit', workers: [enum_1.ImmichWorker.Microservices] }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], DatabaseBackupService.prototype, "onConfigInit", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'ConfigUpdate', server: true }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", void 0)
], DatabaseBackupService.prototype, "onConfigUpdate", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.DatabaseBackup, queue: enum_1.QueueName.BackupDatabase }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], DatabaseBackupService.prototype, "handleBackupDatabase", null);
exports.DatabaseBackupService = DatabaseBackupService = __decorate([
    (0, common_1.Injectable)(),
    __param(6, (0, common_1.Optional)()),
    __param(7, (0, common_1.Optional)()),
    __param(8, (0, common_1.Optional)()),
    __metadata("design:paramtypes", [logging_repository_1.LoggingRepository,
        storage_repository_1.StorageRepository,
        config_repository_1.ConfigRepository,
        system_metadata_repository_1.SystemMetadataRepository,
        process_repository_1.ProcessRepository,
        database_repository_1.DatabaseRepository,
        cron_repository_1.CronRepository,
        job_repository_1.JobRepository,
        maintenance_health_repository_1.MaintenanceHealthRepository])
], DatabaseBackupService);
const SQL_DROP_CONNECTIONS = `
  -- drop all other database connections
  SELECT pg_terminate_backend(pid)
  FROM pg_stat_activity
  WHERE datname = current_database()
    AND pid <> pg_backend_pid();
`;
const SQL_RESET_SCHEMA = (username) => `
  -- re-create the default schema
  DROP SCHEMA public CASCADE;
  CREATE SCHEMA public;

  -- restore access to schema
  GRANT ALL ON SCHEMA public TO "${username}";
  GRANT ALL ON SCHEMA public TO public;
`;
async function* sql(inputStream, databaseUsername, isPgClusterDump) {
    yield SQL_DROP_CONNECTIONS;
    yield isPgClusterDump
        ?
            String.raw `
        \c postgres
      `
        : SQL_RESET_SCHEMA(databaseUsername);
    for await (const chunk of inputStream) {
        yield chunk;
    }
}
async function* sqlRollback(inputStream, databaseUsername) {
    yield SQL_DROP_CONNECTIONS;
    yield SQL_RESET_SCHEMA(databaseUsername);
    for await (const chunk of inputStream) {
        yield chunk;
    }
}
function createSqlProgressStreams(cb) {
    const STDIN_START_MARKER = new TextEncoder().encode('FROM stdin');
    const STDIN_END_MARKER = new TextEncoder().encode(String.raw `\.`);
    let readingStdin = false;
    let sequenceIdx = 0;
    let linesSent = 0;
    let linesProcessed = 0;
    const startedAt = +Date.now();
    const cbDebounced = (0, lodash_1.debounce)(() => {
        const progress = source.writableEnded
            ? Math.min(1, linesProcessed / linesSent)
            :
                Math.min(0.3, 0.1 + (Date.now() - startedAt) / 1e4);
        cb(progress);
    }, 100, {
        maxWait: 100,
    });
    let lastByte = -1;
    const source = new node_stream_1.PassThrough({
        transform(chunk, _encoding, callback) {
            for (const byte of chunk) {
                if (!readingStdin && byte === 10 && lastByte !== 10) {
                    linesSent += 1;
                }
                lastByte = byte;
                const sequence = readingStdin ? STDIN_END_MARKER : STDIN_START_MARKER;
                if (sequence[sequenceIdx] === byte) {
                    sequenceIdx += 1;
                    if (sequence.length === sequenceIdx) {
                        sequenceIdx = 0;
                        readingStdin = !readingStdin;
                    }
                }
                else {
                    sequenceIdx = 0;
                }
            }
            cbDebounced();
            this.push(chunk);
            callback();
        },
    });
    const sink = new node_stream_1.Writable({
        write(chunk, _encoding, callback) {
            for (const byte of chunk) {
                if (byte === 10) {
                    linesProcessed++;
                }
            }
            cbDebounced();
            callback();
        },
    });
    return [source, sink];
}
//# sourceMappingURL=database-backup.service.js.map