"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseContext = void 0;
const default_naming_1 = require("../naming/default.naming");
const hash_naming_1 = require("../naming/hash.naming");
const asOverrideKey = (type, name) => `${type}:${name}`;
const isNamingInterface = (strategy) => {
    return typeof strategy === 'object' && typeof strategy.getName === 'function';
};
const asNamingStrategy = (strategy) => {
    if (isNamingInterface(strategy)) {
        return strategy;
    }
    switch (strategy) {
        case 'hash': {
            return new hash_naming_1.HashNamingStrategy();
        }
        default: {
            return new default_naming_1.DefaultNamingStrategy();
        }
    }
};
class BaseContext {
    databaseName;
    schemaName;
    overrideTableName;
    tables = [];
    functions = [];
    enums = [];
    extensions = [];
    parameters = [];
    overrides = [];
    warnings = [];
    namingStrategy;
    constructor(options) {
        this.databaseName = options.databaseName ?? 'postgres';
        this.schemaName = options.schemaName ?? 'public';
        this.overrideTableName = options.overrideTableName ?? 'migration_overrides';
        this.namingStrategy = asNamingStrategy(options.namingStrategy ?? 'hash');
    }
    getNameFor(item) {
        return this.namingStrategy.getName(item);
    }
    getTableByName(name) {
        return this.tables.find((table) => table.name === name);
    }
    warn(context, message) {
        this.warnings.push(`[${context}] ${message}`);
    }
    build() {
        const overrideMap = new Map();
        for (const override of this.overrides) {
            const { type, name } = override.value;
            overrideMap.set(asOverrideKey(type, name), override);
        }
        for (const func of this.functions) {
            func.override = overrideMap.get(asOverrideKey('function', func.name));
        }
        for (const { indexes, triggers } of this.tables) {
            for (const index of indexes) {
                index.override = overrideMap.get(asOverrideKey('index', index.name));
            }
            for (const trigger of triggers) {
                trigger.override = overrideMap.get(asOverrideKey('trigger', trigger.name));
            }
        }
        return {
            databaseName: this.databaseName,
            schemaName: this.schemaName,
            tables: this.tables,
            functions: this.functions,
            enums: this.enums,
            extensions: this.extensions,
            parameters: this.parameters,
            overrides: this.overrides,
            warnings: this.warnings,
        };
    }
}
exports.BaseContext = BaseContext;
//# sourceMappingURL=base-context.js.map