import { DateTime } from "luxon";
import { Maybe } from "./Maybe";
/**
 * Base type for year-only date
 */
export interface ExifDateYearOnly {
    readonly year: number;
}
/**
 * Type for year-month date (extends year-only)
 */
export interface ExifDateYearMonth extends ExifDateYearOnly {
    readonly month: number;
}
/**
 * Type for full date (extends year-month)
 */
export interface ExifDateFull extends ExifDateYearMonth {
    readonly day: number;
}
/**
 * Type for partial ExifDate (year-only or year-month)
 */
export type ExifDatePartial = ExifDateYearOnly | ExifDateYearMonth;
/**
 * Encodes an ExifDate
 */
export declare class ExifDate {
    readonly year: number;
    readonly month?: number | undefined;
    readonly day?: number | undefined;
    readonly rawValue?: string | undefined;
    static from(exifOrIso: string | number): Maybe<ExifDate>;
    static fromISO(text: string): Maybe<ExifDate>;
    private static fromPatterns;
    static fromExifStrict(text: string): Maybe<ExifDate>;
    static fromYearMonth(text: string | number): Maybe<ExifDate>;
    static fromYear(yearValue: string | number): Maybe<ExifDate>;
    static fromExifLoose(text: string): Maybe<ExifDate>;
    static fromEXIF(text: string): Maybe<ExifDate>;
    static fromDateTime(dt: DateTime, rawValue?: string): Maybe<ExifDate>;
    constructor(year: number, // full year (probably 2019-ish, but maybe Japanese 30-ish). See https://ericasadun.com/2018/12/25/iso-8601-yyyy-yyyy-and-why-your-year-may-be-wrong/
    month?: number | undefined, // 1-12, (no crazy 0-11 nonsense from Date!)
    day?: number | undefined, // 1-31
    rawValue?: string | undefined);
    toDate(): Date;
    /**
     * @param deltaMs defaults to 12 hours, so toMillis() is in the middle of the day.
     *
     * @return the epoch milliseconds for this day in UTC, plus `deltaMs` milliseconds.
     */
    toMillis(deltaMs?: number): number;
    toISOString(): string;
    toExifString(): string;
    toString(sep?: string): string;
    toJSON(): {
        _ctor: string;
        year: number;
        month: number | undefined;
        day: number | undefined;
        rawValue: string | undefined;
    };
    static fromJSON(json: ReturnType<ExifDate["toJSON"]>): ExifDate;
    /**
     * @returns true if this is a partial date (year-only or year-month)
     */
    isPartial(): this is ExifDatePartial;
    /**
     * @returns true if this is a year-only date
     */
    isYearOnly(): this is ExifDateYearOnly;
    /**
     * @returns true if this is a year-month date (no day)
     */
    isYearMonth(): this is ExifDateYearMonth;
    /**
     * @returns true if this is a full date (year, month, and day)
     */
    isFullDate(): this is ExifDateFull;
}
