"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const utils_1 = require("@typescript-eslint/utils");
const util_1 = require("../util");
exports.default = (0, util_1.createRule)({
    name: 'no-non-null-assertion',
    meta: {
        type: 'problem',
        docs: {
            description: 'Disallow non-null assertions using the `!` postfix operator',
            recommended: 'strict',
        },
        hasSuggestions: true,
        messages: {
            noNonNull: 'Forbidden non-null assertion.',
            suggestOptionalChain: 'Consider using the optional chain operator `?.` instead. This operator includes runtime checks, so it is safer than the compile-only non-null assertion operator.',
        },
        schema: [],
    },
    defaultOptions: [],
    create(context) {
        const sourceCode = context.getSourceCode();
        return {
            TSNonNullExpression(node) {
                const suggest = [];
                function convertTokenToOptional(replacement) {
                    return (fixer) => {
                        const operator = sourceCode.getTokenAfter(node.expression, util_1.isNonNullAssertionPunctuator);
                        if (operator) {
                            return fixer.replaceText(operator, replacement);
                        }
                        return null;
                    };
                }
                function removeToken() {
                    return (fixer) => {
                        const operator = sourceCode.getTokenAfter(node.expression, util_1.isNonNullAssertionPunctuator);
                        if (operator) {
                            return fixer.remove(operator);
                        }
                        return null;
                    };
                }
                if (node.parent?.type === utils_1.AST_NODE_TYPES.MemberExpression &&
                    node.parent.object === node) {
                    if (!node.parent.optional) {
                        if (node.parent.computed) {
                            // it is x![y]?.z
                            suggest.push({
                                messageId: 'suggestOptionalChain',
                                fix: convertTokenToOptional('?.'),
                            });
                        }
                        else {
                            // it is x!.y?.z
                            suggest.push({
                                messageId: 'suggestOptionalChain',
                                fix: convertTokenToOptional('?'),
                            });
                        }
                    }
                    else {
                        if (node.parent.computed) {
                            // it is x!?.[y].z
                            suggest.push({
                                messageId: 'suggestOptionalChain',
                                fix: removeToken(),
                            });
                        }
                        else {
                            // it is x!?.y.z
                            suggest.push({
                                messageId: 'suggestOptionalChain',
                                fix: removeToken(),
                            });
                        }
                    }
                }
                else if (node.parent?.type === utils_1.AST_NODE_TYPES.CallExpression &&
                    node.parent.callee === node) {
                    if (!node.parent.optional) {
                        // it is x.y?.z!()
                        suggest.push({
                            messageId: 'suggestOptionalChain',
                            fix: convertTokenToOptional('?.'),
                        });
                    }
                    else {
                        // it is x.y.z!?.()
                        suggest.push({
                            messageId: 'suggestOptionalChain',
                            fix: removeToken(),
                        });
                    }
                }
                context.report({
                    node,
                    messageId: 'noNonNull',
                    suggest,
                });
            },
        };
    },
});
//# sourceMappingURL=no-non-null-assertion.js.map