export { AnthropicProviderOptions } from '@ai-sdk/anthropic';
import { z } from 'zod/v4';
import { anthropicTools } from '@ai-sdk/anthropic/internal';
import { ProviderV3, LanguageModelV3, EmbeddingModelV3, ImageModelV3, RerankingModelV3 } from '@ai-sdk/provider';
import { FetchFunction } from '@ai-sdk/provider-utils';

type BedrockChatModelId = 'amazon.titan-tg1-large' | 'amazon.titan-text-express-v1' | 'anthropic.claude-v2' | 'anthropic.claude-v2:1' | 'anthropic.claude-instant-v1' | 'anthropic.claude-haiku-4-5-20251001-v1:0' | 'anthropic.claude-sonnet-4-20250514-v1:0' | 'anthropic.claude-sonnet-4-5-20250929-v1:0' | 'anthropic.claude-opus-4-20250514-v1:0' | 'anthropic.claude-opus-4-1-20250805-v1:0' | 'anthropic.claude-3-7-sonnet-20250219-v1:0' | 'anthropic.claude-3-5-sonnet-20240620-v1:0' | 'anthropic.claude-3-5-sonnet-20241022-v2:0' | 'anthropic.claude-3-5-haiku-20241022-v1:0' | 'anthropic.claude-3-sonnet-20240229-v1:0' | 'anthropic.claude-3-haiku-20240307-v1:0' | 'anthropic.claude-3-opus-20240229-v1:0' | 'cohere.command-text-v14' | 'cohere.command-light-text-v14' | 'cohere.command-r-v1:0' | 'cohere.command-r-plus-v1:0' | 'meta.llama3-70b-instruct-v1:0' | 'meta.llama3-8b-instruct-v1:0' | 'meta.llama3-1-405b-instruct-v1:0' | 'meta.llama3-1-70b-instruct-v1:0' | 'meta.llama3-1-8b-instruct-v1:0' | 'meta.llama3-2-11b-instruct-v1:0' | 'meta.llama3-2-1b-instruct-v1:0' | 'meta.llama3-2-3b-instruct-v1:0' | 'meta.llama3-2-90b-instruct-v1:0' | 'mistral.mistral-7b-instruct-v0:2' | 'mistral.mixtral-8x7b-instruct-v0:1' | 'mistral.mistral-large-2402-v1:0' | 'mistral.mistral-small-2402-v1:0' | 'openai.gpt-oss-120b-1:0' | 'openai.gpt-oss-20b-1:0' | 'amazon.titan-text-express-v1' | 'amazon.titan-text-lite-v1' | 'us.amazon.nova-premier-v1:0' | 'us.amazon.nova-pro-v1:0' | 'us.amazon.nova-micro-v1:0' | 'us.amazon.nova-lite-v1:0' | 'us.anthropic.claude-3-sonnet-20240229-v1:0' | 'us.anthropic.claude-3-opus-20240229-v1:0' | 'us.anthropic.claude-3-haiku-20240307-v1:0' | 'us.anthropic.claude-3-5-sonnet-20240620-v1:0' | 'us.anthropic.claude-3-5-haiku-20241022-v1:0' | 'us.anthropic.claude-3-5-sonnet-20241022-v2:0' | 'us.anthropic.claude-3-7-sonnet-20250219-v1:0' | 'us.anthropic.claude-sonnet-4-20250514-v1:0' | 'us.anthropic.claude-sonnet-4-5-20250929-v1:0' | 'us.anthropic.claude-opus-4-20250514-v1:0' | 'us.anthropic.claude-opus-4-1-20250805-v1:0' | 'us.meta.llama3-2-11b-instruct-v1:0' | 'us.meta.llama3-2-3b-instruct-v1:0' | 'us.meta.llama3-2-90b-instruct-v1:0' | 'us.meta.llama3-2-1b-instruct-v1:0' | 'us.meta.llama3-1-8b-instruct-v1:0' | 'us.meta.llama3-1-70b-instruct-v1:0' | 'us.meta.llama3-3-70b-instruct-v1:0' | 'us.deepseek.r1-v1:0' | 'us.mistral.pixtral-large-2502-v1:0' | 'us.meta.llama4-scout-17b-instruct-v1:0' | 'us.meta.llama4-maverick-17b-instruct-v1:0' | (string & {});
declare const bedrockProviderOptions: z.ZodObject<{
    additionalModelRequestFields: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodAny>>;
    reasoningConfig: z.ZodOptional<z.ZodObject<{
        type: z.ZodOptional<z.ZodUnion<readonly [z.ZodLiteral<"enabled">, z.ZodLiteral<"disabled">]>>;
        budgetTokens: z.ZodOptional<z.ZodNumber>;
        maxReasoningEffort: z.ZodOptional<z.ZodEnum<{
            low: "low";
            medium: "medium";
            high: "high";
        }>>;
    }, z.core.$strip>>;
    anthropicBeta: z.ZodOptional<z.ZodArray<z.ZodString>>;
}, z.core.$strip>;
type BedrockProviderOptions = z.infer<typeof bedrockProviderOptions>;

type BedrockEmbeddingModelId = 'amazon.titan-embed-text-v1' | 'amazon.titan-embed-text-v2:0' | 'cohere.embed-english-v3' | 'cohere.embed-multilingual-v3' | (string & {});

type BedrockImageModelId = 'amazon.nova-canvas-v1:0' | (string & {});

interface BedrockCredentials {
    region: string;
    accessKeyId: string;
    secretAccessKey: string;
    sessionToken?: string;
}

type BedrockRerankingModelId = 'amazon.rerank-v1:0' | 'cohere.rerank-v3-5:0' | (string & {});
type BedrockRerankingOptions = {
    /**
     * If the total number of results was greater than could fit in a response, a token is returned in the nextToken field. You can enter that token in this field to return the next batch of results.
     */
    nextToken?: string;
    /**
     * Additional model request fields to pass to the model.
     */
    additionalModelRequestFields?: Record<string, unknown>;
};

interface AmazonBedrockProviderSettings {
    /**
  The AWS region to use for the Bedrock provider. Defaults to the value of the
  `AWS_REGION` environment variable.
     */
    region?: string;
    /**
  API key for authenticating requests using Bearer token authentication.
  When provided, this will be used instead of AWS SigV4 authentication.
  Defaults to the value of the `AWS_BEARER_TOKEN_BEDROCK` environment variable.
  
  @example
  ```typescript
  // Using API key directly
  const bedrock = createAmazonBedrock({
    apiKey: 'your-api-key-here',
    region: 'us-east-1'
  });
  
  // Using environment variable AWS_BEARER_TOKEN_BEDROCK
  const bedrock = createAmazonBedrock({
    region: 'us-east-1'
  });
  ```
  
  Note: When `apiKey` is provided, it takes precedence over AWS SigV4 authentication.
  If neither `apiKey` nor `AWS_BEARER_TOKEN_BEDROCK` environment variable is set,
  the provider will fall back to AWS SigV4 authentication using AWS credentials.
     */
    apiKey?: string;
    /**
  The AWS access key ID to use for the Bedrock provider. Defaults to the value of the
  `AWS_ACCESS_KEY_ID` environment variable.
     */
    accessKeyId?: string;
    /**
  The AWS secret access key to use for the Bedrock provider. Defaults to the value of the
  `AWS_SECRET_ACCESS_KEY` environment variable.
     */
    secretAccessKey?: string;
    /**
  The AWS session token to use for the Bedrock provider. Defaults to the value of the
  `AWS_SESSION_TOKEN` environment variable.
     */
    sessionToken?: string;
    /**
  Base URL for the Bedrock API calls.
     */
    baseURL?: string;
    /**
  Custom headers to include in the requests.
     */
    headers?: Record<string, string>;
    /**
  Custom fetch implementation. You can use it as a middleware to intercept requests,
  or to provide a custom fetch implementation for e.g. testing.
  */
    fetch?: FetchFunction;
    /**
  The AWS credential provider to use for the Bedrock provider to get dynamic
  credentials similar to the AWS SDK. Setting a provider here will cause its
  credential values to be used instead of the `accessKeyId`, `secretAccessKey`,
  and `sessionToken` settings.
     */
    credentialProvider?: () => PromiseLike<Omit<BedrockCredentials, 'region'>>;
    generateId?: () => string;
}
interface AmazonBedrockProvider extends ProviderV3 {
    (modelId: BedrockChatModelId): LanguageModelV3;
    languageModel(modelId: BedrockChatModelId): LanguageModelV3;
    /**
     * Creates a model for text embeddings.
     */
    embedding(modelId: BedrockEmbeddingModelId): EmbeddingModelV3;
    /**
     * Creates a model for text embeddings.
     */
    embeddingModel(modelId: BedrockEmbeddingModelId): EmbeddingModelV3;
    /**
     * @deprecated Use `embedding` instead.
     */
    textEmbedding(modelId: BedrockEmbeddingModelId): EmbeddingModelV3;
    /**
     * @deprecated Use `embeddingModel` instead.
     */
    textEmbeddingModel(modelId: BedrockEmbeddingModelId): EmbeddingModelV3;
    /**
  Creates a model for image generation.
     */
    image(modelId: BedrockImageModelId): ImageModelV3;
    /**
  Creates a model for image generation.
     */
    imageModel(modelId: BedrockImageModelId): ImageModelV3;
    /**
     * Creates a model for reranking documents.
     */
    reranking(modelId: BedrockRerankingModelId): RerankingModelV3;
    /**
     * Creates a model for reranking documents.
     */
    rerankingModel(modelId: BedrockRerankingModelId): RerankingModelV3;
    /**
  Anthropic-specific tools that can be used with Anthropic models on Bedrock.
     */
    tools: typeof anthropicTools;
}
/**
Create an Amazon Bedrock provider instance.
 */
declare function createAmazonBedrock(options?: AmazonBedrockProviderSettings): AmazonBedrockProvider;
/**
Default Bedrock provider instance.
 */
declare const bedrock: AmazonBedrockProvider;

declare const VERSION: string;

export { type AmazonBedrockProvider, type AmazonBedrockProviderSettings, type BedrockProviderOptions, type BedrockRerankingOptions, VERSION, bedrock, createAmazonBedrock };
