// src/anthropic-provider.ts
import {
  NoSuchModelError
} from "@ai-sdk/provider";
import {
  generateId as generateId2,
  loadApiKey,
  loadOptionalSetting,
  withoutTrailingSlash,
  withUserAgentSuffix
} from "@ai-sdk/provider-utils";

// src/version.ts
var VERSION = true ? "3.0.9" : "0.0.0-test";

// src/anthropic-messages-language-model.ts
import {
  APICallError
} from "@ai-sdk/provider";
import {
  combineHeaders,
  createEventSourceResponseHandler,
  createJsonResponseHandler,
  createToolNameMapping,
  generateId,
  parseProviderOptions as parseProviderOptions2,
  postJsonToApi,
  resolve
} from "@ai-sdk/provider-utils";

// src/anthropic-error.ts
import {
  createJsonErrorResponseHandler,
  lazySchema,
  zodSchema
} from "@ai-sdk/provider-utils";
import { z } from "zod/v4";
var anthropicErrorDataSchema = lazySchema(
  () => zodSchema(
    z.object({
      type: z.literal("error"),
      error: z.object({
        type: z.string(),
        message: z.string()
      })
    })
  )
);
var anthropicFailedResponseHandler = createJsonErrorResponseHandler({
  errorSchema: anthropicErrorDataSchema,
  errorToMessage: (data) => data.error.message
});

// src/anthropic-messages-api.ts
import { lazySchema as lazySchema2, zodSchema as zodSchema2 } from "@ai-sdk/provider-utils";
import { z as z2 } from "zod/v4";
var anthropicMessagesResponseSchema = lazySchema2(
  () => zodSchema2(
    z2.object({
      type: z2.literal("message"),
      id: z2.string().nullish(),
      model: z2.string().nullish(),
      content: z2.array(
        z2.discriminatedUnion("type", [
          z2.object({
            type: z2.literal("text"),
            text: z2.string(),
            citations: z2.array(
              z2.discriminatedUnion("type", [
                z2.object({
                  type: z2.literal("web_search_result_location"),
                  cited_text: z2.string(),
                  url: z2.string(),
                  title: z2.string(),
                  encrypted_index: z2.string()
                }),
                z2.object({
                  type: z2.literal("page_location"),
                  cited_text: z2.string(),
                  document_index: z2.number(),
                  document_title: z2.string().nullable(),
                  start_page_number: z2.number(),
                  end_page_number: z2.number()
                }),
                z2.object({
                  type: z2.literal("char_location"),
                  cited_text: z2.string(),
                  document_index: z2.number(),
                  document_title: z2.string().nullable(),
                  start_char_index: z2.number(),
                  end_char_index: z2.number()
                })
              ])
            ).optional()
          }),
          z2.object({
            type: z2.literal("thinking"),
            thinking: z2.string(),
            signature: z2.string()
          }),
          z2.object({
            type: z2.literal("redacted_thinking"),
            data: z2.string()
          }),
          z2.object({
            type: z2.literal("tool_use"),
            id: z2.string(),
            name: z2.string(),
            input: z2.unknown(),
            // Programmatic tool calling: caller info when triggered from code execution
            caller: z2.union([
              z2.object({
                type: z2.literal("code_execution_20250825"),
                tool_id: z2.string()
              }),
              z2.object({
                type: z2.literal("direct")
              })
            ]).optional()
          }),
          z2.object({
            type: z2.literal("server_tool_use"),
            id: z2.string(),
            name: z2.string(),
            input: z2.record(z2.string(), z2.unknown()).nullish()
          }),
          z2.object({
            type: z2.literal("mcp_tool_use"),
            id: z2.string(),
            name: z2.string(),
            input: z2.unknown(),
            server_name: z2.string()
          }),
          z2.object({
            type: z2.literal("mcp_tool_result"),
            tool_use_id: z2.string(),
            is_error: z2.boolean(),
            content: z2.array(
              z2.union([
                z2.string(),
                z2.object({ type: z2.literal("text"), text: z2.string() })
              ])
            )
          }),
          z2.object({
            type: z2.literal("web_fetch_tool_result"),
            tool_use_id: z2.string(),
            content: z2.union([
              z2.object({
                type: z2.literal("web_fetch_result"),
                url: z2.string(),
                retrieved_at: z2.string(),
                content: z2.object({
                  type: z2.literal("document"),
                  title: z2.string().nullable(),
                  citations: z2.object({ enabled: z2.boolean() }).optional(),
                  source: z2.union([
                    z2.object({
                      type: z2.literal("base64"),
                      media_type: z2.literal("application/pdf"),
                      data: z2.string()
                    }),
                    z2.object({
                      type: z2.literal("text"),
                      media_type: z2.literal("text/plain"),
                      data: z2.string()
                    })
                  ])
                })
              }),
              z2.object({
                type: z2.literal("web_fetch_tool_result_error"),
                error_code: z2.string()
              })
            ])
          }),
          z2.object({
            type: z2.literal("web_search_tool_result"),
            tool_use_id: z2.string(),
            content: z2.union([
              z2.array(
                z2.object({
                  type: z2.literal("web_search_result"),
                  url: z2.string(),
                  title: z2.string(),
                  encrypted_content: z2.string(),
                  page_age: z2.string().nullish()
                })
              ),
              z2.object({
                type: z2.literal("web_search_tool_result_error"),
                error_code: z2.string()
              })
            ])
          }),
          // code execution results for code_execution_20250522 tool:
          z2.object({
            type: z2.literal("code_execution_tool_result"),
            tool_use_id: z2.string(),
            content: z2.union([
              z2.object({
                type: z2.literal("code_execution_result"),
                stdout: z2.string(),
                stderr: z2.string(),
                return_code: z2.number(),
                content: z2.array(
                  z2.object({
                    type: z2.literal("code_execution_output"),
                    file_id: z2.string()
                  })
                ).optional().default([])
              }),
              z2.object({
                type: z2.literal("code_execution_tool_result_error"),
                error_code: z2.string()
              })
            ])
          }),
          // bash code execution results for code_execution_20250825 tool:
          z2.object({
            type: z2.literal("bash_code_execution_tool_result"),
            tool_use_id: z2.string(),
            content: z2.discriminatedUnion("type", [
              z2.object({
                type: z2.literal("bash_code_execution_result"),
                content: z2.array(
                  z2.object({
                    type: z2.literal("bash_code_execution_output"),
                    file_id: z2.string()
                  })
                ),
                stdout: z2.string(),
                stderr: z2.string(),
                return_code: z2.number()
              }),
              z2.object({
                type: z2.literal("bash_code_execution_tool_result_error"),
                error_code: z2.string()
              })
            ])
          }),
          // text editor code execution results for code_execution_20250825 tool:
          z2.object({
            type: z2.literal("text_editor_code_execution_tool_result"),
            tool_use_id: z2.string(),
            content: z2.discriminatedUnion("type", [
              z2.object({
                type: z2.literal("text_editor_code_execution_tool_result_error"),
                error_code: z2.string()
              }),
              z2.object({
                type: z2.literal("text_editor_code_execution_view_result"),
                content: z2.string(),
                file_type: z2.string(),
                num_lines: z2.number().nullable(),
                start_line: z2.number().nullable(),
                total_lines: z2.number().nullable()
              }),
              z2.object({
                type: z2.literal("text_editor_code_execution_create_result"),
                is_file_update: z2.boolean()
              }),
              z2.object({
                type: z2.literal(
                  "text_editor_code_execution_str_replace_result"
                ),
                lines: z2.array(z2.string()).nullable(),
                new_lines: z2.number().nullable(),
                new_start: z2.number().nullable(),
                old_lines: z2.number().nullable(),
                old_start: z2.number().nullable()
              })
            ])
          }),
          // tool search tool results for tool_search_tool_regex_20251119 and tool_search_tool_bm25_20251119:
          z2.object({
            type: z2.literal("tool_search_tool_result"),
            tool_use_id: z2.string(),
            content: z2.union([
              z2.object({
                type: z2.literal("tool_search_tool_search_result"),
                tool_references: z2.array(
                  z2.object({
                    type: z2.literal("tool_reference"),
                    tool_name: z2.string()
                  })
                )
              }),
              z2.object({
                type: z2.literal("tool_search_tool_result_error"),
                error_code: z2.string()
              })
            ])
          })
        ])
      ),
      stop_reason: z2.string().nullish(),
      stop_sequence: z2.string().nullish(),
      usage: z2.looseObject({
        input_tokens: z2.number(),
        output_tokens: z2.number(),
        cache_creation_input_tokens: z2.number().nullish(),
        cache_read_input_tokens: z2.number().nullish()
      }),
      container: z2.object({
        expires_at: z2.string(),
        id: z2.string(),
        skills: z2.array(
          z2.object({
            type: z2.union([z2.literal("anthropic"), z2.literal("custom")]),
            skill_id: z2.string(),
            version: z2.string()
          })
        ).nullish()
      }).nullish(),
      context_management: z2.object({
        applied_edits: z2.array(
          z2.union([
            z2.object({
              type: z2.literal("clear_tool_uses_20250919"),
              cleared_tool_uses: z2.number(),
              cleared_input_tokens: z2.number()
            }),
            z2.object({
              type: z2.literal("clear_thinking_20251015"),
              cleared_thinking_turns: z2.number(),
              cleared_input_tokens: z2.number()
            })
          ])
        )
      }).nullish()
    })
  )
);
var anthropicMessagesChunkSchema = lazySchema2(
  () => zodSchema2(
    z2.discriminatedUnion("type", [
      z2.object({
        type: z2.literal("message_start"),
        message: z2.object({
          id: z2.string().nullish(),
          model: z2.string().nullish(),
          role: z2.string().nullish(),
          usage: z2.looseObject({
            input_tokens: z2.number(),
            cache_creation_input_tokens: z2.number().nullish(),
            cache_read_input_tokens: z2.number().nullish()
          }),
          // Programmatic tool calling: content may be pre-populated for deferred tool calls
          content: z2.array(
            z2.discriminatedUnion("type", [
              z2.object({
                type: z2.literal("tool_use"),
                id: z2.string(),
                name: z2.string(),
                input: z2.unknown(),
                caller: z2.union([
                  z2.object({
                    type: z2.literal("code_execution_20250825"),
                    tool_id: z2.string()
                  }),
                  z2.object({
                    type: z2.literal("direct")
                  })
                ]).optional()
              })
            ])
          ).nullish(),
          stop_reason: z2.string().nullish(),
          container: z2.object({
            expires_at: z2.string(),
            id: z2.string()
          }).nullish()
        })
      }),
      z2.object({
        type: z2.literal("content_block_start"),
        index: z2.number(),
        content_block: z2.discriminatedUnion("type", [
          z2.object({
            type: z2.literal("text"),
            text: z2.string()
          }),
          z2.object({
            type: z2.literal("thinking"),
            thinking: z2.string()
          }),
          z2.object({
            type: z2.literal("tool_use"),
            id: z2.string(),
            name: z2.string(),
            // Programmatic tool calling: input may be present directly for deferred tool calls
            input: z2.record(z2.string(), z2.unknown()).optional(),
            // Programmatic tool calling: caller info when triggered from code execution
            caller: z2.union([
              z2.object({
                type: z2.literal("code_execution_20250825"),
                tool_id: z2.string()
              }),
              z2.object({
                type: z2.literal("direct")
              })
            ]).optional()
          }),
          z2.object({
            type: z2.literal("redacted_thinking"),
            data: z2.string()
          }),
          z2.object({
            type: z2.literal("server_tool_use"),
            id: z2.string(),
            name: z2.string(),
            input: z2.record(z2.string(), z2.unknown()).nullish()
          }),
          z2.object({
            type: z2.literal("mcp_tool_use"),
            id: z2.string(),
            name: z2.string(),
            input: z2.unknown(),
            server_name: z2.string()
          }),
          z2.object({
            type: z2.literal("mcp_tool_result"),
            tool_use_id: z2.string(),
            is_error: z2.boolean(),
            content: z2.array(
              z2.union([
                z2.string(),
                z2.object({ type: z2.literal("text"), text: z2.string() })
              ])
            )
          }),
          z2.object({
            type: z2.literal("web_fetch_tool_result"),
            tool_use_id: z2.string(),
            content: z2.union([
              z2.object({
                type: z2.literal("web_fetch_result"),
                url: z2.string(),
                retrieved_at: z2.string(),
                content: z2.object({
                  type: z2.literal("document"),
                  title: z2.string().nullable(),
                  citations: z2.object({ enabled: z2.boolean() }).optional(),
                  source: z2.union([
                    z2.object({
                      type: z2.literal("base64"),
                      media_type: z2.literal("application/pdf"),
                      data: z2.string()
                    }),
                    z2.object({
                      type: z2.literal("text"),
                      media_type: z2.literal("text/plain"),
                      data: z2.string()
                    })
                  ])
                })
              }),
              z2.object({
                type: z2.literal("web_fetch_tool_result_error"),
                error_code: z2.string()
              })
            ])
          }),
          z2.object({
            type: z2.literal("web_search_tool_result"),
            tool_use_id: z2.string(),
            content: z2.union([
              z2.array(
                z2.object({
                  type: z2.literal("web_search_result"),
                  url: z2.string(),
                  title: z2.string(),
                  encrypted_content: z2.string(),
                  page_age: z2.string().nullish()
                })
              ),
              z2.object({
                type: z2.literal("web_search_tool_result_error"),
                error_code: z2.string()
              })
            ])
          }),
          // code execution results for code_execution_20250522 tool:
          z2.object({
            type: z2.literal("code_execution_tool_result"),
            tool_use_id: z2.string(),
            content: z2.union([
              z2.object({
                type: z2.literal("code_execution_result"),
                stdout: z2.string(),
                stderr: z2.string(),
                return_code: z2.number(),
                content: z2.array(
                  z2.object({
                    type: z2.literal("code_execution_output"),
                    file_id: z2.string()
                  })
                ).optional().default([])
              }),
              z2.object({
                type: z2.literal("code_execution_tool_result_error"),
                error_code: z2.string()
              })
            ])
          }),
          // bash code execution results for code_execution_20250825 tool:
          z2.object({
            type: z2.literal("bash_code_execution_tool_result"),
            tool_use_id: z2.string(),
            content: z2.discriminatedUnion("type", [
              z2.object({
                type: z2.literal("bash_code_execution_result"),
                content: z2.array(
                  z2.object({
                    type: z2.literal("bash_code_execution_output"),
                    file_id: z2.string()
                  })
                ),
                stdout: z2.string(),
                stderr: z2.string(),
                return_code: z2.number()
              }),
              z2.object({
                type: z2.literal("bash_code_execution_tool_result_error"),
                error_code: z2.string()
              })
            ])
          }),
          // text editor code execution results for code_execution_20250825 tool:
          z2.object({
            type: z2.literal("text_editor_code_execution_tool_result"),
            tool_use_id: z2.string(),
            content: z2.discriminatedUnion("type", [
              z2.object({
                type: z2.literal("text_editor_code_execution_tool_result_error"),
                error_code: z2.string()
              }),
              z2.object({
                type: z2.literal("text_editor_code_execution_view_result"),
                content: z2.string(),
                file_type: z2.string(),
                num_lines: z2.number().nullable(),
                start_line: z2.number().nullable(),
                total_lines: z2.number().nullable()
              }),
              z2.object({
                type: z2.literal("text_editor_code_execution_create_result"),
                is_file_update: z2.boolean()
              }),
              z2.object({
                type: z2.literal(
                  "text_editor_code_execution_str_replace_result"
                ),
                lines: z2.array(z2.string()).nullable(),
                new_lines: z2.number().nullable(),
                new_start: z2.number().nullable(),
                old_lines: z2.number().nullable(),
                old_start: z2.number().nullable()
              })
            ])
          }),
          // tool search tool results for tool_search_tool_regex_20251119 and tool_search_tool_bm25_20251119:
          z2.object({
            type: z2.literal("tool_search_tool_result"),
            tool_use_id: z2.string(),
            content: z2.union([
              z2.object({
                type: z2.literal("tool_search_tool_search_result"),
                tool_references: z2.array(
                  z2.object({
                    type: z2.literal("tool_reference"),
                    tool_name: z2.string()
                  })
                )
              }),
              z2.object({
                type: z2.literal("tool_search_tool_result_error"),
                error_code: z2.string()
              })
            ])
          })
        ])
      }),
      z2.object({
        type: z2.literal("content_block_delta"),
        index: z2.number(),
        delta: z2.discriminatedUnion("type", [
          z2.object({
            type: z2.literal("input_json_delta"),
            partial_json: z2.string()
          }),
          z2.object({
            type: z2.literal("text_delta"),
            text: z2.string()
          }),
          z2.object({
            type: z2.literal("thinking_delta"),
            thinking: z2.string()
          }),
          z2.object({
            type: z2.literal("signature_delta"),
            signature: z2.string()
          }),
          z2.object({
            type: z2.literal("citations_delta"),
            citation: z2.discriminatedUnion("type", [
              z2.object({
                type: z2.literal("web_search_result_location"),
                cited_text: z2.string(),
                url: z2.string(),
                title: z2.string(),
                encrypted_index: z2.string()
              }),
              z2.object({
                type: z2.literal("page_location"),
                cited_text: z2.string(),
                document_index: z2.number(),
                document_title: z2.string().nullable(),
                start_page_number: z2.number(),
                end_page_number: z2.number()
              }),
              z2.object({
                type: z2.literal("char_location"),
                cited_text: z2.string(),
                document_index: z2.number(),
                document_title: z2.string().nullable(),
                start_char_index: z2.number(),
                end_char_index: z2.number()
              })
            ])
          })
        ])
      }),
      z2.object({
        type: z2.literal("content_block_stop"),
        index: z2.number()
      }),
      z2.object({
        type: z2.literal("error"),
        error: z2.object({
          type: z2.string(),
          message: z2.string()
        })
      }),
      z2.object({
        type: z2.literal("message_delta"),
        delta: z2.object({
          stop_reason: z2.string().nullish(),
          stop_sequence: z2.string().nullish(),
          container: z2.object({
            expires_at: z2.string(),
            id: z2.string(),
            skills: z2.array(
              z2.object({
                type: z2.union([
                  z2.literal("anthropic"),
                  z2.literal("custom")
                ]),
                skill_id: z2.string(),
                version: z2.string()
              })
            ).nullish()
          }).nullish(),
          context_management: z2.object({
            applied_edits: z2.array(
              z2.union([
                z2.object({
                  type: z2.literal("clear_tool_uses_20250919"),
                  cleared_tool_uses: z2.number(),
                  cleared_input_tokens: z2.number()
                }),
                z2.object({
                  type: z2.literal("clear_thinking_20251015"),
                  cleared_thinking_turns: z2.number(),
                  cleared_input_tokens: z2.number()
                })
              ])
            )
          }).nullish()
        }),
        usage: z2.looseObject({
          output_tokens: z2.number(),
          cache_creation_input_tokens: z2.number().nullish()
        })
      }),
      z2.object({
        type: z2.literal("message_stop")
      }),
      z2.object({
        type: z2.literal("ping")
      })
    ])
  )
);
var anthropicReasoningMetadataSchema = lazySchema2(
  () => zodSchema2(
    z2.object({
      signature: z2.string().optional(),
      redactedData: z2.string().optional()
    })
  )
);

// src/anthropic-messages-options.ts
import { z as z3 } from "zod/v4";
var anthropicFilePartProviderOptions = z3.object({
  /**
   * Citation configuration for this document.
   * When enabled, this document will generate citations in the response.
   */
  citations: z3.object({
    /**
     * Enable citations for this document
     */
    enabled: z3.boolean()
  }).optional(),
  /**
   * Custom title for the document.
   * If not provided, the filename will be used.
   */
  title: z3.string().optional(),
  /**
   * Context about the document that will be passed to the model
   * but not used towards cited content.
   * Useful for storing document metadata as text or stringified JSON.
   */
  context: z3.string().optional()
});
var anthropicProviderOptions = z3.object({
  /**
   * Whether to send reasoning to the model.
   *
   * This allows you to deactivate reasoning inputs for models that do not support them.
   */
  sendReasoning: z3.boolean().optional(),
  /**
   * Determines how structured outputs are generated.
   *
   * - `outputFormat`: Use the `output_format` parameter to specify the structured output format.
   * - `jsonTool`: Use a special 'json' tool to specify the structured output format.
   * - `auto`: Use 'outputFormat' when supported, otherwise use 'jsonTool' (default).
   */
  structuredOutputMode: z3.enum(["outputFormat", "jsonTool", "auto"]).optional(),
  /**
   * Configuration for enabling Claude's extended thinking.
   *
   * When enabled, responses include thinking content blocks showing Claude's thinking process before the final answer.
   * Requires a minimum budget of 1,024 tokens and counts towards the `max_tokens` limit.
   */
  thinking: z3.object({
    type: z3.union([z3.literal("enabled"), z3.literal("disabled")]),
    budgetTokens: z3.number().optional()
  }).optional(),
  /**
   * Whether to disable parallel function calling during tool use. Default is false.
   * When set to true, Claude will use at most one tool per response.
   */
  disableParallelToolUse: z3.boolean().optional(),
  /**
   * Cache control settings for this message.
   * See https://docs.anthropic.com/en/docs/build-with-claude/prompt-caching
   */
  cacheControl: z3.object({
    type: z3.literal("ephemeral"),
    ttl: z3.union([z3.literal("5m"), z3.literal("1h")]).optional()
  }).optional(),
  /**
   * MCP servers to be utilized in this request.
   */
  mcpServers: z3.array(
    z3.object({
      type: z3.literal("url"),
      name: z3.string(),
      url: z3.string(),
      authorizationToken: z3.string().nullish(),
      toolConfiguration: z3.object({
        enabled: z3.boolean().nullish(),
        allowedTools: z3.array(z3.string()).nullish()
      }).nullish()
    })
  ).optional(),
  /**
   * Agent Skills configuration. Skills enable Claude to perform specialized tasks
   * like document processing (PPTX, DOCX, PDF, XLSX) and data analysis.
   * Requires code execution tool to be enabled.
   */
  container: z3.object({
    id: z3.string().optional(),
    skills: z3.array(
      z3.object({
        type: z3.union([z3.literal("anthropic"), z3.literal("custom")]),
        skillId: z3.string(),
        version: z3.string().optional()
      })
    ).optional()
  }).optional(),
  /**
   * Whether to enable tool streaming (and structured output streaming).
   *
   * When set to false, the model will return all tool calls and results
   * at once after a delay.
   *
   * @default true
   */
  toolStreaming: z3.boolean().optional(),
  /**
   * @default 'high'
   */
  effort: z3.enum(["low", "medium", "high"]).optional(),
  contextManagement: z3.object({
    edits: z3.array(
      z3.discriminatedUnion("type", [
        z3.object({
          type: z3.literal("clear_tool_uses_20250919"),
          trigger: z3.discriminatedUnion("type", [
            z3.object({
              type: z3.literal("input_tokens"),
              value: z3.number()
            }),
            z3.object({
              type: z3.literal("tool_uses"),
              value: z3.number()
            })
          ]).optional(),
          keep: z3.object({
            type: z3.literal("tool_uses"),
            value: z3.number()
          }).optional(),
          clearAtLeast: z3.object({
            type: z3.literal("input_tokens"),
            value: z3.number()
          }).optional(),
          clearToolInputs: z3.boolean().optional(),
          excludeTools: z3.array(z3.string()).optional()
        }),
        z3.object({
          type: z3.literal("clear_thinking_20251015"),
          keep: z3.union([
            z3.literal("all"),
            z3.object({
              type: z3.literal("thinking_turns"),
              value: z3.number()
            })
          ]).optional()
        })
      ])
    )
  }).optional()
});

// src/anthropic-prepare-tools.ts
import {
  UnsupportedFunctionalityError
} from "@ai-sdk/provider";

// src/get-cache-control.ts
var MAX_CACHE_BREAKPOINTS = 4;
function getCacheControl(providerMetadata) {
  var _a;
  const anthropic2 = providerMetadata == null ? void 0 : providerMetadata.anthropic;
  const cacheControlValue = (_a = anthropic2 == null ? void 0 : anthropic2.cacheControl) != null ? _a : anthropic2 == null ? void 0 : anthropic2.cache_control;
  return cacheControlValue;
}
var CacheControlValidator = class {
  constructor() {
    this.breakpointCount = 0;
    this.warnings = [];
  }
  getCacheControl(providerMetadata, context) {
    const cacheControlValue = getCacheControl(providerMetadata);
    if (!cacheControlValue) {
      return void 0;
    }
    if (!context.canCache) {
      this.warnings.push({
        type: "unsupported",
        feature: "cache_control on non-cacheable context",
        details: `cache_control cannot be set on ${context.type}. It will be ignored.`
      });
      return void 0;
    }
    this.breakpointCount++;
    if (this.breakpointCount > MAX_CACHE_BREAKPOINTS) {
      this.warnings.push({
        type: "unsupported",
        feature: "cacheControl breakpoint limit",
        details: `Maximum ${MAX_CACHE_BREAKPOINTS} cache breakpoints exceeded (found ${this.breakpointCount}). This breakpoint will be ignored.`
      });
      return void 0;
    }
    return cacheControlValue;
  }
  getWarnings() {
    return this.warnings;
  }
};

// src/tool/text-editor_20250728.ts
import { createProviderToolFactory } from "@ai-sdk/provider-utils";
import { z as z4 } from "zod/v4";
import { lazySchema as lazySchema3, zodSchema as zodSchema3 } from "@ai-sdk/provider-utils";
var textEditor_20250728ArgsSchema = lazySchema3(
  () => zodSchema3(
    z4.object({
      maxCharacters: z4.number().optional()
    })
  )
);
var textEditor_20250728InputSchema = lazySchema3(
  () => zodSchema3(
    z4.object({
      command: z4.enum(["view", "create", "str_replace", "insert"]),
      path: z4.string(),
      file_text: z4.string().optional(),
      insert_line: z4.number().int().optional(),
      new_str: z4.string().optional(),
      old_str: z4.string().optional(),
      view_range: z4.array(z4.number().int()).optional()
    })
  )
);
var factory = createProviderToolFactory({
  id: "anthropic.text_editor_20250728",
  inputSchema: textEditor_20250728InputSchema
});
var textEditor_20250728 = (args = {}) => {
  return factory(args);
};

// src/tool/web-search_20250305.ts
import {
  createProviderToolFactoryWithOutputSchema,
  lazySchema as lazySchema4,
  zodSchema as zodSchema4
} from "@ai-sdk/provider-utils";
import { z as z5 } from "zod/v4";
var webSearch_20250305ArgsSchema = lazySchema4(
  () => zodSchema4(
    z5.object({
      maxUses: z5.number().optional(),
      allowedDomains: z5.array(z5.string()).optional(),
      blockedDomains: z5.array(z5.string()).optional(),
      userLocation: z5.object({
        type: z5.literal("approximate"),
        city: z5.string().optional(),
        region: z5.string().optional(),
        country: z5.string().optional(),
        timezone: z5.string().optional()
      }).optional()
    })
  )
);
var webSearch_20250305OutputSchema = lazySchema4(
  () => zodSchema4(
    z5.array(
      z5.object({
        url: z5.string(),
        title: z5.string().nullable(),
        pageAge: z5.string().nullable(),
        encryptedContent: z5.string(),
        type: z5.literal("web_search_result")
      })
    )
  )
);
var webSearch_20250305InputSchema = lazySchema4(
  () => zodSchema4(
    z5.object({
      query: z5.string()
    })
  )
);
var factory2 = createProviderToolFactoryWithOutputSchema({
  id: "anthropic.web_search_20250305",
  inputSchema: webSearch_20250305InputSchema,
  outputSchema: webSearch_20250305OutputSchema,
  supportsDeferredResults: true
});
var webSearch_20250305 = (args = {}) => {
  return factory2(args);
};

// src/tool/web-fetch-20250910.ts
import {
  createProviderToolFactoryWithOutputSchema as createProviderToolFactoryWithOutputSchema2,
  lazySchema as lazySchema5,
  zodSchema as zodSchema5
} from "@ai-sdk/provider-utils";
import { z as z6 } from "zod/v4";
var webFetch_20250910ArgsSchema = lazySchema5(
  () => zodSchema5(
    z6.object({
      maxUses: z6.number().optional(),
      allowedDomains: z6.array(z6.string()).optional(),
      blockedDomains: z6.array(z6.string()).optional(),
      citations: z6.object({ enabled: z6.boolean() }).optional(),
      maxContentTokens: z6.number().optional()
    })
  )
);
var webFetch_20250910OutputSchema = lazySchema5(
  () => zodSchema5(
    z6.object({
      type: z6.literal("web_fetch_result"),
      url: z6.string(),
      content: z6.object({
        type: z6.literal("document"),
        title: z6.string().nullable(),
        citations: z6.object({ enabled: z6.boolean() }).optional(),
        source: z6.union([
          z6.object({
            type: z6.literal("base64"),
            mediaType: z6.literal("application/pdf"),
            data: z6.string()
          }),
          z6.object({
            type: z6.literal("text"),
            mediaType: z6.literal("text/plain"),
            data: z6.string()
          })
        ])
      }),
      retrievedAt: z6.string().nullable()
    })
  )
);
var webFetch_20250910InputSchema = lazySchema5(
  () => zodSchema5(
    z6.object({
      url: z6.string()
    })
  )
);
var factory3 = createProviderToolFactoryWithOutputSchema2({
  id: "anthropic.web_fetch_20250910",
  inputSchema: webFetch_20250910InputSchema,
  outputSchema: webFetch_20250910OutputSchema,
  supportsDeferredResults: true
});
var webFetch_20250910 = (args = {}) => {
  return factory3(args);
};

// src/anthropic-prepare-tools.ts
import { validateTypes } from "@ai-sdk/provider-utils";
async function prepareTools({
  tools,
  toolChoice,
  disableParallelToolUse,
  cacheControlValidator,
  supportsStructuredOutput
}) {
  var _a;
  tools = (tools == null ? void 0 : tools.length) ? tools : void 0;
  const toolWarnings = [];
  const betas = /* @__PURE__ */ new Set();
  const validator = cacheControlValidator || new CacheControlValidator();
  if (tools == null) {
    return { tools: void 0, toolChoice: void 0, toolWarnings, betas };
  }
  const anthropicTools2 = [];
  for (const tool of tools) {
    switch (tool.type) {
      case "function": {
        const cacheControl = validator.getCacheControl(tool.providerOptions, {
          type: "tool definition",
          canCache: true
        });
        const anthropicOptions = (_a = tool.providerOptions) == null ? void 0 : _a.anthropic;
        const deferLoading = anthropicOptions == null ? void 0 : anthropicOptions.deferLoading;
        const allowedCallers = anthropicOptions == null ? void 0 : anthropicOptions.allowedCallers;
        anthropicTools2.push({
          name: tool.name,
          description: tool.description,
          input_schema: tool.inputSchema,
          cache_control: cacheControl,
          ...supportsStructuredOutput === true && tool.strict != null ? { strict: tool.strict } : {},
          ...deferLoading != null ? { defer_loading: deferLoading } : {},
          ...allowedCallers != null ? { allowed_callers: allowedCallers } : {},
          ...tool.inputExamples != null ? {
            input_examples: tool.inputExamples.map(
              (example) => example.input
            )
          } : {}
        });
        if (supportsStructuredOutput === true) {
          betas.add("structured-outputs-2025-11-13");
        }
        if (tool.inputExamples != null || allowedCallers != null) {
          betas.add("advanced-tool-use-2025-11-20");
        }
        break;
      }
      case "provider": {
        switch (tool.id) {
          case "anthropic.code_execution_20250522": {
            betas.add("code-execution-2025-05-22");
            anthropicTools2.push({
              type: "code_execution_20250522",
              name: "code_execution",
              cache_control: void 0
            });
            break;
          }
          case "anthropic.code_execution_20250825": {
            betas.add("code-execution-2025-08-25");
            anthropicTools2.push({
              type: "code_execution_20250825",
              name: "code_execution"
            });
            break;
          }
          case "anthropic.computer_20250124": {
            betas.add("computer-use-2025-01-24");
            anthropicTools2.push({
              name: "computer",
              type: "computer_20250124",
              display_width_px: tool.args.displayWidthPx,
              display_height_px: tool.args.displayHeightPx,
              display_number: tool.args.displayNumber,
              cache_control: void 0
            });
            break;
          }
          case "anthropic.computer_20241022": {
            betas.add("computer-use-2024-10-22");
            anthropicTools2.push({
              name: "computer",
              type: "computer_20241022",
              display_width_px: tool.args.displayWidthPx,
              display_height_px: tool.args.displayHeightPx,
              display_number: tool.args.displayNumber,
              cache_control: void 0
            });
            break;
          }
          case "anthropic.text_editor_20250124": {
            betas.add("computer-use-2025-01-24");
            anthropicTools2.push({
              name: "str_replace_editor",
              type: "text_editor_20250124",
              cache_control: void 0
            });
            break;
          }
          case "anthropic.text_editor_20241022": {
            betas.add("computer-use-2024-10-22");
            anthropicTools2.push({
              name: "str_replace_editor",
              type: "text_editor_20241022",
              cache_control: void 0
            });
            break;
          }
          case "anthropic.text_editor_20250429": {
            betas.add("computer-use-2025-01-24");
            anthropicTools2.push({
              name: "str_replace_based_edit_tool",
              type: "text_editor_20250429",
              cache_control: void 0
            });
            break;
          }
          case "anthropic.text_editor_20250728": {
            const args = await validateTypes({
              value: tool.args,
              schema: textEditor_20250728ArgsSchema
            });
            anthropicTools2.push({
              name: "str_replace_based_edit_tool",
              type: "text_editor_20250728",
              max_characters: args.maxCharacters,
              cache_control: void 0
            });
            break;
          }
          case "anthropic.bash_20250124": {
            betas.add("computer-use-2025-01-24");
            anthropicTools2.push({
              name: "bash",
              type: "bash_20250124",
              cache_control: void 0
            });
            break;
          }
          case "anthropic.bash_20241022": {
            betas.add("computer-use-2024-10-22");
            anthropicTools2.push({
              name: "bash",
              type: "bash_20241022",
              cache_control: void 0
            });
            break;
          }
          case "anthropic.memory_20250818": {
            betas.add("context-management-2025-06-27");
            anthropicTools2.push({
              name: "memory",
              type: "memory_20250818"
            });
            break;
          }
          case "anthropic.web_fetch_20250910": {
            betas.add("web-fetch-2025-09-10");
            const args = await validateTypes({
              value: tool.args,
              schema: webFetch_20250910ArgsSchema
            });
            anthropicTools2.push({
              type: "web_fetch_20250910",
              name: "web_fetch",
              max_uses: args.maxUses,
              allowed_domains: args.allowedDomains,
              blocked_domains: args.blockedDomains,
              citations: args.citations,
              max_content_tokens: args.maxContentTokens,
              cache_control: void 0
            });
            break;
          }
          case "anthropic.web_search_20250305": {
            const args = await validateTypes({
              value: tool.args,
              schema: webSearch_20250305ArgsSchema
            });
            anthropicTools2.push({
              type: "web_search_20250305",
              name: "web_search",
              max_uses: args.maxUses,
              allowed_domains: args.allowedDomains,
              blocked_domains: args.blockedDomains,
              user_location: args.userLocation,
              cache_control: void 0
            });
            break;
          }
          case "anthropic.tool_search_regex_20251119": {
            betas.add("advanced-tool-use-2025-11-20");
            anthropicTools2.push({
              type: "tool_search_tool_regex_20251119",
              name: "tool_search_tool_regex"
            });
            break;
          }
          case "anthropic.tool_search_bm25_20251119": {
            betas.add("advanced-tool-use-2025-11-20");
            anthropicTools2.push({
              type: "tool_search_tool_bm25_20251119",
              name: "tool_search_tool_bm25"
            });
            break;
          }
          default: {
            toolWarnings.push({
              type: "unsupported",
              feature: `provider-defined tool ${tool.id}`
            });
            break;
          }
        }
        break;
      }
      default: {
        toolWarnings.push({
          type: "unsupported",
          feature: `tool ${tool}`
        });
        break;
      }
    }
  }
  if (toolChoice == null) {
    return {
      tools: anthropicTools2,
      toolChoice: disableParallelToolUse ? { type: "auto", disable_parallel_tool_use: disableParallelToolUse } : void 0,
      toolWarnings,
      betas
    };
  }
  const type = toolChoice.type;
  switch (type) {
    case "auto":
      return {
        tools: anthropicTools2,
        toolChoice: {
          type: "auto",
          disable_parallel_tool_use: disableParallelToolUse
        },
        toolWarnings,
        betas
      };
    case "required":
      return {
        tools: anthropicTools2,
        toolChoice: {
          type: "any",
          disable_parallel_tool_use: disableParallelToolUse
        },
        toolWarnings,
        betas
      };
    case "none":
      return { tools: void 0, toolChoice: void 0, toolWarnings, betas };
    case "tool":
      return {
        tools: anthropicTools2,
        toolChoice: {
          type: "tool",
          name: toolChoice.toolName,
          disable_parallel_tool_use: disableParallelToolUse
        },
        toolWarnings,
        betas
      };
    default: {
      const _exhaustiveCheck = type;
      throw new UnsupportedFunctionalityError({
        functionality: `tool choice type: ${_exhaustiveCheck}`
      });
    }
  }
}

// src/convert-anthropic-messages-usage.ts
function convertAnthropicMessagesUsage(usage) {
  var _a, _b;
  const inputTokens = usage.input_tokens;
  const outputTokens = usage.output_tokens;
  const cacheCreationTokens = (_a = usage.cache_creation_input_tokens) != null ? _a : 0;
  const cacheReadTokens = (_b = usage.cache_read_input_tokens) != null ? _b : 0;
  return {
    inputTokens: {
      total: inputTokens + cacheCreationTokens + cacheReadTokens,
      noCache: inputTokens,
      cacheRead: cacheReadTokens,
      cacheWrite: cacheCreationTokens
    },
    outputTokens: {
      total: outputTokens,
      text: void 0,
      reasoning: void 0
    },
    raw: usage
  };
}

// src/convert-to-anthropic-messages-prompt.ts
import {
  UnsupportedFunctionalityError as UnsupportedFunctionalityError2
} from "@ai-sdk/provider";
import {
  convertToBase64,
  parseProviderOptions,
  validateTypes as validateTypes2,
  isNonNullable
} from "@ai-sdk/provider-utils";

// src/tool/code-execution_20250522.ts
import {
  createProviderToolFactoryWithOutputSchema as createProviderToolFactoryWithOutputSchema3,
  lazySchema as lazySchema6,
  zodSchema as zodSchema6
} from "@ai-sdk/provider-utils";
import { z as z7 } from "zod/v4";
var codeExecution_20250522OutputSchema = lazySchema6(
  () => zodSchema6(
    z7.object({
      type: z7.literal("code_execution_result"),
      stdout: z7.string(),
      stderr: z7.string(),
      return_code: z7.number(),
      content: z7.array(
        z7.object({
          type: z7.literal("code_execution_output"),
          file_id: z7.string()
        })
      ).optional().default([])
    })
  )
);
var codeExecution_20250522InputSchema = lazySchema6(
  () => zodSchema6(
    z7.object({
      code: z7.string()
    })
  )
);
var factory4 = createProviderToolFactoryWithOutputSchema3({
  id: "anthropic.code_execution_20250522",
  inputSchema: codeExecution_20250522InputSchema,
  outputSchema: codeExecution_20250522OutputSchema
});
var codeExecution_20250522 = (args = {}) => {
  return factory4(args);
};

// src/tool/code-execution_20250825.ts
import {
  createProviderToolFactoryWithOutputSchema as createProviderToolFactoryWithOutputSchema4,
  lazySchema as lazySchema7,
  zodSchema as zodSchema7
} from "@ai-sdk/provider-utils";
import { z as z8 } from "zod/v4";
var codeExecution_20250825OutputSchema = lazySchema7(
  () => zodSchema7(
    z8.discriminatedUnion("type", [
      z8.object({
        type: z8.literal("code_execution_result"),
        stdout: z8.string(),
        stderr: z8.string(),
        return_code: z8.number(),
        content: z8.array(
          z8.object({
            type: z8.literal("code_execution_output"),
            file_id: z8.string()
          })
        ).optional().default([])
      }),
      z8.object({
        type: z8.literal("bash_code_execution_result"),
        content: z8.array(
          z8.object({
            type: z8.literal("bash_code_execution_output"),
            file_id: z8.string()
          })
        ),
        stdout: z8.string(),
        stderr: z8.string(),
        return_code: z8.number()
      }),
      z8.object({
        type: z8.literal("bash_code_execution_tool_result_error"),
        error_code: z8.string()
      }),
      z8.object({
        type: z8.literal("text_editor_code_execution_tool_result_error"),
        error_code: z8.string()
      }),
      z8.object({
        type: z8.literal("text_editor_code_execution_view_result"),
        content: z8.string(),
        file_type: z8.string(),
        num_lines: z8.number().nullable(),
        start_line: z8.number().nullable(),
        total_lines: z8.number().nullable()
      }),
      z8.object({
        type: z8.literal("text_editor_code_execution_create_result"),
        is_file_update: z8.boolean()
      }),
      z8.object({
        type: z8.literal("text_editor_code_execution_str_replace_result"),
        lines: z8.array(z8.string()).nullable(),
        new_lines: z8.number().nullable(),
        new_start: z8.number().nullable(),
        old_lines: z8.number().nullable(),
        old_start: z8.number().nullable()
      })
    ])
  )
);
var codeExecution_20250825InputSchema = lazySchema7(
  () => zodSchema7(
    z8.discriminatedUnion("type", [
      // Programmatic tool calling format (mapped from { code } by AI SDK)
      z8.object({
        type: z8.literal("programmatic-tool-call"),
        code: z8.string()
      }),
      z8.object({
        type: z8.literal("bash_code_execution"),
        command: z8.string()
      }),
      z8.discriminatedUnion("command", [
        z8.object({
          type: z8.literal("text_editor_code_execution"),
          command: z8.literal("view"),
          path: z8.string()
        }),
        z8.object({
          type: z8.literal("text_editor_code_execution"),
          command: z8.literal("create"),
          path: z8.string(),
          file_text: z8.string().nullish()
        }),
        z8.object({
          type: z8.literal("text_editor_code_execution"),
          command: z8.literal("str_replace"),
          path: z8.string(),
          old_str: z8.string(),
          new_str: z8.string()
        })
      ])
    ])
  )
);
var factory5 = createProviderToolFactoryWithOutputSchema4({
  id: "anthropic.code_execution_20250825",
  inputSchema: codeExecution_20250825InputSchema,
  outputSchema: codeExecution_20250825OutputSchema,
  // Programmatic tool calling: tool results may be deferred to a later turn
  // when code execution triggers a client-executed tool that needs to be
  // resolved before the code execution result can be returned.
  supportsDeferredResults: true
});
var codeExecution_20250825 = (args = {}) => {
  return factory5(args);
};

// src/tool/tool-search-regex_20251119.ts
import {
  createProviderToolFactoryWithOutputSchema as createProviderToolFactoryWithOutputSchema5,
  lazySchema as lazySchema8,
  zodSchema as zodSchema8
} from "@ai-sdk/provider-utils";
import { z as z9 } from "zod/v4";
var toolSearchRegex_20251119OutputSchema = lazySchema8(
  () => zodSchema8(
    z9.array(
      z9.object({
        type: z9.literal("tool_reference"),
        toolName: z9.string()
      })
    )
  )
);
var toolSearchRegex_20251119InputSchema = lazySchema8(
  () => zodSchema8(
    z9.object({
      /**
       * A regex pattern to search for tools.
       * Uses Python re.search() syntax. Maximum 200 characters.
       *
       * Examples:
       * - "weather" - matches tool names/descriptions containing "weather"
       * - "get_.*_data" - matches tools like get_user_data, get_weather_data
       * - "database.*query|query.*database" - OR patterns for flexibility
       * - "(?i)slack" - case-insensitive search
       */
      pattern: z9.string(),
      /**
       * Maximum number of tools to return. Optional.
       */
      limit: z9.number().optional()
    })
  )
);
var factory6 = createProviderToolFactoryWithOutputSchema5({
  id: "anthropic.tool_search_regex_20251119",
  inputSchema: toolSearchRegex_20251119InputSchema,
  outputSchema: toolSearchRegex_20251119OutputSchema
});
var toolSearchRegex_20251119 = (args = {}) => {
  return factory6(args);
};

// src/convert-to-anthropic-messages-prompt.ts
function convertToString(data) {
  if (typeof data === "string") {
    return Buffer.from(data, "base64").toString("utf-8");
  }
  if (data instanceof Uint8Array) {
    return new TextDecoder().decode(data);
  }
  if (data instanceof URL) {
    throw new UnsupportedFunctionalityError2({
      functionality: "URL-based text documents are not supported for citations"
    });
  }
  throw new UnsupportedFunctionalityError2({
    functionality: `unsupported data type for text documents: ${typeof data}`
  });
}
function isUrlData(data) {
  return data instanceof URL || isUrlString(data);
}
function isUrlString(data) {
  return typeof data === "string" && /^https?:\/\//i.test(data);
}
function getUrlString(data) {
  return data instanceof URL ? data.toString() : data;
}
async function convertToAnthropicMessagesPrompt({
  prompt,
  sendReasoning,
  warnings,
  cacheControlValidator,
  toolNameMapping
}) {
  var _a, _b, _c, _d, _e, _f, _g, _h, _i, _j, _k, _l, _m, _n, _o;
  const betas = /* @__PURE__ */ new Set();
  const blocks = groupIntoBlocks(prompt);
  const validator = cacheControlValidator || new CacheControlValidator();
  let system = void 0;
  const messages = [];
  async function shouldEnableCitations(providerMetadata) {
    var _a2, _b2;
    const anthropicOptions = await parseProviderOptions({
      provider: "anthropic",
      providerOptions: providerMetadata,
      schema: anthropicFilePartProviderOptions
    });
    return (_b2 = (_a2 = anthropicOptions == null ? void 0 : anthropicOptions.citations) == null ? void 0 : _a2.enabled) != null ? _b2 : false;
  }
  async function getDocumentMetadata(providerMetadata) {
    const anthropicOptions = await parseProviderOptions({
      provider: "anthropic",
      providerOptions: providerMetadata,
      schema: anthropicFilePartProviderOptions
    });
    return {
      title: anthropicOptions == null ? void 0 : anthropicOptions.title,
      context: anthropicOptions == null ? void 0 : anthropicOptions.context
    };
  }
  for (let i = 0; i < blocks.length; i++) {
    const block = blocks[i];
    const isLastBlock = i === blocks.length - 1;
    const type = block.type;
    switch (type) {
      case "system": {
        if (system != null) {
          throw new UnsupportedFunctionalityError2({
            functionality: "Multiple system messages that are separated by user/assistant messages"
          });
        }
        system = block.messages.map(({ content, providerOptions }) => ({
          type: "text",
          text: content,
          cache_control: validator.getCacheControl(providerOptions, {
            type: "system message",
            canCache: true
          })
        }));
        break;
      }
      case "user": {
        const anthropicContent = [];
        for (const message of block.messages) {
          const { role, content } = message;
          switch (role) {
            case "user": {
              for (let j = 0; j < content.length; j++) {
                const part = content[j];
                const isLastPart = j === content.length - 1;
                const cacheControl = (_a = validator.getCacheControl(part.providerOptions, {
                  type: "user message part",
                  canCache: true
                })) != null ? _a : isLastPart ? validator.getCacheControl(message.providerOptions, {
                  type: "user message",
                  canCache: true
                }) : void 0;
                switch (part.type) {
                  case "text": {
                    anthropicContent.push({
                      type: "text",
                      text: part.text,
                      cache_control: cacheControl
                    });
                    break;
                  }
                  case "file": {
                    if (part.mediaType.startsWith("image/")) {
                      anthropicContent.push({
                        type: "image",
                        source: isUrlData(part.data) ? {
                          type: "url",
                          url: getUrlString(part.data)
                        } : {
                          type: "base64",
                          media_type: part.mediaType === "image/*" ? "image/jpeg" : part.mediaType,
                          data: convertToBase64(part.data)
                        },
                        cache_control: cacheControl
                      });
                    } else if (part.mediaType === "application/pdf") {
                      betas.add("pdfs-2024-09-25");
                      const enableCitations = await shouldEnableCitations(
                        part.providerOptions
                      );
                      const metadata = await getDocumentMetadata(
                        part.providerOptions
                      );
                      anthropicContent.push({
                        type: "document",
                        source: isUrlData(part.data) ? {
                          type: "url",
                          url: getUrlString(part.data)
                        } : {
                          type: "base64",
                          media_type: "application/pdf",
                          data: convertToBase64(part.data)
                        },
                        title: (_b = metadata.title) != null ? _b : part.filename,
                        ...metadata.context && { context: metadata.context },
                        ...enableCitations && {
                          citations: { enabled: true }
                        },
                        cache_control: cacheControl
                      });
                    } else if (part.mediaType === "text/plain") {
                      const enableCitations = await shouldEnableCitations(
                        part.providerOptions
                      );
                      const metadata = await getDocumentMetadata(
                        part.providerOptions
                      );
                      anthropicContent.push({
                        type: "document",
                        source: isUrlData(part.data) ? {
                          type: "url",
                          url: getUrlString(part.data)
                        } : {
                          type: "text",
                          media_type: "text/plain",
                          data: convertToString(part.data)
                        },
                        title: (_c = metadata.title) != null ? _c : part.filename,
                        ...metadata.context && { context: metadata.context },
                        ...enableCitations && {
                          citations: { enabled: true }
                        },
                        cache_control: cacheControl
                      });
                    } else {
                      throw new UnsupportedFunctionalityError2({
                        functionality: `media type: ${part.mediaType}`
                      });
                    }
                    break;
                  }
                }
              }
              break;
            }
            case "tool": {
              for (let i2 = 0; i2 < content.length; i2++) {
                const part = content[i2];
                if (part.type === "tool-approval-response") {
                  continue;
                }
                const isLastPart = i2 === content.length - 1;
                const cacheControl = (_d = validator.getCacheControl(part.providerOptions, {
                  type: "tool result part",
                  canCache: true
                })) != null ? _d : isLastPart ? validator.getCacheControl(message.providerOptions, {
                  type: "tool result message",
                  canCache: true
                }) : void 0;
                const output = part.output;
                let contentValue;
                switch (output.type) {
                  case "content":
                    contentValue = output.value.map((contentPart) => {
                      switch (contentPart.type) {
                        case "text":
                          return {
                            type: "text",
                            text: contentPart.text
                          };
                        case "image-data": {
                          return {
                            type: "image",
                            source: {
                              type: "base64",
                              media_type: contentPart.mediaType,
                              data: contentPart.data
                            }
                          };
                        }
                        case "image-url": {
                          return {
                            type: "image",
                            source: {
                              type: "url",
                              url: contentPart.url
                            }
                          };
                        }
                        case "file-url": {
                          return {
                            type: "document",
                            source: {
                              type: "url",
                              url: contentPart.url
                            }
                          };
                        }
                        case "file-data": {
                          if (contentPart.mediaType === "application/pdf") {
                            betas.add("pdfs-2024-09-25");
                            return {
                              type: "document",
                              source: {
                                type: "base64",
                                media_type: contentPart.mediaType,
                                data: contentPart.data
                              }
                            };
                          }
                          warnings.push({
                            type: "other",
                            message: `unsupported tool content part type: ${contentPart.type} with media type: ${contentPart.mediaType}`
                          });
                          return void 0;
                        }
                        default: {
                          warnings.push({
                            type: "other",
                            message: `unsupported tool content part type: ${contentPart.type}`
                          });
                          return void 0;
                        }
                      }
                    }).filter(isNonNullable);
                    break;
                  case "text":
                  case "error-text":
                    contentValue = output.value;
                    break;
                  case "execution-denied":
                    contentValue = (_e = output.reason) != null ? _e : "Tool execution denied.";
                    break;
                  case "json":
                  case "error-json":
                  default:
                    contentValue = JSON.stringify(output.value);
                    break;
                }
                anthropicContent.push({
                  type: "tool_result",
                  tool_use_id: part.toolCallId,
                  content: contentValue,
                  is_error: output.type === "error-text" || output.type === "error-json" ? true : void 0,
                  cache_control: cacheControl
                });
              }
              break;
            }
            default: {
              const _exhaustiveCheck = role;
              throw new Error(`Unsupported role: ${_exhaustiveCheck}`);
            }
          }
        }
        messages.push({ role: "user", content: anthropicContent });
        break;
      }
      case "assistant": {
        const anthropicContent = [];
        const mcpToolUseIds = /* @__PURE__ */ new Set();
        for (let j = 0; j < block.messages.length; j++) {
          const message = block.messages[j];
          const isLastMessage = j === block.messages.length - 1;
          const { content } = message;
          for (let k = 0; k < content.length; k++) {
            const part = content[k];
            const isLastContentPart = k === content.length - 1;
            const cacheControl = (_f = validator.getCacheControl(part.providerOptions, {
              type: "assistant message part",
              canCache: true
            })) != null ? _f : isLastContentPart ? validator.getCacheControl(message.providerOptions, {
              type: "assistant message",
              canCache: true
            }) : void 0;
            switch (part.type) {
              case "text": {
                anthropicContent.push({
                  type: "text",
                  text: (
                    // trim the last text part if it's the last message in the block
                    // because Anthropic does not allow trailing whitespace
                    // in pre-filled assistant responses
                    isLastBlock && isLastMessage && isLastContentPart ? part.text.trim() : part.text
                  ),
                  cache_control: cacheControl
                });
                break;
              }
              case "reasoning": {
                if (sendReasoning) {
                  const reasoningMetadata = await parseProviderOptions({
                    provider: "anthropic",
                    providerOptions: part.providerOptions,
                    schema: anthropicReasoningMetadataSchema
                  });
                  if (reasoningMetadata != null) {
                    if (reasoningMetadata.signature != null) {
                      validator.getCacheControl(part.providerOptions, {
                        type: "thinking block",
                        canCache: false
                      });
                      anthropicContent.push({
                        type: "thinking",
                        thinking: part.text,
                        signature: reasoningMetadata.signature
                      });
                    } else if (reasoningMetadata.redactedData != null) {
                      validator.getCacheControl(part.providerOptions, {
                        type: "redacted thinking block",
                        canCache: false
                      });
                      anthropicContent.push({
                        type: "redacted_thinking",
                        data: reasoningMetadata.redactedData
                      });
                    } else {
                      warnings.push({
                        type: "other",
                        message: "unsupported reasoning metadata"
                      });
                    }
                  } else {
                    warnings.push({
                      type: "other",
                      message: "unsupported reasoning metadata"
                    });
                  }
                } else {
                  warnings.push({
                    type: "other",
                    message: "sending reasoning content is disabled for this model"
                  });
                }
                break;
              }
              case "tool-call": {
                if (part.providerExecuted) {
                  const providerToolName = toolNameMapping.toProviderToolName(
                    part.toolName
                  );
                  const isMcpToolUse = ((_h = (_g = part.providerOptions) == null ? void 0 : _g.anthropic) == null ? void 0 : _h.type) === "mcp-tool-use";
                  if (isMcpToolUse) {
                    mcpToolUseIds.add(part.toolCallId);
                    const serverName = (_j = (_i = part.providerOptions) == null ? void 0 : _i.anthropic) == null ? void 0 : _j.serverName;
                    if (serverName == null || typeof serverName !== "string") {
                      warnings.push({
                        type: "other",
                        message: "mcp tool use server name is required and must be a string"
                      });
                      break;
                    }
                    anthropicContent.push({
                      type: "mcp_tool_use",
                      id: part.toolCallId,
                      name: part.toolName,
                      input: part.input,
                      server_name: serverName,
                      cache_control: cacheControl
                    });
                  } else if (
                    // code execution 20250825:
                    providerToolName === "code_execution" && part.input != null && typeof part.input === "object" && "type" in part.input && typeof part.input.type === "string" && (part.input.type === "bash_code_execution" || part.input.type === "text_editor_code_execution")
                  ) {
                    anthropicContent.push({
                      type: "server_tool_use",
                      id: part.toolCallId,
                      name: part.input.type,
                      // map back to subtool name
                      input: part.input,
                      cache_control: cacheControl
                    });
                  } else if (
                    // code execution 20250825 programmatic tool calling:
                    // Strip the fake 'programmatic-tool-call' type before sending to Anthropic
                    providerToolName === "code_execution" && part.input != null && typeof part.input === "object" && "type" in part.input && part.input.type === "programmatic-tool-call"
                  ) {
                    const { type: _, ...inputWithoutType } = part.input;
                    anthropicContent.push({
                      type: "server_tool_use",
                      id: part.toolCallId,
                      name: "code_execution",
                      input: inputWithoutType,
                      cache_control: cacheControl
                    });
                  } else {
                    if (providerToolName === "code_execution" || // code execution 20250522
                    providerToolName === "web_fetch" || providerToolName === "web_search") {
                      anthropicContent.push({
                        type: "server_tool_use",
                        id: part.toolCallId,
                        name: providerToolName,
                        input: part.input,
                        cache_control: cacheControl
                      });
                    } else if (providerToolName === "tool_search_tool_regex" || providerToolName === "tool_search_tool_bm25") {
                      anthropicContent.push({
                        type: "server_tool_use",
                        id: part.toolCallId,
                        name: providerToolName,
                        input: part.input,
                        cache_control: cacheControl
                      });
                    } else {
                      warnings.push({
                        type: "other",
                        message: `provider executed tool call for tool ${part.toolName} is not supported`
                      });
                    }
                  }
                  break;
                }
                const callerOptions = (_k = part.providerOptions) == null ? void 0 : _k.anthropic;
                const caller = (callerOptions == null ? void 0 : callerOptions.caller) ? callerOptions.caller.type === "code_execution_20250825" && callerOptions.caller.toolId ? {
                  type: "code_execution_20250825",
                  tool_id: callerOptions.caller.toolId
                } : callerOptions.caller.type === "direct" ? { type: "direct" } : void 0 : void 0;
                anthropicContent.push({
                  type: "tool_use",
                  id: part.toolCallId,
                  name: part.toolName,
                  input: part.input,
                  ...caller && { caller },
                  cache_control: cacheControl
                });
                break;
              }
              case "tool-result": {
                const providerToolName = toolNameMapping.toProviderToolName(
                  part.toolName
                );
                if (mcpToolUseIds.has(part.toolCallId)) {
                  const output = part.output;
                  if (output.type !== "json" && output.type !== "error-json") {
                    warnings.push({
                      type: "other",
                      message: `provider executed tool result output type ${output.type} for tool ${part.toolName} is not supported`
                    });
                    break;
                  }
                  anthropicContent.push({
                    type: "mcp_tool_result",
                    tool_use_id: part.toolCallId,
                    is_error: output.type === "error-json",
                    content: output.value,
                    cache_control: cacheControl
                  });
                } else if (providerToolName === "code_execution") {
                  const output = part.output;
                  if (output.type === "error-text" || output.type === "error-json") {
                    let errorInfo = {};
                    try {
                      if (typeof output.value === "string") {
                        errorInfo = JSON.parse(output.value);
                      } else if (typeof output.value === "object" && output.value !== null) {
                        errorInfo = output.value;
                      }
                    } catch (e) {
                    }
                    if (errorInfo.type === "code_execution_tool_result_error") {
                      anthropicContent.push({
                        type: "code_execution_tool_result",
                        tool_use_id: part.toolCallId,
                        content: {
                          type: "code_execution_tool_result_error",
                          error_code: (_l = errorInfo.errorCode) != null ? _l : "unknown"
                        },
                        cache_control: cacheControl
                      });
                    } else {
                      anthropicContent.push({
                        type: "bash_code_execution_tool_result",
                        tool_use_id: part.toolCallId,
                        cache_control: cacheControl,
                        content: {
                          type: "bash_code_execution_tool_result_error",
                          error_code: (_m = errorInfo.errorCode) != null ? _m : "unknown"
                        }
                      });
                    }
                    break;
                  }
                  if (output.type !== "json") {
                    warnings.push({
                      type: "other",
                      message: `provider executed tool result output type ${output.type} for tool ${part.toolName} is not supported`
                    });
                    break;
                  }
                  if (output.value == null || typeof output.value !== "object" || !("type" in output.value) || typeof output.value.type !== "string") {
                    warnings.push({
                      type: "other",
                      message: `provider executed tool result output value is not a valid code execution result for tool ${part.toolName}`
                    });
                    break;
                  }
                  if (output.value.type === "code_execution_result") {
                    const codeExecutionOutput = await validateTypes2({
                      value: output.value,
                      schema: codeExecution_20250522OutputSchema
                    });
                    anthropicContent.push({
                      type: "code_execution_tool_result",
                      tool_use_id: part.toolCallId,
                      content: {
                        type: codeExecutionOutput.type,
                        stdout: codeExecutionOutput.stdout,
                        stderr: codeExecutionOutput.stderr,
                        return_code: codeExecutionOutput.return_code,
                        content: (_n = codeExecutionOutput.content) != null ? _n : []
                      },
                      cache_control: cacheControl
                    });
                  } else {
                    const codeExecutionOutput = await validateTypes2({
                      value: output.value,
                      schema: codeExecution_20250825OutputSchema
                    });
                    if (codeExecutionOutput.type === "code_execution_result") {
                      anthropicContent.push({
                        type: "code_execution_tool_result",
                        tool_use_id: part.toolCallId,
                        content: {
                          type: codeExecutionOutput.type,
                          stdout: codeExecutionOutput.stdout,
                          stderr: codeExecutionOutput.stderr,
                          return_code: codeExecutionOutput.return_code,
                          content: (_o = codeExecutionOutput.content) != null ? _o : []
                        },
                        cache_control: cacheControl
                      });
                    } else if (codeExecutionOutput.type === "bash_code_execution_result" || codeExecutionOutput.type === "bash_code_execution_tool_result_error") {
                      anthropicContent.push({
                        type: "bash_code_execution_tool_result",
                        tool_use_id: part.toolCallId,
                        cache_control: cacheControl,
                        content: codeExecutionOutput
                      });
                    } else {
                      anthropicContent.push({
                        type: "text_editor_code_execution_tool_result",
                        tool_use_id: part.toolCallId,
                        cache_control: cacheControl,
                        content: codeExecutionOutput
                      });
                    }
                  }
                  break;
                }
                if (providerToolName === "web_fetch") {
                  const output = part.output;
                  if (output.type === "error-json") {
                    const errorValue = JSON.parse(output.value);
                    anthropicContent.push({
                      type: "web_fetch_tool_result",
                      tool_use_id: part.toolCallId,
                      content: {
                        type: "web_fetch_tool_result_error",
                        error_code: errorValue.errorCode
                      },
                      cache_control: cacheControl
                    });
                    break;
                  }
                  if (output.type !== "json") {
                    warnings.push({
                      type: "other",
                      message: `provider executed tool result output type ${output.type} for tool ${part.toolName} is not supported`
                    });
                    break;
                  }
                  const webFetchOutput = await validateTypes2({
                    value: output.value,
                    schema: webFetch_20250910OutputSchema
                  });
                  anthropicContent.push({
                    type: "web_fetch_tool_result",
                    tool_use_id: part.toolCallId,
                    content: {
                      type: "web_fetch_result",
                      url: webFetchOutput.url,
                      retrieved_at: webFetchOutput.retrievedAt,
                      content: {
                        type: "document",
                        title: webFetchOutput.content.title,
                        citations: webFetchOutput.content.citations,
                        source: {
                          type: webFetchOutput.content.source.type,
                          media_type: webFetchOutput.content.source.mediaType,
                          data: webFetchOutput.content.source.data
                        }
                      }
                    },
                    cache_control: cacheControl
                  });
                  break;
                }
                if (providerToolName === "web_search") {
                  const output = part.output;
                  if (output.type !== "json") {
                    warnings.push({
                      type: "other",
                      message: `provider executed tool result output type ${output.type} for tool ${part.toolName} is not supported`
                    });
                    break;
                  }
                  const webSearchOutput = await validateTypes2({
                    value: output.value,
                    schema: webSearch_20250305OutputSchema
                  });
                  anthropicContent.push({
                    type: "web_search_tool_result",
                    tool_use_id: part.toolCallId,
                    content: webSearchOutput.map((result) => ({
                      url: result.url,
                      title: result.title,
                      page_age: result.pageAge,
                      encrypted_content: result.encryptedContent,
                      type: result.type
                    })),
                    cache_control: cacheControl
                  });
                  break;
                }
                if (providerToolName === "tool_search_tool_regex" || providerToolName === "tool_search_tool_bm25") {
                  const output = part.output;
                  if (output.type !== "json") {
                    warnings.push({
                      type: "other",
                      message: `provider executed tool result output type ${output.type} for tool ${part.toolName} is not supported`
                    });
                    break;
                  }
                  const toolSearchOutput = await validateTypes2({
                    value: output.value,
                    schema: toolSearchRegex_20251119OutputSchema
                  });
                  const toolReferences = toolSearchOutput.map((ref) => ({
                    type: "tool_reference",
                    tool_name: ref.toolName
                  }));
                  anthropicContent.push({
                    type: "tool_search_tool_result",
                    tool_use_id: part.toolCallId,
                    content: {
                      type: "tool_search_tool_search_result",
                      tool_references: toolReferences
                    },
                    cache_control: cacheControl
                  });
                  break;
                }
                warnings.push({
                  type: "other",
                  message: `provider executed tool result for tool ${part.toolName} is not supported`
                });
                break;
              }
            }
          }
        }
        messages.push({ role: "assistant", content: anthropicContent });
        break;
      }
      default: {
        const _exhaustiveCheck = type;
        throw new Error(`content type: ${_exhaustiveCheck}`);
      }
    }
  }
  return {
    prompt: { system, messages },
    betas
  };
}
function groupIntoBlocks(prompt) {
  const blocks = [];
  let currentBlock = void 0;
  for (const message of prompt) {
    const { role } = message;
    switch (role) {
      case "system": {
        if ((currentBlock == null ? void 0 : currentBlock.type) !== "system") {
          currentBlock = { type: "system", messages: [] };
          blocks.push(currentBlock);
        }
        currentBlock.messages.push(message);
        break;
      }
      case "assistant": {
        if ((currentBlock == null ? void 0 : currentBlock.type) !== "assistant") {
          currentBlock = { type: "assistant", messages: [] };
          blocks.push(currentBlock);
        }
        currentBlock.messages.push(message);
        break;
      }
      case "user": {
        if ((currentBlock == null ? void 0 : currentBlock.type) !== "user") {
          currentBlock = { type: "user", messages: [] };
          blocks.push(currentBlock);
        }
        currentBlock.messages.push(message);
        break;
      }
      case "tool": {
        if ((currentBlock == null ? void 0 : currentBlock.type) !== "user") {
          currentBlock = { type: "user", messages: [] };
          blocks.push(currentBlock);
        }
        currentBlock.messages.push(message);
        break;
      }
      default: {
        const _exhaustiveCheck = role;
        throw new Error(`Unsupported role: ${_exhaustiveCheck}`);
      }
    }
  }
  return blocks;
}

// src/map-anthropic-stop-reason.ts
function mapAnthropicStopReason({
  finishReason,
  isJsonResponseFromTool
}) {
  switch (finishReason) {
    case "pause_turn":
    case "end_turn":
    case "stop_sequence":
      return "stop";
    case "refusal":
      return "content-filter";
    case "tool_use":
      return isJsonResponseFromTool ? "stop" : "tool-calls";
    case "max_tokens":
    case "model_context_window_exceeded":
      return "length";
    default:
      return "other";
  }
}

// src/anthropic-messages-language-model.ts
function createCitationSource(citation, citationDocuments, generateId3) {
  var _a;
  if (citation.type !== "page_location" && citation.type !== "char_location") {
    return;
  }
  const documentInfo = citationDocuments[citation.document_index];
  if (!documentInfo) {
    return;
  }
  return {
    type: "source",
    sourceType: "document",
    id: generateId3(),
    mediaType: documentInfo.mediaType,
    title: (_a = citation.document_title) != null ? _a : documentInfo.title,
    filename: documentInfo.filename,
    providerMetadata: {
      anthropic: citation.type === "page_location" ? {
        citedText: citation.cited_text,
        startPageNumber: citation.start_page_number,
        endPageNumber: citation.end_page_number
      } : {
        citedText: citation.cited_text,
        startCharIndex: citation.start_char_index,
        endCharIndex: citation.end_char_index
      }
    }
  };
}
var AnthropicMessagesLanguageModel = class {
  constructor(modelId, config) {
    this.specificationVersion = "v3";
    var _a;
    this.modelId = modelId;
    this.config = config;
    this.generateId = (_a = config.generateId) != null ? _a : generateId;
  }
  supportsUrl(url) {
    return url.protocol === "https:";
  }
  get provider() {
    return this.config.provider;
  }
  get supportedUrls() {
    var _a, _b, _c;
    return (_c = (_b = (_a = this.config).supportedUrls) == null ? void 0 : _b.call(_a)) != null ? _c : {};
  }
  async getArgs({
    userSuppliedBetas,
    prompt,
    maxOutputTokens,
    temperature,
    topP,
    topK,
    frequencyPenalty,
    presencePenalty,
    stopSequences,
    responseFormat,
    seed,
    tools,
    toolChoice,
    providerOptions,
    stream
  }) {
    var _a, _b, _c, _d, _e, _f;
    const warnings = [];
    if (frequencyPenalty != null) {
      warnings.push({ type: "unsupported", feature: "frequencyPenalty" });
    }
    if (presencePenalty != null) {
      warnings.push({ type: "unsupported", feature: "presencePenalty" });
    }
    if (seed != null) {
      warnings.push({ type: "unsupported", feature: "seed" });
    }
    if (temperature != null && temperature > 1) {
      warnings.push({
        type: "unsupported",
        feature: "temperature",
        details: `${temperature} exceeds anthropic maximum of 1.0. clamped to 1.0`
      });
      temperature = 1;
    } else if (temperature != null && temperature < 0) {
      warnings.push({
        type: "unsupported",
        feature: "temperature",
        details: `${temperature} is below anthropic minimum of 0. clamped to 0`
      });
      temperature = 0;
    }
    if ((responseFormat == null ? void 0 : responseFormat.type) === "json") {
      if (responseFormat.schema == null) {
        warnings.push({
          type: "unsupported",
          feature: "responseFormat",
          details: "JSON response format requires a schema. The response format is ignored."
        });
      }
    }
    const anthropicOptions = await parseProviderOptions2({
      provider: "anthropic",
      providerOptions,
      schema: anthropicProviderOptions
    });
    const {
      maxOutputTokens: maxOutputTokensForModel,
      supportsStructuredOutput: modelSupportsStructuredOutput,
      isKnownModel
    } = getModelCapabilities(this.modelId);
    const supportsStructuredOutput = ((_a = this.config.supportsNativeStructuredOutput) != null ? _a : true) && modelSupportsStructuredOutput;
    const structureOutputMode = (_b = anthropicOptions == null ? void 0 : anthropicOptions.structuredOutputMode) != null ? _b : "auto";
    const useStructuredOutput = structureOutputMode === "outputFormat" || structureOutputMode === "auto" && supportsStructuredOutput;
    const jsonResponseTool = (responseFormat == null ? void 0 : responseFormat.type) === "json" && responseFormat.schema != null && !useStructuredOutput ? {
      type: "function",
      name: "json",
      description: "Respond with a JSON object.",
      inputSchema: responseFormat.schema
    } : void 0;
    const contextManagement = anthropicOptions == null ? void 0 : anthropicOptions.contextManagement;
    const cacheControlValidator = new CacheControlValidator();
    const toolNameMapping = createToolNameMapping({
      tools,
      providerToolNames: {
        "anthropic.code_execution_20250522": "code_execution",
        "anthropic.code_execution_20250825": "code_execution",
        "anthropic.computer_20241022": "computer",
        "anthropic.computer_20250124": "computer",
        "anthropic.text_editor_20241022": "str_replace_editor",
        "anthropic.text_editor_20250124": "str_replace_editor",
        "anthropic.text_editor_20250429": "str_replace_based_edit_tool",
        "anthropic.text_editor_20250728": "str_replace_based_edit_tool",
        "anthropic.bash_20241022": "bash",
        "anthropic.bash_20250124": "bash",
        "anthropic.memory_20250818": "memory",
        "anthropic.web_search_20250305": "web_search",
        "anthropic.web_fetch_20250910": "web_fetch",
        "anthropic.tool_search_regex_20251119": "tool_search_tool_regex",
        "anthropic.tool_search_bm25_20251119": "tool_search_tool_bm25"
      }
    });
    const { prompt: messagesPrompt, betas } = await convertToAnthropicMessagesPrompt({
      prompt,
      sendReasoning: (_c = anthropicOptions == null ? void 0 : anthropicOptions.sendReasoning) != null ? _c : true,
      warnings,
      cacheControlValidator,
      toolNameMapping
    });
    const isThinking = ((_d = anthropicOptions == null ? void 0 : anthropicOptions.thinking) == null ? void 0 : _d.type) === "enabled";
    let thinkingBudget = (_e = anthropicOptions == null ? void 0 : anthropicOptions.thinking) == null ? void 0 : _e.budgetTokens;
    const maxTokens = maxOutputTokens != null ? maxOutputTokens : maxOutputTokensForModel;
    const baseArgs = {
      // model id:
      model: this.modelId,
      // standardized settings:
      max_tokens: maxTokens,
      temperature,
      top_k: topK,
      top_p: topP,
      stop_sequences: stopSequences,
      // provider specific settings:
      ...isThinking && {
        thinking: { type: "enabled", budget_tokens: thinkingBudget }
      },
      ...(anthropicOptions == null ? void 0 : anthropicOptions.effort) && {
        output_config: { effort: anthropicOptions.effort }
      },
      // structured output:
      ...useStructuredOutput && (responseFormat == null ? void 0 : responseFormat.type) === "json" && responseFormat.schema != null && {
        output_format: {
          type: "json_schema",
          schema: responseFormat.schema
        }
      },
      // mcp servers:
      ...(anthropicOptions == null ? void 0 : anthropicOptions.mcpServers) && anthropicOptions.mcpServers.length > 0 && {
        mcp_servers: anthropicOptions.mcpServers.map((server) => ({
          type: server.type,
          name: server.name,
          url: server.url,
          authorization_token: server.authorizationToken,
          tool_configuration: server.toolConfiguration ? {
            allowed_tools: server.toolConfiguration.allowedTools,
            enabled: server.toolConfiguration.enabled
          } : void 0
        }))
      },
      // container: For programmatic tool calling (just an ID string) or agent skills (object with id and skills)
      ...(anthropicOptions == null ? void 0 : anthropicOptions.container) && {
        container: anthropicOptions.container.skills && anthropicOptions.container.skills.length > 0 ? (
          // Object format when skills are provided (agent skills feature)
          {
            id: anthropicOptions.container.id,
            skills: anthropicOptions.container.skills.map((skill) => ({
              type: skill.type,
              skill_id: skill.skillId,
              version: skill.version
            }))
          }
        ) : (
          // String format for container ID only (programmatic tool calling)
          anthropicOptions.container.id
        )
      },
      // prompt:
      system: messagesPrompt.system,
      messages: messagesPrompt.messages,
      ...contextManagement && {
        context_management: {
          edits: contextManagement.edits.map((edit) => {
            const strategy = edit.type;
            switch (strategy) {
              case "clear_tool_uses_20250919":
                return {
                  type: edit.type,
                  ...edit.trigger !== void 0 && {
                    trigger: edit.trigger
                  },
                  ...edit.keep !== void 0 && { keep: edit.keep },
                  ...edit.clearAtLeast !== void 0 && {
                    clear_at_least: edit.clearAtLeast
                  },
                  ...edit.clearToolInputs !== void 0 && {
                    clear_tool_inputs: edit.clearToolInputs
                  },
                  ...edit.excludeTools !== void 0 && {
                    exclude_tools: edit.excludeTools
                  }
                };
              case "clear_thinking_20251015":
                return {
                  type: edit.type,
                  ...edit.keep !== void 0 && { keep: edit.keep }
                };
              default:
                warnings.push({
                  type: "other",
                  message: `Unknown context management strategy: ${strategy}`
                });
                return void 0;
            }
          }).filter((edit) => edit !== void 0)
        }
      }
    };
    if (isThinking) {
      if (thinkingBudget == null) {
        warnings.push({
          type: "compatibility",
          feature: "extended thinking",
          details: "thinking budget is required when thinking is enabled. using default budget of 1024 tokens."
        });
        baseArgs.thinking = {
          type: "enabled",
          budget_tokens: 1024
        };
        thinkingBudget = 1024;
      }
      if (baseArgs.temperature != null) {
        baseArgs.temperature = void 0;
        warnings.push({
          type: "unsupported",
          feature: "temperature",
          details: "temperature is not supported when thinking is enabled"
        });
      }
      if (topK != null) {
        baseArgs.top_k = void 0;
        warnings.push({
          type: "unsupported",
          feature: "topK",
          details: "topK is not supported when thinking is enabled"
        });
      }
      if (topP != null) {
        baseArgs.top_p = void 0;
        warnings.push({
          type: "unsupported",
          feature: "topP",
          details: "topP is not supported when thinking is enabled"
        });
      }
      baseArgs.max_tokens = maxTokens + (thinkingBudget != null ? thinkingBudget : 0);
    } else {
      if (topP != null && temperature != null) {
        warnings.push({
          type: "unsupported",
          feature: "topP",
          details: `topP is not supported when temperature is set. topP is ignored.`
        });
        baseArgs.top_p = void 0;
      }
    }
    if (isKnownModel && baseArgs.max_tokens > maxOutputTokensForModel) {
      if (maxOutputTokens != null) {
        warnings.push({
          type: "unsupported",
          feature: "maxOutputTokens",
          details: `${baseArgs.max_tokens} (maxOutputTokens + thinkingBudget) is greater than ${this.modelId} ${maxOutputTokensForModel} max output tokens. The max output tokens have been limited to ${maxOutputTokensForModel}.`
        });
      }
      baseArgs.max_tokens = maxOutputTokensForModel;
    }
    if ((anthropicOptions == null ? void 0 : anthropicOptions.mcpServers) && anthropicOptions.mcpServers.length > 0) {
      betas.add("mcp-client-2025-04-04");
    }
    if (contextManagement) {
      betas.add("context-management-2025-06-27");
    }
    if ((anthropicOptions == null ? void 0 : anthropicOptions.container) && anthropicOptions.container.skills && anthropicOptions.container.skills.length > 0) {
      betas.add("code-execution-2025-08-25");
      betas.add("skills-2025-10-02");
      betas.add("files-api-2025-04-14");
      if (!(tools == null ? void 0 : tools.some(
        (tool) => tool.type === "provider" && tool.id === "anthropic.code_execution_20250825"
      ))) {
        warnings.push({
          type: "other",
          message: "code execution tool is required when using skills"
        });
      }
    }
    if (anthropicOptions == null ? void 0 : anthropicOptions.effort) {
      betas.add("effort-2025-11-24");
    }
    if (stream && ((_f = anthropicOptions == null ? void 0 : anthropicOptions.toolStreaming) != null ? _f : true)) {
      betas.add("fine-grained-tool-streaming-2025-05-14");
    }
    const usingNativeOutputFormat = useStructuredOutput && (responseFormat == null ? void 0 : responseFormat.type) === "json" && responseFormat.schema != null;
    if (usingNativeOutputFormat) {
      betas.add("structured-outputs-2025-11-13");
    }
    const {
      tools: anthropicTools2,
      toolChoice: anthropicToolChoice,
      toolWarnings,
      betas: toolsBetas
    } = await prepareTools(
      jsonResponseTool != null ? {
        tools: [...tools != null ? tools : [], jsonResponseTool],
        toolChoice: { type: "required" },
        disableParallelToolUse: true,
        cacheControlValidator,
        supportsStructuredOutput
      } : {
        tools: tools != null ? tools : [],
        toolChoice,
        disableParallelToolUse: anthropicOptions == null ? void 0 : anthropicOptions.disableParallelToolUse,
        cacheControlValidator,
        supportsStructuredOutput
      }
    );
    const cacheWarnings = cacheControlValidator.getWarnings();
    return {
      args: {
        ...baseArgs,
        tools: anthropicTools2,
        tool_choice: anthropicToolChoice,
        stream: stream === true ? true : void 0
        // do not send when not streaming
      },
      warnings: [...warnings, ...toolWarnings, ...cacheWarnings],
      betas: /* @__PURE__ */ new Set([...betas, ...toolsBetas, ...userSuppliedBetas]),
      usesJsonResponseTool: jsonResponseTool != null,
      toolNameMapping
    };
  }
  async getHeaders({
    betas,
    headers
  }) {
    return combineHeaders(
      await resolve(this.config.headers),
      headers,
      betas.size > 0 ? { "anthropic-beta": Array.from(betas).join(",") } : {}
    );
  }
  async getBetasFromHeaders(requestHeaders) {
    var _a, _b;
    const configHeaders = await resolve(this.config.headers);
    const configBetaHeader = (_a = configHeaders["anthropic-beta"]) != null ? _a : "";
    const requestBetaHeader = (_b = requestHeaders == null ? void 0 : requestHeaders["anthropic-beta"]) != null ? _b : "";
    return new Set(
      [
        ...configBetaHeader.toLowerCase().split(","),
        ...requestBetaHeader.toLowerCase().split(",")
      ].map((beta) => beta.trim()).filter((beta) => beta !== "")
    );
  }
  buildRequestUrl(isStreaming) {
    var _a, _b, _c;
    return (_c = (_b = (_a = this.config).buildRequestUrl) == null ? void 0 : _b.call(_a, this.config.baseURL, isStreaming)) != null ? _c : `${this.config.baseURL}/messages`;
  }
  transformRequestBody(args) {
    var _a, _b, _c;
    return (_c = (_b = (_a = this.config).transformRequestBody) == null ? void 0 : _b.call(_a, args)) != null ? _c : args;
  }
  extractCitationDocuments(prompt) {
    const isCitationPart = (part) => {
      var _a, _b;
      if (part.type !== "file") {
        return false;
      }
      if (part.mediaType !== "application/pdf" && part.mediaType !== "text/plain") {
        return false;
      }
      const anthropic2 = (_a = part.providerOptions) == null ? void 0 : _a.anthropic;
      const citationsConfig = anthropic2 == null ? void 0 : anthropic2.citations;
      return (_b = citationsConfig == null ? void 0 : citationsConfig.enabled) != null ? _b : false;
    };
    return prompt.filter((message) => message.role === "user").flatMap((message) => message.content).filter(isCitationPart).map((part) => {
      var _a;
      const filePart = part;
      return {
        title: (_a = filePart.filename) != null ? _a : "Untitled Document",
        filename: filePart.filename,
        mediaType: filePart.mediaType
      };
    });
  }
  async doGenerate(options) {
    var _a, _b, _c, _d, _e, _f, _g, _h, _i, _j, _k;
    const { args, warnings, betas, usesJsonResponseTool, toolNameMapping } = await this.getArgs({
      ...options,
      stream: false,
      userSuppliedBetas: await this.getBetasFromHeaders(options.headers)
    });
    const citationDocuments = this.extractCitationDocuments(options.prompt);
    const {
      responseHeaders,
      value: response,
      rawValue: rawResponse
    } = await postJsonToApi({
      url: this.buildRequestUrl(false),
      headers: await this.getHeaders({ betas, headers: options.headers }),
      body: this.transformRequestBody(args),
      failedResponseHandler: anthropicFailedResponseHandler,
      successfulResponseHandler: createJsonResponseHandler(
        anthropicMessagesResponseSchema
      ),
      abortSignal: options.abortSignal,
      fetch: this.config.fetch
    });
    const content = [];
    const mcpToolCalls = {};
    const serverToolCalls = {};
    let isJsonResponseFromTool = false;
    for (const part of response.content) {
      switch (part.type) {
        case "text": {
          if (!usesJsonResponseTool) {
            content.push({ type: "text", text: part.text });
            if (part.citations) {
              for (const citation of part.citations) {
                const source = createCitationSource(
                  citation,
                  citationDocuments,
                  this.generateId
                );
                if (source) {
                  content.push(source);
                }
              }
            }
          }
          break;
        }
        case "thinking": {
          content.push({
            type: "reasoning",
            text: part.thinking,
            providerMetadata: {
              anthropic: {
                signature: part.signature
              }
            }
          });
          break;
        }
        case "redacted_thinking": {
          content.push({
            type: "reasoning",
            text: "",
            providerMetadata: {
              anthropic: {
                redactedData: part.data
              }
            }
          });
          break;
        }
        case "tool_use": {
          const isJsonResponseTool = usesJsonResponseTool && part.name === "json";
          if (isJsonResponseTool) {
            isJsonResponseFromTool = true;
            content.push({
              type: "text",
              text: JSON.stringify(part.input)
            });
          } else {
            const caller = part.caller;
            const callerInfo = caller ? {
              type: caller.type,
              toolId: "tool_id" in caller ? caller.tool_id : void 0
            } : void 0;
            content.push({
              type: "tool-call",
              toolCallId: part.id,
              toolName: part.name,
              input: JSON.stringify(part.input),
              ...callerInfo && {
                providerMetadata: {
                  anthropic: {
                    caller: callerInfo
                  }
                }
              }
            });
          }
          break;
        }
        case "server_tool_use": {
          if (part.name === "text_editor_code_execution" || part.name === "bash_code_execution") {
            content.push({
              type: "tool-call",
              toolCallId: part.id,
              toolName: toolNameMapping.toCustomToolName("code_execution"),
              input: JSON.stringify({ type: part.name, ...part.input }),
              providerExecuted: true
            });
          } else if (part.name === "web_search" || part.name === "code_execution" || part.name === "web_fetch") {
            const inputToSerialize = part.name === "code_execution" && part.input != null && typeof part.input === "object" && "code" in part.input && !("type" in part.input) ? { type: "programmatic-tool-call", ...part.input } : part.input;
            content.push({
              type: "tool-call",
              toolCallId: part.id,
              toolName: toolNameMapping.toCustomToolName(part.name),
              input: JSON.stringify(inputToSerialize),
              providerExecuted: true
            });
          } else if (part.name === "tool_search_tool_regex" || part.name === "tool_search_tool_bm25") {
            serverToolCalls[part.id] = part.name;
            content.push({
              type: "tool-call",
              toolCallId: part.id,
              toolName: toolNameMapping.toCustomToolName(part.name),
              input: JSON.stringify(part.input),
              providerExecuted: true
            });
          }
          break;
        }
        case "mcp_tool_use": {
          mcpToolCalls[part.id] = {
            type: "tool-call",
            toolCallId: part.id,
            toolName: part.name,
            input: JSON.stringify(part.input),
            providerExecuted: true,
            dynamic: true,
            providerMetadata: {
              anthropic: {
                type: "mcp-tool-use",
                serverName: part.server_name
              }
            }
          };
          content.push(mcpToolCalls[part.id]);
          break;
        }
        case "mcp_tool_result": {
          content.push({
            type: "tool-result",
            toolCallId: part.tool_use_id,
            toolName: mcpToolCalls[part.tool_use_id].toolName,
            isError: part.is_error,
            result: part.content,
            dynamic: true,
            providerMetadata: mcpToolCalls[part.tool_use_id].providerMetadata
          });
          break;
        }
        case "web_fetch_tool_result": {
          if (part.content.type === "web_fetch_result") {
            content.push({
              type: "tool-result",
              toolCallId: part.tool_use_id,
              toolName: toolNameMapping.toCustomToolName("web_fetch"),
              result: {
                type: "web_fetch_result",
                url: part.content.url,
                retrievedAt: part.content.retrieved_at,
                content: {
                  type: part.content.content.type,
                  title: part.content.content.title,
                  citations: part.content.content.citations,
                  source: {
                    type: part.content.content.source.type,
                    mediaType: part.content.content.source.media_type,
                    data: part.content.content.source.data
                  }
                }
              }
            });
          } else if (part.content.type === "web_fetch_tool_result_error") {
            content.push({
              type: "tool-result",
              toolCallId: part.tool_use_id,
              toolName: toolNameMapping.toCustomToolName("web_fetch"),
              isError: true,
              result: {
                type: "web_fetch_tool_result_error",
                errorCode: part.content.error_code
              }
            });
          }
          break;
        }
        case "web_search_tool_result": {
          if (Array.isArray(part.content)) {
            content.push({
              type: "tool-result",
              toolCallId: part.tool_use_id,
              toolName: toolNameMapping.toCustomToolName("web_search"),
              result: part.content.map((result) => {
                var _a2;
                return {
                  url: result.url,
                  title: result.title,
                  pageAge: (_a2 = result.page_age) != null ? _a2 : null,
                  encryptedContent: result.encrypted_content,
                  type: result.type
                };
              })
            });
            for (const result of part.content) {
              content.push({
                type: "source",
                sourceType: "url",
                id: this.generateId(),
                url: result.url,
                title: result.title,
                providerMetadata: {
                  anthropic: {
                    pageAge: (_a = result.page_age) != null ? _a : null
                  }
                }
              });
            }
          } else {
            content.push({
              type: "tool-result",
              toolCallId: part.tool_use_id,
              toolName: toolNameMapping.toCustomToolName("web_search"),
              isError: true,
              result: {
                type: "web_search_tool_result_error",
                errorCode: part.content.error_code
              }
            });
          }
          break;
        }
        // code execution 20250522:
        case "code_execution_tool_result": {
          if (part.content.type === "code_execution_result") {
            content.push({
              type: "tool-result",
              toolCallId: part.tool_use_id,
              toolName: toolNameMapping.toCustomToolName("code_execution"),
              result: {
                type: part.content.type,
                stdout: part.content.stdout,
                stderr: part.content.stderr,
                return_code: part.content.return_code,
                content: (_b = part.content.content) != null ? _b : []
              }
            });
          } else if (part.content.type === "code_execution_tool_result_error") {
            content.push({
              type: "tool-result",
              toolCallId: part.tool_use_id,
              toolName: toolNameMapping.toCustomToolName("code_execution"),
              isError: true,
              result: {
                type: "code_execution_tool_result_error",
                errorCode: part.content.error_code
              }
            });
          }
          break;
        }
        // code execution 20250825:
        case "bash_code_execution_tool_result":
        case "text_editor_code_execution_tool_result": {
          content.push({
            type: "tool-result",
            toolCallId: part.tool_use_id,
            toolName: toolNameMapping.toCustomToolName("code_execution"),
            result: part.content
          });
          break;
        }
        // tool search tool results:
        case "tool_search_tool_result": {
          const providerToolName = (_c = serverToolCalls[part.tool_use_id]) != null ? _c : "tool_search_tool_regex";
          if (part.content.type === "tool_search_tool_search_result") {
            content.push({
              type: "tool-result",
              toolCallId: part.tool_use_id,
              toolName: toolNameMapping.toCustomToolName(providerToolName),
              result: part.content.tool_references.map((ref) => ({
                type: ref.type,
                toolName: ref.tool_name
              }))
            });
          } else {
            content.push({
              type: "tool-result",
              toolCallId: part.tool_use_id,
              toolName: toolNameMapping.toCustomToolName(providerToolName),
              isError: true,
              result: {
                type: "tool_search_tool_result_error",
                errorCode: part.content.error_code
              }
            });
          }
          break;
        }
      }
    }
    return {
      content,
      finishReason: {
        unified: mapAnthropicStopReason({
          finishReason: response.stop_reason,
          isJsonResponseFromTool
        }),
        raw: (_d = response.stop_reason) != null ? _d : void 0
      },
      usage: convertAnthropicMessagesUsage(response.usage),
      request: { body: args },
      response: {
        id: (_e = response.id) != null ? _e : void 0,
        modelId: (_f = response.model) != null ? _f : void 0,
        headers: responseHeaders,
        body: rawResponse
      },
      warnings,
      providerMetadata: {
        anthropic: {
          usage: response.usage,
          cacheCreationInputTokens: (_g = response.usage.cache_creation_input_tokens) != null ? _g : null,
          stopSequence: (_h = response.stop_sequence) != null ? _h : null,
          container: response.container ? {
            expiresAt: response.container.expires_at,
            id: response.container.id,
            skills: (_j = (_i = response.container.skills) == null ? void 0 : _i.map((skill) => ({
              type: skill.type,
              skillId: skill.skill_id,
              version: skill.version
            }))) != null ? _j : null
          } : null,
          contextManagement: (_k = mapAnthropicResponseContextManagement(
            response.context_management
          )) != null ? _k : null
        }
      }
    };
  }
  async doStream(options) {
    var _a, _b;
    const {
      args: body,
      warnings,
      betas,
      usesJsonResponseTool,
      toolNameMapping
    } = await this.getArgs({
      ...options,
      stream: true,
      userSuppliedBetas: await this.getBetasFromHeaders(options.headers)
    });
    const citationDocuments = this.extractCitationDocuments(options.prompt);
    const url = this.buildRequestUrl(true);
    const { responseHeaders, value: response } = await postJsonToApi({
      url,
      headers: await this.getHeaders({ betas, headers: options.headers }),
      body: this.transformRequestBody(body),
      failedResponseHandler: anthropicFailedResponseHandler,
      successfulResponseHandler: createEventSourceResponseHandler(
        anthropicMessagesChunkSchema
      ),
      abortSignal: options.abortSignal,
      fetch: this.config.fetch
    });
    let finishReason = {
      unified: "other",
      raw: void 0
    };
    const usage = {
      input_tokens: 0,
      output_tokens: 0,
      cache_creation_input_tokens: 0,
      cache_read_input_tokens: 0
    };
    const contentBlocks = {};
    const mcpToolCalls = {};
    const serverToolCalls = {};
    let contextManagement = null;
    let rawUsage = void 0;
    let cacheCreationInputTokens = null;
    let stopSequence = null;
    let container = null;
    let isJsonResponseFromTool = false;
    let blockType = void 0;
    const generateId3 = this.generateId;
    const transformedStream = response.pipeThrough(
      new TransformStream({
        start(controller) {
          controller.enqueue({ type: "stream-start", warnings });
        },
        transform(chunk, controller) {
          var _a2, _b2, _c, _d, _e, _f, _g, _h, _i, _j, _k, _l, _m;
          if (options.includeRawChunks) {
            controller.enqueue({ type: "raw", rawValue: chunk.rawValue });
          }
          if (!chunk.success) {
            controller.enqueue({ type: "error", error: chunk.error });
            return;
          }
          const value = chunk.value;
          switch (value.type) {
            case "ping": {
              return;
            }
            case "content_block_start": {
              const part = value.content_block;
              const contentBlockType = part.type;
              blockType = contentBlockType;
              switch (contentBlockType) {
                case "text": {
                  if (usesJsonResponseTool) {
                    return;
                  }
                  contentBlocks[value.index] = { type: "text" };
                  controller.enqueue({
                    type: "text-start",
                    id: String(value.index)
                  });
                  return;
                }
                case "thinking": {
                  contentBlocks[value.index] = { type: "reasoning" };
                  controller.enqueue({
                    type: "reasoning-start",
                    id: String(value.index)
                  });
                  return;
                }
                case "redacted_thinking": {
                  contentBlocks[value.index] = { type: "reasoning" };
                  controller.enqueue({
                    type: "reasoning-start",
                    id: String(value.index),
                    providerMetadata: {
                      anthropic: {
                        redactedData: part.data
                      }
                    }
                  });
                  return;
                }
                case "tool_use": {
                  const isJsonResponseTool = usesJsonResponseTool && part.name === "json";
                  if (isJsonResponseTool) {
                    isJsonResponseFromTool = true;
                    contentBlocks[value.index] = { type: "text" };
                    controller.enqueue({
                      type: "text-start",
                      id: String(value.index)
                    });
                  } else {
                    const caller = part.caller;
                    const callerInfo = caller ? {
                      type: caller.type,
                      toolId: "tool_id" in caller ? caller.tool_id : void 0
                    } : void 0;
                    const hasNonEmptyInput = part.input && Object.keys(part.input).length > 0;
                    const initialInput = hasNonEmptyInput ? JSON.stringify(part.input) : "";
                    contentBlocks[value.index] = {
                      type: "tool-call",
                      toolCallId: part.id,
                      toolName: part.name,
                      input: initialInput,
                      firstDelta: initialInput.length === 0,
                      ...callerInfo && { caller: callerInfo }
                    };
                    controller.enqueue({
                      type: "tool-input-start",
                      id: part.id,
                      toolName: part.name
                    });
                  }
                  return;
                }
                case "server_tool_use": {
                  if ([
                    "web_fetch",
                    "web_search",
                    // code execution 20250825:
                    "code_execution",
                    // code execution 20250825 text editor:
                    "text_editor_code_execution",
                    // code execution 20250825 bash:
                    "bash_code_execution"
                  ].includes(part.name)) {
                    const providerToolName = part.name === "text_editor_code_execution" || part.name === "bash_code_execution" ? "code_execution" : part.name;
                    const customToolName = toolNameMapping.toCustomToolName(providerToolName);
                    contentBlocks[value.index] = {
                      type: "tool-call",
                      toolCallId: part.id,
                      toolName: customToolName,
                      input: "",
                      providerExecuted: true,
                      firstDelta: true,
                      providerToolName: part.name
                    };
                    controller.enqueue({
                      type: "tool-input-start",
                      id: part.id,
                      toolName: customToolName,
                      providerExecuted: true
                    });
                  } else if (part.name === "tool_search_tool_regex" || part.name === "tool_search_tool_bm25") {
                    serverToolCalls[part.id] = part.name;
                    const customToolName = toolNameMapping.toCustomToolName(
                      part.name
                    );
                    contentBlocks[value.index] = {
                      type: "tool-call",
                      toolCallId: part.id,
                      toolName: customToolName,
                      input: "",
                      providerExecuted: true,
                      firstDelta: true,
                      providerToolName: part.name
                    };
                    controller.enqueue({
                      type: "tool-input-start",
                      id: part.id,
                      toolName: customToolName,
                      providerExecuted: true
                    });
                  }
                  return;
                }
                case "web_fetch_tool_result": {
                  if (part.content.type === "web_fetch_result") {
                    controller.enqueue({
                      type: "tool-result",
                      toolCallId: part.tool_use_id,
                      toolName: toolNameMapping.toCustomToolName("web_fetch"),
                      result: {
                        type: "web_fetch_result",
                        url: part.content.url,
                        retrievedAt: part.content.retrieved_at,
                        content: {
                          type: part.content.content.type,
                          title: part.content.content.title,
                          citations: part.content.content.citations,
                          source: {
                            type: part.content.content.source.type,
                            mediaType: part.content.content.source.media_type,
                            data: part.content.content.source.data
                          }
                        }
                      }
                    });
                  } else if (part.content.type === "web_fetch_tool_result_error") {
                    controller.enqueue({
                      type: "tool-result",
                      toolCallId: part.tool_use_id,
                      toolName: toolNameMapping.toCustomToolName("web_fetch"),
                      isError: true,
                      result: {
                        type: "web_fetch_tool_result_error",
                        errorCode: part.content.error_code
                      }
                    });
                  }
                  return;
                }
                case "web_search_tool_result": {
                  if (Array.isArray(part.content)) {
                    controller.enqueue({
                      type: "tool-result",
                      toolCallId: part.tool_use_id,
                      toolName: toolNameMapping.toCustomToolName("web_search"),
                      result: part.content.map((result) => {
                        var _a3;
                        return {
                          url: result.url,
                          title: result.title,
                          pageAge: (_a3 = result.page_age) != null ? _a3 : null,
                          encryptedContent: result.encrypted_content,
                          type: result.type
                        };
                      })
                    });
                    for (const result of part.content) {
                      controller.enqueue({
                        type: "source",
                        sourceType: "url",
                        id: generateId3(),
                        url: result.url,
                        title: result.title,
                        providerMetadata: {
                          anthropic: {
                            pageAge: (_a2 = result.page_age) != null ? _a2 : null
                          }
                        }
                      });
                    }
                  } else {
                    controller.enqueue({
                      type: "tool-result",
                      toolCallId: part.tool_use_id,
                      toolName: toolNameMapping.toCustomToolName("web_search"),
                      isError: true,
                      result: {
                        type: "web_search_tool_result_error",
                        errorCode: part.content.error_code
                      }
                    });
                  }
                  return;
                }
                // code execution 20250522:
                case "code_execution_tool_result": {
                  if (part.content.type === "code_execution_result") {
                    controller.enqueue({
                      type: "tool-result",
                      toolCallId: part.tool_use_id,
                      toolName: toolNameMapping.toCustomToolName("code_execution"),
                      result: {
                        type: part.content.type,
                        stdout: part.content.stdout,
                        stderr: part.content.stderr,
                        return_code: part.content.return_code,
                        content: (_b2 = part.content.content) != null ? _b2 : []
                      }
                    });
                  } else if (part.content.type === "code_execution_tool_result_error") {
                    controller.enqueue({
                      type: "tool-result",
                      toolCallId: part.tool_use_id,
                      toolName: toolNameMapping.toCustomToolName("code_execution"),
                      isError: true,
                      result: {
                        type: "code_execution_tool_result_error",
                        errorCode: part.content.error_code
                      }
                    });
                  }
                  return;
                }
                // code execution 20250825:
                case "bash_code_execution_tool_result":
                case "text_editor_code_execution_tool_result": {
                  controller.enqueue({
                    type: "tool-result",
                    toolCallId: part.tool_use_id,
                    toolName: toolNameMapping.toCustomToolName("code_execution"),
                    result: part.content
                  });
                  return;
                }
                // tool search tool results:
                case "tool_search_tool_result": {
                  const providerToolName = (_c = serverToolCalls[part.tool_use_id]) != null ? _c : "tool_search_tool_regex";
                  if (part.content.type === "tool_search_tool_search_result") {
                    controller.enqueue({
                      type: "tool-result",
                      toolCallId: part.tool_use_id,
                      toolName: toolNameMapping.toCustomToolName(providerToolName),
                      result: part.content.tool_references.map((ref) => ({
                        type: ref.type,
                        toolName: ref.tool_name
                      }))
                    });
                  } else {
                    controller.enqueue({
                      type: "tool-result",
                      toolCallId: part.tool_use_id,
                      toolName: toolNameMapping.toCustomToolName(providerToolName),
                      isError: true,
                      result: {
                        type: "tool_search_tool_result_error",
                        errorCode: part.content.error_code
                      }
                    });
                  }
                  return;
                }
                case "mcp_tool_use": {
                  mcpToolCalls[part.id] = {
                    type: "tool-call",
                    toolCallId: part.id,
                    toolName: part.name,
                    input: JSON.stringify(part.input),
                    providerExecuted: true,
                    dynamic: true,
                    providerMetadata: {
                      anthropic: {
                        type: "mcp-tool-use",
                        serverName: part.server_name
                      }
                    }
                  };
                  controller.enqueue(mcpToolCalls[part.id]);
                  return;
                }
                case "mcp_tool_result": {
                  controller.enqueue({
                    type: "tool-result",
                    toolCallId: part.tool_use_id,
                    toolName: mcpToolCalls[part.tool_use_id].toolName,
                    isError: part.is_error,
                    result: part.content,
                    dynamic: true,
                    providerMetadata: mcpToolCalls[part.tool_use_id].providerMetadata
                  });
                  return;
                }
                default: {
                  const _exhaustiveCheck = contentBlockType;
                  throw new Error(
                    `Unsupported content block type: ${_exhaustiveCheck}`
                  );
                }
              }
            }
            case "content_block_stop": {
              if (contentBlocks[value.index] != null) {
                const contentBlock = contentBlocks[value.index];
                switch (contentBlock.type) {
                  case "text": {
                    controller.enqueue({
                      type: "text-end",
                      id: String(value.index)
                    });
                    break;
                  }
                  case "reasoning": {
                    controller.enqueue({
                      type: "reasoning-end",
                      id: String(value.index)
                    });
                    break;
                  }
                  case "tool-call":
                    const isJsonResponseTool = usesJsonResponseTool && contentBlock.toolName === "json";
                    if (!isJsonResponseTool) {
                      controller.enqueue({
                        type: "tool-input-end",
                        id: contentBlock.toolCallId
                      });
                      let finalInput = contentBlock.input === "" ? "{}" : contentBlock.input;
                      if (contentBlock.providerToolName === "code_execution") {
                        try {
                          const parsed = JSON.parse(finalInput);
                          if (parsed != null && typeof parsed === "object" && "code" in parsed && !("type" in parsed)) {
                            finalInput = JSON.stringify({
                              type: "programmatic-tool-call",
                              ...parsed
                            });
                          }
                        } catch (e) {
                        }
                      }
                      controller.enqueue({
                        type: "tool-call",
                        toolCallId: contentBlock.toolCallId,
                        toolName: contentBlock.toolName,
                        input: finalInput,
                        providerExecuted: contentBlock.providerExecuted,
                        ...contentBlock.caller && {
                          providerMetadata: {
                            anthropic: {
                              caller: contentBlock.caller
                            }
                          }
                        }
                      });
                    }
                    break;
                }
                delete contentBlocks[value.index];
              }
              blockType = void 0;
              return;
            }
            case "content_block_delta": {
              const deltaType = value.delta.type;
              switch (deltaType) {
                case "text_delta": {
                  if (usesJsonResponseTool) {
                    return;
                  }
                  controller.enqueue({
                    type: "text-delta",
                    id: String(value.index),
                    delta: value.delta.text
                  });
                  return;
                }
                case "thinking_delta": {
                  controller.enqueue({
                    type: "reasoning-delta",
                    id: String(value.index),
                    delta: value.delta.thinking
                  });
                  return;
                }
                case "signature_delta": {
                  if (blockType === "thinking") {
                    controller.enqueue({
                      type: "reasoning-delta",
                      id: String(value.index),
                      delta: "",
                      providerMetadata: {
                        anthropic: {
                          signature: value.delta.signature
                        }
                      }
                    });
                  }
                  return;
                }
                case "input_json_delta": {
                  const contentBlock = contentBlocks[value.index];
                  let delta = value.delta.partial_json;
                  if (delta.length === 0) {
                    return;
                  }
                  if (isJsonResponseFromTool) {
                    if ((contentBlock == null ? void 0 : contentBlock.type) !== "text") {
                      return;
                    }
                    controller.enqueue({
                      type: "text-delta",
                      id: String(value.index),
                      delta
                    });
                  } else {
                    if ((contentBlock == null ? void 0 : contentBlock.type) !== "tool-call") {
                      return;
                    }
                    if (contentBlock.firstDelta && (contentBlock.providerToolName === "bash_code_execution" || contentBlock.providerToolName === "text_editor_code_execution")) {
                      delta = `{"type": "${contentBlock.providerToolName}",${delta.substring(1)}`;
                    }
                    controller.enqueue({
                      type: "tool-input-delta",
                      id: contentBlock.toolCallId,
                      delta
                    });
                    contentBlock.input += delta;
                    contentBlock.firstDelta = false;
                  }
                  return;
                }
                case "citations_delta": {
                  const citation = value.delta.citation;
                  const source = createCitationSource(
                    citation,
                    citationDocuments,
                    generateId3
                  );
                  if (source) {
                    controller.enqueue(source);
                  }
                  return;
                }
                default: {
                  const _exhaustiveCheck = deltaType;
                  throw new Error(
                    `Unsupported delta type: ${_exhaustiveCheck}`
                  );
                }
              }
            }
            case "message_start": {
              usage.input_tokens = value.message.usage.input_tokens;
              usage.cache_read_input_tokens = (_d = value.message.usage.cache_read_input_tokens) != null ? _d : 0;
              usage.cache_creation_input_tokens = (_e = value.message.usage.cache_creation_input_tokens) != null ? _e : 0;
              rawUsage = {
                ...value.message.usage
              };
              cacheCreationInputTokens = (_f = value.message.usage.cache_creation_input_tokens) != null ? _f : null;
              if (value.message.container != null) {
                container = {
                  expiresAt: value.message.container.expires_at,
                  id: value.message.container.id,
                  skills: null
                };
              }
              if (value.message.stop_reason != null) {
                finishReason = {
                  unified: mapAnthropicStopReason({
                    finishReason: value.message.stop_reason,
                    isJsonResponseFromTool
                  }),
                  raw: value.message.stop_reason
                };
              }
              controller.enqueue({
                type: "response-metadata",
                id: (_g = value.message.id) != null ? _g : void 0,
                modelId: (_h = value.message.model) != null ? _h : void 0
              });
              if (value.message.content != null) {
                for (let contentIndex = 0; contentIndex < value.message.content.length; contentIndex++) {
                  const part = value.message.content[contentIndex];
                  if (part.type === "tool_use") {
                    const caller = part.caller;
                    const callerInfo = caller ? {
                      type: caller.type,
                      toolId: "tool_id" in caller ? caller.tool_id : void 0
                    } : void 0;
                    controller.enqueue({
                      type: "tool-input-start",
                      id: part.id,
                      toolName: part.name
                    });
                    const inputStr = JSON.stringify((_i = part.input) != null ? _i : {});
                    controller.enqueue({
                      type: "tool-input-delta",
                      id: part.id,
                      delta: inputStr
                    });
                    controller.enqueue({
                      type: "tool-input-end",
                      id: part.id
                    });
                    controller.enqueue({
                      type: "tool-call",
                      toolCallId: part.id,
                      toolName: part.name,
                      input: inputStr,
                      ...callerInfo && {
                        providerMetadata: {
                          anthropic: {
                            caller: callerInfo
                          }
                        }
                      }
                    });
                  }
                }
              }
              return;
            }
            case "message_delta": {
              usage.output_tokens = value.usage.output_tokens;
              finishReason = {
                unified: mapAnthropicStopReason({
                  finishReason: value.delta.stop_reason,
                  isJsonResponseFromTool
                }),
                raw: (_j = value.delta.stop_reason) != null ? _j : void 0
              };
              stopSequence = (_k = value.delta.stop_sequence) != null ? _k : null;
              container = value.delta.container != null ? {
                expiresAt: value.delta.container.expires_at,
                id: value.delta.container.id,
                skills: (_m = (_l = value.delta.container.skills) == null ? void 0 : _l.map((skill) => ({
                  type: skill.type,
                  skillId: skill.skill_id,
                  version: skill.version
                }))) != null ? _m : null
              } : null;
              if (value.delta.context_management) {
                contextManagement = mapAnthropicResponseContextManagement(
                  value.delta.context_management
                );
              }
              rawUsage = {
                ...rawUsage,
                ...value.usage
              };
              return;
            }
            case "message_stop": {
              controller.enqueue({
                type: "finish",
                finishReason,
                usage: convertAnthropicMessagesUsage(usage),
                providerMetadata: {
                  anthropic: {
                    usage: rawUsage != null ? rawUsage : null,
                    cacheCreationInputTokens,
                    stopSequence,
                    container,
                    contextManagement
                  }
                }
              });
              return;
            }
            case "error": {
              controller.enqueue({ type: "error", error: value.error });
              return;
            }
            default: {
              const _exhaustiveCheck = value;
              throw new Error(`Unsupported chunk type: ${_exhaustiveCheck}`);
            }
          }
        }
      })
    );
    const [streamForFirstChunk, streamForConsumer] = transformedStream.tee();
    const firstChunkReader = streamForFirstChunk.getReader();
    try {
      await firstChunkReader.read();
      let result = await firstChunkReader.read();
      if (((_a = result.value) == null ? void 0 : _a.type) === "raw") {
        result = await firstChunkReader.read();
      }
      if (((_b = result.value) == null ? void 0 : _b.type) === "error") {
        const error = result.value.error;
        throw new APICallError({
          message: error.message,
          url,
          requestBodyValues: body,
          statusCode: error.type === "overloaded_error" ? 529 : 500,
          responseHeaders,
          responseBody: JSON.stringify(error),
          isRetryable: error.type === "overloaded_error"
        });
      }
    } finally {
      firstChunkReader.cancel().catch(() => {
      });
      firstChunkReader.releaseLock();
    }
    return {
      stream: streamForConsumer,
      request: { body },
      response: { headers: responseHeaders }
    };
  }
};
function getModelCapabilities(modelId) {
  if (modelId.includes("claude-sonnet-4-5") || modelId.includes("claude-opus-4-5") || modelId.includes("claude-haiku-4-5")) {
    return {
      maxOutputTokens: 64e3,
      supportsStructuredOutput: true,
      isKnownModel: true
    };
  } else if (modelId.includes("claude-opus-4-1")) {
    return {
      maxOutputTokens: 32e3,
      supportsStructuredOutput: true,
      isKnownModel: true
    };
  } else if (modelId.includes("claude-sonnet-4-") || modelId.includes("claude-3-7-sonnet")) {
    return {
      maxOutputTokens: 64e3,
      supportsStructuredOutput: false,
      isKnownModel: true
    };
  } else if (modelId.includes("claude-opus-4-")) {
    return {
      maxOutputTokens: 32e3,
      supportsStructuredOutput: false,
      isKnownModel: true
    };
  } else if (modelId.includes("claude-3-5-haiku")) {
    return {
      maxOutputTokens: 8192,
      supportsStructuredOutput: false,
      isKnownModel: true
    };
  } else if (modelId.includes("claude-3-haiku")) {
    return {
      maxOutputTokens: 4096,
      supportsStructuredOutput: false,
      isKnownModel: true
    };
  } else {
    return {
      maxOutputTokens: 4096,
      supportsStructuredOutput: false,
      isKnownModel: false
    };
  }
}
function mapAnthropicResponseContextManagement(contextManagement) {
  return contextManagement ? {
    appliedEdits: contextManagement.applied_edits.map((edit) => {
      const strategy = edit.type;
      switch (strategy) {
        case "clear_tool_uses_20250919":
          return {
            type: edit.type,
            clearedToolUses: edit.cleared_tool_uses,
            clearedInputTokens: edit.cleared_input_tokens
          };
        case "clear_thinking_20251015":
          return {
            type: edit.type,
            clearedThinkingTurns: edit.cleared_thinking_turns,
            clearedInputTokens: edit.cleared_input_tokens
          };
      }
    }).filter((edit) => edit !== void 0)
  } : null;
}

// src/tool/bash_20241022.ts
import {
  createProviderToolFactory as createProviderToolFactory2,
  lazySchema as lazySchema9,
  zodSchema as zodSchema9
} from "@ai-sdk/provider-utils";
import { z as z10 } from "zod/v4";
var bash_20241022InputSchema = lazySchema9(
  () => zodSchema9(
    z10.object({
      command: z10.string(),
      restart: z10.boolean().optional()
    })
  )
);
var bash_20241022 = createProviderToolFactory2({
  id: "anthropic.bash_20241022",
  inputSchema: bash_20241022InputSchema
});

// src/tool/bash_20250124.ts
import {
  createProviderToolFactory as createProviderToolFactory3,
  lazySchema as lazySchema10,
  zodSchema as zodSchema10
} from "@ai-sdk/provider-utils";
import { z as z11 } from "zod/v4";
var bash_20250124InputSchema = lazySchema10(
  () => zodSchema10(
    z11.object({
      command: z11.string(),
      restart: z11.boolean().optional()
    })
  )
);
var bash_20250124 = createProviderToolFactory3({
  id: "anthropic.bash_20250124",
  inputSchema: bash_20250124InputSchema
});

// src/tool/computer_20241022.ts
import {
  createProviderToolFactory as createProviderToolFactory4,
  lazySchema as lazySchema11,
  zodSchema as zodSchema11
} from "@ai-sdk/provider-utils";
import { z as z12 } from "zod/v4";
var computer_20241022InputSchema = lazySchema11(
  () => zodSchema11(
    z12.object({
      action: z12.enum([
        "key",
        "type",
        "mouse_move",
        "left_click",
        "left_click_drag",
        "right_click",
        "middle_click",
        "double_click",
        "screenshot",
        "cursor_position"
      ]),
      coordinate: z12.array(z12.number().int()).optional(),
      text: z12.string().optional()
    })
  )
);
var computer_20241022 = createProviderToolFactory4({
  id: "anthropic.computer_20241022",
  inputSchema: computer_20241022InputSchema
});

// src/tool/computer_20250124.ts
import {
  createProviderToolFactory as createProviderToolFactory5,
  lazySchema as lazySchema12,
  zodSchema as zodSchema12
} from "@ai-sdk/provider-utils";
import { z as z13 } from "zod/v4";
var computer_20250124InputSchema = lazySchema12(
  () => zodSchema12(
    z13.object({
      action: z13.enum([
        "key",
        "hold_key",
        "type",
        "cursor_position",
        "mouse_move",
        "left_mouse_down",
        "left_mouse_up",
        "left_click",
        "left_click_drag",
        "right_click",
        "middle_click",
        "double_click",
        "triple_click",
        "scroll",
        "wait",
        "screenshot"
      ]),
      coordinate: z13.tuple([z13.number().int(), z13.number().int()]).optional(),
      duration: z13.number().optional(),
      scroll_amount: z13.number().optional(),
      scroll_direction: z13.enum(["up", "down", "left", "right"]).optional(),
      start_coordinate: z13.tuple([z13.number().int(), z13.number().int()]).optional(),
      text: z13.string().optional()
    })
  )
);
var computer_20250124 = createProviderToolFactory5({
  id: "anthropic.computer_20250124",
  inputSchema: computer_20250124InputSchema
});

// src/tool/memory_20250818.ts
import {
  createProviderToolFactory as createProviderToolFactory6,
  lazySchema as lazySchema13,
  zodSchema as zodSchema13
} from "@ai-sdk/provider-utils";
import { z as z14 } from "zod/v4";
var memory_20250818InputSchema = lazySchema13(
  () => zodSchema13(
    z14.discriminatedUnion("command", [
      z14.object({
        command: z14.literal("view"),
        path: z14.string(),
        view_range: z14.tuple([z14.number(), z14.number()]).optional()
      }),
      z14.object({
        command: z14.literal("create"),
        path: z14.string(),
        file_text: z14.string()
      }),
      z14.object({
        command: z14.literal("str_replace"),
        path: z14.string(),
        old_str: z14.string(),
        new_str: z14.string()
      }),
      z14.object({
        command: z14.literal("insert"),
        path: z14.string(),
        insert_line: z14.number(),
        insert_text: z14.string()
      }),
      z14.object({
        command: z14.literal("delete"),
        path: z14.string()
      }),
      z14.object({
        command: z14.literal("rename"),
        old_path: z14.string(),
        new_path: z14.string()
      })
    ])
  )
);
var memory_20250818 = createProviderToolFactory6({
  id: "anthropic.memory_20250818",
  inputSchema: memory_20250818InputSchema
});

// src/tool/text-editor_20241022.ts
import {
  createProviderToolFactory as createProviderToolFactory7,
  lazySchema as lazySchema14,
  zodSchema as zodSchema14
} from "@ai-sdk/provider-utils";
import { z as z15 } from "zod/v4";
var textEditor_20241022InputSchema = lazySchema14(
  () => zodSchema14(
    z15.object({
      command: z15.enum(["view", "create", "str_replace", "insert", "undo_edit"]),
      path: z15.string(),
      file_text: z15.string().optional(),
      insert_line: z15.number().int().optional(),
      new_str: z15.string().optional(),
      old_str: z15.string().optional(),
      view_range: z15.array(z15.number().int()).optional()
    })
  )
);
var textEditor_20241022 = createProviderToolFactory7({
  id: "anthropic.text_editor_20241022",
  inputSchema: textEditor_20241022InputSchema
});

// src/tool/text-editor_20250124.ts
import {
  createProviderToolFactory as createProviderToolFactory8,
  lazySchema as lazySchema15,
  zodSchema as zodSchema15
} from "@ai-sdk/provider-utils";
import { z as z16 } from "zod/v4";
var textEditor_20250124InputSchema = lazySchema15(
  () => zodSchema15(
    z16.object({
      command: z16.enum(["view", "create", "str_replace", "insert", "undo_edit"]),
      path: z16.string(),
      file_text: z16.string().optional(),
      insert_line: z16.number().int().optional(),
      new_str: z16.string().optional(),
      old_str: z16.string().optional(),
      view_range: z16.array(z16.number().int()).optional()
    })
  )
);
var textEditor_20250124 = createProviderToolFactory8({
  id: "anthropic.text_editor_20250124",
  inputSchema: textEditor_20250124InputSchema
});

// src/tool/text-editor_20250429.ts
import {
  createProviderToolFactory as createProviderToolFactory9,
  lazySchema as lazySchema16,
  zodSchema as zodSchema16
} from "@ai-sdk/provider-utils";
import { z as z17 } from "zod/v4";
var textEditor_20250429InputSchema = lazySchema16(
  () => zodSchema16(
    z17.object({
      command: z17.enum(["view", "create", "str_replace", "insert"]),
      path: z17.string(),
      file_text: z17.string().optional(),
      insert_line: z17.number().int().optional(),
      new_str: z17.string().optional(),
      old_str: z17.string().optional(),
      view_range: z17.array(z17.number().int()).optional()
    })
  )
);
var textEditor_20250429 = createProviderToolFactory9({
  id: "anthropic.text_editor_20250429",
  inputSchema: textEditor_20250429InputSchema
});

// src/tool/tool-search-bm25_20251119.ts
import {
  createProviderToolFactoryWithOutputSchema as createProviderToolFactoryWithOutputSchema6,
  lazySchema as lazySchema17,
  zodSchema as zodSchema17
} from "@ai-sdk/provider-utils";
import { z as z18 } from "zod/v4";
var toolSearchBm25_20251119OutputSchema = lazySchema17(
  () => zodSchema17(
    z18.array(
      z18.object({
        type: z18.literal("tool_reference"),
        toolName: z18.string()
      })
    )
  )
);
var toolSearchBm25_20251119InputSchema = lazySchema17(
  () => zodSchema17(
    z18.object({
      /**
       * A natural language query to search for tools.
       * Claude will use BM25 text search to find relevant tools.
       */
      query: z18.string(),
      /**
       * Maximum number of tools to return. Optional.
       */
      limit: z18.number().optional()
    })
  )
);
var factory7 = createProviderToolFactoryWithOutputSchema6({
  id: "anthropic.tool_search_bm25_20251119",
  inputSchema: toolSearchBm25_20251119InputSchema,
  outputSchema: toolSearchBm25_20251119OutputSchema
});
var toolSearchBm25_20251119 = (args = {}) => {
  return factory7(args);
};

// src/anthropic-tools.ts
var anthropicTools = {
  /**
   * The bash tool enables Claude to execute shell commands in a persistent bash session,
   * allowing system operations, script execution, and command-line automation.
   *
   * Image results are supported.
   */
  bash_20241022,
  /**
   * The bash tool enables Claude to execute shell commands in a persistent bash session,
   * allowing system operations, script execution, and command-line automation.
   *
   * Image results are supported.
   */
  bash_20250124,
  /**
   * Claude can analyze data, create visualizations, perform complex calculations,
   * run system commands, create and edit files, and process uploaded files directly within
   * the API conversation.
   *
   * The code execution tool allows Claude to run Bash commands and manipulate files,
   * including writing code, in a secure, sandboxed environment.
   */
  codeExecution_20250522,
  /**
   * Claude can analyze data, create visualizations, perform complex calculations,
   * run system commands, create and edit files, and process uploaded files directly within
   * the API conversation.
   *
   * The code execution tool allows Claude to run both Python and Bash commands and manipulate files,
   * including writing code, in a secure, sandboxed environment.
   *
   * This is the latest version with enhanced Bash support and file operations.
   */
  codeExecution_20250825,
  /**
   * Claude can interact with computer environments through the computer use tool, which
   * provides screenshot capabilities and mouse/keyboard control for autonomous desktop interaction.
   *
   * Image results are supported.
   *
   * @param displayWidthPx - The width of the display being controlled by the model in pixels.
   * @param displayHeightPx - The height of the display being controlled by the model in pixels.
   * @param displayNumber - The display number to control (only relevant for X11 environments). If specified, the tool will be provided a display number in the tool definition.
   */
  computer_20241022,
  /**
   * Claude can interact with computer environments through the computer use tool, which
   * provides screenshot capabilities and mouse/keyboard control for autonomous desktop interaction.
   *
   * Image results are supported.
   *
   * @param displayWidthPx - The width of the display being controlled by the model in pixels.
   * @param displayHeightPx - The height of the display being controlled by the model in pixels.
   * @param displayNumber - The display number to control (only relevant for X11 environments). If specified, the tool will be provided a display number in the tool definition.
   */
  computer_20250124,
  /**
   * The memory tool enables Claude to store and retrieve information across conversations through a memory file directory.
   * Claude can create, read, update, and delete files that persist between sessions,
   * allowing it to build knowledge over time without keeping everything in the context window.
   * The memory tool operates client-side—you control where and how the data is stored through your own infrastructure.
   *
   * Supported models: Claude Sonnet 4.5, Claude Sonnet 4, Claude Opus 4.1, Claude Opus 4.
   */
  memory_20250818,
  /**
   * Claude can use an Anthropic-defined text editor tool to view and modify text files,
   * helping you debug, fix, and improve your code or other text documents. This allows Claude
   * to directly interact with your files, providing hands-on assistance rather than just suggesting changes.
   *
   * Supported models: Claude Sonnet 3.5
   */
  textEditor_20241022,
  /**
   * Claude can use an Anthropic-defined text editor tool to view and modify text files,
   * helping you debug, fix, and improve your code or other text documents. This allows Claude
   * to directly interact with your files, providing hands-on assistance rather than just suggesting changes.
   *
   * Supported models: Claude Sonnet 3.7
   */
  textEditor_20250124,
  /**
   * Claude can use an Anthropic-defined text editor tool to view and modify text files,
   * helping you debug, fix, and improve your code or other text documents. This allows Claude
   * to directly interact with your files, providing hands-on assistance rather than just suggesting changes.
   *
   * Note: This version does not support the "undo_edit" command.
   *
   * @deprecated Use textEditor_20250728 instead
   */
  textEditor_20250429,
  /**
   * Claude can use an Anthropic-defined text editor tool to view and modify text files,
   * helping you debug, fix, and improve your code or other text documents. This allows Claude
   * to directly interact with your files, providing hands-on assistance rather than just suggesting changes.
   *
   * Note: This version does not support the "undo_edit" command and adds optional max_characters parameter.
   *
   * Supported models: Claude Sonnet 4, Opus 4, and Opus 4.1
   *
   * @param maxCharacters - Optional maximum number of characters to view in the file
   */
  textEditor_20250728,
  /**
   * Creates a web fetch tool that gives Claude direct access to real-time web content.
   *
   * @param maxUses - The max_uses parameter limits the number of web fetches performed
   * @param allowedDomains - Only fetch from these domains
   * @param blockedDomains - Never fetch from these domains
   * @param citations - Unlike web search where citations are always enabled, citations are optional for web fetch. Set "citations": {"enabled": true} to enable Claude to cite specific passages from fetched documents.
   * @param maxContentTokens - The max_content_tokens parameter limits the amount of content that will be included in the context.
   */
  webFetch_20250910,
  /**
   * Creates a web search tool that gives Claude direct access to real-time web content.
   *
   * @param maxUses - Maximum number of web searches Claude can perform during the conversation.
   * @param allowedDomains - Optional list of domains that Claude is allowed to search.
   * @param blockedDomains - Optional list of domains that Claude should avoid when searching.
   * @param userLocation - Optional user location information to provide geographically relevant search results.
   */
  webSearch_20250305,
  /**
   * Creates a tool search tool that uses regex patterns to find tools.
   *
   * The tool search tool enables Claude to work with hundreds or thousands of tools
   * by dynamically discovering and loading them on-demand. Instead of loading all
   * tool definitions into the context window upfront, Claude searches your tool
   * catalog and loads only the tools it needs.
   *
   * Use `providerOptions: { anthropic: { deferLoading: true } }` on other tools
   * to mark them for deferred loading.
   *
   * Supported models: Claude Opus 4.5, Claude Sonnet 4.5
   */
  toolSearchRegex_20251119,
  /**
   * Creates a tool search tool that uses BM25 (natural language) to find tools.
   *
   * The tool search tool enables Claude to work with hundreds or thousands of tools
   * by dynamically discovering and loading them on-demand. Instead of loading all
   * tool definitions into the context window upfront, Claude searches your tool
   * catalog and loads only the tools it needs.
   *
   * Use `providerOptions: { anthropic: { deferLoading: true } }` on other tools
   * to mark them for deferred loading.
   *
   * Supported models: Claude Opus 4.5, Claude Sonnet 4.5
   */
  toolSearchBm25_20251119
};

// src/anthropic-provider.ts
function createAnthropic(options = {}) {
  var _a, _b;
  const baseURL = (_a = withoutTrailingSlash(
    loadOptionalSetting({
      settingValue: options.baseURL,
      environmentVariableName: "ANTHROPIC_BASE_URL"
    })
  )) != null ? _a : "https://api.anthropic.com/v1";
  const providerName = (_b = options.name) != null ? _b : "anthropic.messages";
  const getHeaders = () => withUserAgentSuffix(
    {
      "anthropic-version": "2023-06-01",
      "x-api-key": loadApiKey({
        apiKey: options.apiKey,
        environmentVariableName: "ANTHROPIC_API_KEY",
        description: "Anthropic"
      }),
      ...options.headers
    },
    `ai-sdk/anthropic/${VERSION}`
  );
  const createChatModel = (modelId) => {
    var _a2;
    return new AnthropicMessagesLanguageModel(modelId, {
      provider: providerName,
      baseURL,
      headers: getHeaders,
      fetch: options.fetch,
      generateId: (_a2 = options.generateId) != null ? _a2 : generateId2,
      supportedUrls: () => ({
        "image/*": [/^https?:\/\/.*$/]
      })
    });
  };
  const provider = function(modelId) {
    if (new.target) {
      throw new Error(
        "The Anthropic model function cannot be called with the new keyword."
      );
    }
    return createChatModel(modelId);
  };
  provider.specificationVersion = "v3";
  provider.languageModel = createChatModel;
  provider.chat = createChatModel;
  provider.messages = createChatModel;
  provider.embeddingModel = (modelId) => {
    throw new NoSuchModelError({ modelId, modelType: "embeddingModel" });
  };
  provider.textEmbeddingModel = provider.embeddingModel;
  provider.imageModel = (modelId) => {
    throw new NoSuchModelError({ modelId, modelType: "imageModel" });
  };
  provider.tools = anthropicTools;
  return provider;
}
var anthropic = createAnthropic();

// src/forward-anthropic-container-id-from-last-step.ts
function forwardAnthropicContainerIdFromLastStep({
  steps
}) {
  var _a, _b, _c;
  for (let i = steps.length - 1; i >= 0; i--) {
    const containerId = (_c = (_b = (_a = steps[i].providerMetadata) == null ? void 0 : _a.anthropic) == null ? void 0 : _b.container) == null ? void 0 : _c.id;
    if (containerId) {
      return {
        providerOptions: {
          anthropic: {
            container: { id: containerId }
          }
        }
      };
    }
  }
  return void 0;
}
export {
  VERSION,
  anthropic,
  createAnthropic,
  forwardAnthropicContainerIdFromLastStep
};
//# sourceMappingURL=index.mjs.map