'use strict';

var middlewareHostHeader = require('@aws-sdk/middleware-host-header');
var middlewareLogger = require('@aws-sdk/middleware-logger');
var middlewareRecursionDetection = require('@aws-sdk/middleware-recursion-detection');
var middlewareUserAgent = require('@aws-sdk/middleware-user-agent');
var configResolver = require('@smithy/config-resolver');
var core = require('@smithy/core');
var schema = require('@smithy/core/schema');
var middlewareContentLength = require('@smithy/middleware-content-length');
var middlewareEndpoint = require('@smithy/middleware-endpoint');
var middlewareRetry = require('@smithy/middleware-retry');
var smithyClient = require('@smithy/smithy-client');
var httpAuthSchemeProvider = require('./auth/httpAuthSchemeProvider');
var runtimeConfig = require('./runtimeConfig');
var regionConfigResolver = require('@aws-sdk/region-config-resolver');
var protocolHttp = require('@smithy/protocol-http');

const resolveClientEndpointParameters = (options) => {
    return Object.assign(options, {
        useDualstackEndpoint: options.useDualstackEndpoint ?? false,
        useFipsEndpoint: options.useFipsEndpoint ?? false,
        defaultSigningName: "cognito-idp",
    });
};
const commonParams = {
    UseFIPS: { type: "builtInParams", name: "useFipsEndpoint" },
    Endpoint: { type: "builtInParams", name: "endpoint" },
    Region: { type: "builtInParams", name: "region" },
    UseDualStack: { type: "builtInParams", name: "useDualstackEndpoint" },
};

const getHttpAuthExtensionConfiguration = (runtimeConfig) => {
    const _httpAuthSchemes = runtimeConfig.httpAuthSchemes;
    let _httpAuthSchemeProvider = runtimeConfig.httpAuthSchemeProvider;
    let _credentials = runtimeConfig.credentials;
    return {
        setHttpAuthScheme(httpAuthScheme) {
            const index = _httpAuthSchemes.findIndex((scheme) => scheme.schemeId === httpAuthScheme.schemeId);
            if (index === -1) {
                _httpAuthSchemes.push(httpAuthScheme);
            }
            else {
                _httpAuthSchemes.splice(index, 1, httpAuthScheme);
            }
        },
        httpAuthSchemes() {
            return _httpAuthSchemes;
        },
        setHttpAuthSchemeProvider(httpAuthSchemeProvider) {
            _httpAuthSchemeProvider = httpAuthSchemeProvider;
        },
        httpAuthSchemeProvider() {
            return _httpAuthSchemeProvider;
        },
        setCredentials(credentials) {
            _credentials = credentials;
        },
        credentials() {
            return _credentials;
        },
    };
};
const resolveHttpAuthRuntimeConfig = (config) => {
    return {
        httpAuthSchemes: config.httpAuthSchemes(),
        httpAuthSchemeProvider: config.httpAuthSchemeProvider(),
        credentials: config.credentials(),
    };
};

const resolveRuntimeExtensions = (runtimeConfig, extensions) => {
    const extensionConfiguration = Object.assign(regionConfigResolver.getAwsRegionExtensionConfiguration(runtimeConfig), smithyClient.getDefaultExtensionConfiguration(runtimeConfig), protocolHttp.getHttpHandlerExtensionConfiguration(runtimeConfig), getHttpAuthExtensionConfiguration(runtimeConfig));
    extensions.forEach((extension) => extension.configure(extensionConfiguration));
    return Object.assign(runtimeConfig, regionConfigResolver.resolveAwsRegionExtensionConfiguration(extensionConfiguration), smithyClient.resolveDefaultRuntimeConfig(extensionConfiguration), protocolHttp.resolveHttpHandlerRuntimeConfig(extensionConfiguration), resolveHttpAuthRuntimeConfig(extensionConfiguration));
};

class CognitoIdentityProviderClient extends smithyClient.Client {
    config;
    constructor(...[configuration]) {
        const _config_0 = runtimeConfig.getRuntimeConfig(configuration || {});
        super(_config_0);
        this.initConfig = _config_0;
        const _config_1 = resolveClientEndpointParameters(_config_0);
        const _config_2 = middlewareUserAgent.resolveUserAgentConfig(_config_1);
        const _config_3 = middlewareRetry.resolveRetryConfig(_config_2);
        const _config_4 = configResolver.resolveRegionConfig(_config_3);
        const _config_5 = middlewareHostHeader.resolveHostHeaderConfig(_config_4);
        const _config_6 = middlewareEndpoint.resolveEndpointConfig(_config_5);
        const _config_7 = httpAuthSchemeProvider.resolveHttpAuthSchemeConfig(_config_6);
        const _config_8 = resolveRuntimeExtensions(_config_7, configuration?.extensions || []);
        this.config = _config_8;
        this.middlewareStack.use(schema.getSchemaSerdePlugin(this.config));
        this.middlewareStack.use(middlewareUserAgent.getUserAgentPlugin(this.config));
        this.middlewareStack.use(middlewareRetry.getRetryPlugin(this.config));
        this.middlewareStack.use(middlewareContentLength.getContentLengthPlugin(this.config));
        this.middlewareStack.use(middlewareHostHeader.getHostHeaderPlugin(this.config));
        this.middlewareStack.use(middlewareLogger.getLoggerPlugin(this.config));
        this.middlewareStack.use(middlewareRecursionDetection.getRecursionDetectionPlugin(this.config));
        this.middlewareStack.use(core.getHttpAuthSchemeEndpointRuleSetPlugin(this.config, {
            httpAuthSchemeParametersProvider: httpAuthSchemeProvider.defaultCognitoIdentityProviderHttpAuthSchemeParametersProvider,
            identityProviderConfigProvider: async (config) => new core.DefaultIdentityProviderConfig({
                "aws.auth#sigv4": config.credentials,
            }),
        }));
        this.middlewareStack.use(core.getHttpSigningPlugin(this.config));
    }
    destroy() {
        super.destroy();
    }
}

class CognitoIdentityProviderServiceException extends smithyClient.ServiceException {
    constructor(options) {
        super(options);
        Object.setPrototypeOf(this, CognitoIdentityProviderServiceException.prototype);
    }
}

class InternalErrorException extends CognitoIdentityProviderServiceException {
    name = "InternalErrorException";
    $fault = "server";
    constructor(opts) {
        super({
            name: "InternalErrorException",
            $fault: "server",
            ...opts,
        });
        Object.setPrototypeOf(this, InternalErrorException.prototype);
    }
}
class InvalidParameterException extends CognitoIdentityProviderServiceException {
    name = "InvalidParameterException";
    $fault = "client";
    reasonCode;
    constructor(opts) {
        super({
            name: "InvalidParameterException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidParameterException.prototype);
        this.reasonCode = opts.reasonCode;
    }
}
class NotAuthorizedException extends CognitoIdentityProviderServiceException {
    name = "NotAuthorizedException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "NotAuthorizedException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NotAuthorizedException.prototype);
    }
}
class ResourceNotFoundException extends CognitoIdentityProviderServiceException {
    name = "ResourceNotFoundException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ResourceNotFoundException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ResourceNotFoundException.prototype);
    }
}
class TooManyRequestsException extends CognitoIdentityProviderServiceException {
    name = "TooManyRequestsException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "TooManyRequestsException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyRequestsException.prototype);
    }
}
class UserImportInProgressException extends CognitoIdentityProviderServiceException {
    name = "UserImportInProgressException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "UserImportInProgressException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, UserImportInProgressException.prototype);
    }
}
class UserNotFoundException extends CognitoIdentityProviderServiceException {
    name = "UserNotFoundException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "UserNotFoundException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, UserNotFoundException.prototype);
    }
}
class InvalidLambdaResponseException extends CognitoIdentityProviderServiceException {
    name = "InvalidLambdaResponseException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "InvalidLambdaResponseException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidLambdaResponseException.prototype);
    }
}
class LimitExceededException extends CognitoIdentityProviderServiceException {
    name = "LimitExceededException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "LimitExceededException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, LimitExceededException.prototype);
    }
}
class TooManyFailedAttemptsException extends CognitoIdentityProviderServiceException {
    name = "TooManyFailedAttemptsException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "TooManyFailedAttemptsException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyFailedAttemptsException.prototype);
    }
}
class UnexpectedLambdaException extends CognitoIdentityProviderServiceException {
    name = "UnexpectedLambdaException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "UnexpectedLambdaException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, UnexpectedLambdaException.prototype);
    }
}
class UserLambdaValidationException extends CognitoIdentityProviderServiceException {
    name = "UserLambdaValidationException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "UserLambdaValidationException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, UserLambdaValidationException.prototype);
    }
}
class CodeDeliveryFailureException extends CognitoIdentityProviderServiceException {
    name = "CodeDeliveryFailureException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "CodeDeliveryFailureException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, CodeDeliveryFailureException.prototype);
    }
}
class InvalidPasswordException extends CognitoIdentityProviderServiceException {
    name = "InvalidPasswordException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "InvalidPasswordException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidPasswordException.prototype);
    }
}
class InvalidSmsRoleAccessPolicyException extends CognitoIdentityProviderServiceException {
    name = "InvalidSmsRoleAccessPolicyException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "InvalidSmsRoleAccessPolicyException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidSmsRoleAccessPolicyException.prototype);
    }
}
class InvalidSmsRoleTrustRelationshipException extends CognitoIdentityProviderServiceException {
    name = "InvalidSmsRoleTrustRelationshipException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "InvalidSmsRoleTrustRelationshipException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidSmsRoleTrustRelationshipException.prototype);
    }
}
class PreconditionNotMetException extends CognitoIdentityProviderServiceException {
    name = "PreconditionNotMetException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "PreconditionNotMetException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, PreconditionNotMetException.prototype);
    }
}
class UnsupportedUserStateException extends CognitoIdentityProviderServiceException {
    name = "UnsupportedUserStateException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "UnsupportedUserStateException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, UnsupportedUserStateException.prototype);
    }
}
class UsernameExistsException extends CognitoIdentityProviderServiceException {
    name = "UsernameExistsException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "UsernameExistsException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, UsernameExistsException.prototype);
    }
}
class AliasExistsException extends CognitoIdentityProviderServiceException {
    name = "AliasExistsException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "AliasExistsException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, AliasExistsException.prototype);
    }
}
class InvalidUserPoolConfigurationException extends CognitoIdentityProviderServiceException {
    name = "InvalidUserPoolConfigurationException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "InvalidUserPoolConfigurationException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidUserPoolConfigurationException.prototype);
    }
}
class InvalidEmailRoleAccessPolicyException extends CognitoIdentityProviderServiceException {
    name = "InvalidEmailRoleAccessPolicyException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "InvalidEmailRoleAccessPolicyException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidEmailRoleAccessPolicyException.prototype);
    }
}
class MFAMethodNotFoundException extends CognitoIdentityProviderServiceException {
    name = "MFAMethodNotFoundException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "MFAMethodNotFoundException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, MFAMethodNotFoundException.prototype);
    }
}
class PasswordResetRequiredException extends CognitoIdentityProviderServiceException {
    name = "PasswordResetRequiredException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "PasswordResetRequiredException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, PasswordResetRequiredException.prototype);
    }
}
class UnsupportedOperationException extends CognitoIdentityProviderServiceException {
    name = "UnsupportedOperationException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "UnsupportedOperationException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, UnsupportedOperationException.prototype);
    }
}
class UserNotConfirmedException extends CognitoIdentityProviderServiceException {
    name = "UserNotConfirmedException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "UserNotConfirmedException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, UserNotConfirmedException.prototype);
    }
}
class UserPoolAddOnNotEnabledException extends CognitoIdentityProviderServiceException {
    name = "UserPoolAddOnNotEnabledException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "UserPoolAddOnNotEnabledException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, UserPoolAddOnNotEnabledException.prototype);
    }
}
class CodeMismatchException extends CognitoIdentityProviderServiceException {
    name = "CodeMismatchException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "CodeMismatchException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, CodeMismatchException.prototype);
    }
}
class ExpiredCodeException extends CognitoIdentityProviderServiceException {
    name = "ExpiredCodeException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ExpiredCodeException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ExpiredCodeException.prototype);
    }
}
class PasswordHistoryPolicyViolationException extends CognitoIdentityProviderServiceException {
    name = "PasswordHistoryPolicyViolationException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "PasswordHistoryPolicyViolationException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, PasswordHistoryPolicyViolationException.prototype);
    }
}
class SoftwareTokenMFANotFoundException extends CognitoIdentityProviderServiceException {
    name = "SoftwareTokenMFANotFoundException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "SoftwareTokenMFANotFoundException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, SoftwareTokenMFANotFoundException.prototype);
    }
}
class ConcurrentModificationException extends CognitoIdentityProviderServiceException {
    name = "ConcurrentModificationException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ConcurrentModificationException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ConcurrentModificationException.prototype);
    }
}
class ForbiddenException extends CognitoIdentityProviderServiceException {
    name = "ForbiddenException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ForbiddenException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ForbiddenException.prototype);
    }
}
class WebAuthnChallengeNotFoundException extends CognitoIdentityProviderServiceException {
    name = "WebAuthnChallengeNotFoundException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "WebAuthnChallengeNotFoundException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, WebAuthnChallengeNotFoundException.prototype);
    }
}
class WebAuthnClientMismatchException extends CognitoIdentityProviderServiceException {
    name = "WebAuthnClientMismatchException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "WebAuthnClientMismatchException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, WebAuthnClientMismatchException.prototype);
    }
}
class WebAuthnCredentialNotSupportedException extends CognitoIdentityProviderServiceException {
    name = "WebAuthnCredentialNotSupportedException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "WebAuthnCredentialNotSupportedException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, WebAuthnCredentialNotSupportedException.prototype);
    }
}
class WebAuthnNotEnabledException extends CognitoIdentityProviderServiceException {
    name = "WebAuthnNotEnabledException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "WebAuthnNotEnabledException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, WebAuthnNotEnabledException.prototype);
    }
}
class WebAuthnOriginNotAllowedException extends CognitoIdentityProviderServiceException {
    name = "WebAuthnOriginNotAllowedException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "WebAuthnOriginNotAllowedException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, WebAuthnOriginNotAllowedException.prototype);
    }
}
class WebAuthnRelyingPartyMismatchException extends CognitoIdentityProviderServiceException {
    name = "WebAuthnRelyingPartyMismatchException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "WebAuthnRelyingPartyMismatchException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, WebAuthnRelyingPartyMismatchException.prototype);
    }
}
class DeviceKeyExistsException extends CognitoIdentityProviderServiceException {
    name = "DeviceKeyExistsException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "DeviceKeyExistsException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, DeviceKeyExistsException.prototype);
    }
}
class GroupExistsException extends CognitoIdentityProviderServiceException {
    name = "GroupExistsException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "GroupExistsException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, GroupExistsException.prototype);
    }
}
class DuplicateProviderException extends CognitoIdentityProviderServiceException {
    name = "DuplicateProviderException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "DuplicateProviderException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, DuplicateProviderException.prototype);
    }
}
class ManagedLoginBrandingExistsException extends CognitoIdentityProviderServiceException {
    name = "ManagedLoginBrandingExistsException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ManagedLoginBrandingExistsException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ManagedLoginBrandingExistsException.prototype);
    }
}
class TermsExistsException extends CognitoIdentityProviderServiceException {
    name = "TermsExistsException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "TermsExistsException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TermsExistsException.prototype);
    }
}
class FeatureUnavailableInTierException extends CognitoIdentityProviderServiceException {
    name = "FeatureUnavailableInTierException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "FeatureUnavailableInTierException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, FeatureUnavailableInTierException.prototype);
    }
}
class TierChangeNotAllowedException extends CognitoIdentityProviderServiceException {
    name = "TierChangeNotAllowedException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "TierChangeNotAllowedException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TierChangeNotAllowedException.prototype);
    }
}
class UserPoolTaggingException extends CognitoIdentityProviderServiceException {
    name = "UserPoolTaggingException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "UserPoolTaggingException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, UserPoolTaggingException.prototype);
    }
}
class InvalidOAuthFlowException extends CognitoIdentityProviderServiceException {
    name = "InvalidOAuthFlowException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "InvalidOAuthFlowException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidOAuthFlowException.prototype);
    }
}
class ScopeDoesNotExistException extends CognitoIdentityProviderServiceException {
    name = "ScopeDoesNotExistException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ScopeDoesNotExistException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ScopeDoesNotExistException.prototype);
    }
}
class UnsupportedIdentityProviderException extends CognitoIdentityProviderServiceException {
    name = "UnsupportedIdentityProviderException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "UnsupportedIdentityProviderException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, UnsupportedIdentityProviderException.prototype);
    }
}
class RefreshTokenReuseException extends CognitoIdentityProviderServiceException {
    name = "RefreshTokenReuseException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "RefreshTokenReuseException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, RefreshTokenReuseException.prototype);
    }
}
class UnauthorizedException extends CognitoIdentityProviderServiceException {
    name = "UnauthorizedException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "UnauthorizedException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, UnauthorizedException.prototype);
    }
}
class UnsupportedTokenTypeException extends CognitoIdentityProviderServiceException {
    name = "UnsupportedTokenTypeException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "UnsupportedTokenTypeException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, UnsupportedTokenTypeException.prototype);
    }
}
class WebAuthnConfigurationMissingException extends CognitoIdentityProviderServiceException {
    name = "WebAuthnConfigurationMissingException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "WebAuthnConfigurationMissingException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, WebAuthnConfigurationMissingException.prototype);
    }
}
class EnableSoftwareTokenMFAException extends CognitoIdentityProviderServiceException {
    name = "EnableSoftwareTokenMFAException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "EnableSoftwareTokenMFAException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, EnableSoftwareTokenMFAException.prototype);
    }
}

const _A = "Actions";
const _AA = "ApplicationArn";
const _AACUO = "AllowAdminCreateUserOnly";
const _AAUTG = "AdminAddUserToGroup";
const _AAUTGR = "AdminAddUserToGroupRequest";
const _AAl = "AliasAttributes";
const _AAu = "AuthenticatorAttachment";
const _AC = "AvailableChallenges";
const _ACA = "AddCustomAttributes";
const _ACAR = "AddCustomAttributesRequest";
const _ACARd = "AddCustomAttributesResponse";
const _ACSU = "AdminConfirmSignUp";
const _ACSUR = "AdminConfirmSignUpRequest";
const _ACSURd = "AdminConfirmSignUpResponse";
const _ACT = "AnalyticsConfigurationType";
const _ACU = "AdminCreateUser";
const _ACUC = "AdminCreateUserConfig";
const _ACUCT = "AdminCreateUserConfigType";
const _ACUR = "AdminCreateUserRequest";
const _ACURd = "AdminCreateUserResponse";
const _ACn = "AnalyticsConfiguration";
const _ADPFU = "AdminDisableProviderForUser";
const _ADPFUR = "AdminDisableProviderForUserRequest";
const _ADPFURd = "AdminDisableProviderForUserResponse";
const _ADT = "AttributeDataType";
const _ADU = "AdminDeleteUser";
const _ADUA = "AdminDeleteUserAttributes";
const _ADUAR = "AdminDeleteUserAttributesRequest";
const _ADUARd = "AdminDeleteUserAttributesResponse";
const _ADUR = "AdminDeleteUserRequest";
const _ADURd = "AdminDisableUserRequest";
const _ADURdm = "AdminDisableUserResponse";
const _ADUd = "AdminDisableUser";
const _AE = "AuthEvents";
const _AEE = "AliasExistsException";
const _AEI = "AnalyticsEndpointId";
const _AET = "AuthEventType";
const _AETu = "AuthEventsType";
const _AEU = "AdminEnableUser";
const _AEUR = "AdminEnableUserRequest";
const _AEURd = "AdminEnableUserResponse";
const _AF = "AuthFlow";
const _AFAF = "AllowedFirstAuthFactors";
const _AFD = "AdminForgetDevice";
const _AFDR = "AdminForgetDeviceRequest";
const _AGD = "AdminGetDevice";
const _AGDR = "AdminGetDeviceRequest";
const _AGDRd = "AdminGetDeviceResponse";
const _AGU = "AdminGetUser";
const _AGUR = "AdminGetUserRequest";
const _AGURd = "AdminGetUserResponse";
const _AI = "ApplicationId";
const _AIA = "AdminInitiateAuth";
const _AIAR = "AdminInitiateAuthRequest";
const _AIARd = "AdminInitiateAuthResponse";
const _ALD = "AdminListDevices";
const _ALDR = "AdminListDevicesRequest";
const _ALDRd = "AdminListDevicesResponse";
const _ALGFU = "AdminListGroupsForUser";
const _ALGFUR = "AdminListGroupsForUserRequest";
const _ALGFURd = "AdminListGroupsForUserResponse";
const _ALPFU = "AdminLinkProviderForUser";
const _ALPFUR = "AdminLinkProviderForUserRequest";
const _ALPFURd = "AdminLinkProviderForUserResponse";
const _ALT = "AssetListType";
const _ALTt = "AttributeListType";
const _ALUAE = "AdminListUserAuthEvents";
const _ALUAER = "AdminListUserAuthEventsRequest";
const _ALUAERd = "AdminListUserAuthEventsResponse";
const _AM = "AnalyticsMetadata";
const _AMT = "AnalyticsMetadataType";
const _AMt = "AttributeMapping";
const _AN = "AttributeName";
const _AOAF = "AllowedOAuthFlows";
const _AOAFUPC = "AllowedOAuthFlowsUserPoolClient";
const _AOAS = "AllowedOAuthScopes";
const _AP = "AuthParameters";
const _APT = "AuthParametersType";
const _AR = "AuthenticationResult";
const _ARS = "AccountRecoverySetting";
const _ARST = "AccountRecoverySettingType";
const _ART = "AuthenticationResultType";
const _ARTAC = "AdminRespondToAuthChallenge";
const _ARTACR = "AdminRespondToAuthChallengeRequest";
const _ARTACRd = "AdminRespondToAuthChallengeResponse";
const _ARUFG = "AdminRemoveUserFromGroup";
const _ARUFGR = "AdminRemoveUserFromGroupRequest";
const _ARUP = "AdminResetUserPassword";
const _ARUPR = "AdminResetUserPasswordRequest";
const _ARUPRd = "AdminResetUserPasswordResponse";
const _ARVBU = "AttributesRequireVerificationBeforeUpdate";
const _ASAF = "AdvancedSecurityAdditionalFlows";
const _ASAFT = "AdvancedSecurityAdditionalFlowsType";
const _ASM = "AdvancedSecurityMode";
const _AST = "AssociateSoftwareToken";
const _ASTR = "AssociateSoftwareTokenRequest";
const _ASTRs = "AssociateSoftwareTokenResponse";
const _ASUMFAP = "AdminSetUserMFAPreference";
const _ASUMFAPR = "AdminSetUserMFAPreferenceRequest";
const _ASUMFAPRd = "AdminSetUserMFAPreferenceResponse";
const _ASUP = "AdminSetUserPassword";
const _ASUPR = "AdminSetUserPasswordRequest";
const _ASUPRd = "AdminSetUserPasswordResponse";
const _ASUS = "AdminSetUserSettings";
const _ASUSR = "AdminSetUserSettingsRequest";
const _ASUSRd = "AdminSetUserSettingsResponse";
const _ASV = "AuthSessionValidity";
const _AT = "AssetType";
const _ATAT = "AccountTakeoverActionsType";
const _ATATc = "AccountTakeoverActionType";
const _ATG = "AttributesToGet";
const _ATRC = "AccountTakeoverRiskConfiguration";
const _ATRCT = "AccountTakeoverRiskConfigurationType";
const _ATV = "AccessTokenValidity";
const _ATc = "AccessToken";
const _ATt = "AttributeType";
const _ATu = "AuthenticatorTransports";
const _AUAEF = "AdminUpdateAuthEventFeedback";
const _AUAEFR = "AdminUpdateAuthEventFeedbackRequest";
const _AUAEFRd = "AdminUpdateAuthEventFeedbackResponse";
const _AUDS = "AdminUpdateDeviceStatus";
const _AUDSR = "AdminUpdateDeviceStatusRequest";
const _AUDSRd = "AdminUpdateDeviceStatusResponse";
const _AUGSO = "AdminUserGlobalSignOut";
const _AUGSOR = "AdminUserGlobalSignOutRequest";
const _AUGSORd = "AdminUserGlobalSignOutResponse";
const _AUUA = "AdminUpdateUserAttributes";
const _AUUAR = "AdminUpdateUserAttributesRequest";
const _AUUARd = "AdminUpdateUserAttributesResponse";
const _AVA = "AutoVerifiedAttributes";
const _AVT = "AttributeValueType";
const _AWSAI = "AWSAccountId";
const _Ar = "Arn";
const _As = "Assets";
const _At = "Attributes";
const _B = "Bytes";
const _BA = "BucketArn";
const _BE = "BlockEmail";
const _BIPRL = "BlockedIPRangeList";
const _C = "Category";
const _CA = "CustomAttributes";
const _CAC = "CreateAuthChallenge";
const _CALT = "CustomAttributesListType";
const _CAM = "CustomAuthMode";
const _CAe = "CertificateArn";
const _CAr = "CreatedAt";
const _CC = "ConfirmationCode";
const _CCAT = "CompromisedCredentialsActionsType";
const _CCD = "CompromisedCredentialsDetected";
const _CCO = "CredentialCreationOptions";
const _CCRC = "CompromisedCredentialsRiskConfiguration";
const _CCRCT = "CompromisedCredentialsRiskConfigurationType";
const _CD = "ContextData";
const _CDC = "CustomDomainConfig";
const _CDCT = "CustomDomainConfigType";
const _CDD = "CodeDeliveryDetails";
const _CDDL = "CodeDeliveryDetailsList";
const _CDDLT = "CodeDeliveryDetailsListType";
const _CDDT = "CodeDeliveryDetailsType";
const _CDFE = "CodeDeliveryFailureException";
const _CDR = "ConfirmDeviceRequest";
const _CDRo = "ConfirmDeviceResponse";
const _CDT = "ContextDataType";
const _CDo = "CompletionDate";
const _CDon = "ConfirmDevice";
const _CDr = "CreationDate";
const _CDu = "CustomDomain";
const _CELVCT = "CustomEmailLambdaVersionConfigType";
const _CES = "CustomEmailSender";
const _CFD = "CloudFrontDomain";
const _CFDl = "CloudFrontDistribution";
const _CFP = "ConfirmForgotPassword";
const _CFPR = "ConfirmForgotPasswordRequest";
const _CFPRo = "ConfirmForgotPasswordResponse";
const _CG = "CreateGroup";
const _CGR = "CreateGroupRequest";
const _CGRr = "CreateGroupResponse";
const _CI = "ClientId";
const _CIP = "CreateIdentityProvider";
const _CIPR = "CreateIdentityProviderRequest";
const _CIPRr = "CreateIdentityProviderResponse";
const _CIT = "ClientIdType";
const _CIr = "CredentialId";
const _CM = "ClientMetadata";
const _CME = "CodeMismatchException";
const _CMEo = "ConcurrentModificationException";
const _CMLB = "CreateManagedLoginBranding";
const _CMLBR = "CreateManagedLoginBrandingRequest";
const _CMLBRr = "CreateManagedLoginBrandingResponse";
const _CMo = "ColorMode";
const _CMom = "CompletionMessage";
const _CMu = "CustomMessage";
const _CN = "ChallengeName";
const _CNl = "ClientName";
const _CP = "ChallengeParameters";
const _CPR = "ChangePasswordRequest";
const _CPRh = "ChangePasswordResponse";
const _CPh = "ChangePassword";
const _CR = "ChallengeResponses";
const _CRLT = "ChallengeResponseListType";
const _CROND = "ChallengeRequiredOnNewDevice";
const _CRS = "CreateResourceServer";
const _CRSR = "CreateResourceServerRequest";
const _CRSRr = "CreateResourceServerResponse";
const _CRT = "ChallengeResponseType";
const _CRTh = "ChallengeResponsesType";
const _CRh = "ChallengeResponse";
const _CS = "ConfigurationSet";
const _CSMSLVCT = "CustomSMSLambdaVersionConfigType";
const _CSMSS = "CustomSMSSender";
const _CSS = "CSS";
const _CSSV = "CSSVersion";
const _CST = "ClientSecretType";
const _CSU = "ConfirmSignUp";
const _CSUR = "ConfirmSignUpRequest";
const _CSURo = "ConfirmSignUpResponse";
const _CSVH = "CSVHeader";
const _CSa = "CaseSensitive";
const _CSl = "ClientSecret";
const _CT = "CreateTerms";
const _CTR = "CreateTermsRequest";
const _CTRr = "CreateTermsResponse";
const _CUAF = "ConfiguredUserAuthFactors";
const _CUIJ = "CreateUserImportJob";
const _CUIJR = "CreateUserImportJobRequest";
const _CUIJRr = "CreateUserImportJobResponse";
const _CUP = "CreateUserPool";
const _CUPC = "CreateUserPoolClient";
const _CUPCR = "CreateUserPoolClientRequest";
const _CUPCRr = "CreateUserPoolClientResponse";
const _CUPD = "CreateUserPoolDomain";
const _CUPDR = "CreateUserPoolDomainRequest";
const _CUPDRr = "CreateUserPoolDomainResponse";
const _CUPR = "CreateUserPoolRequest";
const _CUPRr = "CreateUserPoolResponse";
const _CURL = "CallbackURLs";
const _CWAR = "CompleteWebAuthnRegistration";
const _CWARR = "CompleteWebAuthnRegistrationRequest";
const _CWARRo = "CompleteWebAuthnRegistrationResponse";
const _CWLC = "CloudWatchLogsConfiguration";
const _CWLCT = "CloudWatchLogsConfigurationType";
const _CWLRA = "CloudWatchLogsRoleArn";
const _Ce = "Certificate";
const _Ci = "City";
const _Co = "Country";
const _Cod = "Code";
const _Cr = "Credential";
const _Cre = "Credentials";
const _D = "Device";
const _DA = "DeviceAttributes";
const _DAC = "DefineAuthChallenge";
const _DC = "DeviceConfiguration";
const _DCD = "DeviceCreateDate";
const _DCT = "DeviceConfigurationType";
const _DD = "DomainDescription";
const _DDM = "DesiredDeliveryMediums";
const _DDT = "DomainDescriptionType";
const _DEO = "DefaultEmailOption";
const _DG = "DeleteGroup";
const _DGK = "DeviceGroupKey";
const _DGR = "DeleteGroupRequest";
const _DIP = "DeleteIdentityProvider";
const _DIPR = "DeleteIdentityProviderRequest";
const _DIPRe = "DescribeIdentityProviderRequest";
const _DIPRes = "DescribeIdentityProviderResponse";
const _DIPe = "DescribeIdentityProvider";
const _DK = "DeviceKey";
const _DKEE = "DeviceKeyExistsException";
const _DLAD = "DeviceLastAuthenticatedDate";
const _DLMD = "DeviceLastModifiedDate";
const _DLT = "DeviceListType";
const _DM = "DeliveryMedium";
const _DMLB = "DeleteManagedLoginBranding";
const _DMLBBC = "DescribeManagedLoginBrandingByClient";
const _DMLBBCR = "DescribeManagedLoginBrandingByClientRequest";
const _DMLBBCRe = "DescribeManagedLoginBrandingByClientResponse";
const _DMLBR = "DeleteManagedLoginBrandingRequest";
const _DMLBRe = "DescribeManagedLoginBrandingRequest";
const _DMLBRes = "DescribeManagedLoginBrandingResponse";
const _DMLBe = "DescribeManagedLoginBranding";
const _DN = "DeviceName";
const _DOA = "DeveloperOnlyAttribute";
const _DOROUP = "DeviceOnlyRememberedOnUserPrompt";
const _DP = "DeletionProtection";
const _DPE = "DuplicateProviderException";
const _DRC = "DescribeRiskConfiguration";
const _DRCR = "DescribeRiskConfigurationRequest";
const _DRCRe = "DescribeRiskConfigurationResponse";
const _DRS = "DeviceRememberedStatus";
const _DRSR = "DeleteResourceServerRequest";
const _DRSRe = "DescribeResourceServerRequest";
const _DRSRes = "DescribeResourceServerResponse";
const _DRSe = "DeleteResourceServer";
const _DRSes = "DescribeResourceServer";
const _DRURI = "DefaultRedirectURI";
const _DSVC = "DeviceSecretVerifierConfig";
const _DSVCT = "DeviceSecretVerifierConfigType";
const _DT = "DeviceType";
const _DTR = "DeleteTermsRequest";
const _DTRe = "DescribeTermsRequest";
const _DTRes = "DescribeTermsResponse";
const _DTe = "DeleteTerms";
const _DTes = "DescribeTerms";
const _DU = "DestinationUser";
const _DUA = "DeleteUserAttributes";
const _DUAR = "DeleteUserAttributesRequest";
const _DUARe = "DeleteUserAttributesResponse";
const _DUIJ = "DescribeUserImportJob";
const _DUIJR = "DescribeUserImportJobRequest";
const _DUIJRe = "DescribeUserImportJobResponse";
const _DUP = "DeleteUserPool";
const _DUPC = "DeleteUserPoolClient";
const _DUPCR = "DeleteUserPoolClientRequest";
const _DUPCRe = "DescribeUserPoolClientRequest";
const _DUPCRes = "DescribeUserPoolClientResponse";
const _DUPCe = "DescribeUserPoolClient";
const _DUPD = "DeleteUserPoolDomain";
const _DUPDR = "DeleteUserPoolDomainRequest";
const _DUPDRe = "DeleteUserPoolDomainResponse";
const _DUPDRes = "DescribeUserPoolDomainRequest";
const _DUPDResc = "DescribeUserPoolDomainResponse";
const _DUPDe = "DescribeUserPoolDomain";
const _DUPR = "DeleteUserPoolRequest";
const _DUPRe = "DescribeUserPoolRequest";
const _DUPRes = "DescribeUserPoolResponse";
const _DUPe = "DescribeUserPool";
const _DUR = "DeleteUserRequest";
const _DUe = "DeleteUser";
const _DWAC = "DeleteWebAuthnCredential";
const _DWACR = "DeleteWebAuthnCredentialRequest";
const _DWACRe = "DeleteWebAuthnCredentialResponse";
const _De = "Devices";
const _Des = "Destination";
const _Desc = "Description";
const _Do = "Domain";
const _E = "Enabled";
const _EA = "EventAction";
const _EAF = "ExplicitAuthFlows";
const _EC = "EmailConfiguration";
const _ECD = "EventContextData";
const _ECDT = "EventContextDataType";
const _ECE = "ExpiredCodeException";
const _ECF = "EmailConfigurationFailure";
const _ECT = "EmailConfigurationType";
const _ED = "EncodedData";
const _EF = "EventFeedback";
const _EFT = "EventFeedbackType";
const _EFv = "EventFilter";
const _EI = "EventId";
const _EIx = "ExternalId";
const _EIxp = "ExpiresIn";
const _EM = "EmailMessage";
const _EMBL = "EmailMessageByLink";
const _EMC = "EmailMfaConfiguration";
const _EMCT = "EmailMfaConfigType";
const _EMS = "EmailMfaSettings";
const _EMST = "EmailMfaSettingsType";
const _ENOU = "EstimatedNumberOfUsers";
const _EPAUCD = "EnablePropagateAdditionalUserContextData";
const _ER = "EventResponse";
const _ERT = "EventRiskType";
const _ERv = "EventRisk";
const _ES = "EventSource";
const _ESA = "EmailSendingAccount";
const _ESBL = "EmailSubjectByLink";
const _ESTMFAE = "EnableSoftwareTokenMFAException";
const _ESm = "EmailSubject";
const _ET = "EventType";
const _ETR = "EnableTokenRevocation";
const _EVM = "EmailVerificationMessage";
const _EVS = "EmailVerificationSubject";
const _En = "Enforcement";
const _Ex = "Extension";
const _F = "From";
const _FAC = "ForceAliasCreation";
const _FC = "FirehoseConfiguration";
const _FCN = "FriendlyCredentialName";
const _FCT = "FirehoseConfigurationType";
const _FD = "FeedbackDate";
const _FDN = "FriendlyDeviceName";
const _FDR = "ForgetDeviceRequest";
const _FDo = "ForgetDevice";
const _FE = "ForbiddenException";
const _FP = "ForgotPassword";
const _FPR = "ForgotPasswordRequest";
const _FPRo = "ForgotPasswordResponse";
const _FT = "FeedbackToken";
const _FU = "FailedUsers";
const _FUITE = "FeatureUnavailableInTierException";
const _FV = "FeedbackValue";
const _Fe = "Feature";
const _Fi = "Filter";
const _G = "Groups";
const _GCSVH = "GetCSVHeader";
const _GCSVHR = "GetCSVHeaderRequest";
const _GCSVHRe = "GetCSVHeaderResponse";
const _GD = "GetDevice";
const _GDR = "GetDeviceRequest";
const _GDRe = "GetDeviceResponse";
const _GEE = "GroupExistsException";
const _GG = "GetGroup";
const _GGR = "GetGroupRequest";
const _GGRe = "GetGroupResponse";
const _GIPBI = "GetIdentityProviderByIdentifier";
const _GIPBIR = "GetIdentityProviderByIdentifierRequest";
const _GIPBIRe = "GetIdentityProviderByIdentifierResponse";
const _GLDC = "GetLogDeliveryConfiguration";
const _GLDCR = "GetLogDeliveryConfigurationRequest";
const _GLDCRe = "GetLogDeliveryConfigurationResponse";
const _GLT = "GroupListType";
const _GN = "GroupName";
const _GS = "GenerateSecret";
const _GSC = "GetSigningCertificate";
const _GSCR = "GetSigningCertificateRequest";
const _GSCRe = "GetSigningCertificateResponse";
const _GSO = "GlobalSignOut";
const _GSOR = "GlobalSignOutRequest";
const _GSORl = "GlobalSignOutResponse";
const _GT = "GroupType";
const _GTFRT = "GetTokensFromRefreshToken";
const _GTFRTR = "GetTokensFromRefreshTokenRequest";
const _GTFRTRe = "GetTokensFromRefreshTokenResponse";
const _GU = "GetUser";
const _GUAF = "GetUserAuthFactors";
const _GUAFR = "GetUserAuthFactorsRequest";
const _GUAFRe = "GetUserAuthFactorsResponse";
const _GUAVC = "GetUserAttributeVerificationCode";
const _GUAVCR = "GetUserAttributeVerificationCodeRequest";
const _GUAVCRe = "GetUserAttributeVerificationCodeResponse";
const _GUIC = "GetUICustomization";
const _GUICR = "GetUICustomizationRequest";
const _GUICRe = "GetUICustomizationResponse";
const _GUPMC = "GetUserPoolMfaConfig";
const _GUPMCR = "GetUserPoolMfaConfigRequest";
const _GUPMCRe = "GetUserPoolMfaConfigResponse";
const _GUR = "GetUserRequest";
const _GURe = "GetUserResponse";
const _Gr = "Group";
const _HA = "HighAction";
const _HB = "HtmlBody";
const _HH = "HttpHeaders";
const _HHL = "HttpHeaderList";
const _HHt = "HttpHeader";
const _I = "Identifier";
const _IA = "IpAddress";
const _IAR = "InitiateAuthRequest";
const _IARn = "InitiateAuthResponse";
const _IAn = "InitiateAuth";
const _IEE = "InternalErrorException";
const _IERAPE = "InvalidEmailRoleAccessPolicyException";
const _IF = "ImageFile";
const _II = "IdpIdentifiers";
const _IId = "IdpIdentifier";
const _ILRE = "InvalidLambdaResponseException";
const _IMT = "InviteMessageTemplate";
const _IOAFE = "InvalidOAuthFlowException";
const _IP = "IdentityProvider";
const _IPE = "InvalidParameterException";
const _IPEn = "InvalidPasswordException";
const _IPT = "IdentityProviderType";
const _ISRAPE = "InvalidSmsRoleAccessPolicyException";
const _ISRTRE = "InvalidSmsRoleTrustRelationshipException";
const _IT = "IdToken";
const _ITV = "IdTokenValidity";
const _IU = "ImageUrl";
const _IUPCE = "InvalidUserPoolConfigurationException";
const _IUm = "ImportedUsers";
const _Id = "Id";
const _JI = "JobId";
const _JN = "JobName";
const _KMSKID = "KMSKeyID";
const _L = "Limit";
const _LA = "LowAction";
const _LAa = "LambdaArn";
const _LC = "LambdaConfig";
const _LCLT = "LogConfigurationListType";
const _LCT = "LambdaConfigType";
const _LCTo = "LogConfigurationType";
const _LCo = "LogConfigurations";
const _LD = "ListDevices";
const _LDC = "LogDeliveryConfiguration";
const _LDCT = "LogDeliveryConfigurationType";
const _LDR = "ListDevicesRequest";
const _LDRi = "ListDevicesResponse";
const _LEE = "LimitExceededException";
const _LG = "ListGroups";
const _LGA = "LogGroupArn";
const _LGR = "ListGroupsRequest";
const _LGRi = "ListGroupsResponse";
const _LIP = "ListIdentityProviders";
const _LIPR = "ListIdentityProvidersRequest";
const _LIPRi = "ListIdentityProvidersResponse";
const _LL = "LogLevel";
const _LMD = "LastModifiedDate";
const _LRS = "ListResourceServers";
const _LRSR = "ListResourceServersRequest";
const _LRSRi = "ListResourceServersResponse";
const _LT = "ListTerms";
const _LTFR = "ListTagsForResource";
const _LTFRR = "ListTagsForResourceRequest";
const _LTFRRi = "ListTagsForResourceResponse";
const _LTR = "ListTermsRequest";
const _LTRi = "ListTermsResponse";
const _LU = "ListUsers";
const _LUIG = "ListUsersInGroup";
const _LUIGR = "ListUsersInGroupRequest";
const _LUIGRi = "ListUsersInGroupResponse";
const _LUIJ = "ListUserImportJobs";
const _LUIJR = "ListUserImportJobsRequest";
const _LUIJRi = "ListUserImportJobsResponse";
const _LUP = "ListUserPools";
const _LUPC = "ListUserPoolClients";
const _LUPCR = "ListUserPoolClientsRequest";
const _LUPCRi = "ListUserPoolClientsResponse";
const _LUPR = "ListUserPoolsRequest";
const _LUPRi = "ListUserPoolsResponse";
const _LUR = "ListUsersRequest";
const _LURL = "LogoutURLs";
const _LURi = "ListUsersResponse";
const _LV = "LambdaVersion";
const _LWAC = "ListWebAuthnCredentials";
const _LWACR = "ListWebAuthnCredentialsRequest";
const _LWACRi = "ListWebAuthnCredentialsResponse";
const _Li = "Links";
const _M = "Message";
const _MA = "MediumAction";
const _MAe = "MessageAction";
const _MC = "MfaConfiguration";
const _ME = "MfaEmail";
const _MFAMNFE = "MFAMethodNotFoundException";
const _MFAO = "MFAOptions";
const _MFAOLT = "MFAOptionListType";
const _MFAOT = "MFAOptionType";
const _ML = "MinimumLength";
const _MLB = "ManagedLoginBranding";
const _MLBEE = "ManagedLoginBrandingExistsException";
const _MLBI = "ManagedLoginBrandingId";
const _MLBT = "ManagedLoginBrandingType";
const _MLV = "ManagedLoginVersion";
const _MLa = "MaxLength";
const _MLi = "MinLength";
const _MR = "MaxResults";
const _MTT = "MessageTemplateType";
const _MV = "MinValue";
const _MVa = "MaxValue";
const _Mu = "Mutable";
const _N = "Notify";
const _NAC = "NumberAttributeConstraints";
const _NACT = "NumberAttributeConstraintsType";
const _NAE = "NotAuthorizedException";
const _NAEo = "NoActionEmail";
const _NC = "NotifyConfiguration";
const _NCT = "NotifyConfigurationType";
const _NDM = "NewDeviceMetadata";
const _NDMT = "NewDeviceMetadataType";
const _NET = "NotifyEmailType";
const _NT = "NextToken";
const _Na = "Name";
const _P = "Password";
const _PA = "PreAuthentication";
const _PAN = "ProviderAttributeName";
const _PAV = "ProviderAttributeValue";
const _PAo = "PostAuthentication";
const _PC = "PostConfirmation";
const _PD = "ProviderDetails";
const _PDr = "ProviderDescription";
const _PHPVE = "PasswordHistoryPolicyViolationException";
const _PHS = "PasswordHistorySize";
const _PLT = "ProvidersListType";
const _PM = "PreferredMfa";
const _PMS = "PreferredMfaSetting";
const _PN = "ProviderName";
const _PNME = "PreconditionNotMetException";
const _PNo = "PoolName";
const _PP = "PreviousPassword";
const _PPT = "PasswordPolicyType";
const _PPa = "PasswordPolicy";
const _PPr = "ProposedPassword";
const _PRRE = "PasswordResetRequiredException";
const _PSU = "PreSignUp";
const _PSUr = "PreSignedUrl";
const _PT = "PasswordType";
const _PTG = "PreTokenGeneration";
const _PTGC = "PreTokenGenerationConfig";
const _PTGVCT = "PreTokenGenerationVersionConfigType";
const _PTa = "PaginationToken";
const _PTr = "ProviderType";
const _PUEE = "PreventUserExistenceErrors";
const _PUIT = "ProviderUserIdentifierType";
const _PV = "PasswordVerifier";
const _Pe = "Permanent";
const _Po = "Policies";
const _Pr = "Precedence";
const _Pri = "Priority";
const _Pro = "Provider";
const _Prov = "Providers";
const _R = "Required";
const _RA = "RoleArn";
const _RAe = "ReadAttributes";
const _RAes = "ResourceArn";
const _RC = "RiskConfiguration";
const _RCC = "ResendConfirmationCode";
const _RCCR = "ResendConfirmationCodeRequest";
const _RCCRe = "ResendConfirmationCodeResponse";
const _RCT = "RiskConfigurationType";
const _RD = "RiskDecision";
const _REC = "RiskExceptionConfiguration";
const _RECT = "RiskExceptionConfigurationType";
const _RGPS = "RetryGracePeriodSeconds";
const _RI = "ResourceId";
const _RL = "RiskLevel";
const _RLe = "RequireLowercase";
const _RM = "RecoveryMechanisms";
const _RMR = "ReturnMergedResources";
const _RMT = "RecoveryMechanismsType";
const _RN = "RequireNumbers";
const _RNFE = "ResourceNotFoundException";
const _ROT = "RecoveryOptionType";
const _RPI = "RelyingPartyId";
const _RS = "ResourceServer";
const _RSLT = "ResourceServersListType";
const _RSSLT = "ResourceServerScopeListType";
const _RSST = "ResourceServerScopeType";
const _RST = "ResourceServerType";
const _RSe = "ResourceServers";
const _RSeq = "RequireSymbols";
const _RT = "RefreshToken";
const _RTAC = "RespondToAuthChallenge";
const _RTACR = "RespondToAuthChallengeRequest";
const _RTACRe = "RespondToAuthChallengeResponse";
const _RTEA = "ReplyToEmailAddress";
const _RTR = "RefreshTokenRotation";
const _RTRE = "RefreshTokenReuseException";
const _RTRT = "RefreshTokenRotationType";
const _RTRe = "RevokeTokenRequest";
const _RTRev = "RevokeTokenResponse";
const _RTV = "RefreshTokenValidity";
const _RTe = "ReplyTo";
const _RTev = "RevokeToken";
const _RU = "RequireUppercase";
const _S = "Session";
const _SA = "SourceArn";
const _SAC = "StringAttributeConstraints";
const _SACT = "StringAttributeConstraintsType";
const _SALT = "SchemaAttributesListType";
const _SAM = "SmsAuthenticationMessage";
const _SAT = "SchemaAttributeType";
const _SAc = "SchemaAttributes";
const _SAt = "StreamArn";
const _SB = "S3Bucket";
const _SC = "SecretCode";
const _SCA = "SnsCallerArn";
const _SCF = "SmsConfigurationFailure";
const _SCT = "SecretCodeType";
const _SCTm = "SmsConfigurationType";
const _SCTo = "S3ConfigurationType";
const _SCm = "SmsConfiguration";
const _SCo = "S3Configuration";
const _SD = "ScopeDescription";
const _SDNEE = "ScopeDoesNotExistException";
const _SDt = "StartDate";
const _SH = "SecretHash";
const _SHT = "SecretHashType";
const _SIP = "SupportedIdentityProviders";
const _SIPRL = "SkippedIPRangeList";
const _SIPT = "SignInPolicyType";
const _SIPi = "SignInPolicy";
const _SLDC = "SetLogDeliveryConfiguration";
const _SLDCR = "SetLogDeliveryConfigurationRequest";
const _SLDCRe = "SetLogDeliveryConfigurationResponse";
const _SM = "SmsMessage";
const _SMC = "SmsMfaConfiguration";
const _SMCT = "SmsMfaConfigType";
const _SMSM = "SMSMessage";
const _SMSMS = "SMSMfaSettings";
const _SMSMST = "SMSMfaSettingsType";
const _SN = "ServerName";
const _SNc = "ScopeName";
const _SP = "ServerPath";
const _SR = "SnsRegion";
const _SRC = "SetRiskConfiguration";
const _SRCR = "SetRiskConfigurationRequest";
const _SRCRe = "SetRiskConfigurationResponse";
const _ST = "SessionType";
const _STMC = "SoftwareTokenMfaConfiguration";
const _STMCT = "SoftwareTokenMfaConfigType";
const _STMFANFE = "SoftwareTokenMFANotFoundException";
const _STMFAUCT = "SoftwareTokenMFAUserCodeType";
const _STMS = "SoftwareTokenMfaSettings";
const _STMST = "SoftwareTokenMfaSettingsType";
const _SU = "SourceUser";
const _SUIC = "SetUICustomization";
const _SUICR = "SetUICustomizationRequest";
const _SUICRe = "SetUICustomizationResponse";
const _SUIJ = "StartUserImportJob";
const _SUIJR = "StartUserImportJobRequest";
const _SUIJRt = "StartUserImportJobResponse";
const _SUIJRto = "StopUserImportJobRequest";
const _SUIJRtop = "StopUserImportJobResponse";
const _SUIJt = "StopUserImportJob";
const _SUMFAP = "SetUserMFAPreference";
const _SUMFAPR = "SetUserMFAPreferenceRequest";
const _SUMFAPRe = "SetUserMFAPreferenceResponse";
const _SUPMC = "SetUserPoolMfaConfig";
const _SUPMCR = "SetUserPoolMfaConfigRequest";
const _SUPMCRe = "SetUserPoolMfaConfigResponse";
const _SUR = "SignUpRequest";
const _SURi = "SignUpResponse";
const _SUS = "SetUserSettings";
const _SUSR = "SetUserSettingsRequest";
const _SUSRe = "SetUserSettingsResponse";
const _SUi = "SignUp";
const _SUk = "SkippedUsers";
const _SVM = "SmsVerificationMessage";
const _SWAR = "StartWebAuthnRegistration";
const _SWARR = "StartWebAuthnRegistrationRequest";
const _SWARRt = "StartWebAuthnRegistrationResponse";
const _Sa = "Salt";
const _Sc = "Scopes";
const _Sch = "Schema";
const _Se = "Settings";
const _St = "Status";
const _Su = "Subject";
const _T = "Terms";
const _TB = "TextBody";
const _TCNAE = "TierChangeNotAllowedException";
const _TDLT = "TermsDescriptionListType";
const _TDT = "TermsDescriptionType";
const _TEE = "TermsExistsException";
const _TI = "TermsId";
const _TK = "TagKeys";
const _TMFAE = "TooManyFailedAttemptsException";
const _TMRE = "TooManyRequestsException";
const _TMT = "TokenModelType";
const _TN = "TermsName";
const _TP = "TemporaryPassword";
const _TPVD = "TemporaryPasswordValidityDays";
const _TR = "TagResource";
const _TRR = "TagResourceRequest";
const _TRRa = "TagResourceResponse";
const _TS = "TermsSource";
const _TT = "TokenType";
const _TTe = "TermsType";
const _TVU = "TokenValidityUnits";
const _TVUT = "TokenValidityUnitsType";
const _Ta = "Tags";
const _Ti = "Timezone";
const _To = "Token";
const _U = "Username";
const _UA = "UserAttributes";
const _UAEF = "UpdateAuthEventFeedback";
const _UAEFR = "UpdateAuthEventFeedbackRequest";
const _UAEFRp = "UpdateAuthEventFeedbackResponse";
const _UAN = "UserAttributeNames";
const _UAUS = "UserAttributeUpdateSettings";
const _UAUST = "UserAttributeUpdateSettingsType";
const _UAVD = "UnusedAccountValidityDays";
const _UAs = "UsernameAttributes";
const _UC = "UsernameConfiguration";
const _UCD = "UserCreateDate";
const _UCDT = "UserContextDataType";
const _UCDs = "UserContextData";
const _UCN = "UserConfirmationNecessary";
const _UCPV = "UseCognitoProvidedValues";
const _UCT = "UsernameConfigurationType";
const _UCs = "UserConfirmed";
const _UCse = "UserCode";
const _UDS = "UserDataShared";
const _UDSR = "UpdateDeviceStatusRequest";
const _UDSRp = "UpdateDeviceStatusResponse";
const _UDSp = "UpdateDeviceStatus";
const _UE = "UnauthorizedException";
const _UEE = "UsernameExistsException";
const _UG = "UpdateGroup";
const _UGR = "UpdateGroupRequest";
const _UGRp = "UpdateGroupResponse";
const _UIC = "UICustomization";
const _UICT = "UICustomizationType";
const _UIIPE = "UserImportInProgressException";
const _UIJ = "UserImportJob";
const _UIJLT = "UserImportJobsListType";
const _UIJT = "UserImportJobType";
const _UIJs = "UserImportJobs";
const _UIP = "UpdateIdentityProvider";
const _UIPE = "UnsupportedIdentityProviderException";
const _UIPR = "UpdateIdentityProviderRequest";
const _UIPRp = "UpdateIdentityProviderResponse";
const _ULE = "UnexpectedLambdaException";
const _ULMD = "UserLastModifiedDate";
const _ULT = "UsersListType";
const _ULVE = "UserLambdaValidationException";
const _UM = "UserMigration";
const _UMFASL = "UserMFASettingList";
const _UMLB = "UpdateManagedLoginBranding";
const _UMLBR = "UpdateManagedLoginBrandingRequest";
const _UMLBRp = "UpdateManagedLoginBrandingResponse";
const _UNCE = "UserNotConfirmedException";
const _UNFE = "UserNotFoundException";
const _UOE = "UnsupportedOperationException";
const _UP = "UserPool";
const _UPAO = "UserPoolAddOns";
const _UPAONEE = "UserPoolAddOnNotEnabledException";
const _UPAOT = "UserPoolAddOnsType";
const _UPC = "UserPoolClient";
const _UPCD = "UserPoolClientDescription";
const _UPCLT = "UserPoolClientListType";
const _UPCT = "UserPoolClientType";
const _UPCs = "UserPoolClients";
const _UPDT = "UserPoolDescriptionType";
const _UPI = "UserPoolId";
const _UPLT = "UserPoolListType";
const _UPPT = "UserPoolPolicyType";
const _UPT = "UserPoolTags";
const _UPTE = "UserPoolTaggingException";
const _UPTs = "UserPoolTier";
const _UPTse = "UserPoolType";
const _UPs = "UserPools";
const _UR = "UntagResource";
const _URR = "UntagResourceRequest";
const _URRn = "UntagResourceResponse";
const _URS = "UpdateResourceServer";
const _URSR = "UpdateResourceServerRequest";
const _URSRp = "UpdateResourceServerResponse";
const _US = "UserStatus";
const _USs = "UserSub";
const _UT = "UsernameType";
const _UTR = "UpdateTermsRequest";
const _UTRp = "UpdateTermsResponse";
const _UTTE = "UnsupportedTokenTypeException";
const _UTp = "UpdateTerms";
const _UTs = "UserType";
const _UUA = "UpdateUserAttributes";
const _UUAR = "UpdateUserAttributesRequest";
const _UUARp = "UpdateUserAttributesResponse";
const _UUP = "UpdateUserPool";
const _UUPC = "UpdateUserPoolClient";
const _UUPCR = "UpdateUserPoolClientRequest";
const _UUPCRp = "UpdateUserPoolClientResponse";
const _UUPD = "UpdateUserPoolDomain";
const _UUPDR = "UpdateUserPoolDomainRequest";
const _UUPDRp = "UpdateUserPoolDomainResponse";
const _UUPR = "UpdateUserPoolRequest";
const _UUPRp = "UpdateUserPoolResponse";
const _UUSE = "UnsupportedUserStateException";
const _UV = "UserVerification";
const _Us = "User";
const _Use = "Users";
const _V = "Value";
const _VACR = "VerifyAuthChallengeResponse";
const _VD = "ValidationData";
const _VMT = "VerificationMessageTemplate";
const _VMTT = "VerificationMessageTemplateType";
const _VST = "VerifySoftwareToken";
const _VSTR = "VerifySoftwareTokenRequest";
const _VSTRe = "VerifySoftwareTokenResponse";
const _VUA = "VerifyUserAttribute";
const _VUAR = "VerifyUserAttributeRequest";
const _VUARe = "VerifyUserAttributeResponse";
const _Ve = "Version";
const _WA = "WriteAttributes";
const _WAC = "WebAuthnConfiguration";
const _WACD = "WebAuthnCredentialDescription";
const _WACDLT = "WebAuthnCredentialDescriptionListType";
const _WACME = "WebAuthnClientMismatchException";
const _WACMEe = "WebAuthnConfigurationMissingException";
const _WACNFE = "WebAuthnChallengeNotFoundException";
const _WACNSE = "WebAuthnCredentialNotSupportedException";
const _WACT = "WebAuthnConfigurationType";
const _WANEE = "WebAuthnNotEnabledException";
const _WAONAE = "WebAuthnOriginNotAllowedException";
const _WARPME = "WebAuthnRelyingPartyMismatchException";
const _c = "client";
const _e = "error";
const _hE = "httpError";
const _hN = "headerName";
const _hV = "headerValue";
const _m = "message";
const _rC = "reasonCode";
const _s = "server";
const _sm = "smithy.ts.sdk.synthetic.com.amazonaws.cognitoidentityprovider";
const n0 = "com.amazonaws.cognitoidentityprovider";
var AttributeValueType = [0, n0, _AVT, 8, 0];
var ClientIdType = [0, n0, _CIT, 8, 0];
var ClientSecretType = [0, n0, _CST, 8, 0];
var PasswordType = [0, n0, _PT, 8, 0];
var SecretCodeType = [0, n0, _SCT, 8, 0];
var SecretHashType = [0, n0, _SHT, 8, 0];
var SessionType = [0, n0, _ST, 8, 0];
var SoftwareTokenMFAUserCodeType = [0, n0, _STMFAUCT, 8, 0];
var TokenModelType = [0, n0, _TMT, 8, 0];
var UsernameType = [0, n0, _UT, 8, 0];
var AccountRecoverySettingType$ = [3, n0, _ARST,
    0,
    [_RM],
    [() => RecoveryMechanismsType]
];
var AccountTakeoverActionsType$ = [3, n0, _ATAT,
    0,
    [_LA, _MA, _HA],
    [() => AccountTakeoverActionType$, () => AccountTakeoverActionType$, () => AccountTakeoverActionType$]
];
var AccountTakeoverActionType$ = [3, n0, _ATATc,
    0,
    [_N, _EA],
    [2, 0]
];
var AccountTakeoverRiskConfigurationType$ = [3, n0, _ATRCT,
    0,
    [_NC, _A],
    [() => NotifyConfigurationType$, () => AccountTakeoverActionsType$]
];
var AddCustomAttributesRequest$ = [3, n0, _ACAR,
    0,
    [_UPI, _CA],
    [0, () => CustomAttributesListType]
];
var AddCustomAttributesResponse$ = [3, n0, _ACARd,
    0,
    [],
    []
];
var AdminAddUserToGroupRequest$ = [3, n0, _AAUTGR,
    0,
    [_UPI, _U, _GN],
    [0, [() => UsernameType, 0], 0]
];
var AdminConfirmSignUpRequest$ = [3, n0, _ACSUR,
    0,
    [_UPI, _U, _CM],
    [0, [() => UsernameType, 0], 128 | 0]
];
var AdminConfirmSignUpResponse$ = [3, n0, _ACSURd,
    0,
    [],
    []
];
var AdminCreateUserConfigType$ = [3, n0, _ACUCT,
    0,
    [_AACUO, _UAVD, _IMT],
    [2, 1, () => MessageTemplateType$]
];
var AdminCreateUserRequest$ = [3, n0, _ACUR,
    0,
    [_UPI, _U, _UA, _VD, _TP, _FAC, _MAe, _DDM, _CM],
    [0, [() => UsernameType, 0], [() => AttributeListType, 0], [() => AttributeListType, 0], [() => PasswordType, 0], 2, 0, 64 | 0, 128 | 0]
];
var AdminCreateUserResponse$ = [3, n0, _ACURd,
    0,
    [_Us],
    [[() => UserType$, 0]]
];
var AdminDeleteUserAttributesRequest$ = [3, n0, _ADUAR,
    0,
    [_UPI, _U, _UAN],
    [0, [() => UsernameType, 0], 64 | 0]
];
var AdminDeleteUserAttributesResponse$ = [3, n0, _ADUARd,
    0,
    [],
    []
];
var AdminDeleteUserRequest$ = [3, n0, _ADUR,
    0,
    [_UPI, _U],
    [0, [() => UsernameType, 0]]
];
var AdminDisableProviderForUserRequest$ = [3, n0, _ADPFUR,
    0,
    [_UPI, _Us],
    [0, () => ProviderUserIdentifierType$]
];
var AdminDisableProviderForUserResponse$ = [3, n0, _ADPFURd,
    0,
    [],
    []
];
var AdminDisableUserRequest$ = [3, n0, _ADURd,
    0,
    [_UPI, _U],
    [0, [() => UsernameType, 0]]
];
var AdminDisableUserResponse$ = [3, n0, _ADURdm,
    0,
    [],
    []
];
var AdminEnableUserRequest$ = [3, n0, _AEUR,
    0,
    [_UPI, _U],
    [0, [() => UsernameType, 0]]
];
var AdminEnableUserResponse$ = [3, n0, _AEURd,
    0,
    [],
    []
];
var AdminForgetDeviceRequest$ = [3, n0, _AFDR,
    0,
    [_UPI, _U, _DK],
    [0, [() => UsernameType, 0], 0]
];
var AdminGetDeviceRequest$ = [3, n0, _AGDR,
    0,
    [_DK, _UPI, _U],
    [0, 0, [() => UsernameType, 0]]
];
var AdminGetDeviceResponse$ = [3, n0, _AGDRd,
    0,
    [_D],
    [[() => DeviceType$, 0]]
];
var AdminGetUserRequest$ = [3, n0, _AGUR,
    0,
    [_UPI, _U],
    [0, [() => UsernameType, 0]]
];
var AdminGetUserResponse$ = [3, n0, _AGURd,
    0,
    [_U, _UA, _UCD, _ULMD, _E, _US, _MFAO, _PMS, _UMFASL],
    [[() => UsernameType, 0], [() => AttributeListType, 0], 4, 4, 2, 0, () => MFAOptionListType, 0, 64 | 0]
];
var AdminInitiateAuthRequest$ = [3, n0, _AIAR,
    0,
    [_UPI, _CI, _AF, _AP, _CM, _AM, _CD, _S],
    [0, [() => ClientIdType, 0], 0, [() => AuthParametersType, 0], 128 | 0, () => AnalyticsMetadataType$, () => ContextDataType$, [() => SessionType, 0]]
];
var AdminInitiateAuthResponse$ = [3, n0, _AIARd,
    0,
    [_CN, _S, _CP, _AR, _AC],
    [0, [() => SessionType, 0], 128 | 0, [() => AuthenticationResultType$, 0], 64 | 0]
];
var AdminLinkProviderForUserRequest$ = [3, n0, _ALPFUR,
    0,
    [_UPI, _DU, _SU],
    [0, () => ProviderUserIdentifierType$, () => ProviderUserIdentifierType$]
];
var AdminLinkProviderForUserResponse$ = [3, n0, _ALPFURd,
    0,
    [],
    []
];
var AdminListDevicesRequest$ = [3, n0, _ALDR,
    0,
    [_UPI, _U, _L, _PTa],
    [0, [() => UsernameType, 0], 1, 0]
];
var AdminListDevicesResponse$ = [3, n0, _ALDRd,
    0,
    [_De, _PTa],
    [[() => DeviceListType, 0], 0]
];
var AdminListGroupsForUserRequest$ = [3, n0, _ALGFUR,
    0,
    [_U, _UPI, _L, _NT],
    [[() => UsernameType, 0], 0, 1, 0]
];
var AdminListGroupsForUserResponse$ = [3, n0, _ALGFURd,
    0,
    [_G, _NT],
    [() => GroupListType, 0]
];
var AdminListUserAuthEventsRequest$ = [3, n0, _ALUAER,
    0,
    [_UPI, _U, _MR, _NT],
    [0, [() => UsernameType, 0], 1, 0]
];
var AdminListUserAuthEventsResponse$ = [3, n0, _ALUAERd,
    0,
    [_AE, _NT],
    [() => AuthEventsType, 0]
];
var AdminRemoveUserFromGroupRequest$ = [3, n0, _ARUFGR,
    0,
    [_UPI, _U, _GN],
    [0, [() => UsernameType, 0], 0]
];
var AdminResetUserPasswordRequest$ = [3, n0, _ARUPR,
    0,
    [_UPI, _U, _CM],
    [0, [() => UsernameType, 0], 128 | 0]
];
var AdminResetUserPasswordResponse$ = [3, n0, _ARUPRd,
    0,
    [],
    []
];
var AdminRespondToAuthChallengeRequest$ = [3, n0, _ARTACR,
    0,
    [_UPI, _CI, _CN, _CR, _S, _AM, _CD, _CM],
    [0, [() => ClientIdType, 0], 0, [() => ChallengeResponsesType, 0], [() => SessionType, 0], () => AnalyticsMetadataType$, () => ContextDataType$, 128 | 0]
];
var AdminRespondToAuthChallengeResponse$ = [3, n0, _ARTACRd,
    0,
    [_CN, _S, _CP, _AR],
    [0, [() => SessionType, 0], 128 | 0, [() => AuthenticationResultType$, 0]]
];
var AdminSetUserMFAPreferenceRequest$ = [3, n0, _ASUMFAPR,
    0,
    [_SMSMS, _STMS, _EMS, _U, _UPI],
    [() => SMSMfaSettingsType$, () => SoftwareTokenMfaSettingsType$, () => EmailMfaSettingsType$, [() => UsernameType, 0], 0]
];
var AdminSetUserMFAPreferenceResponse$ = [3, n0, _ASUMFAPRd,
    0,
    [],
    []
];
var AdminSetUserPasswordRequest$ = [3, n0, _ASUPR,
    0,
    [_UPI, _U, _P, _Pe],
    [0, [() => UsernameType, 0], [() => PasswordType, 0], 2]
];
var AdminSetUserPasswordResponse$ = [3, n0, _ASUPRd,
    0,
    [],
    []
];
var AdminSetUserSettingsRequest$ = [3, n0, _ASUSR,
    0,
    [_UPI, _U, _MFAO],
    [0, [() => UsernameType, 0], () => MFAOptionListType]
];
var AdminSetUserSettingsResponse$ = [3, n0, _ASUSRd,
    0,
    [],
    []
];
var AdminUpdateAuthEventFeedbackRequest$ = [3, n0, _AUAEFR,
    0,
    [_UPI, _U, _EI, _FV],
    [0, [() => UsernameType, 0], 0, 0]
];
var AdminUpdateAuthEventFeedbackResponse$ = [3, n0, _AUAEFRd,
    0,
    [],
    []
];
var AdminUpdateDeviceStatusRequest$ = [3, n0, _AUDSR,
    0,
    [_UPI, _U, _DK, _DRS],
    [0, [() => UsernameType, 0], 0, 0]
];
var AdminUpdateDeviceStatusResponse$ = [3, n0, _AUDSRd,
    0,
    [],
    []
];
var AdminUpdateUserAttributesRequest$ = [3, n0, _AUUAR,
    0,
    [_UPI, _U, _UA, _CM],
    [0, [() => UsernameType, 0], [() => AttributeListType, 0], 128 | 0]
];
var AdminUpdateUserAttributesResponse$ = [3, n0, _AUUARd,
    0,
    [],
    []
];
var AdminUserGlobalSignOutRequest$ = [3, n0, _AUGSOR,
    0,
    [_UPI, _U],
    [0, [() => UsernameType, 0]]
];
var AdminUserGlobalSignOutResponse$ = [3, n0, _AUGSORd,
    0,
    [],
    []
];
var AdvancedSecurityAdditionalFlowsType$ = [3, n0, _ASAFT,
    0,
    [_CAM],
    [0]
];
var AliasExistsException$ = [-3, n0, _AEE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(AliasExistsException$, AliasExistsException);
var AnalyticsConfigurationType$ = [3, n0, _ACT,
    0,
    [_AI, _AA, _RA, _EIx, _UDS],
    [0, 0, 0, 0, 2]
];
var AnalyticsMetadataType$ = [3, n0, _AMT,
    0,
    [_AEI],
    [0]
];
var AssetType$ = [3, n0, _AT,
    0,
    [_C, _CMo, _Ex, _B, _RI],
    [0, 0, 0, 21, 0]
];
var AssociateSoftwareTokenRequest$ = [3, n0, _ASTR,
    0,
    [_ATc, _S],
    [[() => TokenModelType, 0], [() => SessionType, 0]]
];
var AssociateSoftwareTokenResponse$ = [3, n0, _ASTRs,
    0,
    [_SC, _S],
    [[() => SecretCodeType, 0], [() => SessionType, 0]]
];
var AttributeType$ = [3, n0, _ATt,
    0,
    [_Na, _V],
    [0, [() => AttributeValueType, 0]]
];
var AuthenticationResultType$ = [3, n0, _ART,
    0,
    [_ATc, _EIxp, _TT, _RT, _IT, _NDM],
    [[() => TokenModelType, 0], 1, 0, [() => TokenModelType, 0], [() => TokenModelType, 0], () => NewDeviceMetadataType$]
];
var AuthEventType$ = [3, n0, _AET,
    0,
    [_EI, _ET, _CDr, _ER, _ERv, _CR, _ECD, _EF],
    [0, 0, 4, 0, () => EventRiskType$, () => ChallengeResponseListType, () => EventContextDataType$, () => EventFeedbackType$]
];
var ChallengeResponseType$ = [3, n0, _CRT,
    0,
    [_CN, _CRh],
    [0, 0]
];
var ChangePasswordRequest$ = [3, n0, _CPR,
    0,
    [_PP, _PPr, _ATc],
    [[() => PasswordType, 0], [() => PasswordType, 0], [() => TokenModelType, 0]]
];
var ChangePasswordResponse$ = [3, n0, _CPRh,
    0,
    [],
    []
];
var CloudWatchLogsConfigurationType$ = [3, n0, _CWLCT,
    0,
    [_LGA],
    [0]
];
var CodeDeliveryDetailsType$ = [3, n0, _CDDT,
    0,
    [_Des, _DM, _AN],
    [0, 0, 0]
];
var CodeDeliveryFailureException$ = [-3, n0, _CDFE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(CodeDeliveryFailureException$, CodeDeliveryFailureException);
var CodeMismatchException$ = [-3, n0, _CME,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(CodeMismatchException$, CodeMismatchException);
var CompleteWebAuthnRegistrationRequest$ = [3, n0, _CWARR,
    0,
    [_ATc, _Cr],
    [[() => TokenModelType, 0], 15]
];
var CompleteWebAuthnRegistrationResponse$ = [3, n0, _CWARRo,
    0,
    [],
    []
];
var CompromisedCredentialsActionsType$ = [3, n0, _CCAT,
    0,
    [_EA],
    [0]
];
var CompromisedCredentialsRiskConfigurationType$ = [3, n0, _CCRCT,
    0,
    [_EFv, _A],
    [64 | 0, () => CompromisedCredentialsActionsType$]
];
var ConcurrentModificationException$ = [-3, n0, _CMEo,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ConcurrentModificationException$, ConcurrentModificationException);
var ConfirmDeviceRequest$ = [3, n0, _CDR,
    0,
    [_ATc, _DK, _DSVC, _DN],
    [[() => TokenModelType, 0], 0, () => DeviceSecretVerifierConfigType$, 0]
];
var ConfirmDeviceResponse$ = [3, n0, _CDRo,
    0,
    [_UCN],
    [2]
];
var ConfirmForgotPasswordRequest$ = [3, n0, _CFPR,
    0,
    [_CI, _SH, _U, _CC, _P, _AM, _UCDs, _CM],
    [[() => ClientIdType, 0], [() => SecretHashType, 0], [() => UsernameType, 0], 0, [() => PasswordType, 0], () => AnalyticsMetadataType$, [() => UserContextDataType$, 0], 128 | 0]
];
var ConfirmForgotPasswordResponse$ = [3, n0, _CFPRo,
    0,
    [],
    []
];
var ConfirmSignUpRequest$ = [3, n0, _CSUR,
    0,
    [_CI, _SH, _U, _CC, _FAC, _AM, _UCDs, _CM, _S],
    [[() => ClientIdType, 0], [() => SecretHashType, 0], [() => UsernameType, 0], 0, 2, () => AnalyticsMetadataType$, [() => UserContextDataType$, 0], 128 | 0, [() => SessionType, 0]]
];
var ConfirmSignUpResponse$ = [3, n0, _CSURo,
    0,
    [_S],
    [[() => SessionType, 0]]
];
var ContextDataType$ = [3, n0, _CDT,
    0,
    [_IA, _SN, _SP, _HH, _ED],
    [0, 0, 0, () => HttpHeaderList, 0]
];
var CreateGroupRequest$ = [3, n0, _CGR,
    0,
    [_GN, _UPI, _Desc, _RA, _Pr],
    [0, 0, 0, 0, 1]
];
var CreateGroupResponse$ = [3, n0, _CGRr,
    0,
    [_Gr],
    [() => GroupType$]
];
var CreateIdentityProviderRequest$ = [3, n0, _CIPR,
    0,
    [_UPI, _PN, _PTr, _PD, _AMt, _II],
    [0, 0, 0, 128 | 0, 128 | 0, 64 | 0]
];
var CreateIdentityProviderResponse$ = [3, n0, _CIPRr,
    0,
    [_IP],
    [() => IdentityProviderType$]
];
var CreateManagedLoginBrandingRequest$ = [3, n0, _CMLBR,
    0,
    [_UPI, _CI, _UCPV, _Se, _As],
    [0, [() => ClientIdType, 0], 2, 15, () => AssetListType]
];
var CreateManagedLoginBrandingResponse$ = [3, n0, _CMLBRr,
    0,
    [_MLB],
    [() => ManagedLoginBrandingType$]
];
var CreateResourceServerRequest$ = [3, n0, _CRSR,
    0,
    [_UPI, _I, _Na, _Sc],
    [0, 0, 0, () => ResourceServerScopeListType]
];
var CreateResourceServerResponse$ = [3, n0, _CRSRr,
    0,
    [_RS],
    [() => ResourceServerType$]
];
var CreateTermsRequest$ = [3, n0, _CTR,
    0,
    [_UPI, _CI, _TN, _TS, _En, _Li],
    [0, [() => ClientIdType, 0], 0, 0, 0, 128 | 0]
];
var CreateTermsResponse$ = [3, n0, _CTRr,
    0,
    [_T],
    [[() => TermsType$, 0]]
];
var CreateUserImportJobRequest$ = [3, n0, _CUIJR,
    0,
    [_JN, _UPI, _CWLRA],
    [0, 0, 0]
];
var CreateUserImportJobResponse$ = [3, n0, _CUIJRr,
    0,
    [_UIJ],
    [() => UserImportJobType$]
];
var CreateUserPoolClientRequest$ = [3, n0, _CUPCR,
    0,
    [_UPI, _CNl, _GS, _RTV, _ATV, _ITV, _TVU, _RAe, _WA, _EAF, _SIP, _CURL, _LURL, _DRURI, _AOAF, _AOAS, _AOAFUPC, _ACn, _PUEE, _ETR, _EPAUCD, _ASV, _RTR],
    [0, 0, 2, 1, 1, 1, () => TokenValidityUnitsType$, 64 | 0, 64 | 0, 64 | 0, 64 | 0, 64 | 0, 64 | 0, 0, 64 | 0, 64 | 0, 2, () => AnalyticsConfigurationType$, 0, 2, 2, 1, () => RefreshTokenRotationType$]
];
var CreateUserPoolClientResponse$ = [3, n0, _CUPCRr,
    0,
    [_UPC],
    [[() => UserPoolClientType$, 0]]
];
var CreateUserPoolDomainRequest$ = [3, n0, _CUPDR,
    0,
    [_Do, _UPI, _MLV, _CDC],
    [0, 0, 1, () => CustomDomainConfigType$]
];
var CreateUserPoolDomainResponse$ = [3, n0, _CUPDRr,
    0,
    [_MLV, _CFD],
    [1, 0]
];
var CreateUserPoolRequest$ = [3, n0, _CUPR,
    0,
    [_PNo, _Po, _DP, _LC, _AVA, _AAl, _UAs, _SVM, _EVM, _EVS, _VMT, _SAM, _MC, _UAUS, _DC, _EC, _SCm, _UPT, _ACUC, _Sch, _UPAO, _UC, _ARS, _UPTs],
    [0, () => UserPoolPolicyType$, 0, () => LambdaConfigType$, 64 | 0, 64 | 0, 64 | 0, 0, 0, 0, () => VerificationMessageTemplateType$, 0, 0, () => UserAttributeUpdateSettingsType$, () => DeviceConfigurationType$, () => EmailConfigurationType$, () => SmsConfigurationType$, 128 | 0, () => AdminCreateUserConfigType$, () => SchemaAttributesListType, () => UserPoolAddOnsType$, () => UsernameConfigurationType$, () => AccountRecoverySettingType$, 0]
];
var CreateUserPoolResponse$ = [3, n0, _CUPRr,
    0,
    [_UP],
    [() => UserPoolType$]
];
var CustomDomainConfigType$ = [3, n0, _CDCT,
    0,
    [_CAe],
    [0]
];
var CustomEmailLambdaVersionConfigType$ = [3, n0, _CELVCT,
    0,
    [_LV, _LAa],
    [0, 0]
];
var CustomSMSLambdaVersionConfigType$ = [3, n0, _CSMSLVCT,
    0,
    [_LV, _LAa],
    [0, 0]
];
var DeleteGroupRequest$ = [3, n0, _DGR,
    0,
    [_GN, _UPI],
    [0, 0]
];
var DeleteIdentityProviderRequest$ = [3, n0, _DIPR,
    0,
    [_UPI, _PN],
    [0, 0]
];
var DeleteManagedLoginBrandingRequest$ = [3, n0, _DMLBR,
    0,
    [_MLBI, _UPI],
    [0, 0]
];
var DeleteResourceServerRequest$ = [3, n0, _DRSR,
    0,
    [_UPI, _I],
    [0, 0]
];
var DeleteTermsRequest$ = [3, n0, _DTR,
    0,
    [_TI, _UPI],
    [0, 0]
];
var DeleteUserAttributesRequest$ = [3, n0, _DUAR,
    0,
    [_UAN, _ATc],
    [64 | 0, [() => TokenModelType, 0]]
];
var DeleteUserAttributesResponse$ = [3, n0, _DUARe,
    0,
    [],
    []
];
var DeleteUserPoolClientRequest$ = [3, n0, _DUPCR,
    0,
    [_UPI, _CI],
    [0, [() => ClientIdType, 0]]
];
var DeleteUserPoolDomainRequest$ = [3, n0, _DUPDR,
    0,
    [_Do, _UPI],
    [0, 0]
];
var DeleteUserPoolDomainResponse$ = [3, n0, _DUPDRe,
    0,
    [],
    []
];
var DeleteUserPoolRequest$ = [3, n0, _DUPR,
    0,
    [_UPI],
    [0]
];
var DeleteUserRequest$ = [3, n0, _DUR,
    0,
    [_ATc],
    [[() => TokenModelType, 0]]
];
var DeleteWebAuthnCredentialRequest$ = [3, n0, _DWACR,
    0,
    [_ATc, _CIr],
    [[() => TokenModelType, 0], 0]
];
var DeleteWebAuthnCredentialResponse$ = [3, n0, _DWACRe,
    0,
    [],
    []
];
var DescribeIdentityProviderRequest$ = [3, n0, _DIPRe,
    0,
    [_UPI, _PN],
    [0, 0]
];
var DescribeIdentityProviderResponse$ = [3, n0, _DIPRes,
    0,
    [_IP],
    [() => IdentityProviderType$]
];
var DescribeManagedLoginBrandingByClientRequest$ = [3, n0, _DMLBBCR,
    0,
    [_UPI, _CI, _RMR],
    [0, [() => ClientIdType, 0], 2]
];
var DescribeManagedLoginBrandingByClientResponse$ = [3, n0, _DMLBBCRe,
    0,
    [_MLB],
    [() => ManagedLoginBrandingType$]
];
var DescribeManagedLoginBrandingRequest$ = [3, n0, _DMLBRe,
    0,
    [_UPI, _MLBI, _RMR],
    [0, 0, 2]
];
var DescribeManagedLoginBrandingResponse$ = [3, n0, _DMLBRes,
    0,
    [_MLB],
    [() => ManagedLoginBrandingType$]
];
var DescribeResourceServerRequest$ = [3, n0, _DRSRe,
    0,
    [_UPI, _I],
    [0, 0]
];
var DescribeResourceServerResponse$ = [3, n0, _DRSRes,
    0,
    [_RS],
    [() => ResourceServerType$]
];
var DescribeRiskConfigurationRequest$ = [3, n0, _DRCR,
    0,
    [_UPI, _CI],
    [0, [() => ClientIdType, 0]]
];
var DescribeRiskConfigurationResponse$ = [3, n0, _DRCRe,
    0,
    [_RC],
    [[() => RiskConfigurationType$, 0]]
];
var DescribeTermsRequest$ = [3, n0, _DTRe,
    0,
    [_TI, _UPI],
    [0, 0]
];
var DescribeTermsResponse$ = [3, n0, _DTRes,
    0,
    [_T],
    [[() => TermsType$, 0]]
];
var DescribeUserImportJobRequest$ = [3, n0, _DUIJR,
    0,
    [_UPI, _JI],
    [0, 0]
];
var DescribeUserImportJobResponse$ = [3, n0, _DUIJRe,
    0,
    [_UIJ],
    [() => UserImportJobType$]
];
var DescribeUserPoolClientRequest$ = [3, n0, _DUPCRe,
    0,
    [_UPI, _CI],
    [0, [() => ClientIdType, 0]]
];
var DescribeUserPoolClientResponse$ = [3, n0, _DUPCRes,
    0,
    [_UPC],
    [[() => UserPoolClientType$, 0]]
];
var DescribeUserPoolDomainRequest$ = [3, n0, _DUPDRes,
    0,
    [_Do],
    [0]
];
var DescribeUserPoolDomainResponse$ = [3, n0, _DUPDResc,
    0,
    [_DD],
    [() => DomainDescriptionType$]
];
var DescribeUserPoolRequest$ = [3, n0, _DUPRe,
    0,
    [_UPI],
    [0]
];
var DescribeUserPoolResponse$ = [3, n0, _DUPRes,
    0,
    [_UP],
    [() => UserPoolType$]
];
var DeviceConfigurationType$ = [3, n0, _DCT,
    0,
    [_CROND, _DOROUP],
    [2, 2]
];
var DeviceKeyExistsException$ = [-3, n0, _DKEE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(DeviceKeyExistsException$, DeviceKeyExistsException);
var DeviceSecretVerifierConfigType$ = [3, n0, _DSVCT,
    0,
    [_PV, _Sa],
    [0, 0]
];
var DeviceType$ = [3, n0, _DT,
    0,
    [_DK, _DA, _DCD, _DLMD, _DLAD],
    [0, [() => AttributeListType, 0], 4, 4, 4]
];
var DomainDescriptionType$ = [3, n0, _DDT,
    0,
    [_UPI, _AWSAI, _Do, _SB, _CFDl, _Ve, _St, _CDC, _MLV],
    [0, 0, 0, 0, 0, 0, 0, () => CustomDomainConfigType$, 1]
];
var DuplicateProviderException$ = [-3, n0, _DPE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(DuplicateProviderException$, DuplicateProviderException);
var EmailConfigurationType$ = [3, n0, _ECT,
    0,
    [_SA, _RTEA, _ESA, _F, _CS],
    [0, 0, 0, 0, 0]
];
var EmailMfaConfigType$ = [3, n0, _EMCT,
    0,
    [_M, _Su],
    [0, 0]
];
var EmailMfaSettingsType$ = [3, n0, _EMST,
    0,
    [_E, _PM],
    [2, 2]
];
var EnableSoftwareTokenMFAException$ = [-3, n0, _ESTMFAE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(EnableSoftwareTokenMFAException$, EnableSoftwareTokenMFAException);
var EventContextDataType$ = [3, n0, _ECDT,
    0,
    [_IA, _DN, _Ti, _Ci, _Co],
    [0, 0, 0, 0, 0]
];
var EventFeedbackType$ = [3, n0, _EFT,
    0,
    [_FV, _Pro, _FD],
    [0, 0, 4]
];
var EventRiskType$ = [3, n0, _ERT,
    0,
    [_RD, _RL, _CCD],
    [0, 0, 2]
];
var ExpiredCodeException$ = [-3, n0, _ECE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ExpiredCodeException$, ExpiredCodeException);
var FeatureUnavailableInTierException$ = [-3, n0, _FUITE,
    { [_e]: _c, [_hE]: 403 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(FeatureUnavailableInTierException$, FeatureUnavailableInTierException);
var FirehoseConfigurationType$ = [3, n0, _FCT,
    0,
    [_SAt],
    [0]
];
var ForbiddenException$ = [-3, n0, _FE,
    { [_e]: _c, [_hE]: 403 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ForbiddenException$, ForbiddenException);
var ForgetDeviceRequest$ = [3, n0, _FDR,
    0,
    [_ATc, _DK],
    [[() => TokenModelType, 0], 0]
];
var ForgotPasswordRequest$ = [3, n0, _FPR,
    0,
    [_CI, _SH, _UCDs, _U, _AM, _CM],
    [[() => ClientIdType, 0], [() => SecretHashType, 0], [() => UserContextDataType$, 0], [() => UsernameType, 0], () => AnalyticsMetadataType$, 128 | 0]
];
var ForgotPasswordResponse$ = [3, n0, _FPRo,
    0,
    [_CDD],
    [() => CodeDeliveryDetailsType$]
];
var GetCSVHeaderRequest$ = [3, n0, _GCSVHR,
    0,
    [_UPI],
    [0]
];
var GetCSVHeaderResponse$ = [3, n0, _GCSVHRe,
    0,
    [_UPI, _CSVH],
    [0, 64 | 0]
];
var GetDeviceRequest$ = [3, n0, _GDR,
    0,
    [_DK, _ATc],
    [0, [() => TokenModelType, 0]]
];
var GetDeviceResponse$ = [3, n0, _GDRe,
    0,
    [_D],
    [[() => DeviceType$, 0]]
];
var GetGroupRequest$ = [3, n0, _GGR,
    0,
    [_GN, _UPI],
    [0, 0]
];
var GetGroupResponse$ = [3, n0, _GGRe,
    0,
    [_Gr],
    [() => GroupType$]
];
var GetIdentityProviderByIdentifierRequest$ = [3, n0, _GIPBIR,
    0,
    [_UPI, _IId],
    [0, 0]
];
var GetIdentityProviderByIdentifierResponse$ = [3, n0, _GIPBIRe,
    0,
    [_IP],
    [() => IdentityProviderType$]
];
var GetLogDeliveryConfigurationRequest$ = [3, n0, _GLDCR,
    0,
    [_UPI],
    [0]
];
var GetLogDeliveryConfigurationResponse$ = [3, n0, _GLDCRe,
    0,
    [_LDC],
    [() => LogDeliveryConfigurationType$]
];
var GetSigningCertificateRequest$ = [3, n0, _GSCR,
    0,
    [_UPI],
    [0]
];
var GetSigningCertificateResponse$ = [3, n0, _GSCRe,
    0,
    [_Ce],
    [0]
];
var GetTokensFromRefreshTokenRequest$ = [3, n0, _GTFRTR,
    0,
    [_RT, _CI, _CSl, _DK, _CM],
    [[() => TokenModelType, 0], [() => ClientIdType, 0], [() => ClientSecretType, 0], 0, 128 | 0]
];
var GetTokensFromRefreshTokenResponse$ = [3, n0, _GTFRTRe,
    0,
    [_AR],
    [[() => AuthenticationResultType$, 0]]
];
var GetUICustomizationRequest$ = [3, n0, _GUICR,
    0,
    [_UPI, _CI],
    [0, [() => ClientIdType, 0]]
];
var GetUICustomizationResponse$ = [3, n0, _GUICRe,
    0,
    [_UIC],
    [[() => UICustomizationType$, 0]]
];
var GetUserAttributeVerificationCodeRequest$ = [3, n0, _GUAVCR,
    0,
    [_ATc, _AN, _CM],
    [[() => TokenModelType, 0], 0, 128 | 0]
];
var GetUserAttributeVerificationCodeResponse$ = [3, n0, _GUAVCRe,
    0,
    [_CDD],
    [() => CodeDeliveryDetailsType$]
];
var GetUserAuthFactorsRequest$ = [3, n0, _GUAFR,
    0,
    [_ATc],
    [[() => TokenModelType, 0]]
];
var GetUserAuthFactorsResponse$ = [3, n0, _GUAFRe,
    0,
    [_U, _PMS, _UMFASL, _CUAF],
    [[() => UsernameType, 0], 0, 64 | 0, 64 | 0]
];
var GetUserPoolMfaConfigRequest$ = [3, n0, _GUPMCR,
    0,
    [_UPI],
    [0]
];
var GetUserPoolMfaConfigResponse$ = [3, n0, _GUPMCRe,
    0,
    [_SMC, _STMC, _EMC, _MC, _WAC],
    [() => SmsMfaConfigType$, () => SoftwareTokenMfaConfigType$, () => EmailMfaConfigType$, 0, () => WebAuthnConfigurationType$]
];
var GetUserRequest$ = [3, n0, _GUR,
    0,
    [_ATc],
    [[() => TokenModelType, 0]]
];
var GetUserResponse$ = [3, n0, _GURe,
    0,
    [_U, _UA, _MFAO, _PMS, _UMFASL],
    [[() => UsernameType, 0], [() => AttributeListType, 0], () => MFAOptionListType, 0, 64 | 0]
];
var GlobalSignOutRequest$ = [3, n0, _GSOR,
    0,
    [_ATc],
    [[() => TokenModelType, 0]]
];
var GlobalSignOutResponse$ = [3, n0, _GSORl,
    0,
    [],
    []
];
var GroupExistsException$ = [-3, n0, _GEE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(GroupExistsException$, GroupExistsException);
var GroupType$ = [3, n0, _GT,
    0,
    [_GN, _UPI, _Desc, _RA, _Pr, _LMD, _CDr],
    [0, 0, 0, 0, 1, 4, 4]
];
var HttpHeader$ = [3, n0, _HHt,
    0,
    [_hN, _hV],
    [0, 0]
];
var IdentityProviderType$ = [3, n0, _IPT,
    0,
    [_UPI, _PN, _PTr, _PD, _AMt, _II, _LMD, _CDr],
    [0, 0, 0, 128 | 0, 128 | 0, 64 | 0, 4, 4]
];
var InitiateAuthRequest$ = [3, n0, _IAR,
    0,
    [_AF, _AP, _CM, _CI, _AM, _UCDs, _S],
    [0, [() => AuthParametersType, 0], 128 | 0, [() => ClientIdType, 0], () => AnalyticsMetadataType$, [() => UserContextDataType$, 0], [() => SessionType, 0]]
];
var InitiateAuthResponse$ = [3, n0, _IARn,
    0,
    [_CN, _S, _CP, _AR, _AC],
    [0, [() => SessionType, 0], 128 | 0, [() => AuthenticationResultType$, 0], 64 | 0]
];
var InternalErrorException$ = [-3, n0, _IEE,
    { [_e]: _s },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InternalErrorException$, InternalErrorException);
var InvalidEmailRoleAccessPolicyException$ = [-3, n0, _IERAPE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidEmailRoleAccessPolicyException$, InvalidEmailRoleAccessPolicyException);
var InvalidLambdaResponseException$ = [-3, n0, _ILRE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidLambdaResponseException$, InvalidLambdaResponseException);
var InvalidOAuthFlowException$ = [-3, n0, _IOAFE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidOAuthFlowException$, InvalidOAuthFlowException);
var InvalidParameterException$ = [-3, n0, _IPE,
    { [_e]: _c, [_hE]: 400 },
    [_m, _rC],
    [0, 0]
];
schema.TypeRegistry.for(n0).registerError(InvalidParameterException$, InvalidParameterException);
var InvalidPasswordException$ = [-3, n0, _IPEn,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidPasswordException$, InvalidPasswordException);
var InvalidSmsRoleAccessPolicyException$ = [-3, n0, _ISRAPE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidSmsRoleAccessPolicyException$, InvalidSmsRoleAccessPolicyException);
var InvalidSmsRoleTrustRelationshipException$ = [-3, n0, _ISRTRE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidSmsRoleTrustRelationshipException$, InvalidSmsRoleTrustRelationshipException);
var InvalidUserPoolConfigurationException$ = [-3, n0, _IUPCE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidUserPoolConfigurationException$, InvalidUserPoolConfigurationException);
var LambdaConfigType$ = [3, n0, _LCT,
    0,
    [_PSU, _CMu, _PC, _PA, _PAo, _DAC, _CAC, _VACR, _PTG, _UM, _PTGC, _CSMSS, _CES, _KMSKID],
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, () => PreTokenGenerationVersionConfigType$, () => CustomSMSLambdaVersionConfigType$, () => CustomEmailLambdaVersionConfigType$, 0]
];
var LimitExceededException$ = [-3, n0, _LEE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(LimitExceededException$, LimitExceededException);
var ListDevicesRequest$ = [3, n0, _LDR,
    0,
    [_ATc, _L, _PTa],
    [[() => TokenModelType, 0], 1, 0]
];
var ListDevicesResponse$ = [3, n0, _LDRi,
    0,
    [_De, _PTa],
    [[() => DeviceListType, 0], 0]
];
var ListGroupsRequest$ = [3, n0, _LGR,
    0,
    [_UPI, _L, _NT],
    [0, 1, 0]
];
var ListGroupsResponse$ = [3, n0, _LGRi,
    0,
    [_G, _NT],
    [() => GroupListType, 0]
];
var ListIdentityProvidersRequest$ = [3, n0, _LIPR,
    0,
    [_UPI, _MR, _NT],
    [0, 1, 0]
];
var ListIdentityProvidersResponse$ = [3, n0, _LIPRi,
    0,
    [_Prov, _NT],
    [() => ProvidersListType, 0]
];
var ListResourceServersRequest$ = [3, n0, _LRSR,
    0,
    [_UPI, _MR, _NT],
    [0, 1, 0]
];
var ListResourceServersResponse$ = [3, n0, _LRSRi,
    0,
    [_RSe, _NT],
    [() => ResourceServersListType, 0]
];
var ListTagsForResourceRequest$ = [3, n0, _LTFRR,
    0,
    [_RAes],
    [0]
];
var ListTagsForResourceResponse$ = [3, n0, _LTFRRi,
    0,
    [_Ta],
    [128 | 0]
];
var ListTermsRequest$ = [3, n0, _LTR,
    0,
    [_UPI, _MR, _NT],
    [0, 1, 0]
];
var ListTermsResponse$ = [3, n0, _LTRi,
    0,
    [_T, _NT],
    [() => TermsDescriptionListType, 0]
];
var ListUserImportJobsRequest$ = [3, n0, _LUIJR,
    0,
    [_UPI, _MR, _PTa],
    [0, 1, 0]
];
var ListUserImportJobsResponse$ = [3, n0, _LUIJRi,
    0,
    [_UIJs, _PTa],
    [() => UserImportJobsListType, 0]
];
var ListUserPoolClientsRequest$ = [3, n0, _LUPCR,
    0,
    [_UPI, _MR, _NT],
    [0, 1, 0]
];
var ListUserPoolClientsResponse$ = [3, n0, _LUPCRi,
    0,
    [_UPCs, _NT],
    [[() => UserPoolClientListType, 0], 0]
];
var ListUserPoolsRequest$ = [3, n0, _LUPR,
    0,
    [_NT, _MR],
    [0, 1]
];
var ListUserPoolsResponse$ = [3, n0, _LUPRi,
    0,
    [_UPs, _NT],
    [() => UserPoolListType, 0]
];
var ListUsersInGroupRequest$ = [3, n0, _LUIGR,
    0,
    [_UPI, _GN, _L, _NT],
    [0, 0, 1, 0]
];
var ListUsersInGroupResponse$ = [3, n0, _LUIGRi,
    0,
    [_Use, _NT],
    [[() => UsersListType, 0], 0]
];
var ListUsersRequest$ = [3, n0, _LUR,
    0,
    [_UPI, _ATG, _L, _PTa, _Fi],
    [0, 64 | 0, 1, 0, 0]
];
var ListUsersResponse$ = [3, n0, _LURi,
    0,
    [_Use, _PTa],
    [[() => UsersListType, 0], 0]
];
var ListWebAuthnCredentialsRequest$ = [3, n0, _LWACR,
    0,
    [_ATc, _NT, _MR],
    [[() => TokenModelType, 0], 0, 1]
];
var ListWebAuthnCredentialsResponse$ = [3, n0, _LWACRi,
    0,
    [_Cre, _NT],
    [() => WebAuthnCredentialDescriptionListType, 0]
];
var LogConfigurationType$ = [3, n0, _LCTo,
    0,
    [_LL, _ES, _CWLC, _SCo, _FC],
    [0, 0, () => CloudWatchLogsConfigurationType$, () => S3ConfigurationType$, () => FirehoseConfigurationType$]
];
var LogDeliveryConfigurationType$ = [3, n0, _LDCT,
    0,
    [_UPI, _LCo],
    [0, () => LogConfigurationListType]
];
var ManagedLoginBrandingExistsException$ = [-3, n0, _MLBEE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ManagedLoginBrandingExistsException$, ManagedLoginBrandingExistsException);
var ManagedLoginBrandingType$ = [3, n0, _MLBT,
    0,
    [_MLBI, _UPI, _UCPV, _Se, _As, _CDr, _LMD],
    [0, 0, 2, 15, () => AssetListType, 4, 4]
];
var MessageTemplateType$ = [3, n0, _MTT,
    0,
    [_SMSM, _EM, _ESm],
    [0, 0, 0]
];
var MFAMethodNotFoundException$ = [-3, n0, _MFAMNFE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(MFAMethodNotFoundException$, MFAMethodNotFoundException);
var MFAOptionType$ = [3, n0, _MFAOT,
    0,
    [_DM, _AN],
    [0, 0]
];
var NewDeviceMetadataType$ = [3, n0, _NDMT,
    0,
    [_DK, _DGK],
    [0, 0]
];
var NotAuthorizedException$ = [-3, n0, _NAE,
    { [_e]: _c, [_hE]: 403 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NotAuthorizedException$, NotAuthorizedException);
var NotifyConfigurationType$ = [3, n0, _NCT,
    0,
    [_F, _RTe, _SA, _BE, _NAEo, _ME],
    [0, 0, 0, () => NotifyEmailType$, () => NotifyEmailType$, () => NotifyEmailType$]
];
var NotifyEmailType$ = [3, n0, _NET,
    0,
    [_Su, _HB, _TB],
    [0, 0, 0]
];
var NumberAttributeConstraintsType$ = [3, n0, _NACT,
    0,
    [_MV, _MVa],
    [0, 0]
];
var PasswordHistoryPolicyViolationException$ = [-3, n0, _PHPVE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(PasswordHistoryPolicyViolationException$, PasswordHistoryPolicyViolationException);
var PasswordPolicyType$ = [3, n0, _PPT,
    0,
    [_ML, _RU, _RLe, _RN, _RSeq, _PHS, _TPVD],
    [1, 2, 2, 2, 2, 1, 1]
];
var PasswordResetRequiredException$ = [-3, n0, _PRRE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(PasswordResetRequiredException$, PasswordResetRequiredException);
var PreconditionNotMetException$ = [-3, n0, _PNME,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(PreconditionNotMetException$, PreconditionNotMetException);
var PreTokenGenerationVersionConfigType$ = [3, n0, _PTGVCT,
    0,
    [_LV, _LAa],
    [0, 0]
];
var ProviderDescription$ = [3, n0, _PDr,
    0,
    [_PN, _PTr, _LMD, _CDr],
    [0, 0, 4, 4]
];
var ProviderUserIdentifierType$ = [3, n0, _PUIT,
    0,
    [_PN, _PAN, _PAV],
    [0, 0, 0]
];
var RecoveryOptionType$ = [3, n0, _ROT,
    0,
    [_Pri, _Na],
    [1, 0]
];
var RefreshTokenReuseException$ = [-3, n0, _RTRE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(RefreshTokenReuseException$, RefreshTokenReuseException);
var RefreshTokenRotationType$ = [3, n0, _RTRT,
    0,
    [_Fe, _RGPS],
    [0, 1]
];
var ResendConfirmationCodeRequest$ = [3, n0, _RCCR,
    0,
    [_CI, _SH, _UCDs, _U, _AM, _CM],
    [[() => ClientIdType, 0], [() => SecretHashType, 0], [() => UserContextDataType$, 0], [() => UsernameType, 0], () => AnalyticsMetadataType$, 128 | 0]
];
var ResendConfirmationCodeResponse$ = [3, n0, _RCCRe,
    0,
    [_CDD],
    [() => CodeDeliveryDetailsType$]
];
var ResourceNotFoundException$ = [-3, n0, _RNFE,
    { [_e]: _c, [_hE]: 404 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ResourceNotFoundException$, ResourceNotFoundException);
var ResourceServerScopeType$ = [3, n0, _RSST,
    0,
    [_SNc, _SD],
    [0, 0]
];
var ResourceServerType$ = [3, n0, _RST,
    0,
    [_UPI, _I, _Na, _Sc],
    [0, 0, 0, () => ResourceServerScopeListType]
];
var RespondToAuthChallengeRequest$ = [3, n0, _RTACR,
    0,
    [_CI, _CN, _S, _CR, _AM, _UCDs, _CM],
    [[() => ClientIdType, 0], 0, [() => SessionType, 0], [() => ChallengeResponsesType, 0], () => AnalyticsMetadataType$, [() => UserContextDataType$, 0], 128 | 0]
];
var RespondToAuthChallengeResponse$ = [3, n0, _RTACRe,
    0,
    [_CN, _S, _CP, _AR],
    [0, [() => SessionType, 0], 128 | 0, [() => AuthenticationResultType$, 0]]
];
var RevokeTokenRequest$ = [3, n0, _RTRe,
    0,
    [_To, _CI, _CSl],
    [[() => TokenModelType, 0], [() => ClientIdType, 0], [() => ClientSecretType, 0]]
];
var RevokeTokenResponse$ = [3, n0, _RTRev,
    0,
    [],
    []
];
var RiskConfigurationType$ = [3, n0, _RCT,
    0,
    [_UPI, _CI, _CCRC, _ATRC, _REC, _LMD],
    [0, [() => ClientIdType, 0], () => CompromisedCredentialsRiskConfigurationType$, () => AccountTakeoverRiskConfigurationType$, () => RiskExceptionConfigurationType$, 4]
];
var RiskExceptionConfigurationType$ = [3, n0, _RECT,
    0,
    [_BIPRL, _SIPRL],
    [64 | 0, 64 | 0]
];
var S3ConfigurationType$ = [3, n0, _SCTo,
    0,
    [_BA],
    [0]
];
var SchemaAttributeType$ = [3, n0, _SAT,
    0,
    [_Na, _ADT, _DOA, _Mu, _R, _NAC, _SAC],
    [0, 0, 2, 2, 2, () => NumberAttributeConstraintsType$, () => StringAttributeConstraintsType$]
];
var ScopeDoesNotExistException$ = [-3, n0, _SDNEE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ScopeDoesNotExistException$, ScopeDoesNotExistException);
var SetLogDeliveryConfigurationRequest$ = [3, n0, _SLDCR,
    0,
    [_UPI, _LCo],
    [0, () => LogConfigurationListType]
];
var SetLogDeliveryConfigurationResponse$ = [3, n0, _SLDCRe,
    0,
    [_LDC],
    [() => LogDeliveryConfigurationType$]
];
var SetRiskConfigurationRequest$ = [3, n0, _SRCR,
    0,
    [_UPI, _CI, _CCRC, _ATRC, _REC],
    [0, [() => ClientIdType, 0], () => CompromisedCredentialsRiskConfigurationType$, () => AccountTakeoverRiskConfigurationType$, () => RiskExceptionConfigurationType$]
];
var SetRiskConfigurationResponse$ = [3, n0, _SRCRe,
    0,
    [_RC],
    [[() => RiskConfigurationType$, 0]]
];
var SetUICustomizationRequest$ = [3, n0, _SUICR,
    0,
    [_UPI, _CI, _CSS, _IF],
    [0, [() => ClientIdType, 0], 0, 21]
];
var SetUICustomizationResponse$ = [3, n0, _SUICRe,
    0,
    [_UIC],
    [[() => UICustomizationType$, 0]]
];
var SetUserMFAPreferenceRequest$ = [3, n0, _SUMFAPR,
    0,
    [_SMSMS, _STMS, _EMS, _ATc],
    [() => SMSMfaSettingsType$, () => SoftwareTokenMfaSettingsType$, () => EmailMfaSettingsType$, [() => TokenModelType, 0]]
];
var SetUserMFAPreferenceResponse$ = [3, n0, _SUMFAPRe,
    0,
    [],
    []
];
var SetUserPoolMfaConfigRequest$ = [3, n0, _SUPMCR,
    0,
    [_UPI, _SMC, _STMC, _EMC, _MC, _WAC],
    [0, () => SmsMfaConfigType$, () => SoftwareTokenMfaConfigType$, () => EmailMfaConfigType$, 0, () => WebAuthnConfigurationType$]
];
var SetUserPoolMfaConfigResponse$ = [3, n0, _SUPMCRe,
    0,
    [_SMC, _STMC, _EMC, _MC, _WAC],
    [() => SmsMfaConfigType$, () => SoftwareTokenMfaConfigType$, () => EmailMfaConfigType$, 0, () => WebAuthnConfigurationType$]
];
var SetUserSettingsRequest$ = [3, n0, _SUSR,
    0,
    [_ATc, _MFAO],
    [[() => TokenModelType, 0], () => MFAOptionListType]
];
var SetUserSettingsResponse$ = [3, n0, _SUSRe,
    0,
    [],
    []
];
var SignInPolicyType$ = [3, n0, _SIPT,
    0,
    [_AFAF],
    [64 | 0]
];
var SignUpRequest$ = [3, n0, _SUR,
    0,
    [_CI, _SH, _U, _P, _UA, _VD, _AM, _UCDs, _CM],
    [[() => ClientIdType, 0], [() => SecretHashType, 0], [() => UsernameType, 0], [() => PasswordType, 0], [() => AttributeListType, 0], [() => AttributeListType, 0], () => AnalyticsMetadataType$, [() => UserContextDataType$, 0], 128 | 0]
];
var SignUpResponse$ = [3, n0, _SURi,
    0,
    [_UCs, _CDD, _USs, _S],
    [2, () => CodeDeliveryDetailsType$, 0, [() => SessionType, 0]]
];
var SmsConfigurationType$ = [3, n0, _SCTm,
    0,
    [_SCA, _EIx, _SR],
    [0, 0, 0]
];
var SmsMfaConfigType$ = [3, n0, _SMCT,
    0,
    [_SAM, _SCm],
    [0, () => SmsConfigurationType$]
];
var SMSMfaSettingsType$ = [3, n0, _SMSMST,
    0,
    [_E, _PM],
    [2, 2]
];
var SoftwareTokenMfaConfigType$ = [3, n0, _STMCT,
    0,
    [_E],
    [2]
];
var SoftwareTokenMFANotFoundException$ = [-3, n0, _STMFANFE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(SoftwareTokenMFANotFoundException$, SoftwareTokenMFANotFoundException);
var SoftwareTokenMfaSettingsType$ = [3, n0, _STMST,
    0,
    [_E, _PM],
    [2, 2]
];
var StartUserImportJobRequest$ = [3, n0, _SUIJR,
    0,
    [_UPI, _JI],
    [0, 0]
];
var StartUserImportJobResponse$ = [3, n0, _SUIJRt,
    0,
    [_UIJ],
    [() => UserImportJobType$]
];
var StartWebAuthnRegistrationRequest$ = [3, n0, _SWARR,
    0,
    [_ATc],
    [[() => TokenModelType, 0]]
];
var StartWebAuthnRegistrationResponse$ = [3, n0, _SWARRt,
    0,
    [_CCO],
    [15]
];
var StopUserImportJobRequest$ = [3, n0, _SUIJRto,
    0,
    [_UPI, _JI],
    [0, 0]
];
var StopUserImportJobResponse$ = [3, n0, _SUIJRtop,
    0,
    [_UIJ],
    [() => UserImportJobType$]
];
var StringAttributeConstraintsType$ = [3, n0, _SACT,
    0,
    [_MLi, _MLa],
    [0, 0]
];
var TagResourceRequest$ = [3, n0, _TRR,
    0,
    [_RAes, _Ta],
    [0, 128 | 0]
];
var TagResourceResponse$ = [3, n0, _TRRa,
    0,
    [],
    []
];
var TermsDescriptionType$ = [3, n0, _TDT,
    0,
    [_TI, _TN, _En, _CDr, _LMD],
    [0, 0, 0, 4, 4]
];
var TermsExistsException$ = [-3, n0, _TEE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TermsExistsException$, TermsExistsException);
var TermsType$ = [3, n0, _TTe,
    0,
    [_TI, _UPI, _CI, _TN, _TS, _En, _Li, _CDr, _LMD],
    [0, 0, [() => ClientIdType, 0], 0, 0, 0, 128 | 0, 4, 4]
];
var TierChangeNotAllowedException$ = [-3, n0, _TCNAE,
    { [_e]: _c, [_hE]: 403 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TierChangeNotAllowedException$, TierChangeNotAllowedException);
var TokenValidityUnitsType$ = [3, n0, _TVUT,
    0,
    [_ATc, _IT, _RT],
    [0, 0, 0]
];
var TooManyFailedAttemptsException$ = [-3, n0, _TMFAE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyFailedAttemptsException$, TooManyFailedAttemptsException);
var TooManyRequestsException$ = [-3, n0, _TMRE,
    { [_e]: _c, [_hE]: 429 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyRequestsException$, TooManyRequestsException);
var UICustomizationType$ = [3, n0, _UICT,
    0,
    [_UPI, _CI, _IU, _CSS, _CSSV, _LMD, _CDr],
    [0, [() => ClientIdType, 0], 0, 0, 0, 4, 4]
];
var UnauthorizedException$ = [-3, n0, _UE,
    { [_e]: _c, [_hE]: 401 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(UnauthorizedException$, UnauthorizedException);
var UnexpectedLambdaException$ = [-3, n0, _ULE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(UnexpectedLambdaException$, UnexpectedLambdaException);
var UnsupportedIdentityProviderException$ = [-3, n0, _UIPE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(UnsupportedIdentityProviderException$, UnsupportedIdentityProviderException);
var UnsupportedOperationException$ = [-3, n0, _UOE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(UnsupportedOperationException$, UnsupportedOperationException);
var UnsupportedTokenTypeException$ = [-3, n0, _UTTE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(UnsupportedTokenTypeException$, UnsupportedTokenTypeException);
var UnsupportedUserStateException$ = [-3, n0, _UUSE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(UnsupportedUserStateException$, UnsupportedUserStateException);
var UntagResourceRequest$ = [3, n0, _URR,
    0,
    [_RAes, _TK],
    [0, 64 | 0]
];
var UntagResourceResponse$ = [3, n0, _URRn,
    0,
    [],
    []
];
var UpdateAuthEventFeedbackRequest$ = [3, n0, _UAEFR,
    0,
    [_UPI, _U, _EI, _FT, _FV],
    [0, [() => UsernameType, 0], 0, [() => TokenModelType, 0], 0]
];
var UpdateAuthEventFeedbackResponse$ = [3, n0, _UAEFRp,
    0,
    [],
    []
];
var UpdateDeviceStatusRequest$ = [3, n0, _UDSR,
    0,
    [_ATc, _DK, _DRS],
    [[() => TokenModelType, 0], 0, 0]
];
var UpdateDeviceStatusResponse$ = [3, n0, _UDSRp,
    0,
    [],
    []
];
var UpdateGroupRequest$ = [3, n0, _UGR,
    0,
    [_GN, _UPI, _Desc, _RA, _Pr],
    [0, 0, 0, 0, 1]
];
var UpdateGroupResponse$ = [3, n0, _UGRp,
    0,
    [_Gr],
    [() => GroupType$]
];
var UpdateIdentityProviderRequest$ = [3, n0, _UIPR,
    0,
    [_UPI, _PN, _PD, _AMt, _II],
    [0, 0, 128 | 0, 128 | 0, 64 | 0]
];
var UpdateIdentityProviderResponse$ = [3, n0, _UIPRp,
    0,
    [_IP],
    [() => IdentityProviderType$]
];
var UpdateManagedLoginBrandingRequest$ = [3, n0, _UMLBR,
    0,
    [_UPI, _MLBI, _UCPV, _Se, _As],
    [0, 0, 2, 15, () => AssetListType]
];
var UpdateManagedLoginBrandingResponse$ = [3, n0, _UMLBRp,
    0,
    [_MLB],
    [() => ManagedLoginBrandingType$]
];
var UpdateResourceServerRequest$ = [3, n0, _URSR,
    0,
    [_UPI, _I, _Na, _Sc],
    [0, 0, 0, () => ResourceServerScopeListType]
];
var UpdateResourceServerResponse$ = [3, n0, _URSRp,
    0,
    [_RS],
    [() => ResourceServerType$]
];
var UpdateTermsRequest$ = [3, n0, _UTR,
    0,
    [_TI, _UPI, _TN, _TS, _En, _Li],
    [0, 0, 0, 0, 0, 128 | 0]
];
var UpdateTermsResponse$ = [3, n0, _UTRp,
    0,
    [_T],
    [[() => TermsType$, 0]]
];
var UpdateUserAttributesRequest$ = [3, n0, _UUAR,
    0,
    [_UA, _ATc, _CM],
    [[() => AttributeListType, 0], [() => TokenModelType, 0], 128 | 0]
];
var UpdateUserAttributesResponse$ = [3, n0, _UUARp,
    0,
    [_CDDL],
    [() => CodeDeliveryDetailsListType]
];
var UpdateUserPoolClientRequest$ = [3, n0, _UUPCR,
    0,
    [_UPI, _CI, _CNl, _RTV, _ATV, _ITV, _TVU, _RAe, _WA, _EAF, _SIP, _CURL, _LURL, _DRURI, _AOAF, _AOAS, _AOAFUPC, _ACn, _PUEE, _ETR, _EPAUCD, _ASV, _RTR],
    [0, [() => ClientIdType, 0], 0, 1, 1, 1, () => TokenValidityUnitsType$, 64 | 0, 64 | 0, 64 | 0, 64 | 0, 64 | 0, 64 | 0, 0, 64 | 0, 64 | 0, 2, () => AnalyticsConfigurationType$, 0, 2, 2, 1, () => RefreshTokenRotationType$]
];
var UpdateUserPoolClientResponse$ = [3, n0, _UUPCRp,
    0,
    [_UPC],
    [[() => UserPoolClientType$, 0]]
];
var UpdateUserPoolDomainRequest$ = [3, n0, _UUPDR,
    0,
    [_Do, _UPI, _MLV, _CDC],
    [0, 0, 1, () => CustomDomainConfigType$]
];
var UpdateUserPoolDomainResponse$ = [3, n0, _UUPDRp,
    0,
    [_MLV, _CFD],
    [1, 0]
];
var UpdateUserPoolRequest$ = [3, n0, _UUPR,
    0,
    [_UPI, _Po, _DP, _LC, _AVA, _SVM, _EVM, _EVS, _VMT, _SAM, _UAUS, _MC, _DC, _EC, _SCm, _UPT, _ACUC, _UPAO, _ARS, _PNo, _UPTs],
    [0, () => UserPoolPolicyType$, 0, () => LambdaConfigType$, 64 | 0, 0, 0, 0, () => VerificationMessageTemplateType$, 0, () => UserAttributeUpdateSettingsType$, 0, () => DeviceConfigurationType$, () => EmailConfigurationType$, () => SmsConfigurationType$, 128 | 0, () => AdminCreateUserConfigType$, () => UserPoolAddOnsType$, () => AccountRecoverySettingType$, 0, 0]
];
var UpdateUserPoolResponse$ = [3, n0, _UUPRp,
    0,
    [],
    []
];
var UserAttributeUpdateSettingsType$ = [3, n0, _UAUST,
    0,
    [_ARVBU],
    [64 | 0]
];
var UserContextDataType$ = [3, n0, _UCDT,
    8,
    [_IA, _ED],
    [0, 0]
];
var UserImportInProgressException$ = [-3, n0, _UIIPE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(UserImportInProgressException$, UserImportInProgressException);
var UserImportJobType$ = [3, n0, _UIJT,
    0,
    [_JN, _JI, _UPI, _PSUr, _CDr, _SDt, _CDo, _St, _CWLRA, _IUm, _SUk, _FU, _CMom],
    [0, 0, 0, 0, 4, 4, 4, 0, 0, 1, 1, 1, 0]
];
var UserLambdaValidationException$ = [-3, n0, _ULVE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(UserLambdaValidationException$, UserLambdaValidationException);
var UsernameConfigurationType$ = [3, n0, _UCT,
    0,
    [_CSa],
    [2]
];
var UsernameExistsException$ = [-3, n0, _UEE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(UsernameExistsException$, UsernameExistsException);
var UserNotConfirmedException$ = [-3, n0, _UNCE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(UserNotConfirmedException$, UserNotConfirmedException);
var UserNotFoundException$ = [-3, n0, _UNFE,
    { [_e]: _c, [_hE]: 404 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(UserNotFoundException$, UserNotFoundException);
var UserPoolAddOnNotEnabledException$ = [-3, n0, _UPAONEE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(UserPoolAddOnNotEnabledException$, UserPoolAddOnNotEnabledException);
var UserPoolAddOnsType$ = [3, n0, _UPAOT,
    0,
    [_ASM, _ASAF],
    [0, () => AdvancedSecurityAdditionalFlowsType$]
];
var UserPoolClientDescription$ = [3, n0, _UPCD,
    0,
    [_CI, _UPI, _CNl],
    [[() => ClientIdType, 0], 0, 0]
];
var UserPoolClientType$ = [3, n0, _UPCT,
    0,
    [_UPI, _CNl, _CI, _CSl, _LMD, _CDr, _RTV, _ATV, _ITV, _TVU, _RAe, _WA, _EAF, _SIP, _CURL, _LURL, _DRURI, _AOAF, _AOAS, _AOAFUPC, _ACn, _PUEE, _ETR, _EPAUCD, _ASV, _RTR],
    [0, 0, [() => ClientIdType, 0], [() => ClientSecretType, 0], 4, 4, 1, 1, 1, () => TokenValidityUnitsType$, 64 | 0, 64 | 0, 64 | 0, 64 | 0, 64 | 0, 64 | 0, 0, 64 | 0, 64 | 0, 2, () => AnalyticsConfigurationType$, 0, 2, 2, 1, () => RefreshTokenRotationType$]
];
var UserPoolDescriptionType$ = [3, n0, _UPDT,
    0,
    [_Id, _Na, _LC, _St, _LMD, _CDr],
    [0, 0, () => LambdaConfigType$, 0, 4, 4]
];
var UserPoolPolicyType$ = [3, n0, _UPPT,
    0,
    [_PPa, _SIPi],
    [() => PasswordPolicyType$, () => SignInPolicyType$]
];
var UserPoolTaggingException$ = [-3, n0, _UPTE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(UserPoolTaggingException$, UserPoolTaggingException);
var UserPoolType$ = [3, n0, _UPTse,
    0,
    [_Id, _Na, _Po, _DP, _LC, _St, _LMD, _CDr, _SAc, _AVA, _AAl, _UAs, _SVM, _EVM, _EVS, _VMT, _SAM, _UAUS, _MC, _DC, _ENOU, _EC, _SCm, _UPT, _SCF, _ECF, _Do, _CDu, _ACUC, _UPAO, _UC, _Ar, _ARS, _UPTs],
    [0, 0, () => UserPoolPolicyType$, 0, () => LambdaConfigType$, 0, 4, 4, () => SchemaAttributesListType, 64 | 0, 64 | 0, 64 | 0, 0, 0, 0, () => VerificationMessageTemplateType$, 0, () => UserAttributeUpdateSettingsType$, 0, () => DeviceConfigurationType$, 1, () => EmailConfigurationType$, () => SmsConfigurationType$, 128 | 0, 0, 0, 0, 0, () => AdminCreateUserConfigType$, () => UserPoolAddOnsType$, () => UsernameConfigurationType$, 0, () => AccountRecoverySettingType$, 0]
];
var UserType$ = [3, n0, _UTs,
    0,
    [_U, _At, _UCD, _ULMD, _E, _US, _MFAO],
    [[() => UsernameType, 0], [() => AttributeListType, 0], 4, 4, 2, 0, () => MFAOptionListType]
];
var VerificationMessageTemplateType$ = [3, n0, _VMTT,
    0,
    [_SM, _EM, _ESm, _EMBL, _ESBL, _DEO],
    [0, 0, 0, 0, 0, 0]
];
var VerifySoftwareTokenRequest$ = [3, n0, _VSTR,
    0,
    [_ATc, _S, _UCse, _FDN],
    [[() => TokenModelType, 0], [() => SessionType, 0], [() => SoftwareTokenMFAUserCodeType, 0], 0]
];
var VerifySoftwareTokenResponse$ = [3, n0, _VSTRe,
    0,
    [_St, _S],
    [0, [() => SessionType, 0]]
];
var VerifyUserAttributeRequest$ = [3, n0, _VUAR,
    0,
    [_ATc, _AN, _Cod],
    [[() => TokenModelType, 0], 0, 0]
];
var VerifyUserAttributeResponse$ = [3, n0, _VUARe,
    0,
    [],
    []
];
var WebAuthnChallengeNotFoundException$ = [-3, n0, _WACNFE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(WebAuthnChallengeNotFoundException$, WebAuthnChallengeNotFoundException);
var WebAuthnClientMismatchException$ = [-3, n0, _WACME,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(WebAuthnClientMismatchException$, WebAuthnClientMismatchException);
var WebAuthnConfigurationMissingException$ = [-3, n0, _WACMEe,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(WebAuthnConfigurationMissingException$, WebAuthnConfigurationMissingException);
var WebAuthnConfigurationType$ = [3, n0, _WACT,
    0,
    [_RPI, _UV],
    [0, 0]
];
var WebAuthnCredentialDescription$ = [3, n0, _WACD,
    0,
    [_CIr, _FCN, _RPI, _AAu, _ATu, _CAr],
    [0, 0, 0, 0, 64 | 0, 4]
];
var WebAuthnCredentialNotSupportedException$ = [-3, n0, _WACNSE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(WebAuthnCredentialNotSupportedException$, WebAuthnCredentialNotSupportedException);
var WebAuthnNotEnabledException$ = [-3, n0, _WANEE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(WebAuthnNotEnabledException$, WebAuthnNotEnabledException);
var WebAuthnOriginNotAllowedException$ = [-3, n0, _WAONAE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(WebAuthnOriginNotAllowedException$, WebAuthnOriginNotAllowedException);
var WebAuthnRelyingPartyMismatchException$ = [-3, n0, _WARPME,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(WebAuthnRelyingPartyMismatchException$, WebAuthnRelyingPartyMismatchException);
var __Unit = "unit";
var CognitoIdentityProviderServiceException$ = [-3, _sm, "CognitoIdentityProviderServiceException", 0, [], []];
schema.TypeRegistry.for(_sm).registerError(CognitoIdentityProviderServiceException$, CognitoIdentityProviderServiceException);
var AssetListType = [1, n0, _ALT,
    0, () => AssetType$
];
var AttributeListType = [1, n0, _ALTt,
    0, [() => AttributeType$,
        0]
];
var AuthEventsType = [1, n0, _AETu,
    0, () => AuthEventType$
];
var ChallengeResponseListType = [1, n0, _CRLT,
    0, () => ChallengeResponseType$
];
var CodeDeliveryDetailsListType = [1, n0, _CDDLT,
    0, () => CodeDeliveryDetailsType$
];
var CustomAttributesListType = [1, n0, _CALT,
    0, () => SchemaAttributeType$
];
var DeviceListType = [1, n0, _DLT,
    0, [() => DeviceType$,
        0]
];
var GroupListType = [1, n0, _GLT,
    0, () => GroupType$
];
var HttpHeaderList = [1, n0, _HHL,
    0, () => HttpHeader$
];
var LogConfigurationListType = [1, n0, _LCLT,
    0, () => LogConfigurationType$
];
var MFAOptionListType = [1, n0, _MFAOLT,
    0, () => MFAOptionType$
];
var ProvidersListType = [1, n0, _PLT,
    0, () => ProviderDescription$
];
var RecoveryMechanismsType = [1, n0, _RMT,
    0, () => RecoveryOptionType$
];
var ResourceServerScopeListType = [1, n0, _RSSLT,
    0, () => ResourceServerScopeType$
];
var ResourceServersListType = [1, n0, _RSLT,
    0, () => ResourceServerType$
];
var SchemaAttributesListType = [1, n0, _SALT,
    0, () => SchemaAttributeType$
];
var TermsDescriptionListType = [1, n0, _TDLT,
    0, () => TermsDescriptionType$
];
var UserImportJobsListType = [1, n0, _UIJLT,
    0, () => UserImportJobType$
];
var UserPoolClientListType = [1, n0, _UPCLT,
    0, [() => UserPoolClientDescription$,
        0]
];
var UserPoolListType = [1, n0, _UPLT,
    0, () => UserPoolDescriptionType$
];
var UsersListType = [1, n0, _ULT,
    0, [() => UserType$,
        0]
];
var WebAuthnCredentialDescriptionListType = [1, n0, _WACDLT,
    0, () => WebAuthnCredentialDescription$
];
var AuthParametersType = [2, n0, _APT,
    8, 0, 0
];
var ChallengeResponsesType = [2, n0, _CRTh,
    8, 0, 0
];
var AddCustomAttributes$ = [9, n0, _ACA,
    0, () => AddCustomAttributesRequest$, () => AddCustomAttributesResponse$
];
var AdminAddUserToGroup$ = [9, n0, _AAUTG,
    0, () => AdminAddUserToGroupRequest$, () => __Unit
];
var AdminConfirmSignUp$ = [9, n0, _ACSU,
    0, () => AdminConfirmSignUpRequest$, () => AdminConfirmSignUpResponse$
];
var AdminCreateUser$ = [9, n0, _ACU,
    0, () => AdminCreateUserRequest$, () => AdminCreateUserResponse$
];
var AdminDeleteUser$ = [9, n0, _ADU,
    0, () => AdminDeleteUserRequest$, () => __Unit
];
var AdminDeleteUserAttributes$ = [9, n0, _ADUA,
    0, () => AdminDeleteUserAttributesRequest$, () => AdminDeleteUserAttributesResponse$
];
var AdminDisableProviderForUser$ = [9, n0, _ADPFU,
    0, () => AdminDisableProviderForUserRequest$, () => AdminDisableProviderForUserResponse$
];
var AdminDisableUser$ = [9, n0, _ADUd,
    0, () => AdminDisableUserRequest$, () => AdminDisableUserResponse$
];
var AdminEnableUser$ = [9, n0, _AEU,
    0, () => AdminEnableUserRequest$, () => AdminEnableUserResponse$
];
var AdminForgetDevice$ = [9, n0, _AFD,
    0, () => AdminForgetDeviceRequest$, () => __Unit
];
var AdminGetDevice$ = [9, n0, _AGD,
    0, () => AdminGetDeviceRequest$, () => AdminGetDeviceResponse$
];
var AdminGetUser$ = [9, n0, _AGU,
    0, () => AdminGetUserRequest$, () => AdminGetUserResponse$
];
var AdminInitiateAuth$ = [9, n0, _AIA,
    0, () => AdminInitiateAuthRequest$, () => AdminInitiateAuthResponse$
];
var AdminLinkProviderForUser$ = [9, n0, _ALPFU,
    0, () => AdminLinkProviderForUserRequest$, () => AdminLinkProviderForUserResponse$
];
var AdminListDevices$ = [9, n0, _ALD,
    0, () => AdminListDevicesRequest$, () => AdminListDevicesResponse$
];
var AdminListGroupsForUser$ = [9, n0, _ALGFU,
    0, () => AdminListGroupsForUserRequest$, () => AdminListGroupsForUserResponse$
];
var AdminListUserAuthEvents$ = [9, n0, _ALUAE,
    0, () => AdminListUserAuthEventsRequest$, () => AdminListUserAuthEventsResponse$
];
var AdminRemoveUserFromGroup$ = [9, n0, _ARUFG,
    0, () => AdminRemoveUserFromGroupRequest$, () => __Unit
];
var AdminResetUserPassword$ = [9, n0, _ARUP,
    0, () => AdminResetUserPasswordRequest$, () => AdminResetUserPasswordResponse$
];
var AdminRespondToAuthChallenge$ = [9, n0, _ARTAC,
    0, () => AdminRespondToAuthChallengeRequest$, () => AdminRespondToAuthChallengeResponse$
];
var AdminSetUserMFAPreference$ = [9, n0, _ASUMFAP,
    0, () => AdminSetUserMFAPreferenceRequest$, () => AdminSetUserMFAPreferenceResponse$
];
var AdminSetUserPassword$ = [9, n0, _ASUP,
    0, () => AdminSetUserPasswordRequest$, () => AdminSetUserPasswordResponse$
];
var AdminSetUserSettings$ = [9, n0, _ASUS,
    0, () => AdminSetUserSettingsRequest$, () => AdminSetUserSettingsResponse$
];
var AdminUpdateAuthEventFeedback$ = [9, n0, _AUAEF,
    0, () => AdminUpdateAuthEventFeedbackRequest$, () => AdminUpdateAuthEventFeedbackResponse$
];
var AdminUpdateDeviceStatus$ = [9, n0, _AUDS,
    0, () => AdminUpdateDeviceStatusRequest$, () => AdminUpdateDeviceStatusResponse$
];
var AdminUpdateUserAttributes$ = [9, n0, _AUUA,
    0, () => AdminUpdateUserAttributesRequest$, () => AdminUpdateUserAttributesResponse$
];
var AdminUserGlobalSignOut$ = [9, n0, _AUGSO,
    0, () => AdminUserGlobalSignOutRequest$, () => AdminUserGlobalSignOutResponse$
];
var AssociateSoftwareToken$ = [9, n0, _AST,
    0, () => AssociateSoftwareTokenRequest$, () => AssociateSoftwareTokenResponse$
];
var ChangePassword$ = [9, n0, _CPh,
    0, () => ChangePasswordRequest$, () => ChangePasswordResponse$
];
var CompleteWebAuthnRegistration$ = [9, n0, _CWAR,
    0, () => CompleteWebAuthnRegistrationRequest$, () => CompleteWebAuthnRegistrationResponse$
];
var ConfirmDevice$ = [9, n0, _CDon,
    0, () => ConfirmDeviceRequest$, () => ConfirmDeviceResponse$
];
var ConfirmForgotPassword$ = [9, n0, _CFP,
    0, () => ConfirmForgotPasswordRequest$, () => ConfirmForgotPasswordResponse$
];
var ConfirmSignUp$ = [9, n0, _CSU,
    0, () => ConfirmSignUpRequest$, () => ConfirmSignUpResponse$
];
var CreateGroup$ = [9, n0, _CG,
    0, () => CreateGroupRequest$, () => CreateGroupResponse$
];
var CreateIdentityProvider$ = [9, n0, _CIP,
    0, () => CreateIdentityProviderRequest$, () => CreateIdentityProviderResponse$
];
var CreateManagedLoginBranding$ = [9, n0, _CMLB,
    0, () => CreateManagedLoginBrandingRequest$, () => CreateManagedLoginBrandingResponse$
];
var CreateResourceServer$ = [9, n0, _CRS,
    0, () => CreateResourceServerRequest$, () => CreateResourceServerResponse$
];
var CreateTerms$ = [9, n0, _CT,
    0, () => CreateTermsRequest$, () => CreateTermsResponse$
];
var CreateUserImportJob$ = [9, n0, _CUIJ,
    0, () => CreateUserImportJobRequest$, () => CreateUserImportJobResponse$
];
var CreateUserPool$ = [9, n0, _CUP,
    0, () => CreateUserPoolRequest$, () => CreateUserPoolResponse$
];
var CreateUserPoolClient$ = [9, n0, _CUPC,
    0, () => CreateUserPoolClientRequest$, () => CreateUserPoolClientResponse$
];
var CreateUserPoolDomain$ = [9, n0, _CUPD,
    0, () => CreateUserPoolDomainRequest$, () => CreateUserPoolDomainResponse$
];
var DeleteGroup$ = [9, n0, _DG,
    0, () => DeleteGroupRequest$, () => __Unit
];
var DeleteIdentityProvider$ = [9, n0, _DIP,
    0, () => DeleteIdentityProviderRequest$, () => __Unit
];
var DeleteManagedLoginBranding$ = [9, n0, _DMLB,
    0, () => DeleteManagedLoginBrandingRequest$, () => __Unit
];
var DeleteResourceServer$ = [9, n0, _DRSe,
    0, () => DeleteResourceServerRequest$, () => __Unit
];
var DeleteTerms$ = [9, n0, _DTe,
    0, () => DeleteTermsRequest$, () => __Unit
];
var DeleteUser$ = [9, n0, _DUe,
    0, () => DeleteUserRequest$, () => __Unit
];
var DeleteUserAttributes$ = [9, n0, _DUA,
    0, () => DeleteUserAttributesRequest$, () => DeleteUserAttributesResponse$
];
var DeleteUserPool$ = [9, n0, _DUP,
    0, () => DeleteUserPoolRequest$, () => __Unit
];
var DeleteUserPoolClient$ = [9, n0, _DUPC,
    0, () => DeleteUserPoolClientRequest$, () => __Unit
];
var DeleteUserPoolDomain$ = [9, n0, _DUPD,
    0, () => DeleteUserPoolDomainRequest$, () => DeleteUserPoolDomainResponse$
];
var DeleteWebAuthnCredential$ = [9, n0, _DWAC,
    0, () => DeleteWebAuthnCredentialRequest$, () => DeleteWebAuthnCredentialResponse$
];
var DescribeIdentityProvider$ = [9, n0, _DIPe,
    0, () => DescribeIdentityProviderRequest$, () => DescribeIdentityProviderResponse$
];
var DescribeManagedLoginBranding$ = [9, n0, _DMLBe,
    0, () => DescribeManagedLoginBrandingRequest$, () => DescribeManagedLoginBrandingResponse$
];
var DescribeManagedLoginBrandingByClient$ = [9, n0, _DMLBBC,
    0, () => DescribeManagedLoginBrandingByClientRequest$, () => DescribeManagedLoginBrandingByClientResponse$
];
var DescribeResourceServer$ = [9, n0, _DRSes,
    0, () => DescribeResourceServerRequest$, () => DescribeResourceServerResponse$
];
var DescribeRiskConfiguration$ = [9, n0, _DRC,
    0, () => DescribeRiskConfigurationRequest$, () => DescribeRiskConfigurationResponse$
];
var DescribeTerms$ = [9, n0, _DTes,
    0, () => DescribeTermsRequest$, () => DescribeTermsResponse$
];
var DescribeUserImportJob$ = [9, n0, _DUIJ,
    0, () => DescribeUserImportJobRequest$, () => DescribeUserImportJobResponse$
];
var DescribeUserPool$ = [9, n0, _DUPe,
    0, () => DescribeUserPoolRequest$, () => DescribeUserPoolResponse$
];
var DescribeUserPoolClient$ = [9, n0, _DUPCe,
    0, () => DescribeUserPoolClientRequest$, () => DescribeUserPoolClientResponse$
];
var DescribeUserPoolDomain$ = [9, n0, _DUPDe,
    0, () => DescribeUserPoolDomainRequest$, () => DescribeUserPoolDomainResponse$
];
var ForgetDevice$ = [9, n0, _FDo,
    0, () => ForgetDeviceRequest$, () => __Unit
];
var ForgotPassword$ = [9, n0, _FP,
    0, () => ForgotPasswordRequest$, () => ForgotPasswordResponse$
];
var GetCSVHeader$ = [9, n0, _GCSVH,
    0, () => GetCSVHeaderRequest$, () => GetCSVHeaderResponse$
];
var GetDevice$ = [9, n0, _GD,
    0, () => GetDeviceRequest$, () => GetDeviceResponse$
];
var GetGroup$ = [9, n0, _GG,
    0, () => GetGroupRequest$, () => GetGroupResponse$
];
var GetIdentityProviderByIdentifier$ = [9, n0, _GIPBI,
    0, () => GetIdentityProviderByIdentifierRequest$, () => GetIdentityProviderByIdentifierResponse$
];
var GetLogDeliveryConfiguration$ = [9, n0, _GLDC,
    0, () => GetLogDeliveryConfigurationRequest$, () => GetLogDeliveryConfigurationResponse$
];
var GetSigningCertificate$ = [9, n0, _GSC,
    0, () => GetSigningCertificateRequest$, () => GetSigningCertificateResponse$
];
var GetTokensFromRefreshToken$ = [9, n0, _GTFRT,
    0, () => GetTokensFromRefreshTokenRequest$, () => GetTokensFromRefreshTokenResponse$
];
var GetUICustomization$ = [9, n0, _GUIC,
    0, () => GetUICustomizationRequest$, () => GetUICustomizationResponse$
];
var GetUser$ = [9, n0, _GU,
    0, () => GetUserRequest$, () => GetUserResponse$
];
var GetUserAttributeVerificationCode$ = [9, n0, _GUAVC,
    0, () => GetUserAttributeVerificationCodeRequest$, () => GetUserAttributeVerificationCodeResponse$
];
var GetUserAuthFactors$ = [9, n0, _GUAF,
    0, () => GetUserAuthFactorsRequest$, () => GetUserAuthFactorsResponse$
];
var GetUserPoolMfaConfig$ = [9, n0, _GUPMC,
    0, () => GetUserPoolMfaConfigRequest$, () => GetUserPoolMfaConfigResponse$
];
var GlobalSignOut$ = [9, n0, _GSO,
    0, () => GlobalSignOutRequest$, () => GlobalSignOutResponse$
];
var InitiateAuth$ = [9, n0, _IAn,
    0, () => InitiateAuthRequest$, () => InitiateAuthResponse$
];
var ListDevices$ = [9, n0, _LD,
    0, () => ListDevicesRequest$, () => ListDevicesResponse$
];
var ListGroups$ = [9, n0, _LG,
    0, () => ListGroupsRequest$, () => ListGroupsResponse$
];
var ListIdentityProviders$ = [9, n0, _LIP,
    0, () => ListIdentityProvidersRequest$, () => ListIdentityProvidersResponse$
];
var ListResourceServers$ = [9, n0, _LRS,
    0, () => ListResourceServersRequest$, () => ListResourceServersResponse$
];
var ListTagsForResource$ = [9, n0, _LTFR,
    0, () => ListTagsForResourceRequest$, () => ListTagsForResourceResponse$
];
var ListTerms$ = [9, n0, _LT,
    0, () => ListTermsRequest$, () => ListTermsResponse$
];
var ListUserImportJobs$ = [9, n0, _LUIJ,
    0, () => ListUserImportJobsRequest$, () => ListUserImportJobsResponse$
];
var ListUserPoolClients$ = [9, n0, _LUPC,
    0, () => ListUserPoolClientsRequest$, () => ListUserPoolClientsResponse$
];
var ListUserPools$ = [9, n0, _LUP,
    0, () => ListUserPoolsRequest$, () => ListUserPoolsResponse$
];
var ListUsers$ = [9, n0, _LU,
    0, () => ListUsersRequest$, () => ListUsersResponse$
];
var ListUsersInGroup$ = [9, n0, _LUIG,
    0, () => ListUsersInGroupRequest$, () => ListUsersInGroupResponse$
];
var ListWebAuthnCredentials$ = [9, n0, _LWAC,
    0, () => ListWebAuthnCredentialsRequest$, () => ListWebAuthnCredentialsResponse$
];
var ResendConfirmationCode$ = [9, n0, _RCC,
    0, () => ResendConfirmationCodeRequest$, () => ResendConfirmationCodeResponse$
];
var RespondToAuthChallenge$ = [9, n0, _RTAC,
    0, () => RespondToAuthChallengeRequest$, () => RespondToAuthChallengeResponse$
];
var RevokeToken$ = [9, n0, _RTev,
    0, () => RevokeTokenRequest$, () => RevokeTokenResponse$
];
var SetLogDeliveryConfiguration$ = [9, n0, _SLDC,
    0, () => SetLogDeliveryConfigurationRequest$, () => SetLogDeliveryConfigurationResponse$
];
var SetRiskConfiguration$ = [9, n0, _SRC,
    0, () => SetRiskConfigurationRequest$, () => SetRiskConfigurationResponse$
];
var SetUICustomization$ = [9, n0, _SUIC,
    0, () => SetUICustomizationRequest$, () => SetUICustomizationResponse$
];
var SetUserMFAPreference$ = [9, n0, _SUMFAP,
    0, () => SetUserMFAPreferenceRequest$, () => SetUserMFAPreferenceResponse$
];
var SetUserPoolMfaConfig$ = [9, n0, _SUPMC,
    0, () => SetUserPoolMfaConfigRequest$, () => SetUserPoolMfaConfigResponse$
];
var SetUserSettings$ = [9, n0, _SUS,
    0, () => SetUserSettingsRequest$, () => SetUserSettingsResponse$
];
var SignUp$ = [9, n0, _SUi,
    0, () => SignUpRequest$, () => SignUpResponse$
];
var StartUserImportJob$ = [9, n0, _SUIJ,
    0, () => StartUserImportJobRequest$, () => StartUserImportJobResponse$
];
var StartWebAuthnRegistration$ = [9, n0, _SWAR,
    0, () => StartWebAuthnRegistrationRequest$, () => StartWebAuthnRegistrationResponse$
];
var StopUserImportJob$ = [9, n0, _SUIJt,
    0, () => StopUserImportJobRequest$, () => StopUserImportJobResponse$
];
var TagResource$ = [9, n0, _TR,
    0, () => TagResourceRequest$, () => TagResourceResponse$
];
var UntagResource$ = [9, n0, _UR,
    0, () => UntagResourceRequest$, () => UntagResourceResponse$
];
var UpdateAuthEventFeedback$ = [9, n0, _UAEF,
    0, () => UpdateAuthEventFeedbackRequest$, () => UpdateAuthEventFeedbackResponse$
];
var UpdateDeviceStatus$ = [9, n0, _UDSp,
    0, () => UpdateDeviceStatusRequest$, () => UpdateDeviceStatusResponse$
];
var UpdateGroup$ = [9, n0, _UG,
    0, () => UpdateGroupRequest$, () => UpdateGroupResponse$
];
var UpdateIdentityProvider$ = [9, n0, _UIP,
    0, () => UpdateIdentityProviderRequest$, () => UpdateIdentityProviderResponse$
];
var UpdateManagedLoginBranding$ = [9, n0, _UMLB,
    0, () => UpdateManagedLoginBrandingRequest$, () => UpdateManagedLoginBrandingResponse$
];
var UpdateResourceServer$ = [9, n0, _URS,
    0, () => UpdateResourceServerRequest$, () => UpdateResourceServerResponse$
];
var UpdateTerms$ = [9, n0, _UTp,
    0, () => UpdateTermsRequest$, () => UpdateTermsResponse$
];
var UpdateUserAttributes$ = [9, n0, _UUA,
    0, () => UpdateUserAttributesRequest$, () => UpdateUserAttributesResponse$
];
var UpdateUserPool$ = [9, n0, _UUP,
    0, () => UpdateUserPoolRequest$, () => UpdateUserPoolResponse$
];
var UpdateUserPoolClient$ = [9, n0, _UUPC,
    0, () => UpdateUserPoolClientRequest$, () => UpdateUserPoolClientResponse$
];
var UpdateUserPoolDomain$ = [9, n0, _UUPD,
    0, () => UpdateUserPoolDomainRequest$, () => UpdateUserPoolDomainResponse$
];
var VerifySoftwareToken$ = [9, n0, _VST,
    0, () => VerifySoftwareTokenRequest$, () => VerifySoftwareTokenResponse$
];
var VerifyUserAttribute$ = [9, n0, _VUA,
    0, () => VerifyUserAttributeRequest$, () => VerifyUserAttributeResponse$
];

class AddCustomAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AddCustomAttributes", {})
    .n("CognitoIdentityProviderClient", "AddCustomAttributesCommand")
    .sc(AddCustomAttributes$)
    .build() {
}

class AdminAddUserToGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminAddUserToGroup", {})
    .n("CognitoIdentityProviderClient", "AdminAddUserToGroupCommand")
    .sc(AdminAddUserToGroup$)
    .build() {
}

class AdminConfirmSignUpCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminConfirmSignUp", {})
    .n("CognitoIdentityProviderClient", "AdminConfirmSignUpCommand")
    .sc(AdminConfirmSignUp$)
    .build() {
}

class AdminCreateUserCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminCreateUser", {})
    .n("CognitoIdentityProviderClient", "AdminCreateUserCommand")
    .sc(AdminCreateUser$)
    .build() {
}

class AdminDeleteUserAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminDeleteUserAttributes", {})
    .n("CognitoIdentityProviderClient", "AdminDeleteUserAttributesCommand")
    .sc(AdminDeleteUserAttributes$)
    .build() {
}

class AdminDeleteUserCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminDeleteUser", {})
    .n("CognitoIdentityProviderClient", "AdminDeleteUserCommand")
    .sc(AdminDeleteUser$)
    .build() {
}

class AdminDisableProviderForUserCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminDisableProviderForUser", {})
    .n("CognitoIdentityProviderClient", "AdminDisableProviderForUserCommand")
    .sc(AdminDisableProviderForUser$)
    .build() {
}

class AdminDisableUserCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminDisableUser", {})
    .n("CognitoIdentityProviderClient", "AdminDisableUserCommand")
    .sc(AdminDisableUser$)
    .build() {
}

class AdminEnableUserCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminEnableUser", {})
    .n("CognitoIdentityProviderClient", "AdminEnableUserCommand")
    .sc(AdminEnableUser$)
    .build() {
}

class AdminForgetDeviceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminForgetDevice", {})
    .n("CognitoIdentityProviderClient", "AdminForgetDeviceCommand")
    .sc(AdminForgetDevice$)
    .build() {
}

class AdminGetDeviceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminGetDevice", {})
    .n("CognitoIdentityProviderClient", "AdminGetDeviceCommand")
    .sc(AdminGetDevice$)
    .build() {
}

class AdminGetUserCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminGetUser", {})
    .n("CognitoIdentityProviderClient", "AdminGetUserCommand")
    .sc(AdminGetUser$)
    .build() {
}

class AdminInitiateAuthCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminInitiateAuth", {})
    .n("CognitoIdentityProviderClient", "AdminInitiateAuthCommand")
    .sc(AdminInitiateAuth$)
    .build() {
}

class AdminLinkProviderForUserCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminLinkProviderForUser", {})
    .n("CognitoIdentityProviderClient", "AdminLinkProviderForUserCommand")
    .sc(AdminLinkProviderForUser$)
    .build() {
}

class AdminListDevicesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminListDevices", {})
    .n("CognitoIdentityProviderClient", "AdminListDevicesCommand")
    .sc(AdminListDevices$)
    .build() {
}

class AdminListGroupsForUserCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminListGroupsForUser", {})
    .n("CognitoIdentityProviderClient", "AdminListGroupsForUserCommand")
    .sc(AdminListGroupsForUser$)
    .build() {
}

class AdminListUserAuthEventsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminListUserAuthEvents", {})
    .n("CognitoIdentityProviderClient", "AdminListUserAuthEventsCommand")
    .sc(AdminListUserAuthEvents$)
    .build() {
}

class AdminRemoveUserFromGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminRemoveUserFromGroup", {})
    .n("CognitoIdentityProviderClient", "AdminRemoveUserFromGroupCommand")
    .sc(AdminRemoveUserFromGroup$)
    .build() {
}

class AdminResetUserPasswordCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminResetUserPassword", {})
    .n("CognitoIdentityProviderClient", "AdminResetUserPasswordCommand")
    .sc(AdminResetUserPassword$)
    .build() {
}

class AdminRespondToAuthChallengeCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminRespondToAuthChallenge", {})
    .n("CognitoIdentityProviderClient", "AdminRespondToAuthChallengeCommand")
    .sc(AdminRespondToAuthChallenge$)
    .build() {
}

class AdminSetUserMFAPreferenceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminSetUserMFAPreference", {})
    .n("CognitoIdentityProviderClient", "AdminSetUserMFAPreferenceCommand")
    .sc(AdminSetUserMFAPreference$)
    .build() {
}

class AdminSetUserPasswordCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminSetUserPassword", {})
    .n("CognitoIdentityProviderClient", "AdminSetUserPasswordCommand")
    .sc(AdminSetUserPassword$)
    .build() {
}

class AdminSetUserSettingsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminSetUserSettings", {})
    .n("CognitoIdentityProviderClient", "AdminSetUserSettingsCommand")
    .sc(AdminSetUserSettings$)
    .build() {
}

class AdminUpdateAuthEventFeedbackCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminUpdateAuthEventFeedback", {})
    .n("CognitoIdentityProviderClient", "AdminUpdateAuthEventFeedbackCommand")
    .sc(AdminUpdateAuthEventFeedback$)
    .build() {
}

class AdminUpdateDeviceStatusCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminUpdateDeviceStatus", {})
    .n("CognitoIdentityProviderClient", "AdminUpdateDeviceStatusCommand")
    .sc(AdminUpdateDeviceStatus$)
    .build() {
}

class AdminUpdateUserAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminUpdateUserAttributes", {})
    .n("CognitoIdentityProviderClient", "AdminUpdateUserAttributesCommand")
    .sc(AdminUpdateUserAttributes$)
    .build() {
}

class AdminUserGlobalSignOutCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AdminUserGlobalSignOut", {})
    .n("CognitoIdentityProviderClient", "AdminUserGlobalSignOutCommand")
    .sc(AdminUserGlobalSignOut$)
    .build() {
}

class AssociateSoftwareTokenCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "AssociateSoftwareToken", {})
    .n("CognitoIdentityProviderClient", "AssociateSoftwareTokenCommand")
    .sc(AssociateSoftwareToken$)
    .build() {
}

class ChangePasswordCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ChangePassword", {})
    .n("CognitoIdentityProviderClient", "ChangePasswordCommand")
    .sc(ChangePassword$)
    .build() {
}

class CompleteWebAuthnRegistrationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "CompleteWebAuthnRegistration", {})
    .n("CognitoIdentityProviderClient", "CompleteWebAuthnRegistrationCommand")
    .sc(CompleteWebAuthnRegistration$)
    .build() {
}

class ConfirmDeviceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ConfirmDevice", {})
    .n("CognitoIdentityProviderClient", "ConfirmDeviceCommand")
    .sc(ConfirmDevice$)
    .build() {
}

class ConfirmForgotPasswordCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ConfirmForgotPassword", {})
    .n("CognitoIdentityProviderClient", "ConfirmForgotPasswordCommand")
    .sc(ConfirmForgotPassword$)
    .build() {
}

class ConfirmSignUpCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ConfirmSignUp", {})
    .n("CognitoIdentityProviderClient", "ConfirmSignUpCommand")
    .sc(ConfirmSignUp$)
    .build() {
}

class CreateGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "CreateGroup", {})
    .n("CognitoIdentityProviderClient", "CreateGroupCommand")
    .sc(CreateGroup$)
    .build() {
}

class CreateIdentityProviderCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "CreateIdentityProvider", {})
    .n("CognitoIdentityProviderClient", "CreateIdentityProviderCommand")
    .sc(CreateIdentityProvider$)
    .build() {
}

class CreateManagedLoginBrandingCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "CreateManagedLoginBranding", {})
    .n("CognitoIdentityProviderClient", "CreateManagedLoginBrandingCommand")
    .sc(CreateManagedLoginBranding$)
    .build() {
}

class CreateResourceServerCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "CreateResourceServer", {})
    .n("CognitoIdentityProviderClient", "CreateResourceServerCommand")
    .sc(CreateResourceServer$)
    .build() {
}

class CreateTermsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "CreateTerms", {})
    .n("CognitoIdentityProviderClient", "CreateTermsCommand")
    .sc(CreateTerms$)
    .build() {
}

class CreateUserImportJobCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "CreateUserImportJob", {})
    .n("CognitoIdentityProviderClient", "CreateUserImportJobCommand")
    .sc(CreateUserImportJob$)
    .build() {
}

class CreateUserPoolClientCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "CreateUserPoolClient", {})
    .n("CognitoIdentityProviderClient", "CreateUserPoolClientCommand")
    .sc(CreateUserPoolClient$)
    .build() {
}

class CreateUserPoolCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "CreateUserPool", {})
    .n("CognitoIdentityProviderClient", "CreateUserPoolCommand")
    .sc(CreateUserPool$)
    .build() {
}

class CreateUserPoolDomainCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "CreateUserPoolDomain", {})
    .n("CognitoIdentityProviderClient", "CreateUserPoolDomainCommand")
    .sc(CreateUserPoolDomain$)
    .build() {
}

class DeleteGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DeleteGroup", {})
    .n("CognitoIdentityProviderClient", "DeleteGroupCommand")
    .sc(DeleteGroup$)
    .build() {
}

class DeleteIdentityProviderCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DeleteIdentityProvider", {})
    .n("CognitoIdentityProviderClient", "DeleteIdentityProviderCommand")
    .sc(DeleteIdentityProvider$)
    .build() {
}

class DeleteManagedLoginBrandingCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DeleteManagedLoginBranding", {})
    .n("CognitoIdentityProviderClient", "DeleteManagedLoginBrandingCommand")
    .sc(DeleteManagedLoginBranding$)
    .build() {
}

class DeleteResourceServerCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DeleteResourceServer", {})
    .n("CognitoIdentityProviderClient", "DeleteResourceServerCommand")
    .sc(DeleteResourceServer$)
    .build() {
}

class DeleteTermsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DeleteTerms", {})
    .n("CognitoIdentityProviderClient", "DeleteTermsCommand")
    .sc(DeleteTerms$)
    .build() {
}

class DeleteUserAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DeleteUserAttributes", {})
    .n("CognitoIdentityProviderClient", "DeleteUserAttributesCommand")
    .sc(DeleteUserAttributes$)
    .build() {
}

class DeleteUserCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DeleteUser", {})
    .n("CognitoIdentityProviderClient", "DeleteUserCommand")
    .sc(DeleteUser$)
    .build() {
}

class DeleteUserPoolClientCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DeleteUserPoolClient", {})
    .n("CognitoIdentityProviderClient", "DeleteUserPoolClientCommand")
    .sc(DeleteUserPoolClient$)
    .build() {
}

class DeleteUserPoolCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DeleteUserPool", {})
    .n("CognitoIdentityProviderClient", "DeleteUserPoolCommand")
    .sc(DeleteUserPool$)
    .build() {
}

class DeleteUserPoolDomainCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DeleteUserPoolDomain", {})
    .n("CognitoIdentityProviderClient", "DeleteUserPoolDomainCommand")
    .sc(DeleteUserPoolDomain$)
    .build() {
}

class DeleteWebAuthnCredentialCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DeleteWebAuthnCredential", {})
    .n("CognitoIdentityProviderClient", "DeleteWebAuthnCredentialCommand")
    .sc(DeleteWebAuthnCredential$)
    .build() {
}

class DescribeIdentityProviderCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DescribeIdentityProvider", {})
    .n("CognitoIdentityProviderClient", "DescribeIdentityProviderCommand")
    .sc(DescribeIdentityProvider$)
    .build() {
}

class DescribeManagedLoginBrandingByClientCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DescribeManagedLoginBrandingByClient", {})
    .n("CognitoIdentityProviderClient", "DescribeManagedLoginBrandingByClientCommand")
    .sc(DescribeManagedLoginBrandingByClient$)
    .build() {
}

class DescribeManagedLoginBrandingCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DescribeManagedLoginBranding", {})
    .n("CognitoIdentityProviderClient", "DescribeManagedLoginBrandingCommand")
    .sc(DescribeManagedLoginBranding$)
    .build() {
}

class DescribeResourceServerCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DescribeResourceServer", {})
    .n("CognitoIdentityProviderClient", "DescribeResourceServerCommand")
    .sc(DescribeResourceServer$)
    .build() {
}

class DescribeRiskConfigurationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DescribeRiskConfiguration", {})
    .n("CognitoIdentityProviderClient", "DescribeRiskConfigurationCommand")
    .sc(DescribeRiskConfiguration$)
    .build() {
}

class DescribeTermsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DescribeTerms", {})
    .n("CognitoIdentityProviderClient", "DescribeTermsCommand")
    .sc(DescribeTerms$)
    .build() {
}

class DescribeUserImportJobCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DescribeUserImportJob", {})
    .n("CognitoIdentityProviderClient", "DescribeUserImportJobCommand")
    .sc(DescribeUserImportJob$)
    .build() {
}

class DescribeUserPoolClientCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DescribeUserPoolClient", {})
    .n("CognitoIdentityProviderClient", "DescribeUserPoolClientCommand")
    .sc(DescribeUserPoolClient$)
    .build() {
}

class DescribeUserPoolCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DescribeUserPool", {})
    .n("CognitoIdentityProviderClient", "DescribeUserPoolCommand")
    .sc(DescribeUserPool$)
    .build() {
}

class DescribeUserPoolDomainCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "DescribeUserPoolDomain", {})
    .n("CognitoIdentityProviderClient", "DescribeUserPoolDomainCommand")
    .sc(DescribeUserPoolDomain$)
    .build() {
}

class ForgetDeviceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ForgetDevice", {})
    .n("CognitoIdentityProviderClient", "ForgetDeviceCommand")
    .sc(ForgetDevice$)
    .build() {
}

class ForgotPasswordCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ForgotPassword", {})
    .n("CognitoIdentityProviderClient", "ForgotPasswordCommand")
    .sc(ForgotPassword$)
    .build() {
}

class GetCSVHeaderCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "GetCSVHeader", {})
    .n("CognitoIdentityProviderClient", "GetCSVHeaderCommand")
    .sc(GetCSVHeader$)
    .build() {
}

class GetDeviceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "GetDevice", {})
    .n("CognitoIdentityProviderClient", "GetDeviceCommand")
    .sc(GetDevice$)
    .build() {
}

class GetGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "GetGroup", {})
    .n("CognitoIdentityProviderClient", "GetGroupCommand")
    .sc(GetGroup$)
    .build() {
}

class GetIdentityProviderByIdentifierCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "GetIdentityProviderByIdentifier", {})
    .n("CognitoIdentityProviderClient", "GetIdentityProviderByIdentifierCommand")
    .sc(GetIdentityProviderByIdentifier$)
    .build() {
}

class GetLogDeliveryConfigurationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "GetLogDeliveryConfiguration", {})
    .n("CognitoIdentityProviderClient", "GetLogDeliveryConfigurationCommand")
    .sc(GetLogDeliveryConfiguration$)
    .build() {
}

class GetSigningCertificateCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "GetSigningCertificate", {})
    .n("CognitoIdentityProviderClient", "GetSigningCertificateCommand")
    .sc(GetSigningCertificate$)
    .build() {
}

class GetTokensFromRefreshTokenCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "GetTokensFromRefreshToken", {})
    .n("CognitoIdentityProviderClient", "GetTokensFromRefreshTokenCommand")
    .sc(GetTokensFromRefreshToken$)
    .build() {
}

class GetUICustomizationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "GetUICustomization", {})
    .n("CognitoIdentityProviderClient", "GetUICustomizationCommand")
    .sc(GetUICustomization$)
    .build() {
}

class GetUserAttributeVerificationCodeCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "GetUserAttributeVerificationCode", {})
    .n("CognitoIdentityProviderClient", "GetUserAttributeVerificationCodeCommand")
    .sc(GetUserAttributeVerificationCode$)
    .build() {
}

class GetUserAuthFactorsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "GetUserAuthFactors", {})
    .n("CognitoIdentityProviderClient", "GetUserAuthFactorsCommand")
    .sc(GetUserAuthFactors$)
    .build() {
}

class GetUserCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "GetUser", {})
    .n("CognitoIdentityProviderClient", "GetUserCommand")
    .sc(GetUser$)
    .build() {
}

class GetUserPoolMfaConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "GetUserPoolMfaConfig", {})
    .n("CognitoIdentityProviderClient", "GetUserPoolMfaConfigCommand")
    .sc(GetUserPoolMfaConfig$)
    .build() {
}

class GlobalSignOutCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "GlobalSignOut", {})
    .n("CognitoIdentityProviderClient", "GlobalSignOutCommand")
    .sc(GlobalSignOut$)
    .build() {
}

class InitiateAuthCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "InitiateAuth", {})
    .n("CognitoIdentityProviderClient", "InitiateAuthCommand")
    .sc(InitiateAuth$)
    .build() {
}

class ListDevicesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ListDevices", {})
    .n("CognitoIdentityProviderClient", "ListDevicesCommand")
    .sc(ListDevices$)
    .build() {
}

class ListGroupsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ListGroups", {})
    .n("CognitoIdentityProviderClient", "ListGroupsCommand")
    .sc(ListGroups$)
    .build() {
}

class ListIdentityProvidersCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ListIdentityProviders", {})
    .n("CognitoIdentityProviderClient", "ListIdentityProvidersCommand")
    .sc(ListIdentityProviders$)
    .build() {
}

class ListResourceServersCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ListResourceServers", {})
    .n("CognitoIdentityProviderClient", "ListResourceServersCommand")
    .sc(ListResourceServers$)
    .build() {
}

class ListTagsForResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ListTagsForResource", {})
    .n("CognitoIdentityProviderClient", "ListTagsForResourceCommand")
    .sc(ListTagsForResource$)
    .build() {
}

class ListTermsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ListTerms", {})
    .n("CognitoIdentityProviderClient", "ListTermsCommand")
    .sc(ListTerms$)
    .build() {
}

class ListUserImportJobsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ListUserImportJobs", {})
    .n("CognitoIdentityProviderClient", "ListUserImportJobsCommand")
    .sc(ListUserImportJobs$)
    .build() {
}

class ListUserPoolClientsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ListUserPoolClients", {})
    .n("CognitoIdentityProviderClient", "ListUserPoolClientsCommand")
    .sc(ListUserPoolClients$)
    .build() {
}

class ListUserPoolsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ListUserPools", {})
    .n("CognitoIdentityProviderClient", "ListUserPoolsCommand")
    .sc(ListUserPools$)
    .build() {
}

class ListUsersCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ListUsers", {})
    .n("CognitoIdentityProviderClient", "ListUsersCommand")
    .sc(ListUsers$)
    .build() {
}

class ListUsersInGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ListUsersInGroup", {})
    .n("CognitoIdentityProviderClient", "ListUsersInGroupCommand")
    .sc(ListUsersInGroup$)
    .build() {
}

class ListWebAuthnCredentialsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ListWebAuthnCredentials", {})
    .n("CognitoIdentityProviderClient", "ListWebAuthnCredentialsCommand")
    .sc(ListWebAuthnCredentials$)
    .build() {
}

class ResendConfirmationCodeCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "ResendConfirmationCode", {})
    .n("CognitoIdentityProviderClient", "ResendConfirmationCodeCommand")
    .sc(ResendConfirmationCode$)
    .build() {
}

class RespondToAuthChallengeCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "RespondToAuthChallenge", {})
    .n("CognitoIdentityProviderClient", "RespondToAuthChallengeCommand")
    .sc(RespondToAuthChallenge$)
    .build() {
}

class RevokeTokenCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "RevokeToken", {})
    .n("CognitoIdentityProviderClient", "RevokeTokenCommand")
    .sc(RevokeToken$)
    .build() {
}

class SetLogDeliveryConfigurationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "SetLogDeliveryConfiguration", {})
    .n("CognitoIdentityProviderClient", "SetLogDeliveryConfigurationCommand")
    .sc(SetLogDeliveryConfiguration$)
    .build() {
}

class SetRiskConfigurationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "SetRiskConfiguration", {})
    .n("CognitoIdentityProviderClient", "SetRiskConfigurationCommand")
    .sc(SetRiskConfiguration$)
    .build() {
}

class SetUICustomizationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "SetUICustomization", {})
    .n("CognitoIdentityProviderClient", "SetUICustomizationCommand")
    .sc(SetUICustomization$)
    .build() {
}

class SetUserMFAPreferenceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "SetUserMFAPreference", {})
    .n("CognitoIdentityProviderClient", "SetUserMFAPreferenceCommand")
    .sc(SetUserMFAPreference$)
    .build() {
}

class SetUserPoolMfaConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "SetUserPoolMfaConfig", {})
    .n("CognitoIdentityProviderClient", "SetUserPoolMfaConfigCommand")
    .sc(SetUserPoolMfaConfig$)
    .build() {
}

class SetUserSettingsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "SetUserSettings", {})
    .n("CognitoIdentityProviderClient", "SetUserSettingsCommand")
    .sc(SetUserSettings$)
    .build() {
}

class SignUpCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "SignUp", {})
    .n("CognitoIdentityProviderClient", "SignUpCommand")
    .sc(SignUp$)
    .build() {
}

class StartUserImportJobCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "StartUserImportJob", {})
    .n("CognitoIdentityProviderClient", "StartUserImportJobCommand")
    .sc(StartUserImportJob$)
    .build() {
}

class StartWebAuthnRegistrationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "StartWebAuthnRegistration", {})
    .n("CognitoIdentityProviderClient", "StartWebAuthnRegistrationCommand")
    .sc(StartWebAuthnRegistration$)
    .build() {
}

class StopUserImportJobCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "StopUserImportJob", {})
    .n("CognitoIdentityProviderClient", "StopUserImportJobCommand")
    .sc(StopUserImportJob$)
    .build() {
}

class TagResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "TagResource", {})
    .n("CognitoIdentityProviderClient", "TagResourceCommand")
    .sc(TagResource$)
    .build() {
}

class UntagResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "UntagResource", {})
    .n("CognitoIdentityProviderClient", "UntagResourceCommand")
    .sc(UntagResource$)
    .build() {
}

class UpdateAuthEventFeedbackCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "UpdateAuthEventFeedback", {})
    .n("CognitoIdentityProviderClient", "UpdateAuthEventFeedbackCommand")
    .sc(UpdateAuthEventFeedback$)
    .build() {
}

class UpdateDeviceStatusCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "UpdateDeviceStatus", {})
    .n("CognitoIdentityProviderClient", "UpdateDeviceStatusCommand")
    .sc(UpdateDeviceStatus$)
    .build() {
}

class UpdateGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "UpdateGroup", {})
    .n("CognitoIdentityProviderClient", "UpdateGroupCommand")
    .sc(UpdateGroup$)
    .build() {
}

class UpdateIdentityProviderCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "UpdateIdentityProvider", {})
    .n("CognitoIdentityProviderClient", "UpdateIdentityProviderCommand")
    .sc(UpdateIdentityProvider$)
    .build() {
}

class UpdateManagedLoginBrandingCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "UpdateManagedLoginBranding", {})
    .n("CognitoIdentityProviderClient", "UpdateManagedLoginBrandingCommand")
    .sc(UpdateManagedLoginBranding$)
    .build() {
}

class UpdateResourceServerCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "UpdateResourceServer", {})
    .n("CognitoIdentityProviderClient", "UpdateResourceServerCommand")
    .sc(UpdateResourceServer$)
    .build() {
}

class UpdateTermsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "UpdateTerms", {})
    .n("CognitoIdentityProviderClient", "UpdateTermsCommand")
    .sc(UpdateTerms$)
    .build() {
}

class UpdateUserAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "UpdateUserAttributes", {})
    .n("CognitoIdentityProviderClient", "UpdateUserAttributesCommand")
    .sc(UpdateUserAttributes$)
    .build() {
}

class UpdateUserPoolClientCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "UpdateUserPoolClient", {})
    .n("CognitoIdentityProviderClient", "UpdateUserPoolClientCommand")
    .sc(UpdateUserPoolClient$)
    .build() {
}

class UpdateUserPoolCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "UpdateUserPool", {})
    .n("CognitoIdentityProviderClient", "UpdateUserPoolCommand")
    .sc(UpdateUserPool$)
    .build() {
}

class UpdateUserPoolDomainCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "UpdateUserPoolDomain", {})
    .n("CognitoIdentityProviderClient", "UpdateUserPoolDomainCommand")
    .sc(UpdateUserPoolDomain$)
    .build() {
}

class VerifySoftwareTokenCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "VerifySoftwareToken", {})
    .n("CognitoIdentityProviderClient", "VerifySoftwareTokenCommand")
    .sc(VerifySoftwareToken$)
    .build() {
}

class VerifyUserAttributeCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AWSCognitoIdentityProviderService", "VerifyUserAttribute", {})
    .n("CognitoIdentityProviderClient", "VerifyUserAttributeCommand")
    .sc(VerifyUserAttribute$)
    .build() {
}

const commands = {
    AddCustomAttributesCommand,
    AdminAddUserToGroupCommand,
    AdminConfirmSignUpCommand,
    AdminCreateUserCommand,
    AdminDeleteUserCommand,
    AdminDeleteUserAttributesCommand,
    AdminDisableProviderForUserCommand,
    AdminDisableUserCommand,
    AdminEnableUserCommand,
    AdminForgetDeviceCommand,
    AdminGetDeviceCommand,
    AdminGetUserCommand,
    AdminInitiateAuthCommand,
    AdminLinkProviderForUserCommand,
    AdminListDevicesCommand,
    AdminListGroupsForUserCommand,
    AdminListUserAuthEventsCommand,
    AdminRemoveUserFromGroupCommand,
    AdminResetUserPasswordCommand,
    AdminRespondToAuthChallengeCommand,
    AdminSetUserMFAPreferenceCommand,
    AdminSetUserPasswordCommand,
    AdminSetUserSettingsCommand,
    AdminUpdateAuthEventFeedbackCommand,
    AdminUpdateDeviceStatusCommand,
    AdminUpdateUserAttributesCommand,
    AdminUserGlobalSignOutCommand,
    AssociateSoftwareTokenCommand,
    ChangePasswordCommand,
    CompleteWebAuthnRegistrationCommand,
    ConfirmDeviceCommand,
    ConfirmForgotPasswordCommand,
    ConfirmSignUpCommand,
    CreateGroupCommand,
    CreateIdentityProviderCommand,
    CreateManagedLoginBrandingCommand,
    CreateResourceServerCommand,
    CreateTermsCommand,
    CreateUserImportJobCommand,
    CreateUserPoolCommand,
    CreateUserPoolClientCommand,
    CreateUserPoolDomainCommand,
    DeleteGroupCommand,
    DeleteIdentityProviderCommand,
    DeleteManagedLoginBrandingCommand,
    DeleteResourceServerCommand,
    DeleteTermsCommand,
    DeleteUserCommand,
    DeleteUserAttributesCommand,
    DeleteUserPoolCommand,
    DeleteUserPoolClientCommand,
    DeleteUserPoolDomainCommand,
    DeleteWebAuthnCredentialCommand,
    DescribeIdentityProviderCommand,
    DescribeManagedLoginBrandingCommand,
    DescribeManagedLoginBrandingByClientCommand,
    DescribeResourceServerCommand,
    DescribeRiskConfigurationCommand,
    DescribeTermsCommand,
    DescribeUserImportJobCommand,
    DescribeUserPoolCommand,
    DescribeUserPoolClientCommand,
    DescribeUserPoolDomainCommand,
    ForgetDeviceCommand,
    ForgotPasswordCommand,
    GetCSVHeaderCommand,
    GetDeviceCommand,
    GetGroupCommand,
    GetIdentityProviderByIdentifierCommand,
    GetLogDeliveryConfigurationCommand,
    GetSigningCertificateCommand,
    GetTokensFromRefreshTokenCommand,
    GetUICustomizationCommand,
    GetUserCommand,
    GetUserAttributeVerificationCodeCommand,
    GetUserAuthFactorsCommand,
    GetUserPoolMfaConfigCommand,
    GlobalSignOutCommand,
    InitiateAuthCommand,
    ListDevicesCommand,
    ListGroupsCommand,
    ListIdentityProvidersCommand,
    ListResourceServersCommand,
    ListTagsForResourceCommand,
    ListTermsCommand,
    ListUserImportJobsCommand,
    ListUserPoolClientsCommand,
    ListUserPoolsCommand,
    ListUsersCommand,
    ListUsersInGroupCommand,
    ListWebAuthnCredentialsCommand,
    ResendConfirmationCodeCommand,
    RespondToAuthChallengeCommand,
    RevokeTokenCommand,
    SetLogDeliveryConfigurationCommand,
    SetRiskConfigurationCommand,
    SetUICustomizationCommand,
    SetUserMFAPreferenceCommand,
    SetUserPoolMfaConfigCommand,
    SetUserSettingsCommand,
    SignUpCommand,
    StartUserImportJobCommand,
    StartWebAuthnRegistrationCommand,
    StopUserImportJobCommand,
    TagResourceCommand,
    UntagResourceCommand,
    UpdateAuthEventFeedbackCommand,
    UpdateDeviceStatusCommand,
    UpdateGroupCommand,
    UpdateIdentityProviderCommand,
    UpdateManagedLoginBrandingCommand,
    UpdateResourceServerCommand,
    UpdateTermsCommand,
    UpdateUserAttributesCommand,
    UpdateUserPoolCommand,
    UpdateUserPoolClientCommand,
    UpdateUserPoolDomainCommand,
    VerifySoftwareTokenCommand,
    VerifyUserAttributeCommand,
};
class CognitoIdentityProvider extends CognitoIdentityProviderClient {
}
smithyClient.createAggregatedClient(commands, CognitoIdentityProvider);

const paginateAdminListGroupsForUser = core.createPaginator(CognitoIdentityProviderClient, AdminListGroupsForUserCommand, "NextToken", "NextToken", "Limit");

const paginateAdminListUserAuthEvents = core.createPaginator(CognitoIdentityProviderClient, AdminListUserAuthEventsCommand, "NextToken", "NextToken", "MaxResults");

const paginateListGroups = core.createPaginator(CognitoIdentityProviderClient, ListGroupsCommand, "NextToken", "NextToken", "Limit");

const paginateListIdentityProviders = core.createPaginator(CognitoIdentityProviderClient, ListIdentityProvidersCommand, "NextToken", "NextToken", "MaxResults");

const paginateListResourceServers = core.createPaginator(CognitoIdentityProviderClient, ListResourceServersCommand, "NextToken", "NextToken", "MaxResults");

const paginateListUserPoolClients = core.createPaginator(CognitoIdentityProviderClient, ListUserPoolClientsCommand, "NextToken", "NextToken", "MaxResults");

const paginateListUserPools = core.createPaginator(CognitoIdentityProviderClient, ListUserPoolsCommand, "NextToken", "NextToken", "MaxResults");

const paginateListUsers = core.createPaginator(CognitoIdentityProviderClient, ListUsersCommand, "PaginationToken", "PaginationToken", "Limit");

const paginateListUsersInGroup = core.createPaginator(CognitoIdentityProviderClient, ListUsersInGroupCommand, "NextToken", "NextToken", "Limit");

const RecoveryOptionNameType = {
    ADMIN_ONLY: "admin_only",
    VERIFIED_EMAIL: "verified_email",
    VERIFIED_PHONE_NUMBER: "verified_phone_number",
};
const AccountTakeoverEventActionType = {
    BLOCK: "BLOCK",
    MFA_IF_CONFIGURED: "MFA_IF_CONFIGURED",
    MFA_REQUIRED: "MFA_REQUIRED",
    NO_ACTION: "NO_ACTION",
};
const AttributeDataType = {
    BOOLEAN: "Boolean",
    DATETIME: "DateTime",
    NUMBER: "Number",
    STRING: "String",
};
const DeliveryMediumType = {
    EMAIL: "EMAIL",
    SMS: "SMS",
};
const MessageActionType = {
    RESEND: "RESEND",
    SUPPRESS: "SUPPRESS",
};
const UserStatusType = {
    ARCHIVED: "ARCHIVED",
    COMPROMISED: "COMPROMISED",
    CONFIRMED: "CONFIRMED",
    EXTERNAL_PROVIDER: "EXTERNAL_PROVIDER",
    FORCE_CHANGE_PASSWORD: "FORCE_CHANGE_PASSWORD",
    RESET_REQUIRED: "RESET_REQUIRED",
    UNCONFIRMED: "UNCONFIRMED",
    UNKNOWN: "UNKNOWN",
};
const AuthFlowType = {
    ADMIN_NO_SRP_AUTH: "ADMIN_NO_SRP_AUTH",
    ADMIN_USER_PASSWORD_AUTH: "ADMIN_USER_PASSWORD_AUTH",
    CUSTOM_AUTH: "CUSTOM_AUTH",
    REFRESH_TOKEN: "REFRESH_TOKEN",
    REFRESH_TOKEN_AUTH: "REFRESH_TOKEN_AUTH",
    USER_AUTH: "USER_AUTH",
    USER_PASSWORD_AUTH: "USER_PASSWORD_AUTH",
    USER_SRP_AUTH: "USER_SRP_AUTH",
};
const ChallengeNameType = {
    ADMIN_NO_SRP_AUTH: "ADMIN_NO_SRP_AUTH",
    CUSTOM_CHALLENGE: "CUSTOM_CHALLENGE",
    DEVICE_PASSWORD_VERIFIER: "DEVICE_PASSWORD_VERIFIER",
    DEVICE_SRP_AUTH: "DEVICE_SRP_AUTH",
    EMAIL_OTP: "EMAIL_OTP",
    MFA_SETUP: "MFA_SETUP",
    NEW_PASSWORD_REQUIRED: "NEW_PASSWORD_REQUIRED",
    PASSWORD: "PASSWORD",
    PASSWORD_SRP: "PASSWORD_SRP",
    PASSWORD_VERIFIER: "PASSWORD_VERIFIER",
    SELECT_CHALLENGE: "SELECT_CHALLENGE",
    SELECT_MFA_TYPE: "SELECT_MFA_TYPE",
    SMS_MFA: "SMS_MFA",
    SMS_OTP: "SMS_OTP",
    SOFTWARE_TOKEN_MFA: "SOFTWARE_TOKEN_MFA",
    WEB_AUTHN: "WEB_AUTHN",
};
const ChallengeName = {
    Mfa: "Mfa",
    Password: "Password",
};
const ChallengeResponse = {
    Failure: "Failure",
    Success: "Success",
};
const FeedbackValueType = {
    INVALID: "Invalid",
    VALID: "Valid",
};
const EventResponseType = {
    Fail: "Fail",
    InProgress: "InProgress",
    Pass: "Pass",
};
const RiskDecisionType = {
    AccountTakeover: "AccountTakeover",
    Block: "Block",
    NoRisk: "NoRisk",
};
const RiskLevelType = {
    High: "High",
    Low: "Low",
    Medium: "Medium",
};
const EventType = {
    ForgotPassword: "ForgotPassword",
    PasswordChange: "PasswordChange",
    ResendCode: "ResendCode",
    SignIn: "SignIn",
    SignUp: "SignUp",
};
const DeviceRememberedStatusType = {
    NOT_REMEMBERED: "not_remembered",
    REMEMBERED: "remembered",
};
const AdvancedSecurityEnabledModeType = {
    AUDIT: "AUDIT",
    ENFORCED: "ENFORCED",
};
const AdvancedSecurityModeType = {
    AUDIT: "AUDIT",
    ENFORCED: "ENFORCED",
    OFF: "OFF",
};
const AliasAttributeType = {
    EMAIL: "email",
    PHONE_NUMBER: "phone_number",
    PREFERRED_USERNAME: "preferred_username",
};
const AuthFactorType = {
    EMAIL_OTP: "EMAIL_OTP",
    PASSWORD: "PASSWORD",
    SMS_OTP: "SMS_OTP",
    WEB_AUTHN: "WEB_AUTHN",
};
const AssetCategoryType = {
    AUTH_APP_GRAPHIC: "AUTH_APP_GRAPHIC",
    EMAIL_GRAPHIC: "EMAIL_GRAPHIC",
    FAVICON_ICO: "FAVICON_ICO",
    FAVICON_SVG: "FAVICON_SVG",
    FORM_BACKGROUND: "FORM_BACKGROUND",
    FORM_LOGO: "FORM_LOGO",
    IDP_BUTTON_ICON: "IDP_BUTTON_ICON",
    PAGE_BACKGROUND: "PAGE_BACKGROUND",
    PAGE_FOOTER_BACKGROUND: "PAGE_FOOTER_BACKGROUND",
    PAGE_FOOTER_LOGO: "PAGE_FOOTER_LOGO",
    PAGE_HEADER_BACKGROUND: "PAGE_HEADER_BACKGROUND",
    PAGE_HEADER_LOGO: "PAGE_HEADER_LOGO",
    PASSKEY_GRAPHIC: "PASSKEY_GRAPHIC",
    PASSWORD_GRAPHIC: "PASSWORD_GRAPHIC",
    SMS_GRAPHIC: "SMS_GRAPHIC",
};
const AssetExtensionType = {
    ICO: "ICO",
    JPEG: "JPEG",
    PNG: "PNG",
    SVG: "SVG",
    WEBP: "WEBP",
};
const ColorSchemeModeType = {
    DARK: "DARK",
    DYNAMIC: "DYNAMIC",
    LIGHT: "LIGHT",
};
const VerifiedAttributeType = {
    EMAIL: "email",
    PHONE_NUMBER: "phone_number",
};
const IdentityProviderTypeType = {
    Facebook: "Facebook",
    Google: "Google",
    LoginWithAmazon: "LoginWithAmazon",
    OIDC: "OIDC",
    SAML: "SAML",
    SignInWithApple: "SignInWithApple",
};
const TermsEnforcementType = {
    NONE: "NONE",
};
const TermsSourceType = {
    LINK: "LINK",
};
const UserImportJobStatusType = {
    Created: "Created",
    Expired: "Expired",
    Failed: "Failed",
    InProgress: "InProgress",
    Pending: "Pending",
    Stopped: "Stopped",
    Stopping: "Stopping",
    Succeeded: "Succeeded",
};
const DeletionProtectionType = {
    ACTIVE: "ACTIVE",
    INACTIVE: "INACTIVE",
};
const EmailSendingAccountType = {
    COGNITO_DEFAULT: "COGNITO_DEFAULT",
    DEVELOPER: "DEVELOPER",
};
const CustomEmailSenderLambdaVersionType = {
    V1_0: "V1_0",
};
const CustomSMSSenderLambdaVersionType = {
    V1_0: "V1_0",
};
const PreTokenGenerationLambdaVersionType = {
    V1_0: "V1_0",
    V2_0: "V2_0",
    V3_0: "V3_0",
};
const UserPoolMfaType = {
    OFF: "OFF",
    ON: "ON",
    OPTIONAL: "OPTIONAL",
};
const UsernameAttributeType = {
    EMAIL: "email",
    PHONE_NUMBER: "phone_number",
};
const UserPoolTierType = {
    ESSENTIALS: "ESSENTIALS",
    LITE: "LITE",
    PLUS: "PLUS",
};
const DefaultEmailOptionType = {
    CONFIRM_WITH_CODE: "CONFIRM_WITH_CODE",
    CONFIRM_WITH_LINK: "CONFIRM_WITH_LINK",
};
const StatusType = {
    Disabled: "Disabled",
    Enabled: "Enabled",
};
const OAuthFlowType = {
    client_credentials: "client_credentials",
    code: "code",
    implicit: "implicit",
};
const ExplicitAuthFlowsType = {
    ADMIN_NO_SRP_AUTH: "ADMIN_NO_SRP_AUTH",
    ALLOW_ADMIN_USER_PASSWORD_AUTH: "ALLOW_ADMIN_USER_PASSWORD_AUTH",
    ALLOW_CUSTOM_AUTH: "ALLOW_CUSTOM_AUTH",
    ALLOW_REFRESH_TOKEN_AUTH: "ALLOW_REFRESH_TOKEN_AUTH",
    ALLOW_USER_AUTH: "ALLOW_USER_AUTH",
    ALLOW_USER_PASSWORD_AUTH: "ALLOW_USER_PASSWORD_AUTH",
    ALLOW_USER_SRP_AUTH: "ALLOW_USER_SRP_AUTH",
    CUSTOM_AUTH_FLOW_ONLY: "CUSTOM_AUTH_FLOW_ONLY",
    USER_PASSWORD_AUTH: "USER_PASSWORD_AUTH",
};
const PreventUserExistenceErrorTypes = {
    ENABLED: "ENABLED",
    LEGACY: "LEGACY",
};
const FeatureType = {
    DISABLED: "DISABLED",
    ENABLED: "ENABLED",
};
const TimeUnitsType = {
    DAYS: "days",
    HOURS: "hours",
    MINUTES: "minutes",
    SECONDS: "seconds",
};
const CompromisedCredentialsEventActionType = {
    BLOCK: "BLOCK",
    NO_ACTION: "NO_ACTION",
};
const EventFilterType = {
    PASSWORD_CHANGE: "PASSWORD_CHANGE",
    SIGN_IN: "SIGN_IN",
    SIGN_UP: "SIGN_UP",
};
const DomainStatusType = {
    ACTIVE: "ACTIVE",
    CREATING: "CREATING",
    DELETING: "DELETING",
    FAILED: "FAILED",
    UPDATING: "UPDATING",
};
const EventSourceName = {
    USER_AUTH_EVENTS: "userAuthEvents",
    USER_NOTIFICATION: "userNotification",
};
const LogLevel = {
    ERROR: "ERROR",
    INFO: "INFO",
};
const UserVerificationType = {
    PREFERRED: "preferred",
    REQUIRED: "required",
};
const VerifySoftwareTokenResponseType = {
    ERROR: "ERROR",
    SUCCESS: "SUCCESS",
};

Object.defineProperty(exports, "$Command", {
    enumerable: true,
    get: function () { return smithyClient.Command; }
});
Object.defineProperty(exports, "__Client", {
    enumerable: true,
    get: function () { return smithyClient.Client; }
});
exports.AccountRecoverySettingType$ = AccountRecoverySettingType$;
exports.AccountTakeoverActionType$ = AccountTakeoverActionType$;
exports.AccountTakeoverActionsType$ = AccountTakeoverActionsType$;
exports.AccountTakeoverEventActionType = AccountTakeoverEventActionType;
exports.AccountTakeoverRiskConfigurationType$ = AccountTakeoverRiskConfigurationType$;
exports.AddCustomAttributes$ = AddCustomAttributes$;
exports.AddCustomAttributesCommand = AddCustomAttributesCommand;
exports.AddCustomAttributesRequest$ = AddCustomAttributesRequest$;
exports.AddCustomAttributesResponse$ = AddCustomAttributesResponse$;
exports.AdminAddUserToGroup$ = AdminAddUserToGroup$;
exports.AdminAddUserToGroupCommand = AdminAddUserToGroupCommand;
exports.AdminAddUserToGroupRequest$ = AdminAddUserToGroupRequest$;
exports.AdminConfirmSignUp$ = AdminConfirmSignUp$;
exports.AdminConfirmSignUpCommand = AdminConfirmSignUpCommand;
exports.AdminConfirmSignUpRequest$ = AdminConfirmSignUpRequest$;
exports.AdminConfirmSignUpResponse$ = AdminConfirmSignUpResponse$;
exports.AdminCreateUser$ = AdminCreateUser$;
exports.AdminCreateUserCommand = AdminCreateUserCommand;
exports.AdminCreateUserConfigType$ = AdminCreateUserConfigType$;
exports.AdminCreateUserRequest$ = AdminCreateUserRequest$;
exports.AdminCreateUserResponse$ = AdminCreateUserResponse$;
exports.AdminDeleteUser$ = AdminDeleteUser$;
exports.AdminDeleteUserAttributes$ = AdminDeleteUserAttributes$;
exports.AdminDeleteUserAttributesCommand = AdminDeleteUserAttributesCommand;
exports.AdminDeleteUserAttributesRequest$ = AdminDeleteUserAttributesRequest$;
exports.AdminDeleteUserAttributesResponse$ = AdminDeleteUserAttributesResponse$;
exports.AdminDeleteUserCommand = AdminDeleteUserCommand;
exports.AdminDeleteUserRequest$ = AdminDeleteUserRequest$;
exports.AdminDisableProviderForUser$ = AdminDisableProviderForUser$;
exports.AdminDisableProviderForUserCommand = AdminDisableProviderForUserCommand;
exports.AdminDisableProviderForUserRequest$ = AdminDisableProviderForUserRequest$;
exports.AdminDisableProviderForUserResponse$ = AdminDisableProviderForUserResponse$;
exports.AdminDisableUser$ = AdminDisableUser$;
exports.AdminDisableUserCommand = AdminDisableUserCommand;
exports.AdminDisableUserRequest$ = AdminDisableUserRequest$;
exports.AdminDisableUserResponse$ = AdminDisableUserResponse$;
exports.AdminEnableUser$ = AdminEnableUser$;
exports.AdminEnableUserCommand = AdminEnableUserCommand;
exports.AdminEnableUserRequest$ = AdminEnableUserRequest$;
exports.AdminEnableUserResponse$ = AdminEnableUserResponse$;
exports.AdminForgetDevice$ = AdminForgetDevice$;
exports.AdminForgetDeviceCommand = AdminForgetDeviceCommand;
exports.AdminForgetDeviceRequest$ = AdminForgetDeviceRequest$;
exports.AdminGetDevice$ = AdminGetDevice$;
exports.AdminGetDeviceCommand = AdminGetDeviceCommand;
exports.AdminGetDeviceRequest$ = AdminGetDeviceRequest$;
exports.AdminGetDeviceResponse$ = AdminGetDeviceResponse$;
exports.AdminGetUser$ = AdminGetUser$;
exports.AdminGetUserCommand = AdminGetUserCommand;
exports.AdminGetUserRequest$ = AdminGetUserRequest$;
exports.AdminGetUserResponse$ = AdminGetUserResponse$;
exports.AdminInitiateAuth$ = AdminInitiateAuth$;
exports.AdminInitiateAuthCommand = AdminInitiateAuthCommand;
exports.AdminInitiateAuthRequest$ = AdminInitiateAuthRequest$;
exports.AdminInitiateAuthResponse$ = AdminInitiateAuthResponse$;
exports.AdminLinkProviderForUser$ = AdminLinkProviderForUser$;
exports.AdminLinkProviderForUserCommand = AdminLinkProviderForUserCommand;
exports.AdminLinkProviderForUserRequest$ = AdminLinkProviderForUserRequest$;
exports.AdminLinkProviderForUserResponse$ = AdminLinkProviderForUserResponse$;
exports.AdminListDevices$ = AdminListDevices$;
exports.AdminListDevicesCommand = AdminListDevicesCommand;
exports.AdminListDevicesRequest$ = AdminListDevicesRequest$;
exports.AdminListDevicesResponse$ = AdminListDevicesResponse$;
exports.AdminListGroupsForUser$ = AdminListGroupsForUser$;
exports.AdminListGroupsForUserCommand = AdminListGroupsForUserCommand;
exports.AdminListGroupsForUserRequest$ = AdminListGroupsForUserRequest$;
exports.AdminListGroupsForUserResponse$ = AdminListGroupsForUserResponse$;
exports.AdminListUserAuthEvents$ = AdminListUserAuthEvents$;
exports.AdminListUserAuthEventsCommand = AdminListUserAuthEventsCommand;
exports.AdminListUserAuthEventsRequest$ = AdminListUserAuthEventsRequest$;
exports.AdminListUserAuthEventsResponse$ = AdminListUserAuthEventsResponse$;
exports.AdminRemoveUserFromGroup$ = AdminRemoveUserFromGroup$;
exports.AdminRemoveUserFromGroupCommand = AdminRemoveUserFromGroupCommand;
exports.AdminRemoveUserFromGroupRequest$ = AdminRemoveUserFromGroupRequest$;
exports.AdminResetUserPassword$ = AdminResetUserPassword$;
exports.AdminResetUserPasswordCommand = AdminResetUserPasswordCommand;
exports.AdminResetUserPasswordRequest$ = AdminResetUserPasswordRequest$;
exports.AdminResetUserPasswordResponse$ = AdminResetUserPasswordResponse$;
exports.AdminRespondToAuthChallenge$ = AdminRespondToAuthChallenge$;
exports.AdminRespondToAuthChallengeCommand = AdminRespondToAuthChallengeCommand;
exports.AdminRespondToAuthChallengeRequest$ = AdminRespondToAuthChallengeRequest$;
exports.AdminRespondToAuthChallengeResponse$ = AdminRespondToAuthChallengeResponse$;
exports.AdminSetUserMFAPreference$ = AdminSetUserMFAPreference$;
exports.AdminSetUserMFAPreferenceCommand = AdminSetUserMFAPreferenceCommand;
exports.AdminSetUserMFAPreferenceRequest$ = AdminSetUserMFAPreferenceRequest$;
exports.AdminSetUserMFAPreferenceResponse$ = AdminSetUserMFAPreferenceResponse$;
exports.AdminSetUserPassword$ = AdminSetUserPassword$;
exports.AdminSetUserPasswordCommand = AdminSetUserPasswordCommand;
exports.AdminSetUserPasswordRequest$ = AdminSetUserPasswordRequest$;
exports.AdminSetUserPasswordResponse$ = AdminSetUserPasswordResponse$;
exports.AdminSetUserSettings$ = AdminSetUserSettings$;
exports.AdminSetUserSettingsCommand = AdminSetUserSettingsCommand;
exports.AdminSetUserSettingsRequest$ = AdminSetUserSettingsRequest$;
exports.AdminSetUserSettingsResponse$ = AdminSetUserSettingsResponse$;
exports.AdminUpdateAuthEventFeedback$ = AdminUpdateAuthEventFeedback$;
exports.AdminUpdateAuthEventFeedbackCommand = AdminUpdateAuthEventFeedbackCommand;
exports.AdminUpdateAuthEventFeedbackRequest$ = AdminUpdateAuthEventFeedbackRequest$;
exports.AdminUpdateAuthEventFeedbackResponse$ = AdminUpdateAuthEventFeedbackResponse$;
exports.AdminUpdateDeviceStatus$ = AdminUpdateDeviceStatus$;
exports.AdminUpdateDeviceStatusCommand = AdminUpdateDeviceStatusCommand;
exports.AdminUpdateDeviceStatusRequest$ = AdminUpdateDeviceStatusRequest$;
exports.AdminUpdateDeviceStatusResponse$ = AdminUpdateDeviceStatusResponse$;
exports.AdminUpdateUserAttributes$ = AdminUpdateUserAttributes$;
exports.AdminUpdateUserAttributesCommand = AdminUpdateUserAttributesCommand;
exports.AdminUpdateUserAttributesRequest$ = AdminUpdateUserAttributesRequest$;
exports.AdminUpdateUserAttributesResponse$ = AdminUpdateUserAttributesResponse$;
exports.AdminUserGlobalSignOut$ = AdminUserGlobalSignOut$;
exports.AdminUserGlobalSignOutCommand = AdminUserGlobalSignOutCommand;
exports.AdminUserGlobalSignOutRequest$ = AdminUserGlobalSignOutRequest$;
exports.AdminUserGlobalSignOutResponse$ = AdminUserGlobalSignOutResponse$;
exports.AdvancedSecurityAdditionalFlowsType$ = AdvancedSecurityAdditionalFlowsType$;
exports.AdvancedSecurityEnabledModeType = AdvancedSecurityEnabledModeType;
exports.AdvancedSecurityModeType = AdvancedSecurityModeType;
exports.AliasAttributeType = AliasAttributeType;
exports.AliasExistsException = AliasExistsException;
exports.AliasExistsException$ = AliasExistsException$;
exports.AnalyticsConfigurationType$ = AnalyticsConfigurationType$;
exports.AnalyticsMetadataType$ = AnalyticsMetadataType$;
exports.AssetCategoryType = AssetCategoryType;
exports.AssetExtensionType = AssetExtensionType;
exports.AssetType$ = AssetType$;
exports.AssociateSoftwareToken$ = AssociateSoftwareToken$;
exports.AssociateSoftwareTokenCommand = AssociateSoftwareTokenCommand;
exports.AssociateSoftwareTokenRequest$ = AssociateSoftwareTokenRequest$;
exports.AssociateSoftwareTokenResponse$ = AssociateSoftwareTokenResponse$;
exports.AttributeDataType = AttributeDataType;
exports.AttributeType$ = AttributeType$;
exports.AuthEventType$ = AuthEventType$;
exports.AuthFactorType = AuthFactorType;
exports.AuthFlowType = AuthFlowType;
exports.AuthenticationResultType$ = AuthenticationResultType$;
exports.ChallengeName = ChallengeName;
exports.ChallengeNameType = ChallengeNameType;
exports.ChallengeResponse = ChallengeResponse;
exports.ChallengeResponseType$ = ChallengeResponseType$;
exports.ChangePassword$ = ChangePassword$;
exports.ChangePasswordCommand = ChangePasswordCommand;
exports.ChangePasswordRequest$ = ChangePasswordRequest$;
exports.ChangePasswordResponse$ = ChangePasswordResponse$;
exports.CloudWatchLogsConfigurationType$ = CloudWatchLogsConfigurationType$;
exports.CodeDeliveryDetailsType$ = CodeDeliveryDetailsType$;
exports.CodeDeliveryFailureException = CodeDeliveryFailureException;
exports.CodeDeliveryFailureException$ = CodeDeliveryFailureException$;
exports.CodeMismatchException = CodeMismatchException;
exports.CodeMismatchException$ = CodeMismatchException$;
exports.CognitoIdentityProvider = CognitoIdentityProvider;
exports.CognitoIdentityProviderClient = CognitoIdentityProviderClient;
exports.CognitoIdentityProviderServiceException = CognitoIdentityProviderServiceException;
exports.CognitoIdentityProviderServiceException$ = CognitoIdentityProviderServiceException$;
exports.ColorSchemeModeType = ColorSchemeModeType;
exports.CompleteWebAuthnRegistration$ = CompleteWebAuthnRegistration$;
exports.CompleteWebAuthnRegistrationCommand = CompleteWebAuthnRegistrationCommand;
exports.CompleteWebAuthnRegistrationRequest$ = CompleteWebAuthnRegistrationRequest$;
exports.CompleteWebAuthnRegistrationResponse$ = CompleteWebAuthnRegistrationResponse$;
exports.CompromisedCredentialsActionsType$ = CompromisedCredentialsActionsType$;
exports.CompromisedCredentialsEventActionType = CompromisedCredentialsEventActionType;
exports.CompromisedCredentialsRiskConfigurationType$ = CompromisedCredentialsRiskConfigurationType$;
exports.ConcurrentModificationException = ConcurrentModificationException;
exports.ConcurrentModificationException$ = ConcurrentModificationException$;
exports.ConfirmDevice$ = ConfirmDevice$;
exports.ConfirmDeviceCommand = ConfirmDeviceCommand;
exports.ConfirmDeviceRequest$ = ConfirmDeviceRequest$;
exports.ConfirmDeviceResponse$ = ConfirmDeviceResponse$;
exports.ConfirmForgotPassword$ = ConfirmForgotPassword$;
exports.ConfirmForgotPasswordCommand = ConfirmForgotPasswordCommand;
exports.ConfirmForgotPasswordRequest$ = ConfirmForgotPasswordRequest$;
exports.ConfirmForgotPasswordResponse$ = ConfirmForgotPasswordResponse$;
exports.ConfirmSignUp$ = ConfirmSignUp$;
exports.ConfirmSignUpCommand = ConfirmSignUpCommand;
exports.ConfirmSignUpRequest$ = ConfirmSignUpRequest$;
exports.ConfirmSignUpResponse$ = ConfirmSignUpResponse$;
exports.ContextDataType$ = ContextDataType$;
exports.CreateGroup$ = CreateGroup$;
exports.CreateGroupCommand = CreateGroupCommand;
exports.CreateGroupRequest$ = CreateGroupRequest$;
exports.CreateGroupResponse$ = CreateGroupResponse$;
exports.CreateIdentityProvider$ = CreateIdentityProvider$;
exports.CreateIdentityProviderCommand = CreateIdentityProviderCommand;
exports.CreateIdentityProviderRequest$ = CreateIdentityProviderRequest$;
exports.CreateIdentityProviderResponse$ = CreateIdentityProviderResponse$;
exports.CreateManagedLoginBranding$ = CreateManagedLoginBranding$;
exports.CreateManagedLoginBrandingCommand = CreateManagedLoginBrandingCommand;
exports.CreateManagedLoginBrandingRequest$ = CreateManagedLoginBrandingRequest$;
exports.CreateManagedLoginBrandingResponse$ = CreateManagedLoginBrandingResponse$;
exports.CreateResourceServer$ = CreateResourceServer$;
exports.CreateResourceServerCommand = CreateResourceServerCommand;
exports.CreateResourceServerRequest$ = CreateResourceServerRequest$;
exports.CreateResourceServerResponse$ = CreateResourceServerResponse$;
exports.CreateTerms$ = CreateTerms$;
exports.CreateTermsCommand = CreateTermsCommand;
exports.CreateTermsRequest$ = CreateTermsRequest$;
exports.CreateTermsResponse$ = CreateTermsResponse$;
exports.CreateUserImportJob$ = CreateUserImportJob$;
exports.CreateUserImportJobCommand = CreateUserImportJobCommand;
exports.CreateUserImportJobRequest$ = CreateUserImportJobRequest$;
exports.CreateUserImportJobResponse$ = CreateUserImportJobResponse$;
exports.CreateUserPool$ = CreateUserPool$;
exports.CreateUserPoolClient$ = CreateUserPoolClient$;
exports.CreateUserPoolClientCommand = CreateUserPoolClientCommand;
exports.CreateUserPoolClientRequest$ = CreateUserPoolClientRequest$;
exports.CreateUserPoolClientResponse$ = CreateUserPoolClientResponse$;
exports.CreateUserPoolCommand = CreateUserPoolCommand;
exports.CreateUserPoolDomain$ = CreateUserPoolDomain$;
exports.CreateUserPoolDomainCommand = CreateUserPoolDomainCommand;
exports.CreateUserPoolDomainRequest$ = CreateUserPoolDomainRequest$;
exports.CreateUserPoolDomainResponse$ = CreateUserPoolDomainResponse$;
exports.CreateUserPoolRequest$ = CreateUserPoolRequest$;
exports.CreateUserPoolResponse$ = CreateUserPoolResponse$;
exports.CustomDomainConfigType$ = CustomDomainConfigType$;
exports.CustomEmailLambdaVersionConfigType$ = CustomEmailLambdaVersionConfigType$;
exports.CustomEmailSenderLambdaVersionType = CustomEmailSenderLambdaVersionType;
exports.CustomSMSLambdaVersionConfigType$ = CustomSMSLambdaVersionConfigType$;
exports.CustomSMSSenderLambdaVersionType = CustomSMSSenderLambdaVersionType;
exports.DefaultEmailOptionType = DefaultEmailOptionType;
exports.DeleteGroup$ = DeleteGroup$;
exports.DeleteGroupCommand = DeleteGroupCommand;
exports.DeleteGroupRequest$ = DeleteGroupRequest$;
exports.DeleteIdentityProvider$ = DeleteIdentityProvider$;
exports.DeleteIdentityProviderCommand = DeleteIdentityProviderCommand;
exports.DeleteIdentityProviderRequest$ = DeleteIdentityProviderRequest$;
exports.DeleteManagedLoginBranding$ = DeleteManagedLoginBranding$;
exports.DeleteManagedLoginBrandingCommand = DeleteManagedLoginBrandingCommand;
exports.DeleteManagedLoginBrandingRequest$ = DeleteManagedLoginBrandingRequest$;
exports.DeleteResourceServer$ = DeleteResourceServer$;
exports.DeleteResourceServerCommand = DeleteResourceServerCommand;
exports.DeleteResourceServerRequest$ = DeleteResourceServerRequest$;
exports.DeleteTerms$ = DeleteTerms$;
exports.DeleteTermsCommand = DeleteTermsCommand;
exports.DeleteTermsRequest$ = DeleteTermsRequest$;
exports.DeleteUser$ = DeleteUser$;
exports.DeleteUserAttributes$ = DeleteUserAttributes$;
exports.DeleteUserAttributesCommand = DeleteUserAttributesCommand;
exports.DeleteUserAttributesRequest$ = DeleteUserAttributesRequest$;
exports.DeleteUserAttributesResponse$ = DeleteUserAttributesResponse$;
exports.DeleteUserCommand = DeleteUserCommand;
exports.DeleteUserPool$ = DeleteUserPool$;
exports.DeleteUserPoolClient$ = DeleteUserPoolClient$;
exports.DeleteUserPoolClientCommand = DeleteUserPoolClientCommand;
exports.DeleteUserPoolClientRequest$ = DeleteUserPoolClientRequest$;
exports.DeleteUserPoolCommand = DeleteUserPoolCommand;
exports.DeleteUserPoolDomain$ = DeleteUserPoolDomain$;
exports.DeleteUserPoolDomainCommand = DeleteUserPoolDomainCommand;
exports.DeleteUserPoolDomainRequest$ = DeleteUserPoolDomainRequest$;
exports.DeleteUserPoolDomainResponse$ = DeleteUserPoolDomainResponse$;
exports.DeleteUserPoolRequest$ = DeleteUserPoolRequest$;
exports.DeleteUserRequest$ = DeleteUserRequest$;
exports.DeleteWebAuthnCredential$ = DeleteWebAuthnCredential$;
exports.DeleteWebAuthnCredentialCommand = DeleteWebAuthnCredentialCommand;
exports.DeleteWebAuthnCredentialRequest$ = DeleteWebAuthnCredentialRequest$;
exports.DeleteWebAuthnCredentialResponse$ = DeleteWebAuthnCredentialResponse$;
exports.DeletionProtectionType = DeletionProtectionType;
exports.DeliveryMediumType = DeliveryMediumType;
exports.DescribeIdentityProvider$ = DescribeIdentityProvider$;
exports.DescribeIdentityProviderCommand = DescribeIdentityProviderCommand;
exports.DescribeIdentityProviderRequest$ = DescribeIdentityProviderRequest$;
exports.DescribeIdentityProviderResponse$ = DescribeIdentityProviderResponse$;
exports.DescribeManagedLoginBranding$ = DescribeManagedLoginBranding$;
exports.DescribeManagedLoginBrandingByClient$ = DescribeManagedLoginBrandingByClient$;
exports.DescribeManagedLoginBrandingByClientCommand = DescribeManagedLoginBrandingByClientCommand;
exports.DescribeManagedLoginBrandingByClientRequest$ = DescribeManagedLoginBrandingByClientRequest$;
exports.DescribeManagedLoginBrandingByClientResponse$ = DescribeManagedLoginBrandingByClientResponse$;
exports.DescribeManagedLoginBrandingCommand = DescribeManagedLoginBrandingCommand;
exports.DescribeManagedLoginBrandingRequest$ = DescribeManagedLoginBrandingRequest$;
exports.DescribeManagedLoginBrandingResponse$ = DescribeManagedLoginBrandingResponse$;
exports.DescribeResourceServer$ = DescribeResourceServer$;
exports.DescribeResourceServerCommand = DescribeResourceServerCommand;
exports.DescribeResourceServerRequest$ = DescribeResourceServerRequest$;
exports.DescribeResourceServerResponse$ = DescribeResourceServerResponse$;
exports.DescribeRiskConfiguration$ = DescribeRiskConfiguration$;
exports.DescribeRiskConfigurationCommand = DescribeRiskConfigurationCommand;
exports.DescribeRiskConfigurationRequest$ = DescribeRiskConfigurationRequest$;
exports.DescribeRiskConfigurationResponse$ = DescribeRiskConfigurationResponse$;
exports.DescribeTerms$ = DescribeTerms$;
exports.DescribeTermsCommand = DescribeTermsCommand;
exports.DescribeTermsRequest$ = DescribeTermsRequest$;
exports.DescribeTermsResponse$ = DescribeTermsResponse$;
exports.DescribeUserImportJob$ = DescribeUserImportJob$;
exports.DescribeUserImportJobCommand = DescribeUserImportJobCommand;
exports.DescribeUserImportJobRequest$ = DescribeUserImportJobRequest$;
exports.DescribeUserImportJobResponse$ = DescribeUserImportJobResponse$;
exports.DescribeUserPool$ = DescribeUserPool$;
exports.DescribeUserPoolClient$ = DescribeUserPoolClient$;
exports.DescribeUserPoolClientCommand = DescribeUserPoolClientCommand;
exports.DescribeUserPoolClientRequest$ = DescribeUserPoolClientRequest$;
exports.DescribeUserPoolClientResponse$ = DescribeUserPoolClientResponse$;
exports.DescribeUserPoolCommand = DescribeUserPoolCommand;
exports.DescribeUserPoolDomain$ = DescribeUserPoolDomain$;
exports.DescribeUserPoolDomainCommand = DescribeUserPoolDomainCommand;
exports.DescribeUserPoolDomainRequest$ = DescribeUserPoolDomainRequest$;
exports.DescribeUserPoolDomainResponse$ = DescribeUserPoolDomainResponse$;
exports.DescribeUserPoolRequest$ = DescribeUserPoolRequest$;
exports.DescribeUserPoolResponse$ = DescribeUserPoolResponse$;
exports.DeviceConfigurationType$ = DeviceConfigurationType$;
exports.DeviceKeyExistsException = DeviceKeyExistsException;
exports.DeviceKeyExistsException$ = DeviceKeyExistsException$;
exports.DeviceRememberedStatusType = DeviceRememberedStatusType;
exports.DeviceSecretVerifierConfigType$ = DeviceSecretVerifierConfigType$;
exports.DeviceType$ = DeviceType$;
exports.DomainDescriptionType$ = DomainDescriptionType$;
exports.DomainStatusType = DomainStatusType;
exports.DuplicateProviderException = DuplicateProviderException;
exports.DuplicateProviderException$ = DuplicateProviderException$;
exports.EmailConfigurationType$ = EmailConfigurationType$;
exports.EmailMfaConfigType$ = EmailMfaConfigType$;
exports.EmailMfaSettingsType$ = EmailMfaSettingsType$;
exports.EmailSendingAccountType = EmailSendingAccountType;
exports.EnableSoftwareTokenMFAException = EnableSoftwareTokenMFAException;
exports.EnableSoftwareTokenMFAException$ = EnableSoftwareTokenMFAException$;
exports.EventContextDataType$ = EventContextDataType$;
exports.EventFeedbackType$ = EventFeedbackType$;
exports.EventFilterType = EventFilterType;
exports.EventResponseType = EventResponseType;
exports.EventRiskType$ = EventRiskType$;
exports.EventSourceName = EventSourceName;
exports.EventType = EventType;
exports.ExpiredCodeException = ExpiredCodeException;
exports.ExpiredCodeException$ = ExpiredCodeException$;
exports.ExplicitAuthFlowsType = ExplicitAuthFlowsType;
exports.FeatureType = FeatureType;
exports.FeatureUnavailableInTierException = FeatureUnavailableInTierException;
exports.FeatureUnavailableInTierException$ = FeatureUnavailableInTierException$;
exports.FeedbackValueType = FeedbackValueType;
exports.FirehoseConfigurationType$ = FirehoseConfigurationType$;
exports.ForbiddenException = ForbiddenException;
exports.ForbiddenException$ = ForbiddenException$;
exports.ForgetDevice$ = ForgetDevice$;
exports.ForgetDeviceCommand = ForgetDeviceCommand;
exports.ForgetDeviceRequest$ = ForgetDeviceRequest$;
exports.ForgotPassword$ = ForgotPassword$;
exports.ForgotPasswordCommand = ForgotPasswordCommand;
exports.ForgotPasswordRequest$ = ForgotPasswordRequest$;
exports.ForgotPasswordResponse$ = ForgotPasswordResponse$;
exports.GetCSVHeader$ = GetCSVHeader$;
exports.GetCSVHeaderCommand = GetCSVHeaderCommand;
exports.GetCSVHeaderRequest$ = GetCSVHeaderRequest$;
exports.GetCSVHeaderResponse$ = GetCSVHeaderResponse$;
exports.GetDevice$ = GetDevice$;
exports.GetDeviceCommand = GetDeviceCommand;
exports.GetDeviceRequest$ = GetDeviceRequest$;
exports.GetDeviceResponse$ = GetDeviceResponse$;
exports.GetGroup$ = GetGroup$;
exports.GetGroupCommand = GetGroupCommand;
exports.GetGroupRequest$ = GetGroupRequest$;
exports.GetGroupResponse$ = GetGroupResponse$;
exports.GetIdentityProviderByIdentifier$ = GetIdentityProviderByIdentifier$;
exports.GetIdentityProviderByIdentifierCommand = GetIdentityProviderByIdentifierCommand;
exports.GetIdentityProviderByIdentifierRequest$ = GetIdentityProviderByIdentifierRequest$;
exports.GetIdentityProviderByIdentifierResponse$ = GetIdentityProviderByIdentifierResponse$;
exports.GetLogDeliveryConfiguration$ = GetLogDeliveryConfiguration$;
exports.GetLogDeliveryConfigurationCommand = GetLogDeliveryConfigurationCommand;
exports.GetLogDeliveryConfigurationRequest$ = GetLogDeliveryConfigurationRequest$;
exports.GetLogDeliveryConfigurationResponse$ = GetLogDeliveryConfigurationResponse$;
exports.GetSigningCertificate$ = GetSigningCertificate$;
exports.GetSigningCertificateCommand = GetSigningCertificateCommand;
exports.GetSigningCertificateRequest$ = GetSigningCertificateRequest$;
exports.GetSigningCertificateResponse$ = GetSigningCertificateResponse$;
exports.GetTokensFromRefreshToken$ = GetTokensFromRefreshToken$;
exports.GetTokensFromRefreshTokenCommand = GetTokensFromRefreshTokenCommand;
exports.GetTokensFromRefreshTokenRequest$ = GetTokensFromRefreshTokenRequest$;
exports.GetTokensFromRefreshTokenResponse$ = GetTokensFromRefreshTokenResponse$;
exports.GetUICustomization$ = GetUICustomization$;
exports.GetUICustomizationCommand = GetUICustomizationCommand;
exports.GetUICustomizationRequest$ = GetUICustomizationRequest$;
exports.GetUICustomizationResponse$ = GetUICustomizationResponse$;
exports.GetUser$ = GetUser$;
exports.GetUserAttributeVerificationCode$ = GetUserAttributeVerificationCode$;
exports.GetUserAttributeVerificationCodeCommand = GetUserAttributeVerificationCodeCommand;
exports.GetUserAttributeVerificationCodeRequest$ = GetUserAttributeVerificationCodeRequest$;
exports.GetUserAttributeVerificationCodeResponse$ = GetUserAttributeVerificationCodeResponse$;
exports.GetUserAuthFactors$ = GetUserAuthFactors$;
exports.GetUserAuthFactorsCommand = GetUserAuthFactorsCommand;
exports.GetUserAuthFactorsRequest$ = GetUserAuthFactorsRequest$;
exports.GetUserAuthFactorsResponse$ = GetUserAuthFactorsResponse$;
exports.GetUserCommand = GetUserCommand;
exports.GetUserPoolMfaConfig$ = GetUserPoolMfaConfig$;
exports.GetUserPoolMfaConfigCommand = GetUserPoolMfaConfigCommand;
exports.GetUserPoolMfaConfigRequest$ = GetUserPoolMfaConfigRequest$;
exports.GetUserPoolMfaConfigResponse$ = GetUserPoolMfaConfigResponse$;
exports.GetUserRequest$ = GetUserRequest$;
exports.GetUserResponse$ = GetUserResponse$;
exports.GlobalSignOut$ = GlobalSignOut$;
exports.GlobalSignOutCommand = GlobalSignOutCommand;
exports.GlobalSignOutRequest$ = GlobalSignOutRequest$;
exports.GlobalSignOutResponse$ = GlobalSignOutResponse$;
exports.GroupExistsException = GroupExistsException;
exports.GroupExistsException$ = GroupExistsException$;
exports.GroupType$ = GroupType$;
exports.HttpHeader$ = HttpHeader$;
exports.IdentityProviderType$ = IdentityProviderType$;
exports.IdentityProviderTypeType = IdentityProviderTypeType;
exports.InitiateAuth$ = InitiateAuth$;
exports.InitiateAuthCommand = InitiateAuthCommand;
exports.InitiateAuthRequest$ = InitiateAuthRequest$;
exports.InitiateAuthResponse$ = InitiateAuthResponse$;
exports.InternalErrorException = InternalErrorException;
exports.InternalErrorException$ = InternalErrorException$;
exports.InvalidEmailRoleAccessPolicyException = InvalidEmailRoleAccessPolicyException;
exports.InvalidEmailRoleAccessPolicyException$ = InvalidEmailRoleAccessPolicyException$;
exports.InvalidLambdaResponseException = InvalidLambdaResponseException;
exports.InvalidLambdaResponseException$ = InvalidLambdaResponseException$;
exports.InvalidOAuthFlowException = InvalidOAuthFlowException;
exports.InvalidOAuthFlowException$ = InvalidOAuthFlowException$;
exports.InvalidParameterException = InvalidParameterException;
exports.InvalidParameterException$ = InvalidParameterException$;
exports.InvalidPasswordException = InvalidPasswordException;
exports.InvalidPasswordException$ = InvalidPasswordException$;
exports.InvalidSmsRoleAccessPolicyException = InvalidSmsRoleAccessPolicyException;
exports.InvalidSmsRoleAccessPolicyException$ = InvalidSmsRoleAccessPolicyException$;
exports.InvalidSmsRoleTrustRelationshipException = InvalidSmsRoleTrustRelationshipException;
exports.InvalidSmsRoleTrustRelationshipException$ = InvalidSmsRoleTrustRelationshipException$;
exports.InvalidUserPoolConfigurationException = InvalidUserPoolConfigurationException;
exports.InvalidUserPoolConfigurationException$ = InvalidUserPoolConfigurationException$;
exports.LambdaConfigType$ = LambdaConfigType$;
exports.LimitExceededException = LimitExceededException;
exports.LimitExceededException$ = LimitExceededException$;
exports.ListDevices$ = ListDevices$;
exports.ListDevicesCommand = ListDevicesCommand;
exports.ListDevicesRequest$ = ListDevicesRequest$;
exports.ListDevicesResponse$ = ListDevicesResponse$;
exports.ListGroups$ = ListGroups$;
exports.ListGroupsCommand = ListGroupsCommand;
exports.ListGroupsRequest$ = ListGroupsRequest$;
exports.ListGroupsResponse$ = ListGroupsResponse$;
exports.ListIdentityProviders$ = ListIdentityProviders$;
exports.ListIdentityProvidersCommand = ListIdentityProvidersCommand;
exports.ListIdentityProvidersRequest$ = ListIdentityProvidersRequest$;
exports.ListIdentityProvidersResponse$ = ListIdentityProvidersResponse$;
exports.ListResourceServers$ = ListResourceServers$;
exports.ListResourceServersCommand = ListResourceServersCommand;
exports.ListResourceServersRequest$ = ListResourceServersRequest$;
exports.ListResourceServersResponse$ = ListResourceServersResponse$;
exports.ListTagsForResource$ = ListTagsForResource$;
exports.ListTagsForResourceCommand = ListTagsForResourceCommand;
exports.ListTagsForResourceRequest$ = ListTagsForResourceRequest$;
exports.ListTagsForResourceResponse$ = ListTagsForResourceResponse$;
exports.ListTerms$ = ListTerms$;
exports.ListTermsCommand = ListTermsCommand;
exports.ListTermsRequest$ = ListTermsRequest$;
exports.ListTermsResponse$ = ListTermsResponse$;
exports.ListUserImportJobs$ = ListUserImportJobs$;
exports.ListUserImportJobsCommand = ListUserImportJobsCommand;
exports.ListUserImportJobsRequest$ = ListUserImportJobsRequest$;
exports.ListUserImportJobsResponse$ = ListUserImportJobsResponse$;
exports.ListUserPoolClients$ = ListUserPoolClients$;
exports.ListUserPoolClientsCommand = ListUserPoolClientsCommand;
exports.ListUserPoolClientsRequest$ = ListUserPoolClientsRequest$;
exports.ListUserPoolClientsResponse$ = ListUserPoolClientsResponse$;
exports.ListUserPools$ = ListUserPools$;
exports.ListUserPoolsCommand = ListUserPoolsCommand;
exports.ListUserPoolsRequest$ = ListUserPoolsRequest$;
exports.ListUserPoolsResponse$ = ListUserPoolsResponse$;
exports.ListUsers$ = ListUsers$;
exports.ListUsersCommand = ListUsersCommand;
exports.ListUsersInGroup$ = ListUsersInGroup$;
exports.ListUsersInGroupCommand = ListUsersInGroupCommand;
exports.ListUsersInGroupRequest$ = ListUsersInGroupRequest$;
exports.ListUsersInGroupResponse$ = ListUsersInGroupResponse$;
exports.ListUsersRequest$ = ListUsersRequest$;
exports.ListUsersResponse$ = ListUsersResponse$;
exports.ListWebAuthnCredentials$ = ListWebAuthnCredentials$;
exports.ListWebAuthnCredentialsCommand = ListWebAuthnCredentialsCommand;
exports.ListWebAuthnCredentialsRequest$ = ListWebAuthnCredentialsRequest$;
exports.ListWebAuthnCredentialsResponse$ = ListWebAuthnCredentialsResponse$;
exports.LogConfigurationType$ = LogConfigurationType$;
exports.LogDeliveryConfigurationType$ = LogDeliveryConfigurationType$;
exports.LogLevel = LogLevel;
exports.MFAMethodNotFoundException = MFAMethodNotFoundException;
exports.MFAMethodNotFoundException$ = MFAMethodNotFoundException$;
exports.MFAOptionType$ = MFAOptionType$;
exports.ManagedLoginBrandingExistsException = ManagedLoginBrandingExistsException;
exports.ManagedLoginBrandingExistsException$ = ManagedLoginBrandingExistsException$;
exports.ManagedLoginBrandingType$ = ManagedLoginBrandingType$;
exports.MessageActionType = MessageActionType;
exports.MessageTemplateType$ = MessageTemplateType$;
exports.NewDeviceMetadataType$ = NewDeviceMetadataType$;
exports.NotAuthorizedException = NotAuthorizedException;
exports.NotAuthorizedException$ = NotAuthorizedException$;
exports.NotifyConfigurationType$ = NotifyConfigurationType$;
exports.NotifyEmailType$ = NotifyEmailType$;
exports.NumberAttributeConstraintsType$ = NumberAttributeConstraintsType$;
exports.OAuthFlowType = OAuthFlowType;
exports.PasswordHistoryPolicyViolationException = PasswordHistoryPolicyViolationException;
exports.PasswordHistoryPolicyViolationException$ = PasswordHistoryPolicyViolationException$;
exports.PasswordPolicyType$ = PasswordPolicyType$;
exports.PasswordResetRequiredException = PasswordResetRequiredException;
exports.PasswordResetRequiredException$ = PasswordResetRequiredException$;
exports.PreTokenGenerationLambdaVersionType = PreTokenGenerationLambdaVersionType;
exports.PreTokenGenerationVersionConfigType$ = PreTokenGenerationVersionConfigType$;
exports.PreconditionNotMetException = PreconditionNotMetException;
exports.PreconditionNotMetException$ = PreconditionNotMetException$;
exports.PreventUserExistenceErrorTypes = PreventUserExistenceErrorTypes;
exports.ProviderDescription$ = ProviderDescription$;
exports.ProviderUserIdentifierType$ = ProviderUserIdentifierType$;
exports.RecoveryOptionNameType = RecoveryOptionNameType;
exports.RecoveryOptionType$ = RecoveryOptionType$;
exports.RefreshTokenReuseException = RefreshTokenReuseException;
exports.RefreshTokenReuseException$ = RefreshTokenReuseException$;
exports.RefreshTokenRotationType$ = RefreshTokenRotationType$;
exports.ResendConfirmationCode$ = ResendConfirmationCode$;
exports.ResendConfirmationCodeCommand = ResendConfirmationCodeCommand;
exports.ResendConfirmationCodeRequest$ = ResendConfirmationCodeRequest$;
exports.ResendConfirmationCodeResponse$ = ResendConfirmationCodeResponse$;
exports.ResourceNotFoundException = ResourceNotFoundException;
exports.ResourceNotFoundException$ = ResourceNotFoundException$;
exports.ResourceServerScopeType$ = ResourceServerScopeType$;
exports.ResourceServerType$ = ResourceServerType$;
exports.RespondToAuthChallenge$ = RespondToAuthChallenge$;
exports.RespondToAuthChallengeCommand = RespondToAuthChallengeCommand;
exports.RespondToAuthChallengeRequest$ = RespondToAuthChallengeRequest$;
exports.RespondToAuthChallengeResponse$ = RespondToAuthChallengeResponse$;
exports.RevokeToken$ = RevokeToken$;
exports.RevokeTokenCommand = RevokeTokenCommand;
exports.RevokeTokenRequest$ = RevokeTokenRequest$;
exports.RevokeTokenResponse$ = RevokeTokenResponse$;
exports.RiskConfigurationType$ = RiskConfigurationType$;
exports.RiskDecisionType = RiskDecisionType;
exports.RiskExceptionConfigurationType$ = RiskExceptionConfigurationType$;
exports.RiskLevelType = RiskLevelType;
exports.S3ConfigurationType$ = S3ConfigurationType$;
exports.SMSMfaSettingsType$ = SMSMfaSettingsType$;
exports.SchemaAttributeType$ = SchemaAttributeType$;
exports.ScopeDoesNotExistException = ScopeDoesNotExistException;
exports.ScopeDoesNotExistException$ = ScopeDoesNotExistException$;
exports.SetLogDeliveryConfiguration$ = SetLogDeliveryConfiguration$;
exports.SetLogDeliveryConfigurationCommand = SetLogDeliveryConfigurationCommand;
exports.SetLogDeliveryConfigurationRequest$ = SetLogDeliveryConfigurationRequest$;
exports.SetLogDeliveryConfigurationResponse$ = SetLogDeliveryConfigurationResponse$;
exports.SetRiskConfiguration$ = SetRiskConfiguration$;
exports.SetRiskConfigurationCommand = SetRiskConfigurationCommand;
exports.SetRiskConfigurationRequest$ = SetRiskConfigurationRequest$;
exports.SetRiskConfigurationResponse$ = SetRiskConfigurationResponse$;
exports.SetUICustomization$ = SetUICustomization$;
exports.SetUICustomizationCommand = SetUICustomizationCommand;
exports.SetUICustomizationRequest$ = SetUICustomizationRequest$;
exports.SetUICustomizationResponse$ = SetUICustomizationResponse$;
exports.SetUserMFAPreference$ = SetUserMFAPreference$;
exports.SetUserMFAPreferenceCommand = SetUserMFAPreferenceCommand;
exports.SetUserMFAPreferenceRequest$ = SetUserMFAPreferenceRequest$;
exports.SetUserMFAPreferenceResponse$ = SetUserMFAPreferenceResponse$;
exports.SetUserPoolMfaConfig$ = SetUserPoolMfaConfig$;
exports.SetUserPoolMfaConfigCommand = SetUserPoolMfaConfigCommand;
exports.SetUserPoolMfaConfigRequest$ = SetUserPoolMfaConfigRequest$;
exports.SetUserPoolMfaConfigResponse$ = SetUserPoolMfaConfigResponse$;
exports.SetUserSettings$ = SetUserSettings$;
exports.SetUserSettingsCommand = SetUserSettingsCommand;
exports.SetUserSettingsRequest$ = SetUserSettingsRequest$;
exports.SetUserSettingsResponse$ = SetUserSettingsResponse$;
exports.SignInPolicyType$ = SignInPolicyType$;
exports.SignUp$ = SignUp$;
exports.SignUpCommand = SignUpCommand;
exports.SignUpRequest$ = SignUpRequest$;
exports.SignUpResponse$ = SignUpResponse$;
exports.SmsConfigurationType$ = SmsConfigurationType$;
exports.SmsMfaConfigType$ = SmsMfaConfigType$;
exports.SoftwareTokenMFANotFoundException = SoftwareTokenMFANotFoundException;
exports.SoftwareTokenMFANotFoundException$ = SoftwareTokenMFANotFoundException$;
exports.SoftwareTokenMfaConfigType$ = SoftwareTokenMfaConfigType$;
exports.SoftwareTokenMfaSettingsType$ = SoftwareTokenMfaSettingsType$;
exports.StartUserImportJob$ = StartUserImportJob$;
exports.StartUserImportJobCommand = StartUserImportJobCommand;
exports.StartUserImportJobRequest$ = StartUserImportJobRequest$;
exports.StartUserImportJobResponse$ = StartUserImportJobResponse$;
exports.StartWebAuthnRegistration$ = StartWebAuthnRegistration$;
exports.StartWebAuthnRegistrationCommand = StartWebAuthnRegistrationCommand;
exports.StartWebAuthnRegistrationRequest$ = StartWebAuthnRegistrationRequest$;
exports.StartWebAuthnRegistrationResponse$ = StartWebAuthnRegistrationResponse$;
exports.StatusType = StatusType;
exports.StopUserImportJob$ = StopUserImportJob$;
exports.StopUserImportJobCommand = StopUserImportJobCommand;
exports.StopUserImportJobRequest$ = StopUserImportJobRequest$;
exports.StopUserImportJobResponse$ = StopUserImportJobResponse$;
exports.StringAttributeConstraintsType$ = StringAttributeConstraintsType$;
exports.TagResource$ = TagResource$;
exports.TagResourceCommand = TagResourceCommand;
exports.TagResourceRequest$ = TagResourceRequest$;
exports.TagResourceResponse$ = TagResourceResponse$;
exports.TermsDescriptionType$ = TermsDescriptionType$;
exports.TermsEnforcementType = TermsEnforcementType;
exports.TermsExistsException = TermsExistsException;
exports.TermsExistsException$ = TermsExistsException$;
exports.TermsSourceType = TermsSourceType;
exports.TermsType$ = TermsType$;
exports.TierChangeNotAllowedException = TierChangeNotAllowedException;
exports.TierChangeNotAllowedException$ = TierChangeNotAllowedException$;
exports.TimeUnitsType = TimeUnitsType;
exports.TokenValidityUnitsType$ = TokenValidityUnitsType$;
exports.TooManyFailedAttemptsException = TooManyFailedAttemptsException;
exports.TooManyFailedAttemptsException$ = TooManyFailedAttemptsException$;
exports.TooManyRequestsException = TooManyRequestsException;
exports.TooManyRequestsException$ = TooManyRequestsException$;
exports.UICustomizationType$ = UICustomizationType$;
exports.UnauthorizedException = UnauthorizedException;
exports.UnauthorizedException$ = UnauthorizedException$;
exports.UnexpectedLambdaException = UnexpectedLambdaException;
exports.UnexpectedLambdaException$ = UnexpectedLambdaException$;
exports.UnsupportedIdentityProviderException = UnsupportedIdentityProviderException;
exports.UnsupportedIdentityProviderException$ = UnsupportedIdentityProviderException$;
exports.UnsupportedOperationException = UnsupportedOperationException;
exports.UnsupportedOperationException$ = UnsupportedOperationException$;
exports.UnsupportedTokenTypeException = UnsupportedTokenTypeException;
exports.UnsupportedTokenTypeException$ = UnsupportedTokenTypeException$;
exports.UnsupportedUserStateException = UnsupportedUserStateException;
exports.UnsupportedUserStateException$ = UnsupportedUserStateException$;
exports.UntagResource$ = UntagResource$;
exports.UntagResourceCommand = UntagResourceCommand;
exports.UntagResourceRequest$ = UntagResourceRequest$;
exports.UntagResourceResponse$ = UntagResourceResponse$;
exports.UpdateAuthEventFeedback$ = UpdateAuthEventFeedback$;
exports.UpdateAuthEventFeedbackCommand = UpdateAuthEventFeedbackCommand;
exports.UpdateAuthEventFeedbackRequest$ = UpdateAuthEventFeedbackRequest$;
exports.UpdateAuthEventFeedbackResponse$ = UpdateAuthEventFeedbackResponse$;
exports.UpdateDeviceStatus$ = UpdateDeviceStatus$;
exports.UpdateDeviceStatusCommand = UpdateDeviceStatusCommand;
exports.UpdateDeviceStatusRequest$ = UpdateDeviceStatusRequest$;
exports.UpdateDeviceStatusResponse$ = UpdateDeviceStatusResponse$;
exports.UpdateGroup$ = UpdateGroup$;
exports.UpdateGroupCommand = UpdateGroupCommand;
exports.UpdateGroupRequest$ = UpdateGroupRequest$;
exports.UpdateGroupResponse$ = UpdateGroupResponse$;
exports.UpdateIdentityProvider$ = UpdateIdentityProvider$;
exports.UpdateIdentityProviderCommand = UpdateIdentityProviderCommand;
exports.UpdateIdentityProviderRequest$ = UpdateIdentityProviderRequest$;
exports.UpdateIdentityProviderResponse$ = UpdateIdentityProviderResponse$;
exports.UpdateManagedLoginBranding$ = UpdateManagedLoginBranding$;
exports.UpdateManagedLoginBrandingCommand = UpdateManagedLoginBrandingCommand;
exports.UpdateManagedLoginBrandingRequest$ = UpdateManagedLoginBrandingRequest$;
exports.UpdateManagedLoginBrandingResponse$ = UpdateManagedLoginBrandingResponse$;
exports.UpdateResourceServer$ = UpdateResourceServer$;
exports.UpdateResourceServerCommand = UpdateResourceServerCommand;
exports.UpdateResourceServerRequest$ = UpdateResourceServerRequest$;
exports.UpdateResourceServerResponse$ = UpdateResourceServerResponse$;
exports.UpdateTerms$ = UpdateTerms$;
exports.UpdateTermsCommand = UpdateTermsCommand;
exports.UpdateTermsRequest$ = UpdateTermsRequest$;
exports.UpdateTermsResponse$ = UpdateTermsResponse$;
exports.UpdateUserAttributes$ = UpdateUserAttributes$;
exports.UpdateUserAttributesCommand = UpdateUserAttributesCommand;
exports.UpdateUserAttributesRequest$ = UpdateUserAttributesRequest$;
exports.UpdateUserAttributesResponse$ = UpdateUserAttributesResponse$;
exports.UpdateUserPool$ = UpdateUserPool$;
exports.UpdateUserPoolClient$ = UpdateUserPoolClient$;
exports.UpdateUserPoolClientCommand = UpdateUserPoolClientCommand;
exports.UpdateUserPoolClientRequest$ = UpdateUserPoolClientRequest$;
exports.UpdateUserPoolClientResponse$ = UpdateUserPoolClientResponse$;
exports.UpdateUserPoolCommand = UpdateUserPoolCommand;
exports.UpdateUserPoolDomain$ = UpdateUserPoolDomain$;
exports.UpdateUserPoolDomainCommand = UpdateUserPoolDomainCommand;
exports.UpdateUserPoolDomainRequest$ = UpdateUserPoolDomainRequest$;
exports.UpdateUserPoolDomainResponse$ = UpdateUserPoolDomainResponse$;
exports.UpdateUserPoolRequest$ = UpdateUserPoolRequest$;
exports.UpdateUserPoolResponse$ = UpdateUserPoolResponse$;
exports.UserAttributeUpdateSettingsType$ = UserAttributeUpdateSettingsType$;
exports.UserContextDataType$ = UserContextDataType$;
exports.UserImportInProgressException = UserImportInProgressException;
exports.UserImportInProgressException$ = UserImportInProgressException$;
exports.UserImportJobStatusType = UserImportJobStatusType;
exports.UserImportJobType$ = UserImportJobType$;
exports.UserLambdaValidationException = UserLambdaValidationException;
exports.UserLambdaValidationException$ = UserLambdaValidationException$;
exports.UserNotConfirmedException = UserNotConfirmedException;
exports.UserNotConfirmedException$ = UserNotConfirmedException$;
exports.UserNotFoundException = UserNotFoundException;
exports.UserNotFoundException$ = UserNotFoundException$;
exports.UserPoolAddOnNotEnabledException = UserPoolAddOnNotEnabledException;
exports.UserPoolAddOnNotEnabledException$ = UserPoolAddOnNotEnabledException$;
exports.UserPoolAddOnsType$ = UserPoolAddOnsType$;
exports.UserPoolClientDescription$ = UserPoolClientDescription$;
exports.UserPoolClientType$ = UserPoolClientType$;
exports.UserPoolDescriptionType$ = UserPoolDescriptionType$;
exports.UserPoolMfaType = UserPoolMfaType;
exports.UserPoolPolicyType$ = UserPoolPolicyType$;
exports.UserPoolTaggingException = UserPoolTaggingException;
exports.UserPoolTaggingException$ = UserPoolTaggingException$;
exports.UserPoolTierType = UserPoolTierType;
exports.UserPoolType$ = UserPoolType$;
exports.UserStatusType = UserStatusType;
exports.UserType$ = UserType$;
exports.UserVerificationType = UserVerificationType;
exports.UsernameAttributeType = UsernameAttributeType;
exports.UsernameConfigurationType$ = UsernameConfigurationType$;
exports.UsernameExistsException = UsernameExistsException;
exports.UsernameExistsException$ = UsernameExistsException$;
exports.VerificationMessageTemplateType$ = VerificationMessageTemplateType$;
exports.VerifiedAttributeType = VerifiedAttributeType;
exports.VerifySoftwareToken$ = VerifySoftwareToken$;
exports.VerifySoftwareTokenCommand = VerifySoftwareTokenCommand;
exports.VerifySoftwareTokenRequest$ = VerifySoftwareTokenRequest$;
exports.VerifySoftwareTokenResponse$ = VerifySoftwareTokenResponse$;
exports.VerifySoftwareTokenResponseType = VerifySoftwareTokenResponseType;
exports.VerifyUserAttribute$ = VerifyUserAttribute$;
exports.VerifyUserAttributeCommand = VerifyUserAttributeCommand;
exports.VerifyUserAttributeRequest$ = VerifyUserAttributeRequest$;
exports.VerifyUserAttributeResponse$ = VerifyUserAttributeResponse$;
exports.WebAuthnChallengeNotFoundException = WebAuthnChallengeNotFoundException;
exports.WebAuthnChallengeNotFoundException$ = WebAuthnChallengeNotFoundException$;
exports.WebAuthnClientMismatchException = WebAuthnClientMismatchException;
exports.WebAuthnClientMismatchException$ = WebAuthnClientMismatchException$;
exports.WebAuthnConfigurationMissingException = WebAuthnConfigurationMissingException;
exports.WebAuthnConfigurationMissingException$ = WebAuthnConfigurationMissingException$;
exports.WebAuthnConfigurationType$ = WebAuthnConfigurationType$;
exports.WebAuthnCredentialDescription$ = WebAuthnCredentialDescription$;
exports.WebAuthnCredentialNotSupportedException = WebAuthnCredentialNotSupportedException;
exports.WebAuthnCredentialNotSupportedException$ = WebAuthnCredentialNotSupportedException$;
exports.WebAuthnNotEnabledException = WebAuthnNotEnabledException;
exports.WebAuthnNotEnabledException$ = WebAuthnNotEnabledException$;
exports.WebAuthnOriginNotAllowedException = WebAuthnOriginNotAllowedException;
exports.WebAuthnOriginNotAllowedException$ = WebAuthnOriginNotAllowedException$;
exports.WebAuthnRelyingPartyMismatchException = WebAuthnRelyingPartyMismatchException;
exports.WebAuthnRelyingPartyMismatchException$ = WebAuthnRelyingPartyMismatchException$;
exports.paginateAdminListGroupsForUser = paginateAdminListGroupsForUser;
exports.paginateAdminListUserAuthEvents = paginateAdminListUserAuthEvents;
exports.paginateListGroups = paginateListGroups;
exports.paginateListIdentityProviders = paginateListIdentityProviders;
exports.paginateListResourceServers = paginateListResourceServers;
exports.paginateListUserPoolClients = paginateListUserPoolClients;
exports.paginateListUserPools = paginateListUserPools;
exports.paginateListUsers = paginateListUsers;
exports.paginateListUsersInGroup = paginateListUsersInGroup;
