'use strict';

var middlewareHostHeader = require('@aws-sdk/middleware-host-header');
var middlewareLogger = require('@aws-sdk/middleware-logger');
var middlewareRecursionDetection = require('@aws-sdk/middleware-recursion-detection');
var middlewareUserAgent = require('@aws-sdk/middleware-user-agent');
var configResolver = require('@smithy/config-resolver');
var core = require('@smithy/core');
var schema = require('@smithy/core/schema');
var eventstreamSerdeConfigResolver = require('@smithy/eventstream-serde-config-resolver');
var middlewareContentLength = require('@smithy/middleware-content-length');
var middlewareEndpoint = require('@smithy/middleware-endpoint');
var middlewareRetry = require('@smithy/middleware-retry');
var smithyClient = require('@smithy/smithy-client');
var httpAuthSchemeProvider = require('./auth/httpAuthSchemeProvider');
var runtimeConfig = require('./runtimeConfig');
var regionConfigResolver = require('@aws-sdk/region-config-resolver');
var protocolHttp = require('@smithy/protocol-http');
var utilWaiter = require('@smithy/util-waiter');

const resolveClientEndpointParameters = (options) => {
    return Object.assign(options, {
        useDualstackEndpoint: options.useDualstackEndpoint ?? false,
        useFipsEndpoint: options.useFipsEndpoint ?? false,
        defaultSigningName: "kinesis",
    });
};
const commonParams = {
    UseFIPS: { type: "builtInParams", name: "useFipsEndpoint" },
    Endpoint: { type: "builtInParams", name: "endpoint" },
    Region: { type: "builtInParams", name: "region" },
    UseDualStack: { type: "builtInParams", name: "useDualstackEndpoint" },
};

const getHttpAuthExtensionConfiguration = (runtimeConfig) => {
    const _httpAuthSchemes = runtimeConfig.httpAuthSchemes;
    let _httpAuthSchemeProvider = runtimeConfig.httpAuthSchemeProvider;
    let _credentials = runtimeConfig.credentials;
    return {
        setHttpAuthScheme(httpAuthScheme) {
            const index = _httpAuthSchemes.findIndex((scheme) => scheme.schemeId === httpAuthScheme.schemeId);
            if (index === -1) {
                _httpAuthSchemes.push(httpAuthScheme);
            }
            else {
                _httpAuthSchemes.splice(index, 1, httpAuthScheme);
            }
        },
        httpAuthSchemes() {
            return _httpAuthSchemes;
        },
        setHttpAuthSchemeProvider(httpAuthSchemeProvider) {
            _httpAuthSchemeProvider = httpAuthSchemeProvider;
        },
        httpAuthSchemeProvider() {
            return _httpAuthSchemeProvider;
        },
        setCredentials(credentials) {
            _credentials = credentials;
        },
        credentials() {
            return _credentials;
        },
    };
};
const resolveHttpAuthRuntimeConfig = (config) => {
    return {
        httpAuthSchemes: config.httpAuthSchemes(),
        httpAuthSchemeProvider: config.httpAuthSchemeProvider(),
        credentials: config.credentials(),
    };
};

const resolveRuntimeExtensions = (runtimeConfig, extensions) => {
    const extensionConfiguration = Object.assign(regionConfigResolver.getAwsRegionExtensionConfiguration(runtimeConfig), smithyClient.getDefaultExtensionConfiguration(runtimeConfig), protocolHttp.getHttpHandlerExtensionConfiguration(runtimeConfig), getHttpAuthExtensionConfiguration(runtimeConfig));
    extensions.forEach((extension) => extension.configure(extensionConfiguration));
    return Object.assign(runtimeConfig, regionConfigResolver.resolveAwsRegionExtensionConfiguration(extensionConfiguration), smithyClient.resolveDefaultRuntimeConfig(extensionConfiguration), protocolHttp.resolveHttpHandlerRuntimeConfig(extensionConfiguration), resolveHttpAuthRuntimeConfig(extensionConfiguration));
};

class KinesisClient extends smithyClient.Client {
    config;
    constructor(...[configuration]) {
        const _config_0 = runtimeConfig.getRuntimeConfig(configuration || {});
        super(_config_0);
        this.initConfig = _config_0;
        const _config_1 = resolveClientEndpointParameters(_config_0);
        const _config_2 = middlewareUserAgent.resolveUserAgentConfig(_config_1);
        const _config_3 = middlewareRetry.resolveRetryConfig(_config_2);
        const _config_4 = configResolver.resolveRegionConfig(_config_3);
        const _config_5 = middlewareHostHeader.resolveHostHeaderConfig(_config_4);
        const _config_6 = middlewareEndpoint.resolveEndpointConfig(_config_5);
        const _config_7 = eventstreamSerdeConfigResolver.resolveEventStreamSerdeConfig(_config_6);
        const _config_8 = httpAuthSchemeProvider.resolveHttpAuthSchemeConfig(_config_7);
        const _config_9 = resolveRuntimeExtensions(_config_8, configuration?.extensions || []);
        this.config = _config_9;
        this.middlewareStack.use(schema.getSchemaSerdePlugin(this.config));
        this.middlewareStack.use(middlewareUserAgent.getUserAgentPlugin(this.config));
        this.middlewareStack.use(middlewareRetry.getRetryPlugin(this.config));
        this.middlewareStack.use(middlewareContentLength.getContentLengthPlugin(this.config));
        this.middlewareStack.use(middlewareHostHeader.getHostHeaderPlugin(this.config));
        this.middlewareStack.use(middlewareLogger.getLoggerPlugin(this.config));
        this.middlewareStack.use(middlewareRecursionDetection.getRecursionDetectionPlugin(this.config));
        this.middlewareStack.use(core.getHttpAuthSchemeEndpointRuleSetPlugin(this.config, {
            httpAuthSchemeParametersProvider: httpAuthSchemeProvider.defaultKinesisHttpAuthSchemeParametersProvider,
            identityProviderConfigProvider: async (config) => new core.DefaultIdentityProviderConfig({
                "aws.auth#sigv4": config.credentials,
            }),
        }));
        this.middlewareStack.use(core.getHttpSigningPlugin(this.config));
    }
    destroy() {
        super.destroy();
    }
}

class KinesisServiceException extends smithyClient.ServiceException {
    constructor(options) {
        super(options);
        Object.setPrototypeOf(this, KinesisServiceException.prototype);
    }
}

class AccessDeniedException extends KinesisServiceException {
    name = "AccessDeniedException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "AccessDeniedException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, AccessDeniedException.prototype);
    }
}
class InvalidArgumentException extends KinesisServiceException {
    name = "InvalidArgumentException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "InvalidArgumentException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidArgumentException.prototype);
    }
}
class LimitExceededException extends KinesisServiceException {
    name = "LimitExceededException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "LimitExceededException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, LimitExceededException.prototype);
    }
}
class ResourceInUseException extends KinesisServiceException {
    name = "ResourceInUseException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ResourceInUseException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ResourceInUseException.prototype);
    }
}
class ResourceNotFoundException extends KinesisServiceException {
    name = "ResourceNotFoundException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ResourceNotFoundException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ResourceNotFoundException.prototype);
    }
}
class ValidationException extends KinesisServiceException {
    name = "ValidationException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ValidationException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ValidationException.prototype);
    }
}
class ExpiredIteratorException extends KinesisServiceException {
    name = "ExpiredIteratorException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ExpiredIteratorException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ExpiredIteratorException.prototype);
    }
}
class ExpiredNextTokenException extends KinesisServiceException {
    name = "ExpiredNextTokenException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ExpiredNextTokenException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ExpiredNextTokenException.prototype);
    }
}
class InternalFailureException extends KinesisServiceException {
    name = "InternalFailureException";
    $fault = "server";
    constructor(opts) {
        super({
            name: "InternalFailureException",
            $fault: "server",
            ...opts,
        });
        Object.setPrototypeOf(this, InternalFailureException.prototype);
    }
}
class KMSAccessDeniedException extends KinesisServiceException {
    name = "KMSAccessDeniedException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "KMSAccessDeniedException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, KMSAccessDeniedException.prototype);
    }
}
class KMSDisabledException extends KinesisServiceException {
    name = "KMSDisabledException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "KMSDisabledException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, KMSDisabledException.prototype);
    }
}
class KMSInvalidStateException extends KinesisServiceException {
    name = "KMSInvalidStateException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "KMSInvalidStateException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, KMSInvalidStateException.prototype);
    }
}
class KMSNotFoundException extends KinesisServiceException {
    name = "KMSNotFoundException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "KMSNotFoundException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, KMSNotFoundException.prototype);
    }
}
class KMSOptInRequired extends KinesisServiceException {
    name = "KMSOptInRequired";
    $fault = "client";
    constructor(opts) {
        super({
            name: "KMSOptInRequired",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, KMSOptInRequired.prototype);
    }
}
class KMSThrottlingException extends KinesisServiceException {
    name = "KMSThrottlingException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "KMSThrottlingException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, KMSThrottlingException.prototype);
    }
}
class ProvisionedThroughputExceededException extends KinesisServiceException {
    name = "ProvisionedThroughputExceededException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ProvisionedThroughputExceededException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ProvisionedThroughputExceededException.prototype);
    }
}

const _AAT = "ApproximateArrivalTimestamp";
const _ADE = "AccessDeniedException";
const _APSI = "AdjacentParentShardId";
const _ASTM = "AdjacentShardToMerge";
const _ATTS = "AddTagsToStream";
const _ATTSI = "AddTagsToStreamInput";
const _C = "Consumer";
const _CARN = "ConsumerARN";
const _CC = "ConsumerCount";
const _CCT = "ConsumerCreationTimestamp";
const _CD = "ConsumerDescription";
const _CL = "ConsumerList";
const _CMB = "CurrentMiBps";
const _CN = "ConsumerName";
const _CS = "ChildShard";
const _CSC = "CurrentShardCount";
const _CSI = "CreateStreamInput";
const _CSL = "ChildShardList";
const _CSLM = "CurrentShardLevelMetrics";
const _CSN = "ContinuationSequenceNumber";
const _CSh = "ChildShards";
const _CSo = "ConsumerStatus";
const _CSr = "CreateStream";
const _Co = "Consumers";
const _D = "Data";
const _DAS = "DescribeAccountSettings";
const _DASI = "DescribeAccountSettingsInput";
const _DASO = "DescribeAccountSettingsOutput";
const _DEM = "DisableEnhancedMonitoring";
const _DEMI = "DisableEnhancedMonitoringInput";
const _DL = "DescribeLimits";
const _DLI = "DescribeLimitsInput";
const _DLO = "DescribeLimitsOutput";
const _DRP = "DeleteResourcePolicy";
const _DRPI = "DeleteResourcePolicyInput";
const _DS = "DeleteStream";
const _DSC = "DeregisterStreamConsumer";
const _DSCI = "DeregisterStreamConsumerInput";
const _DSCIe = "DescribeStreamConsumerInput";
const _DSCO = "DescribeStreamConsumerOutput";
const _DSCe = "DescribeStreamConsumer";
const _DSI = "DeleteStreamInput";
const _DSIe = "DescribeStreamInput";
const _DSLM = "DesiredShardLevelMetrics";
const _DSO = "DescribeStreamOutput";
const _DSRP = "DecreaseStreamRetentionPeriod";
const _DSRPI = "DecreaseStreamRetentionPeriodInput";
const _DSS = "DescribeStreamSummary";
const _DSSI = "DescribeStreamSummaryInput";
const _DSSO = "DescribeStreamSummaryOutput";
const _DSe = "DescribeStream";
const _EA = "EndedAt";
const _EAEA = "EarliestAllowedEndAt";
const _EC = "ErrorCode";
const _ECD = "EnforceConsumerDeletion";
const _EEM = "EnableEnhancedMonitoring";
const _EEMI = "EnableEnhancedMonitoringInput";
const _EHK = "EndingHashKey";
const _EHKx = "ExplicitHashKey";
const _EIE = "ExpiredIteratorException";
const _EM = "EnhancedMetrics";
const _EML = "EnhancedMonitoringList";
const _EMO = "EnhancedMonitoringOutput";
const _EMn = "EnhancedMonitoring";
const _EMr = "ErrorMessage";
const _ENTE = "ExpiredNextTokenException";
const _ES = "EventStream";
const _ESN = "EndingSequenceNumber";
const _ESSI = "ExclusiveStartShardId";
const _ESSN = "ExclusiveStartStreamName";
const _ESTK = "ExclusiveStartTagKey";
const _ET = "EncryptionType";
const _FRC = "FailedRecordCount";
const _GR = "GetRecords";
const _GRI = "GetRecordsInput";
const _GRO = "GetRecordsOutput";
const _GRP = "GetResourcePolicy";
const _GRPI = "GetResourcePolicyInput";
const _GRPO = "GetResourcePolicyOutput";
const _GSI = "GetShardIterator";
const _GSII = "GetShardIteratorInput";
const _GSIO = "GetShardIteratorOutput";
const _HKR = "HashKeyRange";
const _HMS = "HasMoreStreams";
const _HMSa = "HasMoreShards";
const _HMT = "HasMoreTags";
const _IAE = "InvalidArgumentException";
const _IFE = "InternalFailureException";
const _ISRP = "IncreaseStreamRetentionPeriod";
const _ISRPI = "IncreaseStreamRetentionPeriodInput";
const _K = "Key";
const _KI = "KeyId";
const _KMSADE = "KMSAccessDeniedException";
const _KMSDE = "KMSDisabledException";
const _KMSISE = "KMSInvalidStateException";
const _KMSNFE = "KMSNotFoundException";
const _KMSOIR = "KMSOptInRequired";
const _KMSTE = "KMSThrottlingException";
const _L = "Limit";
const _LEE = "LimitExceededException";
const _LS = "ListShards";
const _LSC = "ListStreamConsumers";
const _LSCI = "ListStreamConsumersInput";
const _LSCO = "ListStreamConsumersOutput";
const _LSI = "ListShardsInput";
const _LSIi = "ListStreamsInput";
const _LSO = "ListShardsOutput";
const _LSOi = "ListStreamsOutput";
const _LSi = "ListStreams";
const _LTFR = "ListTagsForResource";
const _LTFRI = "ListTagsForResourceInput";
const _LTFRO = "ListTagsForResourceOutput";
const _LTFS = "ListTagsForStream";
const _LTFSI = "ListTagsForStreamInput";
const _LTFSO = "ListTagsForStreamOutput";
const _MBL = "MillisBehindLatest";
const _MR = "MaxResults";
const _MRSIKB = "MaxRecordSizeInKiB";
const _MS = "MergeShards";
const _MSI = "MergeShardsInput";
const _MTBC = "MinimumThroughputBillingCommitment";
const _MTBCI = "MinimumThroughputBillingCommitmentInput";
const _MTBCO = "MinimumThroughputBillingCommitmentOutput";
const _NSHK = "NewStartingHashKey";
const _NSI = "NextShardIterator";
const _NT = "NextToken";
const _ODSC = "OnDemandStreamCount";
const _ODSCL = "OnDemandStreamCountLimit";
const _OSC = "OpenShardCount";
const _P = "Policy";
const _PK = "PartitionKey";
const _PR = "PutRecord";
const _PRI = "PutRecordInput";
const _PRIu = "PutRecordsInput";
const _PRO = "PutRecordOutput";
const _PROu = "PutRecordsOutput";
const _PRP = "PutResourcePolicy";
const _PRPI = "PutResourcePolicyInput";
const _PRRE = "PutRecordsRequestEntry";
const _PRREL = "PutRecordsRequestEntryList";
const _PRRELu = "PutRecordsResultEntryList";
const _PRREu = "PutRecordsResultEntry";
const _PRu = "PutRecords";
const _PS = "ParentShards";
const _PSI = "ParentShardId";
const _PTEE = "ProvisionedThroughputExceededException";
const _R = "Records";
const _RARN = "ResourceARN";
const _RIUE = "ResourceInUseException";
const _RL = "RecordList";
const _RNFE = "ResourceNotFoundException";
const _RPH = "RetentionPeriodHours";
const _RSC = "RegisterStreamConsumer";
const _RSCI = "RegisterStreamConsumerInput";
const _RSCO = "RegisterStreamConsumerOutput";
const _RTFS = "RemoveTagsFromStream";
const _RTFSI = "RemoveTagsFromStreamInput";
const _Re = "Record";
const _S = "Shards";
const _SA = "StartedAt";
const _SARN = "StreamARN";
const _SC = "ShardCount";
const _SCT = "StreamCreationTimestamp";
const _SD = "StreamDescription";
const _SDS = "StreamDescriptionSummary";
const _SF = "ShardFilter";
const _SHK = "StartingHashKey";
const _SI = "ShardId";
const _SIT = "ShardIteratorType";
const _SIh = "ShardIterator";
const _SL = "ShardLimit";
const _SLM = "ShardLevelMetrics";
const _SLh = "ShardList";
const _SM = "StreamMode";
const _SMD = "StreamModeDetails";
const _SN = "StreamName";
const _SNFO = "SequenceNumberForOrdering";
const _SNR = "SequenceNumberRange";
const _SNe = "SequenceNumber";
const _SNt = "StreamNames";
const _SP = "StartingPosition";
const _SS = "StreamSummaries";
const _SSE = "StartStreamEncryption";
const _SSEI = "StartStreamEncryptionInput";
const _SSEIt = "StopStreamEncryptionInput";
const _SSEt = "StopStreamEncryption";
const _SSI = "SplitShardInput";
const _SSL = "StreamSummaryList";
const _SSN = "StartingSequenceNumber";
const _SSp = "SplitShard";
const _SSt = "StreamStatus";
const _SStr = "StreamSummary";
const _ST = "ScalingType";
const _STM = "ShardToMerge";
const _STS = "ShardToSplit";
const _STSE = "SubscribeToShardEvent";
const _STSES = "SubscribeToShardEventStream";
const _STSI = "SubscribeToShardInput";
const _STSO = "SubscribeToShardOutput";
const _STSu = "SubscribeToShard";
const _Sh = "Shard";
const _St = "Status";
const _T = "Tags";
const _TK = "TagKeys";
const _TL = "TagList";
const _TMB = "TargetMiBps";
const _TR = "TagResource";
const _TRI = "TagResourceInput";
const _TSC = "TargetShardCount";
const _Ta = "Tag";
const _Ti = "Timestamp";
const _Ty = "Type";
const _UAS = "UpdateAccountSettings";
const _UASI = "UpdateAccountSettingsInput";
const _UASO = "UpdateAccountSettingsOutput";
const _UMRS = "UpdateMaxRecordSize";
const _UMRSI = "UpdateMaxRecordSizeInput";
const _UR = "UntagResource";
const _URI = "UntagResourceInput";
const _USC = "UpdateShardCount";
const _USCI = "UpdateShardCountInput";
const _USCO = "UpdateShardCountOutput";
const _USM = "UpdateStreamMode";
const _USMI = "UpdateStreamModeInput";
const _USWT = "UpdateStreamWarmThroughput";
const _USWTI = "UpdateStreamWarmThroughputInput";
const _USWTO = "UpdateStreamWarmThroughputOutput";
const _V = "Value";
const _VE = "ValidationException";
const _WT = "WarmThroughput";
const _WTMB = "WarmThroughputMiBps";
const _WTO = "WarmThroughputObject";
const _c = "client";
const _e = "error";
const _m = "message";
const _s = "server";
const _sm = "smithy.ts.sdk.synthetic.com.amazonaws.kinesis";
const _st = "streaming";
const n0 = "com.amazonaws.kinesis";
var AccessDeniedException$ = [-3, n0, _ADE,
    { [_e]: _c },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(AccessDeniedException$, AccessDeniedException);
var AddTagsToStreamInput$ = [3, n0, _ATTSI,
    0,
    [_SN, _T, _SARN],
    [0, 128 | 0, 0]
];
var ChildShard$ = [3, n0, _CS,
    0,
    [_SI, _PS, _HKR],
    [0, 64 | 0, () => HashKeyRange$]
];
var Consumer$ = [3, n0, _C,
    0,
    [_CN, _CARN, _CSo, _CCT],
    [0, 0, 0, 4]
];
var ConsumerDescription$ = [3, n0, _CD,
    0,
    [_CN, _CARN, _CSo, _CCT, _SARN],
    [0, 0, 0, 4, 0]
];
var CreateStreamInput$ = [3, n0, _CSI,
    0,
    [_SN, _SC, _SMD, _T, _WTMB, _MRSIKB],
    [0, 1, () => StreamModeDetails$, 128 | 0, 1, 1]
];
var DecreaseStreamRetentionPeriodInput$ = [3, n0, _DSRPI,
    0,
    [_SN, _RPH, _SARN],
    [0, 1, 0]
];
var DeleteResourcePolicyInput$ = [3, n0, _DRPI,
    0,
    [_RARN],
    [0]
];
var DeleteStreamInput$ = [3, n0, _DSI,
    0,
    [_SN, _ECD, _SARN],
    [0, 2, 0]
];
var DeregisterStreamConsumerInput$ = [3, n0, _DSCI,
    0,
    [_SARN, _CN, _CARN],
    [0, 0, 0]
];
var DescribeAccountSettingsInput$ = [3, n0, _DASI,
    0,
    [],
    []
];
var DescribeAccountSettingsOutput$ = [3, n0, _DASO,
    0,
    [_MTBC],
    [() => MinimumThroughputBillingCommitmentOutput$]
];
var DescribeLimitsInput$ = [3, n0, _DLI,
    0,
    [],
    []
];
var DescribeLimitsOutput$ = [3, n0, _DLO,
    0,
    [_SL, _OSC, _ODSC, _ODSCL],
    [1, 1, 1, 1]
];
var DescribeStreamConsumerInput$ = [3, n0, _DSCIe,
    0,
    [_SARN, _CN, _CARN],
    [0, 0, 0]
];
var DescribeStreamConsumerOutput$ = [3, n0, _DSCO,
    0,
    [_CD],
    [() => ConsumerDescription$]
];
var DescribeStreamInput$ = [3, n0, _DSIe,
    0,
    [_SN, _L, _ESSI, _SARN],
    [0, 1, 0, 0]
];
var DescribeStreamOutput$ = [3, n0, _DSO,
    0,
    [_SD],
    [() => StreamDescription$]
];
var DescribeStreamSummaryInput$ = [3, n0, _DSSI,
    0,
    [_SN, _SARN],
    [0, 0]
];
var DescribeStreamSummaryOutput$ = [3, n0, _DSSO,
    0,
    [_SDS],
    [() => StreamDescriptionSummary$]
];
var DisableEnhancedMonitoringInput$ = [3, n0, _DEMI,
    0,
    [_SN, _SLM, _SARN],
    [0, 64 | 0, 0]
];
var EnableEnhancedMonitoringInput$ = [3, n0, _EEMI,
    0,
    [_SN, _SLM, _SARN],
    [0, 64 | 0, 0]
];
var EnhancedMetrics$ = [3, n0, _EM,
    0,
    [_SLM],
    [64 | 0]
];
var EnhancedMonitoringOutput$ = [3, n0, _EMO,
    0,
    [_SN, _CSLM, _DSLM, _SARN],
    [0, 64 | 0, 64 | 0, 0]
];
var ExpiredIteratorException$ = [-3, n0, _EIE,
    { [_e]: _c },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ExpiredIteratorException$, ExpiredIteratorException);
var ExpiredNextTokenException$ = [-3, n0, _ENTE,
    { [_e]: _c },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ExpiredNextTokenException$, ExpiredNextTokenException);
var GetRecordsInput$ = [3, n0, _GRI,
    0,
    [_SIh, _L, _SARN],
    [0, 1, 0]
];
var GetRecordsOutput$ = [3, n0, _GRO,
    0,
    [_R, _NSI, _MBL, _CSh],
    [() => RecordList, 0, 1, () => ChildShardList]
];
var GetResourcePolicyInput$ = [3, n0, _GRPI,
    0,
    [_RARN],
    [0]
];
var GetResourcePolicyOutput$ = [3, n0, _GRPO,
    0,
    [_P],
    [0]
];
var GetShardIteratorInput$ = [3, n0, _GSII,
    0,
    [_SN, _SI, _SIT, _SSN, _Ti, _SARN],
    [0, 0, 0, 0, 4, 0]
];
var GetShardIteratorOutput$ = [3, n0, _GSIO,
    0,
    [_SIh],
    [0]
];
var HashKeyRange$ = [3, n0, _HKR,
    0,
    [_SHK, _EHK],
    [0, 0]
];
var IncreaseStreamRetentionPeriodInput$ = [3, n0, _ISRPI,
    0,
    [_SN, _RPH, _SARN],
    [0, 1, 0]
];
var InternalFailureException$ = [-3, n0, _IFE,
    { [_e]: _s },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InternalFailureException$, InternalFailureException);
var InvalidArgumentException$ = [-3, n0, _IAE,
    { [_e]: _c },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidArgumentException$, InvalidArgumentException);
var KMSAccessDeniedException$ = [-3, n0, _KMSADE,
    { [_e]: _c },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(KMSAccessDeniedException$, KMSAccessDeniedException);
var KMSDisabledException$ = [-3, n0, _KMSDE,
    { [_e]: _c },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(KMSDisabledException$, KMSDisabledException);
var KMSInvalidStateException$ = [-3, n0, _KMSISE,
    { [_e]: _c },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(KMSInvalidStateException$, KMSInvalidStateException);
var KMSNotFoundException$ = [-3, n0, _KMSNFE,
    { [_e]: _c },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(KMSNotFoundException$, KMSNotFoundException);
var KMSOptInRequired$ = [-3, n0, _KMSOIR,
    { [_e]: _c },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(KMSOptInRequired$, KMSOptInRequired);
var KMSThrottlingException$ = [-3, n0, _KMSTE,
    { [_e]: _c },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(KMSThrottlingException$, KMSThrottlingException);
var LimitExceededException$ = [-3, n0, _LEE,
    { [_e]: _c },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(LimitExceededException$, LimitExceededException);
var ListShardsInput$ = [3, n0, _LSI,
    0,
    [_SN, _NT, _ESSI, _MR, _SCT, _SF, _SARN],
    [0, 0, 0, 1, 4, () => ShardFilter$, 0]
];
var ListShardsOutput$ = [3, n0, _LSO,
    0,
    [_S, _NT],
    [() => ShardList, 0]
];
var ListStreamConsumersInput$ = [3, n0, _LSCI,
    0,
    [_SARN, _NT, _MR, _SCT],
    [0, 0, 1, 4]
];
var ListStreamConsumersOutput$ = [3, n0, _LSCO,
    0,
    [_Co, _NT],
    [() => ConsumerList, 0]
];
var ListStreamsInput$ = [3, n0, _LSIi,
    0,
    [_L, _ESSN, _NT],
    [1, 0, 0]
];
var ListStreamsOutput$ = [3, n0, _LSOi,
    0,
    [_SNt, _HMS, _NT, _SS],
    [64 | 0, 2, 0, () => StreamSummaryList]
];
var ListTagsForResourceInput$ = [3, n0, _LTFRI,
    0,
    [_RARN],
    [0]
];
var ListTagsForResourceOutput$ = [3, n0, _LTFRO,
    0,
    [_T],
    [() => TagList]
];
var ListTagsForStreamInput$ = [3, n0, _LTFSI,
    0,
    [_SN, _ESTK, _L, _SARN],
    [0, 0, 1, 0]
];
var ListTagsForStreamOutput$ = [3, n0, _LTFSO,
    0,
    [_T, _HMT],
    [() => TagList, 2]
];
var MergeShardsInput$ = [3, n0, _MSI,
    0,
    [_SN, _STM, _ASTM, _SARN],
    [0, 0, 0, 0]
];
var MinimumThroughputBillingCommitmentInput$ = [3, n0, _MTBCI,
    0,
    [_St],
    [0]
];
var MinimumThroughputBillingCommitmentOutput$ = [3, n0, _MTBCO,
    0,
    [_St, _SA, _EA, _EAEA],
    [0, 4, 4, 4]
];
var ProvisionedThroughputExceededException$ = [-3, n0, _PTEE,
    { [_e]: _c },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ProvisionedThroughputExceededException$, ProvisionedThroughputExceededException);
var PutRecordInput$ = [3, n0, _PRI,
    0,
    [_SN, _D, _PK, _EHKx, _SNFO, _SARN],
    [0, 21, 0, 0, 0, 0]
];
var PutRecordOutput$ = [3, n0, _PRO,
    0,
    [_SI, _SNe, _ET],
    [0, 0, 0]
];
var PutRecordsInput$ = [3, n0, _PRIu,
    0,
    [_R, _SN, _SARN],
    [() => PutRecordsRequestEntryList, 0, 0]
];
var PutRecordsOutput$ = [3, n0, _PROu,
    0,
    [_FRC, _R, _ET],
    [1, () => PutRecordsResultEntryList, 0]
];
var PutRecordsRequestEntry$ = [3, n0, _PRRE,
    0,
    [_D, _EHKx, _PK],
    [21, 0, 0]
];
var PutRecordsResultEntry$ = [3, n0, _PRREu,
    0,
    [_SNe, _SI, _EC, _EMr],
    [0, 0, 0, 0]
];
var PutResourcePolicyInput$ = [3, n0, _PRPI,
    0,
    [_RARN, _P],
    [0, 0]
];
var _Record$ = [3, n0, _Re,
    0,
    [_SNe, _AAT, _D, _PK, _ET],
    [0, 4, 21, 0, 0]
];
var RegisterStreamConsumerInput$ = [3, n0, _RSCI,
    0,
    [_SARN, _CN, _T],
    [0, 0, 128 | 0]
];
var RegisterStreamConsumerOutput$ = [3, n0, _RSCO,
    0,
    [_C],
    [() => Consumer$]
];
var RemoveTagsFromStreamInput$ = [3, n0, _RTFSI,
    0,
    [_SN, _TK, _SARN],
    [0, 64 | 0, 0]
];
var ResourceInUseException$ = [-3, n0, _RIUE,
    { [_e]: _c },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ResourceInUseException$, ResourceInUseException);
var ResourceNotFoundException$ = [-3, n0, _RNFE,
    { [_e]: _c },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ResourceNotFoundException$, ResourceNotFoundException);
var SequenceNumberRange$ = [3, n0, _SNR,
    0,
    [_SSN, _ESN],
    [0, 0]
];
var Shard$ = [3, n0, _Sh,
    0,
    [_SI, _PSI, _APSI, _HKR, _SNR],
    [0, 0, 0, () => HashKeyRange$, () => SequenceNumberRange$]
];
var ShardFilter$ = [3, n0, _SF,
    0,
    [_Ty, _SI, _Ti],
    [0, 0, 4]
];
var SplitShardInput$ = [3, n0, _SSI,
    0,
    [_SN, _STS, _NSHK, _SARN],
    [0, 0, 0, 0]
];
var StartingPosition$ = [3, n0, _SP,
    0,
    [_Ty, _SNe, _Ti],
    [0, 0, 4]
];
var StartStreamEncryptionInput$ = [3, n0, _SSEI,
    0,
    [_SN, _ET, _KI, _SARN],
    [0, 0, 0, 0]
];
var StopStreamEncryptionInput$ = [3, n0, _SSEIt,
    0,
    [_SN, _ET, _KI, _SARN],
    [0, 0, 0, 0]
];
var StreamDescription$ = [3, n0, _SD,
    0,
    [_SN, _SARN, _SSt, _SMD, _S, _HMSa, _RPH, _SCT, _EMn, _ET, _KI],
    [0, 0, 0, () => StreamModeDetails$, () => ShardList, 2, 1, 4, () => EnhancedMonitoringList, 0, 0]
];
var StreamDescriptionSummary$ = [3, n0, _SDS,
    0,
    [_SN, _SARN, _SSt, _SMD, _RPH, _SCT, _EMn, _ET, _KI, _OSC, _CC, _WT, _MRSIKB],
    [0, 0, 0, () => StreamModeDetails$, 1, 4, () => EnhancedMonitoringList, 0, 0, 1, 1, () => WarmThroughputObject$, 1]
];
var StreamModeDetails$ = [3, n0, _SMD,
    0,
    [_SM],
    [0]
];
var StreamSummary$ = [3, n0, _SStr,
    0,
    [_SN, _SARN, _SSt, _SMD, _SCT],
    [0, 0, 0, () => StreamModeDetails$, 4]
];
var SubscribeToShardEvent$ = [3, n0, _STSE,
    0,
    [_R, _CSN, _MBL, _CSh],
    [() => RecordList, 0, 1, () => ChildShardList]
];
var SubscribeToShardInput$ = [3, n0, _STSI,
    0,
    [_CARN, _SI, _SP],
    [0, 0, () => StartingPosition$]
];
var SubscribeToShardOutput$ = [3, n0, _STSO,
    0,
    [_ES],
    [[() => SubscribeToShardEventStream$, 0]]
];
var Tag$ = [3, n0, _Ta,
    0,
    [_K, _V],
    [0, 0]
];
var TagResourceInput$ = [3, n0, _TRI,
    0,
    [_T, _RARN],
    [128 | 0, 0]
];
var UntagResourceInput$ = [3, n0, _URI,
    0,
    [_TK, _RARN],
    [64 | 0, 0]
];
var UpdateAccountSettingsInput$ = [3, n0, _UASI,
    0,
    [_MTBC],
    [() => MinimumThroughputBillingCommitmentInput$]
];
var UpdateAccountSettingsOutput$ = [3, n0, _UASO,
    0,
    [_MTBC],
    [() => MinimumThroughputBillingCommitmentOutput$]
];
var UpdateMaxRecordSizeInput$ = [3, n0, _UMRSI,
    0,
    [_SARN, _MRSIKB],
    [0, 1]
];
var UpdateShardCountInput$ = [3, n0, _USCI,
    0,
    [_SN, _TSC, _ST, _SARN],
    [0, 1, 0, 0]
];
var UpdateShardCountOutput$ = [3, n0, _USCO,
    0,
    [_SN, _CSC, _TSC, _SARN],
    [0, 1, 1, 0]
];
var UpdateStreamModeInput$ = [3, n0, _USMI,
    0,
    [_SARN, _SMD, _WTMB],
    [0, () => StreamModeDetails$, 1]
];
var UpdateStreamWarmThroughputInput$ = [3, n0, _USWTI,
    0,
    [_SARN, _SN, _WTMB],
    [0, 0, 1]
];
var UpdateStreamWarmThroughputOutput$ = [3, n0, _USWTO,
    0,
    [_SARN, _SN, _WT],
    [0, 0, () => WarmThroughputObject$]
];
var ValidationException$ = [-3, n0, _VE,
    { [_e]: _c },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ValidationException$, ValidationException);
var WarmThroughputObject$ = [3, n0, _WTO,
    0,
    [_TMB, _CMB],
    [1, 1]
];
var __Unit = "unit";
var KinesisServiceException$ = [-3, _sm, "KinesisServiceException", 0, [], []];
schema.TypeRegistry.for(_sm).registerError(KinesisServiceException$, KinesisServiceException);
var ChildShardList = [1, n0, _CSL,
    0, () => ChildShard$
];
var ConsumerList = [1, n0, _CL,
    0, () => Consumer$
];
var EnhancedMonitoringList = [1, n0, _EML,
    0, () => EnhancedMetrics$
];
var PutRecordsRequestEntryList = [1, n0, _PRREL,
    0, () => PutRecordsRequestEntry$
];
var PutRecordsResultEntryList = [1, n0, _PRRELu,
    0, () => PutRecordsResultEntry$
];
var RecordList = [1, n0, _RL,
    0, () => _Record$
];
var ShardList = [1, n0, _SLh,
    0, () => Shard$
];
var StreamSummaryList = [1, n0, _SSL,
    0, () => StreamSummary$
];
var TagList = [1, n0, _TL,
    0, () => Tag$
];
var SubscribeToShardEventStream$ = [4, n0, _STSES,
    { [_st]: 1 },
    [_STSE, _RNFE, _RIUE, _KMSDE, _KMSISE, _KMSADE, _KMSNFE, _KMSOIR, _KMSTE, _IFE],
    [() => SubscribeToShardEvent$, [() => ResourceNotFoundException$, 0], [() => ResourceInUseException$, 0], [() => KMSDisabledException$, 0], [() => KMSInvalidStateException$, 0], [() => KMSAccessDeniedException$, 0], [() => KMSNotFoundException$, 0], [() => KMSOptInRequired$, 0], [() => KMSThrottlingException$, 0], [() => InternalFailureException$, 0]]
];
var AddTagsToStream$ = [9, n0, _ATTS,
    0, () => AddTagsToStreamInput$, () => __Unit
];
var CreateStream$ = [9, n0, _CSr,
    0, () => CreateStreamInput$, () => __Unit
];
var DecreaseStreamRetentionPeriod$ = [9, n0, _DSRP,
    0, () => DecreaseStreamRetentionPeriodInput$, () => __Unit
];
var DeleteResourcePolicy$ = [9, n0, _DRP,
    0, () => DeleteResourcePolicyInput$, () => __Unit
];
var DeleteStream$ = [9, n0, _DS,
    0, () => DeleteStreamInput$, () => __Unit
];
var DeregisterStreamConsumer$ = [9, n0, _DSC,
    0, () => DeregisterStreamConsumerInput$, () => __Unit
];
var DescribeAccountSettings$ = [9, n0, _DAS,
    0, () => DescribeAccountSettingsInput$, () => DescribeAccountSettingsOutput$
];
var DescribeLimits$ = [9, n0, _DL,
    0, () => DescribeLimitsInput$, () => DescribeLimitsOutput$
];
var DescribeStream$ = [9, n0, _DSe,
    0, () => DescribeStreamInput$, () => DescribeStreamOutput$
];
var DescribeStreamConsumer$ = [9, n0, _DSCe,
    0, () => DescribeStreamConsumerInput$, () => DescribeStreamConsumerOutput$
];
var DescribeStreamSummary$ = [9, n0, _DSS,
    0, () => DescribeStreamSummaryInput$, () => DescribeStreamSummaryOutput$
];
var DisableEnhancedMonitoring$ = [9, n0, _DEM,
    0, () => DisableEnhancedMonitoringInput$, () => EnhancedMonitoringOutput$
];
var EnableEnhancedMonitoring$ = [9, n0, _EEM,
    0, () => EnableEnhancedMonitoringInput$, () => EnhancedMonitoringOutput$
];
var GetRecords$ = [9, n0, _GR,
    0, () => GetRecordsInput$, () => GetRecordsOutput$
];
var GetResourcePolicy$ = [9, n0, _GRP,
    0, () => GetResourcePolicyInput$, () => GetResourcePolicyOutput$
];
var GetShardIterator$ = [9, n0, _GSI,
    0, () => GetShardIteratorInput$, () => GetShardIteratorOutput$
];
var IncreaseStreamRetentionPeriod$ = [9, n0, _ISRP,
    0, () => IncreaseStreamRetentionPeriodInput$, () => __Unit
];
var ListShards$ = [9, n0, _LS,
    0, () => ListShardsInput$, () => ListShardsOutput$
];
var ListStreamConsumers$ = [9, n0, _LSC,
    0, () => ListStreamConsumersInput$, () => ListStreamConsumersOutput$
];
var ListStreams$ = [9, n0, _LSi,
    0, () => ListStreamsInput$, () => ListStreamsOutput$
];
var ListTagsForResource$ = [9, n0, _LTFR,
    0, () => ListTagsForResourceInput$, () => ListTagsForResourceOutput$
];
var ListTagsForStream$ = [9, n0, _LTFS,
    0, () => ListTagsForStreamInput$, () => ListTagsForStreamOutput$
];
var MergeShards$ = [9, n0, _MS,
    0, () => MergeShardsInput$, () => __Unit
];
var PutRecord$ = [9, n0, _PR,
    0, () => PutRecordInput$, () => PutRecordOutput$
];
var PutRecords$ = [9, n0, _PRu,
    0, () => PutRecordsInput$, () => PutRecordsOutput$
];
var PutResourcePolicy$ = [9, n0, _PRP,
    0, () => PutResourcePolicyInput$, () => __Unit
];
var RegisterStreamConsumer$ = [9, n0, _RSC,
    0, () => RegisterStreamConsumerInput$, () => RegisterStreamConsumerOutput$
];
var RemoveTagsFromStream$ = [9, n0, _RTFS,
    0, () => RemoveTagsFromStreamInput$, () => __Unit
];
var SplitShard$ = [9, n0, _SSp,
    0, () => SplitShardInput$, () => __Unit
];
var StartStreamEncryption$ = [9, n0, _SSE,
    0, () => StartStreamEncryptionInput$, () => __Unit
];
var StopStreamEncryption$ = [9, n0, _SSEt,
    0, () => StopStreamEncryptionInput$, () => __Unit
];
var SubscribeToShard$ = [9, n0, _STSu,
    0, () => SubscribeToShardInput$, () => SubscribeToShardOutput$
];
var TagResource$ = [9, n0, _TR,
    0, () => TagResourceInput$, () => __Unit
];
var UntagResource$ = [9, n0, _UR,
    0, () => UntagResourceInput$, () => __Unit
];
var UpdateAccountSettings$ = [9, n0, _UAS,
    0, () => UpdateAccountSettingsInput$, () => UpdateAccountSettingsOutput$
];
var UpdateMaxRecordSize$ = [9, n0, _UMRS,
    0, () => UpdateMaxRecordSizeInput$, () => __Unit
];
var UpdateShardCount$ = [9, n0, _USC,
    0, () => UpdateShardCountInput$, () => UpdateShardCountOutput$
];
var UpdateStreamMode$ = [9, n0, _USM,
    0, () => UpdateStreamModeInput$, () => __Unit
];
var UpdateStreamWarmThroughput$ = [9, n0, _USWT,
    0, () => UpdateStreamWarmThroughputInput$, () => UpdateStreamWarmThroughputOutput$
];

class AddTagsToStreamCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "AddTagsToStream", {})
    .n("KinesisClient", "AddTagsToStreamCommand")
    .sc(AddTagsToStream$)
    .build() {
}

class CreateStreamCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "CreateStream", {})
    .n("KinesisClient", "CreateStreamCommand")
    .sc(CreateStream$)
    .build() {
}

class DecreaseStreamRetentionPeriodCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "DecreaseStreamRetentionPeriod", {})
    .n("KinesisClient", "DecreaseStreamRetentionPeriodCommand")
    .sc(DecreaseStreamRetentionPeriod$)
    .build() {
}

class DeleteResourcePolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    ResourceARN: { type: "contextParams", name: "ResourceARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "DeleteResourcePolicy", {})
    .n("KinesisClient", "DeleteResourcePolicyCommand")
    .sc(DeleteResourcePolicy$)
    .build() {
}

class DeleteStreamCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "DeleteStream", {})
    .n("KinesisClient", "DeleteStreamCommand")
    .sc(DeleteStream$)
    .build() {
}

class DeregisterStreamConsumerCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    ConsumerARN: { type: "contextParams", name: "ConsumerARN" },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "DeregisterStreamConsumer", {})
    .n("KinesisClient", "DeregisterStreamConsumerCommand")
    .sc(DeregisterStreamConsumer$)
    .build() {
}

class DescribeAccountSettingsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "DescribeAccountSettings", {})
    .n("KinesisClient", "DescribeAccountSettingsCommand")
    .sc(DescribeAccountSettings$)
    .build() {
}

class DescribeLimitsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "DescribeLimits", {})
    .n("KinesisClient", "DescribeLimitsCommand")
    .sc(DescribeLimits$)
    .build() {
}

class DescribeStreamCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "DescribeStream", {})
    .n("KinesisClient", "DescribeStreamCommand")
    .sc(DescribeStream$)
    .build() {
}

class DescribeStreamConsumerCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    ConsumerARN: { type: "contextParams", name: "ConsumerARN" },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "DescribeStreamConsumer", {})
    .n("KinesisClient", "DescribeStreamConsumerCommand")
    .sc(DescribeStreamConsumer$)
    .build() {
}

class DescribeStreamSummaryCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "DescribeStreamSummary", {})
    .n("KinesisClient", "DescribeStreamSummaryCommand")
    .sc(DescribeStreamSummary$)
    .build() {
}

class DisableEnhancedMonitoringCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "DisableEnhancedMonitoring", {})
    .n("KinesisClient", "DisableEnhancedMonitoringCommand")
    .sc(DisableEnhancedMonitoring$)
    .build() {
}

class EnableEnhancedMonitoringCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "EnableEnhancedMonitoring", {})
    .n("KinesisClient", "EnableEnhancedMonitoringCommand")
    .sc(EnableEnhancedMonitoring$)
    .build() {
}

class GetRecordsCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `data` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "GetRecords", {})
    .n("KinesisClient", "GetRecordsCommand")
    .sc(GetRecords$)
    .build() {
}

class GetResourcePolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    ResourceARN: { type: "contextParams", name: "ResourceARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "GetResourcePolicy", {})
    .n("KinesisClient", "GetResourcePolicyCommand")
    .sc(GetResourcePolicy$)
    .build() {
}

class GetShardIteratorCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `data` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "GetShardIterator", {})
    .n("KinesisClient", "GetShardIteratorCommand")
    .sc(GetShardIterator$)
    .build() {
}

class IncreaseStreamRetentionPeriodCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "IncreaseStreamRetentionPeriod", {})
    .n("KinesisClient", "IncreaseStreamRetentionPeriodCommand")
    .sc(IncreaseStreamRetentionPeriod$)
    .build() {
}

class ListShardsCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "ListShards", {})
    .n("KinesisClient", "ListShardsCommand")
    .sc(ListShards$)
    .build() {
}

class ListStreamConsumersCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "ListStreamConsumers", {})
    .n("KinesisClient", "ListStreamConsumersCommand")
    .sc(ListStreamConsumers$)
    .build() {
}

class ListStreamsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "ListStreams", {})
    .n("KinesisClient", "ListStreamsCommand")
    .sc(ListStreams$)
    .build() {
}

class ListTagsForResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    ResourceARN: { type: "contextParams", name: "ResourceARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "ListTagsForResource", {})
    .n("KinesisClient", "ListTagsForResourceCommand")
    .sc(ListTagsForResource$)
    .build() {
}

class ListTagsForStreamCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "ListTagsForStream", {})
    .n("KinesisClient", "ListTagsForStreamCommand")
    .sc(ListTagsForStream$)
    .build() {
}

class MergeShardsCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "MergeShards", {})
    .n("KinesisClient", "MergeShardsCommand")
    .sc(MergeShards$)
    .build() {
}

class PutRecordCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `data` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "PutRecord", {})
    .n("KinesisClient", "PutRecordCommand")
    .sc(PutRecord$)
    .build() {
}

class PutRecordsCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `data` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "PutRecords", {})
    .n("KinesisClient", "PutRecordsCommand")
    .sc(PutRecords$)
    .build() {
}

class PutResourcePolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    ResourceARN: { type: "contextParams", name: "ResourceARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "PutResourcePolicy", {})
    .n("KinesisClient", "PutResourcePolicyCommand")
    .sc(PutResourcePolicy$)
    .build() {
}

class RegisterStreamConsumerCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "RegisterStreamConsumer", {})
    .n("KinesisClient", "RegisterStreamConsumerCommand")
    .sc(RegisterStreamConsumer$)
    .build() {
}

class RemoveTagsFromStreamCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "RemoveTagsFromStream", {})
    .n("KinesisClient", "RemoveTagsFromStreamCommand")
    .sc(RemoveTagsFromStream$)
    .build() {
}

class SplitShardCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "SplitShard", {})
    .n("KinesisClient", "SplitShardCommand")
    .sc(SplitShard$)
    .build() {
}

class StartStreamEncryptionCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "StartStreamEncryption", {})
    .n("KinesisClient", "StartStreamEncryptionCommand")
    .sc(StartStreamEncryption$)
    .build() {
}

class StopStreamEncryptionCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "StopStreamEncryption", {})
    .n("KinesisClient", "StopStreamEncryptionCommand")
    .sc(StopStreamEncryption$)
    .build() {
}

class SubscribeToShardCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `data` },
    ConsumerARN: { type: "contextParams", name: "ConsumerARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "SubscribeToShard", {
    eventStream: {
        output: true,
    },
})
    .n("KinesisClient", "SubscribeToShardCommand")
    .sc(SubscribeToShard$)
    .build() {
}

class TagResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    ResourceARN: { type: "contextParams", name: "ResourceARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "TagResource", {})
    .n("KinesisClient", "TagResourceCommand")
    .sc(TagResource$)
    .build() {
}

class UntagResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    ResourceARN: { type: "contextParams", name: "ResourceARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "UntagResource", {})
    .n("KinesisClient", "UntagResourceCommand")
    .sc(UntagResource$)
    .build() {
}

class UpdateAccountSettingsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "UpdateAccountSettings", {})
    .n("KinesisClient", "UpdateAccountSettingsCommand")
    .sc(UpdateAccountSettings$)
    .build() {
}

class UpdateMaxRecordSizeCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "UpdateMaxRecordSize", {})
    .n("KinesisClient", "UpdateMaxRecordSizeCommand")
    .sc(UpdateMaxRecordSize$)
    .build() {
}

class UpdateShardCountCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "UpdateShardCount", {})
    .n("KinesisClient", "UpdateShardCountCommand")
    .sc(UpdateShardCount$)
    .build() {
}

class UpdateStreamModeCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "UpdateStreamMode", {})
    .n("KinesisClient", "UpdateStreamModeCommand")
    .sc(UpdateStreamMode$)
    .build() {
}

class UpdateStreamWarmThroughputCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    OperationType: { type: "staticContextParams", value: `control` },
    StreamARN: { type: "contextParams", name: "StreamARN" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Kinesis_20131202", "UpdateStreamWarmThroughput", {})
    .n("KinesisClient", "UpdateStreamWarmThroughputCommand")
    .sc(UpdateStreamWarmThroughput$)
    .build() {
}

const commands = {
    AddTagsToStreamCommand,
    CreateStreamCommand,
    DecreaseStreamRetentionPeriodCommand,
    DeleteResourcePolicyCommand,
    DeleteStreamCommand,
    DeregisterStreamConsumerCommand,
    DescribeAccountSettingsCommand,
    DescribeLimitsCommand,
    DescribeStreamCommand,
    DescribeStreamConsumerCommand,
    DescribeStreamSummaryCommand,
    DisableEnhancedMonitoringCommand,
    EnableEnhancedMonitoringCommand,
    GetRecordsCommand,
    GetResourcePolicyCommand,
    GetShardIteratorCommand,
    IncreaseStreamRetentionPeriodCommand,
    ListShardsCommand,
    ListStreamConsumersCommand,
    ListStreamsCommand,
    ListTagsForResourceCommand,
    ListTagsForStreamCommand,
    MergeShardsCommand,
    PutRecordCommand,
    PutRecordsCommand,
    PutResourcePolicyCommand,
    RegisterStreamConsumerCommand,
    RemoveTagsFromStreamCommand,
    SplitShardCommand,
    StartStreamEncryptionCommand,
    StopStreamEncryptionCommand,
    SubscribeToShardCommand,
    TagResourceCommand,
    UntagResourceCommand,
    UpdateAccountSettingsCommand,
    UpdateMaxRecordSizeCommand,
    UpdateShardCountCommand,
    UpdateStreamModeCommand,
    UpdateStreamWarmThroughputCommand,
};
class Kinesis extends KinesisClient {
}
smithyClient.createAggregatedClient(commands, Kinesis);

const paginateListStreamConsumers = core.createPaginator(KinesisClient, ListStreamConsumersCommand, "NextToken", "NextToken", "MaxResults");

const paginateListStreams = core.createPaginator(KinesisClient, ListStreamsCommand, "NextToken", "NextToken", "Limit");

const checkState$1 = async (client, input) => {
    let reason;
    try {
        let result = await client.send(new DescribeStreamCommand(input));
        reason = result;
        try {
            const returnComparator = () => {
                return result.StreamDescription.StreamStatus;
            };
            if (returnComparator() === "ACTIVE") {
                return { state: utilWaiter.WaiterState.SUCCESS, reason };
            }
        }
        catch (e) { }
    }
    catch (exception) {
        reason = exception;
    }
    return { state: utilWaiter.WaiterState.RETRY, reason };
};
const waitForStreamExists = async (params, input) => {
    const serviceDefaults = { minDelay: 10, maxDelay: 120 };
    return utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState$1);
};
const waitUntilStreamExists = async (params, input) => {
    const serviceDefaults = { minDelay: 10, maxDelay: 120 };
    const result = await utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState$1);
    return utilWaiter.checkExceptions(result);
};

const checkState = async (client, input) => {
    let reason;
    try {
        let result = await client.send(new DescribeStreamCommand(input));
        reason = result;
    }
    catch (exception) {
        reason = exception;
        if (exception.name && exception.name == "ResourceNotFoundException") {
            return { state: utilWaiter.WaiterState.SUCCESS, reason };
        }
    }
    return { state: utilWaiter.WaiterState.RETRY, reason };
};
const waitForStreamNotExists = async (params, input) => {
    const serviceDefaults = { minDelay: 10, maxDelay: 120 };
    return utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState);
};
const waitUntilStreamNotExists = async (params, input) => {
    const serviceDefaults = { minDelay: 10, maxDelay: 120 };
    const result = await utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState);
    return utilWaiter.checkExceptions(result);
};

const ConsumerStatus = {
    ACTIVE: "ACTIVE",
    CREATING: "CREATING",
    DELETING: "DELETING",
};
const StreamMode = {
    ON_DEMAND: "ON_DEMAND",
    PROVISIONED: "PROVISIONED",
};
const MinimumThroughputBillingCommitmentOutputStatus = {
    DISABLED: "DISABLED",
    ENABLED: "ENABLED",
    ENABLED_UNTIL_EARLIEST_ALLOWED_END: "ENABLED_UNTIL_EARLIEST_ALLOWED_END",
};
const EncryptionType = {
    KMS: "KMS",
    NONE: "NONE",
};
const MetricsName = {
    ALL: "ALL",
    INCOMING_BYTES: "IncomingBytes",
    INCOMING_RECORDS: "IncomingRecords",
    ITERATOR_AGE_MILLISECONDS: "IteratorAgeMilliseconds",
    OUTGOING_BYTES: "OutgoingBytes",
    OUTGOING_RECORDS: "OutgoingRecords",
    READ_PROVISIONED_THROUGHPUT_EXCEEDED: "ReadProvisionedThroughputExceeded",
    WRITE_PROVISIONED_THROUGHPUT_EXCEEDED: "WriteProvisionedThroughputExceeded",
};
const StreamStatus = {
    ACTIVE: "ACTIVE",
    CREATING: "CREATING",
    DELETING: "DELETING",
    UPDATING: "UPDATING",
};
const ShardIteratorType = {
    AFTER_SEQUENCE_NUMBER: "AFTER_SEQUENCE_NUMBER",
    AT_SEQUENCE_NUMBER: "AT_SEQUENCE_NUMBER",
    AT_TIMESTAMP: "AT_TIMESTAMP",
    LATEST: "LATEST",
    TRIM_HORIZON: "TRIM_HORIZON",
};
const ShardFilterType = {
    AFTER_SHARD_ID: "AFTER_SHARD_ID",
    AT_LATEST: "AT_LATEST",
    AT_TIMESTAMP: "AT_TIMESTAMP",
    AT_TRIM_HORIZON: "AT_TRIM_HORIZON",
    FROM_TIMESTAMP: "FROM_TIMESTAMP",
    FROM_TRIM_HORIZON: "FROM_TRIM_HORIZON",
};
const MinimumThroughputBillingCommitmentInputStatus = {
    DISABLED: "DISABLED",
    ENABLED: "ENABLED",
};
const ScalingType = {
    UNIFORM_SCALING: "UNIFORM_SCALING",
};

Object.defineProperty(exports, "$Command", {
    enumerable: true,
    get: function () { return smithyClient.Command; }
});
Object.defineProperty(exports, "__Client", {
    enumerable: true,
    get: function () { return smithyClient.Client; }
});
exports.AccessDeniedException = AccessDeniedException;
exports.AccessDeniedException$ = AccessDeniedException$;
exports.AddTagsToStream$ = AddTagsToStream$;
exports.AddTagsToStreamCommand = AddTagsToStreamCommand;
exports.AddTagsToStreamInput$ = AddTagsToStreamInput$;
exports.ChildShard$ = ChildShard$;
exports.Consumer$ = Consumer$;
exports.ConsumerDescription$ = ConsumerDescription$;
exports.ConsumerStatus = ConsumerStatus;
exports.CreateStream$ = CreateStream$;
exports.CreateStreamCommand = CreateStreamCommand;
exports.CreateStreamInput$ = CreateStreamInput$;
exports.DecreaseStreamRetentionPeriod$ = DecreaseStreamRetentionPeriod$;
exports.DecreaseStreamRetentionPeriodCommand = DecreaseStreamRetentionPeriodCommand;
exports.DecreaseStreamRetentionPeriodInput$ = DecreaseStreamRetentionPeriodInput$;
exports.DeleteResourcePolicy$ = DeleteResourcePolicy$;
exports.DeleteResourcePolicyCommand = DeleteResourcePolicyCommand;
exports.DeleteResourcePolicyInput$ = DeleteResourcePolicyInput$;
exports.DeleteStream$ = DeleteStream$;
exports.DeleteStreamCommand = DeleteStreamCommand;
exports.DeleteStreamInput$ = DeleteStreamInput$;
exports.DeregisterStreamConsumer$ = DeregisterStreamConsumer$;
exports.DeregisterStreamConsumerCommand = DeregisterStreamConsumerCommand;
exports.DeregisterStreamConsumerInput$ = DeregisterStreamConsumerInput$;
exports.DescribeAccountSettings$ = DescribeAccountSettings$;
exports.DescribeAccountSettingsCommand = DescribeAccountSettingsCommand;
exports.DescribeAccountSettingsInput$ = DescribeAccountSettingsInput$;
exports.DescribeAccountSettingsOutput$ = DescribeAccountSettingsOutput$;
exports.DescribeLimits$ = DescribeLimits$;
exports.DescribeLimitsCommand = DescribeLimitsCommand;
exports.DescribeLimitsInput$ = DescribeLimitsInput$;
exports.DescribeLimitsOutput$ = DescribeLimitsOutput$;
exports.DescribeStream$ = DescribeStream$;
exports.DescribeStreamCommand = DescribeStreamCommand;
exports.DescribeStreamConsumer$ = DescribeStreamConsumer$;
exports.DescribeStreamConsumerCommand = DescribeStreamConsumerCommand;
exports.DescribeStreamConsumerInput$ = DescribeStreamConsumerInput$;
exports.DescribeStreamConsumerOutput$ = DescribeStreamConsumerOutput$;
exports.DescribeStreamInput$ = DescribeStreamInput$;
exports.DescribeStreamOutput$ = DescribeStreamOutput$;
exports.DescribeStreamSummary$ = DescribeStreamSummary$;
exports.DescribeStreamSummaryCommand = DescribeStreamSummaryCommand;
exports.DescribeStreamSummaryInput$ = DescribeStreamSummaryInput$;
exports.DescribeStreamSummaryOutput$ = DescribeStreamSummaryOutput$;
exports.DisableEnhancedMonitoring$ = DisableEnhancedMonitoring$;
exports.DisableEnhancedMonitoringCommand = DisableEnhancedMonitoringCommand;
exports.DisableEnhancedMonitoringInput$ = DisableEnhancedMonitoringInput$;
exports.EnableEnhancedMonitoring$ = EnableEnhancedMonitoring$;
exports.EnableEnhancedMonitoringCommand = EnableEnhancedMonitoringCommand;
exports.EnableEnhancedMonitoringInput$ = EnableEnhancedMonitoringInput$;
exports.EncryptionType = EncryptionType;
exports.EnhancedMetrics$ = EnhancedMetrics$;
exports.EnhancedMonitoringOutput$ = EnhancedMonitoringOutput$;
exports.ExpiredIteratorException = ExpiredIteratorException;
exports.ExpiredIteratorException$ = ExpiredIteratorException$;
exports.ExpiredNextTokenException = ExpiredNextTokenException;
exports.ExpiredNextTokenException$ = ExpiredNextTokenException$;
exports.GetRecords$ = GetRecords$;
exports.GetRecordsCommand = GetRecordsCommand;
exports.GetRecordsInput$ = GetRecordsInput$;
exports.GetRecordsOutput$ = GetRecordsOutput$;
exports.GetResourcePolicy$ = GetResourcePolicy$;
exports.GetResourcePolicyCommand = GetResourcePolicyCommand;
exports.GetResourcePolicyInput$ = GetResourcePolicyInput$;
exports.GetResourcePolicyOutput$ = GetResourcePolicyOutput$;
exports.GetShardIterator$ = GetShardIterator$;
exports.GetShardIteratorCommand = GetShardIteratorCommand;
exports.GetShardIteratorInput$ = GetShardIteratorInput$;
exports.GetShardIteratorOutput$ = GetShardIteratorOutput$;
exports.HashKeyRange$ = HashKeyRange$;
exports.IncreaseStreamRetentionPeriod$ = IncreaseStreamRetentionPeriod$;
exports.IncreaseStreamRetentionPeriodCommand = IncreaseStreamRetentionPeriodCommand;
exports.IncreaseStreamRetentionPeriodInput$ = IncreaseStreamRetentionPeriodInput$;
exports.InternalFailureException = InternalFailureException;
exports.InternalFailureException$ = InternalFailureException$;
exports.InvalidArgumentException = InvalidArgumentException;
exports.InvalidArgumentException$ = InvalidArgumentException$;
exports.KMSAccessDeniedException = KMSAccessDeniedException;
exports.KMSAccessDeniedException$ = KMSAccessDeniedException$;
exports.KMSDisabledException = KMSDisabledException;
exports.KMSDisabledException$ = KMSDisabledException$;
exports.KMSInvalidStateException = KMSInvalidStateException;
exports.KMSInvalidStateException$ = KMSInvalidStateException$;
exports.KMSNotFoundException = KMSNotFoundException;
exports.KMSNotFoundException$ = KMSNotFoundException$;
exports.KMSOptInRequired = KMSOptInRequired;
exports.KMSOptInRequired$ = KMSOptInRequired$;
exports.KMSThrottlingException = KMSThrottlingException;
exports.KMSThrottlingException$ = KMSThrottlingException$;
exports.Kinesis = Kinesis;
exports.KinesisClient = KinesisClient;
exports.KinesisServiceException = KinesisServiceException;
exports.KinesisServiceException$ = KinesisServiceException$;
exports.LimitExceededException = LimitExceededException;
exports.LimitExceededException$ = LimitExceededException$;
exports.ListShards$ = ListShards$;
exports.ListShardsCommand = ListShardsCommand;
exports.ListShardsInput$ = ListShardsInput$;
exports.ListShardsOutput$ = ListShardsOutput$;
exports.ListStreamConsumers$ = ListStreamConsumers$;
exports.ListStreamConsumersCommand = ListStreamConsumersCommand;
exports.ListStreamConsumersInput$ = ListStreamConsumersInput$;
exports.ListStreamConsumersOutput$ = ListStreamConsumersOutput$;
exports.ListStreams$ = ListStreams$;
exports.ListStreamsCommand = ListStreamsCommand;
exports.ListStreamsInput$ = ListStreamsInput$;
exports.ListStreamsOutput$ = ListStreamsOutput$;
exports.ListTagsForResource$ = ListTagsForResource$;
exports.ListTagsForResourceCommand = ListTagsForResourceCommand;
exports.ListTagsForResourceInput$ = ListTagsForResourceInput$;
exports.ListTagsForResourceOutput$ = ListTagsForResourceOutput$;
exports.ListTagsForStream$ = ListTagsForStream$;
exports.ListTagsForStreamCommand = ListTagsForStreamCommand;
exports.ListTagsForStreamInput$ = ListTagsForStreamInput$;
exports.ListTagsForStreamOutput$ = ListTagsForStreamOutput$;
exports.MergeShards$ = MergeShards$;
exports.MergeShardsCommand = MergeShardsCommand;
exports.MergeShardsInput$ = MergeShardsInput$;
exports.MetricsName = MetricsName;
exports.MinimumThroughputBillingCommitmentInput$ = MinimumThroughputBillingCommitmentInput$;
exports.MinimumThroughputBillingCommitmentInputStatus = MinimumThroughputBillingCommitmentInputStatus;
exports.MinimumThroughputBillingCommitmentOutput$ = MinimumThroughputBillingCommitmentOutput$;
exports.MinimumThroughputBillingCommitmentOutputStatus = MinimumThroughputBillingCommitmentOutputStatus;
exports.ProvisionedThroughputExceededException = ProvisionedThroughputExceededException;
exports.ProvisionedThroughputExceededException$ = ProvisionedThroughputExceededException$;
exports.PutRecord$ = PutRecord$;
exports.PutRecordCommand = PutRecordCommand;
exports.PutRecordInput$ = PutRecordInput$;
exports.PutRecordOutput$ = PutRecordOutput$;
exports.PutRecords$ = PutRecords$;
exports.PutRecordsCommand = PutRecordsCommand;
exports.PutRecordsInput$ = PutRecordsInput$;
exports.PutRecordsOutput$ = PutRecordsOutput$;
exports.PutRecordsRequestEntry$ = PutRecordsRequestEntry$;
exports.PutRecordsResultEntry$ = PutRecordsResultEntry$;
exports.PutResourcePolicy$ = PutResourcePolicy$;
exports.PutResourcePolicyCommand = PutResourcePolicyCommand;
exports.PutResourcePolicyInput$ = PutResourcePolicyInput$;
exports.RegisterStreamConsumer$ = RegisterStreamConsumer$;
exports.RegisterStreamConsumerCommand = RegisterStreamConsumerCommand;
exports.RegisterStreamConsumerInput$ = RegisterStreamConsumerInput$;
exports.RegisterStreamConsumerOutput$ = RegisterStreamConsumerOutput$;
exports.RemoveTagsFromStream$ = RemoveTagsFromStream$;
exports.RemoveTagsFromStreamCommand = RemoveTagsFromStreamCommand;
exports.RemoveTagsFromStreamInput$ = RemoveTagsFromStreamInput$;
exports.ResourceInUseException = ResourceInUseException;
exports.ResourceInUseException$ = ResourceInUseException$;
exports.ResourceNotFoundException = ResourceNotFoundException;
exports.ResourceNotFoundException$ = ResourceNotFoundException$;
exports.ScalingType = ScalingType;
exports.SequenceNumberRange$ = SequenceNumberRange$;
exports.Shard$ = Shard$;
exports.ShardFilter$ = ShardFilter$;
exports.ShardFilterType = ShardFilterType;
exports.ShardIteratorType = ShardIteratorType;
exports.SplitShard$ = SplitShard$;
exports.SplitShardCommand = SplitShardCommand;
exports.SplitShardInput$ = SplitShardInput$;
exports.StartStreamEncryption$ = StartStreamEncryption$;
exports.StartStreamEncryptionCommand = StartStreamEncryptionCommand;
exports.StartStreamEncryptionInput$ = StartStreamEncryptionInput$;
exports.StartingPosition$ = StartingPosition$;
exports.StopStreamEncryption$ = StopStreamEncryption$;
exports.StopStreamEncryptionCommand = StopStreamEncryptionCommand;
exports.StopStreamEncryptionInput$ = StopStreamEncryptionInput$;
exports.StreamDescription$ = StreamDescription$;
exports.StreamDescriptionSummary$ = StreamDescriptionSummary$;
exports.StreamMode = StreamMode;
exports.StreamModeDetails$ = StreamModeDetails$;
exports.StreamStatus = StreamStatus;
exports.StreamSummary$ = StreamSummary$;
exports.SubscribeToShard$ = SubscribeToShard$;
exports.SubscribeToShardCommand = SubscribeToShardCommand;
exports.SubscribeToShardEvent$ = SubscribeToShardEvent$;
exports.SubscribeToShardEventStream$ = SubscribeToShardEventStream$;
exports.SubscribeToShardInput$ = SubscribeToShardInput$;
exports.SubscribeToShardOutput$ = SubscribeToShardOutput$;
exports.Tag$ = Tag$;
exports.TagResource$ = TagResource$;
exports.TagResourceCommand = TagResourceCommand;
exports.TagResourceInput$ = TagResourceInput$;
exports.UntagResource$ = UntagResource$;
exports.UntagResourceCommand = UntagResourceCommand;
exports.UntagResourceInput$ = UntagResourceInput$;
exports.UpdateAccountSettings$ = UpdateAccountSettings$;
exports.UpdateAccountSettingsCommand = UpdateAccountSettingsCommand;
exports.UpdateAccountSettingsInput$ = UpdateAccountSettingsInput$;
exports.UpdateAccountSettingsOutput$ = UpdateAccountSettingsOutput$;
exports.UpdateMaxRecordSize$ = UpdateMaxRecordSize$;
exports.UpdateMaxRecordSizeCommand = UpdateMaxRecordSizeCommand;
exports.UpdateMaxRecordSizeInput$ = UpdateMaxRecordSizeInput$;
exports.UpdateShardCount$ = UpdateShardCount$;
exports.UpdateShardCountCommand = UpdateShardCountCommand;
exports.UpdateShardCountInput$ = UpdateShardCountInput$;
exports.UpdateShardCountOutput$ = UpdateShardCountOutput$;
exports.UpdateStreamMode$ = UpdateStreamMode$;
exports.UpdateStreamModeCommand = UpdateStreamModeCommand;
exports.UpdateStreamModeInput$ = UpdateStreamModeInput$;
exports.UpdateStreamWarmThroughput$ = UpdateStreamWarmThroughput$;
exports.UpdateStreamWarmThroughputCommand = UpdateStreamWarmThroughputCommand;
exports.UpdateStreamWarmThroughputInput$ = UpdateStreamWarmThroughputInput$;
exports.UpdateStreamWarmThroughputOutput$ = UpdateStreamWarmThroughputOutput$;
exports.ValidationException = ValidationException;
exports.ValidationException$ = ValidationException$;
exports.WarmThroughputObject$ = WarmThroughputObject$;
exports._Record$ = _Record$;
exports.paginateListStreamConsumers = paginateListStreamConsumers;
exports.paginateListStreams = paginateListStreams;
exports.waitForStreamExists = waitForStreamExists;
exports.waitForStreamNotExists = waitForStreamNotExists;
exports.waitUntilStreamExists = waitUntilStreamExists;
exports.waitUntilStreamNotExists = waitUntilStreamNotExists;
