'use strict';

var middlewareHostHeader = require('@aws-sdk/middleware-host-header');
var middlewareLogger = require('@aws-sdk/middleware-logger');
var middlewareRecursionDetection = require('@aws-sdk/middleware-recursion-detection');
var middlewareUserAgent = require('@aws-sdk/middleware-user-agent');
var configResolver = require('@smithy/config-resolver');
var core = require('@smithy/core');
var schema = require('@smithy/core/schema');
var middlewareContentLength = require('@smithy/middleware-content-length');
var middlewareEndpoint = require('@smithy/middleware-endpoint');
var middlewareRetry = require('@smithy/middleware-retry');
var smithyClient = require('@smithy/smithy-client');
var httpAuthSchemeProvider = require('./auth/httpAuthSchemeProvider');
var runtimeConfig = require('./runtimeConfig');
var regionConfigResolver = require('@aws-sdk/region-config-resolver');
var protocolHttp = require('@smithy/protocol-http');

const resolveClientEndpointParameters = (options) => {
    return Object.assign(options, {
        useDualstackEndpoint: options.useDualstackEndpoint ?? false,
        useFipsEndpoint: options.useFipsEndpoint ?? false,
        defaultSigningName: "sns",
    });
};
const commonParams = {
    UseFIPS: { type: "builtInParams", name: "useFipsEndpoint" },
    Endpoint: { type: "builtInParams", name: "endpoint" },
    Region: { type: "builtInParams", name: "region" },
    UseDualStack: { type: "builtInParams", name: "useDualstackEndpoint" },
};

const getHttpAuthExtensionConfiguration = (runtimeConfig) => {
    const _httpAuthSchemes = runtimeConfig.httpAuthSchemes;
    let _httpAuthSchemeProvider = runtimeConfig.httpAuthSchemeProvider;
    let _credentials = runtimeConfig.credentials;
    return {
        setHttpAuthScheme(httpAuthScheme) {
            const index = _httpAuthSchemes.findIndex((scheme) => scheme.schemeId === httpAuthScheme.schemeId);
            if (index === -1) {
                _httpAuthSchemes.push(httpAuthScheme);
            }
            else {
                _httpAuthSchemes.splice(index, 1, httpAuthScheme);
            }
        },
        httpAuthSchemes() {
            return _httpAuthSchemes;
        },
        setHttpAuthSchemeProvider(httpAuthSchemeProvider) {
            _httpAuthSchemeProvider = httpAuthSchemeProvider;
        },
        httpAuthSchemeProvider() {
            return _httpAuthSchemeProvider;
        },
        setCredentials(credentials) {
            _credentials = credentials;
        },
        credentials() {
            return _credentials;
        },
    };
};
const resolveHttpAuthRuntimeConfig = (config) => {
    return {
        httpAuthSchemes: config.httpAuthSchemes(),
        httpAuthSchemeProvider: config.httpAuthSchemeProvider(),
        credentials: config.credentials(),
    };
};

const resolveRuntimeExtensions = (runtimeConfig, extensions) => {
    const extensionConfiguration = Object.assign(regionConfigResolver.getAwsRegionExtensionConfiguration(runtimeConfig), smithyClient.getDefaultExtensionConfiguration(runtimeConfig), protocolHttp.getHttpHandlerExtensionConfiguration(runtimeConfig), getHttpAuthExtensionConfiguration(runtimeConfig));
    extensions.forEach((extension) => extension.configure(extensionConfiguration));
    return Object.assign(runtimeConfig, regionConfigResolver.resolveAwsRegionExtensionConfiguration(extensionConfiguration), smithyClient.resolveDefaultRuntimeConfig(extensionConfiguration), protocolHttp.resolveHttpHandlerRuntimeConfig(extensionConfiguration), resolveHttpAuthRuntimeConfig(extensionConfiguration));
};

class SNSClient extends smithyClient.Client {
    config;
    constructor(...[configuration]) {
        const _config_0 = runtimeConfig.getRuntimeConfig(configuration || {});
        super(_config_0);
        this.initConfig = _config_0;
        const _config_1 = resolveClientEndpointParameters(_config_0);
        const _config_2 = middlewareUserAgent.resolveUserAgentConfig(_config_1);
        const _config_3 = middlewareRetry.resolveRetryConfig(_config_2);
        const _config_4 = configResolver.resolveRegionConfig(_config_3);
        const _config_5 = middlewareHostHeader.resolveHostHeaderConfig(_config_4);
        const _config_6 = middlewareEndpoint.resolveEndpointConfig(_config_5);
        const _config_7 = httpAuthSchemeProvider.resolveHttpAuthSchemeConfig(_config_6);
        const _config_8 = resolveRuntimeExtensions(_config_7, configuration?.extensions || []);
        this.config = _config_8;
        this.middlewareStack.use(schema.getSchemaSerdePlugin(this.config));
        this.middlewareStack.use(middlewareUserAgent.getUserAgentPlugin(this.config));
        this.middlewareStack.use(middlewareRetry.getRetryPlugin(this.config));
        this.middlewareStack.use(middlewareContentLength.getContentLengthPlugin(this.config));
        this.middlewareStack.use(middlewareHostHeader.getHostHeaderPlugin(this.config));
        this.middlewareStack.use(middlewareLogger.getLoggerPlugin(this.config));
        this.middlewareStack.use(middlewareRecursionDetection.getRecursionDetectionPlugin(this.config));
        this.middlewareStack.use(core.getHttpAuthSchemeEndpointRuleSetPlugin(this.config, {
            httpAuthSchemeParametersProvider: httpAuthSchemeProvider.defaultSNSHttpAuthSchemeParametersProvider,
            identityProviderConfigProvider: async (config) => new core.DefaultIdentityProviderConfig({
                "aws.auth#sigv4": config.credentials,
            }),
        }));
        this.middlewareStack.use(core.getHttpSigningPlugin(this.config));
    }
    destroy() {
        super.destroy();
    }
}

class SNSServiceException extends smithyClient.ServiceException {
    constructor(options) {
        super(options);
        Object.setPrototypeOf(this, SNSServiceException.prototype);
    }
}

class AuthorizationErrorException extends SNSServiceException {
    name = "AuthorizationErrorException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "AuthorizationErrorException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, AuthorizationErrorException.prototype);
    }
}
class InternalErrorException extends SNSServiceException {
    name = "InternalErrorException";
    $fault = "server";
    constructor(opts) {
        super({
            name: "InternalErrorException",
            $fault: "server",
            ...opts,
        });
        Object.setPrototypeOf(this, InternalErrorException.prototype);
    }
}
class InvalidParameterException extends SNSServiceException {
    name = "InvalidParameterException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "InvalidParameterException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidParameterException.prototype);
    }
}
class NotFoundException extends SNSServiceException {
    name = "NotFoundException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "NotFoundException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NotFoundException.prototype);
    }
}
class ThrottledException extends SNSServiceException {
    name = "ThrottledException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ThrottledException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ThrottledException.prototype);
    }
}
class FilterPolicyLimitExceededException extends SNSServiceException {
    name = "FilterPolicyLimitExceededException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "FilterPolicyLimitExceededException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, FilterPolicyLimitExceededException.prototype);
    }
}
class ReplayLimitExceededException extends SNSServiceException {
    name = "ReplayLimitExceededException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ReplayLimitExceededException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ReplayLimitExceededException.prototype);
    }
}
class SubscriptionLimitExceededException extends SNSServiceException {
    name = "SubscriptionLimitExceededException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "SubscriptionLimitExceededException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, SubscriptionLimitExceededException.prototype);
    }
}
class OptedOutException extends SNSServiceException {
    name = "OptedOutException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "OptedOutException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, OptedOutException.prototype);
    }
}
class UserErrorException extends SNSServiceException {
    name = "UserErrorException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "UserErrorException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, UserErrorException.prototype);
    }
}
class ConcurrentAccessException extends SNSServiceException {
    name = "ConcurrentAccessException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ConcurrentAccessException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ConcurrentAccessException.prototype);
    }
}
class InvalidSecurityException extends SNSServiceException {
    name = "InvalidSecurityException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "InvalidSecurityException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidSecurityException.prototype);
    }
}
class StaleTagException extends SNSServiceException {
    name = "StaleTagException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "StaleTagException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, StaleTagException.prototype);
    }
}
class TagLimitExceededException extends SNSServiceException {
    name = "TagLimitExceededException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "TagLimitExceededException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TagLimitExceededException.prototype);
    }
}
class TagPolicyException extends SNSServiceException {
    name = "TagPolicyException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "TagPolicyException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TagPolicyException.prototype);
    }
}
class TopicLimitExceededException extends SNSServiceException {
    name = "TopicLimitExceededException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "TopicLimitExceededException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TopicLimitExceededException.prototype);
    }
}
class ResourceNotFoundException extends SNSServiceException {
    name = "ResourceNotFoundException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ResourceNotFoundException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ResourceNotFoundException.prototype);
    }
}
class InvalidStateException extends SNSServiceException {
    name = "InvalidStateException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "InvalidStateException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidStateException.prototype);
    }
}
class ValidationException extends SNSServiceException {
    name = "ValidationException";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "ValidationException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ValidationException.prototype);
        this.Message = opts.Message;
    }
}
class EndpointDisabledException extends SNSServiceException {
    name = "EndpointDisabledException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "EndpointDisabledException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, EndpointDisabledException.prototype);
    }
}
class InvalidParameterValueException extends SNSServiceException {
    name = "InvalidParameterValueException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "InvalidParameterValueException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidParameterValueException.prototype);
    }
}
class KMSAccessDeniedException extends SNSServiceException {
    name = "KMSAccessDeniedException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "KMSAccessDeniedException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, KMSAccessDeniedException.prototype);
    }
}
class KMSDisabledException extends SNSServiceException {
    name = "KMSDisabledException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "KMSDisabledException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, KMSDisabledException.prototype);
    }
}
class KMSInvalidStateException extends SNSServiceException {
    name = "KMSInvalidStateException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "KMSInvalidStateException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, KMSInvalidStateException.prototype);
    }
}
class KMSNotFoundException extends SNSServiceException {
    name = "KMSNotFoundException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "KMSNotFoundException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, KMSNotFoundException.prototype);
    }
}
class KMSOptInRequired extends SNSServiceException {
    name = "KMSOptInRequired";
    $fault = "client";
    constructor(opts) {
        super({
            name: "KMSOptInRequired",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, KMSOptInRequired.prototype);
    }
}
class KMSThrottlingException extends SNSServiceException {
    name = "KMSThrottlingException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "KMSThrottlingException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, KMSThrottlingException.prototype);
    }
}
class PlatformApplicationDisabledException extends SNSServiceException {
    name = "PlatformApplicationDisabledException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "PlatformApplicationDisabledException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, PlatformApplicationDisabledException.prototype);
    }
}
class BatchEntryIdsNotDistinctException extends SNSServiceException {
    name = "BatchEntryIdsNotDistinctException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "BatchEntryIdsNotDistinctException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, BatchEntryIdsNotDistinctException.prototype);
    }
}
class BatchRequestTooLongException extends SNSServiceException {
    name = "BatchRequestTooLongException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "BatchRequestTooLongException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, BatchRequestTooLongException.prototype);
    }
}
class EmptyBatchRequestException extends SNSServiceException {
    name = "EmptyBatchRequestException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "EmptyBatchRequestException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, EmptyBatchRequestException.prototype);
    }
}
class InvalidBatchEntryIdException extends SNSServiceException {
    name = "InvalidBatchEntryIdException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "InvalidBatchEntryIdException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidBatchEntryIdException.prototype);
    }
}
class TooManyEntriesInBatchRequestException extends SNSServiceException {
    name = "TooManyEntriesInBatchRequestException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "TooManyEntriesInBatchRequestException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyEntriesInBatchRequestException.prototype);
    }
}
class VerificationException extends SNSServiceException {
    name = "VerificationException";
    $fault = "client";
    Message;
    Status;
    constructor(opts) {
        super({
            name: "VerificationException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, VerificationException.prototype);
        this.Message = opts.Message;
        this.Status = opts.Status;
    }
}

const _A = "Attributes";
const _AEE = "AuthorizationErrorException";
const _AN = "ActionName";
const _ANt = "AttributeName";
const _AOU = "AuthenticateOnUnsubscribe";
const _AP = "AddPermission";
const _API = "AddPermissionInput";
const _AV = "AttributeValue";
const _AWSAI = "AWSAccountId";
const _BEINDE = "BatchEntryIdsNotDistinctException";
const _BREE = "BatchResultErrorEntry";
const _BREEL = "BatchResultErrorEntryList";
const _BRTLE = "BatchRequestTooLongException";
const _BV = "BinaryValue";
const _C = "Code";
const _CA = "CreatedAt";
const _CAE = "ConcurrentAccessException";
const _CER = "CreateEndpointResponse";
const _CIPNIOO = "CheckIfPhoneNumberIsOptedOut";
const _CIPNIOOI = "CheckIfPhoneNumberIsOptedOutInput";
const _CIPNIOOR = "CheckIfPhoneNumberIsOptedOutResponse";
const _CPA = "CreatePlatformApplication";
const _CPAI = "CreatePlatformApplicationInput";
const _CPAR = "CreatePlatformApplicationResponse";
const _CPE = "CreatePlatformEndpoint";
const _CPEI = "CreatePlatformEndpointInput";
const _CS = "ConfirmSubscription";
const _CSI = "ConfirmSubscriptionInput";
const _CSMSSPN = "CreateSMSSandboxPhoneNumber";
const _CSMSSPNI = "CreateSMSSandboxPhoneNumberInput";
const _CSMSSPNR = "CreateSMSSandboxPhoneNumberResult";
const _CSR = "ConfirmSubscriptionResponse";
const _CT = "CreateTopic";
const _CTI = "CreateTopicInput";
const _CTR = "CreateTopicResponse";
const _CUD = "CustomUserData";
const _DE = "DeleteEndpoint";
const _DEI = "DeleteEndpointInput";
const _DPA = "DeletePlatformApplication";
const _DPAI = "DeletePlatformApplicationInput";
const _DPP = "DataProtectionPolicy";
const _DSMSSPN = "DeleteSMSSandboxPhoneNumber";
const _DSMSSPNI = "DeleteSMSSandboxPhoneNumberInput";
const _DSMSSPNR = "DeleteSMSSandboxPhoneNumberResult";
const _DT = "DataType";
const _DTI = "DeleteTopicInput";
const _DTe = "DeleteTopic";
const _E = "Endpoint";
const _EA = "EndpointArn";
const _EBRE = "EmptyBatchRequestException";
const _EDE = "EndpointDisabledException";
const _En = "Endpoints";
const _F = "Failed";
const _FPLEE = "FilterPolicyLimitExceededException";
const _GDPP = "GetDataProtectionPolicy";
const _GDPPI = "GetDataProtectionPolicyInput";
const _GDPPR = "GetDataProtectionPolicyResponse";
const _GEA = "GetEndpointAttributes";
const _GEAI = "GetEndpointAttributesInput";
const _GEAR = "GetEndpointAttributesResponse";
const _GPAA = "GetPlatformApplicationAttributes";
const _GPAAI = "GetPlatformApplicationAttributesInput";
const _GPAAR = "GetPlatformApplicationAttributesResponse";
const _GSA = "GetSubscriptionAttributes";
const _GSAI = "GetSubscriptionAttributesInput";
const _GSAR = "GetSubscriptionAttributesResponse";
const _GSMSA = "GetSMSAttributes";
const _GSMSAI = "GetSMSAttributesInput";
const _GSMSAR = "GetSMSAttributesResponse";
const _GSMSSAS = "GetSMSSandboxAccountStatus";
const _GSMSSASI = "GetSMSSandboxAccountStatusInput";
const _GSMSSASR = "GetSMSSandboxAccountStatusResult";
const _GTA = "GetTopicAttributes";
const _GTAI = "GetTopicAttributesInput";
const _GTAR = "GetTopicAttributesResponse";
const _I = "Id";
const _IBEIE = "InvalidBatchEntryIdException";
const _ICC = "Iso2CountryCode";
const _IEE = "InternalErrorException";
const _IIS = "IsInSandbox";
const _IPE = "InvalidParameterException";
const _IPVE = "InvalidParameterValueException";
const _ISE = "InvalidSecurityException";
const _ISEn = "InvalidStateException";
const _K = "Key";
const _KMSADE = "KMSAccessDeniedException";
const _KMSDE = "KMSDisabledException";
const _KMSISE = "KMSInvalidStateException";
const _KMSNFE = "KMSNotFoundException";
const _KMSOIR = "KMSOptInRequired";
const _KMSTE = "KMSThrottlingException";
const _L = "Label";
const _LC = "LanguageCode";
const _LEBPA = "ListEndpointsByPlatformApplication";
const _LEBPAI = "ListEndpointsByPlatformApplicationInput";
const _LEBPAR = "ListEndpointsByPlatformApplicationResponse";
const _LOE = "ListOfEndpoints";
const _LON = "ListOriginationNumbers";
const _LONR = "ListOriginationNumbersRequest";
const _LONRi = "ListOriginationNumbersResult";
const _LOPA = "ListOfPlatformApplications";
const _LPA = "ListPlatformApplications";
const _LPAI = "ListPlatformApplicationsInput";
const _LPAR = "ListPlatformApplicationsResponse";
const _LPNOO = "ListPhoneNumbersOptedOut";
const _LPNOOI = "ListPhoneNumbersOptedOutInput";
const _LPNOOR = "ListPhoneNumbersOptedOutResponse";
const _LS = "ListSubscriptions";
const _LSBT = "ListSubscriptionsByTopic";
const _LSBTI = "ListSubscriptionsByTopicInput";
const _LSBTR = "ListSubscriptionsByTopicResponse";
const _LSI = "ListSubscriptionsInput";
const _LSMSSPN = "ListSMSSandboxPhoneNumbers";
const _LSMSSPNI = "ListSMSSandboxPhoneNumbersInput";
const _LSMSSPNR = "ListSMSSandboxPhoneNumbersResult";
const _LSR = "ListSubscriptionsResponse";
const _LT = "ListTopics";
const _LTFR = "ListTagsForResource";
const _LTFRR = "ListTagsForResourceRequest";
const _LTFRRi = "ListTagsForResourceResponse";
const _LTI = "ListTopicsInput";
const _LTR = "ListTopicsResponse";
const _M = "Message";
const _MA = "MessageAttributes";
const _MAM = "MessageAttributeMap";
const _MAV = "MessageAttributeValue";
const _MDI = "MessageDeduplicationId";
const _MGI = "MessageGroupId";
const _MI = "MessageId";
const _MR = "MaxResults";
const _MS = "MessageStructure";
const _N = "Name";
const _NC = "NumberCapabilities";
const _NFE = "NotFoundException";
const _NT = "NextToken";
const _O = "Owner";
const _OIPN = "OptInPhoneNumber";
const _OIPNI = "OptInPhoneNumberInput";
const _OIPNR = "OptInPhoneNumberResponse";
const _OOE = "OptedOutException";
const _OTP = "OneTimePassword";
const _P = "Platform";
const _PA = "PlatformApplications";
const _PAA = "PlatformApplicationArn";
const _PADE = "PlatformApplicationDisabledException";
const _PAl = "PlatformApplication";
const _PB = "PublishBatch";
const _PBI = "PublishBatchInput";
const _PBR = "PublishBatchResponse";
const _PBRE = "PublishBatchRequestEntries";
const _PBREL = "PublishBatchRequestEntryList";
const _PBRELu = "PublishBatchResultEntryList";
const _PBREu = "PublishBatchRequestEntry";
const _PBREub = "PublishBatchResultEntry";
const _PDPP = "PutDataProtectionPolicy";
const _PDPPI = "PutDataProtectionPolicyInput";
const _PI = "PublishInput";
const _PN = "PhoneNumber";
const _PNI = "PhoneNumberInformation";
const _PNIL = "PhoneNumberInformationList";
const _PNL = "PhoneNumberList";
const _PNS = "PhoneNumberString";
const _PNh = "PhoneNumbers";
const _PR = "PublishResponse";
const _Pr = "Protocol";
const _Pu = "Publish";
const _RA = "ResourceArn";
const _RLEE = "ReplayLimitExceededException";
const _RNFE = "ResourceNotFoundException";
const _RP = "RemovePermission";
const _RPI = "RemovePermissionInput";
const _RSA = "ReturnSubscriptionArn";
const _RT = "RouteType";
const _S = "Subscriptions";
const _SA = "SubscriptionArn";
const _SEA = "SetEndpointAttributes";
const _SEAI = "SetEndpointAttributesInput";
const _SF = "SenderFault";
const _SI = "SubscribeInput";
const _SL = "SubscriptionsList";
const _SLEE = "SubscriptionLimitExceededException";
const _SMSSPN = "SMSSandboxPhoneNumber";
const _SMSSPNL = "SMSSandboxPhoneNumberList";
const _SN = "SequenceNumber";
const _SPAA = "SetPlatformApplicationAttributes";
const _SPAAI = "SetPlatformApplicationAttributesInput";
const _SR = "SubscribeResponse";
const _SSA = "SetSubscriptionAttributes";
const _SSAI = "SetSubscriptionAttributesInput";
const _SSMSA = "SetSMSAttributes";
const _SSMSAI = "SetSMSAttributesInput";
const _SSMSAR = "SetSMSAttributesResponse";
const _STA = "SetTopicAttributes";
const _STAI = "SetTopicAttributesInput";
const _STE = "StaleTagException";
const _SV = "StringValue";
const _St = "Status";
const _Su = "Subject";
const _Sub = "Subscription";
const _Subs = "Subscribe";
const _Suc = "Successful";
const _T = "Token";
const _TA = "TopicArn";
const _TAa = "TargetArn";
const _TE = "ThrottledException";
const _TK = "TagKeys";
const _TL = "TagList";
const _TLEE = "TagLimitExceededException";
const _TLEEo = "TopicLimitExceededException";
const _TLo = "TopicsList";
const _TMEIBRE = "TooManyEntriesInBatchRequestException";
const _TPE = "TagPolicyException";
const _TR = "TagResource";
const _TRR = "TagResourceRequest";
const _TRRa = "TagResourceResponse";
const _Ta = "Tags";
const _Tag = "Tag";
const _To = "Topics";
const _Top = "Topic";
const _U = "Unsubscribe";
const _UEE = "UserErrorException";
const _UI = "UnsubscribeInput";
const _UR = "UntagResource";
const _URR = "UntagResourceRequest";
const _URRn = "UntagResourceResponse";
const _V = "Value";
const _VE = "ValidationException";
const _VEe = "VerificationException";
const _VSMSSPN = "VerifySMSSandboxPhoneNumber";
const _VSMSSPNI = "VerifySMSSandboxPhoneNumberInput";
const _VSMSSPNR = "VerifySMSSandboxPhoneNumberResult";
const _a = "attributes";
const _aQE = "awsQueryError";
const _c = "client";
const _e = "error";
const _hE = "httpError";
const _iOO = "isOptedOut";
const _m = "message";
const _nT = "nextToken";
const _pN = "phoneNumber";
const _pNh = "phoneNumbers";
const _s = "server";
const _sm = "smithy.ts.sdk.synthetic.com.amazonaws.sns";
const _xN = "xmlName";
const n0 = "com.amazonaws.sns";
var PhoneNumber = [0, n0, _PN, 8, 0];
var PhoneNumberString = [0, n0, _PNS, 8, 0];
var AddPermissionInput$ = [3, n0, _API,
    0,
    [_TA, _L, _AWSAI, _AN],
    [0, 0, 64 | 0, 64 | 0]
];
var AuthorizationErrorException$ = [-3, n0, _AEE,
    { [_aQE]: [`AuthorizationError`, 403], [_e]: _c, [_hE]: 403 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(AuthorizationErrorException$, AuthorizationErrorException);
var BatchEntryIdsNotDistinctException$ = [-3, n0, _BEINDE,
    { [_aQE]: [`BatchEntryIdsNotDistinct`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(BatchEntryIdsNotDistinctException$, BatchEntryIdsNotDistinctException);
var BatchRequestTooLongException$ = [-3, n0, _BRTLE,
    { [_aQE]: [`BatchRequestTooLong`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(BatchRequestTooLongException$, BatchRequestTooLongException);
var BatchResultErrorEntry$ = [3, n0, _BREE,
    0,
    [_I, _C, _M, _SF],
    [0, 0, 0, 2]
];
var CheckIfPhoneNumberIsOptedOutInput$ = [3, n0, _CIPNIOOI,
    0,
    [_pN],
    [[() => PhoneNumber, 0]]
];
var CheckIfPhoneNumberIsOptedOutResponse$ = [3, n0, _CIPNIOOR,
    0,
    [_iOO],
    [2]
];
var ConcurrentAccessException$ = [-3, n0, _CAE,
    { [_aQE]: [`ConcurrentAccess`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ConcurrentAccessException$, ConcurrentAccessException);
var ConfirmSubscriptionInput$ = [3, n0, _CSI,
    0,
    [_TA, _T, _AOU],
    [0, 0, 0]
];
var ConfirmSubscriptionResponse$ = [3, n0, _CSR,
    0,
    [_SA],
    [0]
];
var CreateEndpointResponse$ = [3, n0, _CER,
    0,
    [_EA],
    [0]
];
var CreatePlatformApplicationInput$ = [3, n0, _CPAI,
    0,
    [_N, _P, _A],
    [0, 0, 128 | 0]
];
var CreatePlatformApplicationResponse$ = [3, n0, _CPAR,
    0,
    [_PAA],
    [0]
];
var CreatePlatformEndpointInput$ = [3, n0, _CPEI,
    0,
    [_PAA, _T, _CUD, _A],
    [0, 0, 0, 128 | 0]
];
var CreateSMSSandboxPhoneNumberInput$ = [3, n0, _CSMSSPNI,
    0,
    [_PN, _LC],
    [[() => PhoneNumberString, 0], 0]
];
var CreateSMSSandboxPhoneNumberResult$ = [3, n0, _CSMSSPNR,
    0,
    [],
    []
];
var CreateTopicInput$ = [3, n0, _CTI,
    0,
    [_N, _A, _Ta, _DPP],
    [0, 128 | 0, () => TagList, 0]
];
var CreateTopicResponse$ = [3, n0, _CTR,
    0,
    [_TA],
    [0]
];
var DeleteEndpointInput$ = [3, n0, _DEI,
    0,
    [_EA],
    [0]
];
var DeletePlatformApplicationInput$ = [3, n0, _DPAI,
    0,
    [_PAA],
    [0]
];
var DeleteSMSSandboxPhoneNumberInput$ = [3, n0, _DSMSSPNI,
    0,
    [_PN],
    [[() => PhoneNumberString, 0]]
];
var DeleteSMSSandboxPhoneNumberResult$ = [3, n0, _DSMSSPNR,
    0,
    [],
    []
];
var DeleteTopicInput$ = [3, n0, _DTI,
    0,
    [_TA],
    [0]
];
var EmptyBatchRequestException$ = [-3, n0, _EBRE,
    { [_aQE]: [`EmptyBatchRequest`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(EmptyBatchRequestException$, EmptyBatchRequestException);
var Endpoint$ = [3, n0, _E,
    0,
    [_EA, _A],
    [0, 128 | 0]
];
var EndpointDisabledException$ = [-3, n0, _EDE,
    { [_aQE]: [`EndpointDisabled`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(EndpointDisabledException$, EndpointDisabledException);
var FilterPolicyLimitExceededException$ = [-3, n0, _FPLEE,
    { [_aQE]: [`FilterPolicyLimitExceeded`, 403], [_e]: _c, [_hE]: 403 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(FilterPolicyLimitExceededException$, FilterPolicyLimitExceededException);
var GetDataProtectionPolicyInput$ = [3, n0, _GDPPI,
    0,
    [_RA],
    [0]
];
var GetDataProtectionPolicyResponse$ = [3, n0, _GDPPR,
    0,
    [_DPP],
    [0]
];
var GetEndpointAttributesInput$ = [3, n0, _GEAI,
    0,
    [_EA],
    [0]
];
var GetEndpointAttributesResponse$ = [3, n0, _GEAR,
    0,
    [_A],
    [128 | 0]
];
var GetPlatformApplicationAttributesInput$ = [3, n0, _GPAAI,
    0,
    [_PAA],
    [0]
];
var GetPlatformApplicationAttributesResponse$ = [3, n0, _GPAAR,
    0,
    [_A],
    [128 | 0]
];
var GetSMSAttributesInput$ = [3, n0, _GSMSAI,
    0,
    [_a],
    [64 | 0]
];
var GetSMSAttributesResponse$ = [3, n0, _GSMSAR,
    0,
    [_a],
    [128 | 0]
];
var GetSMSSandboxAccountStatusInput$ = [3, n0, _GSMSSASI,
    0,
    [],
    []
];
var GetSMSSandboxAccountStatusResult$ = [3, n0, _GSMSSASR,
    0,
    [_IIS],
    [2]
];
var GetSubscriptionAttributesInput$ = [3, n0, _GSAI,
    0,
    [_SA],
    [0]
];
var GetSubscriptionAttributesResponse$ = [3, n0, _GSAR,
    0,
    [_A],
    [128 | 0]
];
var GetTopicAttributesInput$ = [3, n0, _GTAI,
    0,
    [_TA],
    [0]
];
var GetTopicAttributesResponse$ = [3, n0, _GTAR,
    0,
    [_A],
    [128 | 0]
];
var InternalErrorException$ = [-3, n0, _IEE,
    { [_aQE]: [`InternalError`, 500], [_e]: _s, [_hE]: 500 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InternalErrorException$, InternalErrorException);
var InvalidBatchEntryIdException$ = [-3, n0, _IBEIE,
    { [_aQE]: [`InvalidBatchEntryId`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidBatchEntryIdException$, InvalidBatchEntryIdException);
var InvalidParameterException$ = [-3, n0, _IPE,
    { [_aQE]: [`InvalidParameter`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidParameterException$, InvalidParameterException);
var InvalidParameterValueException$ = [-3, n0, _IPVE,
    { [_aQE]: [`ParameterValueInvalid`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidParameterValueException$, InvalidParameterValueException);
var InvalidSecurityException$ = [-3, n0, _ISE,
    { [_aQE]: [`InvalidSecurity`, 403], [_e]: _c, [_hE]: 403 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidSecurityException$, InvalidSecurityException);
var InvalidStateException$ = [-3, n0, _ISEn,
    { [_aQE]: [`InvalidState`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidStateException$, InvalidStateException);
var KMSAccessDeniedException$ = [-3, n0, _KMSADE,
    { [_aQE]: [`KMSAccessDenied`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(KMSAccessDeniedException$, KMSAccessDeniedException);
var KMSDisabledException$ = [-3, n0, _KMSDE,
    { [_aQE]: [`KMSDisabled`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(KMSDisabledException$, KMSDisabledException);
var KMSInvalidStateException$ = [-3, n0, _KMSISE,
    { [_aQE]: [`KMSInvalidState`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(KMSInvalidStateException$, KMSInvalidStateException);
var KMSNotFoundException$ = [-3, n0, _KMSNFE,
    { [_aQE]: [`KMSNotFound`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(KMSNotFoundException$, KMSNotFoundException);
var KMSOptInRequired$ = [-3, n0, _KMSOIR,
    { [_aQE]: [`KMSOptInRequired`, 403], [_e]: _c, [_hE]: 403 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(KMSOptInRequired$, KMSOptInRequired);
var KMSThrottlingException$ = [-3, n0, _KMSTE,
    { [_aQE]: [`KMSThrottling`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(KMSThrottlingException$, KMSThrottlingException);
var ListEndpointsByPlatformApplicationInput$ = [3, n0, _LEBPAI,
    0,
    [_PAA, _NT],
    [0, 0]
];
var ListEndpointsByPlatformApplicationResponse$ = [3, n0, _LEBPAR,
    0,
    [_En, _NT],
    [() => ListOfEndpoints, 0]
];
var ListOriginationNumbersRequest$ = [3, n0, _LONR,
    0,
    [_NT, _MR],
    [0, 1]
];
var ListOriginationNumbersResult$ = [3, n0, _LONRi,
    0,
    [_NT, _PNh],
    [0, [() => PhoneNumberInformationList, 0]]
];
var ListPhoneNumbersOptedOutInput$ = [3, n0, _LPNOOI,
    0,
    [_nT],
    [0]
];
var ListPhoneNumbersOptedOutResponse$ = [3, n0, _LPNOOR,
    0,
    [_pNh, _nT],
    [[() => PhoneNumberList, 0], 0]
];
var ListPlatformApplicationsInput$ = [3, n0, _LPAI,
    0,
    [_NT],
    [0]
];
var ListPlatformApplicationsResponse$ = [3, n0, _LPAR,
    0,
    [_PA, _NT],
    [() => ListOfPlatformApplications, 0]
];
var ListSMSSandboxPhoneNumbersInput$ = [3, n0, _LSMSSPNI,
    0,
    [_NT, _MR],
    [0, 1]
];
var ListSMSSandboxPhoneNumbersResult$ = [3, n0, _LSMSSPNR,
    0,
    [_PNh, _NT],
    [[() => SMSSandboxPhoneNumberList, 0], 0]
];
var ListSubscriptionsByTopicInput$ = [3, n0, _LSBTI,
    0,
    [_TA, _NT],
    [0, 0]
];
var ListSubscriptionsByTopicResponse$ = [3, n0, _LSBTR,
    0,
    [_S, _NT],
    [() => SubscriptionsList, 0]
];
var ListSubscriptionsInput$ = [3, n0, _LSI,
    0,
    [_NT],
    [0]
];
var ListSubscriptionsResponse$ = [3, n0, _LSR,
    0,
    [_S, _NT],
    [() => SubscriptionsList, 0]
];
var ListTagsForResourceRequest$ = [3, n0, _LTFRR,
    0,
    [_RA],
    [0]
];
var ListTagsForResourceResponse$ = [3, n0, _LTFRRi,
    0,
    [_Ta],
    [() => TagList]
];
var ListTopicsInput$ = [3, n0, _LTI,
    0,
    [_NT],
    [0]
];
var ListTopicsResponse$ = [3, n0, _LTR,
    0,
    [_To, _NT],
    [() => TopicsList, 0]
];
var MessageAttributeValue$ = [3, n0, _MAV,
    0,
    [_DT, _SV, _BV],
    [0, 0, 21]
];
var NotFoundException$ = [-3, n0, _NFE,
    { [_aQE]: [`NotFound`, 404], [_e]: _c, [_hE]: 404 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NotFoundException$, NotFoundException);
var OptedOutException$ = [-3, n0, _OOE,
    { [_aQE]: [`OptedOut`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(OptedOutException$, OptedOutException);
var OptInPhoneNumberInput$ = [3, n0, _OIPNI,
    0,
    [_pN],
    [[() => PhoneNumber, 0]]
];
var OptInPhoneNumberResponse$ = [3, n0, _OIPNR,
    0,
    [],
    []
];
var PhoneNumberInformation$ = [3, n0, _PNI,
    0,
    [_CA, _PN, _St, _ICC, _RT, _NC],
    [4, [() => PhoneNumber, 0], 0, 0, 0, 64 | 0]
];
var PlatformApplication$ = [3, n0, _PAl,
    0,
    [_PAA, _A],
    [0, 128 | 0]
];
var PlatformApplicationDisabledException$ = [-3, n0, _PADE,
    { [_aQE]: [`PlatformApplicationDisabled`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(PlatformApplicationDisabledException$, PlatformApplicationDisabledException);
var PublishBatchInput$ = [3, n0, _PBI,
    0,
    [_TA, _PBRE],
    [0, [() => PublishBatchRequestEntryList, 0]]
];
var PublishBatchRequestEntry$ = [3, n0, _PBREu,
    0,
    [_I, _M, _Su, _MS, _MA, _MDI, _MGI],
    [0, 0, 0, 0, [() => MessageAttributeMap, 0], 0, 0]
];
var PublishBatchResponse$ = [3, n0, _PBR,
    0,
    [_Suc, _F],
    [() => PublishBatchResultEntryList, () => BatchResultErrorEntryList]
];
var PublishBatchResultEntry$ = [3, n0, _PBREub,
    0,
    [_I, _MI, _SN],
    [0, 0, 0]
];
var PublishInput$ = [3, n0, _PI,
    0,
    [_TA, _TAa, _PN, _M, _Su, _MS, _MA, _MDI, _MGI],
    [0, 0, [() => PhoneNumber, 0], 0, 0, 0, [() => MessageAttributeMap, 0], 0, 0]
];
var PublishResponse$ = [3, n0, _PR,
    0,
    [_MI, _SN],
    [0, 0]
];
var PutDataProtectionPolicyInput$ = [3, n0, _PDPPI,
    0,
    [_RA, _DPP],
    [0, 0]
];
var RemovePermissionInput$ = [3, n0, _RPI,
    0,
    [_TA, _L],
    [0, 0]
];
var ReplayLimitExceededException$ = [-3, n0, _RLEE,
    { [_aQE]: [`ReplayLimitExceeded`, 403], [_e]: _c, [_hE]: 403 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ReplayLimitExceededException$, ReplayLimitExceededException);
var ResourceNotFoundException$ = [-3, n0, _RNFE,
    { [_aQE]: [`ResourceNotFound`, 404], [_e]: _c, [_hE]: 404 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ResourceNotFoundException$, ResourceNotFoundException);
var SetEndpointAttributesInput$ = [3, n0, _SEAI,
    0,
    [_EA, _A],
    [0, 128 | 0]
];
var SetPlatformApplicationAttributesInput$ = [3, n0, _SPAAI,
    0,
    [_PAA, _A],
    [0, 128 | 0]
];
var SetSMSAttributesInput$ = [3, n0, _SSMSAI,
    0,
    [_a],
    [128 | 0]
];
var SetSMSAttributesResponse$ = [3, n0, _SSMSAR,
    0,
    [],
    []
];
var SetSubscriptionAttributesInput$ = [3, n0, _SSAI,
    0,
    [_SA, _ANt, _AV],
    [0, 0, 0]
];
var SetTopicAttributesInput$ = [3, n0, _STAI,
    0,
    [_TA, _ANt, _AV],
    [0, 0, 0]
];
var SMSSandboxPhoneNumber$ = [3, n0, _SMSSPN,
    0,
    [_PN, _St],
    [[() => PhoneNumberString, 0], 0]
];
var StaleTagException$ = [-3, n0, _STE,
    { [_aQE]: [`StaleTag`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(StaleTagException$, StaleTagException);
var SubscribeInput$ = [3, n0, _SI,
    0,
    [_TA, _Pr, _E, _A, _RSA],
    [0, 0, 0, 128 | 0, 2]
];
var SubscribeResponse$ = [3, n0, _SR,
    0,
    [_SA],
    [0]
];
var Subscription$ = [3, n0, _Sub,
    0,
    [_SA, _O, _Pr, _E, _TA],
    [0, 0, 0, 0, 0]
];
var SubscriptionLimitExceededException$ = [-3, n0, _SLEE,
    { [_aQE]: [`SubscriptionLimitExceeded`, 403], [_e]: _c, [_hE]: 403 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(SubscriptionLimitExceededException$, SubscriptionLimitExceededException);
var Tag$ = [3, n0, _Tag,
    0,
    [_K, _V],
    [0, 0]
];
var TagLimitExceededException$ = [-3, n0, _TLEE,
    { [_aQE]: [`TagLimitExceeded`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TagLimitExceededException$, TagLimitExceededException);
var TagPolicyException$ = [-3, n0, _TPE,
    { [_aQE]: [`TagPolicy`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TagPolicyException$, TagPolicyException);
var TagResourceRequest$ = [3, n0, _TRR,
    0,
    [_RA, _Ta],
    [0, () => TagList]
];
var TagResourceResponse$ = [3, n0, _TRRa,
    0,
    [],
    []
];
var ThrottledException$ = [-3, n0, _TE,
    { [_aQE]: [`Throttled`, 429], [_e]: _c, [_hE]: 429 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ThrottledException$, ThrottledException);
var TooManyEntriesInBatchRequestException$ = [-3, n0, _TMEIBRE,
    { [_aQE]: [`TooManyEntriesInBatchRequest`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyEntriesInBatchRequestException$, TooManyEntriesInBatchRequestException);
var Topic$ = [3, n0, _Top,
    0,
    [_TA],
    [0]
];
var TopicLimitExceededException$ = [-3, n0, _TLEEo,
    { [_aQE]: [`TopicLimitExceeded`, 403], [_e]: _c, [_hE]: 403 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TopicLimitExceededException$, TopicLimitExceededException);
var UnsubscribeInput$ = [3, n0, _UI,
    0,
    [_SA],
    [0]
];
var UntagResourceRequest$ = [3, n0, _URR,
    0,
    [_RA, _TK],
    [0, 64 | 0]
];
var UntagResourceResponse$ = [3, n0, _URRn,
    0,
    [],
    []
];
var UserErrorException$ = [-3, n0, _UEE,
    { [_aQE]: [`UserError`, 400], [_e]: _c, [_hE]: 400 },
    [_m],
    [0]
];
schema.TypeRegistry.for(n0).registerError(UserErrorException$, UserErrorException);
var ValidationException$ = [-3, n0, _VE,
    { [_aQE]: [`ValidationException`, 400], [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ValidationException$, ValidationException);
var VerificationException$ = [-3, n0, _VEe,
    { [_e]: _c },
    [_M, _St],
    [0, 0]
];
schema.TypeRegistry.for(n0).registerError(VerificationException$, VerificationException);
var VerifySMSSandboxPhoneNumberInput$ = [3, n0, _VSMSSPNI,
    0,
    [_PN, _OTP],
    [[() => PhoneNumberString, 0], 0]
];
var VerifySMSSandboxPhoneNumberResult$ = [3, n0, _VSMSSPNR,
    0,
    [],
    []
];
var __Unit = "unit";
var SNSServiceException$ = [-3, _sm, "SNSServiceException", 0, [], []];
schema.TypeRegistry.for(_sm).registerError(SNSServiceException$, SNSServiceException);
var BatchResultErrorEntryList = [1, n0, _BREEL,
    0, () => BatchResultErrorEntry$
];
var ListOfEndpoints = [1, n0, _LOE,
    0, () => Endpoint$
];
var ListOfPlatformApplications = [1, n0, _LOPA,
    0, () => PlatformApplication$
];
var PhoneNumberInformationList = [1, n0, _PNIL,
    0, [() => PhoneNumberInformation$,
        0]
];
var PhoneNumberList = [1, n0, _PNL,
    0, [() => PhoneNumber,
        0]
];
var PublishBatchRequestEntryList = [1, n0, _PBREL,
    0, [() => PublishBatchRequestEntry$,
        0]
];
var PublishBatchResultEntryList = [1, n0, _PBRELu,
    0, () => PublishBatchResultEntry$
];
var SMSSandboxPhoneNumberList = [1, n0, _SMSSPNL,
    0, [() => SMSSandboxPhoneNumber$,
        0]
];
var SubscriptionsList = [1, n0, _SL,
    0, () => Subscription$
];
var TagList = [1, n0, _TL,
    0, () => Tag$
];
var TopicsList = [1, n0, _TLo,
    0, () => Topic$
];
var MessageAttributeMap = [2, n0, _MAM,
    0, [0,
        { [_xN]: _N }],
    [() => MessageAttributeValue$,
        { [_xN]: _V }]
];
var AddPermission$ = [9, n0, _AP,
    0, () => AddPermissionInput$, () => __Unit
];
var CheckIfPhoneNumberIsOptedOut$ = [9, n0, _CIPNIOO,
    0, () => CheckIfPhoneNumberIsOptedOutInput$, () => CheckIfPhoneNumberIsOptedOutResponse$
];
var ConfirmSubscription$ = [9, n0, _CS,
    0, () => ConfirmSubscriptionInput$, () => ConfirmSubscriptionResponse$
];
var CreatePlatformApplication$ = [9, n0, _CPA,
    0, () => CreatePlatformApplicationInput$, () => CreatePlatformApplicationResponse$
];
var CreatePlatformEndpoint$ = [9, n0, _CPE,
    0, () => CreatePlatformEndpointInput$, () => CreateEndpointResponse$
];
var CreateSMSSandboxPhoneNumber$ = [9, n0, _CSMSSPN,
    0, () => CreateSMSSandboxPhoneNumberInput$, () => CreateSMSSandboxPhoneNumberResult$
];
var CreateTopic$ = [9, n0, _CT,
    0, () => CreateTopicInput$, () => CreateTopicResponse$
];
var DeleteEndpoint$ = [9, n0, _DE,
    0, () => DeleteEndpointInput$, () => __Unit
];
var DeletePlatformApplication$ = [9, n0, _DPA,
    0, () => DeletePlatformApplicationInput$, () => __Unit
];
var DeleteSMSSandboxPhoneNumber$ = [9, n0, _DSMSSPN,
    0, () => DeleteSMSSandboxPhoneNumberInput$, () => DeleteSMSSandboxPhoneNumberResult$
];
var DeleteTopic$ = [9, n0, _DTe,
    0, () => DeleteTopicInput$, () => __Unit
];
var GetDataProtectionPolicy$ = [9, n0, _GDPP,
    0, () => GetDataProtectionPolicyInput$, () => GetDataProtectionPolicyResponse$
];
var GetEndpointAttributes$ = [9, n0, _GEA,
    0, () => GetEndpointAttributesInput$, () => GetEndpointAttributesResponse$
];
var GetPlatformApplicationAttributes$ = [9, n0, _GPAA,
    0, () => GetPlatformApplicationAttributesInput$, () => GetPlatformApplicationAttributesResponse$
];
var GetSMSAttributes$ = [9, n0, _GSMSA,
    0, () => GetSMSAttributesInput$, () => GetSMSAttributesResponse$
];
var GetSMSSandboxAccountStatus$ = [9, n0, _GSMSSAS,
    0, () => GetSMSSandboxAccountStatusInput$, () => GetSMSSandboxAccountStatusResult$
];
var GetSubscriptionAttributes$ = [9, n0, _GSA,
    0, () => GetSubscriptionAttributesInput$, () => GetSubscriptionAttributesResponse$
];
var GetTopicAttributes$ = [9, n0, _GTA,
    0, () => GetTopicAttributesInput$, () => GetTopicAttributesResponse$
];
var ListEndpointsByPlatformApplication$ = [9, n0, _LEBPA,
    0, () => ListEndpointsByPlatformApplicationInput$, () => ListEndpointsByPlatformApplicationResponse$
];
var ListOriginationNumbers$ = [9, n0, _LON,
    0, () => ListOriginationNumbersRequest$, () => ListOriginationNumbersResult$
];
var ListPhoneNumbersOptedOut$ = [9, n0, _LPNOO,
    0, () => ListPhoneNumbersOptedOutInput$, () => ListPhoneNumbersOptedOutResponse$
];
var ListPlatformApplications$ = [9, n0, _LPA,
    0, () => ListPlatformApplicationsInput$, () => ListPlatformApplicationsResponse$
];
var ListSMSSandboxPhoneNumbers$ = [9, n0, _LSMSSPN,
    0, () => ListSMSSandboxPhoneNumbersInput$, () => ListSMSSandboxPhoneNumbersResult$
];
var ListSubscriptions$ = [9, n0, _LS,
    0, () => ListSubscriptionsInput$, () => ListSubscriptionsResponse$
];
var ListSubscriptionsByTopic$ = [9, n0, _LSBT,
    0, () => ListSubscriptionsByTopicInput$, () => ListSubscriptionsByTopicResponse$
];
var ListTagsForResource$ = [9, n0, _LTFR,
    0, () => ListTagsForResourceRequest$, () => ListTagsForResourceResponse$
];
var ListTopics$ = [9, n0, _LT,
    0, () => ListTopicsInput$, () => ListTopicsResponse$
];
var OptInPhoneNumber$ = [9, n0, _OIPN,
    0, () => OptInPhoneNumberInput$, () => OptInPhoneNumberResponse$
];
var Publish$ = [9, n0, _Pu,
    0, () => PublishInput$, () => PublishResponse$
];
var PublishBatch$ = [9, n0, _PB,
    0, () => PublishBatchInput$, () => PublishBatchResponse$
];
var PutDataProtectionPolicy$ = [9, n0, _PDPP,
    0, () => PutDataProtectionPolicyInput$, () => __Unit
];
var RemovePermission$ = [9, n0, _RP,
    0, () => RemovePermissionInput$, () => __Unit
];
var SetEndpointAttributes$ = [9, n0, _SEA,
    0, () => SetEndpointAttributesInput$, () => __Unit
];
var SetPlatformApplicationAttributes$ = [9, n0, _SPAA,
    0, () => SetPlatformApplicationAttributesInput$, () => __Unit
];
var SetSMSAttributes$ = [9, n0, _SSMSA,
    0, () => SetSMSAttributesInput$, () => SetSMSAttributesResponse$
];
var SetSubscriptionAttributes$ = [9, n0, _SSA,
    0, () => SetSubscriptionAttributesInput$, () => __Unit
];
var SetTopicAttributes$ = [9, n0, _STA,
    0, () => SetTopicAttributesInput$, () => __Unit
];
var Subscribe$ = [9, n0, _Subs,
    0, () => SubscribeInput$, () => SubscribeResponse$
];
var TagResource$ = [9, n0, _TR,
    0, () => TagResourceRequest$, () => TagResourceResponse$
];
var Unsubscribe$ = [9, n0, _U,
    0, () => UnsubscribeInput$, () => __Unit
];
var UntagResource$ = [9, n0, _UR,
    0, () => UntagResourceRequest$, () => UntagResourceResponse$
];
var VerifySMSSandboxPhoneNumber$ = [9, n0, _VSMSSPN,
    0, () => VerifySMSSandboxPhoneNumberInput$, () => VerifySMSSandboxPhoneNumberResult$
];

class AddPermissionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "AddPermission", {})
    .n("SNSClient", "AddPermissionCommand")
    .sc(AddPermission$)
    .build() {
}

class CheckIfPhoneNumberIsOptedOutCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "CheckIfPhoneNumberIsOptedOut", {})
    .n("SNSClient", "CheckIfPhoneNumberIsOptedOutCommand")
    .sc(CheckIfPhoneNumberIsOptedOut$)
    .build() {
}

class ConfirmSubscriptionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "ConfirmSubscription", {})
    .n("SNSClient", "ConfirmSubscriptionCommand")
    .sc(ConfirmSubscription$)
    .build() {
}

class CreatePlatformApplicationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "CreatePlatformApplication", {})
    .n("SNSClient", "CreatePlatformApplicationCommand")
    .sc(CreatePlatformApplication$)
    .build() {
}

class CreatePlatformEndpointCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "CreatePlatformEndpoint", {})
    .n("SNSClient", "CreatePlatformEndpointCommand")
    .sc(CreatePlatformEndpoint$)
    .build() {
}

class CreateSMSSandboxPhoneNumberCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "CreateSMSSandboxPhoneNumber", {})
    .n("SNSClient", "CreateSMSSandboxPhoneNumberCommand")
    .sc(CreateSMSSandboxPhoneNumber$)
    .build() {
}

class CreateTopicCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "CreateTopic", {})
    .n("SNSClient", "CreateTopicCommand")
    .sc(CreateTopic$)
    .build() {
}

class DeleteEndpointCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "DeleteEndpoint", {})
    .n("SNSClient", "DeleteEndpointCommand")
    .sc(DeleteEndpoint$)
    .build() {
}

class DeletePlatformApplicationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "DeletePlatformApplication", {})
    .n("SNSClient", "DeletePlatformApplicationCommand")
    .sc(DeletePlatformApplication$)
    .build() {
}

class DeleteSMSSandboxPhoneNumberCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "DeleteSMSSandboxPhoneNumber", {})
    .n("SNSClient", "DeleteSMSSandboxPhoneNumberCommand")
    .sc(DeleteSMSSandboxPhoneNumber$)
    .build() {
}

class DeleteTopicCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "DeleteTopic", {})
    .n("SNSClient", "DeleteTopicCommand")
    .sc(DeleteTopic$)
    .build() {
}

class GetDataProtectionPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "GetDataProtectionPolicy", {})
    .n("SNSClient", "GetDataProtectionPolicyCommand")
    .sc(GetDataProtectionPolicy$)
    .build() {
}

class GetEndpointAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "GetEndpointAttributes", {})
    .n("SNSClient", "GetEndpointAttributesCommand")
    .sc(GetEndpointAttributes$)
    .build() {
}

class GetPlatformApplicationAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "GetPlatformApplicationAttributes", {})
    .n("SNSClient", "GetPlatformApplicationAttributesCommand")
    .sc(GetPlatformApplicationAttributes$)
    .build() {
}

class GetSMSAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "GetSMSAttributes", {})
    .n("SNSClient", "GetSMSAttributesCommand")
    .sc(GetSMSAttributes$)
    .build() {
}

class GetSMSSandboxAccountStatusCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "GetSMSSandboxAccountStatus", {})
    .n("SNSClient", "GetSMSSandboxAccountStatusCommand")
    .sc(GetSMSSandboxAccountStatus$)
    .build() {
}

class GetSubscriptionAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "GetSubscriptionAttributes", {})
    .n("SNSClient", "GetSubscriptionAttributesCommand")
    .sc(GetSubscriptionAttributes$)
    .build() {
}

class GetTopicAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "GetTopicAttributes", {})
    .n("SNSClient", "GetTopicAttributesCommand")
    .sc(GetTopicAttributes$)
    .build() {
}

class ListEndpointsByPlatformApplicationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "ListEndpointsByPlatformApplication", {})
    .n("SNSClient", "ListEndpointsByPlatformApplicationCommand")
    .sc(ListEndpointsByPlatformApplication$)
    .build() {
}

class ListOriginationNumbersCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "ListOriginationNumbers", {})
    .n("SNSClient", "ListOriginationNumbersCommand")
    .sc(ListOriginationNumbers$)
    .build() {
}

class ListPhoneNumbersOptedOutCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "ListPhoneNumbersOptedOut", {})
    .n("SNSClient", "ListPhoneNumbersOptedOutCommand")
    .sc(ListPhoneNumbersOptedOut$)
    .build() {
}

class ListPlatformApplicationsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "ListPlatformApplications", {})
    .n("SNSClient", "ListPlatformApplicationsCommand")
    .sc(ListPlatformApplications$)
    .build() {
}

class ListSMSSandboxPhoneNumbersCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "ListSMSSandboxPhoneNumbers", {})
    .n("SNSClient", "ListSMSSandboxPhoneNumbersCommand")
    .sc(ListSMSSandboxPhoneNumbers$)
    .build() {
}

class ListSubscriptionsByTopicCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "ListSubscriptionsByTopic", {})
    .n("SNSClient", "ListSubscriptionsByTopicCommand")
    .sc(ListSubscriptionsByTopic$)
    .build() {
}

class ListSubscriptionsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "ListSubscriptions", {})
    .n("SNSClient", "ListSubscriptionsCommand")
    .sc(ListSubscriptions$)
    .build() {
}

class ListTagsForResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "ListTagsForResource", {})
    .n("SNSClient", "ListTagsForResourceCommand")
    .sc(ListTagsForResource$)
    .build() {
}

class ListTopicsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "ListTopics", {})
    .n("SNSClient", "ListTopicsCommand")
    .sc(ListTopics$)
    .build() {
}

class OptInPhoneNumberCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "OptInPhoneNumber", {})
    .n("SNSClient", "OptInPhoneNumberCommand")
    .sc(OptInPhoneNumber$)
    .build() {
}

class PublishBatchCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "PublishBatch", {})
    .n("SNSClient", "PublishBatchCommand")
    .sc(PublishBatch$)
    .build() {
}

class PublishCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "Publish", {})
    .n("SNSClient", "PublishCommand")
    .sc(Publish$)
    .build() {
}

class PutDataProtectionPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "PutDataProtectionPolicy", {})
    .n("SNSClient", "PutDataProtectionPolicyCommand")
    .sc(PutDataProtectionPolicy$)
    .build() {
}

class RemovePermissionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "RemovePermission", {})
    .n("SNSClient", "RemovePermissionCommand")
    .sc(RemovePermission$)
    .build() {
}

class SetEndpointAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "SetEndpointAttributes", {})
    .n("SNSClient", "SetEndpointAttributesCommand")
    .sc(SetEndpointAttributes$)
    .build() {
}

class SetPlatformApplicationAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "SetPlatformApplicationAttributes", {})
    .n("SNSClient", "SetPlatformApplicationAttributesCommand")
    .sc(SetPlatformApplicationAttributes$)
    .build() {
}

class SetSMSAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "SetSMSAttributes", {})
    .n("SNSClient", "SetSMSAttributesCommand")
    .sc(SetSMSAttributes$)
    .build() {
}

class SetSubscriptionAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "SetSubscriptionAttributes", {})
    .n("SNSClient", "SetSubscriptionAttributesCommand")
    .sc(SetSubscriptionAttributes$)
    .build() {
}

class SetTopicAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "SetTopicAttributes", {})
    .n("SNSClient", "SetTopicAttributesCommand")
    .sc(SetTopicAttributes$)
    .build() {
}

class SubscribeCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "Subscribe", {})
    .n("SNSClient", "SubscribeCommand")
    .sc(Subscribe$)
    .build() {
}

class TagResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "TagResource", {})
    .n("SNSClient", "TagResourceCommand")
    .sc(TagResource$)
    .build() {
}

class UnsubscribeCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "Unsubscribe", {})
    .n("SNSClient", "UnsubscribeCommand")
    .sc(Unsubscribe$)
    .build() {
}

class UntagResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "UntagResource", {})
    .n("SNSClient", "UntagResourceCommand")
    .sc(UntagResource$)
    .build() {
}

class VerifySMSSandboxPhoneNumberCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("AmazonSimpleNotificationService", "VerifySMSSandboxPhoneNumber", {})
    .n("SNSClient", "VerifySMSSandboxPhoneNumberCommand")
    .sc(VerifySMSSandboxPhoneNumber$)
    .build() {
}

const commands = {
    AddPermissionCommand,
    CheckIfPhoneNumberIsOptedOutCommand,
    ConfirmSubscriptionCommand,
    CreatePlatformApplicationCommand,
    CreatePlatformEndpointCommand,
    CreateSMSSandboxPhoneNumberCommand,
    CreateTopicCommand,
    DeleteEndpointCommand,
    DeletePlatformApplicationCommand,
    DeleteSMSSandboxPhoneNumberCommand,
    DeleteTopicCommand,
    GetDataProtectionPolicyCommand,
    GetEndpointAttributesCommand,
    GetPlatformApplicationAttributesCommand,
    GetSMSAttributesCommand,
    GetSMSSandboxAccountStatusCommand,
    GetSubscriptionAttributesCommand,
    GetTopicAttributesCommand,
    ListEndpointsByPlatformApplicationCommand,
    ListOriginationNumbersCommand,
    ListPhoneNumbersOptedOutCommand,
    ListPlatformApplicationsCommand,
    ListSMSSandboxPhoneNumbersCommand,
    ListSubscriptionsCommand,
    ListSubscriptionsByTopicCommand,
    ListTagsForResourceCommand,
    ListTopicsCommand,
    OptInPhoneNumberCommand,
    PublishCommand,
    PublishBatchCommand,
    PutDataProtectionPolicyCommand,
    RemovePermissionCommand,
    SetEndpointAttributesCommand,
    SetPlatformApplicationAttributesCommand,
    SetSMSAttributesCommand,
    SetSubscriptionAttributesCommand,
    SetTopicAttributesCommand,
    SubscribeCommand,
    TagResourceCommand,
    UnsubscribeCommand,
    UntagResourceCommand,
    VerifySMSSandboxPhoneNumberCommand,
};
class SNS extends SNSClient {
}
smithyClient.createAggregatedClient(commands, SNS);

const paginateListEndpointsByPlatformApplication = core.createPaginator(SNSClient, ListEndpointsByPlatformApplicationCommand, "NextToken", "NextToken", "");

const paginateListOriginationNumbers = core.createPaginator(SNSClient, ListOriginationNumbersCommand, "NextToken", "NextToken", "MaxResults");

const paginateListPhoneNumbersOptedOut = core.createPaginator(SNSClient, ListPhoneNumbersOptedOutCommand, "nextToken", "nextToken", "");

const paginateListPlatformApplications = core.createPaginator(SNSClient, ListPlatformApplicationsCommand, "NextToken", "NextToken", "");

const paginateListSMSSandboxPhoneNumbers = core.createPaginator(SNSClient, ListSMSSandboxPhoneNumbersCommand, "NextToken", "NextToken", "MaxResults");

const paginateListSubscriptions = core.createPaginator(SNSClient, ListSubscriptionsCommand, "NextToken", "NextToken", "");

const paginateListSubscriptionsByTopic = core.createPaginator(SNSClient, ListSubscriptionsByTopicCommand, "NextToken", "NextToken", "");

const paginateListTopics = core.createPaginator(SNSClient, ListTopicsCommand, "NextToken", "NextToken", "");

const LanguageCodeString = {
    de_DE: "de-DE",
    en_GB: "en-GB",
    en_US: "en-US",
    es_419: "es-419",
    es_ES: "es-ES",
    fr_CA: "fr-CA",
    fr_FR: "fr-FR",
    it_IT: "it-IT",
    jp_JP: "ja-JP",
    kr_KR: "kr-KR",
    pt_BR: "pt-BR",
    zh_CN: "zh-CN",
    zh_TW: "zh-TW",
};
const NumberCapability = {
    MMS: "MMS",
    SMS: "SMS",
    VOICE: "VOICE",
};
const RouteType = {
    Premium: "Premium",
    Promotional: "Promotional",
    Transactional: "Transactional",
};
const SMSSandboxPhoneNumberVerificationStatus = {
    Pending: "Pending",
    Verified: "Verified",
};

Object.defineProperty(exports, "$Command", {
    enumerable: true,
    get: function () { return smithyClient.Command; }
});
Object.defineProperty(exports, "__Client", {
    enumerable: true,
    get: function () { return smithyClient.Client; }
});
exports.AddPermission$ = AddPermission$;
exports.AddPermissionCommand = AddPermissionCommand;
exports.AddPermissionInput$ = AddPermissionInput$;
exports.AuthorizationErrorException = AuthorizationErrorException;
exports.AuthorizationErrorException$ = AuthorizationErrorException$;
exports.BatchEntryIdsNotDistinctException = BatchEntryIdsNotDistinctException;
exports.BatchEntryIdsNotDistinctException$ = BatchEntryIdsNotDistinctException$;
exports.BatchRequestTooLongException = BatchRequestTooLongException;
exports.BatchRequestTooLongException$ = BatchRequestTooLongException$;
exports.BatchResultErrorEntry$ = BatchResultErrorEntry$;
exports.CheckIfPhoneNumberIsOptedOut$ = CheckIfPhoneNumberIsOptedOut$;
exports.CheckIfPhoneNumberIsOptedOutCommand = CheckIfPhoneNumberIsOptedOutCommand;
exports.CheckIfPhoneNumberIsOptedOutInput$ = CheckIfPhoneNumberIsOptedOutInput$;
exports.CheckIfPhoneNumberIsOptedOutResponse$ = CheckIfPhoneNumberIsOptedOutResponse$;
exports.ConcurrentAccessException = ConcurrentAccessException;
exports.ConcurrentAccessException$ = ConcurrentAccessException$;
exports.ConfirmSubscription$ = ConfirmSubscription$;
exports.ConfirmSubscriptionCommand = ConfirmSubscriptionCommand;
exports.ConfirmSubscriptionInput$ = ConfirmSubscriptionInput$;
exports.ConfirmSubscriptionResponse$ = ConfirmSubscriptionResponse$;
exports.CreateEndpointResponse$ = CreateEndpointResponse$;
exports.CreatePlatformApplication$ = CreatePlatformApplication$;
exports.CreatePlatformApplicationCommand = CreatePlatformApplicationCommand;
exports.CreatePlatformApplicationInput$ = CreatePlatformApplicationInput$;
exports.CreatePlatformApplicationResponse$ = CreatePlatformApplicationResponse$;
exports.CreatePlatformEndpoint$ = CreatePlatformEndpoint$;
exports.CreatePlatformEndpointCommand = CreatePlatformEndpointCommand;
exports.CreatePlatformEndpointInput$ = CreatePlatformEndpointInput$;
exports.CreateSMSSandboxPhoneNumber$ = CreateSMSSandboxPhoneNumber$;
exports.CreateSMSSandboxPhoneNumberCommand = CreateSMSSandboxPhoneNumberCommand;
exports.CreateSMSSandboxPhoneNumberInput$ = CreateSMSSandboxPhoneNumberInput$;
exports.CreateSMSSandboxPhoneNumberResult$ = CreateSMSSandboxPhoneNumberResult$;
exports.CreateTopic$ = CreateTopic$;
exports.CreateTopicCommand = CreateTopicCommand;
exports.CreateTopicInput$ = CreateTopicInput$;
exports.CreateTopicResponse$ = CreateTopicResponse$;
exports.DeleteEndpoint$ = DeleteEndpoint$;
exports.DeleteEndpointCommand = DeleteEndpointCommand;
exports.DeleteEndpointInput$ = DeleteEndpointInput$;
exports.DeletePlatformApplication$ = DeletePlatformApplication$;
exports.DeletePlatformApplicationCommand = DeletePlatformApplicationCommand;
exports.DeletePlatformApplicationInput$ = DeletePlatformApplicationInput$;
exports.DeleteSMSSandboxPhoneNumber$ = DeleteSMSSandboxPhoneNumber$;
exports.DeleteSMSSandboxPhoneNumberCommand = DeleteSMSSandboxPhoneNumberCommand;
exports.DeleteSMSSandboxPhoneNumberInput$ = DeleteSMSSandboxPhoneNumberInput$;
exports.DeleteSMSSandboxPhoneNumberResult$ = DeleteSMSSandboxPhoneNumberResult$;
exports.DeleteTopic$ = DeleteTopic$;
exports.DeleteTopicCommand = DeleteTopicCommand;
exports.DeleteTopicInput$ = DeleteTopicInput$;
exports.EmptyBatchRequestException = EmptyBatchRequestException;
exports.EmptyBatchRequestException$ = EmptyBatchRequestException$;
exports.Endpoint$ = Endpoint$;
exports.EndpointDisabledException = EndpointDisabledException;
exports.EndpointDisabledException$ = EndpointDisabledException$;
exports.FilterPolicyLimitExceededException = FilterPolicyLimitExceededException;
exports.FilterPolicyLimitExceededException$ = FilterPolicyLimitExceededException$;
exports.GetDataProtectionPolicy$ = GetDataProtectionPolicy$;
exports.GetDataProtectionPolicyCommand = GetDataProtectionPolicyCommand;
exports.GetDataProtectionPolicyInput$ = GetDataProtectionPolicyInput$;
exports.GetDataProtectionPolicyResponse$ = GetDataProtectionPolicyResponse$;
exports.GetEndpointAttributes$ = GetEndpointAttributes$;
exports.GetEndpointAttributesCommand = GetEndpointAttributesCommand;
exports.GetEndpointAttributesInput$ = GetEndpointAttributesInput$;
exports.GetEndpointAttributesResponse$ = GetEndpointAttributesResponse$;
exports.GetPlatformApplicationAttributes$ = GetPlatformApplicationAttributes$;
exports.GetPlatformApplicationAttributesCommand = GetPlatformApplicationAttributesCommand;
exports.GetPlatformApplicationAttributesInput$ = GetPlatformApplicationAttributesInput$;
exports.GetPlatformApplicationAttributesResponse$ = GetPlatformApplicationAttributesResponse$;
exports.GetSMSAttributes$ = GetSMSAttributes$;
exports.GetSMSAttributesCommand = GetSMSAttributesCommand;
exports.GetSMSAttributesInput$ = GetSMSAttributesInput$;
exports.GetSMSAttributesResponse$ = GetSMSAttributesResponse$;
exports.GetSMSSandboxAccountStatus$ = GetSMSSandboxAccountStatus$;
exports.GetSMSSandboxAccountStatusCommand = GetSMSSandboxAccountStatusCommand;
exports.GetSMSSandboxAccountStatusInput$ = GetSMSSandboxAccountStatusInput$;
exports.GetSMSSandboxAccountStatusResult$ = GetSMSSandboxAccountStatusResult$;
exports.GetSubscriptionAttributes$ = GetSubscriptionAttributes$;
exports.GetSubscriptionAttributesCommand = GetSubscriptionAttributesCommand;
exports.GetSubscriptionAttributesInput$ = GetSubscriptionAttributesInput$;
exports.GetSubscriptionAttributesResponse$ = GetSubscriptionAttributesResponse$;
exports.GetTopicAttributes$ = GetTopicAttributes$;
exports.GetTopicAttributesCommand = GetTopicAttributesCommand;
exports.GetTopicAttributesInput$ = GetTopicAttributesInput$;
exports.GetTopicAttributesResponse$ = GetTopicAttributesResponse$;
exports.InternalErrorException = InternalErrorException;
exports.InternalErrorException$ = InternalErrorException$;
exports.InvalidBatchEntryIdException = InvalidBatchEntryIdException;
exports.InvalidBatchEntryIdException$ = InvalidBatchEntryIdException$;
exports.InvalidParameterException = InvalidParameterException;
exports.InvalidParameterException$ = InvalidParameterException$;
exports.InvalidParameterValueException = InvalidParameterValueException;
exports.InvalidParameterValueException$ = InvalidParameterValueException$;
exports.InvalidSecurityException = InvalidSecurityException;
exports.InvalidSecurityException$ = InvalidSecurityException$;
exports.InvalidStateException = InvalidStateException;
exports.InvalidStateException$ = InvalidStateException$;
exports.KMSAccessDeniedException = KMSAccessDeniedException;
exports.KMSAccessDeniedException$ = KMSAccessDeniedException$;
exports.KMSDisabledException = KMSDisabledException;
exports.KMSDisabledException$ = KMSDisabledException$;
exports.KMSInvalidStateException = KMSInvalidStateException;
exports.KMSInvalidStateException$ = KMSInvalidStateException$;
exports.KMSNotFoundException = KMSNotFoundException;
exports.KMSNotFoundException$ = KMSNotFoundException$;
exports.KMSOptInRequired = KMSOptInRequired;
exports.KMSOptInRequired$ = KMSOptInRequired$;
exports.KMSThrottlingException = KMSThrottlingException;
exports.KMSThrottlingException$ = KMSThrottlingException$;
exports.LanguageCodeString = LanguageCodeString;
exports.ListEndpointsByPlatformApplication$ = ListEndpointsByPlatformApplication$;
exports.ListEndpointsByPlatformApplicationCommand = ListEndpointsByPlatformApplicationCommand;
exports.ListEndpointsByPlatformApplicationInput$ = ListEndpointsByPlatformApplicationInput$;
exports.ListEndpointsByPlatformApplicationResponse$ = ListEndpointsByPlatformApplicationResponse$;
exports.ListOriginationNumbers$ = ListOriginationNumbers$;
exports.ListOriginationNumbersCommand = ListOriginationNumbersCommand;
exports.ListOriginationNumbersRequest$ = ListOriginationNumbersRequest$;
exports.ListOriginationNumbersResult$ = ListOriginationNumbersResult$;
exports.ListPhoneNumbersOptedOut$ = ListPhoneNumbersOptedOut$;
exports.ListPhoneNumbersOptedOutCommand = ListPhoneNumbersOptedOutCommand;
exports.ListPhoneNumbersOptedOutInput$ = ListPhoneNumbersOptedOutInput$;
exports.ListPhoneNumbersOptedOutResponse$ = ListPhoneNumbersOptedOutResponse$;
exports.ListPlatformApplications$ = ListPlatformApplications$;
exports.ListPlatformApplicationsCommand = ListPlatformApplicationsCommand;
exports.ListPlatformApplicationsInput$ = ListPlatformApplicationsInput$;
exports.ListPlatformApplicationsResponse$ = ListPlatformApplicationsResponse$;
exports.ListSMSSandboxPhoneNumbers$ = ListSMSSandboxPhoneNumbers$;
exports.ListSMSSandboxPhoneNumbersCommand = ListSMSSandboxPhoneNumbersCommand;
exports.ListSMSSandboxPhoneNumbersInput$ = ListSMSSandboxPhoneNumbersInput$;
exports.ListSMSSandboxPhoneNumbersResult$ = ListSMSSandboxPhoneNumbersResult$;
exports.ListSubscriptions$ = ListSubscriptions$;
exports.ListSubscriptionsByTopic$ = ListSubscriptionsByTopic$;
exports.ListSubscriptionsByTopicCommand = ListSubscriptionsByTopicCommand;
exports.ListSubscriptionsByTopicInput$ = ListSubscriptionsByTopicInput$;
exports.ListSubscriptionsByTopicResponse$ = ListSubscriptionsByTopicResponse$;
exports.ListSubscriptionsCommand = ListSubscriptionsCommand;
exports.ListSubscriptionsInput$ = ListSubscriptionsInput$;
exports.ListSubscriptionsResponse$ = ListSubscriptionsResponse$;
exports.ListTagsForResource$ = ListTagsForResource$;
exports.ListTagsForResourceCommand = ListTagsForResourceCommand;
exports.ListTagsForResourceRequest$ = ListTagsForResourceRequest$;
exports.ListTagsForResourceResponse$ = ListTagsForResourceResponse$;
exports.ListTopics$ = ListTopics$;
exports.ListTopicsCommand = ListTopicsCommand;
exports.ListTopicsInput$ = ListTopicsInput$;
exports.ListTopicsResponse$ = ListTopicsResponse$;
exports.MessageAttributeValue$ = MessageAttributeValue$;
exports.NotFoundException = NotFoundException;
exports.NotFoundException$ = NotFoundException$;
exports.NumberCapability = NumberCapability;
exports.OptInPhoneNumber$ = OptInPhoneNumber$;
exports.OptInPhoneNumberCommand = OptInPhoneNumberCommand;
exports.OptInPhoneNumberInput$ = OptInPhoneNumberInput$;
exports.OptInPhoneNumberResponse$ = OptInPhoneNumberResponse$;
exports.OptedOutException = OptedOutException;
exports.OptedOutException$ = OptedOutException$;
exports.PhoneNumberInformation$ = PhoneNumberInformation$;
exports.PlatformApplication$ = PlatformApplication$;
exports.PlatformApplicationDisabledException = PlatformApplicationDisabledException;
exports.PlatformApplicationDisabledException$ = PlatformApplicationDisabledException$;
exports.Publish$ = Publish$;
exports.PublishBatch$ = PublishBatch$;
exports.PublishBatchCommand = PublishBatchCommand;
exports.PublishBatchInput$ = PublishBatchInput$;
exports.PublishBatchRequestEntry$ = PublishBatchRequestEntry$;
exports.PublishBatchResponse$ = PublishBatchResponse$;
exports.PublishBatchResultEntry$ = PublishBatchResultEntry$;
exports.PublishCommand = PublishCommand;
exports.PublishInput$ = PublishInput$;
exports.PublishResponse$ = PublishResponse$;
exports.PutDataProtectionPolicy$ = PutDataProtectionPolicy$;
exports.PutDataProtectionPolicyCommand = PutDataProtectionPolicyCommand;
exports.PutDataProtectionPolicyInput$ = PutDataProtectionPolicyInput$;
exports.RemovePermission$ = RemovePermission$;
exports.RemovePermissionCommand = RemovePermissionCommand;
exports.RemovePermissionInput$ = RemovePermissionInput$;
exports.ReplayLimitExceededException = ReplayLimitExceededException;
exports.ReplayLimitExceededException$ = ReplayLimitExceededException$;
exports.ResourceNotFoundException = ResourceNotFoundException;
exports.ResourceNotFoundException$ = ResourceNotFoundException$;
exports.RouteType = RouteType;
exports.SMSSandboxPhoneNumber$ = SMSSandboxPhoneNumber$;
exports.SMSSandboxPhoneNumberVerificationStatus = SMSSandboxPhoneNumberVerificationStatus;
exports.SNS = SNS;
exports.SNSClient = SNSClient;
exports.SNSServiceException = SNSServiceException;
exports.SNSServiceException$ = SNSServiceException$;
exports.SetEndpointAttributes$ = SetEndpointAttributes$;
exports.SetEndpointAttributesCommand = SetEndpointAttributesCommand;
exports.SetEndpointAttributesInput$ = SetEndpointAttributesInput$;
exports.SetPlatformApplicationAttributes$ = SetPlatformApplicationAttributes$;
exports.SetPlatformApplicationAttributesCommand = SetPlatformApplicationAttributesCommand;
exports.SetPlatformApplicationAttributesInput$ = SetPlatformApplicationAttributesInput$;
exports.SetSMSAttributes$ = SetSMSAttributes$;
exports.SetSMSAttributesCommand = SetSMSAttributesCommand;
exports.SetSMSAttributesInput$ = SetSMSAttributesInput$;
exports.SetSMSAttributesResponse$ = SetSMSAttributesResponse$;
exports.SetSubscriptionAttributes$ = SetSubscriptionAttributes$;
exports.SetSubscriptionAttributesCommand = SetSubscriptionAttributesCommand;
exports.SetSubscriptionAttributesInput$ = SetSubscriptionAttributesInput$;
exports.SetTopicAttributes$ = SetTopicAttributes$;
exports.SetTopicAttributesCommand = SetTopicAttributesCommand;
exports.SetTopicAttributesInput$ = SetTopicAttributesInput$;
exports.StaleTagException = StaleTagException;
exports.StaleTagException$ = StaleTagException$;
exports.Subscribe$ = Subscribe$;
exports.SubscribeCommand = SubscribeCommand;
exports.SubscribeInput$ = SubscribeInput$;
exports.SubscribeResponse$ = SubscribeResponse$;
exports.Subscription$ = Subscription$;
exports.SubscriptionLimitExceededException = SubscriptionLimitExceededException;
exports.SubscriptionLimitExceededException$ = SubscriptionLimitExceededException$;
exports.Tag$ = Tag$;
exports.TagLimitExceededException = TagLimitExceededException;
exports.TagLimitExceededException$ = TagLimitExceededException$;
exports.TagPolicyException = TagPolicyException;
exports.TagPolicyException$ = TagPolicyException$;
exports.TagResource$ = TagResource$;
exports.TagResourceCommand = TagResourceCommand;
exports.TagResourceRequest$ = TagResourceRequest$;
exports.TagResourceResponse$ = TagResourceResponse$;
exports.ThrottledException = ThrottledException;
exports.ThrottledException$ = ThrottledException$;
exports.TooManyEntriesInBatchRequestException = TooManyEntriesInBatchRequestException;
exports.TooManyEntriesInBatchRequestException$ = TooManyEntriesInBatchRequestException$;
exports.Topic$ = Topic$;
exports.TopicLimitExceededException = TopicLimitExceededException;
exports.TopicLimitExceededException$ = TopicLimitExceededException$;
exports.Unsubscribe$ = Unsubscribe$;
exports.UnsubscribeCommand = UnsubscribeCommand;
exports.UnsubscribeInput$ = UnsubscribeInput$;
exports.UntagResource$ = UntagResource$;
exports.UntagResourceCommand = UntagResourceCommand;
exports.UntagResourceRequest$ = UntagResourceRequest$;
exports.UntagResourceResponse$ = UntagResourceResponse$;
exports.UserErrorException = UserErrorException;
exports.UserErrorException$ = UserErrorException$;
exports.ValidationException = ValidationException;
exports.ValidationException$ = ValidationException$;
exports.VerificationException = VerificationException;
exports.VerificationException$ = VerificationException$;
exports.VerifySMSSandboxPhoneNumber$ = VerifySMSSandboxPhoneNumber$;
exports.VerifySMSSandboxPhoneNumberCommand = VerifySMSSandboxPhoneNumberCommand;
exports.VerifySMSSandboxPhoneNumberInput$ = VerifySMSSandboxPhoneNumberInput$;
exports.VerifySMSSandboxPhoneNumberResult$ = VerifySMSSandboxPhoneNumberResult$;
exports.paginateListEndpointsByPlatformApplication = paginateListEndpointsByPlatformApplication;
exports.paginateListOriginationNumbers = paginateListOriginationNumbers;
exports.paginateListPhoneNumbersOptedOut = paginateListPhoneNumbersOptedOut;
exports.paginateListPlatformApplications = paginateListPlatformApplications;
exports.paginateListSMSSandboxPhoneNumbers = paginateListSMSSandboxPhoneNumbers;
exports.paginateListSubscriptions = paginateListSubscriptions;
exports.paginateListSubscriptionsByTopic = paginateListSubscriptionsByTopic;
exports.paginateListTopics = paginateListTopics;
