"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OperationRowsIterator = exports.OperationChunksIterator = void 0;
class OperationIterator {
    constructor(operation, options) {
        this.operation = operation;
        this.options = options;
    }
    [Symbol.asyncIterator]() {
        return this;
    }
    async next() {
        var _a;
        const result = await this.getNext();
        if (result.done && ((_a = this.options) === null || _a === void 0 ? void 0 : _a.autoClose)) {
            await this.operation.close();
        }
        return result;
    }
    // This method is intended for a cleanup when the caller does not intend to make any more
    // reads from iterator (e.g. when using `break` in a `for ... of` loop)
    async return(value) {
        var _a;
        if ((_a = this.options) === null || _a === void 0 ? void 0 : _a.autoClose) {
            await this.operation.close();
        }
        return { done: true, value };
    }
}
class OperationChunksIterator extends OperationIterator {
    async getNext() {
        const hasMoreRows = await this.operation.hasMoreRows();
        if (hasMoreRows) {
            const value = await this.operation.fetchChunk(this.options);
            return { done: false, value };
        }
        return { done: true, value: undefined };
    }
}
exports.OperationChunksIterator = OperationChunksIterator;
class OperationRowsIterator extends OperationIterator {
    constructor(operation, options) {
        super(operation, {
            ...options,
            // Tell slicer to return raw chunks. We're going to process rows one by one anyway,
            // so no need to additionally buffer and slice chunks returned by server
            disableBuffering: true,
        });
        this.chunk = [];
        this.index = 0;
    }
    async getNext() {
        if (this.index < this.chunk.length) {
            const value = this.chunk[this.index];
            this.index += 1;
            return { done: false, value };
        }
        const hasMoreRows = await this.operation.hasMoreRows();
        if (hasMoreRows) {
            this.chunk = await this.operation.fetchChunk(this.options);
            this.index = 0;
            // Note: this call is not really a recursion. Since this method is
            // async - the call will be actually scheduled for processing on
            // the next event loop cycle
            return this.getNext();
        }
        return { done: true, value: undefined };
    }
}
exports.OperationRowsIterator = OperationRowsIterator;
//# sourceMappingURL=OperationIterator.js.map