"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.supportsCloudFetch = exports.supportsMultipleCatalogs = exports.supportsArrowMetadata = exports.supportsArrowCompression = exports.supportsResultPersistenceMode = exports.supportsAsyncMetadataOperations = exports.supportsParameterizedQueries = exports.isFeatureSupported = void 0;
const TCLIService_types_1 = require("../../thrift/TCLIService_types");
/**
 * Protocol version information from Thrift TCLIService
 * Each version adds certain features to the Spark/Hive API
 *
 * Databricks only supports SPARK_CLI_SERVICE_PROTOCOL_V1 (0xA501) or higher
 */
/**
 * Check if the current protocol version supports a specific feature
 * @param serverProtocolVersion The protocol version received from server in TOpenSessionResp
 * @param requiredVersion The minimum protocol version required for a feature
 * @returns boolean indicating if the feature is supported
 */
function isFeatureSupported(serverProtocolVersion, requiredVersion) {
    if (serverProtocolVersion === undefined || serverProtocolVersion === null) {
        return false;
    }
    return serverProtocolVersion >= requiredVersion;
}
exports.isFeatureSupported = isFeatureSupported;
/**
 * Check if parameterized queries are supported
 * (Requires SPARK_CLI_SERVICE_PROTOCOL_V8 or higher)
 * @param serverProtocolVersion The protocol version from server
 * @returns boolean indicating if parameterized queries are supported
 */
function supportsParameterizedQueries(serverProtocolVersion) {
    return isFeatureSupported(serverProtocolVersion, TCLIService_types_1.TProtocolVersion.SPARK_CLI_SERVICE_PROTOCOL_V8);
}
exports.supportsParameterizedQueries = supportsParameterizedQueries;
/**
 * Check if async metadata operations are supported
 * (Requires SPARK_CLI_SERVICE_PROTOCOL_V6 or higher)
 * @param serverProtocolVersion The protocol version from server
 * @returns boolean indicating if async metadata operations are supported
 */
function supportsAsyncMetadataOperations(serverProtocolVersion) {
    return isFeatureSupported(serverProtocolVersion, TCLIService_types_1.TProtocolVersion.SPARK_CLI_SERVICE_PROTOCOL_V6);
}
exports.supportsAsyncMetadataOperations = supportsAsyncMetadataOperations;
/**
 * Check if result persistence mode is supported
 * (Requires SPARK_CLI_SERVICE_PROTOCOL_V7 or higher)
 * @param serverProtocolVersion The protocol version from server
 * @returns boolean indicating if result persistence mode is supported
 */
function supportsResultPersistenceMode(serverProtocolVersion) {
    return isFeatureSupported(serverProtocolVersion, TCLIService_types_1.TProtocolVersion.SPARK_CLI_SERVICE_PROTOCOL_V7);
}
exports.supportsResultPersistenceMode = supportsResultPersistenceMode;
/**
 * Check if Arrow compression is supported
 * (Requires SPARK_CLI_SERVICE_PROTOCOL_V6 or higher)
 * @param serverProtocolVersion The protocol version from server
 * @returns boolean indicating if compressed Arrow batches are supported
 */
function supportsArrowCompression(serverProtocolVersion) {
    return isFeatureSupported(serverProtocolVersion, TCLIService_types_1.TProtocolVersion.SPARK_CLI_SERVICE_PROTOCOL_V6);
}
exports.supportsArrowCompression = supportsArrowCompression;
/**
 * Check if Arrow metadata is supported
 * (Requires SPARK_CLI_SERVICE_PROTOCOL_V5 or higher)
 * @param serverProtocolVersion The protocol version from server
 * @returns boolean indicating if Arrow metadata is supported
 */
function supportsArrowMetadata(serverProtocolVersion) {
    return isFeatureSupported(serverProtocolVersion, TCLIService_types_1.TProtocolVersion.SPARK_CLI_SERVICE_PROTOCOL_V5);
}
exports.supportsArrowMetadata = supportsArrowMetadata;
/**
 * Check if multiple catalogs are supported
 * (Requires SPARK_CLI_SERVICE_PROTOCOL_V4 or higher)
 * @param serverProtocolVersion The protocol version from server
 * @returns boolean indicating if multiple catalogs are supported
 */
function supportsMultipleCatalogs(serverProtocolVersion) {
    return isFeatureSupported(serverProtocolVersion, TCLIService_types_1.TProtocolVersion.SPARK_CLI_SERVICE_PROTOCOL_V4);
}
exports.supportsMultipleCatalogs = supportsMultipleCatalogs;
/**
 * Check if cloud object storage fetching is supported
 * (Requires SPARK_CLI_SERVICE_PROTOCOL_V3 or higher)
 * @param serverProtocolVersion The protocol version from server
 * @returns boolean indicating if cloud fetching is supported
 */
function supportsCloudFetch(serverProtocolVersion) {
    return isFeatureSupported(serverProtocolVersion, TCLIService_types_1.TProtocolVersion.SPARK_CLI_SERVICE_PROTOCOL_V3);
}
exports.supportsCloudFetch = supportsCloudFetch;
//# sourceMappingURL=protocolVersion.js.map