import { BaseResource, BaseResourceOptions } from '@gitbeaker/requester-utils';

interface UserAgentDetailSchema extends Record<string, unknown> {
    user_agent: string;
    ip_address: string;
    akismet_submitted: boolean;
}
type CamelizeString<T extends PropertyKey> = T extends string ? string extends T ? string : T extends `${infer F}_${infer R}` ? `${F}${Capitalize<CamelizeString<R>>}` : T : T;
type Camelize<T> = {
    [K in keyof T as CamelizeString<K>]: Camelize<T[K]>;
};
type Simplify<T> = T extends infer S ? {
    [K in keyof S]: S[K];
} : never;
type Never<T> = Simplify<{
    [P in keyof T]?: never;
}>;
type SomeOf<T> = {
    [K in keyof T]: Pick<Required<T>, K>;
}[keyof T];
type OneOf<T> = {
    [K in keyof T]: Simplify<Pick<T, K> & Never<Omit<T, K>>>;
}[keyof T];
type OneOrNoneOf<T> = Never<T> | OneOf<T>;
type AllOrNone<T extends Record<string, any>> = T | Partial<Record<keyof T, never>>;
type MappedOmit<T, K extends keyof T> = {
    [P in keyof T as P extends K ? never : P]: T[P];
};
/**
 * A special class to mark path segments that should not be URL-encoded.
 * Used for paths that already contain special characters like '/' that should be preserved.
 * For example, in API paths like 'repository/commits', the slash should not be encoded to '%2F'.
 */
declare class RawPathSegment {
    readonly value: string;
    constructor(value: string);
    toString(): string;
}

interface IsForm {
    isForm?: boolean;
}
interface Sudo {
    sudo?: string | number;
}
interface AsStream {
    asStream?: boolean;
}
interface ShowExpanded<E extends boolean = false> {
    showExpanded?: E;
}
interface AsAdmin<A extends boolean = false> {
    asAdmin?: A;
}
type BaseRequestOptions<E extends boolean = false> = Sudo & ShowExpanded<E> & {
    [Key in string]?: any;
};
type PaginationTypes = 'keyset' | 'offset';
interface KeysetPaginationRequestOptions {
    orderBy: string;
    sort: 'asc' | 'desc';
}
interface OffsetPaginationRequestOptions {
    page?: number | string;
}
interface BasePaginationRequestOptions<P extends PaginationTypes | void> {
    pagination?: P;
    perPage?: number | string;
    maxPages?: number;
}
type PaginationRequestSubOptions<P extends PaginationTypes | void> = P extends 'keyset' ? AllOrNone<KeysetPaginationRequestOptions> : P extends 'offset' ? OffsetPaginationRequestOptions : AllOrNone<KeysetPaginationRequestOptions> & OffsetPaginationRequestOptions;
type PaginationRequestOptions<P extends PaginationTypes | void = void> = BasePaginationRequestOptions<P> & PaginationRequestSubOptions<P>;
type CamelizedResponse<T, C> = C extends true ? Camelize<T> : T;
interface OffsetPagination {
    total: number;
    next: number | null;
    current: number;
    previous: number | null;
    perPage: number;
    totalPages: number;
}
interface KeysetPagination {
    idAfter: number;
    perPage: number;
    orderBy: string;
    sort: 'asc' | 'desc';
}
interface ExpandedResponse<T> {
    data: T;
    headers: Record<string, string>;
    status: number;
}
type PaginatedResponse<T, P extends PaginationTypes = PaginationTypes> = {
    [U in P]: {
        paginationInfo: P extends 'keyset' ? KeysetPagination : OffsetPagination;
        data: T;
    };
}[P];
type GitlabAPIExpandedResponse<T, E extends boolean | void, P> = E extends true ? P extends PaginationTypes ? PaginatedResponse<T, P> : ExpandedResponse<T> : T;
type GitlabAPISingleResponse<T, C extends boolean | void, E extends boolean | void> = T extends Record<string, unknown> ? GitlabAPIExpandedResponse<CamelizedResponse<T, C>, E, undefined> : GitlabAPIExpandedResponse<T, E, undefined>;
type GitlabAPIMultiResponse<T, C extends boolean | void, E extends boolean | void, P extends PaginationTypes | void> = T extends Record<string, unknown> ? GitlabAPIExpandedResponse<CamelizedResponse<T, C>[], E, P> : GitlabAPIExpandedResponse<T[], E, P>;
type GitlabAPIResponse<T, C extends boolean | void, E extends boolean | void, P extends PaginationTypes | void> = T extends (infer R)[] ? GitlabAPIMultiResponse<R, C, E, P> : GitlabAPISingleResponse<T, C, E>;

interface ProjectRemoteMirrorSchema extends Record<string, unknown> {
    enabled: boolean;
    id: number;
    last_error?: string;
    last_successful_update_at: string;
    last_update_at: string;
    last_update_started_at: string;
    only_protected_branches: boolean;
    keep_divergent_refs: boolean;
    update_status: string;
    url: string;
}
declare class ProjectRemoteMirrors<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<ProjectRemoteMirrorSchema[], C, E, P>>;
    createPullMirror<E extends boolean = false>(projectId: string | number, url: string, mirror: boolean, options?: {
        onlyProtectedBranches?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectRemoteMirrorSchema, C, E, void>>;
    createPushMirror<E extends boolean = false>(projectId: string | number, url: string, options?: {
        enabled?: boolean;
        onlyProtectedBranches?: boolean;
        keepDivergentRefs?: boolean;
        mirrorBranchRegex?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectRemoteMirrorSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, mirrorId: number, options?: {
        enabled?: boolean;
        onlyProtectedBranches?: boolean;
        keepDivergentRefs?: boolean;
        mirrorBranchRegex?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectRemoteMirrorSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, mirrorId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, mirrorId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectRemoteMirrorSchema, C, E, void>>;
    sync<E extends boolean = false>(projectId: string | number, mirrorId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

type AllEventOptions = {
    action?: 'created' | 'updated' | 'closed' | 'reopened' | 'pushed' | 'commented' | 'merged' | 'joined' | 'left' | 'destroyed' | 'expired';
    targetType?: 'issue' | 'milestone' | 'merge_request' | 'note' | 'project' | 'snippet' | 'user';
    before?: string;
    after?: string;
    scope?: string;
    sort?: 'asc' | 'desc';
};
interface EventSchema extends Record<string, unknown> {
    id: number;
    title?: string;
    project_id: number;
    action_name: string;
    target_id: number;
    target_type: string;
    author_id: number;
    target_title: string;
    created_at: string;
    author: MappedOmit<SimpleUserSchema, 'created_at'>;
    author_username: string;
}
declare class Events<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>({ projectId, userId, ...options }?: OneOrNoneOf<{
        projectId?: string | number;
        userId: string | number;
    }> & AllEventOptions & PaginationRequestOptions<P> & BaseRequestOptions<E>): Promise<GitlabAPIResponse<EventSchema[], C, E, P>>;
}

interface PersonalAccessTokenSchema extends Record<string, unknown> {
    id: number;
    name: string;
    revoked: boolean;
    created_at: string;
    scopes?: string[];
    user_id: number;
    last_used_at: string;
    active: boolean;
    expires_at?: string;
    token: string;
}
type PersonalAccessTokenScopes = 'api' | 'read_api' | 'read_user' | 'create_runner' | 'read_repository' | 'write_repository' | 'read_registry' | 'write_registry' | 'sudo' | 'admin_mode';
type AllPersonalAccessTokenOptions = {
    userId?: string | number;
    state?: 'active' | 'inactive';
    search?: string;
    revoked?: boolean;
    lastUsedBefore?: string;
    lastUsedAfter?: string;
    createdBefore?: string;
    createdAfter?: string;
};
declare class PersonalAccessTokens<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: AllPersonalAccessTokenOptions & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<PersonalAccessTokenSchema[], C, E, P>>;
    create<E extends boolean = false>(userId: number, name: string, scopes: string[], options?: {
        expiresAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PersonalAccessTokenSchema, C, E, void>>;
    remove<E extends boolean = false>({ tokenId, ...options }?: {
        tokenId?: string | number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    rotate<E extends boolean = false>(tokenId: number | 'self', options?: {
        expiresAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PersonalAccessTokenSchema, C, E, void>>;
    show<E extends boolean = false>({ tokenId, ...options }?: {
        tokenId?: string | number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PersonalAccessTokenSchema, C, E, void>>;
}

interface CustomAttributeSchema extends Record<string, unknown> {
    key: string;
    value: string;
}
declare class ResourceCustomAttributes<C extends boolean = false> extends BaseResource<C> {
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CustomAttributeSchema[], C, E, P>>;
    remove<E extends boolean = false>(resourceId: string | number, customAttributeId: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    set<E extends boolean = false>(resourceId: string | number, customAttributeId: string, value: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CustomAttributeSchema, C, E, void>>;
    show<E extends boolean = false>(resourceId: string | number, customAttributeId: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CustomAttributeSchema, C, E, void>>;
}

declare enum AccessLevel {
    NO_ACCESS = 0,
    MINIMAL_ACCESS = 5,
    GUEST = 10,
    REPORTER = 20,
    DEVELOPER = 30,
    MAINTAINER = 40,
    OWNER = 50,
    ADMIN = 60
}

interface SimpleUserSchema extends Record<string, unknown> {
    id: number;
    name: string;
    username: string;
    state: string;
    avatar_url: string;
    web_url: string;
    created_at: string;
}
interface UserSchema extends SimpleUserSchema {
    locked: boolean | null;
    bio: string | null;
    bot: boolean | null;
    location: string | null;
    public_email: string | null;
    skype: string | null;
    linkedin: string | null;
    twitter: string | null;
    discord: string | null;
    website_url: string | null;
    pronouns: string | null;
    organization: string | null;
    job_title: string | null;
    work_information: string | null;
    followers: number | null;
    following: number | null;
    local_time: string | null;
    is_followed: boolean | null;
}
interface ExpandedUserSchema extends UserSchema {
    is_admin: boolean | null;
    bot: boolean;
    last_sign_in_at: string;
    confirmed_at: string;
    last_activity_on: string;
    email: string;
    theme_id: number;
    color_scheme_id: number;
    projects_limit: number;
    current_sign_in_at: string | null;
    note: string | null;
    identities: {
        provider: string;
        extern_uid: string;
        saml_provider_id: number;
    }[] | null;
    can_create_group: boolean;
    can_create_project: boolean;
    two_factor_enabled: boolean;
    external: boolean;
    private_profile: string | null;
    namespace_id: number | null;
    created_by: string | null;
}
interface AdminUserSchema extends ExpandedUserSchema {
    current_sign_in_ip: string;
    last_sign_in_ip: string;
    using_license_seat: boolean | null;
    email_reset_offered_at: string | null;
    shared_runners_minutes_limit?: number | null;
    extra_shared_runners_minutes_limit?: number | null;
    is_auditor?: boolean | null;
    provisioned_by_group_id?: number | null;
    plan?: string;
    trial?: boolean;
}
interface UserActivitySchema extends Record<string, unknown> {
    username: string;
    last_activity_on: string;
    last_activity_at: string;
}
interface UserStatusSchema extends Record<string, unknown> {
    emoji: string;
    availability: string;
    message: string;
    message_html: string;
    clear_status_at: string;
}
interface UserPreferenceSchema extends Record<string, unknown> {
    id: number;
    user_id: number;
    view_diffs_file_by_file: boolean;
    show_whitespace_in_diffs: boolean;
}
interface UserCountSchema extends Record<string, unknown> {
    merge_requests: number;
    assigned_issues: number;
    assigned_merge_requests: number;
    review_requested_merge_requests: number;
    todos: number;
}
interface UserAssociationCountSchema extends Record<string, unknown> {
    groups_count: number;
    projects_count: number;
    issues_count: number;
    merge_requests_count: number;
}
interface UserMembershipSchema extends Record<string, unknown> {
    source_id: number;
    source_name: string;
    source_type: 'Project' | 'Namespace';
    access_level: Exclude<AccessLevel, AccessLevel.ADMIN>;
}
interface UserRunnerSchema extends Record<string, unknown> {
    id: number;
    token: string;
    token_expires_at: string | null;
}
type AllUsersOptions = {
    orderBy?: 'name' | 'username' | 'created_at' | 'updated_at';
    createdBy?: string;
    sort?: 'asc' | 'desc';
    twoFactor?: string;
    withoutProjects?: boolean;
    admins?: boolean;
    samlProviderId?: number;
    skipLdap?: boolean;
    search?: string;
    username?: string;
    active?: boolean;
    blocked?: boolean;
    external?: boolean;
    excludeInternal?: boolean;
    excludeExternal?: boolean;
    withoutProjectBots?: boolean;
    createdBefore?: string;
    createdAfter?: string;
    withCustomAttributes?: boolean;
    customAttributes?: Record<string, string>;
} & AllOrNone<{
    provider: string;
    externUid: string;
}>;
type CreateUserOptions = {
    admin?: boolean;
    auditor?: boolean;
    avatar?: {
        content: Blob;
        filename?: string;
    };
    bio?: string;
    canCreateGroup?: boolean;
    colorSchemeId?: number;
    commitEmail?: string;
    email?: string;
    externUid?: string;
    external?: boolean;
    extraSharedRunnersMinutesLimit?: number;
    forceRandomPassword?: boolean;
    groupIdForSaml?: number;
    linkedin?: string;
    location?: string;
    name?: string;
    note?: string;
    organization?: string;
    password?: string;
    privateProfile?: string;
    projectsLimit?: number;
    pronouns?: string;
    provider?: string;
    publicEmail?: string;
    resetPassword?: boolean;
    sharedRunnersMinutesLimit?: number;
    skipConfirmation?: boolean;
    skype?: string;
    themeId?: number;
    twitter?: string;
    discord?: string;
    username?: string;
    viewDiffsFileByFile?: boolean;
    websiteUrl?: string;
};
type EditUserOptions = CreateUserOptions;
type CreateUserCIRunnerOptions = {
    groupId?: number;
    projectId?: number;
    description?: string;
    paused?: boolean;
    locked?: boolean;
    runUntagged?: boolean;
    tagList?: string[];
    accessLevel?: 'not_protected' | 'ref_protected';
    maximumTimeout?: number;
    maintenanceNote?: string;
};
type AllUserProjectsOptions = {
    archived?: boolean;
    idAfter?: number;
    idBefore?: number;
    membership?: boolean;
    minAccessLevel?: Exclude<AccessLevel, AccessLevel.ADMIN>;
    orderBy?: 'id' | 'name' | 'path' | 'created_at' | 'updated_at' | 'last_activity_at';
    owned?: boolean;
    search?: string;
    simple?: boolean;
    sort?: 'asc' | 'desc';
    starred?: boolean;
    statistics?: boolean;
    visibility?: 'public' | 'internal' | 'private';
    withCustomAttributes?: boolean;
    withIssuesEnabled?: boolean;
    withMergeRequestsEnabled?: boolean;
    withProgrammingLanguage?: string;
    updatedBefore?: string;
    updatedAfter?: string;
};
declare class Users<C extends boolean = false> extends BaseResource<C> {
    activate<E extends boolean = false>(userId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    all<A extends boolean = false, E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: {
        withCustomAttributes: true;
    } & AsAdmin<A> & AllUsersOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<((A extends false ? SimpleUserSchema : AdminUserSchema) & {
        custom_attributes: CustomAttributeSchema[];
    })[], C, E, P>>;
    all<A extends boolean = false, E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: AllUsersOptions & AsAdmin<A> & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<(A extends false ? SimpleUserSchema : AdminUserSchema)[], C, E, P>>;
    allActivities<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: {
        from?: string;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserActivitySchema[], C, E, P>>;
    allEvents<E extends boolean = false, P extends PaginationTypes = 'offset'>(userId: number, options?: AllEventOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<CamelizedResponse<EventSchema, C>[], E, void>>;
    allFollowers<E extends boolean = false, P extends PaginationTypes = 'offset'>(userId: number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SimpleUserSchema[], C, E, P>>;
    allFollowing<E extends boolean = false, P extends PaginationTypes = 'offset'>(userId: number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SimpleUserSchema[], C, E, P>>;
    allMemberships<E extends boolean = false, P extends PaginationTypes = 'offset'>(userId: number, options?: {
        type?: 'Project' | 'Namespace';
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserMembershipSchema[], C, E, P>>;
    allProjects<E extends boolean = false, P extends PaginationTypes = 'keyset'>(userId: string | number, options: PaginationRequestOptions<P> & AllUserProjectsOptions & Sudo & ShowExpanded<E> & {
        simple: true;
    }): Promise<GitlabAPIResponse<SimpleProjectSchema[], C, E, P>>;
    allProjects<E extends boolean = false, P extends PaginationTypes = 'keyset'>(userId: string | number, options: PaginationRequestOptions<P> & AllUserProjectsOptions & Sudo & ShowExpanded<E> & {
        statistics: true;
    }): Promise<GitlabAPIResponse<(ProjectSchema & {
        statistics: ProjectStatisticsSchema;
    })[], C, E, P>>;
    allProjects<E extends boolean = false, P extends PaginationTypes = 'keyset'>(userId: string | number, options?: PaginationRequestOptions<P> & AllUserProjectsOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema[], C, E, P>>;
    allContributedProjects<E extends boolean = false, P extends PaginationTypes = 'keyset'>(userId: string | number, options: PaginationRequestOptions<P> & AllUserProjectsOptions & Sudo & ShowExpanded<E> & {
        simple: true;
    }): Promise<GitlabAPIResponse<SimpleProjectSchema[], C, E, P>>;
    allContributedProjects<E extends boolean = false, P extends PaginationTypes = 'keyset'>(userId: string | number, options: PaginationRequestOptions<P> & AllUserProjectsOptions & Sudo & ShowExpanded<E> & {
        statistics: true;
    }): Promise<GitlabAPIResponse<(ProjectSchema & {
        statistics: ProjectStatisticsSchema;
    })[], C, E, P>>;
    allContributedProjects<E extends boolean = false, P extends PaginationTypes = 'keyset'>(userId: string | number, options?: PaginationRequestOptions<P> & AllUserProjectsOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema[], C, E, P>>;
    allStarredProjects<E extends boolean = false, P extends PaginationTypes = 'keyset'>(userId: string | number, options: PaginationRequestOptions<P> & AllUserProjectsOptions & Sudo & ShowExpanded<E> & {
        simple: true;
    }): Promise<GitlabAPIResponse<SimpleProjectSchema[], C, E, P>>;
    allStarredProjects<E extends boolean = false, P extends PaginationTypes = 'keyset'>(userId: string | number, options: PaginationRequestOptions<P> & AllUserProjectsOptions & Sudo & ShowExpanded<E> & {
        statistics: true;
    }): Promise<GitlabAPIResponse<(ProjectSchema & {
        statistics: ProjectStatisticsSchema;
    })[], C, E, P>>;
    allStarredProjects<E extends boolean = false, P extends PaginationTypes = 'keyset'>(userId: string | number, options?: PaginationRequestOptions<P> & AllUserProjectsOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema[], C, E, P>>;
    approve<E extends boolean = false>(userId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<{
        message: string;
    }, C, E, void>>;
    ban<E extends boolean = false>(userId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    block<E extends boolean = false>(userId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    create<E extends boolean = false>(options?: CreateUserOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedUserSchema, C, E, void>>;
    createPersonalAccessToken<E extends boolean = false>(userId: number, name: string, scopes: string[], options?: {
        expiresAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PersonalAccessTokenSchema, C, E, void>>;
    createCIRunner<E extends boolean = false>(runnerType: 'instance_type' | 'group_type' | 'project_type', options?: CreateUserCIRunnerOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserRunnerSchema, C, E, void>>;
    deactivate<E extends boolean = false>(userId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    disableTwoFactor<E extends boolean = false>(userId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    edit<E extends boolean = false>(userId: number, { avatar, ...options }?: EditUserOptions & Sudo & ShowExpanded<E>): Promise<CamelizedResponse<ExpandedUserSchema, C>>;
    editStatus<E extends boolean = false>(options?: {
        emoji?: string;
        message?: string;
        clearStatusAfter?: '30_minutes' | '3_hours' | '8_hours' | '1_day' | '3_days' | '7_days' | '30_days';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserStatusSchema, C, E, void>>;
    editCurrentUserPreferences<E extends boolean = false>(viewDiffsFileByFile: boolean, showWhitespaceInDiffs: boolean, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserPreferenceSchema, C, E, void>>;
    follow<E extends boolean = false>(userId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SimpleUserSchema, C, E, void>>;
    reject<E extends boolean = false>(userId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<{
        message: string;
    }, C, E, void>>;
    show<A extends boolean = false, E extends boolean = false>(userId: number, options?: AsAdmin<A> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<A extends false ? UserSchema : AdminUserSchema, C, E, void>>;
    showCount<E extends boolean = false>(options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserCountSchema, C, E, void>>;
    showAssociationsCount<E extends boolean = false>(userId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserAssociationCountSchema, C, E, void>>;
    showCurrentUser<A extends boolean = false, E extends boolean = false>(options?: AsAdmin<A> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<A extends false ? ExpandedUserSchema : AdminUserSchema, C, E, void>>;
    showCurrentUserPreferences<E extends boolean = false>(options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserPreferenceSchema, C, E, void>>;
    showStatus<E extends boolean = false>({ iDOrUsername, ...options }?: {
        iDOrUsername?: string | number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserStatusSchema, C, E, void>>;
    remove<E extends boolean = false>(userId: number, options?: {
        hardDelete?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    removeAuthenticationIdentity<E extends boolean = false>(userId: number, provider: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    unban<E extends boolean = false>(userId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    unblock<E extends boolean = false>(userId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    unfollow<E extends boolean = false>(userId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserSchema, C, E, void>>;
}

interface CondensedNamespaceSchema extends Record<string, unknown> {
    id: number;
    name: string;
    path: string;
    kind: string;
    full_path: string;
    parent_id?: number | null;
    avatar_url: string | null;
    web_url: string;
}
interface NamespaceSchema extends CondensedNamespaceSchema {
    members_count_with_descendants: number;
    billable_members_count: number;
    plan: string;
    trial_ends_on?: string;
    trial: boolean;
}
interface NamespaceExistsSchema extends Record<string, unknown> {
    exists: boolean;
    suggests: string[];
}
declare class Namespaces<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: {
        search?: string;
        ownedOnly?: string;
        topLevelOnly?: boolean;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<NamespaceSchema[], C, E, P>>;
    exists<E extends boolean = false>(namespace: string, options?: {
        parentId?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<NamespaceExistsSchema, C, E, void>>;
    show<E extends boolean = false>(namespaceId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<NamespaceSchema, C, E, void>>;
}

interface GroupStatisticsSchema {
    storage_size: number;
    repository_size: number;
    wiki_size: number;
    lfs_objects_size: number;
    job_artifacts_size: number;
    pipeline_artifacts_size: number;
    packages_size: number;
    snippets_size: number;
    uploads_size: number;
}
interface CondensedGroupSchema extends Record<string, unknown> {
    id: number;
    web_url: string;
    name: string;
}
interface SimpleGroupSchema extends CondensedGroupSchema {
    avatar_url: string | null;
    full_name: string;
    full_path: string;
}
interface GroupSchema extends SimpleGroupSchema {
    path: string;
    description: string;
    visibility: 'public' | 'internal' | 'private';
    share_with_group_lock: boolean;
    require_two_factor_authentication: boolean;
    two_factor_grace_period: number;
    project_creation_level: string;
    auto_devops_enabled?: boolean;
    subgroup_creation_level: string;
    emails_enabled?: boolean;
    mentions_disabled?: boolean;
    lfs_enabled: boolean;
    default_branch_protection: number;
    request_access_enabled: boolean;
    created_at: string;
    parent_id: number;
    ldap_cn?: string;
    ldap_access?: string;
    marked_for_deletion_on?: string;
    membership_lock?: boolean;
}
interface ExpandedGroupSchema extends GroupSchema {
    runners_token: string;
    file_template_project_id: number;
    shared_with_groups: {
        group_id: number;
        group_name: string;
        group_full_path: string;
        group_access_level: number;
    }[] | null;
    projects?: ProjectSchema[];
    shared_projects?: ProjectSchema[];
}
type AllGroupsOptions = {
    skipGroups?: number[];
    allAvailable?: boolean;
    search?: string;
    orderBy?: 'name' | 'path' | 'id';
    sort?: 'asc' | 'desc';
    visibility?: 'public' | 'internal' | 'private';
    statistics?: boolean;
    withCustomAttributes?: boolean;
    owned?: boolean;
    minAccessLevel?: Exclude<AccessLevel, AccessLevel.ADMIN>;
    topLevelOnly?: boolean;
    markedForDeletionOn?: string;
    active?: boolean;
    archived?: boolean;
};
type AllGroupProjectsOptions = {
    visibility?: 'public' | 'internal' | 'private';
    orderBy?: 'id' | 'name' | 'path' | 'created_at' | 'updated_at' | 'similarity' | 'last_activity_at';
    topic?: string;
    sort?: 'asc' | 'desc';
    archived?: boolean;
    search?: string;
    simple?: boolean;
    owned?: boolean;
    starred?: boolean;
    withIssuesEnabled?: boolean;
    withMergeRequestsEnabled?: boolean;
    withShared?: boolean;
    includeSubgroups?: boolean;
    minAccessLevel?: Exclude<AccessLevel, AccessLevel.ADMIN>;
    withCustomAttributes?: boolean;
    withSecurityReports?: boolean;
};
type CreateGroupOptions = {
    autoDevopsEnabled?: boolean;
    avatar?: {
        content: Blob;
        filename: string;
    };
    defaultBranchProtection?: 0 | 1 | 2 | 3;
    description?: string;
    emailsDisabled?: boolean;
    lfsEnabled?: boolean;
    mentionsDisabled?: boolean;
    parentId?: number;
    projectCreationLevel?: 'noone' | 'maintainer' | 'developer';
    requestAccessEnabled?: boolean;
    requireTwoFactorAuthentication?: boolean;
    shareWithGroupLock?: boolean;
    subgroupCreationLevel?: string;
    twoFactorGracePeriod?: number;
    visibility?: 'public' | 'internal' | 'private';
    membershipLock?: boolean;
    extraSharedRunnersMinutesLimit?: number;
    sharedRunnersMinutesLimit?: number;
};
type EditGroupOptions = {
    name?: string;
    path?: string;
    autoDevopsEnabled?: boolean;
    avatar?: {
        content: Blob;
        filename: string;
    };
    defaultBranchProtection?: 0 | 1 | 2 | 3;
    description?: string;
    emailsDisabled?: boolean;
    lfsEnabled?: boolean;
    mentionsDisabled?: boolean;
    preventSharingGroupsOutsideHierarchy?: boolean;
    projectCreationLevel?: 'noone' | 'maintainer' | 'developer';
    requestAccessEnabled?: boolean;
    requireTwoFactorAuthentication?: boolean;
    sharedRunnersSetting?: 'enabled' | 'disabled_and_overridable' | 'disabled_and_unoverridable' | 'disabled_with_override';
    shareWithGroupLock?: boolean;
    subgroupCreationLevel?: string;
    twoFactorGracePeriod?: number;
    visibility?: 'public' | 'internal' | 'private';
    extraSharedRunnersMinutesLimit?: number;
    fileTemplateProjectId?: number;
    membershipLock?: boolean;
    preventForkingOutsideGroup?: boolean;
    sharedRunnersMinutesLimit?: number;
    uniqueProjectDownloadLimit?: number;
    uniqueProjectDownloadLimitIntervalInSeconds?: number;
    uniqueProjectDownloadLimitAllowlist?: string[];
    uniqueProjectDownloadLimitAlertlist?: number[];
    autoBanUserOnExcessiveProjectsDownload?: boolean;
    ipRestrictionRanges?: string;
};
type AllProvisionedUsersOptions = {
    username?: string;
    search?: string;
    active?: boolean;
    blocked?: boolean;
    createdAfter?: string;
    createdBefore?: string;
};
declare class Groups<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'keyset'>(options: {
        withCustomAttributes: true;
    } & AllGroupsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<(GroupSchema & {
        custom_attributes: CustomAttributeSchema[];
    })[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'keyset'>(options: {
        statistics: true;
    } & AllGroupsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<(GroupSchema & {
        statistics: GroupStatisticsSchema;
    })[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'keyset'>(options?: AllGroupsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupSchema[], C, E, P>>;
    allDescendantGroups<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options: {
        statistics: true;
    } & AllGroupsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<(GroupSchema & {
        statistics: GroupStatisticsSchema;
    })[], C, E, P>>;
    allDescendantGroups<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options: AllGroupsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupSchema[], C, E, P>>;
    allProjects<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: AllGroupProjectsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema[], C, E, P>>;
    allProjects<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: {
        simple: true;
    } & AllGroupProjectsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CondensedProjectSchema[], C, E, P>>;
    allSharedProjects<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: {
        simple: true;
    } & AllGroupProjectsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CondensedProjectSchema[], C, E, P>>;
    allSharedProjects<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: AllGroupProjectsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema[], C, E, P>>;
    allSubgroups<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: {
        statistics: true;
    } & AllGroupsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<(GroupSchema & {
        statistics: GroupStatisticsSchema;
    })[], C, E, P>>;
    allSubgroups<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: AllGroupsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupSchema[], C, E, P>>;
    allProvisionedUsers<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: AllProvisionedUsersOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SimpleUserSchema[], C, E, P>>;
    allTransferLocations<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: {
        search?: string;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SimpleGroupSchema[], C, E, P>>;
    create<E extends boolean = false>(name: string, path: string, { avatar, ...options }?: CreateGroupOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedGroupSchema, C, E, void>>;
    downloadAvatar<E extends boolean = false>(groupId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    edit<E extends boolean = false>(groupId: string | number, { avatar, ...options }?: EditGroupOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedGroupSchema, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, options?: {
        permanentlyRemove?: boolean | string;
        fullPath?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    removeAvatar<E extends boolean = false>(groupId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    restore<E extends boolean = false>(groupId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    search<E extends boolean = false>(nameOrPath: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupSchema[], C, E, void>>;
    share<E extends boolean = false>(groupId: string | number, sharedGroupId: string | number, groupAccess: number, options: {
        expiresAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedGroupSchema, C, E, void>>;
    show<E extends boolean = false>(groupId: string | number, options?: BaseRequestOptions<E>): Promise<GitlabAPIResponse<ExpandedGroupSchema, C, E, void>>;
    transfer<E extends boolean = false>(groupId: string | number, options?: {
        groupId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    transferProject<E extends boolean = false>(groupId: string | number, projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    unshare<E extends boolean = false>(groupId: string | number, sharedGroupId: string | number, options: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    uploadAvatar<E extends boolean = false>(groupId: string | number, content: Blob, { filename, ...options }?: {
        filename?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<{
        avatar_url: string;
    }, C, E, void>>;
}

type AccessLevelSettingState = 'disabled' | 'enabled' | 'private';
interface ProjectStarrerSchema extends Record<string, unknown> {
    starred_since: string;
    user: MappedOmit<SimpleUserSchema, 'created_at'>;
}
interface ProjectStoragePath extends Record<string, unknown> {
    project_id: string | number;
    disk_path: string;
    created_at: string;
    repository_storage: string;
}
interface ProjectStatisticsSchema {
    commit_count: number;
    storage_size: number;
    repository_size: number;
    wiki_size: number;
    lfs_objects_size: number;
    job_artifacts_size: number;
    pipeline_artifacts_size: number;
    packages_size: number;
    snippets_size: number;
    uploads_size: number;
}
interface ProjectLicenseSchema {
    key: string;
    name: string;
    nickname: string;
    html_url: string;
    source_url: string;
}
interface CondensedProjectSchema extends Record<string, unknown> {
    id: number;
    web_url: string;
    name: string;
    path: string;
}
interface SimpleProjectSchema extends CondensedProjectSchema {
    description: string | null;
    name_with_namespace: string;
    path_with_namespace: string;
    created_at: string;
    default_branch: string;
    topics: string[] | null;
    ssh_url_to_repo: string;
    http_url_to_repo: string;
    readme_url: string | null;
    forks_count: number;
    avatar_url: string | null;
    star_count: number;
    last_activity_at: string;
    namespace: CondensedNamespaceSchema;
}
interface ProjectSchema extends SimpleProjectSchema {
    description_html: string;
    visibility: 'public' | 'internal' | 'private';
    merge_requests_template?: string;
    empty_repo: boolean;
    issues_template?: string;
    owner: Pick<SimpleUserSchema, 'id' | 'name' | 'created_at'>;
    issues_enabled: boolean;
    open_issues_count: number;
    merge_requests_enabled: boolean;
    jobs_enabled: boolean;
    wiki_enabled: boolean;
    snippets_enabled: boolean;
    can_create_merge_request_in: boolean;
    resolve_outdated_diff_discussions: boolean;
    container_registry_access_level: string;
    security_and_compliance_access_level: string;
    container_expiration_policy: {
        cadence: string;
        enabled: boolean;
        keep_n: number | null;
        older_than: string | null;
        name_regex_delete: string | null;
        name_regex_keep: string | null;
        next_run_at: string;
    };
    updated_at: string;
    creator_id: number;
    import_url: string | null;
    import_type: string | null;
    import_status: string;
    import_error: string | null;
    permissions: {
        project_access: {
            access_level: number;
            notification_level: number;
        };
        group_access: {
            access_level: number;
            notification_level: number;
        };
    };
    archived: boolean;
    license_url: string;
    license: ProjectLicenseSchema;
    shared_runners_enabled: boolean;
    group_runners_enabled: boolean;
    runners_token: string;
    ci_default_git_depth: number;
    ci_forward_deployment_enabled: boolean;
    ci_forward_deployment_rollback_allowed: boolean;
    ci_allow_fork_pipelines_to_run_in_parent_project: boolean;
    ci_separated_caches: boolean;
    ci_restrict_pipeline_cancellation_role: string;
    public_jobs: boolean;
    shared_with_groups: {
        group_id: number;
        group_name: string;
        group_full_path: string;
        group_access_level: number;
    }[] | null;
    repository_storage: string;
    only_allow_merge_if_pipeline_succeeds: boolean;
    allow_merge_on_skipped_pipeline: boolean;
    ci_pipeline_variables_minimum_override_role: boolean;
    only_allow_merge_if_all_discussions_are_resolved: boolean;
    remove_source_branch_after_merge: boolean;
    printing_merge_requests_link_enabled: boolean;
    request_access_enabled: boolean;
    merge_method: string;
    squash_option: string;
    auto_devops_enabled: boolean;
    auto_devops_deploy_strategy: string;
    mirror: boolean;
    mirror_user_id: number;
    mirror_trigger_builds: boolean;
    only_mirror_protected_branches: boolean;
    mirror_overwrites_diverged_branches: boolean;
    external_authorization_classification_label: string | null;
    packages_enabled: boolean;
    service_desk_enabled: boolean;
    service_desk_address: string | null;
    autoclose_referenced_issues: boolean;
    suggestion_commit_message: string | null;
    enforce_auth_checks_on_uploads: boolean;
    merge_commit_template: string | null;
    squash_commit_template: string | null;
    issue_branch_template: string;
    marked_for_deletion_on: string;
    compliance_frameworks: string[] | null;
    warn_about_potentially_unwanted_characters: boolean;
    container_registry_image_prefix: string;
    _links: {
        self: string;
        issues: string;
        merge_requests: string;
        repo_branches: string;
        labels: string;
        events: string;
        members: string;
        cluster_agents: string;
    };
}
interface ProjectFileUploadSchema extends Record<string, unknown> {
    alt: string;
    url: string;
    full_path: string;
    markdown: string;
}
type AllProjectsOptions = {
    userId?: string;
    archived?: boolean;
    idAfter?: number;
    idBefore?: number;
    imported?: boolean;
    includeHidden?: boolean;
    includePendingDelete?: boolean;
    lastActivityAfter?: string;
    lastActivityBefore?: string;
    membership?: boolean;
    minAccessLevel?: Exclude<AccessLevel, AccessLevel.ADMIN>;
    orderBy?: 'id' | 'name' | 'path' | 'created_at' | 'updated_at' | 'last_activity_at' | 'similarity' | 'repository_size' | 'storage_size' | 'packages_size' | 'wiki_size';
    owned?: boolean;
    repositoryChecksumFailed?: boolean;
    repositoryStorage?: string;
    searchNamespaces?: boolean;
    search?: string;
    simple?: boolean;
    sort?: 'asc' | 'desc';
    starred?: boolean;
    statistics?: boolean;
    topic?: string;
    topicId?: number;
    visibility?: 'public' | 'internal' | 'private';
    wikiChecksumFailed?: boolean;
    withCustomAttributes?: boolean;
    withIssuesEnabled?: boolean;
    withMergeRequestsEnabled?: boolean;
    withProgrammingLanguage?: string;
    updatedBefore?: string;
    updatedAfter?: string;
    markedForDeletionOn?: string;
    active?: boolean;
};
type CreateProjectOptions = {
    userId?: number;
    avatar?: {
        content: Blob;
        filename: string;
    };
    allowMergeOnSkippedPipeline?: boolean;
    onlyAllowMergeIfAllStatusChecksPassed?: boolean;
    analyticsAccessLevel?: AccessLevelSettingState;
    approvalsBeforeMerge?: number;
    autoCancelPendingPipelines?: string;
    autoDevopsDeployStrategy?: 'continuous' | 'manual' | 'timed_incremental';
    autoDevopsEnabled?: boolean;
    autocloseReferencedIssues?: boolean;
    buildGitStrategy?: string;
    buildTimeout?: number;
    buildsAccessLevel?: AccessLevelSettingState;
    ciConfigPath?: string;
    ciDeletePipelinesInSeconds?: number;
    containerExpirationPolicyAttributes?: Record<string, string>;
    containerRegistryAccessLevel?: AccessLevelSettingState;
    defaultBranch?: string;
    description?: string;
    emailsDisabled?: boolean;
    externalAuthorizationClassificationLabel?: string;
    forkingAccessLevel?: AccessLevelSettingState;
    groupWithProjectTemplatesId?: number;
    importUrl?: string;
    initializeWithReadme?: boolean;
    issuesAccessLevel?: AccessLevelSettingState;
    lfsEnabled?: boolean;
    mergeMethod?: string;
    mergePipelinesEnabled?: boolean;
    mergeRequestsAccessLevel?: AccessLevelSettingState;
    mergeTrainsEnabled?: boolean;
    mirrorTriggerBuilds?: boolean;
    mirror?: boolean;
    namespaceId?: number;
    onlyAllowMergeIfAllDiscussionsAreResolved?: boolean;
    onlyAllowMergeIfPipelineSucceeds?: boolean;
    packagesEnabled?: boolean;
    pagesAccessLevel?: AccessLevelSettingState | 'public';
    printingMergeRequestLinkEnabled?: boolean;
    publicBuilds?: boolean;
    releasesAccessLevel?: AccessLevelSettingState;
    environmentsAccessLevel?: AccessLevelSettingState;
    featureFlagsAccessLevel?: AccessLevelSettingState;
    infrastructureAccessLevel?: AccessLevelSettingState;
    monitorAccessLevel?: AccessLevelSettingState;
    removeSourceBranchAfterMerge?: boolean;
    repositoryAccessLevel?: AccessLevelSettingState;
    repositoryStorage?: string;
    requestAccessEnabled?: boolean;
    requirementsAccessLevel?: AccessLevelSettingState;
    resolveOutdatedDiffDiscussions?: boolean;
    securityAndComplianceAccessLevel?: AccessLevelSettingState;
    sharedRunnersEnabled?: boolean;
    groupRunnersEnabled?: boolean;
    snippetsAccessLevel?: AccessLevelSettingState;
    squashOption?: 'never' | 'always' | 'default_on' | 'default_off';
    templateName?: string;
    templateProjectId?: number;
    topics?: string[];
    useCustomTemplate?: boolean;
    visibility?: 'public' | 'internal' | 'private';
    wikiAccessLevel?: AccessLevelSettingState;
};
type EditProjectOptions = {
    avatar?: {
        content: Blob;
        filename: string;
    };
    allowMergeOnSkippedPipeline?: boolean;
    allowPipelineTriggerApproveDeployment?: boolean;
    onlyAllowMergeIfAllStatusChecksPassed?: boolean;
    analyticsAccessLevel?: AccessLevelSettingState;
    approvalsBeforeMerge?: number;
    autoCancelPendingPipelines?: string;
    autoDevopsDeployStrategy?: 'continuous' | 'manual' | 'timed_incremental';
    autoDevopsEnabled?: boolean;
    autocloseReferencedIssues?: boolean;
    buildGitStrategy?: string;
    buildTimeout?: number;
    buildsAccessLevel?: AccessLevelSettingState;
    ciConfigPath?: string;
    ciDefaultGitDepth?: number;
    ciForwardDeploymentEnabled?: boolean;
    ciAllowForkPipelinesToRunInParentProject?: boolean;
    ciSeparatedCaches?: boolean;
    ciDeletePipelinesInSeconds?: number;
    containerExpirationPolicyAttributes?: Record<string, string>;
    containerRegistryAccessLevel?: string;
    defaultBranch?: string;
    description?: string;
    emailsDisabled?: boolean;
    enforceAuthChecksOnUploads?: boolean;
    externalAuthorizationClassificationLabel?: string;
    forkingAccessLevel?: AccessLevelSettingState;
    importUrl?: string;
    issuesAccessLevel?: AccessLevelSettingState;
    issuesTemplate?: string;
    keepLatestArtifact?: boolean;
    lfsEnabled?: boolean;
    mergeCommitTemplate?: string;
    mergeMethod?: string;
    mergePipelinesEnabled?: boolean;
    mergeRequestsAccessLevel?: AccessLevelSettingState;
    mergeRequestsTemplate?: string;
    mergeTrainsEnabled?: boolean;
    mirrorOverwritesDivergedBranches?: boolean;
    mirrorTriggerBuilds?: boolean;
    mirrorUserId?: number;
    mirror?: boolean;
    mrDefaultTargetSelf?: boolean;
    name?: string;
    onlyAllowMergeIfAllDiscussionsAreResolved?: boolean;
    onlyAllowMergeIfPipelineSucceeds?: boolean;
    onlyMirrorProtectedBranches?: boolean;
    packagesEnabled?: boolean;
    pagesAccessLevel?: string;
    path?: string;
    printingMergeRequestLinkEnabled?: boolean;
    publicBuilds?: boolean;
    releasesAccessLevel?: AccessLevelSettingState;
    environmentsAccessLevel?: AccessLevelSettingState;
    featureFlagsAccessLevel?: AccessLevelSettingState;
    infrastructureAccessLevel?: AccessLevelSettingState;
    monitorAccessLevel?: AccessLevelSettingState;
    removeSourceBranchAfterMerge?: boolean;
    repositoryAccessLevel?: AccessLevelSettingState;
    repositoryStorage?: string;
    requestAccessEnabled?: boolean;
    requirementsAccessLevel?: AccessLevelSettingState;
    resolveOutdatedDiffDiscussions?: boolean;
    restrictUserDefinedVariables?: boolean;
    securityAndComplianceAccessLevel?: AccessLevelSettingState;
    serviceDeskEnabled?: boolean;
    sharedRunnersEnabled?: boolean;
    groupRunnersEnabled?: boolean;
    snippetsAccessLevel?: AccessLevelSettingState;
    issueBranchTemplate?: string;
    squashCommitTemplate?: string;
    squashOption?: 'never' | 'always' | 'default_on' | 'default_off';
    suggestionCommitMessage?: string;
    topics?: string[];
    visibility?: 'public' | 'internal' | 'private';
    wikiAccessLevel?: AccessLevelSettingState;
};
type ForkProjectOptions = {
    branches?: string;
    description?: string;
    mrDefaultTargetSelf?: boolean;
    name?: string;
    namespaceId?: number;
    namespacePath?: string;
    namespace?: number | string;
    path?: string;
    visibility?: 'public' | 'internal' | 'private';
};
type AllForksOptions = {
    archived?: boolean;
    membership?: boolean;
    minAccessLevel?: AccessLevelSettingState;
    orderBy?: 'id' | 'name' | 'path' | 'created_at' | 'updated_at' | 'last_activity_at';
    owned?: boolean;
    search?: string;
    simple?: boolean;
    sort?: 'asc' | 'desc';
    starred?: boolean;
    statistics?: boolean;
    visibility?: 'public' | 'internal' | 'private';
    withCustomAttributes?: boolean;
    withIssuesEnabled?: boolean;
    withMergeRequestsEnabled?: boolean;
    updatedBefore?: string;
    updatedAfter?: string;
};
declare class Projects<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'keyset'>(options: PaginationRequestOptions<P> & AllProjectsOptions & Sudo & ShowExpanded<E> & {
        withCustomAttributes: true;
    }): Promise<GitlabAPIResponse<(ProjectSchema & {
        custom_attributes: CustomAttributeSchema[];
    })[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'keyset'>(options: PaginationRequestOptions<P> & AllProjectsOptions & Sudo & ShowExpanded<E> & {
        simple: true;
    }): Promise<GitlabAPIResponse<SimpleProjectSchema[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'keyset'>(options: PaginationRequestOptions<P> & AllProjectsOptions & Sudo & ShowExpanded<E> & {
        statistics: true;
    }): Promise<GitlabAPIResponse<(ProjectSchema & {
        statistics: ProjectStatisticsSchema;
    })[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'keyset'>(options: PaginationRequestOptions<P> & AllProjectsOptions & Sudo & ShowExpanded<E> & {
        statistics: true;
        withCustomAttributes: true;
    }): Promise<GitlabAPIResponse<(ProjectSchema & {
        statistics: ProjectStatisticsSchema;
        custom_attributes: CustomAttributeSchema[];
    })[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'keyset'>(options?: PaginationRequestOptions<P> & AllProjectsOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema[], C, E, P>>;
    allTransferLocations<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: {
        search?: string;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SimpleGroupSchema[], C, E, P>>;
    allUsers<E extends boolean = false>(projectId: string | number, options?: {
        search?: string;
        skipUsers?: number[];
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MappedOmit<SimpleUserSchema, 'created_at'>[], C, E, void>>;
    allGroups<E extends boolean = false>(projectId: string | number, options?: {
        search?: string;
        skipGroups?: number[];
        withShared?: boolean;
        sharedMinAccessLevel?: Exclude<AccessLevel, AccessLevel.ADMIN>;
        sharedVisibleOnly?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SimpleGroupSchema[], C, E, void>>;
    allInvitedGroups<E extends boolean = false>(projectId: string | number, options?: {
        search?: string;
        sharedMinAccessLevel?: Exclude<AccessLevel, AccessLevel.ADMIN>;
        relation?: 'direct' | 'inherited';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SimpleGroupSchema[], C, E, void>>;
    allSharableGroups<E extends boolean = false>(projectId: string | number, options?: {
        search?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SimpleGroupSchema[], C, E, void>>;
    allForks<E extends boolean = false>(projectId: string | number, options: AllForksOptions & Sudo & ShowExpanded<E> & {
        simple: true;
    }): Promise<GitlabAPIResponse<SimpleProjectSchema[], C, E, void>>;
    allForks<E extends boolean = false>(projectId: string | number, options: AllForksOptions & Sudo & ShowExpanded<E> & {
        statistics: true;
    }): Promise<GitlabAPIResponse<(ProjectSchema & {
        statistics: ProjectStatisticsSchema;
    })[], C, E, void>>;
    allForks<E extends boolean = false>(projectId: string | number, options?: AllForksOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema[], C, E, void>>;
    allStarrers<E extends boolean = false>(projectId: string | number, options?: {
        search?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectStarrerSchema[], C, E, void>>;
    allStoragePaths<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectStoragePath[], C, E, void>>;
    archive<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema, C, E, void>>;
    create<E extends boolean = false>({ userId, avatar, ...options }?: SomeOf<{
        name: string;
        path: string;
    }> & CreateProjectOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema, C, E, void>>;
    createForkRelationship<E extends boolean = false>(projectId: string | number, forkedFromId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    createPullMirror<E extends boolean = false>(projectId: string | number, url: string, mirror: boolean, options?: {
        mirrorTriggerBuilds?: boolean;
        mirrorBranchRegex?: string;
        onlyProtectedBranches?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectRemoteMirrorSchema, C, E, void>>;
    downloadSnapshot<E extends boolean = false>(projectId: string | number, options?: {
        wiki?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, { avatar, ...options }?: EditProjectOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema, C, E, void>>;
    fork<E extends boolean = false>(projectId: string | number, options?: ForkProjectOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema, C, E, void>>;
    housekeeping<E extends boolean = false>(projectId: string | number, options?: {
        task?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    importProjectMembers<E extends boolean = false>(projectId: string | number, sourceProjectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, options?: {
        permanentlyRemove?: boolean;
        fullPath?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    removeForkRelationship<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    removeAvatar<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    restore<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    search<E extends boolean = false>(projectName: string, options?: {
        sort?: 'asc' | 'desc';
        orderBy?: 'id' | 'name' | 'created_at' | 'last_activity_at';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema[], C, E, void>>;
    share<E extends boolean = false>(projectId: string | number, groupId: string | number, groupAccess: number, options?: {
        expiresAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, options: {
        license: true;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema & {
        license: ProjectLicenseSchema;
    }, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, options: {
        withCustomAttributes: true;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema & {
        custom_attributes: CustomAttributeSchema[];
    }, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, options: {
        statistics: true;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema & {
        statistics: ProjectStatisticsSchema;
    }, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, options: {
        withCustomAttributes: true;
        license: true;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema & {
        custom_attributes: CustomAttributeSchema[];
        license: ProjectLicenseSchema;
    }, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, options: {
        withCustomAttributes: true;
        statistics: true;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema & {
        custom_attributes: CustomAttributeSchema[];
        statistics: ProjectStatisticsSchema;
    }, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, options: {
        license: true;
        statistics: true;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema & {
        license: ProjectLicenseSchema;
        statistics: ProjectStatisticsSchema;
    }, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, options: {
        withCustomAttributes: true;
        license: true;
        statistics: true;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema & {
        custom_attributes: CustomAttributeSchema[];
        license: ProjectLicenseSchema;
        statistics: ProjectStatisticsSchema;
    }, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, options?: {
        license?: boolean;
        statistics?: boolean;
        withCustomAttributes?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema, C, E, void>>;
    showLanguages<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<{
        [name: string]: number;
    }, C, E, void>>;
    showPullMirror<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectRemoteMirrorSchema, C, E, void>>;
    star<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema, C, E, void>>;
    transfer<E extends boolean = false>(projectId: string | number, namespaceId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<CamelizedResponse<ProjectSchema, C>, E, undefined>>;
    unarchive<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema, C, E, void>>;
    unshare<E extends boolean = false>(projectId: string | number, groupId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    unstar<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectSchema, C, E, void>>;
    uploadForReference<E extends boolean = false>(projectId: string | number, file: {
        content: Blob;
        filename: string;
    }, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectFileUploadSchema, C, E, void>>;
    uploadAvatar<E extends boolean = false>(projectId: string | number, avatar: {
        content: Blob;
        filename: string;
    }, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<{
        avatar_url: string;
    }, C, E, void>>;
}

interface ClusterAgentSchema extends Record<string, unknown> {
    id: number;
    name: string;
    config_project: SimpleProjectSchema;
    created_at: string;
    created_by_user_id: number;
}
interface ClusterAgentTokenSchema extends Record<string, unknown> {
    id: number;
    name: string;
    description: string;
    agent_id: number;
    status: string;
    token?: string;
    created_at: string;
    created_by_user_id: number;
}
declare class Agents<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ClusterAgentSchema[], C, E, void>>;
    allTokens<E extends boolean = false>(projectId: string | number, agentId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ClusterAgentTokenSchema[], C, E, void>>;
    createToken<E extends boolean = false>(projectId: string | number, agentId: number, name: string, options?: {
        description?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ClusterAgentTokenSchema, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, agentId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ClusterAgentSchema, C, E, void>>;
    showToken<E extends boolean = false>(projectId: string | number, agentId: number, tokenId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ClusterAgentTokenSchema, C, E, void>>;
    register<E extends boolean = false>(projectId: string | number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ClusterAgentSchema, C, E, void>>;
    removeToken<E extends boolean = false>(projectId: string | number, agentId: number, tokenId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    unregister<E extends boolean = false>(projectId: string | number, agentId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface MetricImageSchema extends Record<string, unknown> {
    id: number;
    created_at: string;
    filename: string;
    file_path: string;
    url: string;
    url_text: string;
}
declare class AlertManagement<C extends boolean = false> extends BaseResource<C> {
    allMetricImages<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, alertIId: number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MetricImageSchema[], C, E, P>>;
    editMetricImage<E extends boolean = false>(projectId: string | number, alertIId: number, imageId: number, options?: {
        url?: string;
        urlText?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MetricImageSchema, C, E, void>>;
    removeMetricImage<E extends boolean = false>(projectId: string | number, alertIId: number, imageId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    uploadMetricImage<E extends boolean = false>(projectId: string | number, alertIId: number, metricImage: {
        content: Blob;
        filename: string;
    }, options?: {
        url?: string;
        urlText?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MetricImageSchema, C, E, void>>;
}

interface ApplicationAppearanceSchema extends Record<string, unknown> {
    title: string;
    description: string;
    pwa_name: string;
    pwa_short_name: string;
    pwa_description: string;
    pwa_icon: string;
    logo: string;
    header_logo: string;
    favicon: string;
    new_project_guidelines: string;
    profile_image_guidelines: string;
    header_message: string;
    footer_message: string;
    message_background_color: string;
    message_font_color: string;
    email_header_and_footer_enabled: boolean;
}
declare class ApplicationAppearance<C extends boolean = false> extends BaseResource<C> {
    show<E extends boolean = false>(options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ApplicationAppearanceSchema, C, E, void>>;
    edit<E extends boolean = false>({ logo, pwaIcon, ...options }?: {
        logo?: {
            content: Blob;
            filename: string;
        };
        pwaIcon?: {
            content: Blob;
            filename: string;
        };
    } & BaseRequestOptions<E>): Promise<GitlabAPIResponse<ApplicationAppearanceSchema, C, E, void>>;
}

interface ApplicationPlanLimitSchema extends Record<string, unknown> {
    ci_pipeline_size: number;
    ci_active_jobs: number;
    ci_active_pipelines: number;
    ci_project_subscriptions: number;
    ci_pipeline_schedules: number;
    ci_needs_size_limit: number;
    ci_registered_group_runners: number;
    ci_registered_project_runners: number;
    conan_max_file_size: number;
    generic_packages_max_file_size: number;
    helm_max_file_size: number;
    maven_max_file_size: number;
    npm_max_file_size: number;
    nuget_max_file_size: number;
    pypi_max_file_size: number;
    terraform_module_max_file_size: number;
    storage_size_limit: number;
}
type ApplicationPlanLimitOptions = Partial<Camelize<ApplicationPlanLimitSchema>>;
declare class ApplicationPlanLimits<C extends boolean = false> extends BaseResource<C> {
    show<E extends boolean = false>(options?: {
        planName?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ApplicationPlanLimitSchema, C, E, void>>;
    edit<E extends boolean = false>(planName: string, options?: ApplicationPlanLimitOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ApplicationPlanLimitSchema, C, E, void>>;
}

interface ApplicationSettingsSchema extends Record<string, unknown> {
    admin_mode?: boolean;
    admin_notification_email?: string;
    abuse_notification_email?: string;
    after_sign_out_path?: string;
    after_sign_up_text?: string;
    akismet_api_key?: string;
    akismet_enabled?: boolean;
    allow_group_owners_to_manage_ldap?: boolean;
    allow_local_requests_from_hooks_and_services?: boolean;
    allow_local_requests_from_system_hooks?: boolean;
    allow_local_requests_from_web_hooks_and_services?: boolean;
    allow_runner_registration_token?: boolean;
    archive_builds_in_human_readable?: string;
    asset_proxy_enabled?: boolean;
    asset_proxy_secret_key?: string;
    asset_proxy_url?: string;
    asset_proxy_whitelist?: string | string[];
    asset_proxy_allowlist?: string | string[];
    authorized_keys_enabled?: boolean;
    auto_devops_domain?: string;
    auto_devops_enabled?: boolean;
    automatic_purchased_storage_allocation?: boolean;
    bulk_import_enabled?: boolean;
    can_create_group?: boolean;
    check_namespace_plan?: boolean;
    commit_email_hostname?: string;
    container_expiration_policies_enable_historic_entries?: boolean;
    container_registry_cleanup_tags_service_max_list_size?: number;
    container_registry_delete_tags_service_timeout?: number;
    container_registry_expiration_policies_caching?: boolean;
    container_registry_expiration_policies_worker_capacity?: number;
    container_registry_token_expire_delay?: number;
    package_registry_cleanup_policies_worker_capacity?: number;
    deactivate_dormant_users?: boolean;
    deactivate_dormant_users_period?: number;
    default_artifacts_expire_in?: string;
    default_branch_name?: string;
    default_branch_protection?: number;
    default_ci_config_path?: string;
    default_group_visibility?: string;
    default_preferred_language?: string;
    default_project_creation?: number;
    default_project_visibility?: string;
    default_projects_limit?: number;
    default_snippet_visibility?: string;
    default_syntax_highlighting_theme?: number;
    delayed_project_deletion?: boolean;
    delayed_group_deletion?: boolean;
    deletion_adjourned_period?: number;
    diff_max_patch_bytes?: number;
    diff_max_files?: number;
    diff_max_lines?: number;
    disable_admin_oauth_scopes?: boolean;
    disable_feed_token?: boolean;
    disable_personal_access_token?: boolean;
    disabled_oauth_sign_in_sources?: string[];
    dns_rebinding_protection_enabled?: boolean;
    domain_denylist_enabled?: boolean;
    domain_denylist?: string[];
    domain_allowlist?: string[];
    dsa_key_restriction?: number;
    ecdsa_key_restriction?: number;
    ecdsa_sk_key_restriction?: number;
    ed25519_key_restriction?: number;
    ed25519_sk_key_restriction?: number;
    eks_access_key_id?: string;
    eks_account_id?: string;
    eks_integration_enabled?: boolean;
    eks_secret_access_key?: string;
    elasticsearch_aws_access_key?: string;
    elasticsearch_aws_region?: string;
    elasticsearch_aws_secret_access_key?: string;
    elasticsearch_aws?: boolean;
    elasticsearch_indexed_field_length_limit?: number;
    elasticsearch_indexed_file_size_limit_kb?: number;
    elasticsearch_indexing?: boolean;
    elasticsearch_limit_indexing?: boolean;
    elasticsearch_max_bulk_concurrency?: number;
    elasticsearch_max_bulk_size_mb?: number;
    elasticsearch_namespace_ids?: number[];
    elasticsearch_project_ids?: number[];
    elasticsearch_search?: boolean;
    elasticsearch_url?: string;
    elasticsearch_username?: string;
    elasticsearch_password?: string;
    email_additional_text?: string;
    email_author_in_body?: boolean;
    email_confirmation_setting?: string;
    enabled_git_access_protocol?: string;
    enforce_namespace_storage_limit?: boolean;
    enforce_terms?: boolean;
    external_auth_client_cert?: string;
    external_auth_client_key_pass?: string;
    external_auth_client_key?: string;
    external_authorization_service_default_label?: string;
    external_authorization_service_enabled?: boolean;
    external_authorization_service_timeout?: number;
    external_authorization_service_url?: string;
    external_pipeline_validation_service_url?: string;
    external_pipeline_validation_service_token?: string;
    external_pipeline_validation_service_timeout?: number;
    file_template_project_id?: number;
    first_day_of_week?: number;
    geo_node_allowed_ips?: string;
    geo_status_timeout?: number;
    git_two_factor_session_expiry?: number;
    gitaly_timeout_default?: number;
    gitaly_timeout_fast?: number;
    gitaly_timeout_medium?: number;
    gitlab_dedicated_instance?: boolean;
    grafana_enabled?: boolean;
    grafana_url?: string;
    gravatar_enabled?: boolean;
    group_owners_can_manage_default_branch_protection?: boolean;
    hashed_storage_enabled?: boolean;
    help_page_hide_commercial_content?: boolean;
    help_page_support_url?: string;
    help_page_text?: string;
    help_text?: string;
    hide_third_party_offers?: boolean;
    home_page_url?: string;
    housekeeping_enabled?: boolean;
    housekeeping_optimize_repository_period?: number;
    html_emails_enabled?: boolean;
    import_sources?: string[];
    in_product_marketing_emails_enabled?: boolean;
    invisible_captcha_enabled?: boolean;
    issues_create_limit?: number;
    keep_latest_artifact?: boolean;
    local_markdown_version?: number;
    mailgun_signing_key?: string;
    mailgun_events_enabled?: boolean;
    maintenance_mode_message?: string;
    maintenance_mode?: boolean;
    max_artifacts_size?: number;
    max_attachment_size?: number;
    max_export_size?: number;
    max_import_size?: number;
    max_pages_size?: number;
    max_personal_access_token_lifetime?: number;
    max_ssh_key_lifetime?: number;
    max_terraform_state_size_bytes?: number;
    metrics_method_call_threshold?: number;
    max_number_of_repository_downloads?: number;
    max_number_of_repository_downloads_within_time_period?: number;
    git_rate_limit_users_allowlist?: string[];
    git_rate_limit_users_alertlist?: number[];
    auto_ban_user_on_excessive_projects_download?: boolean;
    mirror_available?: boolean;
    mirror_capacity_threshold?: number;
    mirror_max_capacity?: number;
    mirror_max_delay?: number;
    maven_package_requests_forwarding?: boolean;
    npm_package_requests_forwarding?: boolean;
    pypi_package_requests_forwarding?: boolean;
    outbound_local_requests_whitelist?: string[];
    pages_domain_verification_enabled?: boolean;
    password_authentication_enabled_for_git?: boolean;
    password_authentication_enabled_for_web?: boolean;
    password_number_required?: boolean;
    password_symbol_required?: boolean;
    password_uppercase_required?: boolean;
    password_lowercase_required?: boolean;
    performance_bar_allowed_group_path?: string;
    personal_access_token_prefix?: string;
    pipeline_limit_per_project_user_sha?: number;
    plantuml_enabled?: boolean;
    plantuml_url?: string;
    polling_interval_multiplier?: number;
    project_export_enabled?: boolean;
    projects_api_rate_limit_unauthenticated?: number;
    prometheus_metrics_enabled?: boolean;
    protected_ci_variables?: boolean;
    push_event_activities_limit?: number;
    push_event_hooks_limit?: number;
    rate_limiting_response_text?: string;
    raw_blob_request_limit?: number;
    search_rate_limit?: number;
    search_rate_limit_unauthenticated?: number;
    recaptcha_enabled?: boolean;
    recaptcha_private_key?: string;
    recaptcha_site_key?: string;
    receive_max_input_size?: number;
    repository_checks_enabled?: boolean;
    repository_size_limit?: number;
    repository_storages_weighted?: Record<string, number>;
    repository_storages?: string[];
    require_admin_approval_after_user_signup?: boolean;
    require_two_factor_authentication?: boolean;
    restricted_visibility_levels?: string[];
    rsa_key_restriction?: number;
    session_expire_delay?: number;
    shared_runners_enabled?: boolean;
    shared_runners_minutes?: number;
    shared_runners_text?: string;
    sidekiq_job_limiter_mode?: string;
    sidekiq_job_limiter_compression_threshold_bytes?: number;
    sidekiq_job_limiter_limit_bytes?: number;
    sign_in_text?: string;
    signup_enabled?: boolean;
    slack_app_enabled?: boolean;
    slack_app_id?: string;
    slack_app_secret?: string;
    slack_app_signing_secret?: string;
    slack_app_verification_token?: string;
    snippet_size_limit?: number;
    snowplow_app_id?: string;
    snowplow_collector_hostname?: string;
    snowplow_cookie_domain?: string;
    snowplow_enabled?: boolean;
    sourcegraph_enabled?: boolean;
    sourcegraph_public_only?: boolean;
    sourcegraph_url?: string;
    spam_check_endpoint_enabled?: boolean;
    spam_check_endpoint_url?: string;
    spam_check_api_key?: string;
    suggest_pipeline_enabled?: boolean;
    terminal_max_session_time?: number;
    terms?: string;
    throttle_authenticated_api_enabled?: boolean;
    throttle_authenticated_api_period_in_seconds?: number;
    throttle_authenticated_api_requests_per_period?: number;
    throttle_authenticated_packages_api_enabled?: boolean;
    throttle_authenticated_packages_api_period_in_seconds?: number;
    throttle_authenticated_packages_api_requests_per_period?: number;
    throttle_authenticated_web_enabled?: boolean;
    throttle_authenticated_web_period_in_seconds?: number;
    throttle_authenticated_web_requests_per_period?: number;
    throttle_unauthenticated_api_enabled?: boolean;
    throttle_unauthenticated_api_period_in_seconds?: number;
    throttle_unauthenticated_api_requests_per_period?: number;
    throttle_unauthenticated_packages_api_enabled?: boolean;
    throttle_unauthenticated_packages_api_period_in_seconds?: number;
    throttle_unauthenticated_packages_api_requests_per_period?: number;
    throttle_unauthenticated_web_enabled?: boolean;
    throttle_unauthenticated_web_period_in_seconds?: number;
    throttle_unauthenticated_web_requests_per_period?: number;
    time_tracking_limit_to_hours?: boolean;
    two_factor_grace_period?: number;
    unique_ips_limit_enabled?: boolean;
    unique_ips_limit_per_user?: number;
    unique_ips_limit_time_window?: number;
    usage_ping_enabled?: boolean;
    user_deactivation_emails_enabled?: boolean;
    user_default_external?: boolean;
    user_default_internal_regex?: string;
    user_defaults_to_private_profile?: boolean;
    user_oauth_applications?: boolean;
    user_show_add_ssh_key_message?: boolean;
    version_check_enabled?: boolean;
    whats_new_variant?: string;
    wiki_page_max_content_bytes?: number;
    jira_connect_application_key?: string;
    jira_connect_proxy_url?: string;
}
declare class ApplicationSettings<C extends boolean = false> extends BaseResource<C> {
    show<E extends boolean = false>(options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ApplicationSettingsSchema, C, E, void>>;
    edit<E extends boolean = false>(options?: BaseRequestOptions<E>): Promise<GitlabAPIResponse<ApplicationSettingsSchema, C, E, void>>;
}

interface ApplicationStatisticSchema extends Record<string, unknown> {
    forks: string;
    issues: string;
    merge_requests: string;
    notes: string;
    snippets: string;
    ssh_keys: string;
    milestones: string;
    users: string;
    groups: string;
    projects: string;
    active_users: string;
}
declare class ApplicationStatistics<C extends boolean = false> extends BaseResource<C> {
    show<E extends boolean = false>(options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ApplicationStatisticSchema, C, E, void>>;
}

interface ApplicationSchema extends Record<string, unknown> {
    id: number;
    application_id: string;
    application_name: string;
    secret: string;
    callback_url: string;
    confidential: boolean;
}
declare class Applications<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ApplicationSchema[], C, E, P>>;
    create<E extends boolean = false>(name: string, redirectUri: string, scopes: string, options?: {
        confidential?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ApplicationSchema, C, E, void>>;
    remove<E extends boolean = false>(applicationId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface AuditEventSchema extends Record<string, unknown> {
    id: number;
    author_id: number;
    entity_id: number;
    entity_type: string;
    details: {
        change?: string;
        from?: string;
        to?: string;
        custom_message?: string;
        author_name: string;
        author_email: string;
        target_id: string;
        target_type: string;
        target_details: string;
        ip_address: string;
        entity_path: string;
    };
    created_at: string;
}
interface AllAuditEventOptions {
    createdAfter?: string;
    createdBefore?: string;
    entityType?: string;
    entityId?: number;
}
declare class AuditEvents<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>({ projectId, groupId, ...options }?: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & AllAuditEventOptions & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<AuditEventSchema[], C, E, P>>;
    show<E extends boolean = false>(auditEventId: number, { projectId, groupId, ...options }?: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AuditEventSchema, C, E, void>>;
}

interface AvatarSchema extends Record<string, unknown> {
    avatar_url: string;
}
declare class Avatar<C extends boolean = false> extends BaseResource<C> {
    show<E extends boolean = false>(email: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AvatarSchema, C, E, void>>;
}

interface BroadcastMessageSchema extends Record<string, unknown> {
    message: string;
    starts_at: string;
    ends_at: string;
    color: string;
    font: string;
    id: number;
    active: boolean;
    target_path: string;
    target_access_levels: Exclude<AccessLevel, AccessLevel.MINIMAL_ACCESS | AccessLevel.NO_ACCESS | AccessLevel.ADMIN>[];
    broadcast_type: string;
    dismissable: boolean;
}
interface BroadcastMessageOptions extends Record<string, unknown> {
    message?: string;
    startsAt?: string;
    endsAt?: string;
    color?: string;
    font?: string;
    active?: boolean;
    targetPath?: string;
    targetAccessLevels?: Exclude<AccessLevel, AccessLevel.MINIMAL_ACCESS | AccessLevel.NO_ACCESS | AccessLevel.ADMIN>[];
    broadcastType?: string;
    dismissable?: boolean;
}
declare class BroadcastMessages<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<BroadcastMessageSchema[], C, E, P>>;
    create<E extends boolean = false>(options?: BroadcastMessageOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<BroadcastMessageSchema, C, E, void>>;
    edit<E extends boolean = false>(broadcastMessageId: number, options?: BroadcastMessageOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<BroadcastMessageSchema, C, E, void>>;
    remove<E extends boolean = false>(broadcastMessageId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(broadcastMessageId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<BroadcastMessageSchema, C, E, void>>;
}

interface CodeSuggestionSchema extends Record<string, unknown> {
    access_token: string;
    expires_in: number;
    created_at: number;
}
interface CodeCompletionSchema extends Record<string, unknown> {
    id: string;
    model: {
        engine: string;
        name: string;
    };
    object: string;
    created: number;
    choices: Array<{
        text: string;
        index: number;
        finish_reason: string;
    }>;
}
declare class CodeSuggestions<C extends boolean = false> extends BaseResource<C> {
    createAccessToken<E extends boolean = false>(options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CodeSuggestionSchema, C, E, void>>;
    generateCompletion<E extends boolean = false>(options?: BaseRequestOptions<E>): Promise<GitlabAPIResponse<CodeCompletionSchema, C, E, void>>;
}

interface ComposerV1BaseRepositorySchema extends Record<string, unknown> {
    packages?: string[];
    'metadata-url': string;
    'provider-includes': Record<string, Record<string, string>>;
    'providers-url': string;
}
interface ComposerV2BaseRepositorySchema extends Record<string, unknown> {
    packages?: string[];
    'metadata-url': string;
}
interface ComposerV1PackagesSchema extends Record<string, unknown> {
    providers: {
        [name: string]: {
            sha256: string;
        };
    };
}
interface PackageMetadata {
    name: string;
    type: string;
    license: string;
    version: string;
    dist: {
        type: string;
        url: string;
        reference: string;
        shasum: string;
    };
    source: {
        type: string;
        url: string;
        reference: string;
    };
    uid: number;
}
interface ComposerPackageMetadataSchema extends Record<string, unknown> {
    packages: {
        [name: string]: {
            [version: string]: PackageMetadata;
        };
    };
}
declare class Composer<C extends boolean = false> extends BaseResource<C> {
    create<E extends boolean = false>(projectId: string | number, options?: {
        tag?: string;
        branch?: string;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<{
        message: string;
    }, C, E, void>>;
    download<E extends boolean = false>(projectId: string | number, packageName: string, sha: string, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    showMetadata<E extends boolean = false>(groupId: string | number, packageName: string, options?: {
        sha?: string;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<ComposerPackageMetadataSchema, C, E, void>>;
    showPackages<E extends boolean = false>(groupId: string | number, sha: string, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<ComposerV1PackagesSchema, C, E, void>>;
    showBaseRepository<E extends boolean = false>(groupId: string | number, options?: {
        composerVersion: '1';
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<ComposerV1BaseRepositorySchema, C, E, void>>;
    showBaseRepository<E extends boolean = false>(groupId: string | number, options?: {
        composerVersion: '2';
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<ComposerV2BaseRepositorySchema, C, E, void>>;
}

interface PackageSnapshotSchema extends Record<string, unknown> {
    'conan_package.tgz': string;
    'conanfile.py': string;
    'conanmanifest.txt': string;
}
interface RecipeSnapshotSchema extends Record<string, unknown> {
    'conan_sources.tgz': string;
    'conanfile.py': string;
    'conanmanifest.txt': string;
}
declare class Conan<C extends boolean = false> extends BaseResource<C> {
    authenticate<E extends boolean = false>({ projectId, ...options }?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<string, C, E, void>>;
    checkCredentials<E extends boolean = false>({ projectId, ...options }?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<string, C, E, void>>;
    downloadPackageFile<E extends boolean = false>(packageName: string, packageVersion: string, packageUsername: string, packageChannel: string, conanPackageReference: string, recipeRevision: string, packageRevision: string, filename: string, { projectId, ...options }?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    downloadRecipeFile<E extends boolean = false>(packageName: string, packageVersion: string, packageUsername: string, packageChannel: string, recipeRevision: string, filename: string, { projectId, ...options }?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    showPackageUploadUrls<E extends boolean = false>(packageName: string, packageVersion: string, packageUsername: string, packageChannel: string, conanPackageReference: string, { projectId, ...options }?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<PackageSnapshotSchema, C, E, void>>;
    showPackageDownloadUrls<E extends boolean = false>(packageName: string, packageVersion: string, packageUsername: string, packageChannel: string, conanPackageReference: string, { projectId, ...options }?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<PackageSnapshotSchema, C, E, void>>;
    showPackageManifest<E extends boolean = false>(packageName: string, packageVersion: string, packageUsername: string, packageChannel: string, conanPackageReference: string, { projectId, ...options }?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<PackageSnapshotSchema, C, E, void>>;
    showPackageSnapshot<E extends boolean = false>(packageName: string, packageVersion: string, packageUsername: string, packageChannel: string, conanPackageReference: string, { projectId, ...options }?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<PackageSnapshotSchema, C, E, void>>;
    ping<E extends boolean = false>({ projectId, ...options }?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<string, C, E, void>>;
    showRecipeUploadUrls<E extends boolean = false>(packageName: string, packageVersion: string, packageUsername: string, packageChannel: string, { projectId, ...options }?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<RecipeSnapshotSchema, C, E, void>>;
    showRecipeDownloadUrls<E extends boolean = false>(packageName: string, packageVersion: string, packageUsername: string, packageChannel: string, { projectId, ...options }?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<RecipeSnapshotSchema, C, E, void>>;
    showRecipeManifest<E extends boolean = false>(packageName: string, packageVersion: string, packageUsername: string, packageChannel: string, { projectId, ...options }?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<RecipeSnapshotSchema, C, E, void>>;
    showRecipeSnapshot<E extends boolean = false>(packageName: string, packageVersion: string, packageUsername: string, packageChannel: string, { projectId, ...options }?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<RecipeSnapshotSchema, C, E, void>>;
    removePackageFile<E extends boolean = false>(packageName: string, packageVersion: string, packageUsername: string, packageChannel: string, { projectId, ...options }?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    search<E extends boolean = false>({ projectId, ...options }?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<{
        results: string[];
    }, C, E, void>>;
    uploadPackageFile<E extends boolean = false>(packageFile: {
        content: Blob;
        filename: string;
    }, packageName: string, packageVersion: string, packageUsername: string, packageChannel: string, conanPackageReference: string, recipeRevision: string, packageRevision: string, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    uploadRecipeFile<E extends boolean = false>(packageFile: {
        content: Blob;
        filename: string;
    }, packageName: string, packageVersion: string, packageUsername: string, packageChannel: string, recipeRevision: string, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface DashboardAnnotationSchema extends Record<string, unknown> {
    id: number;
    starting_at: string;
    ending_at?: null;
    dashboard_path: string;
    description: string;
    environment_id: number;
    cluster_id?: null;
}
declare class DashboardAnnotations<C extends boolean = false> extends BaseResource<C> {
    create<E extends boolean = false>(dashboardPath: string, startingAt: string, description: string, { environmentId, clusterId, ...options }?: OneOf<{
        environmentId: number;
        clusterId: number;
    }> & {
        endingAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DashboardAnnotationSchema, C, E, void>>;
}

declare class Debian<C extends boolean = false> extends BaseResource<C> {
    downloadBinaryFileIndex<E extends boolean = false>(distribution: string, component: string, architecture: string, { projectId, groupId, ...options }: OneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    downloadDistributionReleaseFile<E extends boolean = false>(distribution: string, { projectId, groupId, ...options }: OneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    downloadSignedDistributionReleaseFile<E extends boolean = false>(distribution: string, { projectId, groupId, ...options }: OneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    downloadReleaseFileSignature<E extends boolean = false>(distribution: string, { projectId, groupId, ...options }: OneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    downloadPackageFile<E extends boolean = false>(projectId: string | number, distribution: string, letter: string, packageName: string, packageVersion: string, filename: string, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    uploadPackageFile<E extends boolean = false>(projectId: string | number, packageFile: {
        content: Blob;
        filename: string;
    }, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

declare class DependencyProxy<C extends boolean = false> extends BaseResource<C> {
    remove<E extends boolean = false>(groupId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface CondensedDeployKeySchema extends Record<string, unknown> {
    id: number;
    title: string;
    key: string;
    created_at: string;
}
interface DeployKeySchema extends CondensedDeployKeySchema {
    fingerprint: string;
    fingerprint_sha256: string;
    expires_at?: string;
    can_push?: boolean;
}
interface ExpandedDeployKeySchema extends DeployKeySchema {
    projects_with_write_access?: SimpleProjectSchema[];
}
declare class DeployKeys<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>({ projectId, userId, ...options }?: OneOrNoneOf<{
        projectId: string | number;
        userId: string | number;
    }> & {
        public?: boolean;
    } & PaginationRequestOptions<P> & BaseRequestOptions<E>): Promise<GitlabAPIResponse<ExpandedDeployKeySchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, title: string, key: string, options?: {
        canPush?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DeployKeySchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, keyId: number, options?: {
        canPush?: boolean;
        title?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DeployKeySchema, C, E, void>>;
    enable<E extends boolean = false>(projectId: string | number, keyId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CondensedDeployKeySchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, keyId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, keyId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DeployKeySchema, C, E, void>>;
}

type DeployTokenScope = 'read_repository' | 'read_registry' | 'write_registry' | 'read_package_registry' | 'write_package_registry';
interface DeployTokenSchema extends Record<string, unknown> {
    id: number;
    name: string;
    username: string;
    expires_at: string;
    revoked: boolean;
    expired: boolean;
    scopes?: DeployTokenScope[];
}
interface NewDeployTokenSchema extends DeployTokenSchema {
    token: string;
}
declare class DeployTokens<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>({ projectId, groupId, ...options }?: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & {
        active?: boolean;
    } & PaginationRequestOptions<P> & BaseRequestOptions<E>): Promise<GitlabAPIResponse<DeployTokenSchema[], C, E, P>>;
    create<E extends boolean = false>(name: string, scopes: DeployTokenScope[], { projectId, groupId, ...options }?: OneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & {
        expires_at?: string;
        username?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<NewDeployTokenSchema, C, E, void>>;
    remove<E extends boolean = false>(tokenId: number, { projectId, groupId, ...options }?: OneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(tokenId: number, { projectId, groupId, ...options }?: OneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DeployTokenSchema, C, E, void>>;
}

interface AccessRequestSchema extends Record<string, unknown> {
    id: number;
    username: string;
    name: string;
    state: string;
    created_at: string;
    requested_at: string;
}
declare class ResourceAccessRequests<C extends boolean = false> extends BaseResource<C> {
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AccessRequestSchema[], C, E, P>>;
    request<E extends boolean = false>(resourceId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AccessRequestSchema, C, E, void>>;
    approve<E extends boolean = false>(resourceId: string | number, userId: number, options?: {
        accessLevel?: Exclude<AccessLevel, AccessLevel.ADMIN>;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AccessRequestSchema, C, E, void>>;
    deny<E extends boolean = false>(resourceId: string | number, userId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

type AccessTokenScopes = 'api' | 'read_api' | 'create_runner' | 'read_registry' | 'write_registry' | 'read_repository' | 'write_repository';
interface AccessTokenSchema extends Record<string, unknown> {
    user_id: number;
    scopes?: AccessTokenScopes[];
    name: string;
    expires_at: string;
    id: number;
    active: boolean;
    created_at: string;
    revoked: boolean;
    access_level: Exclude<AccessLevel, AccessLevel.NO_ACCESS | AccessLevel.MINIMAL_ACCESS | AccessLevel.ADMIN>;
}
interface AccessTokenExposedSchema extends AccessTokenSchema {
    token: string;
}
declare class ResourceAccessTokens<C extends boolean = false> extends BaseResource<C> {
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<AccessTokenSchema[], C, E, P>>;
    create<E extends boolean = false>(resourceId: string | number, name: string, scopes: AccessTokenScopes[], expiresAt: string, options?: {
        accessLevel?: Exclude<AccessLevel, AccessLevel.NO_ACCESS | AccessLevel.MINIMAL_ACCESS | AccessLevel.ADMIN>;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AccessTokenExposedSchema, C, E, void>>;
    revoke<E extends boolean = false>(resourceId: string | number, tokenId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    rotate<E extends boolean = false>(resourceId: string | number, tokenId: string | number, options?: {
        expiresAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AccessTokenExposedSchema, C, E, void>>;
    show<E extends boolean = false>(resourceId: string | number, tokenId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AccessTokenSchema, C, E, void>>;
}

interface AwardEmojiSchema extends Record<string, unknown> {
    id: number;
    name: string;
    user: SimpleUserSchema;
    created_at: string;
    updated_at: string;
    awardable_id: number;
    awardable_type: string;
}
declare class ResourceAwardEmojis<C extends boolean = false> extends BaseResource<C> {
    protected resourceType2: string;
    constructor(resourceType1: string, resourceType2: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, resourceIId: number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema[], C, E, P>>;
    award<E extends boolean = false>(resourceId: string | number, resourceIId: number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema, C, E, void>>;
    remove<E extends boolean = false>(resourceId: string | number, resourceIId: number, awardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(resourceId: string | number, resourceIId: number, awardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema, C, E, void>>;
}

declare class ResourceNoteAwardEmojis<C extends boolean = false> extends BaseResource<C> {
    protected resourceType: string;
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, resourceIId: number, noteId: number, options?: PaginationRequestOptions<P> & BaseRequestOptions<E>): Promise<GitlabAPIResponse<AwardEmojiSchema[], C, E, P>>;
    award<E extends boolean = false>(projectId: string | number, resourceIId: number, noteId: number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, resourceIId: number, noteId: number, awardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, resourceIId: number, noteId: number, awardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema, C, E, void>>;
}

interface CondensedBadgeSchema extends Record<string, unknown> {
    link_url: string;
    image_url: string;
    rendered_link_url: string;
    rendered_image_url: string;
}
interface BadgeSchema extends CondensedBadgeSchema {
    name: string;
    id: number;
    kind: 'project' | 'group';
}
interface EditBadgeOptions {
    name?: string;
    linkUrl?: string;
    imageUrl?: string;
}
declare class ResourceBadges<C extends boolean = false> extends BaseResource<C> {
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    add<E extends boolean = false>(resourceId: string | number, linkUrl: string, imageUrl: string, options?: {
        name?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<BadgeSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, options?: {
        name?: string;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<BadgeSchema[], C, E, P>>;
    edit<E extends boolean = false>(resourceId: string | number, badgeId: number, options?: EditBadgeOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<BadgeSchema, C, E, void>>;
    preview<E extends boolean = false>(resourceId: string | number, linkUrl: string, imageUrl: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CondensedBadgeSchema, C, E, void>>;
    remove<E extends boolean = false>(resourceId: string | number, badgeId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(resourceId: string | number, badgeId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<BadgeSchema, C, E, void>>;
}

type MetricType = 'deployment_frequency' | 'lead_time_for_changes' | 'time_to_restore_service' | 'change_failure_rate';
interface DORA4MetricSchema extends Record<string, unknown> {
    date: string;
    value: number;
}
declare class ResourceDORA4Metrics<C extends boolean = false> extends BaseResource<C> {
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false>(resourceId: string | number, metric: MetricType, options?: {
        startDate?: string;
        endDate?: string;
        interval?: 'all' | 'monthly' | 'daily';
        environmentTiers?: string[];
    } & BaseRequestOptions<E>): Promise<GitlabAPIResponse<DORA4MetricSchema[], C, E, void>>;
}

interface DiscussionNotePositionBaseSchema extends Record<string, unknown> {
    base_sha: string;
    start_sha: string;
    head_sha: string;
    position_type: 'text' | 'image';
    old_path?: string;
    new_path?: string;
}
interface DiscussionNotePositionTextLineSchema {
    line_code?: string;
    type?: 'new' | 'old';
    old_line?: number;
    new_line?: number;
}
interface DiscussionNotePositionLineRangeSchema {
    start?: DiscussionNotePositionTextLineSchema;
    end?: DiscussionNotePositionTextLineSchema;
}
type DiscussionNotePositionTextSchema = DiscussionNotePositionBaseSchema & {
    position_type: 'text';
    new_line?: string;
    old_line?: string;
    line_range?: DiscussionNotePositionLineRangeSchema;
};
type DiscussionNotePositionImageSchema = DiscussionNotePositionBaseSchema & {
    position_type: 'image';
    width?: string;
    height?: string;
    x?: number;
    y?: number;
};
type DiscussionNotePositionSchema = DiscussionNotePositionTextSchema | DiscussionNotePositionImageSchema;
interface DiscussionNoteSchema extends Record<string, unknown> {
    id: number;
    type: 'DiffNote' | 'DiscussionNote' | null;
    body: string;
    attachment: string | null;
    author: MappedOmit<SimpleUserSchema, 'created_at'>;
    created_at: string;
    updated_at: string;
    system: boolean;
    noteable_id: number;
    noteable_type: 'Issue' | 'Snippet' | 'Epic' | 'Commit' | 'MergeRequest';
    noteable_iid: number | null;
    resolvable: boolean;
}
interface DiscussionSchema extends Record<string, unknown> {
    id: string;
    individual_note: boolean;
    notes?: DiscussionNoteSchema[];
}
type DiscussionNotePositionOptions = Camelize<DiscussionNotePositionSchema>;
declare class ResourceDiscussions<C extends boolean = false> extends BaseResource<C> {
    protected resource2Type: string | RawPathSegment;
    constructor(resourceType: string, resource2Type: string | RawPathSegment, options: BaseResourceOptions<C>);
    addNote<E extends boolean = false>(resourceId: string | number, resource2Id: string | number, discussionId: string | number, body: string, options?: {
        createdAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionNoteSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, resource2Id: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema[], C, E, P>>;
    create<E extends boolean = false>(resourceId: string | number, resource2Id: string | number, body: string, { position, ...options }?: {
        position?: DiscussionNotePositionOptions;
        commitId?: string;
        createdAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema, C, E, void>>;
    editNote<E extends boolean = false>(resourceId: string | number, resource2Id: string | number, discussionId: string | number, noteId: number, options: Sudo & ShowExpanded<E> & {
        body?: string;
        resolved?: boolean;
    }): Promise<GitlabAPIResponse<DiscussionNoteSchema, C, E, void>>;
    removeNote<E extends boolean = false>(resourceId: string | number, resource2Id: string | number, discussionId: string | number, noteId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(resourceId: string | number, resource2Id: string | number, discussionId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema, C, E, void>>;
}

interface PipelineVariableSchema extends Record<string, unknown> {
    key: string;
    variable_type?: string;
    value: string;
}
declare class PipelineScheduleVariables<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, pipelineScheduleId: number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<PipelineVariableSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, pipelineScheduleId: number, key: string, value: string, options?: {
        variableType?: 'env_var' | 'file';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PipelineVariableSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, pipelineScheduleId: number, key: string, value: string, options?: {
        variableType?: 'env_var' | 'file';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PipelineVariableSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, pipelineScheduleId: number, key: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

type CommitablePipelineStatus = 'pending' | 'running' | 'success' | 'failed' | 'canceled';
type PipelineStatus = CommitablePipelineStatus | 'created' | 'waiting_for_resource' | 'preparing' | 'skipped' | 'manual' | 'scheduled';
interface PipelineSchema extends Record<string, unknown> {
    id: number;
    iid: number;
    project_id: number;
    sha: string;
    ref: string;
    status: string;
    source: string;
    created_at: string;
    updated_at: string;
    web_url: string;
}
interface ExpandedPipelineSchema extends PipelineSchema {
    before_sha: string;
    tag: boolean;
    yaml_errors?: unknown;
    user: MappedOmit<SimpleUserSchema, 'created_at'>;
    started_at: string;
    finished_at: string;
    committed_at?: string;
    duration: number;
    queued_duration?: unknown;
    coverage?: unknown;
    detailed_status: {
        icon: string;
        text: string;
        label: string;
        group: string;
        tooltip: string;
        has_details: boolean;
        details_path: string;
        illustration?: null;
        favicon: string;
    };
}
interface PipelineTestCaseSchema {
    status: string;
    name: string;
    classname: string;
    execution_time: number;
    system_output?: string;
    stack_trace?: string;
}
interface PipelineTestSuiteSchema {
    name: string;
    total_time: number;
    total_count: number;
    success_count: number;
    failed_count: number;
    skipped_count: number;
    error_count: number;
    test_cases?: PipelineTestCaseSchema[];
}
interface PipelineTestReportSchema extends Record<string, unknown> {
    total_time: number;
    total_count: number;
    success_count: number;
    failed_count: number;
    skipped_count: number;
    error_count: number;
    test_suites?: PipelineTestSuiteSchema[];
}
interface PipelineTestReportSummarySchema extends Record<string, unknown> {
    total: {
        time: number;
        count: number;
        success: number;
        failed: number;
        skipped: number;
        error: number;
        suite_error: string | null;
    };
    test_suites?: PipelineTestSuiteSchema[];
}
type AllPipelinesOptions = {
    scope?: 'running' | 'pending' | 'finished' | 'branches' | 'tags';
    status?: PipelineStatus;
    source?: string;
    ref?: string;
    sha?: string;
    yamlErrors?: boolean;
    username?: string;
    updatedAfter?: string;
    updatedBefore?: string;
    name?: string;
    orderBy?: 'id' | 'status' | 'updated_at' | 'user_id';
    sort?: 'asc' | 'desc';
};
declare class Pipelines<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false>(projectId: string | number, options?: AllPipelinesOptions & PaginationRequestOptions<'offset'> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PipelineSchema[], C, E, 'offset'>>;
    allVariables<E extends boolean = false>(projectId: string | number, pipelineId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PipelineVariableSchema[], C, E, void>>;
    cancel<E extends boolean = false>(projectId: string | number, pipelineId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedPipelineSchema, C, E, void>>;
    create<E extends boolean = false>(projectId: string | number, ref: string, options?: {
        variables?: PipelineVariableSchema[];
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedPipelineSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, pipelineId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    retry<E extends boolean = false>(projectId: string | number, pipelineId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedPipelineSchema, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, pipelineId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedPipelineSchema, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, pipelineId: 'latest', options?: {
        ref?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedPipelineSchema, C, E, void>>;
    showLatest<E extends boolean = false>(projectId: string | number, options?: {
        ref?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedPipelineSchema, C, E, void>>;
    showTestReport<E extends boolean = false>(projectId: string | number, pipelineId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PipelineTestReportSchema, C, E, void>>;
    showTestReportSummary<E extends boolean = false>(projectId: string | number, pipelineId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PipelineTestReportSummarySchema, C, E, void>>;
}

interface CommitAction {
    /** The action to perform */
    action: 'create' | 'delete' | 'move' | 'update' | 'chmod';
    /** Full path to the file. Ex. lib/class.rb */
    filePath: string;
    /** Original full path to the file being moved.Ex.lib / class1.rb */
    previousPath?: string;
    /** File content, required for all except delete. Optional for move */
    content?: string;
    /** text or base64. text is default. */
    encoding?: string;
    /** Last known file commit id. Will be only considered in update, move and delete actions. */
    lastCommitId?: string;
    /** When true/false enables/disables the execute flag on the file. Only considered for chmod action. */
    execute_filemode?: boolean;
}
interface CommitStatsSchema extends Record<string, unknown> {
    additions: number;
    deletions: number;
    total: number;
}
interface CondensedCommitSchema extends Record<string, unknown> {
    id: string;
    short_id: string;
    message: string;
    title: string;
    author_email: string;
    author_name: string;
    created_at: string;
}
interface CommitSchema extends CondensedCommitSchema {
    parent_ids?: string[];
    message: string;
    authored_date?: string;
    committer_name?: string;
    committer_email?: string;
    committed_date?: string;
    web_url: string;
}
interface ExpandedCommitSchema extends CommitSchema {
    last_pipeline: {
        id: number;
        ref: string;
        sha: string;
        status: string;
    };
    stats: CommitStatsSchema;
    status: string;
}
interface GPGSignatureSchema extends Record<string, unknown> {
    signature_type: 'PGP';
    verification_status: 'verified' | 'unverified';
    gpg_key_id: number;
    gpg_key_primary_keyid: string;
    gpg_key_user_name: string;
    gpg_key_user_email: string;
    gpg_key_subkey_id?: number;
    commit_source: string;
}
interface X509SignatureSchema extends Record<string, unknown> {
    signature_type: 'X509';
    verification_status: 'verified' | 'unverified';
    x509_certificate: {
        id: number;
        subject: string;
        subject_key_identifier: string;
        email: string;
        serial_number: string;
        certificate_status: string;
        x509_issuer: {
            id: number;
            subject: string;
            subject_key_identifier: string;
            crl_url: string;
        };
    };
    commit_source: string;
}
interface MissingSignatureSchema extends Record<string, unknown> {
    message: string;
}
type CommitSignatureSchema = GPGSignatureSchema | X509SignatureSchema | MissingSignatureSchema;
interface CondensedCommitCommentSchema extends Record<string, unknown> {
    note: string;
    author: MappedOmit<SimpleUserSchema, 'created_at'>;
}
interface CommitCommentSchema extends CondensedCommitCommentSchema {
    line_type: 'new' | 'old';
    path: string;
    line: number;
}
interface CommitDiffSchema extends Record<string, unknown> {
    diff: string;
    new_path: string;
    old_path: string;
    a_mode?: string;
    b_mode: string;
    new_file: boolean;
    renamed_file: boolean;
    deleted_file: boolean;
}
interface CommitStatusSchema extends Record<string, unknown> {
    status: CommitablePipelineStatus;
    created_at: string;
    started_at?: string;
    name: string;
    allow_failure: boolean;
    author: MappedOmit<SimpleUserSchema, 'created_at'>;
    description?: string;
    sha: string;
    target_url: string;
    finished_at?: string;
    id: number;
    ref: string;
}
interface CommitReferenceSchema extends Record<string, unknown> {
    type: 'branch' | 'tag' | 'all';
    name: string;
}
interface CommitSequenceSchema extends Record<string, unknown> {
    count: number;
}
interface CommitDiscussionNoteSchema extends MappedOmit<DiscussionNoteSchema, 'position'> {
    confidential?: boolean;
    commands_changes: Record<string, unknown>;
}
interface CommitDiscussionSchema extends Record<string, unknown> {
    id: string;
    individual_note: boolean;
    notes?: CommitDiscussionNoteSchema[];
}
type AllCommitsOptions = {
    refName?: string;
    since?: string;
    until?: string;
    path?: string;
    author?: string;
    all?: boolean;
    withStats?: boolean;
    firstParent?: boolean;
    order?: string;
    trailers?: boolean;
};
type CreateCommitOptions = {
    startBranch?: string;
    startSha?: string;
    startProject?: number | string;
    actions?: CommitAction[];
    authorEmail?: string;
    authorName?: string;
    stats?: boolean;
    force?: boolean;
};
type EditPipelineStatusOptions = {
    ref?: string;
    name?: string;
    context?: string;
    targetUrl?: string;
    description?: string;
    coverage?: number;
    pipelineId?: number;
};
declare class Commits<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options: AllCommitsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E> & {
        withStats: true;
    }): Promise<GitlabAPIResponse<(CommitSchema & {
        stats: CommitStatsSchema;
    })[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options: AllCommitsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E> & {
        trailers: true;
    }): Promise<GitlabAPIResponse<(CommitSchema & {
        trailers: Record<string, unknown>;
    })[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options: AllCommitsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E> & {
        withStats: true;
        trailers: true;
    }): Promise<GitlabAPIResponse<(CommitSchema & {
        trailers: Record<string, unknown>;
        stats: CommitStatsSchema;
    })[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: AllCommitsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CommitSchema[], C, E, P>>;
    allComments<E extends boolean = false>(projectId: string | number, sha: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CondensedCommitCommentSchema[], C, E, void>>;
    allDiscussions<E extends boolean = false>(projectId: string | number, sha: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema[], C, E, void>>;
    allMergeRequests<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, sha: string, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestSchema[], C, E, void>>;
    allReferences<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, sha: string, options?: {
        type?: string;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CommitReferenceSchema[], C, E, P>>;
    allStatuses<E extends boolean = false>(projectId: string | number, sha: string, options?: {
        ref?: string;
        stage?: string;
        name?: string;
        all?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CommitStatusSchema[], C, E, void>>;
    cherryPick<E extends boolean = false>(projectId: string | number, sha: string, branch: string, options?: {
        dryRun?: boolean;
        message?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CommitSchema, C, E, void>>;
    create<E extends boolean = false>(projectId: string | number, branch: string, message: string, actions?: CommitAction[], options?: CreateCommitOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedCommitSchema, C, E, void>>;
    createComment<E extends boolean = false>(projectId: string | number, sha: string, note: string, options?: {
        path?: string;
        line?: number;
        lineType?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CommitCommentSchema, C, E, void>>;
    editStatus<E extends boolean = false>(projectId: string | number, sha: string, state: CommitablePipelineStatus, options?: EditPipelineStatusOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CommitStatusSchema, C, E, void>>;
    revert<E extends boolean = false>(projectId: string | number, sha: string, branch: string, options?: {
        dryRun?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CommitSchema, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, sha: string, options?: {
        stats?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedCommitSchema, C, E, void>>;
    showDiff<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, sha: string, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CommitDiffSchema[], C, E, P>>;
    showGPGSignature<E extends boolean = false>(projectId: string | number, sha: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CommitSignatureSchema, C, E, void>>;
    showSequence<E extends boolean = false>(projectId: string | number, sha: string, options?: {
        firstParent?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CommitSequenceSchema, C, E, void>>;
}

type TodoAction = 'assigned' | 'mentioned' | 'build_failed' | 'marked' | 'approval_required' | 'unmergeable' | 'directly_addressed' | 'merge_train_removed';
type TodoType = 'Issue' | 'MergeRequest' | 'Commit' | 'Epic' | 'DesignManagement::Design' | 'AlertManagement::Alert';
type TodoState = 'pending' | 'done';
interface TodoSchema extends Record<string, unknown> {
    id: number;
    author: MappedOmit<SimpleUserSchema, 'created_at'>;
    project: Pick<SimpleProjectSchema, 'id' | 'name' | 'name_with_namespace' | 'path' | 'path_with_namespace'>;
    action_name: TodoAction;
    target_type: TodoType;
    target: Record<string, unknown>;
    target_url: string;
    body: string;
    state: TodoState;
    created_at: string;
    updated_at: string;
}
declare class TodoLists<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: {
        action?: TodoAction;
        authorId?: number;
        projectId?: string | number;
        groupId?: string | number;
        state?: TodoState;
        type?: TodoType;
    } & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<TodoSchema[], C, E, P>>;
    done<E extends boolean = false>(options: {
        todoId: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TodoSchema, C, E, void>>;
    done<E extends boolean = false>(options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface SimpleLabelSchema extends Record<string, unknown> {
    id: number;
    name: string;
    description: null | string;
    description_html: string;
    text_color: string;
    color: string;
}
interface LabelSchema extends SimpleLabelSchema {
    open_issues_count: number;
    closed_issues_count: number;
    open_merge_requests_count: number;
    subscribed: boolean;
    priority: number;
    is_project_label: boolean;
}
interface LabelCountSchema extends Record<string, unknown> {
    open_issues_count: number;
    closed_issues_count: number;
    open_merge_requests_count: number;
}
declare class ResourceLabels<C extends boolean = false> extends BaseResource<C> {
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, options: {
        withCounts: true;
        includeAncestorGroups?: boolean;
        search?: string;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<(LabelSchema & LabelCountSchema)[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, options?: {
        includeAncestorGroups?: boolean;
        search?: string;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema[], C, E, P>>;
    create<E extends boolean = false>(resourceId: string | number, labelName: string, color: string, options?: {
        description?: string;
        priority?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema, C, E, void>>;
    edit<E extends boolean = false>(resourceId: number | string, labelId: number | string, options: OneOf<{
        newName: string;
        color: string;
    }> & {
        description?: string;
        priority?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema, C, E, void>>;
    promote<E extends boolean = false>(resourceId: string | number, labelId: number | string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema, C, E, void>>;
    remove<E extends boolean = false>(resourceId: string | number, labelId: number | string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(resourceId: string | number, labelId: number | string, options?: {
        includeAncestorGroups?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema, C, E, void>>;
    subscribe<E extends boolean = false>(resourceId: string | number, labelId: number | string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema, C, E, void>>;
    unsubscribe<E extends boolean = false>(resourceId: string | number, labelId: number | string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema, C, E, void>>;
}

interface DiffRefsSchema {
    base_sha: string;
    head_sha: string;
    start_sha: string;
}
interface ReferenceSchema {
    short: string;
    relative: string;
    full: string;
}
interface TaskCompletionStatusSchema {
    count: number;
    completed_count: number;
}
interface MergeRequestDiffSchema extends Record<string, unknown> {
    old_path: string;
    new_path: string;
    a_mode: string;
    b_mode: string;
    new_file: boolean;
    renamed_file: boolean;
    deleted_file: boolean;
    diff: string;
}
interface MergeRequestDiffVersionsSchema extends Record<string, unknown> {
    id: number;
    head_commit_sha: string;
    base_commit_sha: string;
    start_commit_sha: string;
    created_at: string;
    merge_request_id: number;
    state: string;
    real_size: string;
}
interface ExpandedMergeRequestDiffVersionsSchema extends Record<string, unknown> {
    id: number;
    head_commit_sha: string;
    base_commit_sha: string;
    start_commit_sha: string;
    created_at: string;
    merge_request_id: number;
    state: string;
    real_size: string;
    commits: CommitSchema[];
    diffs: CommitDiffSchema[];
}
interface MergeRequestRebaseSchema extends Record<string, unknown> {
    rebase_in_progress?: boolean;
    merge_error?: string;
}
interface CondensedMergeRequestSchema extends Record<string, unknown> {
    id: number;
    iid: number;
    project_id: number;
    title: string;
    description: string | null;
    state: string;
    created_at: string;
    updated_at: string;
    web_url: string;
}
interface MergeRequestSchema extends CondensedMergeRequestSchema {
    merged_by: MappedOmit<SimpleUserSchema, 'created_at'> | null;
    merged_at: string | null;
    closed_by: MappedOmit<SimpleUserSchema, 'created_at'> | null;
    closed_at: string | null;
    target_branch: string;
    source_branch: string;
    user_notes_count: number;
    upvotes: number;
    downvotes: number;
    author: MappedOmit<SimpleUserSchema, 'created_at'>;
    assignees: MappedOmit<SimpleUserSchema, 'created_at'>[] | null;
    assignee: MappedOmit<SimpleUserSchema, 'created_at'> | null;
    reviewers: MappedOmit<SimpleUserSchema, 'created_at'>[] | null;
    source_project_id: number;
    target_project_id: number;
    labels: string[] | SimpleLabelSchema[];
    draft: boolean;
    work_in_progress: boolean;
    milestone: MilestoneSchema | null;
    merge_when_pipeline_succeeds: boolean;
    merge_status: 'unchecked' | 'checking' | 'can_be_merged' | 'cannot_be_merged' | 'cannot_be_merged_recheck';
    detailed_merge_status: 'blocked_status' | 'broken_status' | 'checking' | 'unchecked' | 'ci_must_pass' | 'ci_still_running' | 'discussions_not_resolved' | 'draft_status' | 'external_status_checks' | 'mergeable' | 'not_approved' | 'not_open' | 'policies_denied' | 'jira_association_missing';
    sha: string;
    merge_commit_sha: string | null;
    squash_commit_sha: string | null;
    discussion_locked: boolean | null;
    should_remove_source_branch: boolean | null;
    force_remove_source_branch: boolean;
    reference: string;
    references: ReferenceSchema;
    time_stats: TimeStatsSchema;
    squash: boolean;
    task_completion_status: TaskCompletionStatusSchema;
    prepared_at: string | null;
    has_conflicts: boolean;
    blocking_discussions_resolved: boolean;
    approvals_before_merge: number | null;
}
interface ExpandedMergeRequestSchema extends MergeRequestSchema {
    subscribed: boolean;
    changes_count: string;
    latest_build_started_at: string | null;
    latest_build_finished_at: string | null;
    first_deployed_to_production_at: null;
    pipeline: PipelineSchema | null;
    head_pipeline: ExpandedPipelineSchema | null;
    diff_refs: DiffRefsSchema;
    merge_error: string | null;
    first_contribution: boolean;
    user: {
        can_merge: boolean;
    };
    rebase_in_progress?: boolean;
}
interface MergeRequestSchemaWithExpandedLabels extends MergeRequestSchema {
    labels: SimpleLabelSchema[];
}
interface MergeRequestSchemaWithBasicLabels extends MergeRequestSchema {
    labels: string[];
}
interface MergeRequestTodoSchema extends TodoSchema {
    project: SimpleProjectSchema;
    target_type: 'MergeRequest';
    target: ExpandedMergeRequestSchema;
}
interface MergeRequestChangesSchema extends MappedOmit<MergeRequestSchema, 'has_conflicts' | 'blocking_discussions_resolved' | 'approvals_before_merge'> {
    changes: CommitDiffSchema[];
    overflow: boolean;
}
interface MergeRequestReviewerSchema extends Record<string, unknown> {
    user: MappedOmit<SimpleUserSchema, 'created_at'>;
    state: 'unreviewed' | 'requested_changes' | 'reviewed' | 'approved' | 'review_started';
    created_at: string;
}
type AllMergeRequestsOptions = {
    approvedByIds?: number[];
    approverIds?: number[];
    approved?: string;
    assigneeId?: number | 'Any' | 'None';
    authorId?: number;
    authorUsername?: string;
    createdAfter?: string;
    createdBefore?: string;
    deployedAfter?: string;
    deployedBefore?: string;
    environment?: string;
    iids?: number[];
    labels?: string;
    milestone?: string;
    myReactionEmoji?: string;
    not?: {
        labels?: string | string[];
        milestone?: string;
        authorId?: number;
        authorUsername?: string;
        assigneeId?: number;
        assigneeUsername?: string;
        reviewerId?: number;
        reviewerUsername?: string;
        myReactionEmoji?: string;
    };
    orderBy?: 'created_at' | 'updated_at';
    reviewerId?: number | 'Any' | 'None';
    reviewerUsername?: string;
    scope?: 'created_by_me' | 'assigned_to_me' | 'all';
    search?: string;
    sort?: 'asc' | 'desc';
    sourceBranch?: string;
    state?: 'opened' | 'closed' | 'locked' | 'merged';
    targetBranch?: string;
    updatedAfter?: string;
    updatedBefore?: string;
    view?: string;
    withLabelsDetails?: boolean;
    withMergeStatusRecheck?: boolean;
    wip?: string;
};
type AcceptMergeRequestOptions = {
    mergeCommitMessage?: string;
    squashCommitMessage?: string;
    squash?: boolean;
    shouldRemoveSourceBranch?: boolean;
    /**
     * @deprecated Use `autoMerge` instead. Deprecated in GitLab 17.11.
     */
    mergeWhenPipelineSucceeds?: boolean;
    sha?: string;
    autoMerge?: boolean;
};
type EditMergeRequestOptions = {
    targetBranch?: string;
    title?: string;
    assigneeId?: number;
    assigneeIds?: number[];
    reviewerIds?: number[];
    milestoneId?: number;
    addLabels?: string;
    labels?: string | Array<string>;
    removeLabels?: string;
    description?: string;
    stateEvent?: string;
    removeSourceBranch?: boolean;
    squash?: boolean;
    squashOnMerge?: boolean;
    discussionLocked?: boolean;
    allowCollaboration?: boolean;
    allowMaintainerToPush?: boolean;
};
type CreateMergeRequestOptions = {
    targetProjectId?: number;
} & Pick<EditMergeRequestOptions, 'assigneeId' | 'assigneeIds' | 'description' | 'labels' | 'title' | 'milestoneId' | 'reviewerIds' | 'removeSourceBranch' | 'allowCollaboration' | 'allowMaintainerToPush' | 'squash'>;
declare class MergeRequests<C extends boolean = false> extends BaseResource<C> {
    accept<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: AcceptMergeRequestOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedMergeRequestSchema, C, E, void>>;
    addSpentTime<E extends boolean = false>(projectId: string | number, mergerequestIId: number, duration: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TimeStatsSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & PaginationRequestOptions<P> & AllMergeRequestsOptions & {
        withLabelsDetails: true;
    }): Promise<GitlabAPIResponse<MergeRequestSchemaWithExpandedLabels[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & PaginationRequestOptions<P> & AllMergeRequestsOptions & BaseRequestOptions<E> & {
        withLabelsDetails?: false;
    }): Promise<GitlabAPIResponse<MergeRequestSchemaWithBasicLabels[], C, E, P>>;
    allDiffs<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<MergeRequestDiffSchema[], C, E, P>>;
    allCommits<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<CommitSchema[], C, E, P>>;
    allDiffVersions<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestDiffVersionsSchema[], C, E, void>>;
    allIssuesClosed<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueSchema[], C, E, void>>;
    allIssuesRelated<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueSchema[], C, E, void>>;
    allParticipants<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MappedOmit<SimpleUserSchema, 'created_at'>[], C, E, void>>;
    allPipelines<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, mergerequestIId: number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PipelineSchema[], C, E, P>>;
    cancelOnPipelineSuccess<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedMergeRequestSchema, C, E, void>>;
    create<E extends boolean = false>(projectId: string | number, sourceBranch: string, targetBranch: string, title: string, options?: CreateMergeRequestOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedMergeRequestSchema, C, E, void>>;
    createPipeline<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedPipelineSchema, C, E, void>>;
    createTodo<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestTodoSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: EditMergeRequestOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedMergeRequestSchema, C, E, void>>;
    merge<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: AcceptMergeRequestOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedMergeRequestSchema, C, E, void>>;
    mergeToDefault<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<{
        commit_id: string;
    }, C, E, void>>;
    rebase<E extends boolean = false>(projectId: string | number, mergerequestIId: number, { skipCI, ...options }?: {
        skipCI?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestRebaseSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    resetSpentTime<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TimeStatsSchema, C, E, void>>;
    resetTimeEstimate<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TimeStatsSchema, C, E, void>>;
    setTimeEstimate<E extends boolean = false>(projectId: string | number, mergerequestIId: number, duration: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TimeStatsSchema, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: {
        renderHtml?: boolean;
        includeDivergedCommitsCount?: boolean;
        includeRebaseInProgress?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedMergeRequestSchema, C, E, void>>;
    showChanges<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: {
        accessRawDiffs?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestChangesSchema, C, E, void>>;
    showDiffVersion<E extends boolean = false>(projectId: string | number, mergerequestIId: number, versionId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedMergeRequestDiffVersionsSchema, C, E, void>>;
    showTimeStats<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TimeStatsSchema, C, E, void>>;
    subscribe<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedMergeRequestSchema, C, E, void>>;
    unsubscribe<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedMergeRequestSchema, C, E, void>>;
    showReviewers<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestReviewerSchema[], C, E, void>>;
}

interface TimeStatsSchema extends Record<string, unknown> {
    time_estimate: number;
    total_time_spent: number;
    human_time_estimate: string | null;
    human_total_time_spent: string | null;
}
interface IssueSchema extends Record<string, unknown> {
    state: string;
    description: string;
    health_status?: string;
    weight?: number;
    author: MappedOmit<SimpleUserSchema, 'created_at'>;
    milestone?: MilestoneSchema;
    project_id: number;
    assignees?: MappedOmit<SimpleUserSchema, 'created_at'>[];
    type: string;
    updated_at: string;
    closed_at?: string;
    closed_by?: string;
    id: number;
    title: string;
    created_at: string;
    moved_to_id?: string;
    iid: number;
    labels: string[] | SimpleLabelSchema[];
    upvotes: number;
    downvotes: number;
    merge_requests_count: number;
    user_notes_count: number;
    due_date: string;
    web_url: string;
    references: {
        short: string;
        relative: string;
        full: string;
    };
    time_stats: TimeStatsSchema;
    has_tasks: boolean;
    task_status: string;
    confidential: boolean;
    discussion_locked: boolean;
    _links: {
        self: string;
        notes: string;
        award_emoji: string;
        project: string;
    };
    task_completion_status: {
        count: number;
        completed_count: number;
    };
    subscribed: boolean;
    epic?: {
        id: number;
        iid: number;
        title: string;
        url: string;
        group_id: number;
    };
    service_desk_reply_to?: string;
}
interface IssueSchemaWithExpandedLabels extends IssueSchema {
    labels: SimpleLabelSchema[];
}
interface IssueSchemaWithBasicLabels extends IssueSchema {
    labels: string[];
}
type AllIssuesOptions = {
    assigneeId?: number;
    assigneeUsername?: string[];
    authorId?: number;
    authorUsername?: string;
    confidential?: boolean;
    createdAfter?: string;
    createdBefore?: string;
    dueDate?: string;
    epicId?: number;
    healthStatus?: string;
    iids?: number[];
    in?: string;
    issueType?: string;
    iterationId?: number;
    iterationTitle?: string;
    labels?: string;
    milestone?: string;
    milestoneId?: string;
    myReactionEmoji?: string;
    nonArchived?: boolean;
    not?: Record<string, string>;
    orderBy?: string;
    scope?: string;
    search?: string;
    sort?: string;
    state?: string;
    updatedAfter?: string;
    updatedBefore?: string;
    weight?: number;
    withLabelsDetails?: boolean;
};
type CreateIssueOptions = {
    assigneeId?: number;
    assigneeIds?: number[];
    confidential?: boolean;
    createdAt?: string;
    description?: string;
    discussionToResolve?: string;
    dueDate?: string;
    epicId?: number;
    epicIid?: number;
    iid?: number | string;
    issueType?: string;
    labels?: string;
    mergeRequestToResolveDiscussionsOf?: number;
    milestoneId?: number;
    weight?: number;
};
type EditIssueOptions = {
    addLabels?: string;
    assigneeId?: number;
    assigneeIds?: number[];
    confidential?: boolean;
    description?: string;
    discussionLocked?: boolean;
    dueDate?: string;
    epicId?: number;
    epicIid?: number;
    issueType?: string;
    labels?: string;
    milestoneId?: number;
    removeLabels?: string;
    stateEvent?: string;
    title?: string;
    updatedAt?: string;
    weight?: number;
};
declare class Issues<C extends boolean = false> extends BaseResource<C> {
    addSpentTime<E extends boolean = false>(projectId: string | number, issueIId: number, duration: string, options?: {
        summary?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TimeStatsSchema, C, E, void>>;
    addTimeEstimate<E extends boolean = false>(projectId: string | number, issueIId: number, duration: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TimeStatsSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & PaginationRequestOptions<P> & Sudo & ShowExpanded<E> & AllIssuesOptions & {
        withLabelsDetails: true;
    }): Promise<GitlabAPIResponse<IssueSchemaWithExpandedLabels[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & PaginationRequestOptions<P> & Sudo & ShowExpanded<E> & AllIssuesOptions & BaseRequestOptions<E> & {
        withLabelsDetails?: false;
    }): Promise<GitlabAPIResponse<IssueSchemaWithBasicLabels[], C, E, P>>;
    allMetricImages<E extends boolean = false>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MetricImageSchema[], C, E, void>>;
    allParticipants<E extends boolean = false>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MappedOmit<SimpleUserSchema, 'created_at'>[], C, E, void>>;
    allRelatedMergeRequests<E extends boolean = false>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestSchema[], C, E, void>>;
    create<E extends boolean = false>(projectId: string | number, title: string, options?: CreateIssueOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueSchema, C, E, void>>;
    createTodo<E extends boolean = false>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TodoSchema, C, E, void>>;
    clone<E extends boolean = false>(projectId: string | number, issueIId: number, destinationProjectId: string | number, options?: {
        withNotes?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, issueIId: number, options?: EditIssueOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueSchema, C, E, void>>;
    editMetricImage<E extends boolean = false>(projectId: string | number, issueIId: number, imageId: number, options?: {
        url?: string;
        urlText?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MetricImageSchema, C, E, void>>;
    move<E extends boolean = false>(projectId: string | number, issueIId: number, destinationProjectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueSchema, C, E, void>>;
    promote<E extends boolean = false>(projectId: string | number, issueIId: number, body: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    removeMetricImage<E extends boolean = false>(projectId: string | number, issueIId: number, imageId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    reorder<E extends boolean = false>(projectId: string | number, issueIId: number, options?: {
        moveAfterId?: number;
        moveBeforeId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    resetSpentTime<E extends boolean = false>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TimeStatsSchema, C, E, void>>;
    resetTimeEstimate<E extends boolean = false>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TimeStatsSchema, C, E, void>>;
    show<E extends boolean = false>(issueId: number, { projectId, ...options }?: {
        projectId?: string | number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueSchema, C, E, void>>;
    subscribe<E extends boolean = false>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueSchema, C, E, void>>;
    allClosedByMergeRequestst<E extends boolean = false>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestSchema[], C, E, void>>;
    showTimeStats<E extends boolean = false>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TimeStatsSchema, C, E, void>>;
    unsubscribe<E extends boolean = false>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueSchema, C, E, void>>;
    uploadMetricImage<E extends boolean = false>(projectId: string | number, issueIId: number, metricImage: {
        content: Blob;
        filename: string;
    }, options?: {
        url?: string;
        urlText?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MetricImageSchema, C, E, void>>;
    showUserAgentDetails<E extends boolean = false>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserAgentDetailSchema, C, E, void>>;
}

interface MilestoneSchema extends Record<string, unknown> {
    id: number;
    iid: number;
    project_id: number;
    title: string;
    description: string;
    due_date?: string;
    start_date: string;
    state: string;
    updated_at: string;
    created_at: string;
    expired: boolean;
    web_url: string;
}
interface BurndownChartEventSchema extends Record<string, unknown> {
    created_at: string;
    weight: number;
    action: string;
}
interface AllMilestonesOptions {
    iids?: number[];
    state?: string;
    title?: string;
    search?: string;
    includeParentMilestones?: boolean;
    updatedBefore?: string;
    updatedAfter?: string;
}
declare class ResourceMilestones<C extends boolean = false> extends BaseResource<C> {
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, options?: AllMilestonesOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MilestoneSchema[], C, E, P>>;
    allAssignedIssues<E extends boolean = false>(resourceId: string | number, milestoneId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueSchema[], C, E, void>>;
    allAssignedMergeRequests<E extends boolean = false>(resourceId: string | number, milestoneId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestSchema[], C, E, void>>;
    allBurndownChartEvents<E extends boolean = false>(resourceId: string | number, milestoneId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<BurndownChartEventSchema[], C, E, void>>;
    create<E extends boolean = false>(resourceId: string | number, title: string, options?: {
        description?: string;
        dueDate?: string;
        startDate?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MilestoneSchema, C, E, void>>;
    edit<E extends boolean = false>(resourceId: string | number, milestoneId: number, options?: {
        title?: string;
        description?: string;
        dueDate?: string;
        startDate?: string;
        stateEvent?: 'close' | 'activate';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MilestoneSchema, C, E, void>>;
    remove<E extends boolean = false>(resourceId: string | number, milestoneId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(resourceId: string | number, milestoneId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MilestoneSchema, C, E, void>>;
}

interface IssueBoardListSchema extends Record<string, unknown> {
    id: number;
    label: Pick<LabelSchema, 'id' | 'name' | 'color' | 'description'>;
    position: number;
    max_issue_count: number;
    max_issue_weight: number;
    limit_metric?: string;
}
interface IssueBoardSchema extends Record<string, unknown> {
    id: number;
    name: string;
    milestone: Pick<MilestoneSchema, 'id' | 'title'>;
    lists?: IssueBoardListSchema[];
}
declare class ResourceIssueBoards<C extends boolean = false> extends BaseResource<C> {
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueBoardSchema[], C, E, P>>;
    allLists<E extends boolean = false>(resourceId: string | number, boardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueBoardListSchema[], C, E, void>>;
    create<E extends boolean = false>(resourceId: string | number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueBoardSchema, C, E, void>>;
    createList<E extends boolean = false>(resourceId: string | number, boardId: number, options?: OneOrNoneOf<{
        labelId: number;
        assigneeId: number;
        milestoneId: number;
    }> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueBoardListSchema, C, E, void>>;
    edit<E extends boolean = false>(resourceId: string | number, boardId: number, options?: {
        name?: string;
        assigneeId?: number;
        milestoneId?: number;
        labels?: string;
        weight?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueBoardSchema, C, E, void>>;
    editList<E extends boolean = false>(resourceId: string | number, boardId: number, listId: number, position: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueBoardListSchema, C, E, void>>;
    remove<E extends boolean = false>(resourceId: string | number, boardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    removeList<E extends boolean = false>(resourceId: string | number, boardId: number, listId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(resourceId: string | number, boardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueBoardSchema, C, E, void>>;
    showList<E extends boolean = false>(resourceId: string | number, boardId: number, listId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueBoardListSchema, C, E, void>>;
}

interface MarkdownUploadUserSchema extends Record<string, unknown> {
    id: number;
    name: string;
    username: string;
}
interface MarkdownUploadSchema extends Record<string, unknown> {
    id: number;
    size: number;
    filename: string;
    created_at: string;
    uploaded_by: MarkdownUploadUserSchema;
}
interface MarkdownUploadCreatedSchema extends Record<string, unknown> {
    id: number;
    alt: string;
    url: string;
    full_path: string;
    markdown: string;
}
declare class ResourceMarkdownUploads<C extends boolean> extends BaseResource<C> {
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, options?: Sudo & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<MarkdownUploadSchema[], C, E, P>>;
    download<E extends boolean = false>(resourceId: string | number, uploadId: string | number, options?: Sudo): Promise<GitlabAPIResponse<Blob, C, E, void>>;
    download<E extends boolean = false>(resourceId: string | number, secret: string, filename: string, options?: Sudo): Promise<GitlabAPIResponse<Blob, C, E, void>>;
    remove<E extends boolean = false>(resourceId: string | number, uploadId: string | number, options?: Sudo): Promise<GitlabAPIResponse<void, C, E, void>>;
    remove<E extends boolean = false>(resourceId: string | number, secret: string, filename: string, options?: Sudo): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface IncludeInherited {
    includeInherited?: boolean;
}
interface CondensedMemberSchema extends Record<string, unknown> {
    id: number;
    username: string;
    name: string;
    state: string;
    avatar_url: string;
    web_url: string;
}
interface SimpleMemberSchema extends CondensedMemberSchema {
    expires_at: string;
    access_level: Exclude<AccessLevel, AccessLevel.ADMIN>;
    email: string;
}
interface MemberSchema extends SimpleMemberSchema {
    group_saml_identity: {
        extern_uid: string;
        provider: string;
        saml_provider_id: number;
    };
}
type AddMemberOptions = OneOf<{
    userId: string | number;
    username: string;
}> & {
    expiresAt?: string;
    inviteSource?: string;
    tasksToBeDone?: string[];
    tasksProjectId?: number;
};
interface AllMembersOptions {
    query?: string;
    userIds?: number[];
    skipUsers?: number[];
    showSeatInfo?: boolean;
}
declare class ResourceMembers<C extends boolean = false> extends BaseResource<C> {
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    add<E extends boolean = false>(resourceId: string | number, accessLevel: Exclude<AccessLevel, AccessLevel.ADMIN>, options?: AddMemberOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MemberSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, { includeInherited, ...options }?: IncludeInherited & PaginationRequestOptions<P> & AllMembersOptions & BaseRequestOptions<E>): Promise<GitlabAPIResponse<MemberSchema[], C, E, P>>;
    edit<E extends boolean = false>(resourceId: string | number, userId: number, accessLevel: Exclude<AccessLevel, AccessLevel.ADMIN>, options?: {
        expiresAt?: string;
        memberRoleId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MemberSchema, C, E, void>>;
    show<E extends boolean = false>(resourceId: string | number, userId: number, { includeInherited, ...options }?: IncludeInherited & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MemberSchema, C, E, void>>;
    remove<E extends boolean = false>(resourceId: string | number, userId: number, options?: {
        skipSubresourceS?: boolean;
        unassignIssuables?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface NoteSchema extends Record<string, unknown> {
    id: number;
    body: string;
    attachment: string | null;
    author: MappedOmit<SimpleUserSchema, 'created_at'>;
    created_at: string;
    updated_at: string;
    system: boolean;
    noteable_id: number;
    noteable_type: 'Issue' | 'Snippet' | 'Epic' | 'Commit' | 'MergeRequest';
    noteable_iid: number;
    project_id: number;
    resolvable: boolean;
}
declare class ResourceNotes<C extends boolean = false> extends BaseResource<C> {
    protected resource2Type: string;
    constructor(resourceType: string, resource2Type: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, resource2Id: string | number, options?: {
        sort?: 'asc' | 'desc';
        orderBy?: 'created_at' | 'updated_at';
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<NoteSchema[], C, E, P>>;
    create<E extends boolean = false>(resourceId: string | number, resource2Id: string | number, body: string, options?: {
        internal?: boolean;
        createdAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<NoteSchema, C, E, void>>;
    edit<E extends boolean = false>(resourceId: string | number, resource2Id: string | number, noteId: number, options?: {
        body?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<NoteSchema, C, E, void>>;
    remove<E extends boolean = false>(resourceId: string | number, resource2Id: string | number, noteId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(resourceId: string | number, resource2Id: string | number, noteId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<CamelizedResponse<NoteSchema, C>, E, undefined>>;
}

interface TemplateSchema extends Record<string, unknown> {
    name: string;
    content: string;
}
declare class ResourceTemplates<C extends boolean = false> extends BaseResource<C> {
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TemplateSchema[], C, E, P>>;
    show<E extends boolean = false>(key: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TemplateSchema, C, E, void>>;
}

type VariableType = 'env_var' | 'file';
interface VariableSchema extends Record<string, unknown> {
    variable_type: VariableType;
    value: string;
    protected: boolean;
    masked: boolean;
    key: string;
}
type VariableFilter = Record<'environment_scope', number | string>;
declare class ResourceVariables<C extends boolean> extends BaseResource<C> {
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<VariableSchema[], C, E, P>>;
    create<E extends boolean = false>(resourceId: string | number, key: string, value: string, options?: {
        variableType?: VariableType;
        protected?: boolean;
        masked?: boolean;
        environmentScope?: string;
        description?: string;
        raw?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<VariableSchema, C, E, void>>;
    edit<E extends boolean = false>(resourceId: string | number, key: string, value: string, options?: {
        variableType?: VariableType;
        protected?: boolean;
        masked?: boolean;
        environmentScope?: string;
        filter: VariableFilter;
        description?: string;
        raw?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<VariableSchema, C, E, void>>;
    show<E extends boolean = false>(resourceId: string | number, key: string, options?: {
        filter?: VariableFilter;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<VariableSchema, C, E, void>>;
    remove<E extends boolean = false>(resourceId: string | number, key: string, options?: {
        filter?: VariableFilter;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface WikiSchema extends Record<string, unknown> {
    content: string;
    format: string;
    slug: string;
    title: string;
    encoding: string;
}
interface WikiAttachmentSchema extends Record<string, unknown> {
    file_name: string;
    file_path: string;
    branch: string;
    link: {
        url: string;
        markdown: string;
    };
}
declare class ResourceWikis<C extends boolean = false> extends BaseResource<C> {
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, options: {
        withContent: true;
    } & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<(WikiSchema & {
        content: string;
    })[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<WikiSchema[], C, E, P>>;
    create<E extends boolean = false>(resourceId: string | number, content: string, title: string, options?: {
        format?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<WikiSchema, C, E, void>>;
    edit<E extends boolean = false>(resourceId: string | number, slug: string, options?: OneOf<{
        content: string;
        title: string;
    }> & {
        format?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<WikiSchema, C, E, void>>;
    remove<E extends boolean = false>(resourceId: string | number, slug: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(resourceId: string | number, slug: string, options?: {
        renderHtml?: boolean;
        version?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<WikiSchema, C, E, void>>;
    uploadAttachment<E extends boolean = false>(resourceId: string | number, file: {
        content: Blob;
        filename: string;
    }, options?: {
        branch?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<WikiAttachmentSchema, C, E, void>>;
}

interface HookSchema extends Record<string, unknown> {
    id: number;
    url: string;
    created_at: string;
    push_events: boolean;
    tag_push_events: boolean;
    merge_requests_events: boolean;
    repository_update_events: boolean;
    enable_ssl_verification: boolean;
}
interface ExpandedHookSchema extends HookSchema {
    push_events_branch_filter: string;
    issues_events: boolean;
    confidential_issues_events: boolean;
    note_events: boolean;
    confidential_note_events: boolean;
    job_events: boolean;
    pipeline_events: boolean;
    wiki_page_events: boolean;
    deployment_events: boolean;
    releases_events: boolean;
    alert_status: string;
    disabled_until?: string;
    url_variables: string[];
}
interface AddResourceHookOptions {
    pushEvents?: boolean;
    pushEventsBranchFilter?: string;
    issuesEvents?: boolean;
    confidentialIssuesEvents?: boolean;
    mergeRequestsEvents?: boolean;
    tagPushEvents?: boolean;
    noteEvents?: boolean;
    confidentialNoteEvents?: boolean;
    jobEvents?: boolean;
    pipelineEvents?: boolean;
    wikiPageEvents?: boolean;
    deploymentEvents?: boolean;
    releasesEvents?: boolean;
    subgroupEvents?: boolean;
    enableSslVerification?: boolean;
    token?: string;
}
type EditResourceHookOptions = AddResourceHookOptions;
declare class ResourceHooks<C extends boolean = false> extends BaseResource<C> {
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    add<E extends boolean = false>(resourceId: string | number, url: string, options?: AddResourceHookOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedHookSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedHookSchema[], C, E, P>>;
    edit<E extends boolean = false>(resourceId: string | number, hookId: number, url: string, options?: EditResourceHookOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedHookSchema, C, E, void>>;
    remove<E extends boolean = false>(resourceId: string | number, hookId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(resourceId: string | number, hookId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedHookSchema, C, E, void>>;
}

interface PushRuleSchema extends Record<string, unknown> {
    id: number;
    commit_message_regex: string;
    commit_message_negative_regex: string;
    branch_name_regex: string;
    deny_delete_tag: boolean;
    created_at: string;
    member_check: boolean;
    prevent_secrets: boolean;
    author_email_regex: string;
    file_name_regex: string;
    max_file_size: number;
    commit_committer_check?: boolean;
    reject_unsigned_commits?: boolean;
}
interface CreateAndEditPushRuleOptions {
    denyDeleteTag?: boolean;
    memberCheck?: boolean;
    preventSecrets?: boolean;
    commitMessageRegex?: string;
    commitMessageNegativeRegex?: string;
    branchNameRegex?: string;
    authorEmailRegex?: string;
    fileNameRegex?: string;
    maxFileSize?: number;
    commitCommitterCheck?: boolean;
    rejectUnsignedCommits?: boolean;
}
declare class ResourcePushRules<C extends boolean = false> extends BaseResource<C> {
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    create<E extends boolean = false>(resourceId: string | number, options?: CreateAndEditPushRuleOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PushRuleSchema, C, E, void>>;
    edit<E extends boolean = false>(resourceId: string | number, options?: CreateAndEditPushRuleOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PushRuleSchema, C, E, void>>;
    remove<E extends boolean = false>(resourceId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(resourceId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PushRuleSchema, C, E, void>>;
}

interface RepositoryStorageMoveSchema extends Record<string, unknown> {
    id: number;
    created_at: string;
    state: 'initial' | 'scheduled' | 'started' | 'replicated' | 'failed' | 'finished' | 'cleanup failed';
    source_storage_name: string;
    destination_storage_name: string;
}
declare class ResourceRepositoryStorageMoves<C extends boolean = false> extends BaseResource<C> {
    protected resourceType: string;
    protected resourceTypeSingular: string;
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RepositoryStorageMoveSchema[], C, E, P>>;
    show<E extends boolean = false, P extends PaginationTypes = 'offset'>(repositoryStorageId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RepositoryStorageMoveSchema, C, E, P>>;
    schedule<E extends boolean = false, P extends PaginationTypes = 'offset'>(sourceStorageName: string, options?: {
        destinationStorageName?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RepositoryStorageMoveSchema, C, E, P>>;
}

interface InvitationSchema extends Record<string, unknown> {
    id: number;
    invite_email: string;
    created_at: string;
    access_level: Exclude<AccessLevel, AccessLevel.ADMIN>;
    expires_at: string;
    user_name: string;
    created_by_name: string;
}
declare class ResourceInvitations<C extends boolean = false> extends BaseResource<C> {
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    add<E extends boolean = false>(resourceId: string | number, accessLevel: Exclude<AccessLevel, AccessLevel.ADMIN>, options: OneOf<{
        email: string;
        userId: string;
    }> & {
        expiresAt?: string;
        inviteSource?: string;
        tasksToBeDone?: string[];
        tasksProjectId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<InvitationSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, options?: PaginationRequestOptions<P> & {
        query?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<InvitationSchema[], C, E, P>>;
    edit<E extends boolean = false>(resourceId: string | number, email: string, options?: {
        expiresAt?: string;
        accessLevel?: Exclude<AccessLevel, AccessLevel.ADMIN>;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<InvitationSchema, C, E, void>>;
    remove<E extends boolean = false>(resourceId: string | number, email: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<InvitationSchema, C, E, void>>;
}

interface IterationSchema extends Record<string, unknown> {
    id: number;
    iid: number;
    group_id: number;
    title: string;
    description: string;
    state: number;
    created_at: string;
    updated_at: string;
    due_date: string;
    start_date: string;
    web_url: string;
}
interface AllIterationsOptions {
    state?: 'opened' | 'upcoming' | 'current' | 'closed' | 'all';
    search?: string;
    includeAncestors?: boolean;
    updatedBefore?: string;
    updatedAfter?: string;
}
declare class ResourceIterations<C extends boolean = false> extends BaseResource<C> {
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, options?: AllIterationsOptions & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<IterationSchema[], C, E, P>>;
}

interface ProtectedEnvironmentAccessLevelSummarySchema {
    access_level: AccessLevel.DEVELOPER | AccessLevel.MAINTAINER | AccessLevel.ADMIN;
    access_level_description: string;
    user_id?: number;
    group_id?: number;
}
interface ProtectedEnvironmentSchema extends Record<string, unknown> {
    name: string;
    deploy_access_levels?: ProtectedEnvironmentAccessLevelSummarySchema[];
    required_approval_count: number;
}
type ProtectedEnvironmentAccessLevelEntity = OneOf<{
    userId: number;
    groupId: number;
    accessLevel: AccessLevel.DEVELOPER | AccessLevel.MAINTAINER | AccessLevel.ADMIN;
}>;
declare class ResourceProtectedEnvironments<C extends boolean = false> extends BaseResource<C> {
    constructor(resourceType: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, options?: {
        search?: string;
    } & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<ProtectedEnvironmentSchema[], C, E, P>>;
    create<E extends boolean = false>(resourceId: string | number, name: string, deployAccessLevels: ProtectedEnvironmentAccessLevelEntity[], options?: {
        requiredApprovalCount?: number;
        approvalRules?: ProtectedEnvironmentAccessLevelEntity[];
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProtectedEnvironmentSchema, C, E, void>>;
    edit<E extends boolean = false>(resourceId: string | number, name: string, options?: {
        deployAccessLevels?: ProtectedEnvironmentAccessLevelEntity[];
        requiredApprovalCount?: number;
        approvalRules?: ProtectedEnvironmentAccessLevelEntity[];
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProtectedEnvironmentSchema, C, E, void>>;
    show<E extends boolean = false>(resourceId: string | number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProtectedEnvironmentSchema, C, E, void>>;
    remove<E extends boolean = false>(resourceId: string | number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface IterationEventSchema extends Record<string, unknown> {
    id: number;
    user: MappedOmit<SimpleUserSchema, 'created_at'>;
    created_at: string;
    resource_type: 'Issue';
    resource_id: number;
    iteration: IterationSchema;
    action: 'add' | 'remove';
}
declare class ResourceIterationEvents<C extends boolean = false> extends BaseResource<C> {
    protected resource2Type: string;
    constructor(resourceType: string, resource2Type: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, resource2Id: string | number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<IterationEventSchema[], C, E, P>>;
    show<E extends boolean = false>(resourceId: string | number, resource2Id: string | number, iterationEventId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IterationEventSchema, C, E, void>>;
}

interface LabelEventSchema extends Record<string, unknown> {
    id: number;
    user: MappedOmit<SimpleUserSchema, 'created_at'>;
    created_at: string;
    resource_type: 'Issue' | 'Epic' | 'MergeRequest';
    resource_id: number;
    label: LabelSchema;
    action: 'add' | 'remove';
}
declare class ResourceLabelEvents<C extends boolean = false> extends BaseResource<C> {
    protected resource2Type: string;
    constructor(resourceType: string, resource2Type: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, resource2Id: string | number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<LabelEventSchema[], C, E, P>>;
    show<E extends boolean = false>(resourceId: string | number, resource2Id: string | number, labelEventId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelEventSchema, C, E, void>>;
}

interface MilestoneEventSchema extends Record<string, unknown> {
    id: number;
    user: MappedOmit<SimpleUserSchema, 'created_at'>;
    created_at: string;
    resource_type: 'Issue' | 'MergeRequest';
    resource_id: number;
    milestone: MilestoneSchema;
    action: 'add' | 'remove';
}
declare class ResourceMilestoneEvents<C extends boolean = false> extends BaseResource<C> {
    protected resource2Type: string;
    constructor(resourceType: string, resource2Type: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, resource2Id: string | number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<MilestoneEventSchema[], C, E, P>>;
    show<E extends boolean = false>(resourceId: string | number, resource2Id: string | number, milestoneEventId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MilestoneEventSchema, C, E, void>>;
}

interface StateEventSchema extends Record<string, unknown> {
    id: number;
    user: MappedOmit<SimpleUserSchema, 'created_at'>;
    created_at: string;
    resource_type: 'Issue';
    resource_id: number;
    action: 'add' | 'remove';
    state: 'opened' | 'closed';
}
declare class ResourceStateEvents<C extends boolean = false> extends BaseResource<C> {
    protected resource2Type: string;
    constructor(resourceType: string, resource2Type: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, resource2Id: string | number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<StateEventSchema[], C, E, P>>;
    show<E extends boolean = false>(resourceId: string | number, resource2Id: string | number, stateEventId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<StateEventSchema, C, E, void>>;
}

interface WeightEventSchema extends Record<string, unknown> {
    id: number;
    user: MappedOmit<SimpleUserSchema, 'created_at'>;
    created_at: string;
    issue_id: number;
    weight: number;
}
declare class ResourceWeightEvents<C extends boolean = false> extends BaseResource<C> {
    protected resource2Type: string;
    constructor(resourceType: string, resource2Type: string, options: BaseResourceOptions<C>);
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(resourceId: string | number, resource2Id: string | number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<WeightEventSchema[], C, E, P>>;
    show<E extends boolean = false>(resourceId: string | number, resource2Id: string | number, weightEventId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<WeightEventSchema, C, E, void>>;
}

declare class DockerfileTemplates<C extends boolean = false> extends ResourceTemplates<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface ExperimentGateSchema {
    key: string;
    value: boolean | number;
}
interface ExperimentSchema extends Record<string, unknown> {
    key: string;
    definition: {
        name: string;
        introduced_by_url: string;
        rollout_issue_url: string;
        milestone: string;
        type: string;
        group: string;
        default_enabled: boolean;
    };
    current_status: {
        state: string;
        gates?: ExperimentGateSchema[];
    };
}
declare class Experiments<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: PaginationRequestOptions<P> & BaseRequestOptions<E>): Promise<GitlabAPIResponse<ExperimentSchema[], C, E, P>>;
}

interface GeoNodeSchema extends Record<string, unknown> {
    id: number;
    name: string;
    url: string;
    internal_url: string;
    primary: boolean;
    enabled: boolean;
    current: boolean;
    files_max_capacity: number;
    repos_max_capacity: number;
    verification_max_capacity: number;
    selective_sync_type: string;
    selective_sync_shards?: string[];
    selective_sync_namespace_ids?: number[];
    minimum_reverification_interval: number;
    container_repositories_max_capacity: number;
    sync_object_storage: boolean;
    clone_protocol: string;
    web_edit_url: string;
    web_geo_projects_url: string;
    _links: {
        self: string;
        status: string;
        repair: string;
    };
}
interface GeoNodeFailureSchema extends Record<string, unknown> {
    project_id: number;
    last_repository_synced_at: string;
    last_repository_successful_sync_at: string;
    last_wiki_synced_at: string;
    last_wiki_successful_sync_at: string;
    repository_retry_count?: number;
    wiki_retry_count: number;
    last_repository_sync_failure?: string;
    last_wiki_sync_failure: string;
    last_repository_verification_failure: string;
    last_wiki_verification_failure: string;
    repository_verification_checksum_sha: string;
    wiki_verification_checksum_sha: string;
    repository_checksum_mismatch: boolean;
    wiki_checksum_mismatch: boolean;
}
interface GeoNodeStatusSchema extends Record<string, unknown> {
    geo_node_id: number;
    healthy: boolean;
    health: string;
    health_status: string;
    missing_oauth_application: boolean;
    attachments_count: number;
    attachments_synced_count?: number;
    attachments_failed_count?: number;
    attachments_synced_missing_on_primary_count: number;
    attachments_synced_in_percentage: string;
    db_replication_lag_seconds?: number;
    lfs_objects_count: number;
    lfs_objects_synced_count?: number;
    lfs_objects_failed_count?: number;
    lfs_objects_synced_missing_on_primary_count: number;
    lfs_objects_synced_in_percentage: string;
    job_artifacts_count: number;
    job_artifacts_synced_count?: number;
    job_artifacts_failed_count?: number;
    job_artifacts_synced_missing_on_primary_count: number;
    job_artifacts_synced_in_percentage: string;
    container_repositories_count: number;
    container_repositories_synced_count?: number;
    container_repositories_failed_count?: number;
    container_repositories_synced_in_percentage: string;
    design_repositories_count: number;
    design_repositories_synced_count?: number;
    design_repositories_failed_count?: number;
    design_repositories_synced_in_percentage: string;
    projects_count: number;
    repositories_count: number;
    repositories_failed_count?: number;
    repositories_synced_count?: number;
    repositories_synced_in_percentage: string;
    wikis_count: number;
    wikis_failed_count?: number;
    wikis_synced_count?: number;
    wikis_synced_in_percentage: string;
    replication_slots_count: number;
    replication_slots_used_count: number;
    replication_slots_used_in_percentage: string;
    replication_slots_max_retained_wal_bytes: number;
    repositories_checked_count: number;
    repositories_checked_failed_count: number;
    repositories_checked_in_percentage: string;
    repositories_checksummed_count: number;
    repositories_checksum_failed_count: number;
    repositories_checksummed_in_percentage: string;
    wikis_checksummed_count: number;
    wikis_checksum_failed_count: number;
    wikis_checksummed_in_percentage: string;
    repositories_verified_count: number;
    repositories_verification_failed_count: number;
    repositories_verified_in_percentage: string;
    repositories_checksum_mismatch_count: number;
    wikis_verified_count: number;
    wikis_verification_failed_count: number;
    wikis_verified_in_percentage: string;
    wikis_checksum_mismatch_count: number;
    repositories_retrying_verification_count: number;
    wikis_retrying_verification_count: number;
    last_event_id: number;
    last_event_timestamp: number;
    cursor_last_event_id?: number;
    cursor_last_event_timestamp: number;
    last_successful_status_check_timestamp: number;
    version: string;
    revision: string;
    package_files_count: number;
    package_files_checksummed_count: number;
    package_files_checksum_failed_count: number;
    package_files_registry_count: number;
    package_files_synced_count: number;
    package_files_failed_count: number;
    snippet_repositories_count: number;
    snippet_repositories_checksummed_count: number;
    snippet_repositories_checksum_failed_count: number;
    snippet_repositories_registry_count: number;
    snippet_repositories_synced_count: number;
    snippet_repositories_failed_count: number;
    group_wiki_repositories_checksummed_count: number;
    group_wiki_repositories_checksum_failed_count: number;
    group_wiki_repositories_registry_count: number;
    group_wiki_repositories_synced_count: number;
    group_wiki_repositories_failed_count: number;
}
type CreateGeoNodeOptions = {
    primary?: boolean;
    enabled?: boolean;
    internalUrl?: string;
    filesMaxCapacity?: number;
    reposMaxCapacity?: number;
    verificationMaxCapacity?: number;
    containerRepositoriesMaxCapacity?: number;
    syncObjectStorage?: boolean;
    selectiveSyncType?: 'namespaces' | 'shards';
    selectiveSyncShards?: string[];
    selectiveSyncNamespaceIds?: number[];
    minimumReverificationInterval?: number;
};
type EditGeoNodeOptions = {
    enabled?: boolean;
    name?: string;
    url?: string;
    internalUrl?: string;
    filesMaxCapacity?: number;
    reposMaxCapacity?: number;
    verificationMaxCapacity?: number;
    containerRepositoriesMaxCapacity?: number;
    syncObjectStorage?: boolean;
    selectiveSyncType?: 'namespaces' | 'shards';
    selectiveSyncShards?: string[];
    selectiveSyncNamespaceIds?: number[];
    minimumReverificationInterval?: number;
};
declare class GeoNodes<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GeoNodeSchema[], C, E, P>>;
    allStatuses<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GeoNodeStatusSchema[], C, E, P>>;
    allFailures<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GeoNodeFailureSchema[], C, E, P>>;
    create<E extends boolean = false>(name: string, url: string, options?: CreateGeoNodeOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GeoNodeSchema, C, E, void>>;
    edit<E extends boolean = false>(geonodeId: number, options?: EditGeoNodeOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GeoNodeSchema, C, E, void>>;
    repair<E extends boolean = false>(geonodeId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GeoNodeSchema, C, E, void>>;
    remove<E extends boolean = false>(geonodeId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(geonodeId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GeoNodeSchema, C, E, void>>;
    showStatus<E extends boolean = false>(geonodeId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GeoNodeStatusSchema, C, E, void>>;
}

interface GeoSiteSchema extends Record<string, unknown> {
    id: number;
    name: string;
    url: string;
    internal_url: string;
    primary: boolean;
    enabled: boolean;
    current: boolean;
    files_max_capacity: number;
    repos_max_capacity: number;
    verification_max_capacity: number;
    selective_sync_type: string;
    selective_sync_shards?: string[];
    selective_sync_namespace_ids?: number[];
    minimum_reverification_interval: number;
    sync_object_storage: boolean;
    web_edit_url: string;
    web_geo_projects_url: string;
    _links: {
        self: string;
        status: string;
        repair: string;
    };
}
interface GeoSiteStatusSchema extends Record<string, unknown> {
    geo_node_id: number;
    repository_verification_enabled: boolean;
    repositories_replication_enabled?: boolean;
    repositories_synced_count: number | null;
    repositories_failed_count: number | null;
    repositories_verified_count: number | null;
    repositories_verification_failed_count: number | null;
    repositories_verification_total_count: number | null;
    job_artifacts_synced_missing_on_primary_count: number | null;
    repositories_checksummed_count: number | null;
    repositories_checksum_failed_count: number | null;
    repositories_checksum_mismatch_count: number | null;
    repositories_checksum_total_count: number | null;
    repositories_retrying_verification_count: number | null;
    projects_count: number | null;
    container_repositories_replication_enabled: boolean | null;
    lfs_objects_count: number;
    lfs_objects_checksum_total_count: number;
    lfs_objects_checksummed_count: number;
    lfs_objects_checksum_failed_count: number;
    lfs_objects_synced_count: number | null;
    lfs_objects_failed_count: number | null;
    lfs_objects_registry_count: number | null;
    lfs_objects_verification_total_count: number | null;
    lfs_objects_verified_count: number | null;
    lfs_objects_verification_failed_count?: number | null;
    merge_request_diffs_count: number;
    merge_request_diffs_checksum_total_count: number;
    merge_request_diffs_checksummed_count: number;
    merge_request_diffs_checksum_failed_count: number;
    merge_request_diffs_synced_count: number | null;
    merge_request_diffs_failed_count: number | null;
    merge_request_diffs_registry_count: number | null;
    merge_request_diffs_verification_total_count: number | null;
    merge_request_diffs_verified_count: number | null;
    merge_request_diffs_verification_failed_count: number | null;
    package_files_count: number;
    package_files_checksum_total_count: number;
    package_files_checksummed_count: number;
    package_files_checksum_failed_count: number;
    package_files_synced_count: number | null;
    package_files_failed_count: number | null;
    package_files_registry_count: number | null;
    package_files_verification_total_count: number | null;
    package_files_verified_count: number | null;
    package_files_verification_failed_count: number | null;
    terraform_state_versions_count: number;
    terraform_state_versions_checksum_total_count: number;
    terraform_state_versions_checksummed_count: number;
    terraform_state_versions_checksum_failed_count: number;
    terraform_state_versions_synced_count: number | null;
    terraform_state_versions_failed_count: number | null;
    terraform_state_versions_registry_count: number | null;
    terraform_state_versions_verification_total_count: number | null;
    terraform_state_versions_verified_count: number | null;
    terraform_state_versions_verification_failed_count: number | null;
    snippet_repositories_count: number;
    snippet_repositories_checksum_total_count: number;
    snippet_repositories_checksummed_count: number;
    snippet_repositories_checksum_failed_count: number;
    snippet_repositories_synced_count: number | null;
    snippet_repositories_failed_count: number | null;
    snippet_repositories_registry_count: number | null;
    snippet_repositories_verification_total_count: number | null;
    snippet_repositories_verified_count: number | null;
    snippet_repositories_verification_failed_count: number | null;
    group_wiki_repositories_count: number;
    group_wiki_repositories_checksum_total_count: number | null;
    group_wiki_repositories_checksummed_count: number | null;
    group_wiki_repositories_checksum_failed_count: number | null;
    group_wiki_repositories_synced_count: number | null;
    group_wiki_repositories_failed_count: number | null;
    group_wiki_repositories_registry_count: number | null;
    group_wiki_repositories_verification_total_count: number | null;
    group_wiki_repositories_verified_count: number | null;
    group_wiki_repositories_verification_failed_count: number | null;
    pipeline_artifacts_count: number;
    pipeline_artifacts_checksum_total_count: number;
    pipeline_artifacts_checksummed_count: number;
    pipeline_artifacts_checksum_failed_count: number;
    pipeline_artifacts_synced_count: number | null;
    pipeline_artifacts_failed_count: number | null;
    pipeline_artifacts_registry_count: number | null;
    pipeline_artifacts_verification_total_count: number | null;
    pipeline_artifacts_verified_count: number | null;
    pipeline_artifacts_verification_failed_count: number | null;
    pages_deployments_count: number;
    pages_deployments_checksum_total_count: number;
    pages_deployments_checksummed_count: number;
    pages_deployments_checksum_failed_count: number;
    pages_deployments_synced_count: number | null;
    pages_deployments_failed_count: number | null;
    pages_deployments_registry_count: number | null;
    pages_deployments_verification_total_count: number | null;
    pages_deployments_verified_count: number | null;
    pages_deployments_verification_failed_count: number | null;
    uploads_count: number;
    uploads_checksum_total_count: number;
    uploads_checksummed_count: number;
    uploads_checksum_failed_count: number;
    uploads_synced_count: number | null;
    uploads_failed_count: number | null;
    uploads_registry_count: number | null;
    uploads_verification_total_count: number | null;
    uploads_verified_count: number | null;
    uploads_verification_failed_count: number | null;
    job_artifacts_count: number;
    job_artifacts_checksum_total_count: number;
    job_artifacts_checksummed_count: number;
    job_artifacts_checksum_failed_count: number;
    job_artifacts_synced_count: number | null;
    job_artifacts_failed_count: number | null;
    job_artifacts_registry_count: number | null;
    job_artifacts_verification_total_count: number | null;
    job_artifacts_verified_count: number | null;
    job_artifacts_verification_failed_count: number | null;
    ci_secure_files_count: number;
    ci_secure_files_checksum_total_count: number;
    ci_secure_files_checksummed_count: number;
    ci_secure_files_checksum_failed_count: number;
    ci_secure_files_synced_count: number | null;
    ci_secure_files_failed_count: number | null;
    ci_secure_files_registry_count: number | null;
    ci_secure_files_verification_total_count: number | null;
    ci_secure_files_verified_count: number | null;
    ci_secure_files_verification_failed_count: number | null;
    container_repositories_count: number;
    container_repositories_checksum_total_count: number;
    container_repositories_checksummed_count: number;
    container_repositories_checksum_failed_count: number;
    container_repositories_synced_count: number | null;
    container_repositories_failed_count: number | null;
    container_repositories_registry_count: number | null;
    container_repositories_verification_total_count: number | null;
    container_repositories_verified_count: number | null;
    container_repositories_verification_failed_count: number | null;
    dependency_proxy_blobs_count: number;
    dependency_proxy_blobs_checksum_total_count: number;
    dependency_proxy_blobs_checksummed_count: number;
    dependency_proxy_blobs_checksum_failed_count: number;
    dependency_proxy_blobs_synced_count: number | null;
    dependency_proxy_blobs_failed_count: number | null;
    dependency_proxy_blobs_registry_count: number | null;
    dependency_proxy_blobs_verification_total_count: number | null;
    dependency_proxy_blobs_verified_count: number | null;
    dependency_proxy_blobs_verification_failed_count: number | null;
    dependency_proxy_manifests_count: number;
    dependency_proxy_manifests_checksum_total_count: number;
    dependency_proxy_manifests_checksummed_count: number;
    dependency_proxy_manifests_checksum_failed_count: number;
    dependency_proxy_manifests_synced_count: number | null;
    dependency_proxy_manifests_failed_count: number | null;
    dependency_proxy_manifests_registry_count: number | null;
    dependency_proxy_manifests_verification_total_count: number | null;
    dependency_proxy_manifests_verified_count: number | null;
    dependency_proxy_manifests_verification_failed_count: number | null;
    project_wiki_repositories_count: number;
    project_wiki_repositories_checksum_total_count: number;
    project_wiki_repositories_checksummed_count: number;
    project_wiki_repositories_checksum_failed_count: number;
    project_wiki_repositories_synced_count: number | null;
    project_wiki_repositories_failed_count: number | null;
    project_wiki_repositories_registry_count: number | null;
    project_wiki_repositories_verification_total_count: number | null;
    project_wiki_repositories_verified_count: number | null;
    project_wiki_repositories_verification_failed_count: number | null;
    git_fetch_event_count_weekly: number | null;
    git_push_event_count_weekly: number | null;
    proxy_remote_requests_event_count_weekly: number | null;
    proxy_local_requests_event_count_weekly: number | null;
    repositories_synced_in_percentage: string;
    repositories_checksummed_in_percentage: string;
    repositories_verified_in_percentage: string;
    repositories_checked_in_percentage: string;
    replication_slots_used_in_percentage: string;
    lfs_objects_synced_in_percentage: string;
    lfs_objects_verified_in_percentage: string;
    merge_request_diffs_synced_in_percentage: string;
    merge_request_diffs_verified_in_percentage: string;
    package_files_synced_in_percentage: string;
    package_files_verified_in_percentage: string;
    terraform_state_versions_synced_in_percentage: string;
    terraform_state_versions_verified_in_percentage: string;
    snippet_repositories_synced_in_percentage: string;
    snippet_repositories_verified_in_percentage: string;
    group_wiki_repositories_synced_in_percentage: string;
    group_wiki_repositories_verified_in_percentage: string;
    pipeline_artifacts_synced_in_percentage: string;
    pipeline_artifacts_verified_in_percentage: string;
    pages_deployments_synced_in_percentage: string;
    pages_deployments_verified_in_percentage: string;
    uploads_synced_in_percentage: string;
    uploads_verified_in_percentage: string;
    job_artifacts_synced_in_percentage: string;
    job_artifacts_verified_in_percentage: string;
    ci_secure_files_synced_in_percentage: string;
    ci_secure_files_verified_in_percentage: string;
    container_repositories_synced_in_percentage: string;
    container_repositories_verified_in_percentage: string;
    dependency_proxy_blobs_synced_in_percentage: string;
    dependency_proxy_blobs_verified_in_percentage: string;
    dependency_proxy_manifests_synced_in_percentage: string;
    dependency_proxy_manifests_verified_in_percentage: string;
    project_wiki_repositories_synced_in_percentage: string;
    project_wiki_repositories_verified_in_percentage: string;
    repositories_count: number;
    replication_slots_count: number;
    replication_slots_used_count: number;
    healthy: boolean;
    health: string;
    health_status: string;
    missing_oauth_application: boolean;
    db_replication_lag_seconds: number | null;
    replication_slots_max_retained_wal_bytes: number;
    repositories_checked_count: number | null;
    repositories_checked_failed_count: number | null;
    last_event_id: number;
    last_event_timestamp: number;
    cursor_last_event_id: number | null;
    cursor_last_event_timestamp: number;
    last_successful_status_check_timestamp: number;
    version: string;
    revision: string;
    selective_sync_type: string | null;
    namespaces: string[] | null;
    updated_at: string;
    storage_shards_match: boolean;
    _links: {
        self: string;
        site: string;
    };
}
interface GeoSiteFailureSchema extends Record<string, unknown> {
    project_id: number;
    last_repository_synced_at: string;
    last_repository_successful_sync_at: string;
    last_wiki_synced_at: string;
    last_wiki_successful_sync_at: string;
    repository_retry_count: number | null;
    wiki_retry_count: number;
    last_repository_sync_failure: string | null;
    last_wiki_sync_failure: string;
    last_repository_verification_failure: string;
    last_wiki_verification_failure: string;
    repository_verification_checksum_sha: string;
    wiki_verification_checksum_sha: string;
    repository_checksum_mismatch: boolean;
    wiki_checksum_mismatch: boolean;
}
type CreateGeoSiteOptions = {
    primary?: boolean;
    enabled?: boolean;
    internalUrl?: string;
    filesMaxCapacity?: number;
    reposMaxCapacity?: number;
    verificationMaxCapacity?: number;
    containerRepositoriesMaxCapacity?: number;
    syncObjectStorage?: boolean;
    selectiveSyncType?: 'namespaces' | 'shards';
    selectiveSyncShards?: string[];
    selectiveSyncNamespaceIds?: number[];
    minimumReverificationInterval?: number;
};
type EditGeoSiteOptions = CreateGeoSiteOptions;
declare class GeoSites<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GeoSiteSchema[], C, E, P>>;
    allStatuses<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GeoSiteStatusSchema[], C, E, P>>;
    allFailures<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GeoSiteFailureSchema[], C, E, P>>;
    create<E extends boolean = false>(name: string, url: string, options?: CreateGeoSiteOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GeoSiteSchema, C, E, void>>;
    edit<E extends boolean = false>(geositeId: number, options?: EditGeoSiteOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GeoSiteSchema, C, E, void>>;
    repair<E extends boolean = false>(geositeId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GeoSiteSchema, C, E, void>>;
    remove<E extends boolean = false>(geositeId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(geositeId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GeoSiteSchema, C, E, void>>;
    showStatus<E extends boolean = false>(geositeId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GeoSiteStatusSchema, C, E, void>>;
}

declare class GitLabCIYMLTemplates<C extends boolean = false> extends ResourceTemplates<C> {
    constructor(options: BaseResourceOptions<C>);
}

declare class GitignoreTemplates<C extends boolean = false> extends ResourceTemplates<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface RepositoryImportStatusSchema extends Record<string, unknown> {
    id: number;
    name: string;
    full_path: string;
    full_name: string;
}
interface ExpandedRepositoryImportStatusSchema extends RepositoryImportStatusSchema {
    import_source: string;
    import_status: string;
    human_import_status_name: string;
    provider_link: string;
}
declare class Import<C extends boolean = false> extends BaseResource<C> {
    importGithubRepository<E extends boolean = false>(personalAccessToken: string, repositoryId: number, targetNamespace: string, options?: {
        newName?: string;
        githubHostname?: string;
        optionalStages?: Record<string, string>;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RepositoryImportStatusSchema, C, E, void>>;
    cancelGithubRepositoryImport<E extends boolean = false>(projectId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RepositoryImportStatusSchema, C, E, void>>;
    importGithubGists<E extends boolean = false>(personalAccessToken: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    importBitbucketServerRepository<E extends boolean = false>(bitbucketServerUrl: string, bitbucketServerUsername: string, personalAccessToken: string, bitbucketServerProject: string, bitbucketServerRepository: string, options?: {
        newName?: string;
        targetNamespace?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RepositoryImportStatusSchema, C, E, void>>;
}

interface CICDVariableSchema extends Record<string, unknown> {
    key: string;
    variable_type: string;
    value: string;
    protected: boolean;
    masked: boolean;
    raw: boolean;
}
declare class InstanceLevelCICDVariables<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false>(options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CICDVariableSchema[], C, E, void>>;
    create<E extends boolean = false>(key: string, value: string, options?: {
        variableType?: string;
        protected?: boolean;
        masked?: boolean;
        raw?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CICDVariableSchema, C, E, void>>;
    edit<E extends boolean = false>(keyId: string, value: string, options?: {
        variableType?: string;
        protected?: boolean;
        masked?: boolean;
        raw?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CICDVariableSchema, C, E, void>>;
    show<E extends boolean = false>(keyId: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CICDVariableSchema, C, E, void>>;
    remove<E extends boolean = false>(keyId: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface DeployKeyProjectsSchema extends Record<string, unknown> {
    id: number;
    deploy_key_id: number;
    project_id: number;
    created_at: string;
    updated_at: string;
    can_push: boolean;
}
interface KeySchema extends Record<string, unknown> {
    id: number;
    title: string;
    key: string;
    created_at: string;
    expires_at: string;
    usage_type?: string;
    user: ExpandedUserSchema;
    deploy_keys_projects?: DeployKeyProjectsSchema[];
}
declare class Keys<C extends boolean = false> extends BaseResource<C> {
    show<E extends boolean = false>({ keyId, fingerprint, ...options }?: OneOf<{
        keyId: number;
        fingerprint: string;
    }> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<KeySchema, C, E, void>>;
}

interface LicenseSchema extends Record<string, unknown> {
    id: number;
    plan: string;
    created_at: string;
    starts_at: string;
    expires_at: string;
    historical_max: number;
    maximum_user_count: number;
    expired: boolean;
    overage: number;
    user_limit: number;
    active_users: number;
    licensee: {
        Name: string;
    };
    add_ons: {
        GitLab_FileLocks: number;
        GitLab_Auditor_User: number;
    };
}
declare class License<C extends boolean = false> extends BaseResource<C> {
    add<E extends boolean = false>(license: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LicenseSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<LicenseSchema[], C, E, P>>;
    show<E extends boolean = false>(options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LicenseSchema, C, E, void>>;
    remove<E extends boolean = false>(licenceId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LicenseSchema, C, E, void>>;
    recalculateBillableUsers<E extends boolean = false>(licenceId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<{
        success: boolean;
    }, C, E, void>>;
}

interface LicenseTemplateSchema extends Record<string, unknown> {
    key: string;
    name: string;
    nickname?: string;
    featured: boolean;
    html_url: string;
    source_url: string;
    description: string;
    conditions?: string[];
    permissions?: string[];
    limitations?: string[];
    content: string;
}
interface LicenseTemplates<C extends boolean = false> extends ResourceTemplates<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: {
        popular?: boolean;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LicenseTemplateSchema[], C, E, P>>;
    show<E extends boolean = false>(key: string | number, options?: {
        project?: string;
        fullName?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LicenseTemplateSchema, C, E, void>>;
}
declare class LicenseTemplates<C extends boolean = false> extends ResourceTemplates<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface LintSchema extends Record<string, unknown> {
    valid: boolean;
    merged_yaml?: string;
    errors?: string[];
    warnings?: string[];
}
declare class Lint<C extends boolean = false> extends BaseResource<C> {
    check<E extends boolean = false>(projectId: string | number, options: {
        ref?: string;
        includeJobs?: boolean;
        dryRun?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LintSchema, C, E, void>>;
    lint<E extends boolean = false>(projectId: string | number, content: string, options?: {
        ref?: string;
        includeJobs?: boolean;
        dryRun?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LintSchema, C, E, void>>;
}

interface MarkdownSchema extends Record<string, unknown> {
    html: string;
}
declare class Markdown<C extends boolean = false> extends BaseResource<C> {
    render<E extends boolean = false>(text: string, options?: {
        gfm?: string;
        project?: string | number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MarkdownSchema, C, E, void>>;
}

declare class Maven<C extends boolean = false> extends BaseResource<C> {
    downloadPackageFile<E extends boolean = false>(path: string, filename: string, { projectId, groupId, ...options }: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    uploadPackageFile<E extends boolean = false>(projectId: string | number, path: string, packageFile: {
        content: Blob;
        filename: string;
    }, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface MetadataSchema extends Record<string, unknown> {
    version: string;
    revision: string;
    kas: {
        enabled: boolean;
        externalUrl: string;
        version: string;
    };
    enterprise: boolean;
}
declare class Metadata<C extends boolean = false> extends BaseResource<C> {
    show<E extends boolean = false>(options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MetadataSchema, C, E, void>>;
}

type MigrationStatus = 'created' | 'started' | 'finished' | 'failed';
interface MigrationEntityFailure {
    pipeline_class: string;
    pipeline_step: string;
    exception_class: string;
    correlation_id_value: string;
    created_at: string;
}
interface MigrationEntitySchema extends Record<string, unknown> {
    id: number;
    bulk_import_id: number;
    status: string;
    source_full_path: string;
    destination_name: string;
    destination_namespace: string;
    parent_id?: number;
    namespace_id: number;
    project_id?: string | number;
    created_at: string;
    updated_at: string;
    failures?: MigrationEntityFailure[];
}
interface MigrationStatusSchema extends Record<string, unknown> {
    id: number;
    status: string;
    source_type: string;
    created_at: string;
    updated_at: string;
}
type MigrationEntityOptions = {
    sourceType: string;
    sourceFullPath: string;
    destinationSlug: string;
    destinationNamespace: string;
};
declare class Migrations<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: {
        status?: MigrationStatus;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MigrationStatusSchema[], C, E, P>>;
    create<E extends boolean = false>(configuration: {
        url: string;
        access_token: string;
    }, entities: MigrationEntityOptions[], options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MigrationStatusSchema, C, E, void>>;
    allEntities<E extends boolean = false>({ bulkImportId, ...options }?: {
        status?: MigrationStatus;
        bulkImportId?: number;
    } & PaginationRequestOptions<'offset'> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MigrationEntitySchema[], C, E, 'offset'>>;
    show<E extends boolean = false>(bulkImportId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MigrationStatusSchema, C, E, void>>;
    showEntity<E extends boolean = false>(bulkImportId: number, entitityId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MigrationEntitySchema, C, E, void>>;
}

interface NPMVersionSchema {
    name: string;
    version: string;
    dist: {
        shasum: string;
        tarball: string;
    };
}
interface NPMPackageMetadataSchema extends Record<string, unknown> {
    name: string;
    versions: {
        [version: string]: NPMVersionSchema;
    };
    'dist-tags': {
        [tag: string]: string;
    };
}
declare class NPM<C extends boolean = false> extends BaseResource<C> {
    downloadPackageFile<E extends boolean = false>(projectId: string | number, packageName: string, filename: string, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    removeDistTag<E extends boolean = false>(packageName: string, tag: string, options?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    setDistTag<E extends boolean = false>(packageName: string, tag: string, options?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    showDistTags<E extends boolean = false>(packageName: string, options?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<Record<string, string>, C, E, void>>;
    showMetadata<E extends boolean = false>(packageName: string, options?: {
        projectId?: string | number;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<NPMPackageMetadataSchema, C, E, void>>;
    uploadPackageFile<E extends boolean = false>(projectId: string | number, packageName: string, versions: string, metadata: Record<string, unknown>, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<Record<string, unknown>, C, E, void>>;
}

type NotificationSettingLevel = 'disabled' | 'participating' | 'watch' | 'global' | 'mention' | 'custom';
type CustomSettingLevelEmailEvents = 'new_note' | 'new_issue' | 'reopen_issue' | 'close_issue' | 'reassign_issue' | 'issue_due' | 'new_merge_request' | 'push_to_merge_request' | 'reopen_merge_request' | 'close_merge_request' | 'reassign_merge_request' | 'merge_merge_request' | 'failed_pipeline' | 'fixed_pipeline' | 'success_pipeline' | 'moved_project' | 'merge_when_pipeline_succeeds' | 'new_epic ';
interface NotificationSettingSchema extends Record<string, unknown> {
    level: NotificationSettingLevel;
    notification_email?: string;
}
type EditNotificationSettingsOptions = {
    level?: string;
    notificationEmail?: string;
    newNote?: boolean;
    newIssue?: boolean;
    reopenIssue?: boolean;
    closeIssue?: boolean;
    reassignIssue?: boolean;
    issueDue?: boolean;
    newMergeRequest?: boolean;
    pushToMergeRequest?: boolean;
    reopenMergeRequest?: boolean;
    closeMergeRequest?: boolean;
    reassignMergeRequest?: boolean;
    mergeMergeRequest?: boolean;
    failedPipeline?: boolean;
    fixedPipeline?: boolean;
    successPipeline?: boolean;
    movedProject?: boolean;
    mergeWhenPipelineSucceeds?: boolean;
    newEpic?: boolean;
};
declare class NotificationSettings<C extends boolean = false> extends BaseResource<C> {
    edit<E extends boolean = false>({ groupId, projectId, ...options }?: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & EditNotificationSettingsOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<NotificationSettingSchema, C, E, void>>;
    show<E extends boolean = false>({ groupId, projectId, ...options }?: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<NotificationSettingSchema, C, E, void>>;
}

interface NuGetPackageIndexSchema extends Record<string, unknown> {
    versions: string[];
}
interface NuGetResourceSchema extends Record<string, unknown> {
    '@id': string;
    '@type': string;
    comment: string;
}
interface NuGetServiceIndexSchema extends Record<string, unknown> {
    version: string;
    resources: NuGetResourceSchema[];
}
interface NuGetServiceMetadataVersionSchema extends Record<string, unknown> {
    '@id': string;
    packageContent: string;
    catalogEntry: {
        '@id': string;
        authors: string;
        dependencyGroups: unknown[];
        id: string;
        version: string;
        tags: string;
        packageContent: string;
        summary: string;
    };
}
interface NuGetServiceMetadataItemSchema extends Record<string, unknown> {
    '@id': string;
    lower: string;
    upper: string;
    count: number;
    items: NuGetServiceMetadataVersionSchema;
}
interface NuGetServiceMetadataSchema extends Record<string, unknown> {
    count: number;
    items: NuGetServiceMetadataItemSchema[];
    resources: NuGetResourceSchema[];
}
interface NuGetSearchResultSchema extends Record<string, unknown> {
    '@type': string;
    authors: string;
    id: string;
    title: string;
    version: string;
    verified: boolean;
    summary: string;
    totalDownloads: number;
    versions: {
        '@id': string;
        version: string;
        download: number;
    }[];
}
interface NuGetSearchResultsSchema extends Record<string, unknown> {
    totalHits: number;
    data: NuGetSearchResultSchema[];
}
declare class NuGet<C extends boolean = false> extends BaseResource<C> {
    downloadPackageFile<E extends boolean = false>(projectId: string | number, packageName: string, packageVersion: string, filename: string, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    search<E extends boolean = false>(q: string, { projectId, groupId, ...options }: OneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & {
        skip?: number;
        take?: number;
        prerelease?: boolean;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<NuGetSearchResultsSchema, C, E, void>>;
    showMetadata<E extends boolean = false>(packageName: string, { projectId, groupId, ...options }: OneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & ShowExpanded<E>): Promise<GitlabAPIResponse<NuGetServiceMetadataSchema, C, E, void>>;
    showPackageIndex<E extends boolean = false>(projectId: string | number, packageName: string, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<NuGetPackageIndexSchema, C, E, void>>;
    showServiceIndex<E extends boolean = false>({ projectId, groupId, ...options }: OneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & ShowExpanded<E>): Promise<GitlabAPIResponse<NuGetServiceIndexSchema, C, E, void>>;
    showVersionMetadata<E extends boolean = false>(packageName: string, packageVersion: string, { projectId, groupId, ...options }: OneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & ShowExpanded<E>): Promise<GitlabAPIResponse<NuGetServiceMetadataVersionSchema, C, E, void>>;
    uploadPackageFile<E extends boolean = false>(projectId: string | number, packageName: string, packageVersion: string, packageFile: {
        content: Blob;
        filename: string;
    }, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    uploadSymbolPackage<E extends boolean = false>(projectId: string | number, packageName: string, packageVersion: string, packageFile: {
        content: Blob;
        filename: string;
    }, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

declare class PyPI<C extends boolean = false> extends BaseResource<C> {
    downloadPackageFile<E extends boolean = false>(sha: string, fileIdentifier: string, { projectId, groupId, ...options }?: OneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    showPackageDescriptor<E extends boolean = false>(packageName: string, { projectId, groupId, ...options }: OneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & ShowExpanded<E>): Promise<GitlabAPIResponse<string, C, E, void>>;
    uploadPackageFile<E extends boolean = false>(projectId: string | number, packageFile: {
        content: Blob;
        filename: string;
    }, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

declare class RubyGems<C extends boolean = false> extends BaseResource<C> {
    allDependencies<E extends boolean = false>(projectId: string, options?: {
        gems?: string;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<string, void, E, void>>;
    downloadGemFile<E extends boolean = false>(projectId: string, fileName: string, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    uploadGemFile<E extends boolean = false>(projectId: string | number, packageFile: {
        content: Blob;
        filename: string;
    }, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

type SnippetVisibility = 'private' | 'public' | 'internal';
interface SimpleSnippetSchema extends Record<string, unknown> {
    id: number;
    title: string;
    file_name: string;
    description?: string;
    author: MappedOmit<SimpleUserSchema, 'created_at'>;
    updated_at: string;
    created_at: string;
    project_id?: string | number;
    web_url: string;
}
interface SnippetSchema extends SimpleSnippetSchema {
    visibility: SnippetVisibility;
    raw_url: string;
}
interface ExpandedSnippetSchema extends SnippetSchema {
    expires_at?: string;
    ssh_url_to_repo: string;
    http_url_to_repo: string;
    files?: {
        path: string;
        raw_url: string;
    }[];
}
type CreateSnippetOptions = {
    description?: string;
    visibility?: SnippetVisibility;
    files?: {
        content: string;
        filePath: string;
    }[];
};
type EditSnippetOptions = {
    description?: string;
    visibility?: SnippetVisibility;
    files?: {
        content?: string;
        filePath?: string;
        previousPath?: string;
        action: 'create' | 'update' | 'delete' | 'move';
    }[];
};
declare class Snippets<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false>({ public: ppublic, ...options }?: {
        public?: boolean;
        createdAfter?: string;
        createdBefore?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SnippetSchema[], C, E, void>>;
    create<E extends boolean = false>(title: string, options?: CreateSnippetOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<CamelizedResponse<ExpandedSnippetSchema, C>, E, undefined>>;
    edit<E extends boolean = false>(snippetId: number, options?: EditSnippetOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<CamelizedResponse<ExpandedSnippetSchema, C>, E, undefined>>;
    remove<E extends boolean = false>(snippetId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<void, E, undefined>>;
    show<E extends boolean = false>(snippetId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SnippetSchema, C, E, void>>;
    showContent<E extends boolean = false>(snippetId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<string, C, E, void>>;
    showRepositoryFileContent<E extends boolean = false>(snippetId: number, ref: string, filePath: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<string, C, E, void>>;
    showUserAgentDetails<E extends boolean = false>(snippetId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<CamelizedResponse<UserAgentDetailSchema, C>, E, undefined>>;
}

interface BlobSchema extends Record<string, unknown> {
    id: number;
    basename: string;
    data: string;
    path: string;
    filename: string;
    ref: string;
    startline: number;
    project_id: number;
}
interface SearchCommitSchema extends CommitSchema {
    projectId: string | number;
}
type SearchScopes = 'projects' | 'issues' | 'merge_requests' | 'milestones' | 'snippet_titles' | 'wiki_blobs' | 'commits' | 'blobs' | 'notes' | 'users';
type AllSearchOptions = {
    orderBy?: 'created_at';
    state?: 'issues' | 'merge_requests';
    confidential?: boolean;
};
declare class Search<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(scope: 'users', search: string, options?: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & AllSearchOptions & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<SimpleUserSchema[], C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(scope: 'notes', search: string, options?: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & AllSearchOptions & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<NoteSchema[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(scope: 'blobs', search: string, options?: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & AllSearchOptions & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<BlobSchema[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(scope: 'commits', search: string, options?: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & AllSearchOptions & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<SearchCommitSchema[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(scope: 'wiki_blobs', search: string, options?: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & AllSearchOptions & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<BlobSchema[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(scope: 'snippet_titles', search: string, options?: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & AllSearchOptions & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<SimpleSnippetSchema[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(scope: 'milestones', search: string, options?: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & AllSearchOptions & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<MilestoneSchema[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(scope: 'merge_requests', search: string, options?: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & AllSearchOptions & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<MergeRequestSchema[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(scope: 'issues', search: string, options?: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & AllSearchOptions & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<IssueSchema[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(scope: 'projects', search: string, options?: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & AllSearchOptions & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<ProjectSchema[], C, E, P>>;
}

interface SearchMigrationSchema extends Record<string, unknown> {
    version: number;
    name: string;
    started_at: string;
    completed_at: string;
    completed: boolean;
    obsolete: boolean;
    migration_state: {
        task_id: string | null;
        pause_indexing?: boolean;
        slice?: number;
        max_slices?: number;
        retry_attempt?: number;
        permutation_idx?: number;
        documents_remaining?: number;
        documents_remaining_for_permutation?: number;
    };
}
declare class SearchAdmin<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<SearchMigrationSchema, C, E, void>>;
    show<E extends boolean = false, P extends PaginationTypes = 'offset'>(versionOrName: string, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<SearchMigrationSchema, C, E, void>>;
}

interface ServiceAccountSchema extends Record<string, unknown> {
    id: number;
    name: string;
    username: string;
}
declare class ServiceAccounts<C extends boolean = false> extends BaseResource<C> {
    create<E extends boolean = false>(options?: {
        name?: string;
        username?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ServiceAccountSchema, C, E, void>>;
}

declare class ServiceData<C extends boolean = false> extends BaseResource<C> {
    showMetricDefinitions<E extends boolean = false>(options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<string, C, E, void>>;
    showServicePingSQLQueries<E extends boolean = false>(options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<Record<string, unknown>, C, E, void>>;
    showUsageDataNonSQLMetrics<E extends boolean = false>(options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<Record<string, unknown>, C, E, void>>;
}

interface ProcessMetricSchema {
    hostname: string;
    pid: number;
    tag: string;
    started_at: string;
    queues?: string[];
    labels?: string[];
    concurrency: number;
    busy: number;
}
interface SidekickProcessMetricsSchema extends Record<string, unknown> {
    processes?: ProcessMetricSchema[];
}
interface SidekickQueueMetricsSchema extends Record<string, unknown> {
    queues: {
        default: {
            backlog: number;
            latency: number;
        };
    };
}
interface SidekickJobStatsSchema extends Record<string, unknown> {
    jobs: {
        processed: number;
        failed: number;
        enqueued: number;
        dead: number;
    };
}
type SidekickCompoundMetricsSchema = SidekickJobStatsSchema & SidekickQueueMetricsSchema & SidekickProcessMetricsSchema;
declare class SidekiqMetrics<C extends boolean = false> extends BaseResource<C> {
    queueMetrics<E extends boolean = false>(): Promise<GitlabAPIResponse<SidekickQueueMetricsSchema, C, E, void>>;
    processMetrics<E extends boolean = false>(): Promise<GitlabAPIResponse<SidekickProcessMetricsSchema, C, E, void>>;
    jobStats<E extends boolean = false>(): Promise<GitlabAPIResponse<SidekickJobStatsSchema, C, E, void>>;
    compoundMetrics<E extends boolean = false>(): Promise<GitlabAPIResponse<SidekickCompoundMetricsSchema, C, E, void>>;
}

interface SidekiqQueueStatus extends Record<string, unknown> {
    completed: boolean;
    deleted_jobs: number;
    queue_size: number;
}
type RemoveSidekiqQueueOptions = {
    user?: string;
    project?: string;
    rootNamespace?: string;
    subscriptionPlan?: string;
    callerId?: string;
    featureCategory?: string;
    workerClass?: string;
};
declare class SidekiqQueues<C extends boolean = false> extends BaseResource<C> {
    remove<E extends boolean = false>(queueName: string, options?: RemoveSidekiqQueueOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SidekiqQueueStatus, C, E, void>>;
}

interface SnippetRepositoryStorageMoveSchema extends RepositoryStorageMoveSchema {
    snippet: Pick<SnippetSchema, 'id' | 'title' | 'description' | 'visibility' | 'updated_at' | 'created_at' | 'web_url' | 'raw_url' | 'ssh_url_to_repo' | 'http_url_to_repo'>;
}
interface SnippetRepositoryStorageMoves<C extends boolean = false> extends ResourceRepositoryStorageMoves<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: {
        snippetId?: string | number;
    } & PaginationRequestOptions<P> & BaseRequestOptions<E>): Promise<GitlabAPIResponse<SnippetRepositoryStorageMoveSchema[], C, E, P>>;
    show<E extends boolean = false, P extends PaginationTypes = 'offset'>(repositoryStorageId: number, options?: {
        snippetId?: string | number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SnippetRepositoryStorageMoveSchema, C, E, P>>;
    schedule<E extends boolean = false, P extends PaginationTypes = 'offset'>(sourceStorageName: string, options?: {
        snippetId?: string | number;
        destinationStorageName?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SnippetRepositoryStorageMoveSchema, C, E, P>>;
}
declare class SnippetRepositoryStorageMoves<C extends boolean = false> extends ResourceRepositoryStorageMoves<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface SuggestionSchema extends Record<string, unknown> {
    id: number;
    from_line: number;
    to_line: number;
    appliable: boolean;
    applied: boolean;
    from_content: string;
    to_content: string;
}
declare class Suggestions<C extends boolean = false> extends BaseResource<C> {
    edit<E extends boolean = false>(suggestionId: number, options?: {
        commitMessage?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SuggestionSchema, C, E, void>>;
    editBatch<E extends boolean = false>(suggestionIds: number[], options?: {
        commitMessage?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SuggestionSchema, C, E, void>>;
}

interface SystemHookTestResponse extends Record<string, unknown> {
    project_id: number;
    owner_email: string;
    owner_name: string;
    name: string;
    path: string;
    event_name: string;
}
interface CreateSystemHook {
    token?: string;
    pushEvents?: boolean;
    tagPushEvents?: boolean;
    mergeRequestsEvents?: boolean;
    repositoryUpdateEvents?: boolean;
    enableSslVerification?: boolean;
}
declare class SystemHooks<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false>(options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<HookSchema[], C, E, void>>;
    add<E extends boolean = false>(url: string, options?: CreateSystemHook & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<HookSchema, C, E, void>>;
    create<E extends boolean = false>(url: string, options?: CreateSystemHook & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<HookSchema, C, E, void>>;
    test<E extends boolean = false>(hookId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SystemHookTestResponse, C, E, void>>;
    remove<E extends boolean = false>(hookId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(hookId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<HookSchema, C, E, void>>;
}

interface TopicSchema extends Record<string, unknown> {
    id: number;
    name: string;
    description: string;
    total_projects_count: number;
    avatar_url: string;
}
declare class Topics<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: {
        search?: string;
        withoutProjects?: boolean;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TopicSchema[], C, E, P>>;
    create<E extends boolean = false>(name: string, { avatar, ...options }?: {
        avatar?: {
            content: Blob;
            filename: string;
        };
        description?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TopicSchema, C, E, void>>;
    edit<E extends boolean = false>(topicId: number, { avatar, ...options }?: {
        name?: string;
        title?: string;
        avatar?: {
            content: Blob;
            filename: string;
        };
        description?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TopicSchema, C, E, void>>;
    merge<E extends boolean = false>(sourceTopicId: number, targetTopicId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TopicSchema, C, E, void>>;
    remove<E extends boolean = false>(topicId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(topicId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TopicSchema, C, E, void>>;
}

interface BranchSchema extends Record<string, unknown> {
    name: string;
    merged: boolean;
    protected: boolean;
    default: boolean;
    developers_can_push: boolean;
    developers_can_merge: boolean;
    can_push: boolean;
    web_url: string;
    commit: MappedOmit<CommitSchema, 'web_url' | 'created_at'>;
}
declare class Branches<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: {
        search?: string;
        regex?: string;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<BranchSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, branchName: string, ref: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<BranchSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, branchName: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    removeMerged<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, branchName: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<BranchSchema, C, E, void>>;
}

interface CommitDiscussions<C extends boolean = false> extends ResourceDiscussions<C> {
    addNote<E extends boolean = false>(projectId: string | number, commitId: string, discussionId: string, body: string, options?: {
        createdAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionNoteSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, commitId: string, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, commitId: string, body: string, options?: {
        position?: DiscussionNotePositionOptions;
        commitId?: string;
        createdAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema, C, E, void>>;
    editNote<E extends boolean = false>(projectId: string | number, commitId: string, discussionId: string, noteId: number, options?: Sudo & ShowExpanded<E> & {
        body?: string;
    }): Promise<GitlabAPIResponse<DiscussionNoteSchema, C, E, void>>;
    removeNote<E extends boolean = false>(projectId: string | number, commitId: string, discussionId: string, noteId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, commitId: string, discussionId: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema, C, E, void>>;
}
declare class CommitDiscussions<C extends boolean = false> extends ResourceDiscussions<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface RegistryRepositoryTagSchema extends Record<string, unknown> {
    name: string;
    path: string;
    location: string;
    revision: string;
    short_revision: string;
    digest: string;
    created_at: string;
    total_size: number;
}
type CondensedRegistryRepositoryTagSchema = Pick<RegistryRepositoryTagSchema, 'name' | 'path' | 'location'>;
interface RegistryRepositorySchema extends Record<string, unknown> {
    id: number;
    name: string;
    path: string;
    project_id: number;
    location: string;
    created_at: string;
    cleanup_policy_started_at: string;
    tags_count?: number;
    tags?: Pick<RegistryRepositoryTagSchema, 'name' | 'path' | 'location'>[];
}
type CondensedRegistryRepositorySchema = MappedOmit<RegistryRepositorySchema, 'tags' | 'tags_count'>;
declare class ContainerRegistry<C extends boolean = false> extends BaseResource<C> {
    allRepositories<E extends boolean = false, P extends PaginationTypes = 'offset'>({ groupId, projectId, ...options }?: OneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & {
        tags?: boolean;
        tagsCount?: boolean;
    } & PaginationRequestOptions<P> & BaseRequestOptions<E>): Promise<GitlabAPIResponse<CondensedRegistryRepositorySchema[], C, E, P>>;
    allTags<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, repositoryId: number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CondensedRegistryRepositoryTagSchema[], C, E, P>>;
    editRegistryVisibility<E extends boolean = false>(projectId: string | number, options?: {
        containerRegistryAccessLevel: 'enabled' | 'private' | 'disabled';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<CamelizedResponse<MappedOmit<RegistryRepositorySchema, "tags" | "tags_count">, C>, E, undefined>>;
    removeRepository<E extends boolean = false>(projectId: string | number, repositoryId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    removeTag<E extends boolean = false>(projectId: string | number, repositoryId: number, tagName: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    removeTags<E extends boolean = false>(projectId: string | number, repositoryId: number, nameRegexDelete: string, options?: Sudo & {
        nameRegex?: string;
        nameRegexKeep?: string;
        keepN?: string;
        olderThan?: string;
    } & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    showRepository<E extends boolean = false>(repositoryId: number, options?: {
        tags?: boolean;
        tagsCount?: boolean;
        size?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RegistryRepositorySchema, C, E, void>>;
    showTag<E extends boolean = false>(projectId: string | number, repositoryId: number, tagName: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RegistryRepositoryTagSchema, C, E, void>>;
}

type JobScope = 'created' | 'pending' | 'running' | 'failed' | 'success' | 'canceled' | 'skipped' | 'manual' | 'waiting_for_resource';
interface ArtifactSchema extends Record<string, unknown> {
    file_type: string;
    size: number;
    filename: string;
    file_format?: string;
}
interface CondensedJobSchema extends Record<string, unknown> {
    id: number;
    name: string;
    stage: string;
    project_id: string | number;
    project_name: string;
}
interface JobSchema extends Record<string, unknown> {
    id: number;
    name: string;
    stage: string;
    status: string;
    ref: string;
    tag: boolean;
    coverage?: string;
    allow_failure: boolean;
    created_at: string;
    started_at?: string;
    finished_at?: string;
    failure_reason?: string;
    erased_at?: string;
    duration?: number;
    user: ExpandedUserSchema;
    commit: CondensedCommitSchema;
    pipeline: PipelineSchema;
    web_url: string;
    artifacts: ArtifactSchema[];
    queued_duration: number;
    artifacts_file: {
        filename: string;
        size: number;
    };
    runner: RunnerSchema;
    artifacts_expire_at?: string;
    tag_list?: string[];
    project?: {
        ci_job_token_scope_enabled?: boolean;
    };
}
interface BridgeSchema extends Record<string, unknown> {
    commit: CondensedCommitSchema;
    coverage?: string;
    allow_failure: boolean;
    created_at: string;
    started_at: string;
    finished_at: string;
    erased_at?: string;
    duration: number;
    queued_duration: number;
    id: number;
    name: string;
    pipeline: MappedOmit<PipelineSchema & {
        project_id: number;
    }, 'user'>;
    ref: string;
    stage: string;
    status: string;
    tag: boolean;
    web_url: string;
    user: ExpandedUserSchema;
    downstream_pipeline: MappedOmit<PipelineSchema, 'user'>;
}
interface AllowedAgentSchema extends Record<string, unknown> {
    id: number;
    config_project: MappedOmit<SimpleProjectSchema, 'web_url'>;
}
interface JobKubernetesAgentsSchema extends Record<string, unknown> {
    allowed_agents: AllowedAgentSchema[];
    job: CondensedJobSchema;
    pipeline: PipelineSchema;
    project: MappedOmit<SimpleProjectSchema, 'web_url'>;
    user: SimpleUserSchema;
}
interface JobVariableAttributeOption extends Record<string, unknown> {
    key: string;
    value: string;
}
declare class Jobs<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, { pipelineId, ...options }?: {
        pipelineId?: number;
        scope?: JobScope;
        includeRetried?: boolean;
    } & BaseRequestOptions<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<JobSchema[], C, E, P>>;
    allPipelineBridges<E extends boolean = false>(projectId: string | number, pipelineId: number, options?: {
        scope?: JobScope;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<BridgeSchema[], C, E, void>>;
    cancel<E extends boolean = false>(projectId: string | number, jobId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<JobSchema, C, E, void>>;
    erase<E extends boolean = false>(projectId: string | number, jobId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<JobSchema, C, E, void>>;
    play<E extends boolean = false>(projectId: string | number, jobId: number, options?: {
        jobVariablesAttributes: JobVariableAttributeOption[];
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<JobSchema, C, E, void>>;
    retry<E extends boolean = false>(projectId: string | number, jobId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<JobSchema, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, jobId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<JobSchema, C, E, void>>;
    showConnectedJob<E extends boolean = false>(options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<JobSchema, C, E, void>>;
    showConnectedJobK8Agents<E extends boolean = false>(options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<JobKubernetesAgentsSchema, C, E, void>>;
    showLog<E extends boolean = false>(projectId: string | number, jobId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<string, void, E, void>>;
}

interface RunnerToken extends Record<string, unknown> {
    id: number;
    token: string;
    token_expires_at: string;
}
interface RunnerSchema extends Record<string, unknown> {
    id: number;
    paused: boolean;
    description: string;
    ip_address: string;
    is_shared: boolean;
    runner_type: 'instance_type' | 'group_type' | 'project_type';
    name: string;
    online: boolean;
    status: 'online' | 'offline';
}
interface ExpandedRunnerSchema extends RunnerSchema {
    architecture: string | null;
    description: string;
    contacted_at: string;
    platform: string | null;
    projects: Pick<SimpleProjectSchema, 'id' | 'name' | 'name_with_namespace' | 'path' | 'path_with_namespace'>[];
    groups: CondensedGroupSchema[];
    revision: string | null;
    tag_list: string[] | null;
    version: string | null;
    access_level: string;
    maximum_timeout: number | null;
    run_untagged: boolean;
    locked: boolean;
}
type AllRunnersOptions = {
    type?: 'instance_type' | 'group_type' | 'project_type';
    status?: 'online' | 'offline' | 'stale' | 'never_contacted' | 'active' | 'paused';
    paused?: boolean;
    tagList?: string[];
};
type EditRunnerOptions = {
    description?: string;
    active?: boolean;
    paused?: boolean;
    tagList?: string[];
    runUntagged?: boolean;
    locked?: boolean;
    accessLevel?: 'not_protected' | 'ref_protected';
    maximumTimeout?: number;
};
type CreateRunnerOptions = {
    info?: Record<string, string>;
    description?: string;
    active?: boolean;
    paused?: boolean;
    tagList?: string[];
    runUntagged?: boolean;
    locked?: boolean;
    accessLevel?: 'not_protected' | 'ref_protected';
    maximumTimeout?: number;
    maintainerNote?: string;
    maintenanceNote?: string;
};
declare class Runners<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>({ projectId, groupId, owned, ...options }?: OneOrNoneOf<{
        projectId: string | number;
        owned: boolean;
        groupId: string | number;
    }> & AllRunnersOptions & BaseRequestOptions<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<RunnerSchema[], C, E, P>>;
    allJobs<E extends boolean = false, P extends PaginationTypes = 'offset'>(runnerId: number, options?: Sudo & PaginationRequestOptions<P> & ShowExpanded<E> & {
        status?: string;
        orderBy?: string;
        sort?: string;
    }): Promise<GitlabAPIResponse<JobSchema[], C, E, P>>;
    create<E extends boolean = false>(token: string, options?: CreateRunnerOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RunnerToken, C, E, void>>;
    edit<E extends boolean = false>(runnerId: number, options?: EditRunnerOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedRunnerSchema, C, E, void>>;
    enable<E extends boolean = false>(projectId: string | number, runnerId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RunnerSchema, C, E, void>>;
    disable<E extends boolean = false>(projectId: string | number, runnerId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    register<E extends boolean = false>(token: string, options?: CreateRunnerOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RunnerToken, C, E, void>>;
    remove<E extends boolean = false>({ runnerId, token, ...options }: OneOf<{
        runnerId: number;
        token: string;
    }> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    resetRegistrationToken<E extends boolean = false>({ runnerId, token, ...options }?: OneOrNoneOf<{
        runnerId: string;
        token: string;
    }> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(runnerId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedRunnerSchema, C, E, void>>;
    verify<E extends boolean = false>(options?: {
        systemId?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

type EnvironmentTier = 'production' | 'staging' | 'testing' | 'development' | 'other';
interface EnvironmentSchema extends Record<string, unknown> {
    id: number;
    name: string;
    slug: string;
    external_url: string;
    state: string;
    tier: EnvironmentTier;
    created_at: string;
    updated_at: string;
    enable_advanced_logs_querying: boolean;
    logs_api_path: string;
    flux_resource_path?: string;
    kubernetes_namespace?: string;
    last_deployment: DeploymentSchema;
    deployable: DeployableSchema;
    project?: SimpleProjectSchema;
    auto_stop_at: string | null;
    description: string | null;
    auto_stop_setting: string;
}
type CondensedEnvironmentSchema = MappedOmit<EnvironmentSchema, 'last_deployment' | 'deployable'>;
type ReviewAppSchema = MappedOmit<CondensedEnvironmentSchema, 'state'>;
declare class Environments<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: PaginationRequestOptions<P> & OneOrNoneOf<{
        name: string;
        search: string;
    }> & {
        states?: 'available' | 'stopping' | 'stopped';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CondensedEnvironmentSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, name: string, options?: {
        externalUrl?: string;
        tier?: EnvironmentTier;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CondensedEnvironmentSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, environmentId: number, options?: {
        externalUrl?: string;
        tier?: EnvironmentTier;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CondensedEnvironmentSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, environmentId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    removeReviewApps<E extends boolean = false>(projectId: string | number, options?: {
        before?: string;
        limit?: number;
        dryRun?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<{
        scheduled_entries: ReviewAppSchema[];
        unprocessable_entries: ReviewAppSchema[];
    }, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, environmentId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<EnvironmentSchema, C, E, void>>;
    stop<E extends boolean = false>(projectId: string | number, environmentId: number, options?: {
        force?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CondensedEnvironmentSchema, C, E, void>>;
    stopStale<E extends boolean = false>(projectId: string | number, before: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<{
        message: string;
    }, C, E, void>>;
}

type DeploymentStatus = 'created' | 'running' | 'success' | 'failed' | 'canceled';
interface DeployableSchema extends Record<string, unknown> {
    id: number;
    ref: string;
    name: string;
    runner?: RunnerSchema;
    stage?: string;
    started_at?: string;
    status?: DeploymentStatus;
    tag: boolean;
    commit?: CommitSchema;
    coverage?: string;
    created_at?: string;
    finished_at?: string;
    user?: SimpleUserSchema;
    pipeline?: PipelineSchema;
}
interface DeploymentApprovalStatusSchema extends Record<string, unknown> {
    user: SimpleUserSchema;
    status: 'approved' | 'rejected';
    created_at: string;
    comment: string;
}
interface DeploymentSchema extends Record<string, unknown> {
    id: number;
    iid: number;
    ref: string;
    sha: string;
    created_at: string;
    updated_at: string;
    status: DeploymentStatus;
    user: SimpleUserSchema;
    deployable: DeployableSchema;
    environment: EnvironmentSchema;
    pending_approval_count?: number;
    approvals?: DeploymentApprovalStatusSchema[];
}
type AllDeploymentsOptions = {
    orderBy?: 'id' | 'iid' | 'created_at' | 'updated_at' | 'ref';
    sort?: 'asc' | 'desc';
    updatedAfter?: string;
    updatedBefore?: string;
    environment?: string;
    status?: 'created' | 'running' | 'success' | 'failed' | 'canceled' | 'blocked';
};
declare class Deployments<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: AllDeploymentsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DeploymentSchema[], C, E, P>>;
    allMergeRequests<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, deploymentId: number, options?: AllMergeRequestsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, environment: string, sha: string, ref: string, tag: boolean, options?: {
        status?: 'running' | 'success' | 'failed' | 'canceled';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DeploymentSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, deploymentId: number, status: 'running' | 'success' | 'failed' | 'canceled', options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DeploymentSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, deploymentId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<{
        message: string;
    }, C, E, void>>;
    setApproval<E extends boolean = false>(projectId: string | number, deploymentId: number, status: 'approved' | 'rejected', options?: {
        comment?: string;
        representedAs?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DeploymentApprovalStatusSchema, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, deploymentId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DeploymentSchema, C, E, void>>;
}

interface ErrorTrackingClientKeySchema extends Record<string, unknown> {
    id: number;
    active: boolean;
    public_key: string;
    sentry_dsn: string;
}
declare class ErrorTrackingClientKeys<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ErrorTrackingClientKeySchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ErrorTrackingClientKeySchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface ErrorTrackingSettingsSchema extends Record<string, unknown> {
    active: boolean;
    project_name: string;
    sentry_external_url: string;
    api_url: string;
    integrated: boolean;
}
declare class ErrorTrackingSettings<C extends boolean = false> extends BaseResource<C> {
    create<E extends boolean = false>(projectId: string | number, active: boolean, integrated: boolean, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ErrorTrackingSettingsSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, active: boolean, { integrated, ...options }?: {
        integrated?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ErrorTrackingSettingsSchema, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ErrorTrackingSettingsSchema, C, E, void>>;
}

interface BaseExternalStatusCheckSchema extends Record<string, unknown> {
    id: number;
    name: string;
    external_url: string;
    status: string;
}
type MergeRequestExternalStatusCheckSchema = BaseExternalStatusCheckSchema;
interface ExternalStatusCheckProtectedBranchesSchema {
    id: number;
    project_id: number;
    name: string;
    created_at: string;
    updated_at: string;
    code_owner_approval_required: boolean;
}
interface ProjectExternalStatusCheckSchema extends BaseExternalStatusCheckSchema {
    project_id: number;
    protected_branches?: ExternalStatusCheckProtectedBranchesSchema[];
}
declare class ExternalStatusChecks<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options: {
        mergerequestIId: number;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestExternalStatusCheckSchema[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectExternalStatusCheckSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, name: string, externalUrl: string, options?: {
        protectedBrancheIds: number[];
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectExternalStatusCheckSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, externalStatusCheckId: number, options?: {
        protectedBrancheIds?: number[];
        externalUrl?: string;
        name?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectExternalStatusCheckSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, externalStatusCheckId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    set<E extends boolean = false>(projectId: string | number, mergerequestIId: number, sha: string, externalStatusCheckId: number, options?: {
        status?: 'passed' | 'failed';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectExternalStatusCheckSchema, C, E, void>>;
}

interface FeatureFlagUserListSchema extends Record<string, unknown> {
    name: string;
    user_xids: string;
    id: number;
    iid: number;
    project_id: string | number;
    created_at: string;
    updated_at: string;
}
declare class FeatureFlagUserLists<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: {
        search?: string;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<FeatureFlagUserListSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, name: string, userXids: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<FeatureFlagUserListSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, featureFlagUserListIId: string | number, options?: {
        name?: string;
        userXIds?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<FeatureFlagUserListSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, featureFlagUserListIId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, featureFlagUserListIId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<FeatureFlagUserListSchema, C, E, void>>;
}

interface FeatureFlagStrategyScopeSchema {
    id: number;
    environment_scope: string;
}
interface FeatureFlagStrategySchema {
    id: number;
    name: string;
    parameters: Record<string, unknown>;
    scopes?: FeatureFlagStrategyScopeSchema[];
}
interface FeatureFlagSchema extends Record<string, unknown> {
    name: string;
    description: string;
    active: boolean;
    version: string;
    created_at: string;
    updated_at: string;
    scopes?: string[];
    strategies?: FeatureFlagStrategySchema[];
}
type CreateFeatureFlagOptions = {
    description?: string;
    active?: boolean;
    strategies?: {
        name: string;
        parameters?: Record<string, string>;
        scopes?: MappedOmit<FeatureFlagStrategyScopeSchema, 'id'>[];
    };
};
type EditFeatureFlagOptions = {
    description?: string;
    active?: boolean;
    strategies?: {
        id: string;
        name?: string;
        _destroy?: boolean;
        parameters?: Record<string, string>;
        scopes?: (FeatureFlagStrategyScopeSchema & {
            _destroy?: boolean;
        })[];
    };
};
declare class FeatureFlags<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: {
        scope?: 'enabled' | 'disabled';
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<FeatureFlagSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, flagName: string, version: string, options?: CreateFeatureFlagOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<FeatureFlagSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, featureFlagName: string, options?: EditFeatureFlagOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<FeatureFlagSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, flagName: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, flagName: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<FeatureFlagSchema, C, E, void>>;
}

interface FreezePeriodSchema extends Record<string, unknown> {
    id: number;
    freeze_start: string;
    freeze_end: string;
    cron_timezone: string;
    created_at: string;
    updated_at: string;
}
declare class FreezePeriods<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<FreezePeriodSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, freezeStart: string, freezeEnd: string, options?: {
        cronTimezone?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<FreezePeriodSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, freezePeriodId: number, options?: {
        freezeStart?: string;
        freezeEnd?: string;
        cronTimezone?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<FreezePeriodSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, freezePeriodId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, freezePeriodId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<FreezePeriodSchema, C, E, void>>;
}

interface GitlabPagesSettingsSchema extends Record<string, unknown> {
    url: string;
    is_unique_domain_enabled: boolean;
    force_https: boolean;
    deployments: Array<{
        created_at: string;
        url: string;
        path_prefix: string;
        root_directory: string;
    }>;
}
declare class GitlabPages<C extends boolean = false> extends BaseResource<C> {
    remove<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    showSettings<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GitlabPagesSettingsSchema, C, E, void>>;
}

interface GoProxyModuleVersionSchema extends Record<string, unknown> {
    Version: string;
    Time: string;
}
declare class GoProxy<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false>(projectId: string | number, moduleName: string, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<string, C, E, void>>;
    showVersionMetadata<E extends boolean = false>(projectId: string | number, moduleName: string, moduleVersion: string, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<GoProxyModuleVersionSchema, C, E, void>>;
    downloadModuleFile<E extends boolean = false>(projectId: string | number, moduleName: string, moduleVersion: string, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, C, E, void>>;
    downloadModuleSource<E extends boolean = false>(projectId: string | number, moduleName: string, moduleVersion: string, options?: ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
}

declare class Helm<C extends boolean = false> extends BaseResource<C> {
    downloadChartIndex<E extends boolean = false>(projectId: string | number, channel: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    downloadChart<E extends boolean = false>(projectId: string | number, channel: string, filename: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    import<E extends boolean = false>(projectId: string | number, channel: string, chart: {
        content: Blob;
        filename: string;
    }, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

type SupportedIntegration = 'asana' | 'assembla' | 'bamboo' | 'bugzilla' | 'buildkite' | 'campfire' | 'custom-issue-tracker' | 'drone-ci' | 'emails-on-push' | 'external-wiki' | 'flowdock' | 'hangouts_chat' | 'hipchat' | 'irker' | 'jira' | 'kubernetes' | 'slack-slash-commands' | 'slack' | 'packagist' | 'pipelines-email' | 'pivotaltracker' | 'prometheus' | 'pushover' | 'redmine' | 'microsoft-teams' | 'mattermost' | 'mattermost-slash-commands' | 'teamcity' | 'jenkins' | 'jenkins-deprecated' | 'mock-ci' | 'youtrack';
interface IntegrationSchema extends Record<string, unknown> {
    id: number;
    title: string;
    slug: string;
    created_at: string;
    updated_at: string;
    active: boolean;
    commit_events: boolean;
    push_events: boolean;
    issues_events: boolean;
    confidential_issues_events: boolean;
    merge_requests_events: boolean;
    tag_push_events: boolean;
    note_events: boolean;
    confidential_note_events: boolean;
    pipeline_events: boolean;
    wiki_page_events: boolean;
    job_events: boolean;
    comment_on_event_enabled: boolean;
}
declare class Integrations<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IntegrationSchema[], C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, integrationName: SupportedIntegration, options?: BaseRequestOptions<E>): Promise<GitlabAPIResponse<IntegrationSchema, C, E, void>>;
    disable<E extends boolean = false>(projectId: string | number, integrationName: SupportedIntegration, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, integrationName: SupportedIntegration, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IntegrationSchema, C, E, void>>;
}

interface IssueAwardEmojis<C extends boolean = false> extends ResourceAwardEmojis<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, issueIId: number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema[], C, E, P>>;
    award<E extends boolean = false>(projectId: string | number, issueIId: number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, issueIId: number, awardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, issueIId: number, awardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema, C, E, void>>;
}
declare class IssueAwardEmojis<C extends boolean = false> extends ResourceAwardEmojis<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface IssueDiscussions<C extends boolean = false> extends ResourceDiscussions<C> {
    addNote<E extends boolean = false>(projectId: string | number, issueIId: number, discussionId: string, body: string, options?: {
        createdAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionNoteSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, issueIId: number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, issueIId: number, body: string, options?: {
        createdAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema, C, E, void>>;
    editNote<E extends boolean = false>(projectId: string | number, issueIId: number, discussionId: string, noteId: number, options: Sudo & ShowExpanded<E> & {
        body: string;
    }): Promise<GitlabAPIResponse<DiscussionNoteSchema, C, E, void>>;
    removeNote<E extends boolean = false>(projectId: string | number, issueIId: number, discussionId: string, noteId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, issueIId: number, discussionId: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema, C, E, void>>;
}
declare class IssueDiscussions<C extends boolean = false> extends ResourceDiscussions<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface IssueIterationEvents<C extends boolean = false> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<IterationEventSchema[], C, E, P>>;
    show<E extends boolean = false>(projectId: string | number, issueIId: number, iterationEventId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IterationEventSchema, C, E, void>>;
}
declare class IssueIterationEvents<C extends boolean = false> extends ResourceIterationEvents<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface IssueLabelEvents<C extends boolean = false> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<LabelEventSchema[], C, E, P>>;
    show<E extends boolean = false>(projectId: string | number, issueIId: number, labelEventId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelEventSchema, C, E, void>>;
}
declare class IssueLabelEvents<C extends boolean = false> extends ResourceLabelEvents<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface IssueLinkSchema extends Record<string, unknown> {
    id: number;
    iid: number;
    project_id: number;
    issue_link_id: number;
    state: string;
    description: string;
    weight?: number;
    author: MappedOmit<SimpleUserSchema, 'created_at'>;
    milestone: MilestoneSchema;
    assignees?: MappedOmit<SimpleUserSchema, 'created_at'>[];
    title: string;
    labels?: string[];
    user_notes_count: number;
    due_date: string;
    web_url: string;
    confidential: boolean;
    updated_at: string;
    link_created_at: string;
    link_updated_at: string;
    link_type: 'relates_to' | 'blocks' | 'is_blocked_by';
}
interface ExpandedIssueLinkSchema extends Record<string, unknown> {
    source_issue: MappedOmit<IssueLinkSchema, 'link_type' | 'link_created_at' | 'link_updated_at' | 'issue_link_id'>;
    target_issue: MappedOmit<IssueLinkSchema, 'link_type' | 'link_created_at' | 'link_updated_at' | 'issue_link_id'>;
    link_type: 'relates_to' | 'blocks' | 'is_blocked_by';
}
declare class IssueLinks<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<IssueLinkSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, issueIId: number, targetProjectId: string | number, targetIssueIId: number, options?: {
        linkType?: 'relates_to' | 'blocks' | 'is_blocked_by';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedIssueLinkSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, issueIId: number, issueLinkId: number, options?: {
        linkType?: 'relates_to' | 'blocks' | 'is_blocked_by';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedIssueLinkSchema, C, E, void>>;
}

interface IssueMilestoneEvents<C extends boolean = false> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<MilestoneEventSchema[], C, E, P>>;
    show<E extends boolean = false>(projectId: string | number, issueIId: number, milestoneEventId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MilestoneEventSchema, C, E, void>>;
}
declare class IssueMilestoneEvents<C extends boolean = false> extends ResourceMilestoneEvents<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface IssueNoteAwardEmojis<C extends boolean = false> extends ResourceNoteAwardEmojis<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, issueIId: number, noteId: number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema[], C, E, P>>;
    award<E extends boolean = false>(projectId: string | number, issueIId: number, noteId: number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, issueIId: number, noteId: number, awardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, issueIId: number, noteId: number, awardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema, C, E, void>>;
}
declare class IssueNoteAwardEmojis<C extends boolean = false> extends ResourceNoteAwardEmojis<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface IssueNoteSchema extends NoteSchema {
    noteable_type: 'Issue';
}
interface IssueNotes<C extends boolean = false> extends ResourceNotes<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, issueIId: number, options?: {
        sort?: 'asc' | 'desc';
        orderBy?: 'created_at' | 'updated_at';
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueNoteSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, issueIId: number, body: string, options?: {
        internal?: boolean;
        createdAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueNoteSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, issueIId: number, noteId: number, options: {
        body?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueNoteSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, issueIId: number, noteId: number, options?: Sudo): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, issueIId: number, noteId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueNoteSchema, C, E, void>>;
}
declare class IssueNotes<C extends boolean = false> extends ResourceNotes<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface IssueStateEvents<C extends boolean = false> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<StateEventSchema[], C, E, P>>;
    show<E extends boolean = false>(projectId: string | number, issueIId: number, stateEventId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<StateEventSchema, C, E, void>>;
}
declare class IssueStateEvents<C extends boolean = false> extends ResourceStateEvents<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface IssueWeightEvents<C extends boolean = false> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<StateEventSchema[], C, E, P>>;
    show<E extends boolean = false>(projectId: string | number, issueIId: number, weightEventId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<StateEventSchema, C, E, void>>;
}
declare class IssueWeightEvents<C extends boolean = false> extends ResourceStateEvents<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface StatisticsSchema extends Record<string, unknown> {
    statistics: {
        counts: {
            all: number;
            closed: number;
            opened: number;
        };
    };
}
type AllIssueStatisticsOptions = {
    labels?: string;
    milestone?: string;
    scope?: 'created_by_me' | 'assigned_to_me' | 'all';
    epicId?: number;
    myReactionEmoji?: string;
    iids?: number[];
    search?: string;
    in?: string;
    createdAfter?: string;
    createdBefore?: string;
    updatedAfter?: string;
    updatedBefore?: string;
    confidential?: boolean;
};
declare class IssuesStatistics<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false>({ projectId, groupId, ...options }?: OneOrNoneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & OneOrNoneOf<{
        authorId: number;
        authorUsername: string;
    }> & OneOrNoneOf<{
        assigneeId: number;
        assigneeUsername: string;
    }> & AllIssueStatisticsOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<StatisticsSchema, C, E, void>>;
}

declare class JobArtifacts<C extends boolean = false> extends BaseResource<C> {
    downloadArchive<E extends boolean = false>(projectId: string | number, { jobId, artifactPath, ref, ...options }?: ({
        jobId: number;
        artifactPath?: undefined;
        job?: undefined;
        ref?: undefined;
    } | {
        jobId: number;
        artifactPath: string;
        job?: undefined;
        ref?: undefined;
    } | {
        ref: string;
        job: string;
        jobId?: undefined;
        artifactPath?: undefined;
    } | {
        ref: string;
        job: string;
        artifactPath: string;
        jobId?: undefined;
    }) & {
        jobToken?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    keep<E extends boolean = false>(projectId: string | number, jobId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<JobSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, { jobId, ...options }?: {
        jobId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

type ProtectedBranchAccessLevel = AccessLevel.NO_ACCESS | AccessLevel.DEVELOPER | AccessLevel.MAINTAINER | AccessLevel.ADMIN;
interface ExtendedProtectedBranchAccessLevelSchema {
    id: number;
    access_level: ProtectedBranchAccessLevel;
    access_level_description: string;
    user_id?: number | null;
    group_id?: number | null;
}
interface ProtectedBranchSchema extends Record<string, unknown> {
    id: number;
    name: string;
    push_access_levels?: ExtendedProtectedBranchAccessLevelSchema[];
    merge_access_levels?: ExtendedProtectedBranchAccessLevelSchema[];
    unprotect_access_levels?: ExtendedProtectedBranchAccessLevelSchema[];
    allow_force_push: boolean;
    code_owner_approval_required: boolean;
}
type CreateProtectedBranchAllowOptions = OneOf<{
    userId: number;
    groupId: number;
    accessLevel: ProtectedBranchAccessLevel;
}>;
type EditProtectedBranchAllowOptions = {
    _destroy?: boolean;
} & ({
    userId: number;
} | {
    groupId: number;
} | {
    accessLevel: ProtectedBranchAccessLevel;
    id: number;
});
type CreateProtectedBranchOptions = {
    allowForcePush?: boolean;
    allowedToMerge?: CreateProtectedBranchAllowOptions[];
    allowedToPush?: CreateProtectedBranchAllowOptions[];
    allowedToUnprotect?: CreateProtectedBranchAllowOptions[];
    codeOwnerApprovalRequired?: boolean;
    mergeAccessLevel?: ProtectedBranchAccessLevel;
    pushAccessLevel?: ProtectedBranchAccessLevel;
    unprotectAccessLevel?: ProtectedBranchAccessLevel;
};
type EditProtectedBranchOptions = {
    allowForcePush?: boolean;
    allowedToMerge?: EditProtectedBranchAllowOptions[];
    allowedToPush?: EditProtectedBranchAllowOptions[];
    allowedToUnprotect?: EditProtectedBranchAllowOptions[];
    codeOwnerApprovalRequired?: boolean;
};
declare class ProtectedBranches<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: {
        search?: string;
    } & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<ProtectedBranchSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, branchName: string, options?: CreateProtectedBranchOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProtectedBranchSchema, C, E, void>>;
    protect<E extends boolean = false>(projectId: string | number, branchName: string, options?: CreateProtectedBranchOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProtectedBranchSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, branchName: string, options?: EditProtectedBranchOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProtectedBranchSchema, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, branchName: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProtectedBranchSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, branchName: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    unprotect<E extends boolean = false>(projectId: string | number, branchName: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface ProjectLevelMergeRequestApprovalSchema extends Record<string, unknown> {
    approvals_before_merge: number;
    reset_approvals_on_push: boolean;
    disable_overriding_approvers_per_merge_request: boolean;
    merge_requests_author_approval: boolean;
    merge_requests_disable_committers_approval: boolean;
    require_password_to_approve: boolean;
}
interface ApprovedByEntity {
    user: MappedOmit<SimpleUserSchema, 'created_at'>;
}
interface MergeRequestLevelMergeRequestApprovalSchema extends Record<string, unknown> {
    id: number;
    iid: number;
    project_id: number;
    title: string;
    description: string;
    state: string;
    created_at: string;
    updated_at: string;
    merge_status: string;
    approvals_required: number;
    approvals_left: number;
    approved_by?: ApprovedByEntity[];
}
interface ApprovalRuleSchema extends Record<string, unknown> {
    id: number;
    name: string;
    rule_type: string;
    eligible_approvers?: MappedOmit<SimpleUserSchema, 'created_at'>[];
    approvals_required: number;
    users?: MappedOmit<SimpleUserSchema, 'created_at'>[];
    groups?: GroupSchema[];
    contains_hidden_groups: boolean;
    overridden: boolean;
}
interface ProjectLevelApprovalRuleSchema extends ApprovalRuleSchema {
    protected_branches?: ProtectedBranchSchema[];
}
interface MergeRequestLevelApprovalRuleSchema extends ApprovalRuleSchema {
    source_rule?: string;
}
interface ApprovalStateSchema extends Record<string, unknown> {
    approval_rules_overwritten: boolean;
    rules: ({
        approved: boolean;
    } & MergeRequestLevelApprovalRuleSchema)[];
}
type CreateApprovalRuleOptions = {
    userIds?: number[];
    groupIds?: number[];
    protectedBranchIds?: number[];
    appliesToAllProtectedBranches?: boolean;
    reportType?: string;
    ruleType?: string;
    usernames?: string[];
};
type EditApprovalRuleOptions = {
    userIds?: number[];
    groupIds?: number[];
    protectedBranchIds?: number[];
    appliesToAllProtectedBranches?: boolean;
    usernames?: string[];
    removeHiddenGroups?: boolean;
};
type EditConfigurationOptions = {
    disableOverridingApproversPerMergeRequest?: boolean;
    mergeRequestsAuthorApproval?: boolean;
    mergeRequestsDisableCommittersApproval?: boolean;
    requirePasswordToApprove?: boolean;
    resetApprovalsOnPush?: boolean;
    selectiveCodeOwnerRemovals?: boolean;
};
declare class MergeRequestApprovals<C extends boolean = false> extends BaseResource<C> {
    allApprovalRules<E extends boolean = false>(projectId: string | number, options: {
        mergerequestIId: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestLevelApprovalRuleSchema[], C, E, void>>;
    allApprovalRules<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectLevelApprovalRuleSchema[], C, E, void>>;
    approve<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: {
        sha?: string;
        approvalPassword?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<CamelizedResponse<MergeRequestLevelMergeRequestApprovalSchema, C>, E, undefined>>;
    createApprovalRule<E extends boolean = false>(projectId: string | number, name: string, approvalsRequired: number, options: {
        mergerequestIId: number;
    } & CreateApprovalRuleOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestLevelApprovalRuleSchema, C, E, void>>;
    createApprovalRule<E extends boolean = false>(projectId: string | number, name: string, approvalsRequired: number, options?: CreateApprovalRuleOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectLevelApprovalRuleSchema, C, E, void>>;
    editApprovalRule<E extends boolean = false>(projectId: string | number, approvalRuleId: number, name: string, approvalsRequired: number, options: {
        mergerequestIId: number;
    } & EditApprovalRuleOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestLevelApprovalRuleSchema, C, E, void>>;
    editApprovalRule<E extends boolean = false>(projectId: string | number, approvalRuleId: number, name: string, approvalsRequired: number, options?: EditApprovalRuleOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectLevelApprovalRuleSchema, C, E, void>>;
    editConfiguration<E extends boolean = false>(projectId: string | number, options?: EditConfigurationOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectLevelMergeRequestApprovalSchema, C, E, void>>;
    removeApprovalRule<E extends boolean = false>(projectId: string | number, approvalRuleId: number, { mergerequestIId, ...options }?: {
        mergerequestIId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    showApprovalRule<E extends boolean = false>(projectId: string | number, approvalRuleId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectLevelApprovalRuleSchema, C, E, void>>;
    showApprovalState<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ApprovalStateSchema, C, E, void>>;
    showConfiguration<E extends boolean = false>(projectId: string | number, options: {
        mergerequestIId: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestLevelMergeRequestApprovalSchema, C, E, void>>;
    showConfiguration<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectLevelMergeRequestApprovalSchema, C, E, void>>;
    unapprove<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface MergeRequestAwardEmojis<C extends boolean = false> extends ResourceAwardEmojis<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, mergerequestIId: number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema[], C, E, P>>;
    award<E extends boolean = false>(projectId: string | number, mergerequestIId: number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, mergerequestIId: number, awardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, mergerequestIId: number, awardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema, C, E, void>>;
}
declare class MergeRequestAwardEmojis<C extends boolean = false> extends ResourceAwardEmojis<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface MergeRequestContextCommitSchema extends Record<string, unknown> {
    id: string;
    short_id: string;
    created_at: string;
    parent_ids?: null;
    title: string;
    message: string;
    author_name: string;
    author_email: string;
    authored_date: string;
    committer_name: string;
    committer_email: string;
    committed_date: string;
}
declare class MergeRequestContextCommits<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<MergeRequestContextCommitSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, commits: string[], { mergerequestIId, ...options }?: {
        mergerequestIId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface MergeRequestDiscussionNoteSchema extends DiscussionNoteSchema {
    resolved_by: string;
    resolved_at: string;
    position?: DiscussionNotePositionSchema;
}
type MergeRequestDiscussionNotePositionOptions = DiscussionNotePositionOptions & {
    lineRange?: {
        start?: {
            lineCode: string;
            type: 'new' | 'old';
        };
        end?: {
            lineCode: string;
            type: 'new' | 'old';
        };
    };
};
interface MergeRequestDiscussions<C extends boolean = false> extends ResourceDiscussions<C> {
    addNote<E extends boolean = false>(projectId: string | number, mergerequestId: string | number, discussionId: string, body: string, options?: {
        createdAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestDiscussionNoteSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, mergerequestId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, mergerequestId: string | number, body: string, options?: {
        position?: DiscussionNotePositionOptions;
        commitId?: string;
        createdAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema, C, E, void>>;
    editNote<E extends boolean = false>(projectId: string | number, mergerequestId: string | number, discussionId: string, noteId: number, options: Sudo & ShowExpanded<E> & OneOf<{
        body: string;
        resolved: boolean;
    }>): Promise<GitlabAPIResponse<MergeRequestDiscussionNoteSchema, C, E, void>>;
    removeNote<E extends boolean = false>(projectId: string | number, mergerequestId: string | number, discussionId: string, noteId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    resolve<E extends boolean = false>(projectId: string | number, mergerequestId: string | number, discussionId: string, resolve: boolean, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, mergerequestId: string | number, discussionId: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema, C, E, void>>;
}
declare class MergeRequestDiscussions<C extends boolean = false> extends ResourceDiscussions<C> {
    constructor(options: BaseResourceOptions<C>);
}

type MergeRequestDraftNotePositionSchema = DiscussionNotePositionSchema & {
    line_range?: number;
};
interface MergeRequestDraftNoteSchema extends Record<string, unknown> {
    id: number;
    author_id: number;
    merge_request_id: number;
    resolve_discussion: boolean;
    discussion_id?: number;
    note: string;
    commit_id?: number;
    line_code?: number;
    position: MergeRequestDraftNotePositionSchema;
}
declare class MergeRequestDraftNotes<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, mergerequestIId: number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestDraftNoteSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, mergerequestIId: number, note: string, options?: {
        commitId?: string;
        inReplyToDiscussionId?: number;
        resolveDiscussion?: boolean;
        position?: Camelize<MergeRequestDraftNotePositionSchema>;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestDraftNoteSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, mergerequestIId: number, draftNoteId: number, options?: {
        note?: string;
        position?: Camelize<MergeRequestDraftNotePositionSchema>;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestDraftNoteSchema, C, E, void>>;
    publish<E extends boolean = false>(projectId: string | number, mergerequestIId: number, draftNoteId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestDraftNoteSchema, C, E, void>>;
    publishBulk<E extends boolean = false>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestDraftNoteSchema[], C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, mergerequestIId: number, draftNoteId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, mergerequestIId: number, draftNoteId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestDraftNoteSchema, C, E, void>>;
}

interface MergeRequestLabelEvents<C extends boolean = false> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<LabelEventSchema[], C, E, P>>;
    show<E extends boolean = false>(projectId: string | number, mergerequestIId: number, labelEventId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelEventSchema, C, E, void>>;
}
declare class MergeRequestLabelEvents<C extends boolean = false> extends ResourceLabelEvents<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface MergeRequestMilestoneEvents<C extends boolean = false> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, mergerequestIId: number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<MilestoneEventSchema[], C, E, P>>;
    show<E extends boolean = false>(projectId: string | number, mergerequestIId: number, milestoneEventId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MilestoneEventSchema, C, E, void>>;
}
declare class MergeRequestMilestoneEvents<C extends boolean = false> extends ResourceMilestoneEvents<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface MergeRequestNoteAwardEmojis<C extends boolean = false> extends ResourceNoteAwardEmojis<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, mergeRequestIId: number, noteId: number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema[], C, E, P>>;
    award<E extends boolean = false>(projectId: string | number, mergeRequestIId: number, noteId: number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, mergeRequestIId: number, noteId: number, awardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, mergeRequestIId: number, noteId: number, awardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema, C, E, void>>;
}
declare class MergeRequestNoteAwardEmojis<C extends boolean = false> extends ResourceNoteAwardEmojis<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface MergeRequestNoteSchema extends NoteSchema {
    noteable_type: 'MergeRequest';
}
interface MergeRequestNotes<C extends boolean = false> extends ResourceNotes<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, mergerequestIId: number, options?: {
        sort?: 'asc' | 'desc';
        orderBy?: 'created_at' | 'updated_at';
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestNoteSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, mergerequestIId: number, body: string, options?: {
        mergeRequestDiffSha?: string;
        createdAt?: string;
        internal?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestNoteSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, mergerequestIId: number, noteId: number, options: {
        body: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestNoteSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, mergerequestIId: number, noteId: number, options?: Sudo): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, mergerequestIId: number, noteId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestNoteSchema, C, E, void>>;
}
declare class MergeRequestNotes<C extends boolean = false> extends ResourceNotes<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface MergeTrainSchema extends Record<string, unknown> {
    id: number;
    merge_request: CondensedMergeRequestSchema;
    user: MappedOmit<SimpleUserSchema, 'created_at'>;
    pipeline: PipelineSchema;
    created_at: string;
    updated_at: string;
    target_branch: string;
    status: string;
    merged_at: string;
    duration: number;
}
declare class MergeTrains<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options: {
        targetBranch?: string;
        scope?: 'active' | 'complete';
        sort?: 'asc' | 'desc';
    } & PaginationRequestOptions<P> & BaseRequestOptions<E>): Promise<GitlabAPIResponse<MergeTrainSchema[], C, E, P>>;
    showStatus<E extends boolean = false>(projectId: string | number, mergeRequestIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeTrainSchema, C, E, void>>;
    addMergeRequest<E extends boolean = false>(projectId: string | number, mergeRequestIId: number, options?: {
        whenPipelineSucceeds?: boolean;
        sha?: string;
        squash?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeTrainSchema, C, E, void>>;
}

interface PackageRegistrySchema extends Record<string, unknown> {
    id: number;
    package_id: number;
    created_at: string;
    updated_at: string;
    size: number;
    file_store: number;
    file_md5?: string;
    file_sha1?: string;
    file_name: string;
    file: {
        url: string;
    };
    file_sha256: string;
    verification_retry_at?: string;
    verified_at?: string;
    verification_failure?: string;
    verification_retry_count?: string;
    verification_checksum?: string;
    verification_state: number;
    verification_started_at?: string;
    new_file_path?: string;
}
declare class PackageRegistry<C extends boolean = false> extends BaseResource<C> {
    publish<E extends boolean = false>(projectId: string | number, packageName: string, packageVersion: string, packageFile: {
        content: Blob;
        filename: string;
    }, options: {
        select: 'package_file';
        contentType?: string;
        status?: 'default' | 'hidden';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PackageRegistrySchema, C, E, void>>;
    publish<E extends boolean = false>(projectId: string | number, packageName: string, packageVersion: string, packageFile: {
        content: Blob;
        filename: string;
    }, options?: {
        contentType?: string;
        status?: 'default' | 'hidden';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<{
        message: string;
    }, C, E, void>>;
    download<E extends boolean = false>(projectId: string | number, packageName: string, packageVersion: string, filename: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<CamelizedResponse<{
        message: string;
    }, C>, E, undefined>>;
}

interface PackageSchema extends Record<string, unknown> {
    id: number;
    name: string;
    version: string;
    package_type: string;
    created_at: string;
}
interface ExpandedPackageSchema extends PackageSchema {
    _links: Record<string, string>;
    pipelines: PipelineSchema[];
    versions: MappedOmit<ExpandedPackageSchema, '_links'>;
}
interface PackageFileSchema extends Record<string, unknown> {
    id: number;
    package_id: number;
    created_at: string;
    file_name: string;
    size: number;
    file_md5: string;
    file_sha1: string;
    pipelines?: PipelineSchema[];
}
type AllPackageOptions = {
    excludeSubgroups?: boolean;
    orderBy?: 'created_at' | 'name' | 'version' | 'type' | 'project_path';
    sort?: 'asc' | 'desc';
    packageType?: 'conan' | 'maven' | 'npm' | 'pypi' | 'composer' | 'nuget' | 'helm' | 'golang';
    packageName?: string;
    includeVersionless?: boolean;
    status?: 'default' | 'hidden' | 'processing' | 'error' | 'pending_destruction';
};
declare class Packages<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>({ projectId, groupId, ...options }?: OneOf<{
        projectId: string | number;
        groupId: string | number;
    }> & AllPackageOptions & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<PackageSchema[], C, E, P>>;
    allFiles<E extends boolean = false>(projectId: string | number, packageId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PackageFileSchema[], C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, packageId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    removeFile<E extends boolean = false>(projectId: string | number, packageId: number, projectFileId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, packageId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedPackageSchema, C, E, void>>;
}

interface PagesDomainSchema extends Record<string, unknown> {
    domain: string;
    url: string;
    project_id: number;
    auto_ssl_enabled?: boolean;
    certificate?: {
        expired: boolean;
        expiration: string;
        certificate: string;
        certificate_text: string;
    };
}
declare class PagesDomains<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false>({ projectId, ...options }?: {
        projectId?: string | number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PagesDomainSchema[], C, E, void>>;
    create<E extends boolean = false>(projectId: string | number, domain: string, options?: {
        autoSslEnabled?: string;
        certificate?: string;
        key?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PagesDomainSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, domain: string, options?: {
        autoSslEnabled?: string;
        certificate?: string;
        key?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PagesDomainSchema, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, domain: string, options?: {
        autoSslEnabled?: string;
        certificate?: string;
        key?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PagesDomainSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, domain: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface CondensedPipelineScheduleSchema extends Record<string, unknown> {
    id: number;
    description: string;
    ref: string;
    cron: string;
    cron_timezone: string;
    next_run_at: string;
    active: boolean;
    created_at: string;
    updated_at: string;
    owner: MappedOmit<SimpleUserSchema, 'created_at'>;
}
interface PipelineScheduleSchema extends CondensedPipelineScheduleSchema {
    last_pipeline: Pick<PipelineSchema, 'id' | 'sha' | 'ref' | 'status'>;
}
interface ExpandedPipelineScheduleSchema extends PipelineScheduleSchema {
    last_pipeline: Pick<PipelineSchema, 'id' | 'sha' | 'ref' | 'status'>;
    variables: PipelineVariableSchema[];
}
declare class PipelineSchedules<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: {
        scope?: 'active' | 'inactive';
    } & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<CondensedPipelineScheduleSchema[], C, E, P>>;
    allTriggeredPipelines<E extends boolean = false>(projectId: string | number, pipelineScheduleId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PipelineSchema[], C, E, void>>;
    create<E extends boolean = false>(projectId: string | number, description: string, ref: string, cron: string, options?: {
        cronTimezone?: string;
        active?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PipelineScheduleSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, pipelineScheduleId: number, options?: {
        description?: string;
        ref?: string;
        cron?: string;
        cronTimezone?: string;
        active?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PipelineScheduleSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, pipelineScheduleId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PipelineScheduleSchema, C, E, void>>;
    run<E extends boolean = false>(projectId: string | number, pipelineScheduleId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<{
        message: string;
    }, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, pipelineScheduleId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedPipelineScheduleSchema, C, E, void>>;
    takeOwnership<E extends boolean = false>(projectId: string | number, pipelineScheduleId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PipelineScheduleSchema, C, E, void>>;
}

interface PipelineTriggerTokenSchema extends Record<string, unknown> {
    id: number;
    description: string;
    created_at: string;
    last_used: string | null;
    token: string;
    updated_at: string;
    owner: MappedOmit<SimpleUserSchema, 'created_at'> | null;
}
declare class PipelineTriggerTokens<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<PipelineTriggerTokenSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, description: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PipelineTriggerTokenSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, triggerId: number, options?: {
        description?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PipelineTriggerTokenSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, triggerId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, triggerId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PipelineTriggerTokenSchema, C, E, void>>;
    trigger<E extends boolean = false>(projectId: string | number, ref: string, token: string, { variables, ...options }?: {
        variables?: Record<string, string>;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedPipelineSchema, C, E, void>>;
}

declare class ProductAnalytics<C extends boolean = false> extends BaseResource<C> {
    allFunnels<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<Record<string, unknown>, C, E, void>>;
    load<E extends boolean = false>(projectId: string | number, options?: {
        includeToken?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    dryRun<E extends boolean = false>(projectId: string | number, options?: {
        includeToken?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    showMetadata<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface ProjectAccessRequests<C extends boolean = false> extends ResourceAccessRequests<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AccessRequestSchema[], C, E, P>>;
    request<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AccessRequestSchema, C, E, void>>;
    approve<E extends boolean = false>(projectId: string | number, userId: number, options?: {
        accessLevel?: Exclude<AccessLevel, AccessLevel.ADMIN>;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AccessRequestSchema, C, E, void>>;
    deny<E extends boolean = false>(groupId: string | number, userId: number): Promise<GitlabAPIResponse<void, C, E, void>>;
}
declare class ProjectAccessRequests<C extends boolean = false> extends ResourceAccessRequests<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface ProjectAccessTokens<C extends boolean = false> extends ResourceAccessTokens<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<AccessTokenSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, name: string, scopes: AccessTokenScopes[], expiresAt: string, options?: {
        accessLevel?: Exclude<AccessLevel, AccessLevel.MINIMAL_ACCESS | AccessLevel.NO_ACCESS | AccessLevel.ADMIN>;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AccessTokenExposedSchema, C, E, void>>;
    revoke<E extends boolean = false>(projectId: string | number, tokenId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    rotate<E extends boolean = false>(projectId: string | number, tokenId: string | number, options?: {
        expiresAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AccessTokenExposedSchema, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, tokenId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AccessTokenSchema, C, E, void>>;
}
declare class ProjectAccessTokens<C extends boolean = false> extends ResourceAccessTokens<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface ProjectAliasSchema extends Record<string, unknown> {
    id: number;
    project_id: string | number;
    name: string;
}
declare class ProjectAliases<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<ProjectAliasSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectAliasSchema, C, E, void>>;
    edit<E extends boolean = false>(name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectAliasSchema, C, E, void>>;
    remove<E extends boolean = false>(name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface ProjectBadgeSchema extends BadgeSchema {
    kind: 'project';
}
interface ProjectBadges<C extends boolean = false> extends ResourceBadges<C> {
    add<E extends boolean = false>(groupId: string | number, linkUrl: string, imageUrl: string, options?: {
        name?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectBadgeSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: {
        name?: string;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectBadgeSchema[], C, E, P>>;
    edit<E extends boolean = false>(groupId: string | number, badgeId: number, options?: EditBadgeOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectBadgeSchema, C, E, void>>;
    preview<E extends boolean = false>(groupId: string | number, linkUrl: string, imageUrl: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CondensedBadgeSchema, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, badgeId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(groupId: string | number, badgeId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectBadgeSchema, C, E, void>>;
}
declare class ProjectBadges<C extends boolean = false> extends ResourceBadges<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface ProjectCustomAttributes<C extends boolean = false> extends ResourceCustomAttributes<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CustomAttributeSchema[], C, E, P>>;
    set<E extends boolean = false>(projectId: string | number, customAttributeId: string, value: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CustomAttributeSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, customAttributeId: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, customAttributeId: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CustomAttributeSchema, C, E, void>>;
}
declare class ProjectCustomAttributes<C extends boolean> extends ResourceCustomAttributes<C> {
    constructor(options: BaseResourceOptions<C>);
}

declare class ProjectDORA4Metrics<C extends boolean = false> extends ResourceDORA4Metrics<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface ProjectHookSchema extends ExpandedHookSchema {
    projectId: number;
}
interface ProjectHooks<C extends boolean = false> {
    add<E extends boolean = false>(projectId: string | number, url: string, options?: AddResourceHookOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectHookSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectHookSchema[], C, E, P>>;
    edit<E extends boolean = false>(projectId: string | number, hookId: number, url: string, options?: EditResourceHookOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectHookSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, hookId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, hookId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectHookSchema, C, E, void>>;
}
declare class ProjectHooks<C extends boolean = false> extends ResourceHooks<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface ExportStatusSchema extends Record<string, unknown> {
    id: number;
    description: string;
    name: string;
    name_with_namespace: string;
    path: string;
    path_with_namespace: string;
    created_at: string;
    export_status: string;
    _links: {
        api_url: string;
        web_url: string;
    };
}
interface FailedRelationSchema {
    id: number;
    created_at: string;
    exception_class: string;
    exception_message: string;
    source: string;
    relation_name: string;
}
interface ImportStatusSchema extends Record<string, unknown> {
    id: number;
    description: string;
    name: string;
    name_with_namespace: string;
    path: string;
    path_with_namespace: string;
    created_at: string;
    import_status: string;
    correlation_id: string;
    failed_relations?: FailedRelationSchema[];
    import_error?: string;
}
declare class ProjectImportExports<C extends boolean = false> extends BaseResource<C> {
    download<E extends boolean = false>(projectId: string | number, options: {
        asStream: true;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ReadableStream, void, E, void>>;
    download<E extends boolean = false>(projectId: string | number, options?: {
        asStream?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    import<E extends boolean = false>(file: {
        content: Blob;
        filename: string;
    }, path: string, options?: {
        name?: string;
        namespace?: number | string;
        overrideParams?: Record<string, unknown>;
        overwrite?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ImportStatusSchema, C, E, void>>;
    importRemote<E extends boolean = false>(url: string, path: string, options?: {
        name?: number;
        namespace?: number | string;
        overrideParams?: Record<string, unknown>;
        overwrite?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ImportStatusSchema, C, E, void>>;
    importRemoteS3<E extends boolean = false>(accessKeyId: string, bucketName: string, fileKey: string, path: string, region: string, secretAccessKey: string, options?: {
        name?: number;
        namespace?: number | string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ImportStatusSchema, C, E, void>>;
    showExportStatus<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<CamelizedResponse<ExportStatusSchema, C>, E, undefined>>;
    showImportStatus<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<CamelizedResponse<ImportStatusSchema, C>, E, undefined>>;
    scheduleExport<E extends boolean = false>(projectId: string | number, uploadConfig: {
        url: string;
        http_method?: string;
    }, options?: {
        description?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<{
        message: string;
    }, C, E, void>>;
}

interface ProjectInvitations<C extends boolean = false> {
    add<E extends boolean = false>(projectId: string | number, accessLevel: Exclude<AccessLevel, AccessLevel.ADMIN>, options: OneOf<{
        email: string;
        userId: string;
    }> & {
        expiresAt?: string;
        inviteSource?: string;
        tasksToBeDone?: string[];
        tasksProjectId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<InvitationSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: PaginationRequestOptions<P> & {
        query?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<InvitationSchema[], C, E, P>>;
    edit<E extends boolean = false>(projectId: string | number, email: string, options?: {
        expiresAt?: string;
        accessLevel?: Exclude<AccessLevel, AccessLevel.ADMIN>;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<InvitationSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, email: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<InvitationSchema, C, E, void>>;
}
declare class ProjectInvitations<C extends boolean = false> extends ResourceInvitations<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface ProjectIssueBoardSchema extends IssueBoardSchema {
    project: SimpleProjectSchema;
}
interface ProjectIssueBoards<C extends boolean = false> extends ResourceIssueBoards<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectIssueBoardSchema[], C, E, P>>;
    allLists<E extends boolean = false>(projectId: string | number, boardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueBoardListSchema[], C, E, void>>;
    create<E extends boolean = false>(projectId: string | number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectIssueBoardSchema, C, E, void>>;
    createList<E extends boolean = false>(projectId: string | number, boardId: number, options?: OneOrNoneOf<{
        labelId: number;
        assigneeId: number;
        milestoneId: number;
    }> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueBoardListSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, boardId: number, options?: {
        name?: string;
        assigneeId?: number;
        milestoneId?: number;
        labels?: string;
        weight?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectIssueBoardSchema, C, E, void>>;
    editList<E extends boolean = false>(projectId: string | number, boardId: number, listId: number, position: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueBoardListSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, boardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    removeList<E extends boolean = false>(projectId: string | number, boardId: number, listId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, boardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectIssueBoardSchema, C, E, void>>;
    showList<E extends boolean = false>(projectId: string | number, boardId: number, listId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueBoardListSchema, C, E, void>>;
}
declare class ProjectIssueBoards<C extends boolean = false> extends ResourceIssueBoards<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface ProjectIterations<C extends boolean = false> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: AllIterationsOptions & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<IterationSchema[], C, E, P>>;
}
declare class ProjectIterations<C extends boolean = false> extends ResourceIterations<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface JobTokenScopeSchema extends Record<string, unknown> {
    inbound_enabled: boolean;
    outbound_enabled: boolean;
}
interface ProjectAllowListSchema extends Record<string, unknown> {
    source_project_id: number;
    target_project_id: number;
}
interface GroupAllowListSchema extends Record<string, unknown> {
    source_project_id: number;
    target_group_id: number;
}
declare class ProjectJobTokenScopes<C extends boolean = false> extends BaseResource<C> {
    show<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<JobTokenScopeSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, enabled: boolean, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<JobTokenScopeSchema, C, E, void>>;
    showInboundAllowList<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SimpleProjectSchema[], C, E, void>>;
    addToInboundAllowList<E extends boolean = false>(projectId: string | number, targetProjectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectAllowListSchema, C, E, void>>;
    removeFromInboundAllowList<E extends boolean = false>(projectId: string | number, targetProjectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    showGroupsAllowList<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CondensedGroupSchema[], C, E, void>>;
    addToGroupsAllowList<E extends boolean = false>(projectId: string | number, targetGroupId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupAllowListSchema, C, E, void>>;
    removeFromGroupsAllowList<E extends boolean = false>(projectId: string | number, targetGroupId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface ProjectLabels<C extends boolean = false> extends ResourceLabels<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options: {
        withCounts: true;
        includeAncestorGroups?: boolean;
        search?: string;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<(LabelSchema & LabelCountSchema)[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options: {
        includeAncestorGroups?: boolean;
        search?: string;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: {
        withCounts?: boolean;
        includeAncestorGroups?: boolean;
        search?: string;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, labelName: string, color: string, options?: {
        description?: string;
        priority?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, labelId: number | string, options: OneOf<{
        newName: string;
        color: string;
    }> & {
        description?: string;
        priority?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema, C, E, void>>;
    promote<E extends boolean = false>(projectId: string | number, labelId: number | string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, labelId: number | string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, labelId: number | string, options?: {
        includeAncestorGroups?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema, C, E, void>>;
    subscribe<E extends boolean = false>(projectId: string | number, labelId: number | string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema, C, E, void>>;
    unsubscribe<E extends boolean = false>(projectId: string | number, labelId: number | string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema, C, E, void>>;
}
declare class ProjectLabels<C extends boolean = false> extends ResourceLabels<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface ProjectMarkdownUploads<C extends boolean = false> extends ResourceMarkdownUploads<C> {
    create<E extends boolean = false>(projectId: string | number, file: {
        content: Blob;
        filename: string;
    }, options?: Sudo): Promise<GitlabAPIResponse<MarkdownUploadCreatedSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: Sudo & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<MarkdownUploadSchema[], C, E, P>>;
    download<E extends boolean = false>(projectId: string | number, uploadId: string | number, options?: Sudo): Promise<GitlabAPIResponse<Blob, C, E, void>>;
    download<E extends boolean = false>(projectId: string | number, secret: string, filename: string, options?: Sudo): Promise<GitlabAPIResponse<Blob, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, uploadId: string | number, options?: Sudo): Promise<GitlabAPIResponse<void, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, secret: string, filename: string, options?: Sudo): Promise<GitlabAPIResponse<void, C, E, void>>;
}
declare class ProjectMarkdownUploads<C extends boolean = false> extends ResourceMarkdownUploads<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface ProjectMembers<C extends boolean = false> extends ResourceMembers<C> {
    add<E extends boolean = false>(projectId: string | number, accessLevel: Exclude<AccessLevel, AccessLevel.ADMIN>, options?: AddMemberOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MemberSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: IncludeInherited & PaginationRequestOptions<P> & AllMembersOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MemberSchema[], C, E, P>>;
    edit<E extends boolean = false>(projectId: string | number, userId: number, accessLevel: AccessLevel, options?: {
        expiresAt?: string;
        memberRoleId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MemberSchema, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, userId: number, options?: IncludeInherited & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MemberSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, userId: number, options?: {
        skipSubresourceS?: boolean;
        unassignIssuables?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}
declare class ProjectMembers<C extends boolean = false> extends ResourceMembers<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface ProjectMilestones<C extends boolean = false> extends ResourceMilestones<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: AllMilestonesOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MilestoneSchema[], C, E, P>>;
    allAssignedIssues<E extends boolean = false>(projectId: string | number, milestoneId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueSchema[], C, E, void>>;
    allAssignedMergeRequests<E extends boolean = false>(projectId: string | number, milestoneId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestSchema[], C, E, void>>;
    allBurndownChartEvents<E extends boolean = false>(projectId: string | number, milestoneId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<BurndownChartEventSchema[], C, E, void>>;
    create<E extends boolean = false>(projectId: string | number, title: string, options?: {
        description?: string;
        dueDate?: string;
        startDate?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MilestoneSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, milestoneId: number, options?: {
        title?: string;
        description?: string;
        dueDate?: string;
        startDate?: string;
        stateEvent?: 'close' | 'activate';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MilestoneSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, milestoneId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, milestoneId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MilestoneSchema, C, E, void>>;
}
declare class ProjectMilestones<C extends boolean = false> extends ResourceMilestones<C> {
    constructor(options: BaseResourceOptions<C>);
    promote<E extends boolean = false>(projectId: string | number, milestoneId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface ProjectProtectedEnvironments<C extends boolean = false> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: {
        search?: string;
    } & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<ProtectedEnvironmentSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, name: string, deployAccessLevels: ProtectedEnvironmentAccessLevelEntity[], options?: {
        requiredApprovalCount?: number;
        approvalRules: ProtectedEnvironmentAccessLevelEntity[];
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProtectedEnvironmentSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, name: string, options?: {
        deployAccessLevels?: ProtectedEnvironmentAccessLevelEntity[];
        requiredApprovalCount?: number;
        approvalRules?: ProtectedEnvironmentAccessLevelEntity[];
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProtectedEnvironmentSchema, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProtectedEnvironmentSchema, C, E, void>>;
    unprotect<E extends boolean = false>(projectId: string | number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}
declare class ProjectProtectedEnvironments<C extends boolean = false> extends ResourceProtectedEnvironments<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface ProjectPushRules<C extends boolean = false> extends ResourcePushRules<C> {
    create<E extends boolean = false>(projectId: string | number, options?: CreateAndEditPushRuleOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PushRuleSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, options?: CreateAndEditPushRuleOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PushRuleSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PushRuleSchema, C, E, void>>;
}
declare class ProjectPushRules<C extends boolean = false> extends ResourcePushRules<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface RelationsExportStatusSchema extends Record<string, unknown> {
    relation: string;
    status: number;
    error?: string;
    updated_at: string;
}
declare class ProjectRelationsExport<C extends boolean = false> extends BaseResource<C> {
    download<E extends boolean = false>(projectId: string | number, relation: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    showExportStatus(projectId: string | number, options?: Sudo): Promise<CamelizedResponse<RelationsExportStatusSchema, C>>;
    scheduleExport<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<{
        message: string;
    }, C, E, void>>;
}

interface ReleaseEvidence {
    sha: string;
    filepath: string;
    collected_at: string;
}
interface ReleaseAssetSource {
    format: string;
    url: string;
}
interface ReleaseAssetLink {
    id: number;
    name: string;
    url: string;
    external: boolean;
    link_type: string;
}
interface ReleaseSchema extends Record<string, unknown> {
    name: string | null;
    tag_name: string;
    description: string | null;
    created_at: string;
    released_at: string | null;
    upcoming_release: boolean | null;
    author: MappedOmit<SimpleUserSchema, 'created_at'>;
    commit: CommitSchema;
    milestones: MilestoneSchema[] | null;
    commit_path: string;
    tag_path: string;
    assets: {
        count: number;
        sources?: ReleaseAssetSource[] | null;
        links: ReleaseAssetLink[] | null;
        evidence_file_path: string;
    };
    evidences: ReleaseEvidence[] | null;
    _links: {
        closed_issues_url: string;
        closed_merge_requests_url: string;
        edit_url: string;
        merged_merge_requests_url: string;
        opened_issues_url: string;
        opened_merge_requests_url: string;
        self: string;
    };
}
declare class ProjectReleases<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: PaginationRequestOptions<P> & BaseRequestOptions<E> & {
        includeHtmlDescription: true;
    }): Promise<GitlabAPIResponse<(ReleaseSchema & {
        description_html: string;
    })[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: PaginationRequestOptions<P> & BaseRequestOptions<E>): Promise<GitlabAPIResponse<ReleaseSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, options?: BaseRequestOptions<E>): Promise<GitlabAPIResponse<ReleaseSchema, C, E, void>>;
    createEvidence<E extends boolean = false>(projectId: string | number, tagName: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<number, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, tagName: string, options?: BaseRequestOptions<E>): Promise<GitlabAPIResponse<ReleaseSchema, C, E, void>>;
    download<E extends boolean = false>(projectId: string | number, tagName: string, filepath: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, C, E, void>>;
    downloadLatest<E extends boolean = false>(projectId: string | number, filepath: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, tagName: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, tagName: string, options?: {
        includeHtmlDescription?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ReleaseSchema, C, E, void>>;
    showLatest<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ReleaseSchema, C, E, void>>;
    showLatestEvidence<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ReleaseSchema, C, E, void>>;
}

interface ProjectRepositoryStorageMoveSchema extends RepositoryStorageMoveSchema {
    project: SimpleProjectSchema;
}
interface ProjectRepositoryStorageMoves<C extends boolean = false> extends ResourceRepositoryStorageMoves<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: {
        projectId?: string | number;
    } & PaginationRequestOptions<P> & BaseRequestOptions<E>): Promise<GitlabAPIResponse<ProjectRepositoryStorageMoveSchema[], C, E, P>>;
    show<E extends boolean = false, P extends PaginationTypes = 'offset'>(repositoryStorageId: number, options?: {
        projectId?: string | number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectRepositoryStorageMoveSchema, C, E, P>>;
    schedule<E extends boolean = false, P extends PaginationTypes = 'offset'>(sourceStorageName: string, options?: {
        projectId?: string | number;
        destinationStorageName: any;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectRepositoryStorageMoveSchema, C, E, P>>;
}
declare class ProjectRepositoryStorageMoves<C extends boolean = false> extends ResourceRepositoryStorageMoves<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface ProjectSnippetAwardEmojis<C extends boolean = false> extends ResourceAwardEmojis<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, snippetId: number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema[], C, E, P>>;
    award<E extends boolean = false>(projectId: string | number, snippetId: number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, snippetId: number, awardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, snippetId: number, awardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema, C, E, void>>;
}
declare class ProjectSnippetAwardEmojis<C extends boolean = false> extends ResourceAwardEmojis<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface ProjectSnippetDiscussions<C extends boolean = false> extends ResourceDiscussions<C> {
    addNote<E extends boolean = false>(projectId: string | number, snippetId: string | number, discussionId: string, body: string, options?: {
        createdAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionNoteSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, issueId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, snippetId: string | number, body: string, options?: {
        createdAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema, C, E, void>>;
    editNote<E extends boolean = false>(projectId: string | number, snippetId: string | number, discussionId: string, noteId: number, options: Sudo & ShowExpanded<E> & {
        body: string;
    }): Promise<GitlabAPIResponse<DiscussionNoteSchema, C, E, void>>;
    removeNote<E extends boolean = false>(projectId: string | number, snippetId: string | number, discussionId: string, noteId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, snippetId: string | number, discussionId: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema, C, E, void>>;
}
declare class ProjectSnippetDiscussions<C extends boolean = false> extends ResourceDiscussions<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface SnippetNoteSchema extends NoteSchema {
    noteable_type: 'Snippet';
}
interface ProjectSnippetNotes<C extends boolean = false> extends ResourceNotes<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, snippedId: number, options?: {
        sort?: 'asc' | 'desc';
        orderBy?: 'created_at' | 'updated_at';
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SnippetNoteSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, snippedId: number, body: string, options?: {
        createdAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SnippetNoteSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, snippedId: number, noteId: number, options: {
        body: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SnippetNoteSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, snippedId: number, noteId: number, options?: Sudo): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, snippedId: number, noteId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SnippetNoteSchema, C, E, void>>;
}
declare class ProjectSnippetNotes<C extends boolean = false> extends ResourceNotes<C> {
    constructor(options: BaseResourceOptions<C>);
}

declare class ProjectSnippets<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<SnippetSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, title: string, options?: CreateSnippetOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<CamelizedResponse<ExpandedSnippetSchema, C>, E, undefined>>;
    edit<E extends boolean = false>(projectId: string | number, snippetId: number, options?: EditSnippetOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<CamelizedResponse<ExpandedSnippetSchema, C>, E, undefined>>;
    remove<E extends boolean = false>(projectId: string | number, snippetId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<void, E, undefined>>;
    show<E extends boolean = false>(projectId: string | number, snippetId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SnippetSchema, C, E, void>>;
    showContent<E extends boolean = false>(projectId: string | number, snippetId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<string, C, E, void>>;
    showRepositoryFileContent<E extends boolean = false>(projectId: string | number, snippetId: number, ref: string, filePath: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<string, C, E, void>>;
    showUserAgentDetails<E extends boolean = false>(projectId: string | number, snippetId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<CamelizedResponse<UserAgentDetailSchema, C>, E, undefined>>;
}

interface ProjectStatisticSchema extends Record<string, unknown> {
    fetches: {
        total: number;
        days: {
            count: number;
            date: string;
        }[];
    };
}
declare class ProjectStatistics<C extends boolean = false> extends BaseResource<C> {
    show<E extends boolean = false>(projectId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectStatisticSchema, C, E, void>>;
}

type ProjectTemplateType = 'dockerfiles' | 'gitignores' | 'gitlab_ci_ymls' | 'licenses' | 'issues' | 'merge_requests';
interface ProjectTemplateSchema extends Record<string, unknown> {
    name: string;
    content: string;
}
declare class ProjectTemplates<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, type: ProjectTemplateType, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectTemplateSchema[], C, E, P>>;
    show<E extends boolean = false>(projectId: string | number, type: ProjectTemplateType, name: string, options?: {
        project?: string;
        fullname?: string;
        sourceTemplateProjectId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectTemplateSchema, C, E, void>>;
}

interface TerraformStateSchema extends Record<string, unknown> {
    version: number;
    terraform_version: string;
    serial: number;
    lineage: string;
    outputs?: Record<string, unknown>;
    resources?: Record<string, unknown>[];
    check_results: Record<string, unknown>[];
}
declare class ProjectTerraformState<C extends boolean = false> extends BaseResource<C> {
    show<E extends boolean = false>(projectId: string | number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TerraformStateSchema, C, E, void>>;
    showVersion<E extends boolean = false>(projectId: string | number, name: string, serial: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TerraformStateSchema, C, E, void>>;
    removeVersion<E extends boolean = false>(projectId: string | number, name: string, serial: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    removeTerraformStateLock<E extends boolean = false>(projectId: string | number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    createVersion<E extends boolean = false>(projectId: string | number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TerraformStateSchema, C, E, void>>;
}

interface ProjectVariableSchema extends VariableSchema {
    environment_scope: string;
}
interface ProjectVariables<C extends boolean = false> extends ResourceVariables<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<ProjectVariableSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, key: string, value: string, options?: {
        variableType?: VariableType;
        protected?: boolean;
        masked?: boolean;
        masked_and_hidden?: boolean;
        environmentScope?: string;
        description?: string;
        raw?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectVariableSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, key: string, value: string, options?: {
        variableType?: VariableType;
        protected?: boolean;
        masked?: boolean;
        masked_and_hidden?: boolean;
        environmentScope?: string;
        raw?: boolean;
        filter: VariableFilter;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectVariableSchema, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, key: string, options?: {
        filter?: VariableFilter;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectVariableSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, key: string, options?: {
        filter?: VariableFilter;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}
declare class ProjectVariables<C extends boolean = false> extends ResourceVariables<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface ProjectVulnerabilitySchema extends Record<string, unknown> {
    author_id: number;
    confidence: string;
    created_at: string;
    description?: string;
    dismissed_at?: string;
    dismissed_by_id?: number;
    due_date?: string;
    finding: {
        confidence: string;
        created_at: string;
        id: number;
        location_fingerprint: string;
        metadata_version: string;
        name: string;
        primary_identifier_id: number;
        project_fingerprint: string;
        project_id: number;
        raw_metadata: string;
        report_type: string;
        scanner_id: number;
        severity: string;
        updated_at: string;
        uuid: string;
        vulnerability_id: number;
    };
    id: number;
    last_edited_at?: string;
    last_edited_by_id?: number;
    project: SimpleProjectSchema;
    project_default_branch: string;
    report_type: string;
    resolved_at?: string;
    resolved_by_id?: number;
    resolved_on_default_branch: boolean;
    severity: string;
    start_date?: string;
    state: string;
    title: string;
    updated_at: string;
    updated_by_id?: number;
}
declare class ProjectVulnerabilities<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectVulnerabilitySchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, findingId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProjectVulnerabilitySchema, C, E, void>>;
}

interface ProjectWikis<C extends boolean = false> extends ResourceWikis<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options: {
        withContent: true;
    } & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<(WikiSchema & {
        content: string;
    })[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<WikiSchema[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: {
        withContent?: boolean;
    } & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<WikiSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, content: string, title: string, options?: {
        format?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<WikiSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, slug: string, options?: OneOf<{
        content: string;
        title: string;
    }> & {
        format?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<WikiSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, slug: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, slug: string, options?: {
        renderHtml?: boolean;
        version?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<WikiSchema, C, E, void>>;
    uploadAttachment<E extends boolean = false>(projectId: string | number, file: {
        content: Blob;
        filename: string;
    }, options?: {
        branch?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<WikiAttachmentSchema, C, E, void>>;
}
declare class ProjectWikis<C extends boolean = false> extends ResourceWikis<C> {
    constructor(options: BaseResourceOptions<C>);
}

type ProtectedTagAccessLevel = AccessLevel.NO_ACCESS | AccessLevel.DEVELOPER | AccessLevel.MAINTAINER | AccessLevel.ADMIN;
interface ProtectedTagAccessLevelSummarySchema {
    id: number;
    access_level: ProtectedTagAccessLevel;
    access_level_description: string;
}
interface ProtectedTagSchema extends Record<string, unknown> {
    name: string;
    create_access_levels?: ProtectedTagAccessLevelSummarySchema[];
}
type ProtectedTagAccessLevelEntity = OneOf<{
    userId: number;
    groupId: number;
    accessLevel: ProtectedTagAccessLevel;
}>;
declare class ProtectedTags<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProtectedTagSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, tagName: string, options?: {
        createAccessLevel?: ProtectedTagAccessLevel;
        allowedToCreate?: ProtectedTagAccessLevelEntity[];
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProtectedTagSchema, C, E, void>>;
    protect<E extends boolean = false>(projectId: string | number, tagName: string, options?: {
        createAccessLevel?: ProtectedTagAccessLevel;
        allowedToCreate?: ProtectedTagAccessLevelEntity[];
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProtectedTagSchema, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, tagName: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProtectedTagSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, tagName: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    unprotect<E extends boolean = false>(projectId: string | number, tagName: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface ReleaseLinkSchema extends Record<string, unknown> {
    id: number;
    name: string;
    url: string;
    external: boolean;
    link_type: string;
}
declare class ReleaseLinks<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, tagName: string, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<ReleaseLinkSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, tagName: string, name: string, url: string, options?: Sudo & {
        filePath?: string;
        linkType?: string;
        directAssetPath?: string;
    }): Promise<GitlabAPIResponse<ReleaseLinkSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, tagName: string, linkId: number, options?: Sudo & ShowExpanded<E> & {
        name?: string;
        url?: string;
        filePath?: string;
        linkType?: string;
        directAssetPath?: string;
    }): Promise<GitlabAPIResponse<ReleaseLinkSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, tagName: string, linkId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, tagName: string, linkId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<CamelizedResponse<ReleaseLinkSchema, C>, E, undefined>>;
}

type ArchiveType = 'tar.gz' | 'tar.bz2' | 'tbz' | 'tbz2' | 'tb2' | 'bz2' | 'tar' | 'zip';
interface RepositoryChangelogSchema extends Record<string, unknown> {
    notes: string;
}
interface RepositoryCompareSchema extends Record<string, unknown> {
    commit: MappedOmit<CondensedCommitSchema, 'message'>;
    commits?: MappedOmit<CondensedCommitSchema, 'message'>[];
    diffs?: CommitDiffSchema[];
    compare_timeout: boolean;
    compare_same_ref: boolean;
}
interface RepositoryContributorSchema extends Record<string, unknown> {
    name: string;
    email: string;
    commits: number;
    additions: number;
    deletions: number;
}
interface RepositoryTreeSchema extends Record<string, unknown> {
    id: string;
    name: string;
    type: string;
    path: string;
    mode: string;
}
interface RepositoryBlobSchema extends Record<string, unknown> {
    size: number;
    encoding: string;
    content: string;
    sha: string;
}
type AllRepositoryTreesOptions = {
    pageToken?: string;
    path?: string;
    recursive?: boolean;
    ref?: string;
};
type EditChangelogOptions = {
    branch?: string;
    configFile?: string;
    date?: string;
    file?: string;
    from?: string;
    message?: string;
    to?: string;
    trailer?: string;
};
type ShowChangelogOptions = {
    configFile?: string;
    date?: string;
    from?: string;
    to?: string;
    trailer?: string;
};
declare class Repositories<C extends boolean = false> extends BaseResource<C> {
    allContributors<E extends boolean = false>(projectId: string | number, options?: {
        orderBy?: string;
        sort?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RepositoryContributorSchema[], C, E, void>>;
    allRepositoryTrees<E extends boolean = false>(projectId: string | number, options?: AllRepositoryTreesOptions & PaginationRequestOptions<'keyset'> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RepositoryTreeSchema[], C, E, 'keyset'>>;
    compare<E extends boolean = false>(projectId: string | number, from: string, to: string, options?: {
        fromProjectId?: string | number;
        straight?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RepositoryCompareSchema, C, E, void>>;
    editChangelog<E extends boolean = false>(projectId: string | number, version: string, options?: EditChangelogOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RepositoryChangelogSchema, C, E, void>>;
    mergeBase<E extends boolean = false>(projectId: string | number, refs: string[], options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CommitSchema, C, E, void>>;
    showArchive<E extends boolean = false>(projectId: string | number, options: {
        fileType?: ArchiveType;
        sha?: string;
        path?: string;
        asStream: true;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ReadableStream, void, E, void>>;
    showArchive<E extends boolean = false>(projectId: string | number, options?: {
        fileType?: ArchiveType;
        sha?: string;
        path?: string;
        asStream?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    showBlob<E extends boolean = false>(projectId: string | number, sha: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RepositoryBlobSchema, void, E, void>>;
    showBlobRaw<E extends boolean = false>(projectId: string | number, sha: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    showChangelog<E extends boolean = false>(projectId: string | number, version: string, options?: ShowChangelogOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RepositoryChangelogSchema, C, E, void>>;
}

interface RepositoryFileExpandedSchema extends Record<string, unknown> {
    file_name: string;
    file_path: string;
    size: number;
    encoding: string;
    content: string;
    content_sha256: string;
    ref: string;
    blob_id: string;
    commit_id: string;
    last_commit_id: string;
}
interface RepositoryFileBlameSchema extends Record<string, unknown> {
    commit: CommitSchema;
    lines?: string[];
}
interface RepositoryFileSchema extends Record<string, unknown> {
    file_path: string;
    branch: string;
}
type CreateRepositoryFileOptions = {
    authorEmail?: string;
    authorName?: string;
    encoding?: string;
    executeFilemode?: boolean;
    startBranch?: string;
};
type EditRepositoryFileOptions = {
    authorEmail?: string;
    authorName?: string;
    encoding?: string;
    executeFilemode?: boolean;
    startBranch?: string;
    lastCommitId?: string;
};
type RemoveRepositoryFileOptions = {
    authorEmail?: string;
    authorName?: string;
    startBranch?: string;
    lastCommitId?: string;
};
declare class RepositoryFiles<C extends boolean = false> extends BaseResource<C> {
    allFileBlames<E extends boolean = false>(projectId: string | number, filePath: string, ref: string, options?: {
        range?: {
            start: number;
            end: number;
        };
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RepositoryFileBlameSchema[], C, E, void>>;
    create<E extends boolean = false>(projectId: string | number, filePath: string, branch: string, content: string, commitMessage: string, options?: CreateRepositoryFileOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RepositoryFileSchema, C, E, void>>;
    edit<E extends boolean = false>(projectId: string | number, filePath: string, branch: string, content: string, commitMessage: string, options?: EditRepositoryFileOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RepositoryFileSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, filePath: string, branch: string, commitMessage: string, options?: RemoveRepositoryFileOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, filePath: string, ref: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RepositoryFileExpandedSchema, C, E, void>>;
    showRaw<E extends boolean = false>(projectId: string | number, filePath: string, ref: string, options?: {
        lfs?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<string | Blob, C, E, void>>;
}

interface RepositorySubmoduleSchema extends CommitSchema {
    status?: string;
}
declare class RepositorySubmodules<C extends boolean = false> extends BaseResource<C> {
    edit<E extends boolean = false>(projectId: string | number, submodule: string, branch: string, commitSha: string, options?: {
        commitMessage?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RepositorySubmoduleSchema, C, E, void>>;
}

interface ResourceGroupSchema extends Record<string, unknown> {
    id: number;
    key: string;
    process_mode: string;
    created_at: string;
    updated_at: string;
}
declare class ResourceGroups<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<ResourceGroupSchema[], C, E, P>>;
    edit<E extends boolean = false>(projectId: string | number, key: string, options?: {
        processMode?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<CamelizedResponse<ResourceGroupSchema, C>, E, undefined>>;
    show<E extends boolean = false>(projectId: string | number, key: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ResourceGroupSchema, C, E, void>>;
    allUpcomingJobs<E extends boolean = false>(projectId: string | number, key: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<JobSchema[], C, E, void>>;
}

interface SecureFileSchema extends Record<string, unknown> {
    id: number;
    name: string;
    checksum: string;
    checksum_algorithm: string;
    created_at: string;
    expires_at?: string;
    metadata?: {
        id: string;
        issuer: {
            C: string;
            O: string;
            CN: string;
            OU: string;
        };
        subject: {
            C: string;
            O: string;
            CN: string;
            OU: string;
            UID: string;
        };
        expires_at: string;
    };
}
declare class SecureFiles<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SecureFileSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, name: string, file: {
        content: Blob;
        filename: string;
    }, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SecureFileSchema, C, E, void>>;
    download<E extends boolean = false>(projectId: string | number, secureFileId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, secureFileId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, secureFileId: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SecureFileSchema, C, E, void>>;
}

interface TagSchema extends Record<string, unknown> {
    commit: CommitSchema;
    release: Pick<ReleaseSchema, 'tag_name' | 'description'>;
    name: string;
    target: string;
    message?: string;
    protected: boolean;
    created_at?: string;
}
interface TagSignatureSchema extends Record<string, unknown> {
    signature_type: string;
    verification_status: string;
    x509_certificate: {
        id: number;
        subject: string;
        subject_key_identifier: string;
        email: string;
        serial_number: number;
        certificate_status: string;
        x509_issuer: {
            id: number;
            subject: string;
            subject_key_identifier: string;
            crl_url: string;
        };
    };
}
declare class Tags<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: {
        orderBy?: 'name' | 'updated' | 'version';
        sort?: 'asc' | 'desc';
        search?: string;
    } & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<TagSchema[], C, E, P>>;
    create<E extends boolean = false>(projectId: string | number, tagName: string, ref: string, options?: {
        message?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TagSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, tagName: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, tagName: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TagSchema, C, E, void>>;
    showSignature<E extends boolean = false>(projectId: string | number, tagName: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<TagSignatureSchema | {
        message: string;
    }, C, E, void>>;
}

interface StarredDashboardSchema extends Record<string, unknown> {
    id: number;
    dashboard_path: string;
    user_id: number;
    project_id: number;
}
declare class UserStarredMetricsDashboard<C extends boolean = false> extends BaseResource<C> {
    create<E extends boolean = false>(projectId: string | number, dashboardPath: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<StarredDashboardSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, options?: {
        dashboard_path?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<{
        deleted_rows: number;
    }, C, E, void>>;
}

interface EpicAwardEmojis<C extends boolean = false> extends ResourceAwardEmojis<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(epicId: string | number, issueIId: number, options?: PaginationRequestOptions<P> & BaseRequestOptions<E>): Promise<GitlabAPIResponse<AwardEmojiSchema[], C, E, P>>;
    award<E extends boolean = false>(epicId: string | number, issueIId: number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema, C, E, void>>;
    remove<E extends boolean = false>(epicId: string | number, issueIId: number, awardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(epicId: string | number, issueIId: number, awardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AwardEmojiSchema, C, E, void>>;
}
declare class EpicAwardEmojis<C extends boolean = false> extends ResourceAwardEmojis<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface EpicDiscussions<C extends boolean = false> extends ResourceDiscussions<C> {
    addNote<E extends boolean = false>(groupId: string | number, epicId: number, discussionId: string, body: string, options?: {
        createdAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionNoteSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, epicId: number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema[], C, E, P>>;
    create<E extends boolean = false>(groupId: string | number, epicId: number, body: string, options?: {
        createdAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema, C, E, void>>;
    editNote<E extends boolean = false>(groupId: string | number, epicId: number, discussionId: string, noteId: number, options: Sudo & ShowExpanded<E> & {
        body: string;
    }): Promise<GitlabAPIResponse<DiscussionNoteSchema, C, E, void>>;
    removeNote<E extends boolean = false>(groupId: string | number, epicId: number, discussionId: string, noteId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(groupId: string | number, epicId: number, discussionId: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<DiscussionSchema, C, E, void>>;
}
declare class EpicDiscussions<C extends boolean = false> extends ResourceDiscussions<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface EpicIssueSchema extends MappedOmit<IssueSchema, 'references' | 'task_completion_status'> {
    epic_issue_id: number;
}
interface ExpandedEpicIssueSchema extends EpicIssueSchema {
    subscribed: boolean;
    relative_position: number;
}
declare class EpicIssues<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, epicIId: number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<EpicIssueSchema[], C, E, P>>;
    assign<E extends boolean = false>(groupId: string | number, epicIId: number, epicIssueId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<EpicIssueSchema, C, E, void>>;
    edit<E extends boolean = false>(groupId: string | number, epicIId: number, epicIssueId: number, options?: {
        moveBeforeId?: number;
        moveAfterId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ExpandedEpicIssueSchema, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, epicIId: number, epicIssueId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface EpicLabelEvents<C extends boolean = false> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, epidId: number, options?: BaseRequestOptions<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<LabelEventSchema[], C, E, P>>;
    show<E extends boolean = false>(groupId: string | number, epidId: number, labelEventId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelEventSchema, C, E, void>>;
}
declare class EpicLabelEvents<C extends boolean = false> extends ResourceLabelEvents<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface CondensedEpicLinkSchema extends Record<string, unknown> {
    id: number;
    iid: number;
    group_id: number;
    parent_id: number;
    title: string;
    has_children: boolean;
    has_issues: boolean;
    reference: string;
    url: string;
    relation_url: string;
}
interface EpicLinkSchema extends Record<string, unknown> {
    id: number;
    iid: number;
    group_id: number;
    parent_id: number;
    title: string;
    description: string;
    author: SimpleUserSchema;
    start_date?: string;
    start_date_is_fixed: boolean;
    start_date_fixed?: string;
    start_date_from_milestones?: string;
    start_date_from_inherited_source?: string;
    due_date: string;
    due_date_is_fixed: boolean;
    due_date_fixed?: string;
    due_date_from_milestones?: string;
    due_date_from_inherited_source: string;
    created_at: string;
    updated_at: string;
    labels?: string[];
}
declare class EpicLinks<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, epicIId: number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<EpicLinkSchema[], C, E, P>>;
    assign<E extends boolean = false>(groupId: string | number, epicIId: number, childEpicId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<EpicLinkSchema, C, E, void>>;
    create<E extends boolean = false>(groupId: string | number, epicIId: number, title: string, options?: {
        confidential?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CondensedEpicLinkSchema, C, E, void>>;
    reorder<E extends boolean = false>(groupId: string | number, epicIId: number, childEpicId: number, options?: {
        moveBeforeId?: number;
        moveAfterId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<EpicLinkSchema, C, E, void>>;
    unassign<E extends boolean = false>(groupId: string | number, epicIId: number, childEpicId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<EpicLinkSchema, C, E, void>>;
}

interface EpicNoteSchema extends NoteSchema {
    noteable_type: 'Epic';
}
interface EpicNotes<C extends boolean = false> extends ResourceNotes<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, epicId: number, options?: {
        sort?: 'asc' | 'desc';
        orderBy?: 'created_at' | 'updated_at';
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<EpicNoteSchema[], C, E, P>>;
    create<E extends boolean = false>(groupId: string | number, epicId: number, body: string, options?: {
        internal?: boolean;
        createdAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<EpicNoteSchema, C, E, void>>;
    edit<E extends boolean = false>(groupId: string | number, epicId: number, noteId: number, options: {
        body: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<EpicNoteSchema, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, epicId: number, noteId: number, options?: Sudo): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(groupId: string | number, epicId: number, noteId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<EpicNoteSchema, C, E, void>>;
}
declare class EpicNotes<C extends boolean = false> extends ResourceNotes<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface EpicSchema extends Record<string, unknown> {
    id: number;
    iid: number;
    group_id: number;
    parent_id: number;
    parent_iid: number;
    title: string;
    description: string;
    state: string;
    confidential: string;
    web_url: string;
    references: {
        short: string;
        relative: string;
        full: string;
    };
    author: MappedOmit<SimpleUserSchema, 'created_at'>;
    start_date?: string;
    start_date_is_fixed: boolean;
    start_date_fixed?: string;
    start_date_from_inherited_source?: string;
    due_date: string;
    due_date_is_fixed: boolean;
    due_date_fixed?: string;
    due_date_from_inherited_source: string;
    created_at: string;
    updated_at: string;
    closed_at: string;
    labels: string[] | SimpleLabelSchema[];
    upvotes: number;
    downvotes: number;
    _links: {
        self: string;
        epic_issues: string;
        group: string;
    };
}
interface EpicSchemaWithExpandedLabels extends EpicSchema {
    labels: SimpleLabelSchema[];
}
interface EpicSchemaWithBasicLabels extends EpicSchema {
    labels: string[];
}
interface EpicTodoSchema extends TodoSchema {
    group: Pick<GroupSchema, 'id' | 'name' | 'path' | 'kind' | 'full_path' | 'parent_id'>;
    target_type: 'Epic';
    target: EpicSchema;
}
type AllEpicsOptions = {
    authorId?: number;
    authorUsername?: string;
    labels?: string;
    withLabelsDetails?: boolean;
    orderBy?: 'created_at' | 'updated_at' | 'title';
    sort?: string;
    search?: string;
    state?: string;
    createdAfter?: string;
    createdBefore?: string;
    updatedAfter?: string;
    updatedBefore?: string;
    includeAncestorGroups?: boolean;
    includeDescendantGroups?: boolean;
    myReactionEmoji?: string;
    not?: Record<string, string>;
};
type CreateEpicOptions = {
    labels?: string;
    description?: string;
    color?: string;
    confidential?: boolean;
    createdAt?: string;
    startDateIsFixed?: boolean;
    startDateFixed?: string;
    dueDateIsFixed?: boolean;
    dueDateFixed?: string;
    parentId?: number | string;
};
type EditEpicOptions = {
    addLabels?: string;
    confidential?: boolean;
    description?: string;
    dueDateFixed?: string;
    dueDateIsFixed?: boolean;
    labels?: string;
    parentId?: number | string;
    removeLabels?: string;
    startDateFixed?: string;
    startDateIsFixed?: boolean;
    stateEvent?: string;
    title?: string;
    updatedAt?: string;
    color?: string;
};
declare class Epics<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options: AllEpicsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E> & {
        withLabelsDetails: true;
    }): Promise<GitlabAPIResponse<EpicSchemaWithExpandedLabels[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: AllEpicsOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E> & {
        withLabelsDetails?: false;
    }): Promise<GitlabAPIResponse<EpicSchemaWithBasicLabels[], C, E, P>>;
    create<E extends boolean = false>(groupId: string | number, title: string, options?: CreateEpicOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<EpicSchema, C, E, void>>;
    createTodo<E extends boolean = false>(groupId: string | number, epicIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<EpicTodoSchema, C, E, void>>;
    edit<E extends boolean = false>(groupId: string | number, epicIId: number, options?: EditEpicOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MappedOmit<EpicSchema, '_links'>, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, epicIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(groupId: string | number, epicIId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<EpicSchema, C, E, void>>;
}

interface GroupAccessRequests<C extends boolean = false> extends ResourceAccessRequests<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AccessRequestSchema[], C, E, P>>;
    request<E extends boolean = false>(groupId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AccessRequestSchema, C, E, void>>;
    approve<E extends boolean = false>(groupId: string | number, userId: number, options?: {
        accessLevel?: Exclude<AccessLevel, AccessLevel.ADMIN>;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AccessRequestSchema, C, E, void>>;
    deny<E extends boolean = false>(groupId: string | number, userId: number): Promise<GitlabAPIResponse<void, C, E, void>>;
}
declare class GroupAccessRequests<C extends boolean = false> extends ResourceAccessRequests<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface GroupAccessTokens<C extends boolean = false> extends ResourceAccessTokens<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<AccessTokenSchema[], C, E, P>>;
    create<E extends boolean = false>(groupId: string | number, name: string, scopes: AccessTokenScopes[], expiresAt: string, options?: {
        accessLevel?: Exclude<AccessLevel, AccessLevel.MINIMAL_ACCESS | AccessLevel.NO_ACCESS | AccessLevel.ADMIN>;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AccessTokenExposedSchema, C, E, void>>;
    revoke<E extends boolean = false>(groupId: string | number, tokenId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    rotate<E extends boolean = false>(groupId: string | number, tokenId: string | number, options?: {
        expiresAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AccessTokenExposedSchema, C, E, void>>;
    show<E extends boolean = false>(groupId: string | number, tokenId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<AccessTokenSchema, C, E, void>>;
}
declare class GroupAccessTokens<C extends boolean = false> extends ResourceAccessTokens<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface GroupAnalyticsIssuesCountSchema extends Record<string, unknown> {
    issues_count: number;
}
interface GroupAnalyticsMRsCountSchema extends Record<string, unknown> {
    merge_requests_count: number;
}
interface GroupAnalyticsNewMembersCountSchema extends Record<string, unknown> {
    new_members_count: number;
}
declare class GroupActivityAnalytics<C extends boolean = false> extends BaseResource<C> {
    showIssuesCount<E extends boolean = false>(groupPath: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupAnalyticsIssuesCountSchema, C, E, void>>;
    showMergeRequestsCount<E extends boolean = false>(groupPath: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupAnalyticsMRsCountSchema, C, E, void>>;
    showNewMembersCount<E extends boolean = false>(groupPath: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupAnalyticsNewMembersCountSchema, C, E, void>>;
}

interface GroupBadgeSchema extends BadgeSchema {
    kind: 'group';
}
interface GroupBadges<C extends boolean = false> extends ResourceBadges<C> {
    add<E extends boolean = false>(groupId: string | number, linkUrl: string, imageUrl: string, options?: {
        name?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupBadgeSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: {
        name?: string;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupBadgeSchema[], C, E, P>>;
    edit<E extends boolean = false>(groupId: string | number, badgeId: number, options?: EditBadgeOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupBadgeSchema, C, E, void>>;
    preview<E extends boolean = false>(groupId: string | number, linkUrl: string, imageUrl: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CondensedBadgeSchema, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, badgeId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(groupId: string | number, badgeId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupBadgeSchema, C, E, void>>;
}
declare class GroupBadges<C extends boolean = false> extends ResourceBadges<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface GroupCustomAttributes<C extends boolean = false> extends ResourceCustomAttributes<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CustomAttributeSchema[], C, E, P>>;
    set<E extends boolean = false>(groupId: string | number, customAttributeId: string, value: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CustomAttributeSchema, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, customAttributeId: string, options?: Sudo): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(groupId: string | number, customAttributeId: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CustomAttributeSchema, C, E, void>>;
}
declare class GroupCustomAttributes<C extends boolean = false> extends ResourceCustomAttributes<C> {
    constructor(options: BaseResourceOptions<C>);
}

declare class GroupDORA4Metrics<C extends boolean = false> extends ResourceDORA4Metrics<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface GroupEpicBoardListSchema extends Record<string, unknown> {
    id: number;
    label: Pick<LabelSchema, 'id' | 'name' | 'color' | 'description'>;
    position: number;
    list_type: 'label';
    collapsed: boolean;
}
interface GroupEpicBoardSchema extends Record<string, unknown> {
    id: number;
    name: string;
    group: CondensedGroupSchema;
    hide_backlog_list: boolean;
    hide_closed_list: boolean;
    labels: LabelSchema[] | null;
    lists: GroupEpicBoardListSchema[] | null;
}
declare class GroupEpicBoards<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupEpicBoardSchema[], C, E, P>>;
    allLists<E extends boolean = false>(groupId: string | number, boardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupEpicBoardListSchema[], C, E, void>>;
    show<E extends boolean = false>(groupId: string | number, boardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupEpicBoardSchema, C, E, void>>;
    showList<E extends boolean = false>(groupId: string | number, boardId: number, listId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupEpicBoardListSchema, C, E, void>>;
}

interface GroupHookSchema extends ExpandedHookSchema {
    groupId: number;
    subgroup_events: boolean;
}
interface GroupHooks<C extends boolean = false> {
    add<E extends boolean = false>(groupId: string | number, url: string, options?: AddResourceHookOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupHookSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupHookSchema[], C, E, P>>;
    edit<E extends boolean = false>(groupId: string | number, hookId: number, url: string, options?: EditResourceHookOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupHookSchema, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, hookId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(groupId: string | number, hookId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupHookSchema, C, E, void>>;
}
declare class GroupHooks<C extends boolean = false> extends ResourceHooks<C> {
    constructor(options: BaseResourceOptions<C>);
}

declare class GroupImportExports<C extends boolean = false> extends BaseResource<C> {
    download<E extends boolean = false>(groupId: string | number, options: {
        asStream: true;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ReadableStream, void, E, void>>;
    download<E extends boolean = false>(groupId: string | number, options?: {
        asStream?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    import<E extends boolean = false>(file: {
        content: Blob;
        filename: string;
    }, path: string, { parentId, name, ...options }: {
        parentId?: number;
        name?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ImportStatusSchema, C, E, void>>;
    scheduleExport<E extends boolean = false>(groupId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<{
        message: string;
    }, C, E, void>>;
}

interface GroupInvitations<C extends boolean = false> {
    add<E extends boolean = false>(groupId: string | number, accessLevel: Exclude<AccessLevel, AccessLevel.ADMIN>, options: OneOf<{
        email: string;
        userId: string;
    }> & {
        expiresAt?: string;
        inviteSource?: string;
        tasksToBeDone?: string[];
        tasksProjectId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<InvitationSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: PaginationRequestOptions<P> & {
        query?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<InvitationSchema[], C, E, P>>;
    edit<E extends boolean = false>(groupId: string | number, email: string, options?: {
        expiresAt?: string;
        accessLevel?: Exclude<AccessLevel, AccessLevel.ADMIN>;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<InvitationSchema, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, email: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<InvitationSchema, C, E, void>>;
}
declare class GroupInvitations<C extends boolean = false> extends ResourceInvitations<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface GrouptIssueBoardSchema extends IssueBoardSchema {
    group: CondensedGroupSchema;
}
interface GroupIssueBoards<C extends boolean = false> extends ResourceIssueBoards<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GrouptIssueBoardSchema[], C, E, P>>;
    allLists<E extends boolean = false>(resourceId: string | number, boardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueBoardListSchema[], C, E, void>>;
    create<E extends boolean = false>(groupId: string | number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GrouptIssueBoardSchema, C, E, void>>;
    createList<E extends boolean = false>(groupId: string | number, boardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueBoardListSchema, C, E, void>>;
    edit<E extends boolean = false>(groupId: string | number, boardId: number, options?: {
        hideBacklogList?: boolean;
        hideClosedList?: boolean;
        name?: string;
        assigneeId?: number;
        milestoneId?: number;
        labels?: string;
        weight?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GrouptIssueBoardSchema, C, E, void>>;
    editList<E extends boolean = false>(groupId: string | number, boardId: number, listId: number, position: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueBoardListSchema, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, boardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    removeList<E extends boolean = false>(groupId: string | number, boardId: number, listId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(groupId: string | number, boardId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GrouptIssueBoardSchema, C, E, void>>;
    showList<E extends boolean = false>(groupId: string | number, boardId: number, listId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueBoardListSchema, C, E, void>>;
}
declare class GroupIssueBoards<C extends boolean = false> extends ResourceIssueBoards<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface GroupIterations<C extends boolean = false> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: AllIterationsOptions & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<IterationSchema[], C, E, P>>;
}
declare class GroupIterations<C extends boolean = false> extends ResourceIterations<C> {
    constructor(options: BaseResourceOptions<C>);
}

declare class GroupLDAPLinks<C extends boolean = false> extends BaseResource<C> {
    add<E extends boolean = false>(groupId: string | number, groupAccess: number, provider: string, options?: {
        cn?: string;
        groupAccess?: Exclude<AccessLevel, AccessLevel.ADMIN>;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<string, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<string[], C, E, P>>;
    remove<E extends boolean = false>(groupId: string | number, provider: string, options?: {
        cn?: string;
        filter?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    sync(groupId: string | number, options?: Sudo & ShowExpanded): Promise<string | number | void | ReadableStream<any> | Blob | string[] | CamelizedResponse<Record<string, unknown>, C> | CamelizedResponse<Record<string, unknown>, C>[] | null>;
}

interface GroupLabels<C extends boolean = false> extends ResourceLabels<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options: {
        withCounts: true;
        includeAncestorGroups?: boolean;
        search?: string;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<(LabelSchema & LabelCountSchema)[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options: {
        includeAncestorGroups?: boolean;
        search?: string;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: {
        withCounts?: boolean;
        includeAncestorGroups?: boolean;
        search?: string;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema[], C, E, P>>;
    create<E extends boolean = false>(groupId: string | number, labelName: string, color: string, options?: {
        description?: string;
        priority?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema, C, E, void>>;
    edit<E extends boolean = false>(groupId: string | number, labelId: number | string, options: OneOf<{
        newName: string;
        color: string;
    }> & {
        description?: string;
        priority?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema, C, E, void>>;
    promote<E extends boolean = false>(groupId: string | number, labelId: number | string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, labelId: number | string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(groupId: string | number, labelId: number | string, options?: {
        includeAncestorGroups?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema, C, E, void>>;
    subscribe<E extends boolean = false>(groupId: string | number, labelId: number | string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema, C, E, void>>;
    unsubscribe<E extends boolean = false>(groupId: string | number, labelId: number | string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<LabelSchema, C, E, void>>;
}
declare class GroupLabels<C extends boolean = false> extends ResourceLabels<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface GroupMarkdownUploads<C extends boolean = false> extends ResourceMarkdownUploads<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: Sudo & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<MarkdownUploadSchema[], C, E, P>>;
    download<E extends boolean = false>(groupId: string | number, uploadId: string | number, options?: Sudo): Promise<GitlabAPIResponse<Blob, C, E, void>>;
    download<E extends boolean = false>(projectId: string | number, secret: string, filename: string, options?: Sudo): Promise<GitlabAPIResponse<Blob, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, uploadId: string | number, options?: Sudo): Promise<GitlabAPIResponse<void, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, secret: string, filename: string, options?: Sudo): Promise<GitlabAPIResponse<void, C, E, void>>;
}
declare class GroupMarkdownUploads<C extends boolean = false> extends ResourceMarkdownUploads<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface MemberRoleSchema extends Record<string, unknown> {
    id: number;
    group_id: number;
    base_access_level: number;
    read_code: boolean;
}
declare class GroupMemberRoles<C extends boolean = false> extends BaseResource<C> {
    add<E extends boolean = false>(groupId: string | number, baseAccessLevel: Exclude<AccessLevel, AccessLevel.NO_ACCESS | AccessLevel.MINIMAL_ACCESS | AccessLevel.ADMIN>, options?: {
        readCode?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MemberRoleSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MemberRoleSchema[], C, E, P>>;
    remove<E extends boolean = false>(groupId: string | number, memberRoleId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface BillableGroupMemberSchema extends CondensedMemberSchema {
    last_activity_on: string;
    membership_type: string;
    removable: boolean;
    created_at: string;
}
interface BillableGroupMemberMembershipSchema extends Record<string, unknown> {
    id: number;
    source_id: number;
    source_full_name: string;
    source_members_url: string;
    created_at: string;
    expires_at: string;
    access_level: {
        string_value: string;
        integer_value: Exclude<AccessLevel, AccessLevel.ADMIN>;
    };
}
interface OverrodeGroupMemberSchema extends SimpleMemberSchema {
    override: boolean;
}
interface GroupMembers<C extends boolean = false> extends ResourceMembers<C> {
    add<E extends boolean = false>(projectId: string | number, accessLevel: Exclude<AccessLevel, AccessLevel.ADMIN>, options?: AddMemberOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MemberSchema, C, E, void>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, options?: IncludeInherited & PaginationRequestOptions<P> & AllMembersOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MemberSchema[], C, E, P>>;
    edit<E extends boolean = false>(projectId: string | number, userId: number, accessLevel: Exclude<AccessLevel, AccessLevel.ADMIN>, options?: {
        expiresAt?: string;
        memberRoleId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MemberSchema, C, E, void>>;
    show<E extends boolean = false>(projectId: string | number, userId: number, options?: IncludeInherited & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MemberSchema, C, E, void>>;
    remove<E extends boolean = false>(projectId: string | number, userId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}
declare class GroupMembers<C extends boolean = false> extends ResourceMembers<C> {
    constructor(options: BaseResourceOptions<C>);
    allBillable<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<BillableGroupMemberSchema[], C, E, P>>;
    allPending<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MemberSchema[], C, E, P>>;
    allBillableMemberships<E extends boolean = false>(groupId: string | number, userId: number, options?: {
        search?: string;
        sort?: 'access_level_asc' | 'access_level_desc' | 'last_joined' | 'name_asc' | 'name_desc' | 'oldest_joined' | 'oldest_sign_in' | 'recent_sign_in' | 'last_activity_on_asc' | 'last_activity_on_desc';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<BillableGroupMemberMembershipSchema[], C, E, void>>;
    approve<E extends boolean = false>(groupId: string | number, userId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MemberSchema, C, E, void>>;
    approveAll<E extends boolean = false>(groupId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MemberSchema[], C, E, void>>;
    removeBillable<E extends boolean = false>(groupId: string | number, userId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    removeOverrideFlag<E extends boolean = false>(groupId: string | number, userId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<OverrodeGroupMemberSchema, C, E, void>>;
    setOverrideFlag<E extends boolean = false>(groupId: string | number, userId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<OverrodeGroupMemberSchema, C, E, void>>;
}

interface GroupMilestones<C extends boolean = false> extends ResourceMilestones<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: AllMilestonesOptions & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MilestoneSchema[], C, E, P>>;
    allAssignedIssues<E extends boolean = false>(groupId: string | number, milestoneId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IssueSchema[], C, E, void>>;
    allAssignedMergeRequests<E extends boolean = false>(groupId: string | number, milestoneId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MergeRequestSchema[], C, E, void>>;
    allBurndownChartEvents<E extends boolean = false>(projectId: string | number, milestoneId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<BurndownChartEventSchema[], C, E, void>>;
    create<E extends boolean = false>(groupId: string | number, title: string, options?: {
        description?: string;
        dueDate?: string;
        startDate?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MilestoneSchema, C, E, void>>;
    edit<E extends boolean = false>(groupId: string | number, milestoneId: number, options?: {
        title?: string;
        description?: string;
        dueDate?: string;
        startDate?: string;
        stateEvent?: 'close' | 'activate';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MilestoneSchema, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, milestoneId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(groupId: string | number, milestoneId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<MilestoneSchema, C, E, void>>;
}
declare class GroupMilestones<C extends boolean = false> extends ResourceMilestones<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface GroupProtectedEnvironments<C extends boolean = false> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: {
        search?: string;
    } & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<ProtectedEnvironmentSchema[], C, E, P>>;
    create<E extends boolean = false>(groupId: string | number, name: string, deployAccessLevels: ProtectedEnvironmentAccessLevelEntity[], options?: {
        requiredApprovalCount?: number;
        approvalRules?: ProtectedEnvironmentAccessLevelEntity[];
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProtectedEnvironmentSchema, C, E, void>>;
    edit<E extends boolean = false>(groupId: string | number, name: string, options?: {
        deployAccessLevels?: ProtectedEnvironmentAccessLevelEntity[];
        requiredApprovalCount?: number;
        approvalRules?: ProtectedEnvironmentAccessLevelEntity[];
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProtectedEnvironmentSchema, C, E, void>>;
    show<E extends boolean = false>(groupId: string | number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ProtectedEnvironmentSchema, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, name: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}
declare class GroupProtectedEnvironments<C extends boolean = false> extends ResourceProtectedEnvironments<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface GroupPushRules<C extends boolean = false> extends ResourcePushRules<C> {
    create<E extends boolean = false>(groupId: string | number, options?: CreateAndEditPushRuleOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PushRuleSchema, C, E, void>>;
    edit<E extends boolean = false>(groupId: string | number, options?: CreateAndEditPushRuleOptions & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PushRuleSchema, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(groupId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<PushRuleSchema, C, E, void>>;
}
declare class GroupPushRules<C extends boolean = false> extends ResourcePushRules<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface GroupRelationExportStatusSchema extends Record<string, unknown> {
    relation: string;
    status: number;
    error?: string;
    updated_at: string;
}
declare class GroupRelationExports<C extends boolean = false> extends BaseResource<C> {
    download<E extends boolean = false>(groupId: string | number, relation: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<Blob, void, E, void>>;
    exportStatus<E extends boolean = false>(groupId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupRelationExportStatusSchema[], C, E, void>>;
    scheduleExport<E extends boolean = false>(groupId: string | number, options?: BaseRequestOptions<E>): Promise<GitlabAPIResponse<{
        message: string;
    }, C, E, void>>;
}

declare class GroupReleases<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: PaginationRequestOptions<P> & BaseRequestOptions<E>): Promise<GitlabAPIResponse<ReleaseSchema[], C, E, P>>;
}

interface GroupRepositoryStorageMoveSchema extends RepositoryStorageMoveSchema {
    group: Pick<GroupSchema, 'id' | 'web_url' | 'name'>;
}
interface GroupRepositoryStorageMoves<C extends boolean = false> extends ResourceRepositoryStorageMoves<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(options?: {
        groupId?: string | number;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupRepositoryStorageMoveSchema[], C, E, P>>;
    show<E extends boolean = false, P extends PaginationTypes = 'offset'>(repositoryStorageId: number, options?: {
        groupId?: string | number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupRepositoryStorageMoveSchema, C, E, P>>;
    schedule<E extends boolean = false, P extends PaginationTypes = 'offset'>(sourceStorageName: string, options?: {
        groupId?: string | number;
        destinationStorageName?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<GroupRepositoryStorageMoveSchema, C, E, P>>;
}
declare class GroupRepositoryStorageMoves<C extends boolean = false> extends ResourceRepositoryStorageMoves<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface IdentitySchema extends Record<string, unknown> {
    extern_uid: string;
    user_id: number;
}
declare class GroupSAMLIdentities<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IdentitySchema[], C, E, P>>;
    edit<E extends boolean = false>(groupId: string | number, identityId: string, options: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface SAMLGroupSchema extends Record<string, unknown> {
    name: string;
    access_level: number;
}
declare class GroupSAMLLinks<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SAMLGroupSchema[], C, E, P>>;
    create<E extends boolean = false>(groupId: string | number, samlGroupName: string, accessLevel: Exclude<AccessLevel, AccessLevel.ADMIN>, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SAMLGroupSchema, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, samlGroupName: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(groupId: string | number, samlGroupName: string, options: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<SAMLGroupSchema, C, E, void>>;
}

declare class GroupSCIMIdentities<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<IdentitySchema[], C, E, P>>;
    edit<E extends boolean = false>(groupId: string | number, identityId: string, options: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

type ServiceAccountAccessTokenSchema = MappedOmit<AccessTokenSchema, 'access_level'>;
declare class GroupServiceAccounts<C extends boolean = false> extends BaseResource<C> {
    create<E extends boolean = false>(groupId: string | number, options?: {
        name?: string;
        username?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ServiceAccountSchema, C, E, void>>;
    addPersonalAccessToken<E extends boolean = false>(groupId: string | number, serviceAccountId: number, options?: {
        expiresAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ServiceAccountAccessTokenSchema, C, E, void>>;
    createPersonalAccessToken<E extends boolean = false>(groupId: string | number, serviceAccountId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ServiceAccountAccessTokenSchema, C, E, void>>;
    rotatePersonalAccessToken<E extends boolean = false>(groupId: string | number, serviceAccountId: number, tokenId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<ServiceAccountAccessTokenSchema, C, E, void>>;
}

interface GroupVariables<C extends boolean = false> extends ResourceVariables<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<VariableSchema[], C, E, P>>;
    create<E extends boolean = false>(groupId: string | number, key: string, value: string, options?: {
        variableType?: VariableType;
        protected?: boolean;
        masked?: boolean;
        environmentScope?: string;
        description?: string;
        raw?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<VariableSchema, C, E, void>>;
    edit<E extends boolean = false>(groupId: string | number, key: string, value: string, options?: {
        variableType?: VariableType;
        protected?: boolean;
        masked?: boolean;
        environmentScope?: string;
        description?: string;
        raw?: boolean;
        filter: VariableFilter;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<VariableSchema, C, E, void>>;
    show<E extends boolean = false>(groupId: string | number, key: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<VariableSchema, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, key: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}
declare class GroupVariables<C extends boolean = false> extends ResourceVariables<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface GroupWikis<C extends boolean = false> extends ResourceWikis<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options: {
        withContent: true;
    } & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<(WikiSchema & {
        content: string;
    })[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<WikiSchema[], C, E, P>>;
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, options?: {
        withContent?: boolean;
    } & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<WikiSchema[], C, E, P>>;
    create<E extends boolean = false>(groupId: string | number, content: string, title: string, options?: {
        format?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<WikiSchema, C, E, void>>;
    edit<E extends boolean = false>(groupId: string | number, slug: string, options?: OneOf<{
        content: string;
        title: string;
    }> & {
        format?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<WikiSchema, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, slug: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
    show<E extends boolean = false>(groupId: string | number, slug: string, options?: {
        renderHtml?: boolean;
        version?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<WikiSchema, C, E, void>>;
    uploadAttachment<E extends boolean = false>(groupId: string | number, file: {
        content: Blob;
        filename: string;
    }, options?: {
        branch?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<WikiAttachmentSchema, C, E, void>>;
}
declare class GroupWikis<C extends boolean = false> extends ResourceWikis<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface RelatedEpicSchema extends EpicSchema {
    related_epic_link_id: number;
}
interface RelatedEpicLinkSchema extends Record<string, unknown> {
    source_epic: RelatedEpicSchema;
    target_epic: RelatedEpicSchema;
}
type RelatedEpicLinkType = 'relates_to' | 'blocks' | 'is_blocked_by';
declare class LinkedEpics<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, epicIId: number, options?: {
        createdAfter?: string;
        createdBefore?: string;
        updatedAfter?: string;
        updatedBefore?: string;
    } & Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<RelatedEpicSchema[], C, E, P>>;
    create<E extends boolean = false>(groupId: string | number, epicIId: number, targetEpicIId: string | number, targetGroupId: string | number, options?: {
        linkType?: RelatedEpicLinkType;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RelatedEpicLinkSchema, C, E, void>>;
    remove<E extends boolean = false>(groupId: string | number, epicIId: number, relatedEpicLinkId: string | number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<RelatedEpicLinkSchema, C, E, void>>;
}

interface UserCustomAttributes<C extends boolean = false> extends ResourceCustomAttributes<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(userId: string | number, options?: PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CustomAttributeSchema[], C, E, P>>;
    remove<E extends boolean = false>(userId: string | number, customAttributeId: string, options?: Sudo): Promise<GitlabAPIResponse<void, C, E, void>>;
    set<E extends boolean = false>(userId: string | number, customAttributeId: string, value: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CustomAttributeSchema, C, E, void>>;
    show<E extends boolean = false>(userId: string | number, customAttributeId: string, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<CustomAttributeSchema, C, E, void>>;
}
declare class UserCustomAttributes<C extends boolean = false> extends ResourceCustomAttributes<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface UserEmailSchema extends Record<string, unknown> {
    id: number;
    email: string;
    confirmed_at: string;
}
declare class UserEmails<C extends boolean = false> extends BaseResource<C> {
    add<E extends boolean = false>(email: string, options?: {
        userId?: number;
        skipConfirmation?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserEmailSchema, C, E, void>>;
    all<E extends boolean = false>({ userId, ...options }?: {
        userId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserEmailSchema[], C, E, void>>;
    create<E extends boolean = false>(email: string, { userId, ...options }?: {
        userId?: number;
        skipConfirmation?: boolean;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserEmailSchema, C, E, void>>;
    show<E extends boolean = false>(emailId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserEmailSchema, C, E, void>>;
    remove<E extends boolean = false>(emailId: number, { userId, ...options }?: {
        userId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface UserGPGKeySchema extends Record<string, unknown> {
    id: number;
    key: string;
    created_at: string;
}
declare class UserGPGKeys<C extends boolean = false> extends BaseResource<C> {
    add<E extends boolean = false>(key: string, options?: {
        userId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserGPGKeySchema, C, E, void>>;
    all<E extends boolean = false>({ userId, ...options }?: {
        userId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserGPGKeySchema[], C, E, void>>;
    create<E extends boolean = false>(key: string, { userId, ...options }?: {
        userId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserGPGKeySchema, C, E, void>>;
    show<E extends boolean = false>(keyId: number, { userId, ...options }?: {
        userId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserGPGKeySchema, C, E, void>>;
    remove<E extends boolean = false>(keyId: number, { userId, ...options }?: {
        userId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

type ImpersonationTokenScope = 'api' | 'read_api' | 'read_user' | 'create_runner' | 'read_repository' | 'write_repository' | 'read_registry' | 'write_registry' | 'sudo' | 'admin_mode';
type ImpersonationTokenState = 'all' | 'active' | 'inactive';
interface UserImpersonationTokenSchema extends Record<string, unknown> {
    active: boolean;
    user_id: number;
    scopes?: string[];
    revoked: boolean;
    name: string;
    id: number;
    created_at: string;
    impersonation: boolean;
    expires_at: string;
    token?: string;
}
declare class UserImpersonationTokens<C extends boolean = false> extends BaseResource<C> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(userId: number, options?: {
        state?: ImpersonationTokenState;
    } & PaginationRequestOptions<P> & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserImpersonationTokenSchema[], C, E, P>>;
    create<E extends boolean = false>(userId: number, name: string, scopes: ImpersonationTokenScope[], options?: {
        expiresAt?: string;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserImpersonationTokenSchema, C, E, void>>;
    show<E extends boolean = false>(userId: number, tokenId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserImpersonationTokenSchema, C, E, void>>;
    remove<E extends boolean = false>(userId: number, tokenId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<void, E, undefined>>;
    revoke<E extends boolean = false>(userId: number, tokenId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIExpandedResponse<void, E, undefined>>;
}

interface UserSSHKeySchema extends Record<string, unknown> {
    id: number;
    key: string;
    title: string;
    created_at: string;
}
declare class UserSSHKeys<C extends boolean = false> extends BaseResource<C> {
    add<E extends boolean = false>(title: string, key: string, options?: {
        userId?: number;
        expiresAt?: string;
        usageType?: 'auth' | 'signing' | 'auth_and_signing';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserSSHKeySchema, C, E, void>>;
    all<E extends boolean = false>({ userId, ...options }?: {
        userId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserSSHKeySchema[], C, E, void>>;
    create<E extends boolean = false>(title: string, key: string, { userId, ...options }?: {
        userId?: number;
        expiresAt?: string;
        usageType?: 'auth' | 'signing' | 'auth_and_signing';
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserSSHKeySchema, C, E, void>>;
    show<E extends boolean = false>(keyId: number, { userId, ...options }?: {
        userId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<UserSSHKeySchema, C, E, void>>;
    remove<E extends boolean = false>(keyId: number, { userId, ...options }?: {
        userId?: number;
    } & Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<void, C, E, void>>;
}

interface MergeRequestStateEvents<C extends boolean = false> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(projectId: string | number, issueIId: number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<StateEventSchema[], C, E, P>>;
    show<E extends boolean = false>(projectId: string | number, issueIId: number, stateEventId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<StateEventSchema, C, E, void>>;
}
declare class MergeRequestStateEvents<C extends boolean = false> extends ResourceStateEvents<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface EpicStateEvents<C extends boolean = false> {
    all<E extends boolean = false, P extends PaginationTypes = 'offset'>(groupId: string | number, epicId: number, options?: Sudo & ShowExpanded<E> & PaginationRequestOptions<P>): Promise<GitlabAPIResponse<StateEventSchema[], C, E, P>>;
    show<E extends boolean = false>(groupId: string | number, epicId: number, stateEventId: number, options?: Sudo & ShowExpanded<E>): Promise<GitlabAPIResponse<StateEventSchema, C, E, void>>;
}
declare class EpicStateEvents<C extends boolean = false> extends ResourceStateEvents<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface Gitlab<C extends boolean = false> extends BaseResource<C> {
    Agents: Agents<C>;
    AlertManagement: AlertManagement<C>;
    ApplicationAppearance: ApplicationAppearance<C>;
    ApplicationPlanLimits: ApplicationPlanLimits<C>;
    Applications: Applications<C>;
    ApplicationSettings: ApplicationSettings<C>;
    ApplicationStatistics: ApplicationStatistics<C>;
    AuditEvents: AuditEvents<C>;
    Avatar: Avatar<C>;
    BroadcastMessages: BroadcastMessages<C>;
    CodeSuggestions: CodeSuggestions<C>;
    Composer: Composer<C>;
    Conan: Conan<C>;
    DashboardAnnotations: DashboardAnnotations<C>;
    Debian: Debian<C>;
    DependencyProxy: DependencyProxy<C>;
    DeployKeys: DeployKeys<C>;
    DeployTokens: DeployTokens<C>;
    DockerfileTemplates: DockerfileTemplates<C>;
    Events: Events<C>;
    Experiments: Experiments<C>;
    GeoNodes: GeoNodes<C>;
    GeoSites: GeoSites<C>;
    GitignoreTemplates: GitignoreTemplates<C>;
    GitLabCIYMLTemplates: GitLabCIYMLTemplates<C>;
    Import: Import<C>;
    InstanceLevelCICDVariables: InstanceLevelCICDVariables<C>;
    Keys: Keys<C>;
    License: License<C>;
    LicenseTemplates: LicenseTemplates<C>;
    Lint: Lint<C>;
    Markdown: Markdown<C>;
    Maven: Maven<C>;
    Metadata: Metadata<C>;
    Migrations: Migrations<C>;
    Namespaces: Namespaces<C>;
    NotificationSettings: NotificationSettings<C>;
    NPM: NPM<C>;
    NuGet: NuGet<C>;
    PersonalAccessTokens: PersonalAccessTokens<C>;
    PyPI: PyPI<C>;
    RubyGems: RubyGems<C>;
    Search: Search<C>;
    SearchAdmin: SearchAdmin<C>;
    ServiceAccounts: ServiceAccounts<C>;
    ServiceData: ServiceData<C>;
    SidekiqMetrics: SidekiqMetrics<C>;
    SidekiqQueues: SidekiqQueues<C>;
    SnippetRepositoryStorageMoves: SnippetRepositoryStorageMoves<C>;
    Snippets: Snippets<C>;
    Suggestions: Suggestions<C>;
    SystemHooks: SystemHooks<C>;
    TodoLists: TodoLists<C>;
    Topics: Topics<C>;
    Branches: Branches<C>;
    CommitDiscussions: CommitDiscussions<C>;
    Commits: Commits<C>;
    ContainerRegistry: ContainerRegistry<C>;
    Deployments: Deployments<C>;
    Environments: Environments<C>;
    ErrorTrackingClientKeys: ErrorTrackingClientKeys<C>;
    ErrorTrackingSettings: ErrorTrackingSettings<C>;
    ExternalStatusChecks: ExternalStatusChecks<C>;
    FeatureFlags: FeatureFlags<C>;
    FeatureFlagUserLists: FeatureFlagUserLists<C>;
    FreezePeriods: FreezePeriods<C>;
    GitlabPages: GitlabPages<C>;
    GoProxy: GoProxy<C>;
    Helm: Helm<C>;
    Integrations: Integrations<C>;
    IssueAwardEmojis: IssueAwardEmojis<C>;
    IssueDiscussions: IssueDiscussions<C>;
    IssueIterationEvents: IssueIterationEvents<C>;
    IssueLabelEvents: IssueLabelEvents<C>;
    IssueLinks: IssueLinks<C>;
    IssueMilestoneEvents: IssueMilestoneEvents<C>;
    IssueNoteAwardEmojis: IssueNoteAwardEmojis<C>;
    IssueNotes: IssueNotes<C>;
    Issues: Issues<C>;
    IssuesStatistics: IssuesStatistics<C>;
    IssueStateEvents: IssueStateEvents<C>;
    IssueWeightEvents: IssueWeightEvents<C>;
    JobArtifacts: JobArtifacts<C>;
    Jobs: Jobs<C>;
    MergeRequestApprovals: MergeRequestApprovals<C>;
    MergeRequestAwardEmojis: MergeRequestAwardEmojis<C>;
    MergeRequestContextCommits: MergeRequestContextCommits<C>;
    MergeRequestDiscussions: MergeRequestDiscussions<C>;
    MergeRequestLabelEvents: MergeRequestLabelEvents<C>;
    MergeRequestMilestoneEvents: MergeRequestMilestoneEvents<C>;
    MergeRequestStateEvents: MergeRequestStateEvents<C>;
    MergeRequestDraftNotes: MergeRequestDraftNotes<C>;
    MergeRequestNotes: MergeRequestNotes<C>;
    MergeRequestNoteAwardEmojis: MergeRequestNoteAwardEmojis<C>;
    MergeRequests: MergeRequests<C>;
    MergeTrains: MergeTrains<C>;
    PackageRegistry: PackageRegistry<C>;
    Packages: Packages<C>;
    PagesDomains: PagesDomains<C>;
    Pipelines: Pipelines<C>;
    PipelineSchedules: PipelineSchedules<C>;
    PipelineScheduleVariables: PipelineScheduleVariables<C>;
    PipelineTriggerTokens: PipelineTriggerTokens<C>;
    ProductAnalytics: ProductAnalytics<C>;
    ProjectAccessRequests: ProjectAccessRequests<C>;
    ProjectAccessTokens: ProjectAccessTokens<C>;
    ProjectAliases: ProjectAliases<C>;
    ProjectBadges: ProjectBadges<C>;
    ProjectCustomAttributes: ProjectCustomAttributes<C>;
    ProjectDORA4Metrics: ProjectDORA4Metrics<C>;
    ProjectHooks: ProjectHooks<C>;
    ProjectImportExports: ProjectImportExports<C>;
    ProjectInvitations: ProjectInvitations<C>;
    ProjectIssueBoards: ProjectIssueBoards<C>;
    ProjectIterations: ProjectIterations<C>;
    ProjectJobTokenScopes: ProjectJobTokenScopes<C>;
    ProjectLabels: ProjectLabels<C>;
    ProjectMarkdownUploads: ProjectMarkdownUploads<C>;
    ProjectMembers: ProjectMembers<C>;
    ProjectMilestones: ProjectMilestones<C>;
    ProjectProtectedEnvironments: ProjectProtectedEnvironments<C>;
    ProjectPushRules: ProjectPushRules<C>;
    ProjectRelationsExport: ProjectRelationsExport<C>;
    ProjectReleases: ProjectReleases<C>;
    ProjectRemoteMirrors: ProjectRemoteMirrors<C>;
    ProjectRepositoryStorageMoves: ProjectRepositoryStorageMoves<C>;
    Projects: Projects<C>;
    ProjectSnippetAwardEmojis: ProjectSnippetAwardEmojis<C>;
    ProjectSnippetDiscussions: ProjectSnippetDiscussions<C>;
    ProjectSnippetNotes: ProjectSnippetNotes<C>;
    ProjectSnippets: ProjectSnippets<C>;
    ProjectStatistics: ProjectStatistics<C>;
    ProjectTemplates: ProjectTemplates<C>;
    ProjectTerraformState: ProjectTerraformState<C>;
    ProjectVariables: ProjectVariables<C>;
    ProjectVulnerabilities: ProjectVulnerabilities<C>;
    ProjectWikis: ProjectWikis<C>;
    ProtectedBranches: ProtectedBranches<C>;
    ProtectedTags: ProtectedTags<C>;
    ReleaseLinks: ReleaseLinks<C>;
    Repositories: Repositories<C>;
    RepositoryFiles: RepositoryFiles<C>;
    RepositorySubmodules: RepositorySubmodules<C>;
    ResourceGroups: ResourceGroups<C>;
    Runners: Runners<C>;
    SecureFiles: SecureFiles<C>;
    Tags: Tags<C>;
    UserStarredMetricsDashboard: UserStarredMetricsDashboard<C>;
    EpicAwardEmojis: EpicAwardEmojis<C>;
    EpicDiscussions: EpicDiscussions<C>;
    EpicIssues: EpicIssues<C>;
    EpicLabelEvents: EpicLabelEvents<C>;
    EpicLinks: EpicLinks<C>;
    EpicNotes: EpicNotes<C>;
    Epics: Epics<C>;
    EpicStateEvents: EpicStateEvents<C>;
    GroupAccessRequests: GroupAccessRequests<C>;
    GroupAccessTokens: GroupAccessTokens<C>;
    GroupActivityAnalytics: GroupActivityAnalytics<C>;
    GroupBadges: GroupBadges<C>;
    GroupCustomAttributes: GroupCustomAttributes<C>;
    GroupDORA4Metrics: GroupDORA4Metrics<C>;
    GroupEpicBoards: GroupEpicBoards<C>;
    GroupHooks: GroupHooks<C>;
    GroupImportExports: GroupImportExports<C>;
    GroupInvitations: GroupInvitations<C>;
    GroupIssueBoards: GroupIssueBoards<C>;
    GroupIterations: GroupIterations<C>;
    GroupLabels: GroupLabels<C>;
    GroupLDAPLinks: GroupLDAPLinks<C>;
    GroupMarkdownUploads: GroupMarkdownUploads<C>;
    GroupMembers: GroupMembers<C>;
    GroupMemberRoles: GroupMemberRoles<C>;
    GroupMilestones: GroupMilestones<C>;
    GroupProtectedEnvironments: GroupProtectedEnvironments<C>;
    GroupPushRules: GroupPushRules<C>;
    GroupRelationExports: GroupRelationExports<C>;
    GroupReleases: GroupReleases<C>;
    GroupRepositoryStorageMoves: GroupRepositoryStorageMoves<C>;
    Groups: Groups<C>;
    GroupSAMLIdentities: GroupSAMLIdentities<C>;
    GroupSAMLLinks: GroupSAMLLinks<C>;
    GroupSCIMIdentities: GroupSCIMIdentities<C>;
    GroupServiceAccounts: GroupServiceAccounts<C>;
    GroupVariables: GroupVariables<C>;
    GroupWikis: GroupWikis<C>;
    LinkedEpics: LinkedEpics<C>;
    UserCustomAttributes: UserCustomAttributes<C>;
    UserEmails: UserEmails<C>;
    UserGPGKeys: UserGPGKeys<C>;
    UserImpersonationTokens: UserImpersonationTokens<C>;
    Users: Users<C>;
    UserSSHKeys: UserSSHKeys<C>;
}
declare class Gitlab<C extends boolean = false> extends BaseResource<C> {
    constructor(options: BaseResourceOptions<C>);
}

interface WebhookRepositorySchema {
    name: string;
    url: string;
    description: string;
    homepage: string;
    git_http_url: string;
    git_ssh_url: string;
    visibility_level: number;
}
interface WebhookLabelSchema {
    id: number;
    title: string;
    color: string;
    project_id: number;
    created_at: string;
    updated_at: string;
    template: boolean;
    description: string;
    type: string;
    group_id: number;
}
interface WebhookProjectSchema {
    id: number;
    name: string;
    description: string;
    web_url: string;
    avatar_url: string | null;
    git_ssh_url: string;
    git_http_url: string;
    namespace: string;
    visibility_level: number;
    path_with_namespace: string;
    default_branch: string;
    homepage: string;
    url: string;
    ssh_url: string;
    http_url: string;
}
interface WebhookPipelineSchema {
    project: {
        id: number;
        web_url: string;
        path_with_namespace: string;
    };
    pipeline_id: number;
    job_id: number;
}
interface WebhookDiffSchema {
    diff: string;
    new_path: string;
    old_path: string;
    a_mode: string;
    b_mode: string;
    new_file: boolean;
    renamed_file: boolean;
    deleted_file: boolean;
}
type WebhookUserSchema = Pick<SimpleUserSchema, 'id' | 'name' | 'username' | 'avatar_url'>;
interface BaseWebhookEventSchema {
    object_kind: string;
    event_name: string;
    project: WebhookProjectSchema;
    user: MappedOmit<SimpleUserSchema, 'created_at'>;
}
interface WebhookBaseNoteEventSchema extends BaseWebhookEventSchema {
    object_kind: 'note';
    event_type: 'note';
    project_id: number;
    repository: {
        name: string;
        url: string;
        description: string;
        homepage: string;
    };
    object_attributes: {
        id: number;
        note: string;
        noteable_type: string;
        discussion_id: string;
        author_id: number;
        created_at: string;
        updated_at: string;
        project_id: number;
        attachment: string | null;
        line_code: string;
        commit_id: string;
        noteable_id: string | null;
        system: boolean;
        st_diff: WebhookDiffSchema | null;
        url: string;
    };
}
interface WebhookBasePushEventSchema extends MappedOmit<BaseWebhookEventSchema, 'user'> {
    before: string;
    after: string;
    ref: string;
    ref_protected: boolean;
    checkout_sha: string;
    user_id: number;
    user_name: string;
    user_avatar: string;
    project_id: number;
    repository: WebhookRepositorySchema;
    commits: {
        id: string;
        message: string;
        title: string;
        timestamp: string;
        url: string;
        author: {
            name: string;
            email: string;
        };
        added: string[] | null;
        modified: string[] | null;
        removed: string[] | null;
    }[] | null;
    total_commits_count: number;
}
interface WebhookTagEventSchema extends WebhookBasePushEventSchema {
    object_kind: 'tag_push';
    event_name: 'tag_push';
}
interface WebhookPushEventSchema extends WebhookBasePushEventSchema {
    object_kind: 'push';
    event_name: 'push';
    user_username: string;
    user_email: string;
}
interface WebhookIssueEventSchema extends BaseWebhookEventSchema {
    object_kind: 'issue';
    event_type: 'issue';
    object_attributes: {
        id: number;
        title: string;
        assignee_ids: number[] | null;
        assignee_id: number;
        author_id: number;
        project_id: number;
        created_at: string;
        updated_at: string;
        updated_by_id: number;
        last_edited_at: string | null;
        last_edited_by_id: number | null;
        relative_position: number;
        description: string;
        milestone_id: number | null;
        state_id: number;
        confidential: boolean;
        discussion_locked: boolean;
        due_date: string | null;
        moved_to_id: number | null;
        duplicated_to_id: number | null;
        time_estimate: number;
        total_time_spent: number;
        time_change: number;
        human_total_time_spent: string | null;
        human_time_estimate: string | null;
        human_time_change: string | null;
        weight: number | null;
        health_status: string;
        iid: number;
        url: string;
        state: string;
        action: string;
        severity: string;
        escalation_status: string;
        escalation_policy: {
            id: number;
            name: string;
        };
        labels: WebhookLabelSchema[] | null;
    };
    repository: {
        name: string;
        url: string;
        description: string;
        homepage: string;
    };
    assignees: WebhookUserSchema[] | null;
    assignee: WebhookUserSchema | null;
    labels: WebhookLabelSchema[] | null;
    changes: {
        updated_by_id: {
            previous: string | null;
            current: string;
        };
        updated_at: {
            previous: string;
            current: string;
        };
        labels: {
            previous: WebhookLabelSchema[] | null;
            current: WebhookLabelSchema[] | null;
        };
    };
}
interface WebhookCommitNoteEventSchema extends WebhookBaseNoteEventSchema {
    commit: {
        id: string;
        message: string;
        timestamp: string;
        url: string;
        author: {
            name: string;
            email: string;
        };
    };
}
interface WebhookMergeRequestNoteEventSchema extends WebhookBaseNoteEventSchema {
    merge_request: {
        id: number;
        target_branch: string;
        source_branch: string;
        source_project_id: number;
        author_id: number;
        assignee_id: number;
        title: string;
        created_at: string;
        updated_at: string;
        milestone_id: number;
        state: string;
        merge_status: string;
        target_project_id: number;
        iid: number;
        description: string;
        position: number;
        labels: WebhookLabelSchema[] | null;
        source: WebhookProjectSchema;
        target: WebhookProjectSchema;
        last_commit: {
            id: string;
            message: string;
            timestamp: string;
            url: string;
            author: {
                name: string;
                email: string;
            };
        };
        work_in_progress: boolean;
        draft: boolean;
        assignee: WebhookUserSchema | null;
        detailed_merge_status: string;
    };
}
interface WebhookIssueNoteEventSchema extends WebhookBaseNoteEventSchema {
    issue: {
        id: number;
        title: string;
        assignee_ids: number[] | null;
        assignee_id: number | null;
        author_id: number;
        project_id: number;
        created_at: string;
        updated_at: string;
        position: number;
        branch_name: string | null;
        description: string;
        milestone_id: number | null;
        state: string;
        iid: number;
        labels: WebhookLabelSchema[] | null;
    };
}
interface WebhookSnippetNoteEventSchema extends WebhookBaseNoteEventSchema {
    snippet: {
        id: number;
        title: string;
        content: string;
        author_id: number;
        project_id: number;
        created_at: string;
        updated_at: string;
        file_name: string;
        expires_at: string | null;
        type: string;
        visibility_level: number;
        url: string;
    };
}
interface WebhookMergeRequestEventSchema extends BaseWebhookEventSchema {
    object_kind: 'merge_request';
    event_type: 'merge_request';
    repository: WebhookRepositorySchema;
    object_attributes: {
        id: number;
        iid: number;
        target_branch: string;
        source_branch: string;
        source_project_id: number;
        author_id: number;
        assignee_ids: number[] | null;
        reviewer_ids: number[] | null;
        title: string;
        created_at: string;
        updated_at: string;
        last_edited_at: string;
        last_edited_by_id: number;
        milestone_id: number | null;
        state_id: number;
        state: string;
        blocking_discussions_resolved: boolean;
        work_in_progress: boolean;
        draft: boolean;
        first_contribution: boolean;
        merge_commit_sha: string;
        target_project_id: number;
        description: string;
        total_time_spent: number;
        time_change: number;
        human_total_time_spent: string;
        human_time_change: string;
        human_time_estimate: string;
        url: string;
        source: WebhookProjectSchema;
        target: WebhookProjectSchema;
        last_commit: {
            id: string;
            message: string;
            title: string;
            timestamp: string;
            url: string;
            author: {
                name: string;
                email: string;
            };
        };
        labels: WebhookLabelSchema[] | null;
        action: string;
        detailed_merge_status: string;
    };
    labels: WebhookLabelSchema[] | null;
    changes: {
        target_branch: {
            previous: string | null;
            current: string | null;
        };
        updated_by_id: {
            previous: number | null;
            current: number | null;
        };
        draft: {
            previous: boolean | null;
            current: boolean | null;
        };
        updated_at: {
            previous: string | null;
            current: string | null;
        };
        assignees: {
            previous: WebhookUserSchema[] | null;
            current: WebhookUserSchema[] | null;
        };
        reviewers: {
            previous: WebhookUserSchema[] | null;
            current: WebhookUserSchema[] | null;
        };
        labels: {
            previous: WebhookLabelSchema[] | null;
            current: WebhookLabelSchema[] | null;
        };
        last_edited_at: {
            previous: string | null;
            current: string | null;
        };
        last_edited_by_id: {
            previous: number | null;
            current: number | null;
        };
    };
    assignees: WebhookUserSchema[] | null;
    reviewers: WebhookUserSchema[] | null;
}
interface WebhookWikiEventSchema extends MappedOmit<BaseWebhookEventSchema, 'event_name'> {
    object_kind: 'wiki_page';
    wiki: {
        web_url: string;
        git_ssh_url: string;
        git_http_url: string;
        path_with_namespace: string;
        default_branch: string;
    };
    object_attributes: {
        title: string;
        content: string;
        format: string;
        message: string;
        slug: string;
        url: string;
        action: string;
        diff_url: string;
    };
}
interface WebhookPipelineEventSchema extends MappedOmit<BaseWebhookEventSchema, 'event_name'> {
    object_kind: 'pipeline';
    object_attributes: {
        id: number;
        iid: number;
        name: string;
        ref: string;
        tag: boolean;
        sha: string;
        before_sha: string;
        source: string;
        status: string;
        detailed_status: string;
        stages: string[] | null;
        created_at: string;
        finished_at: string;
        duration: number;
        variables: {
            key: string;
            value: string;
        }[] | null;
        url: string;
    };
    merge_request: null | {
        id: number;
        iid: number;
        title: string;
        source_branch: string;
        source_project_id: number;
        target_branch: string;
        target_project_id: number;
        state: string;
        merge_status: string;
        detailed_merge_status: string;
        url: string;
    };
    commit: {
        id: string;
        message: string;
        timestamp: string;
        url: string;
        author: {
            name: string;
            email: string;
        };
    };
    source_pipeline: WebhookPipelineSchema;
    builds: {
        id: number;
        stage: string;
        name: string;
        status: string;
        created_at: string;
        started_at: string | null;
        finished_at: string | null;
        duration: number | null;
        queued_duration: number | null;
        failure_reason: string | null;
        when: string;
        manual: boolean;
        allow_failure: boolean;
        user: MappedOmit<SimpleUserSchema, 'created_at'>;
        runner: {
            id: number;
            description: string;
            active: boolean;
            runner_type: string;
            is_shared: boolean;
            tags: string[] | null;
        } | null;
        artifacts_file: {
            filename: string | null;
            size: string | null;
        };
        environment: {
            name: string;
            action: string;
            deployment_tier: string;
        } | null;
    }[] | null;
}
interface WebhookJobEventSchema extends MappedOmit<BaseWebhookEventSchema, 'event_name'> {
    object_kind: 'build';
    ref: string;
    tag: boolean;
    before_sha: string;
    sha: string;
    build_id: number;
    build_name: string;
    build_stage: string;
    build_status: string;
    build_created_at: string;
    build_started_at: string | null;
    build_finished_at: string | null;
    build_duration: number | null;
    build_queued_duration: number;
    build_allow_failure: boolean;
    build_failure_reason: string;
    retries_count: number;
    pipeline_id: number;
    project_id: number;
    project_name: string;
    commit: {
        id: number;
        name: string;
        sha: string;
        message: string;
        author_name: string;
        author_email: string;
        status: string;
        duration: number | null;
        started_at: string | null;
        finished_at: string | null;
    };
    repository: WebhookRepositorySchema;
    runner: {
        active: boolean;
        runner_type: string;
        is_shared: boolean;
        id: number;
        description: string;
        tags: string[] | null;
    };
    environment: {
        name: string;
        action: string;
        deployment_tier: string;
    } | null;
    source_pipeline: WebhookPipelineSchema;
}
interface WebhookDeploymentEventSchema extends MappedOmit<BaseWebhookEventSchema, 'event_name'> {
    object_kind: 'deployment';
    status: string;
    status_changed_at: string;
    deployment_id: number;
    deployable_id: number;
    deployable_url: string;
    environment: string;
    environment_tier: string;
    environment_slug: string;
    environment_external_url: string;
    short_sha: string;
    user_url: string;
    commit_url: string;
    commit_title: string;
}
interface WebhookGroupMemberEventSchema {
    event_name: 'user_remove_from_group' | 'user_update_for_group';
    created_at: string;
    updated_at: string;
    group_name: string;
    group_path: string;
    group_id: number;
    user_username: string;
    user_name: string;
    user_email: string;
    user_id: number;
    group_access: string;
    group_plan: string | null;
    expires_at: string;
}
interface WebhookSubGroupEventSchema {
    event_name: 'subgroup_create' | 'subgroup_destroy';
    created_at: string;
    updated_at: string;
    name: string;
    path: string;
    full_path: string;
    group_id: number;
    parent_group_id: number;
    parent_name: string;
    parent_path: string;
    parent_full_path: string;
}
interface WebhookFeatureFlagEventSchema extends MappedOmit<BaseWebhookEventSchema, 'event_name'> {
    object_kind: 'feature_flag';
    user_url: string;
    object_attributes: {
        id: number;
        name: string;
        description: string;
        active: boolean;
    };
}
interface WebhookReleaseEventSchema {
    object_kind: 'release';
    project: WebhookProjectSchema;
    id: number;
    created_at: string;
    description: string;
    name: string;
    released_at: string;
    tag: string;
    url: string;
    action: string;
    assets: {
        count: number;
        links: {
            id: number;
            link_type: string;
            name: string;
            url: string;
        }[] | null;
        sources: {
            format: string;
            url: string;
        }[] | null;
    };
    commit: {
        id: string;
        message: string;
        title: string;
        timestamp: string;
        url: string;
        author: {
            name: string;
            email: string;
        };
    };
}
interface WebhookEmojiEventSchema extends BaseWebhookEventSchema {
    object_kind: 'emoji';
    event_type: 'award';
    merge_request?: WebhookMergeRequestEventSchema['object_attributes'];
    project_id: number;
    object_attributes: {
        user_id: number;
        created_at: string;
        id: number;
        name: string;
        awardable_type: string;
        awardable_id: number;
        updated_at: string;
        action: string;
        awarded_on_url: string;
    };
    note: {
        attachment: string | null;
        author_id: number;
        change_position: boolean | null;
        commit_id: number | null;
        created_at: string;
        discussion_id: string;
        id: number;
        line_code: string | null;
        note: string;
        noteable_id: number;
        noteable_type: string;
        original_position: number | null;
        position: number | null;
        project_id: number;
        resolved_at: string | null;
        resolved_by_id: number | null;
        resolved_by_push: boolean | null;
        st_diff: WebhookDiffSchema | null;
        system: boolean;
        type: string | null;
        updated_at: string;
        updated_by_id: number | null;
        description: string;
        url: string;
    };
    issue: {
        author_id: number;
        closed_at: string | null;
        confidential: boolean;
        created_at: string;
        description: string;
        discussion_locked: boolean | null;
        due_date: string | null;
        id: number;
        iid: number;
        last_edited_at: string | null;
        last_edited_by_id: number | null;
        milestone_id: number | null;
        moved_to_id: number | null;
        duplicated_to_id: number | null;
        project_id: number;
        relative_position: number;
        state_id: number;
        time_estimate: number;
        title: string;
        updated_at: string;
        updated_by_id: number | null;
        weight: number | null;
        health_status: string | null;
        url: string;
        total_time_spent: number;
        time_change: number;
        human_total_time_spent: string | null;
        human_time_change: string | null;
        human_time_estimate: string | null;
        assignee_ids: number[] | null;
        assignee_id: number;
        labels: WebhookLabelSchema[] | null;
        state: string;
        severity: string;
    };
}

export { type AcceptMergeRequestOptions, AccessLevel, type AccessLevelSettingState, type AccessRequestSchema, type AccessTokenExposedSchema, type AccessTokenSchema, type AccessTokenScopes, type AddMemberOptions, type AddResourceHookOptions, type AdminUserSchema, Agents, AlertManagement, type AllAuditEventOptions, type AllCommitsOptions, type AllDeploymentsOptions, type AllEpicsOptions, type AllEventOptions, type AllForksOptions, type AllGroupProjectsOptions, type AllGroupsOptions, type AllIssueStatisticsOptions, type AllIssuesOptions, type AllIterationsOptions, type AllMembersOptions, type AllMergeRequestsOptions, type AllMilestonesOptions, type AllPackageOptions, type AllPersonalAccessTokenOptions, type AllPipelinesOptions, type AllProjectsOptions, type AllProvisionedUsersOptions, type AllRepositoryTreesOptions, type AllRunnersOptions, type AllSearchOptions, type AllUserProjectsOptions, type AllUsersOptions, type AllowedAgentSchema, ApplicationAppearance, type ApplicationAppearanceSchema, type ApplicationPlanLimitOptions, type ApplicationPlanLimitSchema, ApplicationPlanLimits, type ApplicationSchema, ApplicationSettings, type ApplicationSettingsSchema, type ApplicationStatisticSchema, ApplicationStatistics, Applications, type ApprovalRuleSchema, type ApprovalStateSchema, type ApprovedByEntity, type ArchiveType, type ArtifactSchema, type AsStream, type AuditEventSchema, AuditEvents, Avatar, type AvatarSchema, type AwardEmojiSchema, type BadgeSchema, type BaseExternalStatusCheckSchema, type BasePaginationRequestOptions, type BaseRequestOptions, type BaseWebhookEventSchema, type BillableGroupMemberMembershipSchema, type BillableGroupMemberSchema, type BlobSchema, type BranchSchema, Branches, type BridgeSchema, type BroadcastMessageOptions, type BroadcastMessageSchema, BroadcastMessages, type BurndownChartEventSchema, type CICDVariableSchema, type Camelize, type ClusterAgentSchema, type ClusterAgentTokenSchema, type CodeCompletionSchema, type CodeSuggestionSchema, CodeSuggestions, type CommitAction, type CommitCommentSchema, type CommitDiffSchema, type CommitDiscussionNoteSchema, type CommitDiscussionSchema, CommitDiscussions, type CommitReferenceSchema, type CommitSchema, type CommitSequenceSchema, type CommitSignatureSchema, type CommitStatsSchema, type CommitStatusSchema, type CommitablePipelineStatus, Commits, Composer, type ComposerPackageMetadataSchema, type ComposerV1BaseRepositorySchema, type ComposerV1PackagesSchema, type ComposerV2BaseRepositorySchema, Conan, type CondensedBadgeSchema, type CondensedCommitCommentSchema, type CondensedCommitSchema, type CondensedDeployKeySchema, type CondensedEnvironmentSchema, type CondensedEpicLinkSchema, type CondensedGroupSchema, type CondensedJobSchema, type CondensedMemberSchema, type CondensedMergeRequestSchema, type CondensedNamespaceSchema, type CondensedPipelineScheduleSchema, type CondensedProjectSchema, type CondensedRegistryRepositorySchema, type CondensedRegistryRepositoryTagSchema, ContainerRegistry, type CreateAndEditPushRuleOptions, type CreateApprovalRuleOptions, type CreateCommitOptions, type CreateEpicOptions, type CreateFeatureFlagOptions, type CreateGeoNodeOptions, type CreateGeoSiteOptions, type CreateGroupOptions, type CreateIssueOptions, type CreateMergeRequestOptions, type CreateProjectOptions, type CreateProtectedBranchAllowOptions, type CreateProtectedBranchOptions, type CreateRepositoryFileOptions, type CreateRunnerOptions, type CreateSnippetOptions, type CreateSystemHook, type CreateUserCIRunnerOptions, type CreateUserOptions, type CustomAttributeSchema, type CustomSettingLevelEmailEvents, type DORA4MetricSchema, type DashboardAnnotationSchema, DashboardAnnotations, Debian, DependencyProxy, type DeployKeyProjectsSchema, type DeployKeySchema, DeployKeys, type DeployTokenSchema, type DeployTokenScope, DeployTokens, type DeployableSchema, type DeploymentApprovalStatusSchema, type DeploymentSchema, type DeploymentStatus, Deployments, type DiffRefsSchema, type DiscussionNotePositionBaseSchema, type DiscussionNotePositionImageSchema, type DiscussionNotePositionLineRangeSchema, type DiscussionNotePositionOptions, type DiscussionNotePositionSchema, type DiscussionNotePositionTextLineSchema, type DiscussionNotePositionTextSchema, type DiscussionNoteSchema, type DiscussionSchema, DockerfileTemplates, type EditApprovalRuleOptions, type EditBadgeOptions, type EditChangelogOptions, type EditConfigurationOptions, type EditEpicOptions, type EditFeatureFlagOptions, type EditGeoNodeOptions, type EditGeoSiteOptions, type EditGroupOptions, type EditIssueOptions, type EditMergeRequestOptions, type EditNotificationSettingsOptions, type EditPipelineStatusOptions, type EditProjectOptions, type EditProtectedBranchAllowOptions, type EditProtectedBranchOptions, type EditRepositoryFileOptions, type EditResourceHookOptions, type EditRunnerOptions, type EditSnippetOptions, type EditUserOptions, type EnvironmentSchema, type EnvironmentTier, Environments, EpicAwardEmojis, EpicDiscussions, type EpicIssueSchema, EpicIssues, EpicLabelEvents, type EpicLinkSchema, EpicLinks, type EpicNoteSchema, EpicNotes, type EpicSchema, type EpicSchemaWithBasicLabels, type EpicSchemaWithExpandedLabels, type EpicTodoSchema, Epics, type ErrorTrackingClientKeySchema, ErrorTrackingClientKeys, ErrorTrackingSettings, type ErrorTrackingSettingsSchema, type EventSchema, Events, type ExpandedCommitSchema, type ExpandedDeployKeySchema, type ExpandedEpicIssueSchema, type ExpandedGroupSchema, type ExpandedHookSchema, type ExpandedIssueLinkSchema, type ExpandedMergeRequestDiffVersionsSchema, type ExpandedMergeRequestSchema, type ExpandedPackageSchema, type ExpandedPipelineScheduleSchema, type ExpandedPipelineSchema, type ExpandedRepositoryImportStatusSchema, type ExpandedResponse, type ExpandedRunnerSchema, type ExpandedSnippetSchema, type ExpandedUserSchema, type ExperimentGateSchema, type ExperimentSchema, Experiments, type ExportStatusSchema, type ExtendedProtectedBranchAccessLevelSchema, type ExternalStatusCheckProtectedBranchesSchema, ExternalStatusChecks, type FailedRelationSchema, type FeatureFlagSchema, type FeatureFlagStrategySchema, type FeatureFlagStrategyScopeSchema, type FeatureFlagUserListSchema, FeatureFlagUserLists, FeatureFlags, type ForkProjectOptions, type FreezePeriodSchema, FreezePeriods, type GPGSignatureSchema, type GeoNodeFailureSchema, type GeoNodeSchema, type GeoNodeStatusSchema, GeoNodes, type GeoSiteFailureSchema, type GeoSiteSchema, type GeoSiteStatusSchema, GeoSites, GitLabCIYMLTemplates, GitignoreTemplates, Gitlab, type GitlabAPIResponse, GitlabPages, type GitlabPagesSettingsSchema, GoProxy, type GoProxyModuleVersionSchema, GroupAccessRequests, GroupAccessTokens, GroupActivityAnalytics, type GroupAllowListSchema, type GroupAnalyticsIssuesCountSchema, type GroupAnalyticsMRsCountSchema, type GroupAnalyticsNewMembersCountSchema, type GroupBadgeSchema, GroupBadges, GroupCustomAttributes, GroupDORA4Metrics, type GroupEpicBoardListSchema, type GroupEpicBoardSchema, GroupEpicBoards, type GroupHookSchema, GroupHooks, GroupImportExports, GroupInvitations, GroupIssueBoards, GroupIterations, GroupLDAPLinks, GroupLabels, GroupMarkdownUploads, GroupMemberRoles, GroupMembers, GroupMilestones, GroupProtectedEnvironments, GroupPushRules, type GroupRelationExportStatusSchema, GroupRelationExports, GroupReleases, type GroupRepositoryStorageMoveSchema, GroupRepositoryStorageMoves, GroupSAMLIdentities, GroupSAMLLinks, GroupSCIMIdentities, type GroupSchema, GroupServiceAccounts, type GroupStatisticsSchema, GroupVariables, GroupWikis, Groups, type GrouptIssueBoardSchema, Helm, type HookSchema, type IdentitySchema, type ImpersonationTokenScope, type ImpersonationTokenState, Import, type ImportStatusSchema, type IncludeInherited, InstanceLevelCICDVariables, type IntegrationSchema, Integrations, type InvitationSchema, type IsForm, IssueAwardEmojis, type IssueBoardListSchema, type IssueBoardSchema, IssueDiscussions, IssueIterationEvents, IssueLabelEvents, type IssueLinkSchema, IssueLinks, IssueMilestoneEvents, IssueNoteAwardEmojis, type IssueNoteSchema, IssueNotes, type IssueSchema, type IssueSchemaWithBasicLabels, type IssueSchemaWithExpandedLabels, IssueStateEvents, IssueWeightEvents, Issues, IssuesStatistics, type IterationEventSchema, type IterationSchema, JobArtifacts, type JobKubernetesAgentsSchema, type JobSchema, type JobScope, type JobTokenScopeSchema, type JobVariableAttributeOption, Jobs, type KeySchema, Keys, type KeysetPagination, type KeysetPaginationRequestOptions, type LabelCountSchema, type LabelEventSchema, type LabelSchema, License, type LicenseSchema, type LicenseTemplateSchema, LicenseTemplates, LinkedEpics, Lint, type LintSchema, Markdown, type MarkdownSchema, type MarkdownUploadCreatedSchema, type MarkdownUploadSchema, type MarkdownUploadUserSchema, Maven, type MemberRoleSchema, type MemberSchema, MergeRequestApprovals, MergeRequestAwardEmojis, type MergeRequestChangesSchema, type MergeRequestContextCommitSchema, MergeRequestContextCommits, type MergeRequestDiffSchema, type MergeRequestDiffVersionsSchema, type MergeRequestDiscussionNotePositionOptions, type MergeRequestDiscussionNoteSchema, MergeRequestDiscussions, type MergeRequestDraftNotePositionSchema, type MergeRequestDraftNoteSchema, MergeRequestDraftNotes, type MergeRequestExternalStatusCheckSchema, MergeRequestLabelEvents, type MergeRequestLevelApprovalRuleSchema, type MergeRequestLevelMergeRequestApprovalSchema, MergeRequestMilestoneEvents, MergeRequestNoteAwardEmojis, type MergeRequestNoteSchema, MergeRequestNotes, type MergeRequestRebaseSchema, type MergeRequestReviewerSchema, type MergeRequestSchema, type MergeRequestSchemaWithBasicLabels, type MergeRequestSchemaWithExpandedLabels, type MergeRequestTodoSchema, MergeRequests, type MergeTrainSchema, MergeTrains, Metadata, type MetadataSchema, type MetricImageSchema, type MetricType, type MigrationEntityFailure, type MigrationEntityOptions, type MigrationEntitySchema, type MigrationStatus, type MigrationStatusSchema, Migrations, type MilestoneEventSchema, type MilestoneSchema, type MissingSignatureSchema, NPM, type NPMPackageMetadataSchema, type NPMVersionSchema, type NamespaceExistsSchema, type NamespaceSchema, Namespaces, type NewDeployTokenSchema, type NoteSchema, type NotificationSettingLevel, type NotificationSettingSchema, NotificationSettings, NuGet, type NuGetPackageIndexSchema, type NuGetResourceSchema, type NuGetSearchResultSchema, type NuGetSearchResultsSchema, type NuGetServiceIndexSchema, type NuGetServiceMetadataItemSchema, type NuGetServiceMetadataSchema, type NuGetServiceMetadataVersionSchema, type OffsetPagination, type OffsetPaginationRequestOptions, type OverrodeGroupMemberSchema, type PackageFileSchema, type PackageMetadata, PackageRegistry, type PackageRegistrySchema, type PackageSchema, type PackageSnapshotSchema, Packages, type PagesDomainSchema, PagesDomains, type PaginatedResponse, type PaginationRequestOptions, type PaginationRequestSubOptions, type PaginationTypes, type PersonalAccessTokenSchema, type PersonalAccessTokenScopes, PersonalAccessTokens, type PipelineScheduleSchema, PipelineScheduleVariables, PipelineSchedules, type PipelineSchema, type PipelineStatus, type PipelineTestCaseSchema, type PipelineTestReportSchema, type PipelineTestReportSummarySchema, type PipelineTestSuiteSchema, type PipelineTriggerTokenSchema, PipelineTriggerTokens, type PipelineVariableSchema, Pipelines, type ProcessMetricSchema, ProductAnalytics, ProjectAccessRequests, ProjectAccessTokens, type ProjectAliasSchema, ProjectAliases, type ProjectAllowListSchema, type ProjectBadgeSchema, ProjectBadges, ProjectCustomAttributes, ProjectDORA4Metrics, type ProjectExternalStatusCheckSchema, type ProjectFileUploadSchema, type ProjectHookSchema, ProjectHooks, ProjectImportExports, ProjectInvitations, type ProjectIssueBoardSchema, ProjectIssueBoards, ProjectIterations, ProjectJobTokenScopes, ProjectLabels, type ProjectLevelApprovalRuleSchema, type ProjectLevelMergeRequestApprovalSchema, type ProjectLicenseSchema, ProjectMarkdownUploads, ProjectMembers, ProjectMilestones, ProjectProtectedEnvironments, ProjectPushRules, ProjectRelationsExport, ProjectReleases, type ProjectRemoteMirrorSchema, ProjectRemoteMirrors, type ProjectRepositoryStorageMoveSchema, ProjectRepositoryStorageMoves, type ProjectSchema, ProjectSnippetAwardEmojis, ProjectSnippetDiscussions, ProjectSnippetNotes, ProjectSnippets, type ProjectStarrerSchema, type ProjectStatisticSchema, ProjectStatistics, type ProjectStatisticsSchema, type ProjectStoragePath, type ProjectTemplateSchema, type ProjectTemplateType, ProjectTemplates, ProjectTerraformState, type ProjectVariableSchema, ProjectVariables, ProjectVulnerabilities, type ProjectVulnerabilitySchema, ProjectWikis, Projects, type ProtectedBranchAccessLevel, type ProtectedBranchSchema, ProtectedBranches, type ProtectedEnvironmentAccessLevelEntity, type ProtectedEnvironmentAccessLevelSummarySchema, type ProtectedEnvironmentSchema, type ProtectedTagAccessLevel, type ProtectedTagAccessLevelEntity, type ProtectedTagAccessLevelSummarySchema, type ProtectedTagSchema, ProtectedTags, type PushRuleSchema, PyPI, type RecipeSnapshotSchema, type ReferenceSchema, type RegistryRepositorySchema, type RegistryRepositoryTagSchema, type RelatedEpicLinkSchema, type RelatedEpicLinkType, type RelatedEpicSchema, type RelationsExportStatusSchema, type ReleaseAssetLink, type ReleaseAssetSource, type ReleaseEvidence, type ReleaseLinkSchema, ReleaseLinks, type ReleaseSchema, type RemoveRepositoryFileOptions, type RemoveSidekiqQueueOptions, Repositories, type RepositoryBlobSchema, type RepositoryChangelogSchema, type RepositoryCompareSchema, type RepositoryContributorSchema, type RepositoryFileBlameSchema, type RepositoryFileExpandedSchema, type RepositoryFileSchema, RepositoryFiles, type RepositoryImportStatusSchema, type RepositoryStorageMoveSchema, type RepositorySubmoduleSchema, RepositorySubmodules, type RepositoryTreeSchema, ResourceAccessRequests, ResourceAccessTokens, ResourceAwardEmojis, ResourceBadges, ResourceCustomAttributes, ResourceDORA4Metrics, ResourceDiscussions, type ResourceGroupSchema, ResourceGroups, ResourceHooks, ResourceInvitations, ResourceIssueBoards, ResourceIterationEvents, ResourceIterations, ResourceLabelEvents, ResourceLabels, ResourceMarkdownUploads, ResourceMembers, ResourceMilestoneEvents, ResourceMilestones, ResourceNoteAwardEmojis, ResourceNotes, ResourceProtectedEnvironments, ResourcePushRules, ResourceRepositoryStorageMoves, ResourceStateEvents, ResourceTemplates, ResourceVariables, ResourceWeightEvents, ResourceWikis, type ReviewAppSchema, RubyGems, type RunnerSchema, type RunnerToken, Runners, type SAMLGroupSchema, Search, SearchAdmin, type SearchCommitSchema, type SearchMigrationSchema, type SearchScopes, type SecureFileSchema, SecureFiles, type ServiceAccountAccessTokenSchema, type ServiceAccountSchema, ServiceAccounts, ServiceData, type ShowChangelogOptions, type ShowExpanded, type SidekickCompoundMetricsSchema, type SidekickJobStatsSchema, type SidekickProcessMetricsSchema, type SidekickQueueMetricsSchema, SidekiqMetrics, type SidekiqQueueStatus, SidekiqQueues, type SimpleGroupSchema, type SimpleLabelSchema, type SimpleMemberSchema, type SimpleProjectSchema, type SimpleSnippetSchema, type SimpleUserSchema, type SnippetNoteSchema, type SnippetRepositoryStorageMoveSchema, SnippetRepositoryStorageMoves, type SnippetSchema, type SnippetVisibility, Snippets, type StarredDashboardSchema, type StateEventSchema, type StatisticsSchema, type Sudo, type SuggestionSchema, Suggestions, type SupportedIntegration, type SystemHookTestResponse, SystemHooks, type TagSchema, type TagSignatureSchema, Tags, type TaskCompletionStatusSchema, type TemplateSchema, type TerraformStateSchema, type TimeStatsSchema, type TodoAction, TodoLists, type TodoSchema, type TodoState, type TodoType, type TopicSchema, Topics, type UserActivitySchema, type UserAgentDetailSchema, type UserAssociationCountSchema, type UserCountSchema, UserCustomAttributes, type UserEmailSchema, UserEmails, type UserGPGKeySchema, UserGPGKeys, type UserImpersonationTokenSchema, UserImpersonationTokens, type UserMembershipSchema, type UserPreferenceSchema, type UserRunnerSchema, type UserSSHKeySchema, UserSSHKeys, type UserSchema, UserStarredMetricsDashboard, type UserStatusSchema, Users, type VariableFilter, type VariableSchema, type VariableType, type WebhookBaseNoteEventSchema, type WebhookBasePushEventSchema, type WebhookCommitNoteEventSchema, type WebhookDeploymentEventSchema, type WebhookDiffSchema, type WebhookEmojiEventSchema, type WebhookFeatureFlagEventSchema, type WebhookGroupMemberEventSchema, type WebhookIssueEventSchema, type WebhookIssueNoteEventSchema, type WebhookJobEventSchema, type WebhookLabelSchema, type WebhookMergeRequestEventSchema, type WebhookMergeRequestNoteEventSchema, type WebhookPipelineEventSchema, type WebhookPipelineSchema, type WebhookProjectSchema, type WebhookPushEventSchema, type WebhookReleaseEventSchema, type WebhookRepositorySchema, type WebhookSnippetNoteEventSchema, type WebhookSubGroupEventSchema, type WebhookTagEventSchema, type WebhookUserSchema, type WebhookWikiEventSchema, type WeightEventSchema, type WikiAttachmentSchema, type WikiSchema, type X509SignatureSchema };
