import { BaseResource } from '@gitbeaker/requester-utils';
import { camelizeKeys, decamelizeKeys } from 'xcase';
import QS, { parse } from 'qs';

// src/resources/Agents.ts
function appendFormFromObject(object) {
  const form = new FormData();
  Object.entries(object).forEach(([k, v]) => {
    if (v == null) return;
    if (Array.isArray(v)) form.append(k, v[0], v[1]);
    else form.append(k, v);
  });
  return form;
}
var RawPathSegment = class {
  value;
  constructor(value) {
    this.value = value;
  }
  toString() {
    return this.value;
  }
};
function endpoint(strings, ...values) {
  return values.reduce((result, value, index) => {
    const encodedValue = value instanceof RawPathSegment ? value.value : encodeURIComponent(String(value));
    return result + encodedValue + strings[index + 1];
  }, strings[0]);
}
function parseLinkHeader(linkString) {
  const output = {};
  const regex = /<([^>]+)>; rel="([^"]+)"/g;
  let m;
  while (m = regex.exec(linkString)) {
    const [, v, k] = m;
    output[k] = v;
  }
  return output;
}
function reformatObjectOptions(obj, prefixKey, decamelizeValues = false) {
  const formatted = decamelizeValues ? decamelizeKeys(obj) : obj;
  return QS.stringify({ [prefixKey]: formatted }, { encode: false }).split("&").reduce((acc, cur) => {
    const [key, val] = cur.split(/=(.*)/);
    acc[key] = val;
    return acc;
  }, {});
}
function packageResponse(response, showExpanded) {
  return showExpanded ? {
    data: response.body,
    status: response.status,
    headers: response.headers
  } : response.body;
}
function getStream(response, showExpanded) {
  return packageResponse(response, showExpanded);
}
function getSingle(camelize, response, showExpanded) {
  const { status, headers } = response;
  let { body } = response;
  if (camelize) body = camelizeKeys(body);
  return packageResponse({ body, status, headers }, showExpanded);
}
async function getManyMore(camelize, getFn, endpoint2, response, requestOptions, acc) {
  const { sudo, showExpanded, maxPages, pagination, page, perPage, idAfter, orderBy, sort } = requestOptions;
  if (camelize) response.body = camelizeKeys(response?.body);
  const newAcc = [...acc || [], ...response.body];
  const withinBounds = maxPages && perPage ? newAcc.length / +perPage < maxPages : true;
  const { next = "" } = parseLinkHeader(response.headers.link);
  if (!(page && (acc || []).length === 0) && next && withinBounds) {
    const parsedQueryString = parse(next.split("?")[1]);
    const qs = { ...camelizeKeys(parsedQueryString) };
    const newOpts = {
      ...qs,
      maxPages,
      sudo,
      showExpanded
    };
    const nextResponse = await getFn(endpoint2, {
      searchParams: qs,
      sudo
    });
    return getManyMore(camelize, getFn, endpoint2, nextResponse, newOpts, newAcc);
  }
  if (!showExpanded) return newAcc;
  const paginationInfo = pagination === "keyset" ? {
    idAfter: idAfter ? +idAfter : null,
    perPage: perPage ? +perPage : null,
    orderBy,
    sort
  } : {
    total: parseInt(response.headers["x-total"], 10),
    next: parseInt(response.headers["x-next-page"], 10) || null,
    current: parseInt(response.headers["x-page"], 10) || 1,
    previous: parseInt(response.headers["x-prev-page"], 10) || null,
    perPage: parseInt(response.headers["x-per-page"], 10),
    totalPages: parseInt(response.headers["x-total-pages"], 10)
  };
  return {
    data: newAcc,
    paginationInfo
  };
}
function get() {
  return async (service, endpoint2, options) => {
    const { asStream, sudo, showExpanded, maxPages, ...searchParams } = options || {};
    const signal = service.queryTimeout ? AbortSignal.timeout(service.queryTimeout) : void 0;
    const response = await service.requester.get(endpoint2, {
      searchParams,
      sudo,
      asStream,
      signal
    });
    const camelizeResponseBody = service.camelize || false;
    if (asStream) return getStream(response, showExpanded);
    if (!Array.isArray(response.body))
      return getSingle(
        camelizeResponseBody,
        response,
        showExpanded
      );
    const reqOpts = {
      sudo,
      showExpanded,
      maxPages,
      ...searchParams
    };
    return getManyMore(
      camelizeResponseBody,
      (ep, op) => service.requester.get(ep, { ...op, signal }),
      endpoint2,
      response,
      reqOpts
    );
  };
}
function post() {
  return async (service, endpoint2, { searchParams, isForm, sudo, showExpanded, ...options } = {}) => {
    const body = isForm ? appendFormFromObject(options) : options;
    const response = await service.requester.post(endpoint2, {
      searchParams,
      body,
      sudo,
      signal: service.queryTimeout ? AbortSignal.timeout(service.queryTimeout) : void 0
    });
    if (service.camelize) response.body = camelizeKeys(response.body);
    return packageResponse(response, showExpanded);
  };
}
function put() {
  return async (service, endpoint2, { searchParams, isForm, sudo, showExpanded, ...options } = {}) => {
    const body = isForm ? appendFormFromObject(options) : options;
    const response = await service.requester.put(endpoint2, {
      body,
      searchParams,
      sudo,
      signal: service.queryTimeout ? AbortSignal.timeout(service.queryTimeout) : void 0
    });
    if (service.camelize) response.body = camelizeKeys(response.body);
    return packageResponse(response, showExpanded);
  };
}
function patch() {
  return async (service, endpoint2, { searchParams, isForm, sudo, showExpanded, ...options } = {}) => {
    const body = isForm ? appendFormFromObject(options) : options;
    const response = await service.requester.patch(endpoint2, {
      body,
      searchParams,
      sudo,
      signal: service.queryTimeout ? AbortSignal.timeout(service.queryTimeout) : void 0
    });
    if (service.camelize) response.body = camelizeKeys(response.body);
    return packageResponse(response, showExpanded);
  };
}
function del() {
  return async (service, endpoint2, { sudo, showExpanded, searchParams, ...options } = {}) => {
    const response = await service.requester.delete(endpoint2, {
      body: options,
      searchParams,
      sudo,
      signal: service.queryTimeout ? AbortSignal.timeout(service.queryTimeout) : void 0
    });
    return packageResponse(response, showExpanded);
  };
}
var RequestHelper = {
  post,
  put,
  patch,
  get,
  del
};

// src/resources/Agents.ts
var Agents = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/cluster_agents`,
      options
    );
  }
  allTokens(projectId, agentId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/cluster_agents/${agentId}/tokens`,
      options
    );
  }
  createToken(projectId, agentId, name, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/cluster_agents/${agentId}/tokens`,
      {
        name,
        ...options
      }
    );
  }
  show(projectId, agentId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/cluster_agents/${agentId}`,
      options
    );
  }
  showToken(projectId, agentId, tokenId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/cluster_agents/${agentId}/tokens/${tokenId}`,
      options
    );
  }
  register(projectId, name, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/cluster_agents`,
      {
        name,
        ...options
      }
    );
  }
  removeToken(projectId, agentId, tokenId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/cluster_agents/${agentId}/tokens/${tokenId}`,
      options
    );
  }
  unregister(projectId, agentId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/cluster_agents/${agentId}`,
      options
    );
  }
};
var AlertManagement = class extends BaseResource {
  allMetricImages(projectId, alertIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/alert_management_alerts/${alertIId}/metric_images`,
      options
    );
  }
  editMetricImage(projectId, alertIId, imageId, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/alert_management_alerts/${alertIId}/metric_images/${imageId}`,
      options
    );
  }
  removeMetricImage(projectId, alertIId, imageId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/alert_management_alerts/${alertIId}/metric_images/${imageId}`,
      options
    );
  }
  uploadMetricImage(projectId, alertIId, metricImage, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/alert_management_alerts/${alertIId}/metric_images`,
      {
        isForm: true,
        file: [metricImage.content, metricImage.filename],
        ...options
      }
    );
  }
};
var ApplicationAppearance = class extends BaseResource {
  show(options) {
    return RequestHelper.get()(
      this,
      "application/appearence",
      options
    );
  }
  edit({
    logo,
    pwaIcon,
    ...options
  } = {}) {
    if (logo || pwaIcon) {
      const opts = {
        ...options,
        isForm: true
      };
      if (logo) opts.logo = [logo.content, logo.filename];
      if (pwaIcon) opts.pwaIcon = [pwaIcon.content, pwaIcon.filename];
      return RequestHelper.put()(this, "application/appearence", opts);
    }
    return RequestHelper.put()(
      this,
      "application/appearence",
      options
    );
  }
};
var ApplicationPlanLimits = class extends BaseResource {
  show(options) {
    return RequestHelper.get()(
      this,
      "application/plan_limits",
      options
    );
  }
  edit(planName, options = {}) {
    const {
      ciPipelineSize,
      ciActiveJobs,
      ciActivePipelines,
      ciProjectSubscriptions,
      ciPipelineSchedules,
      ciNeedsSizeLimit,
      ciRegisteredGroupRunners,
      ciRegisteredProjectRunners,
      conanMaxFileSize,
      genericPackagesMaxFileSize,
      helmMaxFileSize,
      mavenMaxFileSize,
      npmMaxFileSize,
      nugetMaxFileSize,
      pypiMaxFileSize,
      terraformModuleMaxFileSize,
      storageSizeLimit,
      ...opts
    } = options;
    return RequestHelper.put()(this, "application/plan_limits", {
      ...opts,
      searchParams: {
        planName,
        ciPipelineSize,
        ciActiveJobs,
        ciActivePipelines,
        ciProjectSubscriptions,
        ciPipelineSchedules,
        ciNeedsSizeLimit,
        ciRegisteredGroupRunners,
        ciRegisteredProjectRunners,
        conanMaxFileSize,
        genericPackagesMaxFileSize,
        helmMaxFileSize,
        mavenMaxFileSize,
        npmMaxFileSize,
        nugetMaxFileSize,
        pypiMaxFileSize,
        terraformModuleMaxFileSize,
        storageSizeLimit
      }
    });
  }
};
var ApplicationSettings = class extends BaseResource {
  show(options) {
    return RequestHelper.get()(this, "application/settings", options);
  }
  edit(options) {
    return RequestHelper.put()(this, "application/settings", options);
  }
};
var ApplicationStatistics = class extends BaseResource {
  show(options) {
    return RequestHelper.get()(this, "application/statistics", options);
  }
};
var Applications = class extends BaseResource {
  all(options) {
    return RequestHelper.get()(this, "applications", options);
  }
  create(name, redirectUri, scopes, options) {
    return RequestHelper.post()(this, "applications", {
      name,
      redirectUri,
      scopes,
      ...options
    });
  }
  remove(applicationId, options) {
    return RequestHelper.del()(this, `applications/${applicationId}`, options);
  }
};
function url({
  projectId,
  groupId
} = {}) {
  let prefix = "";
  if (projectId) prefix = endpoint`projects/${projectId}/`;
  else if (groupId) prefix = endpoint`groups/${groupId}/`;
  return `${prefix}audit_events`;
}
var AuditEvents = class extends BaseResource {
  all({
    projectId,
    groupId,
    ...options
  } = {}) {
    const uri = url({ projectId, groupId });
    return RequestHelper.get()(
      this,
      uri,
      options
    );
  }
  show(auditEventId, {
    projectId,
    groupId,
    ...options
  } = {}) {
    const uri = url({ projectId, groupId });
    return RequestHelper.get()(this, `${uri}/${auditEventId}`, options);
  }
};
var Avatar = class extends BaseResource {
  show(email, options) {
    return RequestHelper.get()(this, "avatar", { email, ...options });
  }
};
var BroadcastMessages = class extends BaseResource {
  all(options) {
    return RequestHelper.get()(this, "broadcast_messages", options);
  }
  create(options) {
    return RequestHelper.post()(this, "broadcast_messages", options);
  }
  edit(broadcastMessageId, options) {
    return RequestHelper.put()(
      this,
      `broadcast_messages/${broadcastMessageId}`,
      options
    );
  }
  remove(broadcastMessageId, options) {
    return RequestHelper.del()(this, `broadcast_messages/${broadcastMessageId}`, options);
  }
  show(broadcastMessageId, options) {
    return RequestHelper.get()(
      this,
      `broadcast_messages/${broadcastMessageId}`,
      options
    );
  }
};
var CodeSuggestions = class extends BaseResource {
  createAccessToken(options) {
    return RequestHelper.post()(this, "code_suggestions/tokens", options);
  }
  generateCompletion(options) {
    return RequestHelper.post()(
      this,
      "code_suggestions/completions",
      options
    );
  }
};
var Composer = class extends BaseResource {
  create(projectId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/packages/composer`,
      options
    );
  }
  download(projectId, packageName, sha, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/packages/composer/archives/${packageName}`,
      {
        searchParams: { sha },
        ...options
      }
    );
  }
  showMetadata(groupId, packageName, options) {
    let url12;
    if (options && options.sha) {
      url12 = endpoint`groups/${groupId}/-/packages/composer/${packageName}$${options.sha}`;
    } else {
      url12 = endpoint`groups/${groupId}/-/packages/composer/p2/${packageName}`;
    }
    return RequestHelper.get()(this, url12, options);
  }
  showPackages(groupId, sha, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/-/packages/composer/p/${sha}`,
      options
    );
  }
  showBaseRepository(groupId, options) {
    const clonedService = { ...this };
    if (options && options.composerVersion === "2") {
      clonedService.headers["User-Agent"] = "Composer/2";
    }
    return RequestHelper.get()(
      clonedService,
      endpoint`groups/${groupId}/-/packages/composer/packages`,
      options
    );
  }
};
function url2(projectId) {
  return projectId ? endpoint`projects/${projectId}/packages/conan/v1` : "packages/conan/v1";
}
var Conan = class extends BaseResource {
  authenticate({
    projectId,
    ...options
  } = {}) {
    return RequestHelper.get()(this, `${url2(projectId)}/users/authenticate`, options);
  }
  checkCredentials({
    projectId,
    ...options
  } = {}) {
    const prefix = url2(projectId);
    return RequestHelper.get()(this, `${prefix}/users/check_credentials`, options);
  }
  downloadPackageFile(packageName, packageVersion, packageUsername, packageChannel, conanPackageReference, recipeRevision, packageRevision, filename, { projectId, ...options } = {}) {
    const prefix = url2(projectId);
    return RequestHelper.get()(
      this,
      `${prefix}/conans/${packageName}/${packageVersion}/${packageUsername}/${packageChannel}/${recipeRevision}/package/${conanPackageReference}/${packageRevision}/${filename}`,
      options
    );
  }
  downloadRecipeFile(packageName, packageVersion, packageUsername, packageChannel, recipeRevision, filename, { projectId, ...options } = {}) {
    const prefix = url2(projectId);
    return RequestHelper.get()(
      this,
      `${prefix}/conans/${packageName}/${packageVersion}/${packageUsername}/${packageChannel}/${recipeRevision}/export/${filename}`,
      options
    );
  }
  showPackageUploadUrls(packageName, packageVersion, packageUsername, packageChannel, conanPackageReference, { projectId, ...options } = {}) {
    const prefix = url2(projectId);
    return RequestHelper.get()(
      this,
      `${prefix}/conans/${packageName}/${packageVersion}/${packageUsername}/${packageChannel}/packages/${conanPackageReference}/upload_urls`,
      options
    );
  }
  showPackageDownloadUrls(packageName, packageVersion, packageUsername, packageChannel, conanPackageReference, { projectId, ...options } = {}) {
    const prefix = url2(projectId);
    return RequestHelper.get()(
      this,
      `${prefix}/conans/${packageName}/${packageVersion}/${packageUsername}/${packageChannel}/packages/${conanPackageReference}/download_urls`,
      options
    );
  }
  showPackageManifest(packageName, packageVersion, packageUsername, packageChannel, conanPackageReference, { projectId, ...options } = {}) {
    const prefix = url2(projectId);
    return RequestHelper.get()(
      this,
      `${prefix}/conans/${packageName}/${packageVersion}/${packageUsername}/${packageChannel}/packages/${conanPackageReference}/digest`,
      options
    );
  }
  showPackageSnapshot(packageName, packageVersion, packageUsername, packageChannel, conanPackageReference, { projectId, ...options } = {}) {
    const prefix = url2(projectId);
    return RequestHelper.get()(
      this,
      `${prefix}/conans/${packageName}/${packageVersion}/${packageUsername}/${packageChannel}/packages/${conanPackageReference}`,
      options
    );
  }
  ping({
    projectId,
    ...options
  } = {}) {
    return RequestHelper.post()(this, `${url2(projectId)}/ping`, options);
  }
  showRecipeUploadUrls(packageName, packageVersion, packageUsername, packageChannel, { projectId, ...options } = {}) {
    const prefix = url2(projectId);
    return RequestHelper.get()(
      this,
      `${prefix}/conans/${packageName}/${packageVersion}/${packageUsername}/${packageChannel}/upload_urls`,
      options
    );
  }
  showRecipeDownloadUrls(packageName, packageVersion, packageUsername, packageChannel, { projectId, ...options } = {}) {
    const prefix = url2(projectId);
    return RequestHelper.get()(
      this,
      `${prefix}/conans/${packageName}/${packageVersion}/${packageUsername}/${packageChannel}/download_urls`,
      options
    );
  }
  showRecipeManifest(packageName, packageVersion, packageUsername, packageChannel, { projectId, ...options } = {}) {
    const prefix = url2(projectId);
    return RequestHelper.get()(
      this,
      `${prefix}/conans/${packageName}/${packageVersion}/${packageUsername}/${packageChannel}/digest`,
      options
    );
  }
  showRecipeSnapshot(packageName, packageVersion, packageUsername, packageChannel, { projectId, ...options } = {}) {
    const prefix = url2(projectId);
    return RequestHelper.get()(
      this,
      `${prefix}/conans/${packageName}/${packageVersion}/${packageUsername}/${packageChannel}`,
      options
    );
  }
  removePackageFile(packageName, packageVersion, packageUsername, packageChannel, { projectId, ...options } = {}) {
    const prefix = url2(projectId);
    return RequestHelper.get()(
      this,
      `${prefix}/conans/${packageName}/${packageVersion}/${packageUsername}/${packageChannel}`,
      options
    );
  }
  search({
    projectId,
    ...options
  } = {}) {
    const prefix = url2(projectId);
    return RequestHelper.get()(this, `${prefix}/conans/search`, options);
  }
  uploadPackageFile(packageFile, packageName, packageVersion, packageUsername, packageChannel, conanPackageReference, recipeRevision, packageRevision, options) {
    const prefix = url2();
    return RequestHelper.get()(
      this,
      `${prefix}/files/${packageName}/${packageVersion}/${packageUsername}/${packageChannel}/${recipeRevision}/package/${conanPackageReference}/${packageRevision}/${packageFile.filename}`,
      {
        isForm: true,
        ...options,
        file: [packageFile.content, packageFile.filename]
      }
    );
  }
  uploadRecipeFile(packageFile, packageName, packageVersion, packageUsername, packageChannel, recipeRevision, options) {
    const prefix = url2();
    return RequestHelper.get()(
      this,
      `${prefix}/files/${packageName}/${packageVersion}/${packageUsername}/${packageChannel}/${recipeRevision}/export/${packageFile.filename}`,
      {
        isForm: true,
        ...options,
        file: [packageFile.content, packageFile.filename]
      }
    );
  }
};
var DashboardAnnotations = class extends BaseResource {
  create(dashboardPath, startingAt, description, {
    environmentId,
    clusterId,
    ...options
  } = {}) {
    let url12;
    if (environmentId) url12 = endpoint`environments/${environmentId}/metrics_dashboard/annotations`;
    else if (clusterId) url12 = endpoint`clusters/${clusterId}/metrics_dashboard/annotations`;
    else
      throw new Error(
        "Missing required argument. Please supply a environmentId or a cluserId in the options parameter."
      );
    return RequestHelper.post()(this, url12, {
      dashboardPath,
      startingAt,
      description,
      ...options
    });
  }
};
function url3({
  projectId,
  groupId
} = {}) {
  if (projectId) return endpoint`/projects/${projectId}/packages/debian`;
  if (groupId) return endpoint`/groups/${groupId}/-/packages/debian`;
  throw new Error(
    "Missing required argument. Please supply a projectId or a groupId in the options parameter"
  );
}
var Debian = class extends BaseResource {
  downloadBinaryFileIndex(distribution, component, architecture, {
    projectId,
    groupId,
    ...options
  }) {
    const prefix = url3({
      projectId,
      groupId
    });
    return RequestHelper.get()(
      this,
      `${prefix}/dists/${distribution}/${component}/binary-${architecture}/Packages`,
      options
    );
  }
  downloadDistributionReleaseFile(distribution, {
    projectId,
    groupId,
    ...options
  }) {
    const prefix = url3({
      projectId,
      groupId
    });
    return RequestHelper.get()(
      this,
      `${prefix}/dists/${distribution}/Release`,
      options
    );
  }
  downloadSignedDistributionReleaseFile(distribution, {
    projectId,
    groupId,
    ...options
  }) {
    const prefix = url3({
      projectId,
      groupId
    });
    return RequestHelper.get()(
      this,
      `${prefix}/dists/${distribution}/InRelease`,
      options
    );
  }
  downloadReleaseFileSignature(distribution, {
    projectId,
    groupId,
    ...options
  }) {
    const prefix = url3({
      projectId,
      groupId
    });
    return RequestHelper.get()(
      this,
      `${prefix}/dists/${distribution}/Release.gpg`,
      options
    );
  }
  downloadPackageFile(projectId, distribution, letter, packageName, packageVersion, filename, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/packages/debian/pool/${distribution}/${letter}/${packageName}/${packageVersion}/${filename}`,
      options
    );
  }
  uploadPackageFile(projectId, packageFile, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/packages/debian/${packageFile.filename}`,
      {
        isForm: true,
        ...options,
        file: [packageFile.content, packageFile.filename]
      }
    );
  }
};
var DependencyProxy = class extends BaseResource {
  remove(groupId, options) {
    return RequestHelper.post()(this, `groups/${groupId}/dependency_proxy/cache`, options);
  }
};
var DeployKeys = class extends BaseResource {
  all({
    projectId,
    userId,
    ...options
  } = {}) {
    let url12;
    if (projectId) {
      url12 = endpoint`projects/${projectId}/deploy_keys`;
    } else if (userId) {
      url12 = endpoint`users/${userId}/project_deploy_keys`;
    } else {
      url12 = "deploy_keys";
    }
    return RequestHelper.get()(this, url12, options);
  }
  create(projectId, title, key, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/deploy_keys`,
      {
        title,
        key,
        ...options
      }
    );
  }
  edit(projectId, keyId, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/deploy_keys/${keyId}`,
      options
    );
  }
  enable(projectId, keyId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/deploy_keys/${keyId}/enable`,
      options
    );
  }
  remove(projectId, keyId, options) {
    return RequestHelper.del()(this, endpoint`projects/${projectId}/deploy_keys/${keyId}`, options);
  }
  show(projectId, keyId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/deploy_keys/${keyId}`,
      options
    );
  }
};
var DeployTokens = class extends BaseResource {
  all({
    projectId,
    groupId,
    ...options
  } = {}) {
    let url12;
    if (projectId) url12 = endpoint`projects/${projectId}/deploy_tokens`;
    else if (groupId) url12 = endpoint`groups/${groupId}/deploy_tokens`;
    else url12 = "deploy_tokens";
    return RequestHelper.get()(this, url12, options);
  }
  create(name, scopes, {
    projectId,
    groupId,
    ...options
  } = {}) {
    let url12;
    if (projectId) url12 = endpoint`projects/${projectId}/deploy_tokens`;
    else if (groupId) url12 = endpoint`groups/${groupId}/deploy_tokens`;
    else {
      throw new Error(
        "Missing required argument. Please supply a projectId or a groupId in the options parameter."
      );
    }
    return RequestHelper.post()(this, url12, {
      name,
      scopes,
      ...options
    });
  }
  remove(tokenId, {
    projectId,
    groupId,
    ...options
  } = {}) {
    let url12;
    if (projectId) url12 = endpoint`projects/${projectId}/deploy_tokens/${tokenId}`;
    else if (groupId) url12 = endpoint`groups/${groupId}/deploy_tokens/${tokenId}`;
    else {
      throw new Error(
        "Missing required argument. Please supply a projectId or a groupId in the options parameter."
      );
    }
    return RequestHelper.del()(this, url12, options);
  }
  show(tokenId, {
    projectId,
    groupId,
    ...options
  } = {}) {
    let url12;
    if (projectId) url12 = endpoint`projects/${projectId}/deploy_tokens/${tokenId}`;
    else if (groupId) url12 = endpoint`groups/${groupId}/deploy_tokens/${tokenId}`;
    else {
      throw new Error(
        "Missing required argument. Please supply a projectId or a groupId in the options parameter."
      );
    }
    return RequestHelper.get()(
      this,
      url12,
      options
    );
  }
};
var ResourceAccessRequests = class extends BaseResource {
  constructor(resourceType, options) {
    super({ prefixUrl: resourceType, ...options });
  }
  all(resourceId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/access_requests`,
      options
    );
  }
  request(resourceId, options) {
    return RequestHelper.post()(
      this,
      endpoint`${resourceId}/access_requests`,
      options
    );
  }
  approve(resourceId, userId, options) {
    return RequestHelper.put()(
      this,
      endpoint`${resourceId}/access_requests/${userId}/approve`,
      options
    );
  }
  deny(resourceId, userId, options) {
    return RequestHelper.del()(this, endpoint`${resourceId}/access_requests/${userId}`, options);
  }
};
var ResourceAccessTokens = class extends BaseResource {
  constructor(resourceType, options) {
    super({ prefixUrl: resourceType, ...options });
  }
  all(resourceId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/access_tokens`,
      options
    );
  }
  create(resourceId, name, scopes, expiresAt, options) {
    return RequestHelper.post()(
      this,
      endpoint`${resourceId}/access_tokens`,
      {
        name,
        scopes,
        expiresAt,
        ...options
      }
    );
  }
  revoke(resourceId, tokenId, options) {
    return RequestHelper.del()(this, endpoint`${resourceId}/access_tokens/${tokenId}`, options);
  }
  rotate(resourceId, tokenId, options) {
    return RequestHelper.post()(
      this,
      endpoint`${resourceId}/access_tokens/${tokenId}/rotate`,
      options
    );
  }
  show(resourceId, tokenId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/access_tokens/${tokenId}`,
      options
    );
  }
};
function url4(resourceId, resourceType2, resourceId2, awardId) {
  const [rId, rId2] = [resourceId, resourceId2].map(encodeURIComponent);
  const output = [rId, resourceType2, rId2];
  output.push("award_emoji");
  if (awardId) output.push(awardId);
  return output.join("/");
}
var ResourceAwardEmojis = class extends BaseResource {
  resourceType2;
  constructor(resourceType1, resourceType2, options) {
    super({ prefixUrl: resourceType1, ...options });
    this.resourceType2 = resourceType2;
  }
  all(resourceId, resourceIId, options) {
    return RequestHelper.get()(
      this,
      url4(resourceId, this.resourceType2, resourceIId),
      options
    );
  }
  award(resourceId, resourceIId, name, options) {
    return RequestHelper.post()(
      this,
      url4(resourceId, this.resourceType2, resourceIId),
      {
        name,
        ...options
      }
    );
  }
  remove(resourceId, resourceIId, awardId, options) {
    return RequestHelper.del()(
      this,
      url4(resourceId, this.resourceType2, resourceIId, awardId),
      options
    );
  }
  show(resourceId, resourceIId, awardId, options) {
    return RequestHelper.get()(
      this,
      url4(resourceId, this.resourceType2, resourceIId, awardId),
      options
    );
  }
};
function url5(resourceId, resourceType2, resourceId2, noteId, awardId) {
  const [rId, rId2] = [resourceId, resourceId2].map(encodeURIComponent);
  const output = [rId, resourceType2, rId2];
  output.push("notes");
  output.push(noteId);
  output.push("award_emoji");
  if (awardId) output.push(awardId);
  return output.join("/");
}
var ResourceNoteAwardEmojis = class extends BaseResource {
  resourceType;
  constructor(resourceType, options) {
    super({ prefixUrl: "projects", ...options });
    this.resourceType = resourceType;
  }
  all(projectId, resourceIId, noteId, options) {
    return RequestHelper.get()(
      this,
      url5(projectId, this.resourceType, resourceIId, noteId),
      options
    );
  }
  award(projectId, resourceIId, noteId, name, options) {
    return RequestHelper.post()(
      this,
      url5(projectId, this.resourceType, resourceIId, noteId),
      {
        name,
        ...options
      }
    );
  }
  remove(projectId, resourceIId, noteId, awardId, options) {
    return RequestHelper.del()(
      this,
      url5(projectId, this.resourceType, resourceIId, noteId, awardId),
      options
    );
  }
  show(projectId, resourceIId, noteId, awardId, options) {
    return RequestHelper.get()(
      this,
      url5(projectId, this.resourceType, resourceIId, noteId, awardId),
      options
    );
  }
};
var ResourceBadges = class extends BaseResource {
  constructor(resourceType, options) {
    super({ prefixUrl: resourceType, ...options });
  }
  add(resourceId, linkUrl, imageUrl, options) {
    return RequestHelper.post()(this, endpoint`${resourceId}/badges`, {
      linkUrl,
      imageUrl,
      ...options
    });
  }
  all(resourceId, options) {
    return RequestHelper.get()(this, endpoint`${resourceId}/badges`, options);
  }
  edit(resourceId, badgeId, options) {
    return RequestHelper.put()(
      this,
      endpoint`${resourceId}/badges/${badgeId}`,
      options
    );
  }
  preview(resourceId, linkUrl, imageUrl, options) {
    return RequestHelper.get()(this, endpoint`${resourceId}/badges/render`, {
      linkUrl,
      imageUrl,
      ...options
    });
  }
  remove(resourceId, badgeId, options) {
    return RequestHelper.del()(this, endpoint`${resourceId}/badges/${badgeId}`, options);
  }
  show(resourceId, badgeId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/badges/${badgeId}`,
      options
    );
  }
};
var ResourceCustomAttributes = class extends BaseResource {
  constructor(resourceType, options) {
    super({ prefixUrl: resourceType, ...options });
  }
  all(resourceId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/custom_attributes`,
      options
    );
  }
  remove(resourceId, customAttributeId, options) {
    return RequestHelper.del()(
      this,
      endpoint`${resourceId}/custom_attributes/${customAttributeId}`,
      options
    );
  }
  set(resourceId, customAttributeId, value, options) {
    return RequestHelper.put()(
      this,
      endpoint`${resourceId}/custom_attributes/${customAttributeId}`,
      {
        value,
        ...options
      }
    );
  }
  show(resourceId, customAttributeId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/custom_attributes/${customAttributeId}`,
      options
    );
  }
};
var ResourceDORA4Metrics = class extends BaseResource {
  constructor(resourceType, options) {
    super({ prefixUrl: resourceType, ...options });
  }
  all(resourceId, metric, options) {
    return RequestHelper.get()(this, endpoint`${resourceId}/dora/metrics`, {
      metric,
      ...options
    });
  }
};
var ResourceDiscussions = class extends BaseResource {
  resource2Type;
  constructor(resourceType, resource2Type, options) {
    super({ prefixUrl: resourceType, ...options });
    this.resource2Type = resource2Type;
  }
  addNote(resourceId, resource2Id, discussionId, body, options) {
    return RequestHelper.post()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/discussions/${discussionId}/notes`,
      { ...options, body }
    );
  }
  all(resourceId, resource2Id, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/discussions`,
      options
    );
  }
  create(resourceId, resource2Id, body, {
    position,
    ...options
  } = {}) {
    const opts = { ...options, body };
    if (position) {
      Object.assign(opts, reformatObjectOptions(position, "position", true));
      opts.isForm = true;
    }
    return RequestHelper.post()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/discussions`,
      opts
    );
  }
  editNote(resourceId, resource2Id, discussionId, noteId, options) {
    return RequestHelper.put()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/discussions/${discussionId}/notes/${noteId}`,
      options
    );
  }
  removeNote(resourceId, resource2Id, discussionId, noteId, options) {
    return RequestHelper.del()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/discussions/${discussionId}/notes/${noteId}`,
      options
    );
  }
  show(resourceId, resource2Id, discussionId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/discussions/${discussionId}`,
      options
    );
  }
};
var ResourceIssueBoards = class extends BaseResource {
  constructor(resourceType, options) {
    super({ prefixUrl: resourceType, ...options });
  }
  all(resourceId, options) {
    return RequestHelper.get()(this, endpoint`${resourceId}/boards`, options);
  }
  allLists(resourceId, boardId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/boards/${boardId}/lists`,
      options
    );
  }
  create(resourceId, name, options) {
    return RequestHelper.post()(this, endpoint`${resourceId}/boards`, {
      name,
      ...options
    });
  }
  createList(resourceId, boardId, options) {
    return RequestHelper.post()(
      this,
      endpoint`${resourceId}/boards/${boardId}/lists`,
      options
    );
  }
  edit(resourceId, boardId, options) {
    return RequestHelper.put()(
      this,
      endpoint`${resourceId}/boards/${boardId}`,
      options
    );
  }
  editList(resourceId, boardId, listId, position, options) {
    return RequestHelper.put()(
      this,
      endpoint`${resourceId}/boards/${boardId}/lists/${listId}`,
      {
        position,
        ...options
      }
    );
  }
  remove(resourceId, boardId, options) {
    return RequestHelper.del()(this, endpoint`${resourceId}/boards/${boardId}`, options);
  }
  removeList(resourceId, boardId, listId, options) {
    return RequestHelper.del()(
      this,
      endpoint`${resourceId}/boards/${boardId}/lists/${listId}`,
      options
    );
  }
  show(resourceId, boardId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/boards/${boardId}`,
      options
    );
  }
  showList(resourceId, boardId, listId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/boards/${boardId}/lists/${listId}`,
      options
    );
  }
};
var ResourceLabels = class extends BaseResource {
  constructor(resourceType, options) {
    super({ prefixUrl: resourceType, ...options });
  }
  all(resourceId, options) {
    return RequestHelper.get()(this, endpoint`${resourceId}/labels`, options);
  }
  create(resourceId, labelName, color, options) {
    return RequestHelper.post()(this, endpoint`${resourceId}/labels`, {
      name: labelName,
      color,
      ...options
    });
  }
  edit(resourceId, labelId, options) {
    if (!options?.newName && !options?.color)
      throw new Error(
        "Missing required argument. Please supply a color or a newName in the options parameter."
      );
    return RequestHelper.put()(
      this,
      endpoint`${resourceId}/labels/${labelId}`,
      options
    );
  }
  promote(resourceId, labelId, options) {
    return RequestHelper.put()(
      this,
      endpoint`${resourceId}/labels/${labelId}/promote`,
      options
    );
  }
  remove(resourceId, labelId, options) {
    return RequestHelper.del()(this, endpoint`${resourceId}/labels/${labelId}`, options);
  }
  show(resourceId, labelId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/labels/${labelId}`,
      options
    );
  }
  subscribe(resourceId, labelId, options) {
    return RequestHelper.post()(
      this,
      endpoint`${resourceId}/issues/${labelId}/subscribe`,
      options
    );
  }
  unsubscribe(resourceId, labelId, options) {
    return RequestHelper.post()(
      this,
      endpoint`${resourceId}/issues/${labelId}/unsubscribe`,
      options
    );
  }
};
var ResourceMarkdownUploads = class extends BaseResource {
  constructor(resourceType, options) {
    super({ prefixUrl: resourceType, ...options });
  }
  all(resourceId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/uploads`,
      options
    );
  }
  download(resourceId, uploadIdOrSecret, filename, options) {
    if (filename && typeof filename === "string") {
      return RequestHelper.get()(
        this,
        endpoint`${resourceId}/uploads/${uploadIdOrSecret}/${filename}`,
        options
      );
    }
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/uploads/${uploadIdOrSecret}`,
      options
    );
  }
  remove(resourceId, uploadIdOrSecret, filename, options) {
    if (filename && typeof filename === "string") {
      return RequestHelper.del()(
        this,
        endpoint`${resourceId}/uploads/${uploadIdOrSecret}/${filename}`,
        options
      );
    }
    return RequestHelper.del()(this, endpoint`${resourceId}/uploads/${uploadIdOrSecret}`, options);
  }
};
var ResourceMembers = class extends BaseResource {
  constructor(resourceType, options) {
    super({ prefixUrl: resourceType, ...options });
  }
  add(resourceId, accessLevel, options) {
    return RequestHelper.post()(this, endpoint`${resourceId}/members`, {
      accessLevel,
      ...options
    });
  }
  all(resourceId, {
    includeInherited,
    ...options
  } = {}) {
    let url12 = endpoint`${resourceId}/members`;
    if (includeInherited) url12 += "/all";
    return RequestHelper.get()(this, url12, options);
  }
  edit(resourceId, userId, accessLevel, options) {
    return RequestHelper.put()(this, endpoint`${resourceId}/members/${userId}`, {
      accessLevel,
      ...options
    });
  }
  show(resourceId, userId, { includeInherited, ...options } = {}) {
    const [rId, uId] = [resourceId, userId].map(encodeURIComponent);
    const url12 = [rId, "members"];
    if (includeInherited) url12.push("all");
    url12.push(uId);
    return RequestHelper.get()(this, url12.join("/"), options);
  }
  remove(resourceId, userId, options) {
    return RequestHelper.del()(this, endpoint`${resourceId}/members/${userId}`, options);
  }
};
var ResourceMilestones = class extends BaseResource {
  constructor(resourceType, options) {
    super({ prefixUrl: resourceType, ...options });
  }
  all(resourceId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/milestones`,
      options
    );
  }
  allAssignedIssues(resourceId, milestoneId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/milestones/${milestoneId}/issues`,
      options
    );
  }
  allAssignedMergeRequests(resourceId, milestoneId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/milestones/${milestoneId}/merge_requests`,
      options
    );
  }
  allBurndownChartEvents(resourceId, milestoneId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/milestones/${milestoneId}/burndown_events`,
      options
    );
  }
  create(resourceId, title, options) {
    return RequestHelper.post()(this, endpoint`${resourceId}/milestones`, {
      title,
      ...options
    });
  }
  edit(resourceId, milestoneId, options) {
    return RequestHelper.put()(
      this,
      endpoint`${resourceId}/milestones/${milestoneId}`,
      options
    );
  }
  remove(resourceId, milestoneId, options) {
    return RequestHelper.del()(this, endpoint`${resourceId}/milestones/${milestoneId}`, options);
  }
  show(resourceId, milestoneId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/milestones/${milestoneId}`,
      options
    );
  }
};
var ResourceNotes = class extends BaseResource {
  resource2Type;
  constructor(resourceType, resource2Type, options) {
    super({ prefixUrl: resourceType, ...options });
    this.resource2Type = resource2Type;
  }
  all(resourceId, resource2Id, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/notes`,
      options
    );
  }
  create(resourceId, resource2Id, body, options) {
    return RequestHelper.post()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/notes`,
      {
        body,
        ...options
      }
    );
  }
  edit(resourceId, resource2Id, noteId, options) {
    return RequestHelper.put()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/notes/${noteId}`,
      options
    );
  }
  remove(resourceId, resource2Id, noteId, options) {
    return RequestHelper.del()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/notes/${noteId}`,
      options
    );
  }
  show(resourceId, resource2Id, noteId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/notes/${noteId}`,
      options
    );
  }
};
var ResourceTemplates = class extends BaseResource {
  constructor(resourceType, options) {
    super({ prefixUrl: ["templates", resourceType].join("/"), ...options });
  }
  all(options) {
    process.emitWarning(
      'This API will be deprecated as of Gitlabs v5 API. Please make the switch to "ProjectTemplates".',
      "DeprecationWarning"
    );
    return RequestHelper.get()(this, "", options);
  }
  show(key, options) {
    process.emitWarning(
      'This API will be deprecated as of Gitlabs v5 API. Please make the switch to "ProjectTemplates".',
      "DeprecationWarning"
    );
    return RequestHelper.get()(this, encodeURIComponent(key), options);
  }
};
var ResourceVariables = class extends BaseResource {
  constructor(resourceType, options) {
    super({ prefixUrl: resourceType, ...options });
  }
  all(resourceId, options) {
    return RequestHelper.get()(this, endpoint`${resourceId}/variables`, options);
  }
  create(resourceId, key, value, options) {
    return RequestHelper.post()(this, endpoint`${resourceId}/variables`, {
      key,
      value,
      ...options
    });
  }
  edit(resourceId, key, value, options) {
    return RequestHelper.put()(this, endpoint`${resourceId}/variables/${key}`, {
      value,
      ...options
    });
  }
  show(resourceId, key, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/variables/${key}`,
      options
    );
  }
  remove(resourceId, key, options) {
    return RequestHelper.del()(this, endpoint`${resourceId}/variables/${key}`, options);
  }
};
var ResourceWikis = class extends BaseResource {
  constructor(resourceType, options) {
    super({ prefixUrl: resourceType, ...options });
  }
  all(resourceId, options) {
    return RequestHelper.get()(this, endpoint`${resourceId}/wikis`, options);
  }
  create(resourceId, content, title, options) {
    return RequestHelper.post()(this, endpoint`${resourceId}/wikis`, {
      content,
      title,
      ...options
    });
  }
  edit(resourceId, slug, options) {
    return RequestHelper.put()(this, endpoint`${resourceId}/wikis/${slug}`, options);
  }
  remove(resourceId, slug, options) {
    return RequestHelper.del()(this, endpoint`${resourceId}/wikis/${slug}`, options);
  }
  show(resourceId, slug, options) {
    return RequestHelper.get()(this, endpoint`${resourceId}/wikis/${slug}`, options);
  }
  uploadAttachment(resourceId, file, options) {
    return RequestHelper.post()(
      this,
      endpoint`${resourceId}/wikis/attachments`,
      {
        ...options,
        isForm: true,
        file: [file.content, file.filename]
      }
    );
  }
};
var ResourceHooks = class extends BaseResource {
  constructor(resourceType, options) {
    super({ prefixUrl: resourceType, ...options });
  }
  add(resourceId, url12, options) {
    return RequestHelper.post()(this, endpoint`${resourceId}/hooks`, {
      url: url12,
      ...options
    });
  }
  all(resourceId, options) {
    return RequestHelper.get()(this, endpoint`${resourceId}/hooks`, options);
  }
  edit(resourceId, hookId, url12, options) {
    return RequestHelper.put()(this, endpoint`${resourceId}/hooks/${hookId}`, {
      url: url12,
      ...options
    });
  }
  remove(resourceId, hookId, options) {
    return RequestHelper.del()(this, endpoint`${resourceId}/hooks/${hookId}`, options);
  }
  show(resourceId, hookId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/hooks/${hookId}`,
      options
    );
  }
};
var ResourcePushRules = class extends BaseResource {
  constructor(resourceType, options) {
    super({ prefixUrl: resourceType, ...options });
  }
  create(resourceId, options) {
    return RequestHelper.post()(this, endpoint`${resourceId}/push_rule`, options);
  }
  edit(resourceId, options) {
    return RequestHelper.put()(this, endpoint`${resourceId}/push_rule`, options);
  }
  remove(resourceId, options) {
    return RequestHelper.del()(this, endpoint`${resourceId}/push_rule`, options);
  }
  show(resourceId, options) {
    return RequestHelper.get()(this, endpoint`${resourceId}/push_rule`, options);
  }
};
var ResourceRepositoryStorageMoves = class extends BaseResource {
  resourceType;
  resourceTypeSingular;
  constructor(resourceType, options) {
    super(options);
    this.resourceType = resourceType;
    this.resourceTypeSingular = resourceType.substring(0, resourceType.length - 1);
  }
  all(options) {
    const resourceId = options?.[`${this.resourceTypeSingular}Id`];
    const url12 = resourceId ? endpoint`${this.resourceType}/${resourceId}/repository_storage_moves` : `${this.resourceTypeSingular}_repository_storage_moves`;
    return RequestHelper.get()(this, url12, options);
  }
  show(repositoryStorageId, options) {
    const resourceId = options?.[`${this.resourceTypeSingular}Id`];
    const url12 = resourceId ? endpoint`${this.resourceType}/${resourceId}/repository_storage_moves` : `${this.resourceTypeSingular}_repository_storage_moves`;
    return RequestHelper.get()(
      this,
      `${url12}/${repositoryStorageId}`,
      options
    );
  }
  schedule(sourceStorageName, options) {
    const resourceId = options?.[`${this.resourceTypeSingular}Id`];
    const url12 = resourceId ? endpoint`${this.resourceType}/${resourceId}/repository_storage_moves` : `${this.resourceTypeSingular}_repository_storage_moves`;
    return RequestHelper.post()(this, url12, {
      sourceStorageName,
      ...options
    });
  }
};
var ResourceInvitations = class extends BaseResource {
  constructor(resourceType, options) {
    super({ prefixUrl: resourceType, ...options });
  }
  add(resourceId, accessLevel, options) {
    if (!options?.email && !options?.userId)
      throw new Error(
        "Missing required argument. Please supply a email or a userId in the options parameter."
      );
    return RequestHelper.post()(this, endpoint`${resourceId}/invitations`, {
      accessLevel,
      ...options
    });
  }
  all(resourceId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/invitations`,
      options
    );
  }
  edit(resourceId, email, options) {
    return RequestHelper.put()(
      this,
      endpoint`${resourceId}/invitations/${email}`,
      options
    );
  }
  remove(resourceId, email, options) {
    return RequestHelper.del()(
      this,
      endpoint`${resourceId}/invitations/${email}`,
      options
    );
  }
};
var ResourceIterations = class extends BaseResource {
  constructor(resourceType, options) {
    super({ prefixUrl: resourceType, ...options });
  }
  all(resourceId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/iterations`,
      options
    );
  }
};
var ResourceProtectedEnvironments = class extends BaseResource {
  constructor(resourceType, options) {
    super({ prefixUrl: resourceType, ...options });
  }
  all(resourceId, options) {
    return RequestHelper.get()(
      this,
      `${resourceId}/protected_environments`,
      options
    );
  }
  create(resourceId, name, deployAccessLevels, options) {
    return RequestHelper.post()(
      this,
      `${resourceId}/protected_environments`,
      {
        name,
        deployAccessLevels,
        ...options
      }
    );
  }
  edit(resourceId, name, options) {
    return RequestHelper.put()(
      this,
      `${resourceId}/protected_environments/${name}`,
      options
    );
  }
  show(resourceId, name, options) {
    return RequestHelper.get()(
      this,
      `${resourceId}/protected_environments/${name}`,
      options
    );
  }
  remove(resourceId, name, options) {
    return RequestHelper.del()(this, `${resourceId}/protected_environments/${name}`, options);
  }
};
var ResourceIterationEvents = class extends BaseResource {
  resource2Type;
  constructor(resourceType, resource2Type, options) {
    super({ prefixUrl: resourceType, ...options });
    this.resource2Type = resource2Type;
  }
  all(resourceId, resource2Id, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/resource_iteration_events`,
      options
    );
  }
  show(resourceId, resource2Id, iterationEventId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/resource_iteration_events/${iterationEventId}`,
      options
    );
  }
};
var ResourceLabelEvents = class extends BaseResource {
  resource2Type;
  constructor(resourceType, resource2Type, options) {
    super({ prefixUrl: resourceType, ...options });
    this.resource2Type = resource2Type;
  }
  all(resourceId, resource2Id, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/resource_label_events`,
      options
    );
  }
  show(resourceId, resource2Id, labelEventId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/resource_label_events/${labelEventId}`,
      options
    );
  }
};
var ResourceMilestoneEvents = class extends BaseResource {
  resource2Type;
  constructor(resourceType, resource2Type, options) {
    super({ prefixUrl: resourceType, ...options });
    this.resource2Type = resource2Type;
  }
  all(resourceId, resource2Id, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/resource_milestone_events`,
      options
    );
  }
  show(resourceId, resource2Id, milestoneEventId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/resource_milestone_events/${milestoneEventId}`,
      options
    );
  }
};
var ResourceStateEvents = class extends BaseResource {
  resource2Type;
  constructor(resourceType, resource2Type, options) {
    super({ prefixUrl: resourceType, ...options });
    this.resource2Type = resource2Type;
  }
  all(resourceId, resource2Id, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/resource_state_events`,
      options
    );
  }
  show(resourceId, resource2Id, stateEventId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${resourceId}/${this.resource2Type}/${resource2Id}/resource_state_events/${stateEventId}`,
      options
    );
  }
};

// src/resources/DockerfileTemplates.ts
var DockerfileTemplates = class extends ResourceTemplates {
  constructor(options) {
    super("dockerfiles", options);
  }
};
var Events = class extends BaseResource {
  all({
    projectId,
    userId,
    ...options
  } = {}) {
    let url12;
    if (projectId) url12 = endpoint`projects/${projectId}/events`;
    else if (userId) url12 = endpoint`users/${userId}/events`;
    else url12 = "events";
    return RequestHelper.get()(this, url12, options);
  }
};
var Experiments = class extends BaseResource {
  all(options) {
    return RequestHelper.get()(this, "experiments", options);
  }
};
var GeoNodes = class extends BaseResource {
  all(options) {
    return RequestHelper.get()(this, "geo_nodes", options);
  }
  allStatuses(options) {
    return RequestHelper.get()(this, "geo_nodes/statuses", options);
  }
  allFailures(options) {
    return RequestHelper.get()(this, "geo_nodes/current/failures", options);
  }
  create(name, url12, options) {
    return RequestHelper.post()(this, "geo_nodes", { name, url: url12, ...options });
  }
  edit(geonodeId, options) {
    return RequestHelper.put()(this, `geo_nodes/${geonodeId}`, options);
  }
  repair(geonodeId, options) {
    return RequestHelper.post()(this, `geo_nodes/${geonodeId}/repair`, options);
  }
  remove(geonodeId, options) {
    return RequestHelper.del()(this, `geo_nodes/${geonodeId}`, options);
  }
  show(geonodeId, options) {
    return RequestHelper.get()(this, `geo_nodes/${geonodeId}`, options);
  }
  showStatus(geonodeId, options) {
    return RequestHelper.get()(this, `geo_nodes/${geonodeId}/status`, options);
  }
};
var GeoSites = class extends BaseResource {
  all(options) {
    return RequestHelper.get()(this, "geo_sites", options);
  }
  allStatuses(options) {
    return RequestHelper.get()(this, "geo_sites/statuses", options);
  }
  allFailures(options) {
    return RequestHelper.get()(this, "geo_sites/current/failures", options);
  }
  create(name, url12, options) {
    return RequestHelper.post()(this, "geo_sites", { name, url: url12, ...options });
  }
  edit(geositeId, options) {
    return RequestHelper.put()(this, `geo_sites/${geositeId}`, options);
  }
  repair(geositeId, options) {
    return RequestHelper.post()(this, `geo_sites/${geositeId}/repair`, options);
  }
  remove(geositeId, options) {
    return RequestHelper.del()(this, `geo_sites/${geositeId}`, options);
  }
  show(geositeId, options) {
    return RequestHelper.get()(this, `geo_sites/${geositeId}`, options);
  }
  showStatus(geositeId, options) {
    return RequestHelper.get()(this, `geo_sites/${geositeId}/status`, options);
  }
};

// src/resources/GitLabCIYMLTemplates.ts
var GitLabCIYMLTemplates = class extends ResourceTemplates {
  constructor(options) {
    super("gitlab_ci_ymls", options);
  }
};

// src/resources/GitignoreTemplates.ts
var GitignoreTemplates = class extends ResourceTemplates {
  constructor(options) {
    super("gitignores", options);
  }
};
var Import = class extends BaseResource {
  importGithubRepository(personalAccessToken, repositoryId, targetNamespace, options) {
    return RequestHelper.post()(this, "import/github", {
      personalAccessToken,
      repoId: repositoryId,
      targetNamespace,
      ...options
    });
  }
  cancelGithubRepositoryImport(projectId, options) {
    return RequestHelper.post()(this, "import/github/cancel", {
      projectId,
      ...options
    });
  }
  importGithubGists(personalAccessToken, options) {
    return RequestHelper.post()(this, "import/github/gists", {
      personalAccessToken,
      ...options
    });
  }
  importBitbucketServerRepository(bitbucketServerUrl, bitbucketServerUsername, personalAccessToken, bitbucketServerProject, bitbucketServerRepository, options) {
    return RequestHelper.post()(this, "import/bitbucket_server", {
      bitbucketServerUrl,
      bitbucketServerUsername,
      personalAccessToken,
      bitbucketServerProject,
      bitbucketServerRepo: bitbucketServerRepository,
      ...options
    });
  }
};
var InstanceLevelCICDVariables = class extends BaseResource {
  all(options) {
    return RequestHelper.get()(this, "admin/ci/variables", options);
  }
  create(key, value, options) {
    return RequestHelper.post()(this, "admin/ci/variables", {
      key,
      value,
      ...options
    });
  }
  edit(keyId, value, options) {
    return RequestHelper.put()(this, endpoint`admin/ci/variables/${keyId}`, {
      value,
      ...options
    });
  }
  show(keyId, options) {
    return RequestHelper.get()(
      this,
      endpoint`admin/ci/variables/${keyId}`,
      options
    );
  }
  remove(keyId, options) {
    return RequestHelper.get()(this, endpoint`admin/ci/variables/${keyId}`, options);
  }
};
var Keys = class extends BaseResource {
  show({
    keyId,
    fingerprint,
    ...options
  } = {}) {
    let url12;
    if (keyId) url12 = `keys/${keyId}`;
    else if (fingerprint) url12 = `keys?fingerprint=${fingerprint}`;
    else {
      throw new Error(
        "Missing required argument. Please supply a fingerprint or a keyId in the options parameter"
      );
    }
    return RequestHelper.get()(this, url12, options);
  }
};
var License = class extends BaseResource {
  add(license, options) {
    return RequestHelper.post()(this, "license", {
      searchParams: { license },
      ...options
    });
  }
  all(options) {
    return RequestHelper.get()(this, "licenses", options);
  }
  show(options) {
    return RequestHelper.get()(this, "license", options);
  }
  remove(licenceId, options) {
    return RequestHelper.del()(this, `license/${licenceId}`, options);
  }
  recalculateBillableUsers(licenceId, options) {
    return RequestHelper.put()(
      this,
      `license/${licenceId}/refresh_billable_users`,
      options
    );
  }
};

// src/resources/LicenseTemplates.ts
var LicenseTemplates = class extends ResourceTemplates {
  constructor(options) {
    super("Licenses", options);
  }
};
var Lint = class extends BaseResource {
  check(projectId, options) {
    return RequestHelper.get()(this, endpoint`projects/${projectId}/ci/lint`, options);
  }
  lint(projectId, content, options) {
    return RequestHelper.post()(this, endpoint`projects/${projectId}/ci/lint`, {
      ...options,
      content
    });
  }
};
var Markdown = class extends BaseResource {
  render(text, options) {
    return RequestHelper.post()(this, "markdown", { text, ...options });
  }
};
var Maven = class extends BaseResource {
  downloadPackageFile(path, filename, {
    projectId,
    groupId,
    ...options
  }) {
    let url12 = endpoint`packages/maven/${path}/${filename}`;
    if (projectId) url12 = endpoint`projects/${projectId}/${url12}`;
    else if (groupId) url12 = endpoint`groups/${groupId}/-/${url12}`;
    return RequestHelper.get()(this, url12, options);
  }
  uploadPackageFile(projectId, path, packageFile, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/packages/maven/${path}/${packageFile.filename}`,
      {
        isForm: true,
        ...options,
        file: [packageFile.content, packageFile.filename]
      }
    );
  }
};
var Metadata = class extends BaseResource {
  show(options) {
    return RequestHelper.get()(this, "metadata", options);
  }
};
var Migrations = class extends BaseResource {
  all(options) {
    return RequestHelper.get()(this, "bulk_imports", options);
  }
  create(configuration, entities, options) {
    return RequestHelper.post()(this, "bulk_imports", {
      configuration,
      entities,
      ...options
    });
  }
  allEntities({
    bulkImportId,
    ...options
  } = {}) {
    const url12 = bulkImportId ? endpoint`bulk_imports/${bulkImportId}/entities` : "bulk_imports/entities";
    return RequestHelper.get()(this, url12, options);
  }
  show(bulkImportId, options) {
    return RequestHelper.get()(
      this,
      `bulk_imports/${bulkImportId}`,
      options
    );
  }
  showEntity(bulkImportId, entitityId, options) {
    return RequestHelper.get()(
      this,
      `bulk_imports/${bulkImportId}/entities/${entitityId}`,
      options
    );
  }
};
function url6(projectId) {
  return projectId ? endpoint`/projects/${projectId}/packages/npm` : "packages/npm";
}
var NPM = class extends BaseResource {
  downloadPackageFile(projectId, packageName, filename, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/packages/npm/${packageName}/-/${filename}`,
      options
    );
  }
  removeDistTag(packageName, tag, options) {
    const prefix = url6(options?.projectId);
    return RequestHelper.del()(
      this,
      `${prefix}/-/package/${packageName}/dist-tags/${tag}`,
      options
    );
  }
  setDistTag(packageName, tag, options) {
    const prefix = url6(options?.projectId);
    return RequestHelper.put()(
      this,
      `${prefix}/-/package/${packageName}/dist-tags/${tag}`,
      options
    );
  }
  showDistTags(packageName, options) {
    const prefix = url6(options?.projectId);
    return RequestHelper.get()(
      this,
      `${prefix}/-/package/${packageName}/dist-tags`,
      options
    );
  }
  showMetadata(packageName, options) {
    const prefix = url6(options?.projectId);
    return RequestHelper.get()(this, `${prefix}/${packageName}`, options);
  }
  uploadPackageFile(projectId, packageName, versions, metadata, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/packages/npm/${packageName}`,
      {
        ...options,
        versions,
        ...metadata
      }
    );
  }
};
var Namespaces = class extends BaseResource {
  all(options) {
    return RequestHelper.get()(this, "namespaces", options);
  }
  exists(namespace, options) {
    return RequestHelper.get()(
      this,
      endpoint`namespaces/${namespace}/exists`,
      options
    );
  }
  show(namespaceId, options) {
    return RequestHelper.get()(this, endpoint`namespaces/${namespaceId}`, options);
  }
};
function url7({
  projectId,
  groupId
} = {}) {
  let prefix = "";
  if (projectId) prefix = endpoint`projects/${projectId}/`;
  if (groupId) prefix = endpoint`groups/${groupId}/`;
  return `${prefix}notification_settings`;
}
var NotificationSettings = class extends BaseResource {
  edit({
    groupId,
    projectId,
    ...options
  } = {}) {
    const uri = url7({ groupId, projectId });
    return RequestHelper.put()(this, uri, options);
  }
  show({
    groupId,
    projectId,
    ...options
  } = {}) {
    const uri = url7({ groupId, projectId });
    return RequestHelper.get()(this, uri, options);
  }
};
function url8({
  projectId,
  groupId
} = {}) {
  if (projectId) return endpoint`/projects/${projectId}/packages/nuget`;
  if (groupId) return endpoint`/groups/${groupId}/-/packages/nuget`;
  throw new Error(
    "Missing required argument. Please supply a projectId or a groupId in the options parameter"
  );
}
var NuGet = class extends BaseResource {
  downloadPackageFile(projectId, packageName, packageVersion, filename, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/packages/nuget/download/${packageName}/${packageVersion}/${filename}`,
      options
    );
  }
  search(q, {
    projectId,
    groupId,
    ...options
  }) {
    const uri = url8({ projectId, groupId });
    return RequestHelper.get()(this, `${uri}/query`, { q, ...options });
  }
  showMetadata(packageName, {
    projectId,
    groupId,
    ...options
  }) {
    const uri = url8({ projectId, groupId });
    return RequestHelper.get()(
      this,
      `${uri}/metadata/${packageName}/index`,
      options
    );
  }
  showPackageIndex(projectId, packageName, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/packages/nuget/download/${packageName}/index`,
      options
    );
  }
  showServiceIndex({
    projectId,
    groupId,
    ...options
  }) {
    const uri = url8({ projectId, groupId });
    return RequestHelper.get()(
      this,
      `${uri}/index`,
      options
    );
  }
  showVersionMetadata(packageName, packageVersion, {
    projectId,
    groupId,
    ...options
  }) {
    const uri = url8({ projectId, groupId });
    return RequestHelper.get()(
      this,
      `${uri}/metadata/${packageName}/${packageVersion}`,
      options
    );
  }
  uploadPackageFile(projectId, packageName, packageVersion, packageFile, options) {
    return RequestHelper.put()(this, endpoint`projects/${projectId}/packages/nuget`, {
      isForm: true,
      ...options,
      packageName,
      packageVersion,
      file: [packageFile.content, packageFile.filename]
    });
  }
  uploadSymbolPackage(projectId, packageName, packageVersion, packageFile, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/packages/nuget/symbolpackage`,
      {
        isForm: true,
        ...options,
        packageName,
        packageVersion,
        file: [packageFile.content, packageFile.filename]
      }
    );
  }
};
var PersonalAccessTokens = class extends BaseResource {
  all(options) {
    return RequestHelper.get()(
      this,
      "personal_access_tokens",
      options
    );
  }
  // Convience method - Also located in Users
  create(userId, name, scopes, options) {
    return RequestHelper.post()(
      this,
      endpoint`users/${userId}/personal_access_tokens`,
      {
        name,
        scopes,
        ...options
      }
    );
  }
  remove({
    tokenId,
    ...options
  } = {}) {
    const url12 = tokenId ? endpoint`personal_access_tokens/${tokenId}` : "personal_access_tokens/self";
    return RequestHelper.del()(this, url12, options);
  }
  rotate(tokenId, options) {
    return RequestHelper.post()(
      this,
      endpoint`personal_access_tokens/${tokenId}/rotate`,
      options
    );
  }
  show({
    tokenId,
    ...options
  } = {}) {
    const url12 = tokenId ? endpoint`personal_access_tokens/${tokenId}` : "personal_access_tokens/self";
    return RequestHelper.get()(this, url12, options);
  }
};
var PyPI = class extends BaseResource {
  downloadPackageFile(sha, fileIdentifier, {
    projectId,
    groupId,
    ...options
  } = {}) {
    let url12;
    if (projectId) {
      url12 = endpoint`projects/${projectId}/packages/pypi/files/${sha}/${fileIdentifier}`;
    } else if (groupId) {
      url12 = endpoint`groups/${groupId}/packages/pypi/files/${sha}/${fileIdentifier}`;
    } else {
      throw new Error(
        "Missing required argument. Please supply a projectId or a groupId in the options parameter"
      );
    }
    return RequestHelper.get()(this, url12, options);
  }
  showPackageDescriptor(packageName, {
    projectId,
    groupId,
    ...options
  }) {
    let url12;
    if (projectId) {
      url12 = endpoint`projects/${projectId}/packages/pypi/simple/${packageName}`;
    } else if (groupId) {
      url12 = endpoint`groups/${groupId}/packages/pypi/simple/${packageName}`;
    } else {
      throw new Error(
        "Missing required argument. Please supply a projectId or a groupId in the options parameter"
      );
    }
    return RequestHelper.get()(this, url12, options);
  }
  uploadPackageFile(projectId, packageFile, options) {
    return RequestHelper.put()(this, endpoint`projects/${projectId}/packages/pypi`, {
      ...options,
      isForm: true,
      file: [packageFile.content, packageFile.filename]
    });
  }
};
var RubyGems = class extends BaseResource {
  allDependencies(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/packages/rubygems/api/v1/dependencies`,
      options
    );
  }
  downloadGemFile(projectId, fileName, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/packages/rubygems/gems/${fileName}`,
      options
    );
  }
  uploadGemFile(projectId, packageFile, options) {
    return RequestHelper.post()(this, `projects/${projectId}/packages/rubygems/api/v1/gems`, {
      isForm: true,
      ...options,
      file: [packageFile.content, packageFile.filename]
    });
  }
};
var Search = class extends BaseResource {
  all(scope, search, options) {
    const { projectId, groupId, ...opts } = options || {};
    let url12;
    if (projectId) url12 = endpoint`projects/${projectId}/`;
    else if (groupId) url12 = endpoint`groups/${groupId}/`;
    else url12 = "";
    return RequestHelper.get()(this, `${url12}search`, {
      scope,
      search,
      ...opts
    });
  }
};
var SearchAdmin = class extends BaseResource {
  all(options) {
    return RequestHelper.get()(this, "admin/search/migrations", options);
  }
  show(versionOrName, options) {
    return RequestHelper.get()(
      this,
      endpoint`admin/search/migrations/${versionOrName}`,
      options
    );
  }
};
var ServiceAccounts = class extends BaseResource {
  create(options) {
    return RequestHelper.post()(this, endpoint`service_accounts`, options);
  }
};
var ServiceData = class extends BaseResource {
  showMetricDefinitions(options) {
    return RequestHelper.get()(this, "usage_data/metric_definitions", options);
  }
  showServicePingSQLQueries(options) {
    return RequestHelper.get()(this, "usage_data/queries", options);
  }
  showUsageDataNonSQLMetrics(options) {
    return RequestHelper.get()(
      this,
      "usage_data/non_sql_metrics",
      options
    );
  }
};
var SidekiqMetrics = class extends BaseResource {
  queueMetrics() {
    return RequestHelper.get()(this, "sidekiq/queue_metrics");
  }
  processMetrics() {
    return RequestHelper.get()(this, "sidekiq/process_metrics");
  }
  jobStats() {
    return RequestHelper.get()(this, "sidekiq/job_stats");
  }
  compoundMetrics() {
    return RequestHelper.get()(this, "sidekiq/compound_metrics");
  }
};
var SidekiqQueues = class extends BaseResource {
  remove(queueName, options) {
    return RequestHelper.get()(
      this,
      endpoint`admin/sidekiq/queues/${queueName}`,
      options
    );
  }
};

// src/resources/SnippetRepositoryStorageMoves.ts
var SnippetRepositoryStorageMoves = class extends ResourceRepositoryStorageMoves {
  constructor(options) {
    super("snippets", options);
  }
};
var Snippets = class extends BaseResource {
  all({
    public: ppublic,
    ...options
  } = {}) {
    const url12 = ppublic ? "snippets/public" : "snippets";
    return RequestHelper.get()(this, url12, options);
  }
  create(title, options) {
    return RequestHelper.post()(this, "snippets", {
      title,
      ...options
    });
  }
  edit(snippetId, options) {
    return RequestHelper.put()(this, `snippets/${snippetId}`, options);
  }
  remove(snippetId, options) {
    return RequestHelper.del()(this, `snippets/${snippetId}`, options);
  }
  show(snippetId, options) {
    return RequestHelper.get()(this, `snippets/${snippetId}`, options);
  }
  showContent(snippetId, options) {
    return RequestHelper.get()(this, `snippets/${snippetId}/raw`, options);
  }
  showRepositoryFileContent(snippetId, ref, filePath, options) {
    return RequestHelper.get()(
      this,
      endpoint`snippets/${snippetId}/files/${ref}/${filePath}/raw`,
      options
    );
  }
  showUserAgentDetails(snippetId, options) {
    return RequestHelper.get()(
      this,
      `snippets/${snippetId}/user_agent_detail`,
      options
    );
  }
};
var Suggestions = class extends BaseResource {
  edit(suggestionId, options) {
    return RequestHelper.put()(
      this,
      `suggestions/${suggestionId}/apply`,
      options
    );
  }
  editBatch(suggestionIds, options) {
    return RequestHelper.put()(this, `suggestions/batch_apply`, {
      ...options,
      ids: suggestionIds
    });
  }
};
var SystemHooks = class extends BaseResource {
  all(options) {
    return RequestHelper.get()(this, "hooks", options);
  }
  // Convenience method
  add(url12, options) {
    return this.create(url12, options);
  }
  create(url12, options) {
    return RequestHelper.post()(this, "hooks", {
      url: url12,
      ...options
    });
  }
  test(hookId, options) {
    return RequestHelper.post()(this, `hooks/${hookId}`, options);
  }
  remove(hookId, options) {
    return RequestHelper.del()(this, `hooks/${hookId}`, options);
  }
  show(hookId, options) {
    return RequestHelper.post()(this, `hooks/${hookId}`, options);
  }
};
var TodoLists = class extends BaseResource {
  all(options) {
    return RequestHelper.get()(this, "todos", options);
  }
  done({
    todoId,
    ...options
  } = {}) {
    let prefix = "todos";
    if (todoId) prefix += `/${todoId}`;
    return RequestHelper.post()(
      this,
      `${prefix}/mark_as_done`,
      options
    );
  }
};
var Topics = class extends BaseResource {
  all(options) {
    return RequestHelper.get()(this, "topics", options);
  }
  create(name, {
    avatar,
    ...options
  } = {}) {
    const opts = {
      name,
      ...options
    };
    if (avatar) {
      opts.isForm = true;
      opts.file = [avatar.content, avatar.filename];
    }
    return RequestHelper.post()(this, "topics", opts);
  }
  edit(topicId, {
    avatar,
    ...options
  } = {}) {
    const opts = { ...options };
    if (avatar) {
      opts.isForm = true;
      opts.file = [avatar.content, avatar.filename];
    }
    return RequestHelper.put()(this, `topics/${topicId}`, opts);
  }
  merge(sourceTopicId, targetTopicId, options) {
    return RequestHelper.post()(this, `topics/merge`, {
      sourceTopicId,
      targetTopicId,
      ...options
    });
  }
  remove(topicId, options) {
    return RequestHelper.del()(this, `topics/${topicId}`, options);
  }
  show(topicId, options) {
    return RequestHelper.get()(this, `topics/${topicId}`, options);
  }
};
var Branches = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/branches`,
      options
    );
  }
  create(projectId, branchName, ref, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/repository/branches`,
      {
        branch: branchName,
        ref,
        ...options
      }
    );
  }
  remove(projectId, branchName, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/repository/branches/${branchName}`,
      options
    );
  }
  removeMerged(projectId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/repository/merged_branches`,
      options
    );
  }
  show(projectId, branchName, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/branches/${branchName}`,
      options
    );
  }
};

// src/resources/CommitDiscussions.ts
var CommitDiscussions = class extends ResourceDiscussions {
  constructor(options) {
    super("projects", new RawPathSegment("repository/commits"), options);
  }
};
var Commits = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/commits`,
      options
    );
  }
  allComments(projectId, sha, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/commits/${sha}/comments`,
      options
    );
  }
  allDiscussions(projectId, sha, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/commits/${sha}/discussions`,
      options
    );
  }
  allMergeRequests(projectId, sha, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/commits/${sha}/merge_requests`,
      options
    );
  }
  allReferences(projectId, sha, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/commits/${sha}/refs`,
      options
    );
  }
  allStatuses(projectId, sha, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/commits/${sha}/statuses`,
      options
    );
  }
  cherryPick(projectId, sha, branch, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/repository/commits/${sha}/cherry_pick`,
      {
        branch,
        ...options
      }
    );
  }
  create(projectId, branch, message, actions = [], options = {}) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/repository/commits`,
      {
        branch,
        commitMessage: message,
        actions,
        ...options
      }
    );
  }
  createComment(projectId, sha, note, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/repository/commits/${sha}/comments`,
      {
        note,
        ...options
      }
    );
  }
  editStatus(projectId, sha, state, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/statuses/${sha}`,
      {
        state,
        ...options
      }
    );
  }
  revert(projectId, sha, branch, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/repository/commits/${sha}/revert`,
      {
        ...options,
        branch
      }
    );
  }
  show(projectId, sha, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/commits/${sha}`,
      options
    );
  }
  showDiff(projectId, sha, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/commits/${sha}/diff`,
      options
    );
  }
  showGPGSignature(projectId, sha, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/commits/${sha}/signature`,
      options
    );
  }
  showSequence(projectId, sha, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/commits/${sha}/sequence`,
      options
    );
  }
};
var ContainerRegistry = class extends BaseResource {
  allRepositories({
    groupId,
    projectId,
    ...options
  } = {}) {
    let url12;
    if (groupId) url12 = endpoint`groups/${groupId}/registry/repositories`;
    else if (projectId) url12 = endpoint`projects/${projectId}/registry/repositories`;
    else
      throw new Error(
        "Missing required argument. Please supply a groupId or a projectId in the options parameter."
      );
    return RequestHelper.get()(this, url12, options);
  }
  allTags(projectId, repositoryId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/registry/repositories/${repositoryId}/tags`,
      options
    );
  }
  editRegistryVisibility(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}`,
      options
    );
  }
  removeRepository(projectId, repositoryId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/registry/repositories/${repositoryId}`,
      options
    );
  }
  removeTag(projectId, repositoryId, tagName, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/registry/repositories/${repositoryId}/tags/${tagName}`,
      options
    );
  }
  removeTags(projectId, repositoryId, nameRegexDelete, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/registry/repositories/${repositoryId}/tags`,
      {
        nameRegexDelete,
        ...options
      }
    );
  }
  showRepository(repositoryId, options) {
    return RequestHelper.get()(
      this,
      endpoint`registry/repositories/${repositoryId}`,
      options
    );
  }
  showTag(projectId, repositoryId, tagName, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/registry/repositories/${repositoryId}/tags/${tagName}`,
      options
    );
  }
};
var Deployments = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/deployments`,
      options
    );
  }
  allMergeRequests(projectId, deploymentId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/deployments/${deploymentId}/merge_requests`,
      options
    );
  }
  create(projectId, environment, sha, ref, tag, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/deployments`,
      {
        environment,
        sha,
        ref,
        tag,
        ...options
      }
    );
  }
  edit(projectId, deploymentId, status, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/deployments/${deploymentId}`,
      {
        ...options,
        status
      }
    );
  }
  remove(projectId, deploymentId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/deployments/${deploymentId}`,
      options
    );
  }
  setApproval(projectId, deploymentId, status, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/deployments/${deploymentId}/approval`,
      {
        ...options,
        status
      }
    );
  }
  show(projectId, deploymentId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/deployments/${deploymentId}`,
      options
    );
  }
};
var Environments = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/environments`,
      options
    );
  }
  create(projectId, name, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/environments`,
      {
        name,
        ...options
      }
    );
  }
  edit(projectId, environmentId, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/environments/${environmentId}`,
      options
    );
  }
  remove(projectId, environmentId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/environments/${environmentId}`,
      options
    );
  }
  removeReviewApps(projectId, options) {
    return RequestHelper.del()(this, endpoint`projects/${projectId}/environments/review_apps`, options);
  }
  show(projectId, environmentId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/environments/${environmentId}`,
      options
    );
  }
  stop(projectId, environmentId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/environments/${environmentId}/stop`,
      options
    );
  }
  stopStale(projectId, before, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/environments/stop_stale`,
      {
        searchParams: { before },
        ...options
      }
    );
  }
};
var ErrorTrackingClientKeys = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/error_tracking/client_keys`,
      options
    );
  }
  create(projectId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/error_tracking/client_keys`,
      options
    );
  }
  remove(projectId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/error_tracking/client_keys`,
      options
    );
  }
};
var ErrorTrackingSettings = class extends BaseResource {
  create(projectId, active, integrated, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/error_tracking/settings`,
      {
        searchParams: {
          active,
          integrated
        },
        ...options
      }
    );
  }
  edit(projectId, active, { integrated, ...options } = {}) {
    return RequestHelper.patch()(
      this,
      endpoint`projects/${projectId}/error_tracking/settings`,
      {
        searchParams: {
          active,
          integrated
        },
        ...options
      }
    );
  }
  show(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/error_tracking/settings`,
      options
    );
  }
};
var ExternalStatusChecks = class extends BaseResource {
  all(projectId, options) {
    const { mergerequestIId, ...opts } = options || {};
    let url12 = endpoint`projects/${projectId}`;
    if (mergerequestIId) {
      url12 += endpoint`/merge_requests/${mergerequestIId}/status_checks`;
    } else {
      url12 += "/external_status_checks";
    }
    return RequestHelper.get()(this, url12, opts);
  }
  create(projectId, name, externalUrl, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/external_status_checks`,
      {
        name,
        externalUrl,
        ...options
      }
    );
  }
  edit(projectId, externalStatusCheckId, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/external_status_checks/${externalStatusCheckId}`,
      options
    );
  }
  remove(projectId, externalStatusCheckId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/external_status_checks/${externalStatusCheckId}`,
      options
    );
  }
  set(projectId, mergerequestIId, sha, externalStatusCheckId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/status_check_responses`,
      {
        sha,
        externalStatusCheckId,
        ...options
      }
    );
  }
};
var FeatureFlagUserLists = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/feature_flags_user_lists`,
      options
    );
  }
  create(projectId, name, userXids, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/feature_flags_user_lists`,
      {
        name,
        userXids,
        ...options
      }
    );
  }
  edit(projectId, featureFlagUserListIId, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/feature_flags_user_lists/${featureFlagUserListIId}`,
      options
    );
  }
  remove(projectId, featureFlagUserListIId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/feature_flags_user_lists/${featureFlagUserListIId}`,
      options
    );
  }
  show(projectId, featureFlagUserListIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/feature_flags_user_lists/${featureFlagUserListIId}`,
      options
    );
  }
};
var FeatureFlags = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/feature_flags`,
      options
    );
  }
  create(projectId, flagName, version, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/feature_flags`,
      {
        name: flagName,
        version,
        ...options
      }
    );
  }
  edit(projectId, featureFlagName, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/feature_flags/${featureFlagName}`,
      options
    );
  }
  remove(projectId, flagName, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/feature_flags/${flagName}`,
      options
    );
  }
  show(projectId, flagName, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/feature_flags/${flagName}`,
      options
    );
  }
};
var FreezePeriods = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/freeze_periods`,
      options
    );
  }
  create(projectId, freezeStart, freezeEnd, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/freeze_periods`,
      {
        freezeStart,
        freezeEnd,
        ...options
      }
    );
  }
  edit(projectId, freezePeriodId, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/freeze_periods/${freezePeriodId}`,
      options
    );
  }
  remove(projectId, freezePeriodId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/freeze_periods/${freezePeriodId}`,
      options
    );
  }
  show(projectId, freezePeriodId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/freeze_periods/${freezePeriodId}`,
      options
    );
  }
};
var GitlabPages = class extends BaseResource {
  remove(projectId, options) {
    return RequestHelper.del()(this, endpoint`projects/${projectId}/pages`, options);
  }
  showSettings(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/pages`,
      options
    );
  }
};
var GoProxy = class extends BaseResource {
  all(projectId, moduleName, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/packages/go/${moduleName}/@v/list`,
      options
    );
  }
  showVersionMetadata(projectId, moduleName, moduleVersion, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/packages/go/${moduleName}/@v/${moduleVersion}.info`,
      options
    );
  }
  downloadModuleFile(projectId, moduleName, moduleVersion, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/packages/go/${moduleName}/@v/${moduleVersion}.mod`,
      options
    );
  }
  downloadModuleSource(projectId, moduleName, moduleVersion, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/packages/go/${moduleName}/@v/${moduleVersion}.zip`,
      options
    );
  }
};
var Helm = class extends BaseResource {
  downloadChartIndex(projectId, channel, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/packages/helm/${channel}/index.yaml`,
      options
    );
  }
  downloadChart(projectId, channel, filename, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/packages/helm/${channel}/charts/${filename}.tgz`,
      options
    );
  }
  import(projectId, channel, chart, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/packages/helm/api/${channel}/charts`,
      {
        isForm: true,
        ...options,
        chart: [chart.content, chart.filename]
      }
    );
  }
};
var Integrations = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/integrations`,
      options
    );
  }
  edit(projectId, integrationName, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/integrations/${integrationName}`,
      options
    );
  }
  disable(projectId, integrationName, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/integrations/${integrationName}`,
      options
    );
  }
  show(projectId, integrationName, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/integrations/${integrationName}`,
      options
    );
  }
};

// src/resources/IssueAwardEmojis.ts
var IssueAwardEmojis = class extends ResourceAwardEmojis {
  constructor(options) {
    super("projects", "issues", options);
  }
};

// src/resources/IssueDiscussions.ts
var IssueDiscussions = class extends ResourceDiscussions {
  constructor(options) {
    super("projects", "issues", options);
  }
};

// src/resources/IssueIterationEvents.ts
var IssueIterationEvents = class extends ResourceIterationEvents {
  constructor(options) {
    super("projects", "issues", options);
  }
};

// src/resources/IssueLabelEvents.ts
var IssueLabelEvents = class extends ResourceLabelEvents {
  constructor(options) {
    super("projects", "issues", options);
  }
};
var IssueLinks = class extends BaseResource {
  all(projectId, issueIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/links`,
      options
    );
  }
  create(projectId, issueIId, targetProjectId, targetIssueIId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/links`,
      {
        targetProjectId,
        targetIssueIid: targetIssueIId,
        ...options
      }
    );
  }
  remove(projectId, issueIId, issueLinkId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/links/${issueLinkId}`,
      options
    );
  }
};

// src/resources/IssueMilestoneEvents.ts
var IssueMilestoneEvents = class extends ResourceMilestoneEvents {
  constructor(options) {
    super("projects", "issues", options);
  }
};

// src/resources/IssueNoteAwardEmojis.ts
var IssueNoteAwardEmojis = class extends ResourceNoteAwardEmojis {
  constructor(options) {
    super("issues", options);
  }
};

// src/resources/IssueNotes.ts
var IssueNotes = class extends ResourceNotes {
  constructor(options) {
    super("projects", "issues", options);
  }
};

// src/resources/IssueStateEvents.ts
var IssueStateEvents = class extends ResourceStateEvents {
  constructor(options) {
    super("projects", "issues", options);
  }
};

// src/resources/IssueWeightEvents.ts
var IssueWeightEvents = class extends ResourceStateEvents {
  constructor(options) {
    super("projects", "issues", options);
  }
};
var Issues = class extends BaseResource {
  addSpentTime(projectId, issueIId, duration, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/add_spent_time`,
      {
        duration,
        ...options
      }
    );
  }
  addTimeEstimate(projectId, issueIId, duration, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/time_estimate`,
      {
        duration,
        ...options
      }
    );
  }
  all({
    projectId,
    groupId,
    ...options
  } = {}) {
    let url12;
    if (projectId) url12 = endpoint`projects/${projectId}/issues`;
    else if (groupId) url12 = endpoint`groups/${groupId}/issues`;
    else url12 = "issues";
    return RequestHelper.get()(this, url12, options);
  }
  allMetricImages(projectId, issueIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/metric_images`,
      options
    );
  }
  allParticipants(projectId, issueIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/participants`,
      options
    );
  }
  allRelatedMergeRequests(projectId, issueIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/related_merge_requests`,
      options
    );
  }
  create(projectId, title, options) {
    return RequestHelper.post()(this, endpoint`projects/${projectId}/issues`, {
      ...options,
      title
    });
  }
  createTodo(projectId, issueIId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/todo`,
      options
    );
  }
  clone(projectId, issueIId, destinationProjectId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/clone`,
      {
        toProjectId: destinationProjectId,
        ...options
      }
    );
  }
  edit(projectId, issueIId, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}`,
      options
    );
  }
  editMetricImage(projectId, issueIId, imageId, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/metric_images/${imageId}`,
      options
    );
  }
  move(projectId, issueIId, destinationProjectId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/move`,
      {
        toProjectId: destinationProjectId,
        ...options
      }
    );
  }
  // Includes /promote already!
  promote(projectId, issueIId, body, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/notes`,
      {
        searchParams: {
          body: `${body} 
 /promote`
        },
        ...options
      }
    );
  }
  remove(projectId, issueIId, options) {
    return RequestHelper.del()(this, endpoint`projects/${projectId}/issues/${issueIId}`, options);
  }
  removeMetricImage(projectId, issueIId, imageId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/metric_images/${imageId}`,
      options
    );
  }
  reorder(projectId, issueIId, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/reorder`,
      options
    );
  }
  resetSpentTime(projectId, issueIId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/reset_spent_time`,
      options
    );
  }
  resetTimeEstimate(projectId, issueIId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/reset_time_estimate`,
      options
    );
  }
  show(issueId, { projectId, ...options } = {}) {
    const url12 = projectId ? endpoint`projects/${projectId}/issues/${issueId}` : `issues/${issueId}`;
    return RequestHelper.get()(this, url12, options);
  }
  subscribe(projectId, issueIId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/subscribe`,
      options
    );
  }
  allClosedByMergeRequestst(projectId, issueIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/closed_by`,
      options
    );
  }
  showTimeStats(projectId, issueIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/time_stats`,
      options
    );
  }
  unsubscribe(projectId, issueIId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/unsubscribe`,
      options
    );
  }
  uploadMetricImage(projectId, issueIId, metricImage, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/metric_images`,
      {
        isForm: true,
        ...options,
        file: [metricImage.content, metricImage.filename]
      }
    );
  }
  showUserAgentDetails(projectId, issueIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/issues/${issueIId}/user_agent_details`,
      options
    );
  }
};
var IssuesStatistics = class extends BaseResource {
  all({
    projectId,
    groupId,
    ...options
  } = {}) {
    let url12;
    if (projectId) url12 = endpoint`projects/${projectId}/issues_statistics`;
    else if (groupId) url12 = endpoint`groups/${groupId}/issues_statistics`;
    else url12 = "issues_statistics";
    return RequestHelper.get()(this, url12, options);
  }
};
function generateDownloadPathForJob(projectId, jobId, artifactPath) {
  let url12 = endpoint`projects/${projectId}/jobs/${jobId}/artifacts`;
  if (artifactPath) url12 += `/${artifactPath}`;
  return url12;
}
function generateDownloadPath(projectId, ref, artifactPath) {
  let url12 = endpoint`projects/${projectId}/jobs/artifacts/${ref}`;
  if (artifactPath) {
    url12 += endpoint`/raw/${artifactPath}`;
  } else {
    url12 += endpoint`/download`;
  }
  return url12;
}
var JobArtifacts = class extends BaseResource {
  downloadArchive(projectId, {
    jobId,
    artifactPath,
    ref,
    ...options
  } = {}) {
    let url12;
    if (jobId) url12 = generateDownloadPathForJob(projectId, jobId, artifactPath);
    else if (options?.job && ref) url12 = generateDownloadPath(projectId, ref, artifactPath);
    else
      throw new Error(
        "Missing one of the required parameters. See typing documentation for available arguments."
      );
    return RequestHelper.get()(this, url12, options);
  }
  keep(projectId, jobId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/jobs/${jobId}/artifacts/keep`,
      options
    );
  }
  remove(projectId, { jobId, ...options } = {}) {
    let url12;
    if (jobId) {
      url12 = endpoint`projects/${projectId}/jobs/${jobId}/artifacts`;
    } else {
      url12 = endpoint`projects/${projectId}/artifacts`;
    }
    return RequestHelper.del()(this, url12, options);
  }
};
var Jobs = class extends BaseResource {
  all(projectId, {
    pipelineId,
    ...options
  } = {}) {
    const url12 = pipelineId ? endpoint`projects/${projectId}/pipelines/${pipelineId}/jobs` : endpoint`projects/${projectId}/jobs`;
    return RequestHelper.get()(this, url12, options);
  }
  allPipelineBridges(projectId, pipelineId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/pipelines/${pipelineId}/bridges`,
      options
    );
  }
  cancel(projectId, jobId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/jobs/${jobId}/cancel`,
      options
    );
  }
  erase(projectId, jobId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/jobs/${jobId}/erase`,
      options
    );
  }
  play(projectId, jobId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/jobs/${jobId}/play`,
      options
    );
  }
  retry(projectId, jobId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/jobs/${jobId}/retry`,
      options
    );
  }
  show(projectId, jobId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/jobs/${jobId}`,
      options
    );
  }
  showConnectedJob(options) {
    if (!this.headers["job-token"]) throw new Error('Missing required header "job-token"');
    return RequestHelper.get()(this, "job", options);
  }
  showConnectedJobK8Agents(options) {
    if (!this.headers["job-token"]) throw new Error('Missing required header "job-token"');
    return RequestHelper.get()(this, "job/allowed_agents", options);
  }
  showLog(projectId, jobId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/jobs/${jobId}/trace`,
      options
    );
  }
};
var MergeRequestApprovals = class extends BaseResource {
  allApprovalRules(projectId, { mergerequestIId, ...options } = {}) {
    let url12;
    if (mergerequestIId) {
      url12 = endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/approval_rules`;
    } else {
      url12 = endpoint`projects/${projectId}/approval_rules`;
    }
    return RequestHelper.get()(this, url12, options);
  }
  approve(projectId, mergerequestIId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/approve`,
      options
    );
  }
  createApprovalRule(projectId, name, approvalsRequired, {
    mergerequestIId,
    ...options
  } = {}) {
    let url12;
    if (mergerequestIId) {
      url12 = endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/approval_rules`;
    } else {
      url12 = endpoint`projects/${projectId}/approval_rules`;
    }
    return RequestHelper.post()(this, url12, { name, approvalsRequired, ...options });
  }
  editApprovalRule(projectId, approvalRuleId, name, approvalsRequired, {
    mergerequestIId,
    ...options
  } = {}) {
    let url12;
    if (mergerequestIId) {
      url12 = endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/approval_rules/${approvalRuleId}`;
    } else {
      url12 = endpoint`projects/${projectId}/approval_rules/${approvalRuleId}`;
    }
    return RequestHelper.put()(this, url12, { name, approvalsRequired, ...options });
  }
  editConfiguration(projectId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/approvals`,
      options
    );
  }
  removeApprovalRule(projectId, approvalRuleId, {
    mergerequestIId,
    ...options
  } = {}) {
    let url12;
    if (mergerequestIId) {
      url12 = endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/approval_rules/${approvalRuleId}`;
    } else {
      url12 = endpoint`projects/${projectId}/approval_rules/${approvalRuleId}`;
    }
    return RequestHelper.del()(this, url12, options);
  }
  showApprovalRule(projectId, approvalRuleId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/approval_rules/${approvalRuleId}`,
      options
    );
  }
  showApprovalState(projectId, mergerequestIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/approval_state`,
      options
    );
  }
  showConfiguration(projectId, { mergerequestIId, ...options } = {}) {
    let url12;
    if (mergerequestIId) {
      url12 = endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/approvals`;
    } else {
      url12 = endpoint`projects/${projectId}/approvals`;
    }
    return RequestHelper.get()(this, url12, options);
  }
  unapprove(projectId, mergerequestIId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/unapprove`,
      options
    );
  }
};

// src/resources/MergeRequestAwardEmojis.ts
var MergeRequestAwardEmojis = class extends ResourceAwardEmojis {
  constructor(options) {
    super("projects", "merge_requests", options);
  }
};
var MergeRequestContextCommits = class extends BaseResource {
  all(projectId, mergerequestIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/context_commits`,
      options
    );
  }
  create(projectId, commits, { mergerequestIId, ...options } = {}) {
    const prefix = endpoint`projects/${projectId}/merge_requests`;
    const url12 = mergerequestIId ? `${prefix}/${mergerequestIId}/context_commits` : prefix;
    return RequestHelper.post()(this, url12, {
      commits,
      ...options
    });
  }
  remove(projectId, mergerequestIId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/context_commits`,
      options
    );
  }
};

// src/resources/MergeRequestDiscussions.ts
var MergeRequestDiscussions = class extends ResourceDiscussions {
  constructor(options) {
    super("projects", "merge_requests", options);
  }
  resolve(projectId, mergerequestId, discussionId, resolved, options) {
    return RequestHelper.put()(
      this,
      endpoint`${projectId}/merge_requests/${mergerequestId}/discussions/${discussionId}`,
      {
        searchParams: { resolved },
        ...options
      }
    );
  }
};
var MergeRequestDraftNotes = class extends BaseResource {
  all(projectId, mergerequestIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/draft_notes`,
      options
    );
  }
  create(projectId, mergerequestIId, note, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/draft_notes`,
      {
        ...options,
        note
      }
    );
  }
  edit(projectId, mergerequestIId, draftNoteId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/draft_notes/${draftNoteId}`,
      options
    );
  }
  publish(projectId, mergerequestIId, draftNoteId, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/draft_notes/${draftNoteId}/publish`,
      options
    );
  }
  publishBulk(projectId, mergerequestIId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/draft_notes/bulk_publish`,
      options
    );
  }
  remove(projectId, mergerequestIId, draftNoteId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/draft_notes/${draftNoteId}`,
      options
    );
  }
  show(projectId, mergerequestIId, draftNoteId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/draft_notes/${draftNoteId}`,
      options
    );
  }
};

// src/resources/MergeRequestLabelEvents.ts
var MergeRequestLabelEvents = class extends ResourceLabelEvents {
  constructor(options) {
    super("projects", "merge_requests", options);
  }
};

// src/resources/MergeRequestMilestoneEvents.ts
var MergeRequestMilestoneEvents = class extends ResourceMilestoneEvents {
  constructor(options) {
    super("projects", "merge_requests", options);
  }
};

// src/resources/MergeRequestNoteAwardEmojis.ts
var MergeRequestNoteAwardEmojis = class extends ResourceNoteAwardEmojis {
  constructor(options) {
    super("merge_requests", options);
  }
};

// src/resources/MergeRequestNotes.ts
var MergeRequestNotes = class extends ResourceNotes {
  constructor(options) {
    super("projects", "merge_requests", options);
  }
};
var MergeRequests = class extends BaseResource {
  // convenience method
  accept(projectId, mergerequestIId, options) {
    return this.merge(projectId, mergerequestIId, options);
  }
  addSpentTime(projectId, mergerequestIId, duration, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/add_spent_time`,
      {
        duration,
        ...options
      }
    );
  }
  all({
    projectId,
    groupId,
    ...options
  } = {}) {
    let prefix = "";
    if (projectId) {
      prefix = endpoint`projects/${projectId}/`;
    } else if (groupId) {
      prefix = endpoint`groups/${groupId}/`;
    }
    return RequestHelper.get()(this, `${prefix}merge_requests`, options);
  }
  allDiffs(projectId, mergerequestIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/diffs`,
      options
    );
  }
  allCommits(projectId, mergerequestIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/commits`,
      options
    );
  }
  allDiffVersions(projectId, mergerequestIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/versions`,
      options
    );
  }
  allIssuesClosed(projectId, mergerequestIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/closes_issues`,
      options
    );
  }
  allIssuesRelated(projectId, mergerequestIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/related_issues`,
      options
    );
  }
  allParticipants(projectId, mergerequestIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/participants`,
      options
    );
  }
  allPipelines(projectId, mergerequestIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/pipelines`,
      options
    );
  }
  cancelOnPipelineSuccess(projectId, mergerequestIId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/cancel_merge_when_pipeline_succeeds`,
      options
    );
  }
  create(projectId, sourceBranch, targetBranch, title, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/merge_requests`,
      {
        sourceBranch,
        targetBranch,
        title,
        ...options
      }
    );
  }
  createPipeline(projectId, mergerequestIId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/pipelines`,
      options
    );
  }
  createTodo(projectId, mergerequestIId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/todo`,
      options
    );
  }
  edit(projectId, mergerequestIId, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}`,
      options
    );
  }
  merge(projectId, mergerequestIId, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/merge`,
      options
    );
  }
  mergeToDefault(projectId, mergerequestIId, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/merge_ref`,
      options
    );
  }
  rebase(projectId, mergerequestIId, { skipCI, ...options } = {}) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/rebase`,
      {
        ...options,
        skipCi: skipCI
      }
    );
  }
  remove(projectId, mergerequestIId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}`,
      options
    );
  }
  resetSpentTime(projectId, mergerequestIId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/reset_spent_time`,
      options
    );
  }
  resetTimeEstimate(projectId, mergerequestIId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/reset_time_estimate`,
      options
    );
  }
  setTimeEstimate(projectId, mergerequestIId, duration, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/time_estimate`,
      {
        duration,
        ...options
      }
    );
  }
  show(projectId, mergerequestIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}`,
      options
    );
  }
  showChanges(projectId, mergerequestIId, options) {
    process.emitWarning(
      'This endpoint was deprecated in GitLab API 15.7 and will be removed in API v5. Please use the "allDiffs" function instead.',
      "DeprecationWarning"
    );
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/changes`,
      options
    );
  }
  showDiffVersion(projectId, mergerequestIId, versionId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/versions/${versionId}`,
      options
    );
  }
  showTimeStats(projectId, mergerequestIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/time_stats`,
      options
    );
  }
  subscribe(projectId, mergerequestIId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/subscribe`,
      options
    );
  }
  unsubscribe(projectId, mergerequestIId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/unsubscribe`,
      options
    );
  }
  showReviewers(projectId, mergerequestIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/merge_requests/${mergerequestIId}/reviewers`,
      options
    );
  }
};
var MergeTrains = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/merge_trains`,
      options
    );
  }
  showStatus(projectId, mergeRequestIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/merge_trains/merge_requests/${mergeRequestIId}`,
      options
    );
  }
  addMergeRequest(projectId, mergeRequestIId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/merge_trains/merge_requests/${mergeRequestIId}`,
      options
    );
  }
};
var PackageRegistry = class extends BaseResource {
  publish(projectId, packageName, packageVersion, packageFile, {
    contentType,
    ...options
  } = {}) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/packages/generic/${packageName}/${packageVersion}/${packageFile.filename}`,
      {
        isForm: true,
        file: [packageFile.content, packageFile.filename],
        ...options
      }
    );
  }
  download(projectId, packageName, packageVersion, filename, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/packages/generic/${packageName}/${packageVersion}/${filename}`,
      options
    );
  }
};
var Packages = class extends BaseResource {
  all({
    projectId,
    groupId,
    ...options
  } = {}) {
    let url12;
    if (projectId) url12 = endpoint`projects/${projectId}/packages`;
    else if (groupId) url12 = endpoint`groups/${groupId}/packages`;
    else {
      throw new Error(
        "Missing required argument. Please supply a projectId or a groupId in the options parameter."
      );
    }
    return RequestHelper.get()(this, url12, options);
  }
  allFiles(projectId, packageId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/packages/${packageId}/package_files`,
      options
    );
  }
  remove(projectId, packageId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/packages/${packageId}`,
      options
    );
  }
  removeFile(projectId, packageId, projectFileId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/packages/${packageId}/package_files/${projectFileId}`,
      options
    );
  }
  show(projectId, packageId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/packages/${packageId}`,
      options
    );
  }
};
var PagesDomains = class extends BaseResource {
  all({
    projectId,
    ...options
  } = {}) {
    const prefix = projectId ? endpoint`projects/${projectId}/` : "";
    return RequestHelper.get()(this, `${prefix}pages/domains`, options);
  }
  create(projectId, domain, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/pages/domains`,
      {
        domain,
        ...options
      }
    );
  }
  edit(projectId, domain, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/pages/domains/${domain}`,
      options
    );
  }
  show(projectId, domain, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/pages/domains/${domain}`,
      options
    );
  }
  remove(projectId, domain, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/pages/domains/${domain}`,
      options
    );
  }
};
var PipelineScheduleVariables = class extends BaseResource {
  all(projectId, pipelineScheduleId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/pipeline_schedules/${pipelineScheduleId}/variables`,
      options
    );
  }
  create(projectId, pipelineScheduleId, key, value, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/pipeline_schedules/${pipelineScheduleId}/variables`,
      {
        ...options,
        key,
        value
      }
    );
  }
  edit(projectId, pipelineScheduleId, key, value, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/pipeline_schedules/${pipelineScheduleId}/variables/${key}`,
      {
        ...options,
        value
      }
    );
  }
  remove(projectId, pipelineScheduleId, key, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/pipeline_schedules/${pipelineScheduleId}/variables/${key}`,
      options
    );
  }
};
var PipelineSchedules = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/pipeline_schedules`,
      options
    );
  }
  allTriggeredPipelines(projectId, pipelineScheduleId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/pipeline_schedules/${pipelineScheduleId}/pipelines`,
      options
    );
  }
  create(projectId, description, ref, cron, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/pipeline_schedules`,
      {
        description,
        ref,
        cron,
        ...options
      }
    );
  }
  edit(projectId, pipelineScheduleId, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/pipeline_schedules/${pipelineScheduleId}`,
      options
    );
  }
  remove(projectId, pipelineScheduleId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/pipeline_schedules/${pipelineScheduleId}`,
      options
    );
  }
  run(projectId, pipelineScheduleId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/pipeline_schedules/${pipelineScheduleId}/play`,
      options
    );
  }
  show(projectId, pipelineScheduleId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/pipeline_schedules/${pipelineScheduleId}`,
      options
    );
  }
  takeOwnership(projectId, pipelineScheduleId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/pipeline_schedules/${pipelineScheduleId}/take_ownership`,
      options
    );
  }
};
var PipelineTriggerTokens = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/triggers`,
      options
    );
  }
  create(projectId, description, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/triggers`,
      {
        description,
        ...options
      }
    );
  }
  edit(projectId, triggerId, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/triggers/${triggerId}`,
      options
    );
  }
  remove(projectId, triggerId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/triggers/${triggerId}`,
      options
    );
  }
  show(projectId, triggerId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/triggers/${triggerId}`,
      options
    );
  }
  trigger(projectId, ref, token, { variables, ...options } = {}) {
    const opts = {
      ...options,
      searchParams: {
        token,
        ref
      }
    };
    if (variables) {
      opts.isForm = true;
      Object.assign(opts, reformatObjectOptions(variables, "variables"));
    }
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/trigger/pipeline`,
      opts
    );
  }
};
var Pipelines = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/pipelines`,
      options
    );
  }
  allVariables(projectId, pipelineId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/pipelines/${pipelineId}/variables`,
      options
    );
  }
  cancel(projectId, pipelineId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/pipelines/${pipelineId}/cancel`,
      options
    );
  }
  create(projectId, ref, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/pipeline`,
      {
        ref,
        ...options
      }
    );
  }
  remove(projectId, pipelineId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/pipelines/${pipelineId}`,
      options
    );
  }
  retry(projectId, pipelineId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/pipelines/${pipelineId}/retry`,
      options
    );
  }
  show(projectId, pipelineId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/pipelines/${pipelineId}`,
      options
    );
  }
  showLatest(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/pipelines/latest`,
      options
    );
  }
  showTestReport(projectId, pipelineId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/pipelines/${pipelineId}/test_report`,
      options
    );
  }
  showTestReportSummary(projectId, pipelineId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/pipelines/${pipelineId}/test_report_summary`,
      options
    );
  }
};
var ProductAnalytics = class extends BaseResource {
  allFunnels(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/product_analytics/funnels`,
      options
    );
  }
  load(projectId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/product_analytics/request/load`,
      options
    );
  }
  dryRun(projectId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/product_analytics/request/dry-run`,
      options
    );
  }
  showMetadata(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/product_analytics/request/meta`,
      options
    );
  }
};

// src/resources/ProjectAccessRequests.ts
var ProjectAccessRequests = class extends ResourceAccessRequests {
  constructor(options) {
    super("projects", options);
  }
};

// src/resources/ProjectAccessTokens.ts
var ProjectAccessTokens = class extends ResourceAccessTokens {
  constructor(options) {
    super("projects", options);
  }
};
var ProjectAliases = class extends BaseResource {
  all(options) {
    return RequestHelper.get()(this, "project_aliases", options);
  }
  create(projectId, name, options) {
    return RequestHelper.post()(this, "project_aliases", {
      name,
      projectId,
      ...options
    });
  }
  edit(name, options) {
    return RequestHelper.post()(this, `project_aliases/${name}`, options);
  }
  remove(name, options) {
    return RequestHelper.del()(this, `project_aliases/${name}`, options);
  }
};

// src/resources/ProjectBadges.ts
var ProjectBadges = class extends ResourceBadges {
  constructor(options) {
    super("projects", options);
  }
};

// src/resources/ProjectCustomAttributes.ts
var ProjectCustomAttributes = class extends ResourceCustomAttributes {
  constructor(options) {
    super("projects", options);
  }
};

// src/resources/ProjectDORA4Metrics.ts
var ProjectDORA4Metrics = class extends ResourceDORA4Metrics {
  constructor(options) {
    super("projects", options);
  }
};

// src/resources/ProjectHooks.ts
var ProjectHooks = class extends ResourceHooks {
  constructor(options) {
    super("projects", options);
  }
};
var ProjectImportExports = class extends BaseResource {
  download(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/export/download`,
      options
    );
  }
  import(file, path, options) {
    return RequestHelper.post()(this, "projects/import", {
      isForm: true,
      ...options,
      file: [file.content, file.filename],
      path
    });
  }
  importRemote(url12, path, options) {
    return RequestHelper.post()(this, "projects/remote-import", {
      ...options,
      path,
      url: url12
    });
  }
  importRemoteS3(accessKeyId, bucketName, fileKey, path, region, secretAccessKey, options) {
    return RequestHelper.post()(this, "projects/remote-import", {
      ...options,
      accessKeyId,
      bucketName,
      fileKey,
      path,
      region,
      secretAccessKey
    });
  }
  showExportStatus(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/export`,
      options
    );
  }
  showImportStatus(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/import`,
      options
    );
  }
  scheduleExport(projectId, uploadConfig, options) {
    return RequestHelper.post()(this, endpoint`projects/${projectId}/export`, {
      ...options,
      upload: uploadConfig
    });
  }
};

// src/resources/ProjectInvitations.ts
var ProjectInvitations = class extends ResourceInvitations {
  constructor(options) {
    super("projects", options);
  }
};

// src/resources/ProjectIssueBoards.ts
var ProjectIssueBoards = class extends ResourceIssueBoards {
  constructor(options) {
    super("projects", options);
  }
};

// src/resources/ProjectIterations.ts
var ProjectIterations = class extends ResourceIterations {
  constructor(options) {
    super("project", options);
  }
};
var ProjectJobTokenScopes = class extends BaseResource {
  show(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/job_token_scope`,
      options
    );
  }
  edit(projectId, enabled, options) {
    return RequestHelper.patch()(
      this,
      endpoint`projects/${projectId}/job_token_scope`,
      { ...options, enabled }
    );
  }
  showInboundAllowList(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/job_token_scope/allowlist`,
      options
    );
  }
  addToInboundAllowList(projectId, targetProjectId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/job_token_scope/allowlist`,
      { ...options, targetProjectId }
    );
  }
  removeFromInboundAllowList(projectId, targetProjectId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/job_token_scope/allowlist/${targetProjectId}`,
      options
    );
  }
  showGroupsAllowList(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/job_token_scope/groups_allowlist`,
      options
    );
  }
  addToGroupsAllowList(projectId, targetGroupId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/job_token_scope/groups_allowlist`,
      { ...options, targetGroupId }
    );
  }
  removeFromGroupsAllowList(projectId, targetGroupId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/job_token_scope/groups_allowlist/${targetGroupId}`,
      options
    );
  }
};

// src/resources/ProjectLabels.ts
var ProjectLabels = class extends ResourceLabels {
  constructor(options) {
    super("projects", options);
  }
};

// src/resources/ProjectMarkdownUploads.ts
var ProjectMarkdownUploads = class extends ResourceMarkdownUploads {
  constructor(options) {
    super("projects", options);
  }
  create(projectId, file, options) {
    return RequestHelper.post()(this, endpoint`${projectId}/uploads`, {
      isForm: true,
      ...options,
      file: [file.content, file.filename]
    });
  }
};

// src/resources/ProjectMembers.ts
var ProjectMembers = class extends ResourceMembers {
  constructor(options) {
    super("projects", options);
  }
};

// src/resources/ProjectMilestones.ts
var ProjectMilestones = class extends ResourceMilestones {
  constructor(options) {
    super("projects", options);
  }
  promote(projectId, milestoneId, options) {
    return RequestHelper.post()(
      this,
      endpoint`${projectId}/milestones/${milestoneId}/promote`,
      options
    );
  }
};

// src/resources/ProjectProtectedEnvironments.ts
var ProjectProtectedEnvironments = class extends ResourceProtectedEnvironments {
  constructor(options) {
    super("projects", options);
  }
};

// src/resources/ProjectPushRules.ts
var ProjectPushRules = class extends ResourcePushRules {
  constructor(options) {
    super("projects", options);
  }
};
var ProjectRelationsExport = class extends BaseResource {
  download(projectId, relation, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/export_relations/download`,
      {
        relation,
        ...options
      }
    );
  }
  showExportStatus(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/export_relations/status`,
      options
    );
  }
  scheduleExport(projectId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/export_relations`,
      options
    );
  }
};
var ProjectReleases = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/releases`,
      options
    );
  }
  create(projectId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/releases`,
      options
    );
  }
  createEvidence(projectId, tagName, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/releases/${tagName}/evidence`,
      options
    );
  }
  edit(projectId, tagName, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/releases/${tagName}`,
      options
    );
  }
  download(projectId, tagName, filepath, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/releases/${tagName}/downloads/${filepath}`,
      options
    );
  }
  downloadLatest(projectId, filepath, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/releases/permalink/latest/downloads/${filepath}`,
      options
    );
  }
  remove(projectId, tagName, options) {
    return RequestHelper.del()(this, endpoint`projects/${projectId}/releases/${tagName}`, options);
  }
  show(projectId, tagName, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/releases/${tagName}`,
      options
    );
  }
  showLatest(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/releases/permalink/latest`,
      options
    );
  }
  showLatestEvidence(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/releases/permalink/latest/evidence`,
      options
    );
  }
};
var ProjectRemoteMirrors = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/remote_mirrors`,
      options
    );
  }
  // Helper method - Duplicated from Projects
  createPullMirror(projectId, url12, mirror, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/mirror/pull`,
      {
        importUrl: url12,
        mirror,
        ...options
      }
    );
  }
  createPushMirror(projectId, url12, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/remote_mirrors`,
      {
        url: url12,
        ...options
      }
    );
  }
  edit(projectId, mirrorId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/remote_mirrors/${mirrorId}`,
      options
    );
  }
  remove(projectId, mirrorId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/remote_mirrors/${mirrorId}`,
      options
    );
  }
  show(projectId, mirrorId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/remote_mirrors/${mirrorId}`,
      options
    );
  }
  sync(projectId, mirrorId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/remote_mirrors/${mirrorId}/sync`,
      options
    );
  }
};

// src/resources/ProjectRepositoryStorageMoves.ts
var ProjectRepositoryStorageMoves = class extends ResourceRepositoryStorageMoves {
  constructor(options) {
    super("projects", options);
  }
};

// src/resources/ProjectSnippetAwardEmojis.ts
var ProjectSnippetAwardEmojis = class extends ResourceAwardEmojis {
  constructor(options) {
    super("projects", "snippets", options);
  }
};

// src/resources/ProjectSnippetDiscussions.ts
var ProjectSnippetDiscussions = class extends ResourceDiscussions {
  constructor(options) {
    super("projects", "snippets", options);
  }
};

// src/resources/ProjectSnippetNotes.ts
var ProjectSnippetNotes = class extends ResourceNotes {
  constructor(options) {
    super("projects", "snippets", options);
  }
};
var ProjectSnippets = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/snippets`,
      options
    );
  }
  create(projectId, title, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/snippets`,
      {
        title,
        ...options
      }
    );
  }
  edit(projectId, snippetId, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/snippets/${snippetId}`,
      options
    );
  }
  remove(projectId, snippetId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/snippets/${snippetId}`,
      options
    );
  }
  show(projectId, snippetId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/snippets/${snippetId}`,
      options
    );
  }
  showContent(projectId, snippetId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/snippets/${snippetId}/raw`,
      options
    );
  }
  showRepositoryFileContent(projectId, snippetId, ref, filePath, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/snippets/${snippetId}/files/${ref}/${filePath}/raw`,
      options
    );
  }
  showUserAgentDetails(projectId, snippetId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/snippets/${snippetId}/user_agent_detail`,
      options
    );
  }
};
var ProjectStatistics = class extends BaseResource {
  show(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/statistics`,
      options
    );
  }
};
var ProjectTemplates = class extends BaseResource {
  all(projectId, type, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/templates/${type}`,
      options
    );
  }
  show(projectId, type, name, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/templates/${type}/${name}`,
      options
    );
  }
};
var ProjectTerraformState = class extends BaseResource {
  show(projectId, name, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/terraform/state/${name}`,
      options
    );
  }
  showVersion(projectId, name, serial, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/terraform/state/${name}/versions/${serial}`,
      options
    );
  }
  removeVersion(projectId, name, serial, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/terraform/state/${name}/versions/${serial}`,
      options
    );
  }
  remove(projectId, name, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/terraform/state/${name}`,
      options
    );
  }
  removeTerraformStateLock(projectId, name, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/terraform/state/${name}/lock`,
      options
    );
  }
  createVersion(projectId, name, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/terraform/state/${name}`,
      options
    );
  }
};

// src/resources/ProjectVariables.ts
var ProjectVariables = class extends ResourceVariables {
  constructor(options) {
    super("projects", options);
  }
};
var ProjectVulnerabilities = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/vulnerabilities`,
      options
    );
  }
  create(projectId, findingId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/vulnerabilities`,
      {
        ...options,
        searchParams: {
          findingId
        }
      }
    );
  }
};

// src/resources/ProjectWikis.ts
var ProjectWikis = class extends ResourceWikis {
  constructor(options) {
    super("projects", options);
  }
};
var Projects = class extends BaseResource {
  all({
    userId,
    starredOnly,
    ...options
  } = {}) {
    let uri;
    if (userId && starredOnly) uri = endpoint`users/${userId}/starred_projects`;
    else if (userId) uri = endpoint`users/${userId}/projects`;
    else uri = "projects";
    return RequestHelper.get()(this, uri, options);
  }
  allTransferLocations(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/transfer_locations`,
      options
    );
  }
  allUsers(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/users`,
      options
    );
  }
  allGroups(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/groups`,
      options
    );
  }
  allInvitedGroups(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/invited_groups`,
      options
    );
  }
  allSharableGroups(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/share_locations`,
      options
    );
  }
  allForks(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/forks`,
      options
    );
  }
  allStarrers(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/starrers`,
      options
    );
  }
  allStoragePaths(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/storage`,
      options
    );
  }
  archive(projectId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/archive`,
      options
    );
  }
  create({
    userId,
    avatar,
    ...options
  } = {}) {
    const url12 = userId ? `projects/user/${userId}` : "projects";
    if (avatar) {
      return RequestHelper.post()(this, url12, {
        ...options,
        isForm: true,
        avatar: [avatar.content, avatar.filename]
      });
    }
    return RequestHelper.post()(this, url12, { ...options, avatar });
  }
  createForkRelationship(projectId, forkedFromId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/fork/${forkedFromId}`,
      options
    );
  }
  // Helper method - Duplicated from ProjectRemoteMirrors
  createPullMirror(projectId, url12, mirror, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/mirror/pull`,
      {
        importUrl: url12,
        mirror,
        ...options
      }
    );
  }
  downloadSnapshot(projectId, options) {
    return RequestHelper.get()(this, endpoint`projects/${projectId}/snapshot`, options);
  }
  edit(projectId, { avatar, ...options } = {}) {
    const url12 = endpoint`projects/${projectId}`;
    if (avatar) {
      return RequestHelper.put()(this, url12, {
        ...options,
        isForm: true,
        avatar: [avatar.content, avatar.filename]
      });
    }
    return RequestHelper.put()(this, url12, { ...options, avatar });
  }
  fork(projectId, options) {
    return RequestHelper.post()(this, endpoint`projects/${projectId}/fork`, options);
  }
  housekeeping(projectId, options) {
    return RequestHelper.post()(this, endpoint`projects/${projectId}/housekeeping`, options);
  }
  importProjectMembers(projectId, sourceProjectId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/import_project_members/${sourceProjectId}`,
      options
    );
  }
  remove(projectId, options) {
    return RequestHelper.del()(this, endpoint`projects/${projectId}`, options);
  }
  removeForkRelationship(projectId, options) {
    return RequestHelper.del()(this, endpoint`projects/${projectId}/fork`, options);
  }
  removeAvatar(projectId, options) {
    return RequestHelper.put()(this, endpoint`projects/${projectId}`, {
      ...options,
      avatar: ""
    });
  }
  restore(projectId, options) {
    return RequestHelper.post()(this, endpoint`projects/${projectId}/restore`, options);
  }
  search(projectName, options) {
    return RequestHelper.get()(this, "projects", {
      search: projectName,
      ...options
    });
  }
  share(projectId, groupId, groupAccess, options) {
    return RequestHelper.post()(this, endpoint`projects/${projectId}/share`, {
      groupId,
      groupAccess,
      ...options
    });
  }
  show(projectId, options) {
    return RequestHelper.get()(this, endpoint`projects/${projectId}`, options);
  }
  showLanguages(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/languages`,
      options
    );
  }
  showPullMirror(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/mirror/pull`,
      options
    );
  }
  star(projectId, options) {
    return RequestHelper.post()(this, endpoint`projects/${projectId}/star`, options);
  }
  transfer(projectId, namespaceId, options) {
    return RequestHelper.put()(this, endpoint`projects/${projectId}/transfer`, {
      ...options,
      namespace: namespaceId
    });
  }
  unarchive(projectId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/unarchive`,
      options
    );
  }
  unshare(projectId, groupId, options) {
    return RequestHelper.del()(this, endpoint`projects/${projectId}/share/${groupId}`, options);
  }
  unstar(projectId, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/unstar`,
      options
    );
  }
  /* Upload file to be used a reference within an issue, merge request or
     comment
  */
  uploadForReference(projectId, file, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/uploads`,
      {
        ...options,
        isForm: true,
        file: [file.content, file.filename]
      }
    );
  }
  uploadAvatar(projectId, avatar, options) {
    return RequestHelper.put()(this, endpoint`projects/${projectId}`, {
      ...options,
      isForm: true,
      avatar: [avatar.content, avatar.filename]
    });
  }
};
var ProtectedBranches = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/protected_branches`,
      options
    );
  }
  create(projectId, branchName, options) {
    const { sudo, showExpanded, ...opts } = options || {};
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/protected_branches`,
      {
        searchParams: {
          ...opts,
          name: branchName
        },
        sudo,
        showExpanded
      }
    );
  }
  // Convenience method - create
  protect(projectId, branchName, options) {
    return this.create(projectId, branchName, options);
  }
  edit(projectId, branchName, options) {
    return RequestHelper.patch()(
      this,
      endpoint`projects/${projectId}/protected_branches/${branchName}`,
      options
    );
  }
  show(projectId, branchName, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/protected_branches/${branchName}`,
      options
    );
  }
  remove(projectId, branchName, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/protected_branches/${branchName}`,
      options
    );
  }
  // Convenience method - remove
  unprotect(projectId, branchName, options) {
    return this.remove(projectId, branchName, options);
  }
};
var ProtectedTags = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/protected_tags`,
      options
    );
  }
  create(projectId, tagName, options) {
    const { sudo, showExpanded, ...opts } = options || {};
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/protected_tags`,
      {
        searchParams: {
          name: tagName,
          ...opts
        },
        sudo,
        showExpanded
      }
    );
  }
  // Convenience method - create
  protect(projectId, tagName, options) {
    return this.create(projectId, tagName, options);
  }
  show(projectId, tagName, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/protected_tags/${tagName}`,
      options
    );
  }
  remove(projectId, tagName, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/protected_tags/${tagName}`,
      options
    );
  }
  // Convenience method - remove
  unprotect(projectId, tagName, options) {
    return this.remove(projectId, tagName, options);
  }
};
var ReleaseLinks = class extends BaseResource {
  all(projectId, tagName, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/releases/${tagName}/assets/links`,
      options
    );
  }
  create(projectId, tagName, name, url12, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/releases/${tagName}/assets/links`,
      {
        name,
        url: url12,
        ...options
      }
    );
  }
  edit(projectId, tagName, linkId, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/releases/${tagName}/assets/links/${linkId}`,
      options
    );
  }
  remove(projectId, tagName, linkId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/releases/${tagName}/assets/links/${linkId}`,
      options
    );
  }
  show(projectId, tagName, linkId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/releases/${tagName}/assets/links/${linkId}`,
      options
    );
  }
};
var Repositories = class extends BaseResource {
  allContributors(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/contributors`,
      options
    );
  }
  allRepositoryTrees(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/tree`,
      options
    );
  }
  compare(projectId, from, to, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/compare`,
      {
        from,
        to,
        ...options
      }
    );
  }
  editChangelog(projectId, version, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/repository/changelog`,
      { ...options, version }
    );
  }
  mergeBase(projectId, refs, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/merge_base`,
      {
        ...options,
        refs
      }
    );
  }
  showArchive(projectId, {
    fileType = "tar.gz",
    ...options
  } = {}) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/archive.${fileType}`,
      options
    );
  }
  showBlob(projectId, sha, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/blobs/${sha}`,
      options
    );
  }
  showBlobRaw(projectId, sha, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/blobs/${sha}/raw`,
      options
    );
  }
  showChangelog(projectId, version, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/changelog`,
      { ...options, version }
    );
  }
};
var RepositoryFiles = class extends BaseResource {
  allFileBlames(projectId, filePath, ref, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/files/${filePath}/blame`,
      {
        ref,
        ...options
      }
    );
  }
  create(projectId, filePath, branch, content, commitMessage, options) {
    return RequestHelper.post()(
      this,
      endpoint`projects/${projectId}/repository/files/${filePath}`,
      {
        branch,
        content,
        commitMessage,
        ...options
      }
    );
  }
  edit(projectId, filePath, branch, content, commitMessage, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/repository/files/${filePath}`,
      {
        branch,
        content,
        commitMessage,
        ...options
      }
    );
  }
  remove(projectId, filePath, branch, commitMessage, options) {
    return RequestHelper.del()(this, endpoint`projects/${projectId}/repository/files/${filePath}`, {
      branch,
      commitMessage,
      ...options
    });
  }
  show(projectId, filePath, ref, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/files/${filePath}`,
      {
        ref,
        ...options
      }
    );
  }
  showRaw(projectId, filePath, ref, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/files/${filePath}/raw`,
      {
        ref,
        ...options
      }
    );
  }
};
var RepositorySubmodules = class extends BaseResource {
  edit(projectId, submodule, branch, commitSha, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/repository/submodules/${submodule}`,
      {
        branch,
        commitSha,
        ...options
      }
    );
  }
};
var ResourceGroups = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/resource_groups`,
      options
    );
  }
  edit(projectId, key, options) {
    return RequestHelper.put()(
      this,
      endpoint`projects/${projectId}/resource_groups/${key}`,
      options
    );
  }
  show(projectId, key, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/resource_groups/${key}`,
      options
    );
  }
  allUpcomingJobs(projectId, key, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/resource_groups/${key}/upcoming_jobs`,
      options
    );
  }
};
var Runners = class extends BaseResource {
  all({
    projectId,
    groupId,
    owned,
    ...options
  } = {}) {
    let url12;
    if (projectId) url12 = endpoint`projects/${projectId}/runners`;
    else if (groupId) url12 = endpoint`groups/${groupId}/runners`;
    else if (owned) url12 = "runners";
    else url12 = "runners/all";
    return RequestHelper.get()(this, url12, options);
  }
  allJobs(runnerId, options) {
    return RequestHelper.get()(this, `runners/${runnerId}/jobs`, options);
  }
  // https://docs.gitlab.com/15.9/ee/api/runners.html#register-a-new-runner
  create(token, options) {
    return RequestHelper.post()(this, `runners`, {
      token,
      ...options
    });
  }
  edit(runnerId, options) {
    return RequestHelper.put()(this, `runners/${runnerId}`, options);
  }
  enable(projectId, runnerId, options) {
    return RequestHelper.post()(this, endpoint`projects/${projectId}/runners`, {
      runnerId,
      ...options
    });
  }
  disable(projectId, runnerId, options) {
    return RequestHelper.del()(this, endpoint`projects/${projectId}/runners/${runnerId}`, options);
  }
  // Create - Convenience method
  register(token, options) {
    return this.create(token, options);
  }
  remove({
    runnerId,
    token,
    ...options
  }) {
    let url12;
    if (runnerId) url12 = `runners/${runnerId}`;
    else if (token) {
      url12 = "runners";
    } else
      throw new Error(
        "Missing required argument. Please supply a runnerId or a token in the options parameter"
      );
    return RequestHelper.del()(this, url12, {
      token,
      ...options
    });
  }
  resetRegistrationToken({
    runnerId,
    token,
    ...options
  } = {}) {
    let url12;
    if (runnerId) url12 = endpoint`runners/${runnerId}/reset_registration_token`;
    else if (token) url12 = "runners/reset_registration_token";
    else {
      throw new Error("Missing either runnerId or token parameters");
    }
    return RequestHelper.post()(this, url12, {
      token,
      ...options
    });
  }
  show(runnerId, options) {
    return RequestHelper.get()(this, `runners/${runnerId}`, options);
  }
  verify(options) {
    return RequestHelper.post()(this, `runners/verify`, options);
  }
};
var SecureFiles = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/secure_files`,
      options
    );
  }
  create(projectId, name, file, options) {
    return RequestHelper.post()(this, `projects/${projectId}/secure_files`, {
      isForm: true,
      ...options,
      file: [file.content, file.filename],
      name
    });
  }
  download(projectId, secureFileId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/secure_files/${secureFileId}/download`,
      options
    );
  }
  remove(projectId, secureFileId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/secure_files/${secureFileId}`,
      options
    );
  }
  show(projectId, secureFileId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/secure_files/${secureFileId}`,
      options
    );
  }
};
var Tags = class extends BaseResource {
  all(projectId, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/tags`,
      options
    );
  }
  create(projectId, tagName, ref, options) {
    return RequestHelper.post()(this, endpoint`projects/${projectId}/repository/tags`, {
      searchParams: {
        tagName,
        ref
      },
      ...options
    });
  }
  remove(projectId, tagName, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/repository/tags/${tagName}`,
      options
    );
  }
  show(projectId, tagName, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/tags/${tagName}`,
      options
    );
  }
  showSignature(projectId, tagName, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/repository/tags/${tagName}/signature`,
      options
    );
  }
};
var UserStarredMetricsDashboard = class extends BaseResource {
  create(projectId, dashboardPath, options) {
    return RequestHelper.get()(
      this,
      endpoint`projects/${projectId}/metrics/user_starred_dashboards`,
      {
        dashboardPath,
        ...options
      }
    );
  }
  remove(projectId, options) {
    return RequestHelper.del()(
      this,
      endpoint`projects/${projectId}/metrics/user_starred_dashboards`,
      options
    );
  }
};

// src/resources/EpicAwardEmojis.ts
var EpicAwardEmojis = class extends ResourceAwardEmojis {
  constructor(options) {
    super("epics", "issues", options);
  }
};

// src/resources/EpicDiscussions.ts
var EpicDiscussions = class extends ResourceDiscussions {
  constructor(options) {
    super("groups", "epics", options);
  }
};
var EpicIssues = class extends BaseResource {
  all(groupId, epicIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/epics/${epicIId}/issues`,
      options
    );
  }
  assign(groupId, epicIId, epicIssueId, options) {
    return RequestHelper.post()(
      this,
      endpoint`groups/${groupId}/epics/${epicIId}/issues/${epicIssueId}`,
      options
    );
  }
  edit(groupId, epicIId, epicIssueId, options) {
    return RequestHelper.put()(
      this,
      endpoint`groups/${groupId}/epics/${epicIId}/issues/${epicIssueId}`,
      options
    );
  }
  remove(groupId, epicIId, epicIssueId, options) {
    return RequestHelper.del()(
      this,
      endpoint`groups/${groupId}/epics/${epicIId}/issues/${epicIssueId}`,
      options
    );
  }
};

// src/resources/EpicLabelEvents.ts
var EpicLabelEvents = class extends ResourceLabelEvents {
  constructor(options) {
    super("groups", "epics", options);
  }
};
var EpicLinks = class extends BaseResource {
  all(groupId, epicIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/epics/${epicIId}/links`,
      options
    );
  }
  assign(groupId, epicIId, childEpicId, options) {
    return RequestHelper.post()(
      this,
      endpoint`groups/${groupId}/epics/${epicIId}/links/${childEpicId}`,
      options
    );
  }
  create(groupId, epicIId, title, options) {
    return RequestHelper.post()(
      this,
      endpoint`groups/${groupId}/epics/${epicIId}/links`,
      {
        searchParams: {
          title
        },
        ...options
      }
    );
  }
  reorder(groupId, epicIId, childEpicId, options) {
    return RequestHelper.put()(
      this,
      endpoint`groups/${groupId}/epics/${epicIId}/links/${childEpicId}`,
      options
    );
  }
  unassign(groupId, epicIId, childEpicId, options) {
    return RequestHelper.del()(
      this,
      endpoint`groups/${groupId}/epics/${epicIId}/links/${childEpicId}`,
      options
    );
  }
};

// src/resources/EpicNotes.ts
var EpicNotes = class extends ResourceNotes {
  constructor(options) {
    super("groups", "epics", options);
  }
};
var Epics = class extends BaseResource {
  all(groupId, options) {
    return RequestHelper.get()(this, endpoint`groups/${groupId}/epics`, options);
  }
  create(groupId, title, options) {
    return RequestHelper.post()(this, endpoint`groups/${groupId}/epics`, {
      title,
      ...options
    });
  }
  createTodo(groupId, epicIId, options) {
    return RequestHelper.post()(
      this,
      endpoint`groups/${groupId}/epics/${epicIId}/todos`,
      options
    );
  }
  edit(groupId, epicIId, options) {
    return RequestHelper.put()(
      this,
      endpoint`groups/${groupId}/epics/${epicIId}`,
      options
    );
  }
  remove(groupId, epicIId, options) {
    return RequestHelper.del()(this, endpoint`groups/${groupId}/epics/${epicIId}`, options);
  }
  show(groupId, epicIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/epics/${epicIId}`,
      options
    );
  }
};

// src/resources/GroupAccessRequests.ts
var GroupAccessRequests = class extends ResourceAccessRequests {
  constructor(options) {
    super("groups", options);
  }
};

// src/resources/GroupAccessTokens.ts
var GroupAccessTokens = class extends ResourceAccessTokens {
  constructor(options) {
    super("groups", options);
  }
};
var GroupActivityAnalytics = class extends BaseResource {
  showIssuesCount(groupPath, options) {
    return RequestHelper.get()(
      this,
      "analytics/group_activity/issues_count",
      {
        searchParams: {
          groupPath
        },
        ...options
      }
    );
  }
  showMergeRequestsCount(groupPath, options) {
    return RequestHelper.get()(
      this,
      "analytics/group_activity/merge_requests_count",
      {
        searchParams: {
          groupPath
        },
        ...options
      }
    );
  }
  showNewMembersCount(groupPath, options) {
    return RequestHelper.get()(
      this,
      "analytics/group_activity/new_members_count",
      {
        searchParams: {
          groupPath
        },
        ...options
      }
    );
  }
};

// src/resources/GroupBadges.ts
var GroupBadges = class extends ResourceBadges {
  constructor(options) {
    super("groups", options);
  }
};

// src/resources/GroupCustomAttributes.ts
var GroupCustomAttributes = class extends ResourceCustomAttributes {
  constructor(options) {
    super("groups", options);
  }
};

// src/resources/GroupDORA4Metrics.ts
var GroupDORA4Metrics = class extends ResourceDORA4Metrics {
  constructor(options) {
    super("groups", options);
  }
};
var GroupEpicBoards = class extends BaseResource {
  all(groupId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/epic_boards`,
      options
    );
  }
  allLists(groupId, boardId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/epic_boards/${boardId}/lists`,
      options
    );
  }
  show(groupId, boardId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/epic_boards/${boardId}`,
      options
    );
  }
  showList(groupId, boardId, listId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/epic_boards/${boardId}/lists/${listId}`,
      options
    );
  }
};

// src/resources/GroupHooks.ts
var GroupHooks = class extends ResourceHooks {
  constructor(options) {
    super("groups", options);
  }
};
var GroupImportExports = class extends BaseResource {
  download(groupId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/export/download`,
      options
    );
  }
  import(file, path, { parentId, name, ...options }) {
    return RequestHelper.post()(this, "groups/import", {
      isForm: true,
      ...options,
      file: [file.content, file.filename],
      path,
      name: name || path.split("/").at(0),
      parentId
    });
  }
  scheduleExport(groupId, options) {
    return RequestHelper.post()(
      this,
      endpoint`groups/${groupId}/export`,
      options
    );
  }
};

// src/resources/GroupInvitations.ts
var GroupInvitations = class extends ResourceInvitations {
  constructor(options) {
    super("groups", options);
  }
};

// src/resources/GroupIssueBoards.ts
var GroupIssueBoards = class extends ResourceIssueBoards {
  constructor(options) {
    super("groups", options);
  }
};

// src/resources/GroupIterations.ts
var GroupIterations = class extends ResourceIterations {
  constructor(options) {
    super("groups", options);
  }
};
var GroupLDAPLinks = class extends BaseResource {
  add(groupId, groupAccess, provider, options) {
    return RequestHelper.post()(this, endpoint`groups/${groupId}/ldap_group_links`, {
      groupAccess,
      provider,
      ...options
    });
  }
  all(groupId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/ldap_group_links`,
      options
    );
  }
  remove(groupId, provider, options) {
    return RequestHelper.del()(this, endpoint`groups/${groupId}/ldap_group_links`, {
      provider,
      ...options
    });
  }
  sync(groupId, options) {
    return RequestHelper.post()(this, endpoint`groups/${groupId}/ldap_sync`, options);
  }
};

// src/resources/GroupLabels.ts
var GroupLabels = class extends ResourceLabels {
  constructor(options) {
    super("groups", options);
  }
};

// src/resources/GroupMarkdownUploads.ts
var GroupMarkdownUploads = class extends ResourceMarkdownUploads {
  constructor(options) {
    super("groups", options);
  }
};
var GroupMemberRoles = class extends BaseResource {
  add(groupId, baseAccessLevel, options) {
    return RequestHelper.post()(this, endpoint`groups/${groupId}/members`, {
      baseAccessLevel,
      ...options
    });
  }
  all(groupId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/member_roles`,
      options
    );
  }
  remove(groupId, memberRoleId, options) {
    return RequestHelper.del()(
      this,
      endpoint`groups/${groupId}/member_roles/${memberRoleId}`,
      options
    );
  }
};

// src/resources/GroupMembers.ts
var GroupMembers = class extends ResourceMembers {
  constructor(options) {
    super("groups", options);
  }
  allBillable(groupId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${groupId}/billable_members`,
      options
    );
  }
  allPending(groupId, options) {
    return RequestHelper.get()(this, endpoint`${groupId}/pending_members`, options);
  }
  allBillableMemberships(groupId, userId, options) {
    return RequestHelper.get()(
      this,
      endpoint`${groupId}/billable_members/${userId}/memberships`,
      options
    );
  }
  approve(groupId, userId, options) {
    return RequestHelper.put()(
      this,
      endpoint`${groupId}/members/${userId}/approve`,
      options
    );
  }
  approveAll(groupId, options) {
    return RequestHelper.put()(
      this,
      endpoint`${groupId}/members/approve_all`,
      options
    );
  }
  removeBillable(groupId, userId, options) {
    return RequestHelper.del()(this, endpoint`${groupId}/billable_members/${userId}`, options);
  }
  removeOverrideFlag(groupId, userId, options) {
    return RequestHelper.del()(
      this,
      endpoint`${groupId}/members/${userId}/override`,
      options
    );
  }
  setOverrideFlag(groupId, userId, options) {
    return RequestHelper.post()(
      this,
      endpoint`${groupId}/members/${userId}/override`,
      options
    );
  }
};

// src/resources/GroupMilestones.ts
var GroupMilestones = class extends ResourceMilestones {
  constructor(options) {
    super("groups", options);
  }
};

// src/resources/GroupProtectedEnvironments.ts
var GroupProtectedEnvironments = class extends ResourceProtectedEnvironments {
  constructor(options) {
    super("groups", options);
  }
};

// src/resources/GroupPushRules.ts
var GroupPushRules = class extends ResourcePushRules {
  constructor(options) {
    super("groups", options);
  }
};
var GroupRelationExports = class extends BaseResource {
  download(groupId, relation, options) {
    return RequestHelper.get()(this, endpoint`groups/${groupId}/export_relations/download`, {
      searchParams: { relation },
      ...options
    });
  }
  exportStatus(groupId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/export_relations`,
      options
    );
  }
  scheduleExport(groupId, options) {
    return RequestHelper.post()(
      this,
      endpoint`groups/${groupId}/export_relations`,
      options
    );
  }
};
var GroupReleases = class extends BaseResource {
  all(groupId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/releases`,
      options
    );
  }
};

// src/resources/GroupRepositoryStorageMoves.ts
var GroupRepositoryStorageMoves = class extends ResourceRepositoryStorageMoves {
  constructor(options) {
    super("groups", options);
  }
};
var GroupSAMLIdentities = class extends BaseResource {
  all(groupId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/saml/identities`,
      options
    );
  }
  edit(groupId, identityId, options) {
    return RequestHelper.patch()(
      this,
      endpoint`groups/${groupId}/saml/${identityId}`,
      options
    );
  }
};
var GroupSAMLLinks = class extends BaseResource {
  all(groupId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/saml_group_links`,
      options
    );
  }
  create(groupId, samlGroupName, accessLevel, options) {
    return RequestHelper.post()(
      this,
      endpoint`groups/${groupId}/saml_group_links`,
      {
        accessLevel,
        samlGroupName,
        ...options
      }
    );
  }
  remove(groupId, samlGroupName, options) {
    return RequestHelper.del()(
      this,
      endpoint`groups/${groupId}/saml_group_links/${samlGroupName}`,
      options
    );
  }
  show(groupId, samlGroupName, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/saml_group_links/${samlGroupName}`,
      options
    );
  }
};
var GroupSCIMIdentities = class extends BaseResource {
  all(groupId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/scim/identities`,
      options
    );
  }
  edit(groupId, identityId, options) {
    return RequestHelper.patch()(
      this,
      endpoint`groups/${groupId}/scim/${identityId}`,
      options
    );
  }
};
var GroupServiceAccounts = class extends BaseResource {
  create(groupId, options) {
    return RequestHelper.post()(
      this,
      endpoint`groups/${groupId}/service_accounts`,
      options
    );
  }
  // @deprecated In favor of `createPersonalAccessToken`
  addPersonalAccessToken(groupId, serviceAccountId, options) {
    return this.createPersonalAccessToken(groupId, serviceAccountId, options);
  }
  createPersonalAccessToken(groupId, serviceAccountId, options) {
    return RequestHelper.post()(
      this,
      endpoint`groups/${groupId}/service_accounts/${serviceAccountId}`,
      options
    );
  }
  rotatePersonalAccessToken(groupId, serviceAccountId, tokenId, options) {
    return RequestHelper.post()(
      this,
      endpoint`groups/${groupId}/service_accounts/${serviceAccountId}/personal_access_tokens/${tokenId}/rotate`,
      options
    );
  }
};

// src/resources/GroupVariables.ts
var GroupVariables = class extends ResourceVariables {
  constructor(options) {
    super("groups", options);
  }
};

// src/resources/GroupWikis.ts
var GroupWikis = class extends ResourceWikis {
  constructor(options) {
    super("groups", options);
  }
};
var Groups = class extends BaseResource {
  all(options) {
    return RequestHelper.get()(this, "groups", options);
  }
  allDescendantGroups(groupId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/descendant_groups`,
      options
    );
  }
  allProjects(groupId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/projects`,
      options
    );
  }
  allSharedProjects(groupId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/projects/shared`,
      options
    );
  }
  allSubgroups(groupId, options) {
    return RequestHelper.get()(this, endpoint`groups/${groupId}/subgroups`, options);
  }
  allProvisionedUsers(groupId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/provisioned_users`,
      options
    );
  }
  allTransferLocations(groupId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/transfer_locations`,
      options
    );
  }
  create(name, path, { avatar, ...options } = {}) {
    if (avatar) {
      return RequestHelper.post()(this, "groups", {
        ...options,
        isForm: true,
        avatar: [avatar.content, avatar.filename],
        name,
        path
      });
    }
    return RequestHelper.post()(this, "groups", { name, path, ...options });
  }
  downloadAvatar(groupId, options) {
    return RequestHelper.get()(this, endpoint`groups/${groupId}/avatar`, options);
  }
  edit(groupId, { avatar, ...options } = {}) {
    if (avatar) {
      return RequestHelper.post()(this, endpoint`groups/${groupId}`, {
        ...options,
        isForm: true,
        avatar: [avatar.content, avatar.filename]
      });
    }
    return RequestHelper.put()(this, endpoint`groups/${groupId}`, options);
  }
  remove(groupId, options) {
    return RequestHelper.del()(this, endpoint`groups/${groupId}`, options);
  }
  removeAvatar(groupId, options) {
    return RequestHelper.put()(this, endpoint`groups/${groupId}`, {
      ...options,
      avatar: ""
    });
  }
  restore(groupId, options) {
    return RequestHelper.post()(this, endpoint`groups/${groupId}/restore`, options);
  }
  search(nameOrPath, options) {
    return RequestHelper.get()(this, "groups", {
      search: nameOrPath,
      ...options
    });
  }
  share(groupId, sharedGroupId, groupAccess, options) {
    return RequestHelper.post()(this, endpoint`groups/${groupId}/share`, {
      groupId: sharedGroupId,
      groupAccess,
      ...options
    });
  }
  show(groupId, options) {
    return RequestHelper.get()(this, endpoint`groups/${groupId}`, options);
  }
  transfer(groupId, options) {
    return RequestHelper.post()(this, endpoint`groups/${groupId}/transfer`, options);
  }
  transferProject(groupId, projectId, options) {
    return RequestHelper.post()(
      this,
      endpoint`groups/${groupId}/projects/${projectId}`,
      options
    );
  }
  unshare(groupId, sharedGroupId, options) {
    return RequestHelper.del()(this, endpoint`groups/${groupId}/share/${sharedGroupId}`, options);
  }
  uploadAvatar(groupId, content, { filename, ...options } = {}) {
    return RequestHelper.put()(this, endpoint`groups/${groupId}/avatar`, {
      isForm: true,
      ...options,
      file: [content, filename]
    });
  }
};
var LinkedEpics = class extends BaseResource {
  all(groupId, epicIId, options) {
    return RequestHelper.get()(
      this,
      endpoint`groups/${groupId}/epics/${epicIId}/related_epics`,
      options
    );
  }
  create(groupId, epicIId, targetEpicIId, targetGroupId, options) {
    return RequestHelper.post()(
      this,
      endpoint`groups/${groupId}/epics/${epicIId}/related_epics`,
      {
        searchParams: {
          targetGroupId,
          targetEpicIid: targetEpicIId
        },
        ...options
      }
    );
  }
  remove(groupId, epicIId, relatedEpicLinkId, options) {
    return RequestHelper.del()(
      this,
      endpoint`groups/${groupId}/epics/${epicIId}/related_epics/${relatedEpicLinkId}`,
      options
    );
  }
};

// src/resources/UserCustomAttributes.ts
var UserCustomAttributes = class extends ResourceCustomAttributes {
  constructor(options) {
    super("users", options);
  }
};
var url9 = (userId) => userId ? `users/${userId}/emails` : "user/emails";
var UserEmails = class extends BaseResource {
  // Convenience method for create
  add(email, options) {
    return this.create(email, options);
  }
  all({
    userId,
    ...options
  } = {}) {
    return RequestHelper.get()(
      this,
      url9(userId),
      options
    );
  }
  create(email, {
    userId,
    ...options
  } = {}) {
    return RequestHelper.post()(this, url9(userId), {
      email,
      ...options
    });
  }
  show(emailId, options) {
    return RequestHelper.get()(this, `user/emails/${emailId}`, options);
  }
  remove(emailId, { userId, ...options } = {}) {
    return RequestHelper.del()(
      this,
      `${url9(userId)}/${emailId}`,
      options
    );
  }
};
var url10 = (userId) => userId ? `users/${userId}/gpg_keys` : "user/gpg_keys";
var UserGPGKeys = class extends BaseResource {
  // Convienence method
  add(key, options) {
    return this.create(key, options);
  }
  all({
    userId,
    ...options
  } = {}) {
    return RequestHelper.get()(this, url10(userId), options);
  }
  create(key, { userId, ...options } = {}) {
    return RequestHelper.post()(this, url10(userId), {
      key,
      ...options
    });
  }
  show(keyId, { userId, ...options } = {}) {
    return RequestHelper.get()(this, `${url10(userId)}/${keyId}`, options);
  }
  remove(keyId, { userId, ...options } = {}) {
    return RequestHelper.del()(this, `${url10(userId)}/${keyId}`, options);
  }
};
var UserImpersonationTokens = class extends BaseResource {
  all(userId, options) {
    return RequestHelper.get()(
      this,
      `users/${userId}/impersonation_tokens`,
      options
    );
  }
  create(userId, name, scopes, options) {
    return RequestHelper.post()(
      this,
      `users/${userId}/impersonation_tokens`,
      {
        name,
        scopes,
        ...options
      }
    );
  }
  show(userId, tokenId, options) {
    return RequestHelper.get()(
      this,
      `users/${userId}/impersonation_tokens/${tokenId}`,
      options
    );
  }
  remove(userId, tokenId, options) {
    return RequestHelper.del()(this, `users/${userId}/impersonation_tokens/${tokenId}`, options);
  }
  // Convienence method
  revoke(userId, tokenId, options) {
    return this.remove(userId, tokenId, options);
  }
};
var url11 = (userId) => userId ? `users/${userId}/keys` : "user/keys";
var UserSSHKeys = class extends BaseResource {
  // Convienence method for create
  add(title, key, options) {
    return this.create(title, key, options);
  }
  all({
    userId,
    ...options
  } = {}) {
    return RequestHelper.get()(
      this,
      url11(userId),
      options
    );
  }
  create(title, key, {
    userId,
    ...options
  } = {}) {
    return RequestHelper.post()(this, url11(userId), {
      title,
      key,
      ...options
    });
  }
  show(keyId, { userId, ...options } = {}) {
    return RequestHelper.get()(
      this,
      `${url11(userId)}/${keyId}`,
      options
    );
  }
  remove(keyId, { userId, ...options } = {}) {
    return RequestHelper.del()(this, `${url11(userId)}/${keyId}`, options);
  }
};
var Users = class extends BaseResource {
  activate(userId, options) {
    return RequestHelper.post()(this, endpoint`users/${userId}/activate`, options);
  }
  all(options) {
    return RequestHelper.get()(
      this,
      "users",
      options
    );
  }
  allActivities(options) {
    return RequestHelper.get()(this, "user/activities", options);
  }
  allEvents(userId, options) {
    return RequestHelper.get()(this, endpoint`users/${userId}/events`, options);
  }
  allFollowers(userId, options) {
    return RequestHelper.get()(
      this,
      endpoint`users/${userId}/followers`,
      options
    );
  }
  allFollowing(userId, options) {
    return RequestHelper.get()(
      this,
      endpoint`users/${userId}/following`,
      options
    );
  }
  allMemberships(userId, options) {
    return RequestHelper.get()(
      this,
      endpoint`users/${userId}/memberships`,
      options
    );
  }
  allProjects(userId, options) {
    return RequestHelper.get()(this, endpoint`users/${userId}/projects`, options);
  }
  allContributedProjects(userId, options) {
    return RequestHelper.get()(
      this,
      endpoint`users/${userId}/contributed_projects`,
      options
    );
  }
  allStarredProjects(userId, options) {
    return RequestHelper.get()(
      this,
      endpoint`users/${userId}/starred_projects`,
      options
    );
  }
  approve(userId, options) {
    return RequestHelper.post()(
      this,
      endpoint`users/${userId}/approve`,
      options
    );
  }
  ban(userId, options) {
    return RequestHelper.post()(this, endpoint`users/${userId}/ban`, options);
  }
  block(userId, options) {
    return RequestHelper.post()(this, endpoint`users/${userId}/block`, options);
  }
  create(options) {
    return RequestHelper.post()(this, "users", options);
  }
  createPersonalAccessToken(userId, name, scopes, options) {
    return RequestHelper.post()(
      this,
      endpoint`users/${userId}/personal_access_tokens`,
      {
        name,
        scopes,
        ...options
      }
    );
  }
  createCIRunner(runnerType, options) {
    return RequestHelper.post()(this, "user/runners", {
      ...options,
      runnerType
    });
  }
  deactivate(userId, options) {
    return RequestHelper.post()(this, endpoint`users/${userId}/deactivate`, options);
  }
  disableTwoFactor(userId, options) {
    return RequestHelper.patch()(this, endpoint`users/${userId}/disable_two_factor`, options);
  }
  edit(userId, { avatar, ...options } = {}) {
    const opts = {
      ...options,
      isForm: true
    };
    if (avatar) opts.avatar = [avatar.content, avatar.filename];
    return RequestHelper.put()(this, endpoint`users/${userId}`, opts);
  }
  editStatus(options) {
    return RequestHelper.put()(this, "user/status", options);
  }
  editCurrentUserPreferences(viewDiffsFileByFile, showWhitespaceInDiffs, options) {
    return RequestHelper.put()(this, "user/preferences", {
      viewDiffsFileByFile,
      showWhitespaceInDiffs,
      ...options
    });
  }
  follow(userId, options) {
    return RequestHelper.post()(this, endpoint`users/${userId}/follow`, options);
  }
  reject(userId, options) {
    return RequestHelper.post()(
      this,
      endpoint`users/${userId}/reject`,
      options
    );
  }
  show(userId, options) {
    return RequestHelper.get()(
      this,
      endpoint`users/${userId}`,
      options
    );
  }
  showCount(options) {
    return RequestHelper.get()(this, "user_counts", options);
  }
  showAssociationsCount(userId, options) {
    return RequestHelper.get()(
      this,
      `users/${userId}/associations_count`,
      options
    );
  }
  showCurrentUser(options) {
    return RequestHelper.get()(
      this,
      "user",
      options
    );
  }
  showCurrentUserPreferences(options) {
    return RequestHelper.get()(this, "user/preferences", options);
  }
  showStatus({
    iDOrUsername,
    ...options
  } = {}) {
    let url12;
    if (iDOrUsername) url12 = `users/${iDOrUsername}/status`;
    else url12 = "user/status";
    return RequestHelper.get()(this, url12, options);
  }
  remove(userId, options) {
    return RequestHelper.del()(this, endpoint`users/${userId}`, options);
  }
  removeAuthenticationIdentity(userId, provider, options) {
    return RequestHelper.del()(this, endpoint`users/${userId}/identities/${provider}`, options);
  }
  unban(userId, options) {
    return RequestHelper.post()(this, endpoint`users/${userId}/unban`, options);
  }
  unblock(userId, options) {
    return RequestHelper.post()(this, endpoint`users/${userId}/unblock`, options);
  }
  unfollow(userId, options) {
    return RequestHelper.post()(this, endpoint`users/${userId}/unfollow`, options);
  }
};

// src/resources/MergeRequestStateEvents.ts
var MergeRequestStateEvents = class extends ResourceStateEvents {
  constructor(options) {
    super("projects", "merge_requests", options);
  }
};

// src/resources/EpicStateEvents.ts
var EpicStateEvents = class extends ResourceStateEvents {
  constructor(options) {
    super("groups", "epics", options);
  }
};

// src/resources/Gitlab.ts
var resources = {
  Agents,
  AlertManagement,
  ApplicationAppearance,
  ApplicationPlanLimits,
  Applications,
  ApplicationSettings,
  ApplicationStatistics,
  AuditEvents,
  Avatar,
  BroadcastMessages,
  CodeSuggestions,
  Composer,
  Conan,
  DashboardAnnotations,
  Debian,
  DependencyProxy,
  DeployKeys,
  DeployTokens,
  DockerfileTemplates,
  Events,
  Experiments,
  GeoNodes,
  GeoSites,
  GitignoreTemplates,
  GitLabCIYMLTemplates,
  Import,
  InstanceLevelCICDVariables,
  Keys,
  License,
  LicenseTemplates,
  Lint,
  Markdown,
  Maven,
  Metadata,
  Migrations,
  Namespaces,
  NotificationSettings,
  NPM,
  NuGet,
  PersonalAccessTokens,
  PyPI,
  RubyGems,
  Search,
  SearchAdmin,
  ServiceAccounts,
  ServiceData,
  SidekiqMetrics,
  SidekiqQueues,
  SnippetRepositoryStorageMoves,
  Snippets,
  Suggestions,
  SystemHooks,
  TodoLists,
  Topics,
  Branches,
  CommitDiscussions,
  Commits,
  ContainerRegistry,
  Deployments,
  Environments,
  ErrorTrackingClientKeys,
  ErrorTrackingSettings,
  ExternalStatusChecks,
  FeatureFlags,
  FeatureFlagUserLists,
  FreezePeriods,
  GitlabPages,
  GoProxy,
  Helm,
  Integrations,
  IssueAwardEmojis,
  IssueDiscussions,
  IssueIterationEvents,
  IssueLabelEvents,
  IssueLinks,
  IssueMilestoneEvents,
  IssueNoteAwardEmojis,
  IssueNotes,
  Issues,
  IssuesStatistics,
  IssueStateEvents,
  IssueWeightEvents,
  JobArtifacts,
  Jobs,
  MergeRequestApprovals,
  MergeRequestAwardEmojis,
  MergeRequestContextCommits,
  MergeRequestDiscussions,
  MergeRequestLabelEvents,
  MergeRequestMilestoneEvents,
  MergeRequestStateEvents,
  MergeRequestDraftNotes,
  MergeRequestNotes,
  MergeRequestNoteAwardEmojis,
  MergeRequests,
  MergeTrains,
  PackageRegistry,
  Packages,
  PagesDomains,
  Pipelines,
  PipelineSchedules,
  PipelineScheduleVariables,
  PipelineTriggerTokens,
  ProductAnalytics,
  ProjectAccessRequests,
  ProjectAccessTokens,
  ProjectAliases,
  ProjectBadges,
  ProjectCustomAttributes,
  ProjectDORA4Metrics,
  ProjectHooks,
  ProjectImportExports,
  ProjectInvitations,
  ProjectIssueBoards,
  ProjectIterations,
  ProjectJobTokenScopes,
  ProjectLabels,
  ProjectMarkdownUploads,
  ProjectMembers,
  ProjectMilestones,
  ProjectProtectedEnvironments,
  ProjectPushRules,
  ProjectRelationsExport,
  ProjectReleases,
  ProjectRemoteMirrors,
  ProjectRepositoryStorageMoves,
  Projects,
  ProjectSnippetAwardEmojis,
  ProjectSnippetDiscussions,
  ProjectSnippetNotes,
  ProjectSnippets,
  ProjectStatistics,
  ProjectTemplates,
  ProjectTerraformState,
  ProjectVariables,
  ProjectVulnerabilities,
  ProjectWikis,
  ProtectedBranches,
  ProtectedTags,
  ReleaseLinks,
  Repositories,
  RepositoryFiles,
  RepositorySubmodules,
  ResourceGroups,
  Runners,
  SecureFiles,
  Tags,
  UserStarredMetricsDashboard,
  EpicAwardEmojis,
  EpicDiscussions,
  EpicIssues,
  EpicLabelEvents,
  EpicLinks,
  EpicNotes,
  Epics,
  EpicStateEvents,
  GroupAccessRequests,
  GroupAccessTokens,
  GroupActivityAnalytics,
  GroupBadges,
  GroupCustomAttributes,
  GroupDORA4Metrics,
  GroupEpicBoards,
  GroupHooks,
  GroupImportExports,
  GroupInvitations,
  GroupIssueBoards,
  GroupIterations,
  GroupLabels,
  GroupLDAPLinks,
  GroupMarkdownUploads,
  GroupMembers,
  GroupMemberRoles,
  GroupMilestones,
  GroupProtectedEnvironments,
  GroupPushRules,
  GroupRelationExports,
  GroupReleases,
  GroupRepositoryStorageMoves,
  Groups,
  GroupSAMLIdentities,
  GroupSAMLLinks,
  GroupSCIMIdentities,
  GroupServiceAccounts,
  GroupVariables,
  GroupWikis,
  LinkedEpics,
  UserCustomAttributes,
  UserEmails,
  UserGPGKeys,
  UserImpersonationTokens,
  Users,
  UserSSHKeys
};
var Gitlab = class extends BaseResource {
  constructor(options) {
    super(options);
    Object.keys(resources).forEach((s) => {
      this[s] = new resources[s](options);
    });
  }
};

// src/constants.ts
var AccessLevel = /* @__PURE__ */ ((AccessLevel2) => {
  AccessLevel2[AccessLevel2["NO_ACCESS"] = 0] = "NO_ACCESS";
  AccessLevel2[AccessLevel2["MINIMAL_ACCESS"] = 5] = "MINIMAL_ACCESS";
  AccessLevel2[AccessLevel2["GUEST"] = 10] = "GUEST";
  AccessLevel2[AccessLevel2["REPORTER"] = 20] = "REPORTER";
  AccessLevel2[AccessLevel2["DEVELOPER"] = 30] = "DEVELOPER";
  AccessLevel2[AccessLevel2["MAINTAINER"] = 40] = "MAINTAINER";
  AccessLevel2[AccessLevel2["OWNER"] = 50] = "OWNER";
  AccessLevel2[AccessLevel2["ADMIN"] = 60] = "ADMIN";
  return AccessLevel2;
})(AccessLevel || {});

export { AccessLevel, Agents, AlertManagement, ApplicationAppearance, ApplicationPlanLimits, ApplicationSettings, ApplicationStatistics, Applications, AuditEvents, Avatar, Branches, BroadcastMessages, CodeSuggestions, CommitDiscussions, Commits, Composer, Conan, ContainerRegistry, DashboardAnnotations, Debian, DependencyProxy, DeployKeys, DeployTokens, Deployments, DockerfileTemplates, Environments, EpicAwardEmojis, EpicDiscussions, EpicIssues, EpicLabelEvents, EpicLinks, EpicNotes, Epics, ErrorTrackingClientKeys, ErrorTrackingSettings, Events, Experiments, ExternalStatusChecks, FeatureFlagUserLists, FeatureFlags, FreezePeriods, GeoNodes, GeoSites, GitLabCIYMLTemplates, GitignoreTemplates, Gitlab, GitlabPages, GoProxy, GroupAccessRequests, GroupAccessTokens, GroupActivityAnalytics, GroupBadges, GroupCustomAttributes, GroupDORA4Metrics, GroupEpicBoards, GroupHooks, GroupImportExports, GroupInvitations, GroupIssueBoards, GroupIterations, GroupLDAPLinks, GroupLabels, GroupMarkdownUploads, GroupMemberRoles, GroupMembers, GroupMilestones, GroupProtectedEnvironments, GroupPushRules, GroupRelationExports, GroupReleases, GroupRepositoryStorageMoves, GroupSAMLIdentities, GroupSAMLLinks, GroupSCIMIdentities, GroupServiceAccounts, GroupVariables, GroupWikis, Groups, Helm, Import, InstanceLevelCICDVariables, Integrations, IssueAwardEmojis, IssueDiscussions, IssueIterationEvents, IssueLabelEvents, IssueLinks, IssueMilestoneEvents, IssueNoteAwardEmojis, IssueNotes, IssueStateEvents, IssueWeightEvents, Issues, IssuesStatistics, JobArtifacts, Jobs, Keys, License, LicenseTemplates, LinkedEpics, Lint, Markdown, Maven, MergeRequestApprovals, MergeRequestAwardEmojis, MergeRequestContextCommits, MergeRequestDiscussions, MergeRequestDraftNotes, MergeRequestLabelEvents, MergeRequestMilestoneEvents, MergeRequestNoteAwardEmojis, MergeRequestNotes, MergeRequests, MergeTrains, Metadata, Migrations, NPM, Namespaces, NotificationSettings, NuGet, PackageRegistry, Packages, PagesDomains, PersonalAccessTokens, PipelineScheduleVariables, PipelineSchedules, PipelineTriggerTokens, Pipelines, ProductAnalytics, ProjectAccessRequests, ProjectAccessTokens, ProjectAliases, ProjectBadges, ProjectCustomAttributes, ProjectDORA4Metrics, ProjectHooks, ProjectImportExports, ProjectInvitations, ProjectIssueBoards, ProjectIterations, ProjectJobTokenScopes, ProjectLabels, ProjectMarkdownUploads, ProjectMembers, ProjectMilestones, ProjectProtectedEnvironments, ProjectPushRules, ProjectRelationsExport, ProjectReleases, ProjectRemoteMirrors, ProjectRepositoryStorageMoves, ProjectSnippetAwardEmojis, ProjectSnippetDiscussions, ProjectSnippetNotes, ProjectSnippets, ProjectStatistics, ProjectTemplates, ProjectTerraformState, ProjectVariables, ProjectVulnerabilities, ProjectWikis, Projects, ProtectedBranches, ProtectedTags, PyPI, ReleaseLinks, Repositories, RepositoryFiles, RepositorySubmodules, ResourceGroups, RubyGems, Runners, Search, SearchAdmin, SecureFiles, ServiceAccounts, ServiceData, SidekiqMetrics, SidekiqQueues, SnippetRepositoryStorageMoves, Snippets, Suggestions, SystemHooks, Tags, TodoLists, Topics, UserCustomAttributes, UserEmails, UserGPGKeys, UserImpersonationTokens, UserSSHKeys, UserStarredMetricsDashboard, Users };
