import { stringify } from 'qs';
import { decamelizeKeys } from 'xcase';
import { RateLimiterQueue, RateLimiterMemory } from 'rate-limiter-flexible';
import Picomatch from 'picomatch-browser';

// src/RequesterUtils.ts
var { isMatch: isGlobMatch } = Picomatch;
function generateRateLimiterFn(limit, interval) {
  const limiter = new RateLimiterQueue(
    new RateLimiterMemory({ points: limit, duration: interval })
  );
  return () => limiter.removeTokens(1);
}
function formatQuery(params = {}) {
  const decamelized = decamelizeKeys(params);
  return stringify(decamelized, { arrayFormat: "brackets" });
}
async function defaultOptionsHandler(resourceOptions, {
  body,
  searchParams,
  sudo,
  signal,
  asStream = false,
  method = "GET"
} = {}) {
  const { headers: preconfiguredHeaders, authHeaders, url, agent } = resourceOptions;
  const defaultOptions = {
    method,
    asStream,
    signal,
    prefixUrl: url,
    agent
  };
  defaultOptions.headers = { ...preconfiguredHeaders };
  if (sudo) defaultOptions.headers.sudo = `${sudo}`;
  if (body) {
    if (body instanceof FormData) {
      defaultOptions.body = body;
    } else {
      defaultOptions.body = JSON.stringify(decamelizeKeys(body));
      defaultOptions.headers["content-type"] = "application/json";
    }
  }
  if (Object.keys(authHeaders).length > 0) {
    const [authHeaderKey, authHeaderFn] = Object.entries(authHeaders)[0];
    defaultOptions.headers[authHeaderKey] = await authHeaderFn();
  }
  const q = formatQuery(searchParams);
  if (q) defaultOptions.searchParams = q;
  return Promise.resolve(defaultOptions);
}
function createRateLimiters(rateLimitOptions = {}, rateLimitDuration = 60) {
  const rateLimiters = {};
  Object.entries(rateLimitOptions).forEach(([key, config]) => {
    if (typeof config === "number")
      rateLimiters[key] = generateRateLimiterFn(config, rateLimitDuration);
    else
      rateLimiters[key] = {
        method: config.method.toUpperCase(),
        limit: generateRateLimiterFn(config.limit, rateLimitDuration)
      };
  });
  return rateLimiters;
}
function createRequesterFn(optionsHandler, requestHandler) {
  const methods = ["get", "post", "put", "patch", "delete"];
  return (serviceOptions) => {
    const requester = {};
    const rateLimiters = createRateLimiters(
      serviceOptions.rateLimits,
      serviceOptions.rateLimitDuration
    );
    methods.forEach((m) => {
      requester[m] = async (endpoint, options) => {
        const defaultRequestOptions = await defaultOptionsHandler(serviceOptions, {
          ...options,
          method: m.toUpperCase()
        });
        const requestOptions = await optionsHandler(serviceOptions, defaultRequestOptions);
        return requestHandler(endpoint, { ...requestOptions, rateLimiters });
      };
    });
    return requester;
  };
}
function createPresetConstructor(Constructor, presetConfig) {
  return class extends Constructor {
    constructor(...args) {
      const [config, ...rest] = args;
      super({ ...presetConfig, ...config }, ...rest);
    }
  };
}
function presetResourceArguments(resources, customConfig = {}) {
  const result = {};
  Object.entries(resources).forEach(([key, Constructor]) => {
    if (typeof Constructor === "function") {
      result[key] = createPresetConstructor(
        Constructor,
        customConfig
      );
    } else {
      result[key] = Constructor;
    }
  });
  return result;
}
function getMatchingRateLimiter(endpoint, rateLimiters = {}, method = "GET") {
  const sortedEndpoints = Object.keys(rateLimiters).sort().reverse();
  const match = sortedEndpoints.find((ep) => isGlobMatch(endpoint, ep));
  const rateLimitConfig = match && rateLimiters[match];
  if (typeof rateLimitConfig === "function") return rateLimitConfig;
  if (rateLimitConfig && rateLimitConfig?.method?.toUpperCase() === method.toUpperCase()) {
    return rateLimitConfig.limit;
  }
  return generateRateLimiterFn(3e3, 60);
}

// src/BaseResource.ts
function getDynamicToken(tokenArgument) {
  return tokenArgument instanceof Function ? tokenArgument() : Promise.resolve(tokenArgument);
}
var DEFAULT_RATE_LIMITS = Object.freeze({
  // Default rate limit
  "**": 3e3,
  // Import/Export
  "projects/import": 6,
  "projects/*/export": 6,
  "projects/*/download": 1,
  "groups/import": 6,
  "groups/*/export": 6,
  "groups/*/download": 1,
  // Note creation
  "projects/*/issues/*/notes": {
    method: "post",
    limit: 300
  },
  "projects/*/snippets/*/notes": {
    method: "post",
    limit: 300
  },
  "projects/*/merge_requests/*/notes": {
    method: "post",
    limit: 300
  },
  "groups/*/epics/*/notes": {
    method: "post",
    limit: 300
  },
  // Repositories - get file archive
  "projects/*/repository/archive*": 5,
  // Project Jobs
  "projects/*/jobs": 600,
  // Member deletion
  "projects/*/members": 60,
  "groups/*/members": 60
});
var BaseResource = class {
  url;
  requester;
  queryTimeout;
  headers;
  authHeaders;
  camelize;
  constructor({
    sudo,
    profileToken,
    camelize,
    requesterFn,
    agent,
    profileMode = "execution",
    host = "https://gitlab.com",
    prefixUrl = "",
    queryTimeout = 3e5,
    rateLimitDuration = 60,
    rateLimits = DEFAULT_RATE_LIMITS,
    ...tokens
  }) {
    if (!requesterFn) throw new ReferenceError("requesterFn must be passed");
    this.url = [host, "api", "v4", prefixUrl].join("/");
    this.headers = {};
    this.authHeaders = {};
    this.camelize = camelize;
    this.queryTimeout = queryTimeout;
    if ("oauthToken" in tokens)
      this.authHeaders.authorization = async () => {
        const token = await getDynamicToken(tokens.oauthToken);
        return `Bearer ${token}`;
      };
    else if ("jobToken" in tokens)
      this.authHeaders["job-token"] = async () => getDynamicToken(tokens.jobToken);
    else if ("token" in tokens)
      this.authHeaders["private-token"] = async () => getDynamicToken(tokens.token);
    if (profileToken) {
      this.headers["X-Profile-Token"] = profileToken;
      this.headers["X-Profile-Mode"] = profileMode;
    }
    if (sudo) this.headers.Sudo = `${sudo}`;
    this.requester = requesterFn({ ...this, rateLimits, rateLimitDuration, agent });
  }
};

// src/GitbeakerError.ts
var GitbeakerRequestError = class extends Error {
  cause;
  constructor(message, options) {
    super(message, options);
    this.cause = options?.cause;
    this.name = "GitbeakerRequestError";
  }
};
var GitbeakerTimeoutError = class extends Error {
  constructor(message, options) {
    super(message, options);
    this.name = "GitbeakerTimeoutError";
  }
};
var GitbeakerRetryError = class extends Error {
  constructor(message, options) {
    super(message, options);
    this.name = "GitbeakerRetryError";
  }
};

export { BaseResource, GitbeakerRequestError, GitbeakerRetryError, GitbeakerTimeoutError, createRateLimiters, createRequesterFn, defaultOptionsHandler, formatQuery, generateRateLimiterFn, getMatchingRateLimiter, presetResourceArguments };
