import { FIMCompletionRequest$Outbound, FIMCompletionResponse } from '@mistralai/mistralai/models/components';

type Awaitable<T> = T | Promise<T>;
type CopilotAIResponse = {
    text: string | null;
    raw?: unknown;
    error?: string;
};

/**
 * Providers supported by Copilot.
 */
type Provider = 'mistral';
/**
 * Core type mapping for provider-specific implementations
 */
interface ProviderImplementationMap {
    mistral: {
        Model: 'codestral';
        Params: FIMCompletionRequest$Outbound;
        Completion: FIMCompletionResponse;
    };
}
/**
 * Models available for each provider (maintained as individual exports)
 */
type MistralModel = ProviderImplementationMap['mistral']['Model'];
/**
 * Union of all predefined Copilot models
 */
type Model = {
    [K in Provider]: ProviderImplementationMap[K]['Model'];
}[Provider];
/**
 * Utility types for provider-specific implementations
 */
type PickModel<P extends Provider> = ProviderImplementationMap[P]['Model'];
type PickCompletionCreateParams<P extends Provider> = ProviderImplementationMap[P]['Params'];
type PickCompletion<P extends Provider> = ProviderImplementationMap[P]['Completion'];
/**
 * Consolidated chat completion types (maintained as individual exports)
 */
type CompletionCreateParams = {
    [K in Provider]: ProviderImplementationMap[K]['Params'];
}[Provider];
type Completion = {
    [K in Provider]: ProviderImplementationMap[K]['Completion'];
}[Provider];
/**
 * Individual provider type aliases (preserved from original)
 */
type MistralCompletionCreateParams = FIMCompletionRequest$Outbound;
type MistralCompletion = FIMCompletionResponse;
interface ProviderHandler<P extends Provider> {
    createEndpoint(model: PickModel<P>, apiKey: string): string;
    createRequestBody(model: PickModel<P>, prompt: PromptData): PickCompletionCreateParams<P>;
    createHeaders(apiKey: string): Record<string, string>;
    parseCompletion(completion: PickCompletion<P>): string | null;
}

/**
 * Represents the structure of a prompt used for code completion.
 */
interface PromptData {
    /**
     * Contextual information about the code environment
     * @example filename, technologies, etc.
     */
    context: string;
    /** Instructions for the AI model on how to generate the completion */
    instruction: string;
    /** The content of the file being edited */
    fileContent: string;
}
type CustomPrompt<T> = (metadata: T) => Partial<PromptData>;
type CustomOptions = {
    /** Should be undefined when using a custom model */
    provider?: undefined;
    /**
     * Custom model implementation function.
     * @param prompt - The prompt data used to generate the completion.
     * @returns Object containing the generated text.
     * @see {@link https://monacopilot.dev/advanced/custom-model}
     */
    model: CustomCopilotModel;
};
type CustomCopilotModel = (prompt: PromptData) => Awaitable<CustomModelResponse>;
type CustomModelResponse = {
    /** The generated text content, or null if no text was generated for some reason.
     * The text is used to insert into the editor directly. */
    text: string | null;
};
type AIRequestHandler = (params: {
    /** The API endpoint URL for the LLM completion request */
    endpoint: string;
    /** The request body containing the completion parameters (model, prompt, etc.) */
    body: Record<string, unknown>;
    /** HTTP headers required for authentication and content type */
    headers: Record<string, string>;
}) => Promise<Record<string, unknown>>;
type CopilotOptions = ProviderOptions<'mistral'> | CustomOptions;
type ProviderOptions<T extends Provider> = {
    provider: T;
    model: ProviderImplementationMap[T]['Model'];
};

declare abstract class Copilot<Metadata> {
    protected readonly apiKey: string;
    protected provider: Provider | undefined;
    protected model: Model | CustomCopilotModel;
    constructor(apiKey: string | undefined, options: CopilotOptions);
    protected abstract getDefaultPrompt(metadata: Metadata): PromptData;
    protected generatePrompt(metadata: Metadata, customPrompt?: CustomPrompt<Metadata>): PromptData;
    protected makeAIRequest(metadata: Metadata, options?: {
        customPrompt?: CustomPrompt<Metadata>;
        aiRequestHandler?: AIRequestHandler;
    }): Promise<CopilotAIResponse>;
    private prepareRequest;
    private processResponse;
    private isCustomModel;
    protected sendRequest(endpoint: string, requestBody: CompletionCreateParams, headers: Record<string, string>, aiRequestHandler?: AIRequestHandler): Promise<any>;
    protected handleError(error: unknown): CopilotAIResponse;
}

declare const logger: {
    report: (error: unknown) => {
        message: string;
        stack?: string;
    };
    warn: (message: string, error?: unknown) => void;
    warnDeprecated: (oldFeature: string, newFeature: string, location?: string) => void;
};

declare const DEFAULT_COPILOT_MAX_TOKENS: 256;
declare const DEFAULT_COPILOT_STOP_SEQUENCE: "\n\n";
declare const DEFAULT_COPILOT_TEMPERATURE: 0.1;
declare const DEFAULT_COPILOT_TOP_P: 0.1;
declare const DEFAULT_COPILOT_STREAM: false;

declare const PROVIDERS: readonly ["mistral"];
declare const MODEL_IDS: Record<Model, string>;
declare const PROVIDER_MODEL_MAP: Record<Provider, Model[]>;
declare const PROVIDER_ENDPOINT_MAP: Record<Provider, string>;

type Endpoint = string;
type Filename = string;
type Technologies = string[];
type RelatedFile = {
    /**
     * The relative path from the current editing code in the editor to an external file.
     *
     * Examples:
     * - To include a file `utils.js` in the same directory, set as `./utils.js`.
     * - To include a file `utils.js` in the parent directory, set as `../utils.js`.
     * - To include a file `utils.js` in the child directory, set as `./child/utils.js`.
     */
    path: string;
    /**
     * The content of the external file as a string.
     */
    content: string;
};
interface BaseCopilotMetadata {
    /**
     * The programming language of the code.
     */
    language: string | undefined;
    /**
     * The name of the file being edited.
     */
    filename: Filename | undefined;
    /**
     * The technologies used in the completion.
     */
    technologies: Technologies | undefined;
    /**
     * Additional context from related files.
     */
    relatedFiles: RelatedFile[] | undefined;
    /**
     * The text that appears after the cursor.
     */
    textAfterCursor: string;
    /**
     * The text that appears before the cursor.
     */
    textBeforeCursor: string;
    /**
     * The current cursor position.
     */
    cursorPosition: {
        /**
         * line number (starts at 1)
         */
        readonly lineNumber: number;
        /**
         * column (the first character in a line is between column 1 and column 2)
         */
        readonly column: number;
    };
}

export { type AIRequestHandler, type BaseCopilotMetadata, type Completion, type CompletionCreateParams, Copilot, type CopilotOptions, type CustomCopilotModel, type CustomPrompt, DEFAULT_COPILOT_MAX_TOKENS, DEFAULT_COPILOT_STOP_SEQUENCE, DEFAULT_COPILOT_STREAM, DEFAULT_COPILOT_TEMPERATURE, DEFAULT_COPILOT_TOP_P, type Endpoint, type Filename, MODEL_IDS, type MistralCompletion, type MistralCompletionCreateParams, type MistralModel, type Model, PROVIDERS, PROVIDER_ENDPOINT_MAP, PROVIDER_MODEL_MAP, type PickCompletion, type PickCompletionCreateParams, type PickModel, type PromptData, type Provider, type ProviderHandler, type ProviderImplementationMap, type ProviderOptions, type RelatedFile, type Technologies, logger };
