Object.defineProperty(exports, Symbol.toStringTag, { value: 'Module' });

const core = require('@sentry/core');
const Sentry = require('@sentry/node');
const helpers = require('./helpers.js');

/**
 * A decorator wrapping the native nest Cron decorator, sending check-ins to Sentry.
 */
const SentryCron = (monitorSlug, monitorConfig) => {
  return (target, propertyKey, descriptor) => {
    const originalMethod = descriptor.value ;

    descriptor.value = function (...args) {
      return Sentry.withMonitor(
        monitorSlug,
        () => {
          return originalMethod.apply(this, args);
        },
        monitorConfig,
      );
    };

    copyFunctionNameAndMetadata({ originalMethod, descriptor });

    return descriptor;
  };
};

/**
 * A decorator usable to wrap arbitrary functions with spans.
 */
function SentryTraced(op = 'function') {
  return function (_target, propertyKey, descriptor) {
    const originalMethod = descriptor.value ; // function can be sync or async

    descriptor.value = function (...args) {
      return Sentry.startSpan(
        {
          op: op,
          name: propertyKey,
        },
        () => {
          return originalMethod.apply(this, args);
        },
      );
    };

    copyFunctionNameAndMetadata({ originalMethod, descriptor });

    return descriptor;
  };
}

/**
 * A decorator to wrap user-defined exception filters and add Sentry error reporting.
 */
function SentryExceptionCaptured() {
  return function (target, propertyKey, descriptor) {
    const originalCatch = descriptor.value ;

    descriptor.value = function (exception, host, ...args) {
      if (helpers.isExpectedError(exception)) {
        return originalCatch.apply(this, [exception, host, ...args]);
      }

      core.captureException(exception);
      return originalCatch.apply(this, [exception, host, ...args]);
    };

    copyFunctionNameAndMetadata({ originalMethod: originalCatch, descriptor });

    return descriptor;
  };
}

/**
 * Copies the function name and metadata from the original method to the decorated method.
 * This ensures that the decorated method maintains the same name and metadata as the original.
 *
 * @param {Function} params.originalMethod - The original method being decorated
 * @param {PropertyDescriptor} params.descriptor - The property descriptor containing the decorated method
 */
function copyFunctionNameAndMetadata({
  originalMethod,
  descriptor,
}

) {
  // preserve the original name on the decorated function
  Object.defineProperty(descriptor.value, 'name', {
    value: originalMethod.name,
    configurable: true,
    enumerable: true,
    writable: true,
  });

  // copy metadata
  if (typeof Reflect !== 'undefined' && typeof Reflect.getMetadataKeys === 'function') {
    const originalMetaData = Reflect.getMetadataKeys(originalMethod);
    for (const key of originalMetaData) {
      const value = Reflect.getMetadata(key, originalMethod);
      Reflect.defineMetadata(key, value, descriptor.value);
    }
  }
}

exports.SentryCron = SentryCron;
exports.SentryExceptionCaptured = SentryExceptionCaptured;
exports.SentryTraced = SentryTraced;
//# sourceMappingURL=decorators.js.map
