Object.defineProperty(exports, Symbol.toStringTag, { value: 'Module' });

const core = require('@sentry/core');
const Sentry = require('@sentry/node');
const nest = require('./integrations/nest.js');

/**
 * Initializes the NestJS SDK
 */
function init(options = {}) {
  const opts = {
    defaultIntegrations: getDefaultIntegrations(options),
    ...options,
  };

  core.applySdkMetadata(opts, 'nestjs');

  const client = Sentry.init(opts);

  if (client) {
    client.on('spanStart', span => {
      // The NestInstrumentation has no requestHook, so we add NestJS-specific attributes here
      addNestSpanAttributes(span);
    });
  }

  return client;
}

/** Get the default integrations for the NestJS SDK. */
function getDefaultIntegrations(options) {
  return [nest.nestIntegration(), ...Sentry.getDefaultIntegrations(options)];
}

function addNestSpanAttributes(span) {
  const attributes = core.spanToJSON(span).data;

  // this is one of: app_creation, request_context, handler
  const type = attributes['nestjs.type'];

  // Only set the NestJS attributes for spans that are created by the NestJS instrumentation and for spans that do not have an op already.
  if (type && !attributes[core.SEMANTIC_ATTRIBUTE_SENTRY_OP]) {
    span.setAttributes({
      [core.SEMANTIC_ATTRIBUTE_SENTRY_ORIGIN]: 'auto.http.otel.nestjs',
      [core.SEMANTIC_ATTRIBUTE_SENTRY_OP]: `${type}.nestjs`,
    });
  }
}

exports.getDefaultIntegrations = getDefaultIntegrations;
exports.init = init;
//# sourceMappingURL=sdk.js.map
