Object.defineProperty(exports, Symbol.toStringTag, { value: 'Module' });

const common = require('@nestjs/common');
const core = require('@nestjs/core');
const core$1 = require('@sentry/core');
const helpers = require('./helpers.js');

// Partial extract of FastifyRequest interface
// https://github.com/fastify/fastify/blob/87f9f20687c938828f1138f91682d568d2a31e53/types/request.d.ts#L41

/**
 * Note: We cannot use @ syntax to add the decorators, so we add them directly below the classes as function wrappers.
 */

/**
 * Interceptor to add Sentry tracing capabilities to Nest.js applications.
 */
class SentryTracingInterceptor  {
  // used to exclude this class from being auto-instrumented

   constructor() {
    this.__SENTRY_INTERNAL__ = true;
  }

  /**
   * Intercepts HTTP requests to set the transaction name for Sentry tracing.
   */
   intercept(context, next) {
    if (core$1.getIsolationScope() === core$1.getDefaultIsolationScope()) {
      core$1.logger.warn('Isolation scope is still the default isolation scope, skipping setting transactionName.');
      return next.handle();
    }

    if (context.getType() === 'http') {
      const req = context.switchToHttp().getRequest() ;
      if ('routeOptions' in req && req.routeOptions?.url) {
        // fastify case
        core$1.getIsolationScope().setTransactionName(`${(req.method || 'GET').toUpperCase()} ${req.routeOptions.url}`);
      } else if ('route' in req && req.route?.path) {
        // express case
        core$1.getIsolationScope().setTransactionName(`${(req.method || 'GET').toUpperCase()} ${req.route.path}`);
      }
    }

    return next.handle();
  }
}
common.Injectable()(SentryTracingInterceptor);

/**
 * Global filter to handle exceptions and report them to Sentry.
 */
class SentryGlobalFilter extends core.BaseExceptionFilter {

   constructor(applicationRef) {
    super(applicationRef);
    this.__SENTRY_INTERNAL__ = true;
    this._logger = new common.Logger('ExceptionsHandler');
  }

  /**
   * Catches exceptions and reports them to Sentry unless they are expected errors.
   */
   catch(exception, host) {
    const contextType = host.getType();

    // The BaseExceptionFilter does not work well in GraphQL applications.
    // By default, Nest GraphQL applications use the ExternalExceptionFilter, which just rethrows the error:
    // https://github.com/nestjs/nest/blob/master/packages/core/exceptions/external-exception-filter.ts
    if (contextType === 'graphql') {
      // neither report nor log HttpExceptions
      if (exception instanceof common.HttpException) {
        throw exception;
      }

      if (exception instanceof Error) {
        this._logger.error(exception.message, exception.stack);
      }

      core$1.captureException(exception);
      throw exception;
    }

    // Handle microservice context (rpc)
    // We cannot add proper handing here since RpcException depend on the @nestjs/microservices package
    // For these cases we log a warning that the user should be providing a dedicated exception filter
    if (contextType === 'rpc') {
      // Unlikely case
      if (exception instanceof common.HttpException) {
        throw exception;
      }

      // Handle any other kind of error
      if (!(exception instanceof Error)) {
        if (!helpers.isExpectedError(exception)) {
          core$1.captureException(exception);
        }
        throw exception;
      }

      // In this case we're likely running into an RpcException, which the user should handle with a dedicated filter
      // https://github.com/nestjs/nest/blob/master/sample/03-microservices/src/common/filters/rpc-exception.filter.ts
      if (!helpers.isExpectedError(exception)) {
        core$1.captureException(exception);
      }

      this._logger.warn(
        'IMPORTANT: RpcException should be handled with a dedicated Rpc exception filter, not the generic SentryGlobalFilter',
      );

      // Log the error and return, otherwise we may crash the user's app by handling rpc errors in a http context
      this._logger.error(exception.message, exception.stack);
      return;
    }

    // HTTP exceptions
    if (!helpers.isExpectedError(exception)) {
      core$1.captureException(exception);
    }

    return super.catch(exception, host);
  }
}
common.Catch()(SentryGlobalFilter);

/**
 * Set up a root module that can be injected in nest applications.
 */
class SentryModule {
  /**
   * Configures the module as the root module in a Nest.js application.
   */
   static forRoot() {
    return {
      module: SentryModule,
      providers: [
        {
          provide: core.APP_INTERCEPTOR,
          useClass: SentryTracingInterceptor,
        },
      ],
    };
  }
}
common.Global()(SentryModule);
common.Module({
  providers: [
    {
      provide: core.APP_INTERCEPTOR,
      useClass: SentryTracingInterceptor,
    },
  ],
})(SentryModule);

exports.SentryGlobalFilter = SentryGlobalFilter;
exports.SentryModule = SentryModule;
//# sourceMappingURL=setup.js.map
