import { isWrapped } from '@opentelemetry/core';
import { InstrumentationBase, InstrumentationNodeModuleDefinition, InstrumentationNodeModuleFile } from '@opentelemetry/instrumentation';
import { SDK_VERSION, startSpan, captureException } from '@sentry/core';
import { getEventSpanOptions } from './helpers.js';

const supportedVersions = ['>=2.0.0'];
const COMPONENT = '@nestjs/event-emitter';

/**
 * Custom instrumentation for nestjs event-emitter
 *
 * This hooks into the `OnEvent` decorator, which is applied on event handlers.
 */
class SentryNestEventInstrumentation extends InstrumentationBase {
   constructor(config = {}) {
    super('sentry-nestjs-event', SDK_VERSION, config);
  }

  /**
   * Initializes the instrumentation by defining the modules to be patched.
   */
   init() {
    const moduleDef = new InstrumentationNodeModuleDefinition(COMPONENT, supportedVersions);

    moduleDef.files.push(this._getOnEventFileInstrumentation(supportedVersions));
    return moduleDef;
  }

  /**
   * Wraps the @OnEvent decorator.
   */
   _getOnEventFileInstrumentation(versions) {
    return new InstrumentationNodeModuleFile(
      '@nestjs/event-emitter/dist/decorators/on-event.decorator.js',
      versions,
      (moduleExports) => {
        if (isWrapped(moduleExports.OnEvent)) {
          this._unwrap(moduleExports, 'OnEvent');
        }
        this._wrap(moduleExports, 'OnEvent', this._createWrapOnEvent());
        return moduleExports;
      },
      (moduleExports) => {
        this._unwrap(moduleExports, 'OnEvent');
      },
    );
  }

  /**
   * Creates a wrapper function for the @OnEvent decorator.
   */
   _createWrapOnEvent() {
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    return function wrapOnEvent(original) {
      return function wrappedOnEvent(event, options) {
        // Get the original decorator result
        const decoratorResult = original(event, options);

        // Return a new decorator function that wraps the handler
        return (target, propertyKey, descriptor) => {
          if (
            !descriptor.value ||
            typeof descriptor.value !== 'function' ||
            target.__SENTRY_INTERNAL__ ||
            // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
            descriptor.value.__SENTRY_INSTRUMENTED__
          ) {
            return decoratorResult(target, propertyKey, descriptor);
          }

          const originalHandler = descriptor.value;
          // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
          const handlerName = originalHandler.name || propertyKey;
          let eventName = typeof event === 'string' ? event : String(event);

          // Instrument the actual handler
          descriptor.value = async function (...args) {
            // When multiple @OnEvent decorators are used on a single method, we need to get all event names
            // from the reflector metadata as there is no information during execution which event triggered it
            if (Reflect.getMetadataKeys(descriptor.value).includes('EVENT_LISTENER_METADATA')) {
              const eventData = Reflect.getMetadata('EVENT_LISTENER_METADATA', descriptor.value);
              if (Array.isArray(eventData)) {
                eventName = eventData
                  .map((data) => {
                    if (data && typeof data === 'object' && 'event' in data && data.event) {
                      return data.event;
                    }
                    return '';
                  })
                  .reverse() // decorators are evaluated bottom to top
                  .join('|');
              }
            }

            return startSpan(getEventSpanOptions(eventName), async () => {
              try {
                // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
                const result = await originalHandler.apply(this, args);
                return result;
              } catch (error) {
                // exceptions from event handlers are not caught by global error filter
                captureException(error);
                throw error;
              }
            });
          };

          // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
          descriptor.value.__SENTRY_INSTRUMENTED__ = true;

          // Preserve the original function name
          Object.defineProperty(descriptor.value, 'name', {
            value: handlerName,
            configurable: true,
          });

          // Apply the original decorator
          return decoratorResult(target, propertyKey, descriptor);
        };
      };
    };
  }
}

export { SentryNestEventInstrumentation };
//# sourceMappingURL=sentry-nest-event-instrumentation.js.map
