Object.defineProperty(exports, Symbol.toStringTag, { value: 'Module' });

const core = require('@sentry/core');
const instrument = require('../../../otel/instrument.js');
const addOriginToSpan = require('../../../utils/addOriginToSpan.js');
const attributes = require('./attributes.js');
const constants = require('./constants.js');
const instrumentation = require('./instrumentation.js');

const instrumentVercelAi = instrument.generateInstrumentOnce(constants.INTEGRATION_NAME, () => new instrumentation.SentryVercelAiInstrumentation({}));

const _vercelAIIntegration = ((options = {}) => {
  let instrumentation;

  return {
    name: constants.INTEGRATION_NAME,
    options,
    setupOnce() {
      instrumentation = instrumentVercelAi();
    },
    setup(client) {
      instrumentation?.callWhenPatched(() => {
        client.on('spanStart', span => {
          const { data: attributes$1, description: name } = core.spanToJSON(span);

          if (!name) {
            return;
          }

          // The id of the model
          const aiModelId = attributes$1[attributes.AI_MODEL_ID_ATTRIBUTE];

          // the provider of the model
          const aiModelProvider = attributes$1[attributes.AI_MODEL_PROVIDER_ATTRIBUTE];

          // both of these must be defined for the integration to work
          if (typeof aiModelId !== 'string' || typeof aiModelProvider !== 'string' || !aiModelId || !aiModelProvider) {
            return;
          }

          let isPipelineSpan = false;

          switch (name) {
            case 'ai.generateText': {
              span.setAttribute(core.SEMANTIC_ATTRIBUTE_SENTRY_OP, 'ai.pipeline.generateText');
              isPipelineSpan = true;
              break;
            }
            case 'ai.generateText.doGenerate': {
              span.setAttribute(core.SEMANTIC_ATTRIBUTE_SENTRY_OP, 'ai.run.doGenerate');
              break;
            }
            case 'ai.streamText': {
              span.setAttribute(core.SEMANTIC_ATTRIBUTE_SENTRY_OP, 'ai.pipeline.streamText');
              isPipelineSpan = true;
              break;
            }
            case 'ai.streamText.doStream': {
              span.setAttribute(core.SEMANTIC_ATTRIBUTE_SENTRY_OP, 'ai.run.doStream');
              break;
            }
            case 'ai.generateObject': {
              span.setAttribute(core.SEMANTIC_ATTRIBUTE_SENTRY_OP, 'ai.pipeline.generateObject');
              isPipelineSpan = true;
              break;
            }
            case 'ai.generateObject.doGenerate': {
              span.setAttribute(core.SEMANTIC_ATTRIBUTE_SENTRY_OP, 'ai.run.doGenerate');
              break;
            }
            case 'ai.streamObject': {
              span.setAttribute(core.SEMANTIC_ATTRIBUTE_SENTRY_OP, 'ai.pipeline.streamObject');
              isPipelineSpan = true;
              break;
            }
            case 'ai.streamObject.doStream': {
              span.setAttribute(core.SEMANTIC_ATTRIBUTE_SENTRY_OP, 'ai.run.doStream');
              break;
            }
            case 'ai.embed': {
              span.setAttribute(core.SEMANTIC_ATTRIBUTE_SENTRY_OP, 'ai.pipeline.embed');
              isPipelineSpan = true;
              break;
            }
            case 'ai.embed.doEmbed': {
              span.setAttribute(core.SEMANTIC_ATTRIBUTE_SENTRY_OP, 'ai.embeddings');
              break;
            }
            case 'ai.embedMany': {
              span.setAttribute(core.SEMANTIC_ATTRIBUTE_SENTRY_OP, 'ai.pipeline.embedMany');
              isPipelineSpan = true;
              break;
            }
            case 'ai.embedMany.doEmbed': {
              span.setAttribute(core.SEMANTIC_ATTRIBUTE_SENTRY_OP, 'ai.embeddings');
              break;
            }
            case 'ai.toolCall':
            case 'ai.stream.firstChunk':
            case 'ai.stream.finish':
              span.setAttribute(core.SEMANTIC_ATTRIBUTE_SENTRY_OP, 'ai.run');
              break;
          }

          addOriginToSpan.addOriginToSpan(span, 'auto.vercelai.otel');

          const nameWthoutAi = name.replace('ai.', '');
          span.setAttribute('ai.pipeline.name', nameWthoutAi);
          span.updateName(nameWthoutAi);

          // If a Telemetry name is set and it is a pipeline span, use that as the operation name
          const functionId = attributes$1['ai.telemetry.functionId'];
          if (functionId && typeof functionId === 'string' && isPipelineSpan) {
            span.updateName(functionId);
            span.setAttribute('ai.pipeline.name', functionId);
          }

          if (attributes$1[attributes.AI_PROMPT_ATTRIBUTE]) {
            span.setAttribute('gen_ai.prompt', attributes$1[attributes.AI_PROMPT_ATTRIBUTE]);
          }
          if (attributes$1[attributes.AI_MODEL_ID_ATTRIBUTE] && !attributes$1[attributes.GEN_AI_RESPONSE_MODEL_ATTRIBUTE]) {
            span.setAttribute(attributes.GEN_AI_RESPONSE_MODEL_ATTRIBUTE, attributes$1[attributes.AI_MODEL_ID_ATTRIBUTE]);
          }
          span.setAttribute('ai.streaming', name.includes('stream'));
        });

        client.addEventProcessor(event => {
          if (event.type === 'transaction' && event.spans?.length) {
            for (const span of event.spans) {
              const { data: attributes$1, description: name } = span;

              if (!name || span.origin !== 'auto.vercelai.otel') {
                continue;
              }

              if (attributes$1[attributes.AI_USAGE_COMPLETION_TOKENS_ATTRIBUTE] != undefined) {
                attributes$1[attributes.GEN_AI_USAGE_OUTPUT_TOKENS_ATTRIBUTE] = attributes$1[attributes.AI_USAGE_COMPLETION_TOKENS_ATTRIBUTE];
                // eslint-disable-next-line @typescript-eslint/no-dynamic-delete
                delete attributes$1[attributes.AI_USAGE_COMPLETION_TOKENS_ATTRIBUTE];
              }
              if (attributes$1[attributes.AI_USAGE_PROMPT_TOKENS_ATTRIBUTE] != undefined) {
                attributes$1[attributes.GEN_AI_USAGE_INPUT_TOKENS_ATTRIBUTE] = attributes$1[attributes.AI_USAGE_PROMPT_TOKENS_ATTRIBUTE];
                // eslint-disable-next-line @typescript-eslint/no-dynamic-delete
                delete attributes$1[attributes.AI_USAGE_PROMPT_TOKENS_ATTRIBUTE];
              }
              if (
                typeof attributes$1[attributes.GEN_AI_USAGE_OUTPUT_TOKENS_ATTRIBUTE] === 'number' &&
                typeof attributes$1[attributes.GEN_AI_USAGE_INPUT_TOKENS_ATTRIBUTE] === 'number'
              ) {
                attributes$1['gen_ai.usage.total_tokens'] =
                  attributes$1[attributes.GEN_AI_USAGE_OUTPUT_TOKENS_ATTRIBUTE] + attributes$1[attributes.GEN_AI_USAGE_INPUT_TOKENS_ATTRIBUTE];
              }
            }
          }

          return event;
        });
      });
    },
  };
}) ;

/**
 * Adds Sentry tracing instrumentation for the [ai](https://www.npmjs.com/package/ai) library.
 *
 * For more information, see the [`ai` documentation](https://sdk.vercel.ai/docs/ai-sdk-core/telemetry).
 *
 * @example
 * ```javascript
 * const Sentry = require('@sentry/node');
 *
 * Sentry.init({
 *  integrations: [Sentry.vercelAIIntegration()],
 * });
 * ```
 *
 * By default this integration adds tracing support to all `ai` function calls. If you need to disable
 * collecting spans for a specific call, you can do so by setting `experimental_telemetry.isEnabled` to
 * `false` in the first argument of the function call.
 *
 * ```javascript
 * const result = await generateText({
 *   model: openai('gpt-4-turbo'),
 *   experimental_telemetry: { isEnabled: false },
 * });
 * ```
 *
 * If you want to collect inputs and outputs for a specific call, you must specifically opt-in to each
 * function call by setting `experimental_telemetry.recordInputs` and `experimental_telemetry.recordOutputs`
 * to `true`.
 *
 * ```javascript
 * const result = await generateText({
 *  model: openai('gpt-4-turbo'),
 *  experimental_telemetry: { isEnabled: true, recordInputs: true, recordOutputs: true },
 * });
 */
const vercelAIIntegration = core.defineIntegration(_vercelAIIntegration);

exports.instrumentVercelAi = instrumentVercelAi;
exports.vercelAIIntegration = vercelAIIntegration;
//# sourceMappingURL=index.js.map
