//#region src/hooks.ts
var Hooks = class {
	/**
	* Adds the given callback to the list of callbacks for the given hook.
	*
	* The callback will be invoked when the hook it is registered for is run.
	*
	* One callback function can be registered to multiple hooks and the same hook multiple times.
	*
	* @param name The name of the hook, or an object of callbacks keyed by name
	* @param callback The callback function which is given environment variables.
	* @param first Will add the hook to the top of the list (defaults to the bottom)
	*/
	add(name, callback, first) {
		if (typeof name !== "string") for (const hookName in name) this.add(hookName, name[hookName], callback);
		else (Array.isArray(name) ? name : [name]).forEach((hookName) => {
			if (!this[hookName]) this[hookName] = [];
			if (callback && typeof callback === "function") {
				const hooks = this[hookName];
				if (first) hooks.unshift(callback);
				else hooks.push(callback);
			}
		});
	}
	/**
	* Runs a hook invoking all registered callbacks with the given environment variables.
	*
	* Callbacks will be invoked synchronously and in the order in which they were registered.
	*
	* @param name The name of the hook.
	* @param env The environment variables of the hook passed to all callbacks registered.
	*/
	run(name, env) {
		if (!this[name]) this[name] = [];
		this[name].forEach((callback) => {
			callback.call(env && env.context ? env.context : env, env);
		});
	}
};

//#endregion
//#region src/plugins.ts
var Plugins = class {
	jsep;
	registered;
	constructor(jsep$1) {
		this.jsep = jsep$1;
		this.registered = {};
	}
	/**
	* Adds the given plugin(s) to the registry
	*
	* @param plugins The plugins to register
	*/
	register(...plugins) {
		plugins.forEach((plugin) => {
			if (typeof plugin !== "object" || !plugin.name || !plugin.init) throw new Error("Invalid JSEP plugin format");
			if (this.registered[plugin.name]) return;
			plugin.init(this.jsep);
			this.registered[plugin.name] = plugin;
		});
	}
};

//#endregion
//#region src/jsep.ts
var Jsep$1 = class Jsep$1 {
	static ternary = null;
	version = "<%= version %>";
	COMPOUND = "Compound";
	SEQUENCE_EXP = "SequenceExpression";
	IDENTIFIER = "Identifier";
	MEMBER_EXP = "MemberExpression";
	LITERAL = "Literal";
	THIS_EXP = "ThisExpression";
	CALL_EXP = "CallExpression";
	UNARY_EXP = "UnaryExpression";
	BINARY_EXP = "BinaryExpression";
	ARRAY_EXP = "ArrayExpression";
	TAB_CODE = 9;
	LF_CODE = 10;
	CR_CODE = 13;
	SPACE_CODE = 32;
	PERIOD_CODE = 46;
	COMMA_CODE = 44;
	SQUOTE_CODE = 39;
	DQUOTE_CODE = 34;
	OPAREN_CODE = 40;
	CPAREN_CODE = 41;
	OBRACK_CODE = 91;
	CBRACK_CODE = 93;
	QUMARK_CODE = 63;
	SEMCOL_CODE = 59;
	COLON_CODE = 58;
	expr = "";
	index = 0;
	unary_ops = {};
	binary_ops = {};
	max_unop_len = 0;
	max_binop_len = 0;
	right_associative = /* @__PURE__ */ new Set();
	additional_identifier_chars = /* @__PURE__ */ new Set();
	literals = {};
	this_str = "this";
	hooks = new Hooks();
	plugins;
	/**
	* @returns {JsepInstance}
	*/
	static instance() {
		const instance = new Jsep$1();
		const jsep$1 = new Proxy(instance.parse.bind(instance), {
			get: (_t, k) => instance[k],
			set: (_t, k, v) => {
				instance[k] = v;
				return true;
			}
		});
		jsep$1.Jsep = jsep$1;
		return jsep$1;
	}
	/**
	* @returns {string}
	*/
	toString() {
		return "JavaScript Expression Parser (JSEP) v" + this.version;
	}
	constructor() {
		this.plugins = new Plugins(this);
		this.clearConfig();
	}
	/**
	* @returns {JsepInstance}
	*/
	instance() {
		return Jsep$1.instance();
	}
	/**
	* set ternary to static property so it can be used by defaultConfig() method
	* @param {IPlugin} ternary
	*/
	registerTernary(ternary) {
		Jsep$1.ternary = ternary;
		this.plugins.register(ternary);
	}
	/**
	* sets config to default JSEP config
	* @returns {this}
	*/
	defaultConfig() {
		this.unary_ops = {
			"-": 1,
			"!": 1,
			"~": 1,
			"+": 1
		};
		this.max_unop_len = this.getMaxKeyLen(this.unary_ops);
		this.binary_ops = {
			"||": 1,
			"??": 1,
			"&&": 2,
			"|": 3,
			"^": 4,
			"&": 5,
			"==": 6,
			"!=": 6,
			"===": 6,
			"!==": 6,
			"<": 7,
			">": 7,
			"<=": 7,
			">=": 7,
			"<<": 8,
			">>": 8,
			">>>": 8,
			"+": 9,
			"-": 9,
			"*": 10,
			"/": 10,
			"%": 10,
			"**": 11
		};
		this.max_binop_len = this.getMaxKeyLen(this.binary_ops);
		this.right_associative = new Set(["**"]);
		this.additional_identifier_chars = new Set(["$", "_"]);
		this.literals = {
			"true": true,
			"false": false,
			"null": null
		};
		this.this_str = "this";
		this.hooks = new Hooks();
		this.plugins = new Plugins(this);
		if (Jsep$1.ternary) this.plugins.register(Jsep$1.ternary);
		return this;
	}
	/**
	* clears JSEP config
	* @returns {this}
	*/
	clearConfig() {
		this.removeAllUnaryOps();
		this.removeAllBinaryOps();
		this.removeAllIdentifierChars();
		this.removeAllLiterals();
		this.this_str = "this";
		this.hooks = new Hooks();
		this.plugins = new Plugins(this);
		return this;
	}
	/**
	* @param {string} op_name The name of the unary op to add
	* @returns {this}
	*/
	addUnaryOp(op_name) {
		this.max_unop_len = Math.max(op_name.length, this.max_unop_len);
		this.unary_ops[op_name] = 1;
		return this;
	}
	/**
	* @param {string} op_name The name of the binary op to add
	* @param {number} precedence The precedence of the binary op (can be a float). Higher number = higher precedence
	* @param {boolean} [isRightAssociative=false] whether operator is right-associative
	* @returns {this}
	*/
	addBinaryOp(op_name, precedence, isRightAssociative) {
		this.max_binop_len = Math.max(op_name.length, this.max_binop_len);
		this.binary_ops[op_name] = precedence;
		if (isRightAssociative) this.right_associative.add(op_name);
		else this.right_associative.delete(op_name);
		return this;
	}
	/**
	* @param {string} char The additional character to treat as a valid part of an identifier
	* return {this}
	*/
	addIdentifierChar(char) {
		this.additional_identifier_chars.add(char);
		return this;
	}
	/**
	* @param {string} literal_name The name of the literal to add
	* @param {*} literal_value The value of the literal
	* @returns {this}
	*/
	addLiteral(literal_name, literal_value) {
		this.literals[literal_name] = literal_value;
		return this;
	}
	/**
	* @param {string} op_name The name of the unary op to remove
	* return {this}
	*/
	removeUnaryOp(op_name) {
		delete this.unary_ops[op_name];
		if (op_name.length === this.max_unop_len) this.max_unop_len = this.getMaxKeyLen(this.unary_ops);
		return this;
	}
	/**
	* return {this}
	*/
	removeAllUnaryOps() {
		this.unary_ops = {};
		this.max_unop_len = 0;
		return this;
	}
	/**
	* @param {string} char The additional character to stop treating as a valid part of an identifier
	* return {this}
	*/
	removeIdentifierChar(char) {
		this.additional_identifier_chars.delete(char);
		return this;
	}
	/**
	* @param {string} char The additional character to stop treating as a valid part of an identifier
	* return {this}
	*/
	removeAllIdentifierChars() {
		this.additional_identifier_chars = /* @__PURE__ */ new Set();
		return this;
	}
	/**
	* @param {string} op_name The name of the binary op to remove
	* return {this}
	*/
	removeBinaryOp(op_name) {
		delete this.binary_ops[op_name];
		if (op_name.length === this.max_binop_len) this.max_binop_len = this.getMaxKeyLen(this.binary_ops);
		this.right_associative.delete(op_name);
		return this;
	}
	/**
	* return {this}
	*/
	removeAllBinaryOps() {
		this.binary_ops = {};
		this.max_binop_len = 0;
		this.right_associative = /* @__PURE__ */ new Set();
		return this;
	}
	/**
	* @param {string} literal_name The name of the literal to remove
	* return {this}
	*/
	removeLiteral(literal_name) {
		delete this.literals[literal_name];
		return this;
	}
	/**
	* return {this}
	*/
	removeAllLiterals() {
		this.literals = {};
		return this;
	}
	/**
	* @returns {string}
	*/
	get char() {
		return this.expr.charAt(this.index);
	}
	/**
	* @returns {number}
	*/
	get code() {
		return this.expr.charCodeAt(this.index);
	}
	/**
	* Get the longest key length of any object
	* @param {object} obj
	* @returns {number}
	*/
	getMaxKeyLen(obj) {
		return Math.max(0, ...Object.keys(obj).map((k) => k.length));
	}
	/**
	* `ch` is a character code in the next three functions
	* @param {number} ch
	* @returns {boolean}
	*/
	isDecimalDigit(ch) {
		return ch >= 48 && ch <= 57;
	}
	/**
	* Returns the precedence of a binary operator or `0` if it isn't a binary operator. Can be float.
	* @param {string} op_val
	* @returns {number}
	*/
	binaryPrecedence(op_val) {
		return this.binary_ops[op_val] || 0;
	}
	/**
	* Looks for start of identifier
	* @param {number} ch
	* @returns {boolean}
	*/
	isIdentifierStart(ch) {
		return ch >= 65 && ch <= 90 || ch >= 97 && ch <= 122 || ch >= 128 && !this.binary_ops[String.fromCharCode(ch)] || this.additional_identifier_chars.has(String.fromCharCode(ch));
	}
	/**
	* @param {number} ch
	* @returns {boolean}
	*/
	isIdentifierPart(ch) {
		return this.isIdentifierStart(ch) || this.isDecimalDigit(ch);
	}
	/**
	* throw error at index of the expression
	* @param {string} message
	* @throws
	*/
	throwError(message) {
		const error = /* @__PURE__ */ new Error(message + " at character " + this.index);
		error.index = this.index;
		error.description = message;
		throw error;
	}
	/**
	* Run a given hook
	* @param {string} name
	* @param {Expression|false} [node]
	* @returns {Expression|false|undefined}
	*/
	runHook(name, node) {
		if (this.hooks[name]) {
			const env = {
				context: this,
				node
			};
			this.hooks.run(name, env);
			return env.node;
		}
		return node;
	}
	/**
	* Runs a given hook until one returns a node
	* @param {string} name
	* @returns {Expression|undefined}
	*/
	searchHook(name) {
		if (this.hooks[name]) {
			const env = { context: this };
			this.hooks[name].find((callback) => {
				callback.call(env.context, env);
				return env.node;
			});
			return env.node;
		}
	}
	/**
	* Push `index` up to the next non-space character
	*/
	gobbleSpaces() {
		let ch = this.code;
		while (ch === this.SPACE_CODE || ch === this.TAB_CODE || ch === this.LF_CODE || ch === this.CR_CODE) ch = this.expr.charCodeAt(++this.index);
		this.runHook("gobble-spaces");
	}
	/**
	* Top-level method to parse all expressions and returns compound or single node
	* @params {string} expr
	* @returns {Expression}
	*/
	parse(expr) {
		this.expr = expr;
		this.index = 0;
		this.runHook("before-all");
		const nodes = this.gobbleExpressions();
		const node = nodes.length === 1 ? nodes[0] : {
			type: this.COMPOUND,
			body: nodes
		};
		return this.runHook("after-all", node);
	}
	/**
	* top-level parser (but can be reused within as well)
	* @param {number} [untilICode]
	* @returns {Expression[]}
	*/
	gobbleExpressions(untilICode) {
		const nodes = [];
		let ch_i;
		let node;
		while (this.index < this.expr.length) {
			ch_i = this.code;
			if (ch_i === this.SEMCOL_CODE || ch_i === this.COMMA_CODE) this.index++;
			else {
				node = this.gobbleExpression();
				if (node) nodes.push(node);
				else if (this.index < this.expr.length) {
					if (ch_i === untilICode) break;
					this.throwError("Unexpected \"" + this.char + "\"");
				}
			}
		}
		return nodes;
	}
	/**
	* The main parsing function.
	* @returns {Expression|undefined}
	*/
	gobbleExpression() {
		const node = this.searchHook("gobble-expression") || this.gobbleBinaryExpression();
		this.gobbleSpaces();
		return this.runHook("after-expression", node);
	}
	/**
	* Search for the operation portion of the string (e.g. `+`, `===`)
	* Start by taking the longest possible binary operations (3 characters: `===`, `!==`, `>>>`)
	* and move down from 3 to 2 to 1 character until a matching binary operation is found
	* then, return that binary operation
	* @returns {string|false}
	*/
	gobbleBinaryOp() {
		this.gobbleSpaces();
		let to_check = this.expr.substr(this.index, this.max_binop_len);
		let tc_len = to_check.length;
		while (tc_len > 0) {
			if (Object.prototype.hasOwnProperty.call(this.binary_ops, to_check) && (!this.isIdentifierStart(this.code) || this.index + to_check.length < this.expr.length && !this.isIdentifierPart(this.expr.charCodeAt(this.index + to_check.length)))) {
				this.index += tc_len;
				return to_check;
			}
			to_check = to_check.substr(0, --tc_len);
		}
		return false;
	}
	/**
	* This function is responsible for gobbling an individual expression,
	* e.g. `1`, `1+2`, `a+(b*2)-Math.sqrt(2)`
	* @returns {Expression|undefined}
	*/
	gobbleBinaryExpression() {
		let node;
		let biop;
		let prec;
		let stack;
		let biop_info;
		let left;
		let right;
		let i;
		let cur_biop;
		left = this.gobbleToken();
		if (!left) return left;
		biop = this.gobbleBinaryOp();
		if (!biop) return left;
		biop_info = {
			value: biop,
			prec: this.binaryPrecedence(biop),
			right_a: this.right_associative.has(biop)
		};
		right = this.gobbleToken();
		if (!right) this.throwError("Expected expression after " + biop);
		stack = [
			left,
			biop_info,
			right
		];
		while (biop = this.gobbleBinaryOp()) {
			prec = this.binaryPrecedence(biop);
			if (prec === 0) {
				this.index -= biop.length;
				break;
			}
			biop_info = {
				value: biop,
				prec,
				right_a: this.right_associative.has(biop)
			};
			cur_biop = biop;
			const comparePrev = (prev) => biop_info.right_a && prev.right_a ? prec > prev.prec : prec <= prev.prec;
			while (stack.length > 2 && comparePrev(stack[stack.length - 2])) {
				right = stack.pop();
				biop = stack.pop().value;
				left = stack.pop();
				node = {
					type: this.BINARY_EXP,
					operator: biop,
					left,
					right
				};
				stack.push(node);
			}
			node = this.gobbleToken();
			if (!node) this.throwError("Expected expression after " + cur_biop);
			stack.push(biop_info, node);
		}
		i = stack.length - 1;
		node = stack[i];
		while (i > 1) {
			node = {
				type: this.BINARY_EXP,
				operator: stack[i - 1].value,
				left: stack[i - 2],
				right: node
			};
			i -= 2;
		}
		return node;
	}
	/**
	* An individual part of a binary expression:
	* e.g. `foo.bar(baz)`, `1`, `"abc"`, `(a % 2)` (because it's in parenthesis)
	* @returns {Expression|false|undefined}
	*/
	gobbleToken() {
		let ch;
		let to_check;
		let tc_len;
		let node;
		this.gobbleSpaces();
		const hookNode = this.searchHook("gobble-token");
		if (hookNode) return this.runHook("after-token", hookNode);
		ch = this.code;
		if (this.isDecimalDigit(ch) || ch === this.PERIOD_CODE) return this.gobbleNumericLiteral();
		if (ch === this.SQUOTE_CODE || ch === this.DQUOTE_CODE) node = this.gobbleStringLiteral();
		else if (ch === this.OBRACK_CODE) node = this.gobbleArray();
		else {
			to_check = this.expr.substr(this.index, this.max_unop_len);
			tc_len = to_check.length;
			while (tc_len > 0) {
				if (Object.prototype.hasOwnProperty.call(this.unary_ops, to_check) && (!this.isIdentifierStart(this.code) || this.index + to_check.length < this.expr.length && !this.isIdentifierPart(this.expr.charCodeAt(this.index + to_check.length)))) {
					this.index += tc_len;
					const argument = this.gobbleToken();
					if (!argument) this.throwError("missing unaryOp argument");
					return this.runHook("after-token", {
						type: this.UNARY_EXP,
						operator: to_check,
						argument,
						prefix: true
					});
				}
				to_check = to_check.substr(0, --tc_len);
			}
			if (this.isIdentifierStart(ch)) {
				node = this.gobbleIdentifier();
				if (Object.prototype.hasOwnProperty.call(this.literals, node.name)) node = {
					type: this.LITERAL,
					value: this.literals[node.name],
					raw: node.name
				};
				else if (node.name === this.this_str) node = { type: this.THIS_EXP };
			} else if (ch === this.OPAREN_CODE) node = this.gobbleGroup();
		}
		if (!node) return this.runHook("after-token", false);
		node = this.gobbleTokenProperty(node);
		return this.runHook("after-token", node);
	}
	/**
	* Gobble properties of of identifiers/strings/arrays/groups.
	* e.g. `foo`, `bar.baz`, `foo['bar'].baz`
	* It also gobbles function calls:
	* e.g. `Math.acos(obj.angle)`
	* @param {Expression} node
	* @returns {Expression}
	*/
	gobbleTokenProperty(node) {
		this.gobbleSpaces();
		let ch = this.code;
		while (ch === this.PERIOD_CODE || ch === this.OBRACK_CODE || ch === this.OPAREN_CODE || ch === this.QUMARK_CODE) {
			let optional;
			if (ch === this.QUMARK_CODE) {
				if (this.expr.charCodeAt(this.index + 1) !== this.PERIOD_CODE) break;
				optional = true;
				this.index += 2;
				this.gobbleSpaces();
				ch = this.code;
			}
			this.index++;
			if (ch === this.OBRACK_CODE) {
				node = {
					type: this.MEMBER_EXP,
					computed: true,
					object: node,
					property: this.gobbleExpression()
				};
				if (!node.property) this.throwError("Unexpected \"" + this.char + "\"");
				this.gobbleSpaces();
				ch = this.code;
				if (ch !== this.CBRACK_CODE) this.throwError("Unclosed [");
				this.index++;
			} else if (ch === this.OPAREN_CODE) node = {
				type: this.CALL_EXP,
				"arguments": this.gobbleArguments(this.CPAREN_CODE),
				callee: node
			};
			else if (ch === this.PERIOD_CODE || optional) {
				if (optional) this.index--;
				this.gobbleSpaces();
				node = {
					type: this.MEMBER_EXP,
					computed: false,
					object: node,
					property: this.gobbleIdentifier()
				};
			}
			if (optional) node.optional = true;
			this.gobbleSpaces();
			ch = this.code;
		}
		return node;
	}
	/**
	* Parse simple numeric literals: `12`, `3.4`, `.5`. Do this by using a string to
	* keep track of everything in the numeric literal and then calling `parseFloat` on that string
	* @returns {Literal}
	*/
	gobbleNumericLiteral() {
		let number = "";
		let ch;
		while (this.isDecimalDigit(this.code)) number += this.expr.charAt(this.index++);
		if (this.code === this.PERIOD_CODE) {
			number += this.expr.charAt(this.index++);
			while (this.isDecimalDigit(this.code)) number += this.expr.charAt(this.index++);
		}
		ch = this.char;
		if (ch === "e" || ch === "E") {
			number += this.expr.charAt(this.index++);
			ch = this.char;
			if (ch === "+" || ch === "-") number += this.expr.charAt(this.index++);
			while (this.isDecimalDigit(this.code)) number += this.expr.charAt(this.index++);
			if (!this.isDecimalDigit(this.expr.charCodeAt(this.index - 1))) this.throwError("Expected exponent (" + number + this.char + ")");
		}
		const chCode = this.code;
		if (this.isIdentifierStart(chCode)) this.throwError("Variable names cannot start with a number (" + number + this.char + ")");
		else if (chCode === this.PERIOD_CODE || number.length === 1 && number.charCodeAt(0) === this.PERIOD_CODE) this.throwError("Unexpected period");
		return {
			type: this.LITERAL,
			value: parseFloat(number),
			raw: number
		};
	}
	/**
	* Parses a string literal, staring with single or double quotes with basic support for escape codes
	* e.g. `"hello world"`, `'this is\nJSEP'`
	* @returns {Literal}
	*/
	gobbleStringLiteral() {
		let str = "";
		const startIndex = this.index;
		const quote = this.expr.charAt(this.index++);
		let closed = false;
		while (this.index < this.expr.length) {
			let ch = this.expr.charAt(this.index++);
			if (ch === quote) {
				closed = true;
				break;
			} else if (ch === "\\") {
				ch = this.expr.charAt(this.index++);
				switch (ch) {
					case "n":
						str += "\n";
						break;
					case "r":
						str += "\r";
						break;
					case "t":
						str += "	";
						break;
					case "b":
						str += "\b";
						break;
					case "f":
						str += "\f";
						break;
					case "v":
						str += "\v";
						break;
					default: str += ch;
				}
			} else str += ch;
		}
		if (!closed) this.throwError("Unclosed quote after \"" + str + "\"");
		return {
			type: this.LITERAL,
			value: str,
			raw: this.expr.substring(startIndex, this.index)
		};
	}
	/**
	* Gobbles only identifiers
	* e.g.: `foo`, `_value`, `$x1`
	* Also, this function checks if that identifier is a literal:
	* (e.g. `true`, `false`, `null`) or `this`
	* @returns {Identifier}
	*/
	gobbleIdentifier() {
		let ch = this.code;
		const start = this.index;
		if (this.isIdentifierStart(ch)) this.index++;
		else this.throwError("Unexpected " + this.char);
		while (this.index < this.expr.length) {
			ch = this.code;
			if (this.isIdentifierPart(ch)) this.index++;
			else break;
		}
		return {
			type: this.IDENTIFIER,
			name: this.expr.slice(start, this.index)
		};
	}
	/**
	* Gobbles a list of arguments within the context of a function call
	* or array literal. This function also assumes that the opening character
	* `(` or `[` has already been gobbled, and gobbles expressions and commas
	* until the terminator character `)` or `]` is encountered.
	* e.g. `foo(bar, baz)`, `my_func()`, or `[bar, baz]`
	* @param {number} termination
	* @returns {Expression[]}
	*/
	gobbleArguments(termination) {
		const args = [];
		let closed = false;
		let separator_count = 0;
		while (this.index < this.expr.length) {
			this.gobbleSpaces();
			const ch_i = this.code;
			if (ch_i === termination) {
				closed = true;
				this.index++;
				if (termination === this.CPAREN_CODE && separator_count && separator_count >= args.length) this.throwError("Unexpected token " + String.fromCharCode(termination));
				break;
			} else if (ch_i === this.COMMA_CODE) {
				this.index++;
				separator_count++;
				if (separator_count !== args.length) {
					if (termination === this.CPAREN_CODE) this.throwError("Unexpected token ,");
					else if (termination === this.CBRACK_CODE) for (let arg = args.length; arg < separator_count; arg++) args.push(null);
				}
			} else if (args.length !== separator_count && separator_count !== 0) this.throwError("Expected comma");
			else {
				const node = this.gobbleExpression();
				if (!node || node.type === this.COMPOUND) this.throwError("Expected comma");
				args.push(node);
			}
		}
		if (!closed) this.throwError("Expected " + String.fromCharCode(termination));
		return args;
	}
	/**
	* Responsible for parsing a group of things within parentheses `()`
	* that have no identifier in front (so not a function call)
	* This function assumes that it needs to gobble the opening parenthesis
	* and then tries to gobble everything within that parenthesis, assuming
	* that the next thing it should see is the close parenthesis. If not,
	* then the expression probably doesn't have a `)`
	* @returns {Expression|false}
	*/
	gobbleGroup() {
		this.index++;
		const nodes = this.gobbleExpressions(this.CPAREN_CODE);
		if (this.code === this.CPAREN_CODE) {
			this.index++;
			if (nodes.length === 1) return nodes[0];
			else if (!nodes.length) return false;
			else return {
				type: this.SEQUENCE_EXP,
				expressions: nodes
			};
		} else this.throwError("Unclosed (");
	}
	/**
	* Responsible for parsing Array literals `[1, 2, 3]`
	* This function assumes that it needs to gobble the opening bracket
	* and then tries to gobble the expressions as arguments.
	* @returns {ArrayExpression}
	*/
	gobbleArray() {
		this.index++;
		return {
			type: this.ARRAY_EXP,
			elements: this.gobbleArguments(this.CBRACK_CODE)
		};
	}
};
const jsep = Jsep$1.instance();
jsep.defaultConfig();
var jsep_default = jsep;

//#endregion
//#region ../ternary/dist/index.js
const CONDITIONAL_EXP = "ConditionalExpression";
var src_default$1 = {
	name: "ternary",
	init(jsep$1) {
		jsep$1.hooks.add("after-expression", function gobbleTernary(env) {
			if (env.node && this.code === jsep$1.QUMARK_CODE) {
				this.index++;
				const test = env.node;
				const consequent = this.gobbleExpression();
				if (!consequent) this.throwError("Expected expression");
				this.gobbleSpaces();
				if (this.code === jsep$1.COLON_CODE) {
					this.index++;
					const alternate = this.gobbleExpression();
					if (!alternate) this.throwError("Expected expression");
					env.node = {
						type: CONDITIONAL_EXP,
						test,
						consequent,
						alternate
					};
					const testNode = test;
					if (testNode.operator && jsep$1.binary_ops[testNode.operator] <= .9) {
						let newTest = testNode;
						while (newTest.right?.operator && jsep$1.binary_ops[newTest.right.operator] <= .9) newTest = newTest.right;
						env.node.test = newTest.right;
						newTest.right = env.node;
						env.node = test;
					}
				} else this.throwError("Expected :");
			}
		});
	}
};

//#endregion
//#region src/index.ts
jsep_default.registerTernary(src_default$1);
const Jsep = jsep_default;
var src_default = jsep_default;

//#endregion
export { Jsep, src_default as default };
//# sourceMappingURL=index.js.map