import type { HookCallback, HookEnv, IHooks } from './types.js';

export default class Hooks implements IHooks {
  [key: string]: HookCallback[] | ((name: string | Record<string, HookCallback>, callback?: HookCallback | boolean, first?: boolean) => void) | ((name: string, env: HookEnv) => void);

  /**
   * Adds the given callback to the list of callbacks for the given hook.
   *
   * The callback will be invoked when the hook it is registered for is run.
   *
   * One callback function can be registered to multiple hooks and the same hook multiple times.
   *
   * @param name The name of the hook, or an object of callbacks keyed by name
   * @param callback The callback function which is given environment variables.
   * @param first Will add the hook to the top of the list (defaults to the bottom)
   */
  add(name: string | Record<string, HookCallback>, callback?: HookCallback | boolean, first?: boolean): void {
    if (typeof name !== 'string') {
      // Multiple hook callbacks, keyed by name
      for (const hookName in name) {
        this.add(hookName, name[hookName], callback as boolean);
      }
    } else {
      const names = Array.isArray(name) ? name : [name];
      names.forEach((hookName) => {
        if (!this[hookName]) {
          this[hookName] = [];
        }

        if (callback && typeof callback === 'function') {
          const hooks = this[hookName] as HookCallback[];
          if (first) {
            hooks.unshift(callback);
          } else {
            hooks.push(callback);
          }
        }
      });
    }
  }

  /**
   * Runs a hook invoking all registered callbacks with the given environment variables.
   *
   * Callbacks will be invoked synchronously and in the order in which they were registered.
   *
   * @param name The name of the hook.
   * @param env The environment variables of the hook passed to all callbacks registered.
   */
  run(name: string, env: HookEnv): void {
    if (!this[name]) {
      this[name] = [];
    }
    const hooks = this[name] as HookCallback[];
    hooks.forEach((callback) => {
      callback.call(env && env.context ? env.context : env, env);
    });
  }
}
