// Expression Types
export type BaseTypes = string | number | boolean | RegExp | null | undefined | object;

export interface Expression {
  type: string;
  [key: string]: BaseTypes | Expression | Array<BaseTypes | Expression>;
}

export interface ArrayExpression extends Expression {
  type: 'ArrayExpression';
  elements: Array<Expression | null>;
}

export interface BinaryExpression extends Expression {
  type: 'BinaryExpression';
  operator: string;
  left: Expression;
  right: Expression;
}

export interface CallExpression extends Expression {
  type: 'CallExpression';
  arguments: Expression[];
  callee: Expression;
}

export interface Compound extends Expression {
  type: 'Compound';
  body: Expression[];
}

export interface SequenceExpression extends Expression {
  type: 'SequenceExpression';
  expressions: Expression[];
}

export interface ConditionalExpression extends Expression {
  type: 'ConditionalExpression';
  test: Expression;
  consequent: Expression;
  alternate: Expression;
}

export interface Identifier extends Expression {
  type: 'Identifier';
  name: string;
}

export interface Literal extends Expression {
  type: 'Literal';
  value: boolean | number | string | RegExp | null;
  raw: string;
}

export interface MemberExpression extends Expression {
  type: 'MemberExpression';
  computed: boolean;
  object: Expression;
  property: Expression;
  optional?: boolean;
}

export interface ThisExpression extends Expression {
  type: 'ThisExpression';
}

export interface UnaryExpression extends Expression {
  type: 'UnaryExpression';
  operator: string;
  argument: Expression;
  prefix: boolean;
}

export type ExpressionType =
  | 'Compound'
  | 'SequenceExpression'
  | 'Identifier'
  | 'MemberExpression'
  | 'Literal'
  | 'ThisExpression'
  | 'CallExpression'
  | 'UnaryExpression'
  | 'BinaryExpression'
  | 'ConditionalExpression'
  | 'ArrayExpression';

export type CoreExpression =
  | ArrayExpression
  | BinaryExpression
  | CallExpression
  | Compound
  | SequenceExpression
  | ConditionalExpression
  | Identifier
  | Literal
  | MemberExpression
  | ThisExpression
  | UnaryExpression;

export type PossibleExpression = Expression | undefined;

// Hook Types
export type HookType =
  | 'gobble-expression'
  | 'after-expression'
  | 'gobble-token'
  | 'after-token'
  | 'gobble-spaces'
  | 'before-all'
  | 'after-all';

export interface HookEnv {
  context?: JsepInstance;
  node?: Expression | false;
}

export type HookCallback = (this: JsepInstance, env: HookEnv) => void;

export interface IHooks {
  [key: string]: HookCallback[] | ((name: string | Record<string, HookCallback>, callback?: HookCallback | boolean, first?: boolean) => void) | ((name: string, env: HookEnv) => void);
  add(name: string | Record<string, HookCallback>, callback?: HookCallback | boolean, first?: boolean): void;
  run(name: string, env: HookEnv): void;
}

// Plugin Types
export interface IPlugin {
  name: string;
  init: (jsep: JsepInstance) => void;
}

export interface IPlugins {
  jsep: JsepInstance;
  registered: Record<string, IPlugin>;
  register(...plugins: IPlugin[]): void;
}

// Jsep Instance Types
export interface JsepInstance {
  (expr: string): Expression;

  // Version
  version: string;

  // Node Types
  COMPOUND: string;
  SEQUENCE_EXP: string;
  IDENTIFIER: string;
  MEMBER_EXP: string;
  LITERAL: string;
  THIS_EXP: string;
  CALL_EXP: string;
  UNARY_EXP: string;
  BINARY_EXP: string;
  ARRAY_EXP: string;

  // Character Codes
  TAB_CODE: number;
  LF_CODE: number;
  CR_CODE: number;
  SPACE_CODE: number;
  PERIOD_CODE: number;
  COMMA_CODE: number;
  SQUOTE_CODE: number;
  DQUOTE_CODE: number;
  OPAREN_CODE: number;
  CPAREN_CODE: number;
  OBRACK_CODE: number;
  CBRACK_CODE: number;
  QUMARK_CODE: number;
  SEMCOL_CODE: number;
  COLON_CODE: number;

  // Configuration
  unary_ops: Record<string, number>;
  binary_ops: Record<string, number>;
  right_associative: Set<string>;
  additional_identifier_chars: Set<string>;
  literals: Record<string, unknown>;
  this_str: string;
  max_unop_len: number;
  max_binop_len: number;

  // Hooks and Plugins
  hooks: IHooks;
  plugins: IPlugins;

  // Parser State
  expr: string;
  index: number;

  // Getters
  readonly char: string;
  readonly code: number;

  // Configuration Methods
  instance(): JsepInstance;
  defaultConfig(): JsepInstance;
  clearConfig(): JsepInstance;
  registerTernary(ternary: IPlugin): void;

  // Operator Methods
  addUnaryOp(op_name: string): JsepInstance;
  addBinaryOp(op_name: string, precedence: number, isRightAssociative?: boolean): JsepInstance;
  addIdentifierChar(char: string): JsepInstance;
  addLiteral(literal_name: string, literal_value: unknown): JsepInstance;

  removeUnaryOp(op_name: string): JsepInstance;
  removeBinaryOp(op_name: string): JsepInstance;
  removeIdentifierChar(char: string): JsepInstance;
  removeLiteral(literal_name: string): JsepInstance;

  removeAllUnaryOps(): JsepInstance;
  removeAllBinaryOps(): JsepInstance;
  removeAllIdentifierChars(): JsepInstance;
  removeAllLiterals(): JsepInstance;

  // Utility Methods
  getMaxKeyLen(obj: Record<string, unknown>): number;
  isDecimalDigit(ch: number): boolean;
  binaryPrecedence(op_val: string): number;
  isIdentifierStart(ch: number): boolean;
  isIdentifierPart(ch: number): boolean;
  throwError(message: string): never;

  // Hook Methods
  runHook(name: string, node?: Expression | false): Expression | false | undefined;
  searchHook(name: string): Expression | undefined;

  // Parser Methods
  parse(expr: string): Expression;
  gobbleSpaces(): void;
  gobbleExpressions(untilICode?: number): Expression[];
  gobbleExpression(): Expression | undefined;
  gobbleBinaryOp(): string | false;
  gobbleBinaryExpression(): Expression | undefined;
  gobbleToken(): Expression | false | undefined;
  gobbleTokenProperty(node: Expression): Expression;
  gobbleNumericLiteral(): Literal;
  gobbleStringLiteral(): Literal;
  gobbleIdentifier(): Identifier;
  gobbleArguments(termination: number): Expression[];
  gobbleGroup(): Expression | false;
  gobbleArray(): ArrayExpression;

  // For const { Jsep } = require('jsep') compatibility
  Jsep?: JsepInstance;
}
