"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createSubscriberAdapter = createSubscriberAdapter;
exports.createPublisherAdapter = createPublisherAdapter;
/**
 * Creates a Subscriber adapter for a Redis client.
 * @param client - The Redis client to adapt
 * @returns A Subscriber interface compatible with the resumable stream
 */
function createSubscriberAdapter(client) {
    // Track all active handlers by channel
    const handlers = new Map();
    // Store reference to the listener for cleanup
    const messageListener = (channel, message) => {
        const handler = handlers.get(channel);
        if (handler) {
            handler(message);
        }
    };
    return {
        connect: () => {
            // ioredis Redis instances are connected by default. Nothing to do.
            return Promise.resolve();
        },
        subscribe: async (channel, callback) => {
            // Add the global listener on first subscription
            if (handlers.size === 0) {
                client.on("message", messageListener);
            }
            handlers.set(channel, callback);
            await client.subscribe(channel);
        },
        unsubscribe: async (channel) => {
            handlers.delete(channel);
            // Remove the global listener when no more subscriptions
            if (handlers.size === 0) {
                client.removeListener("message", messageListener);
            }
            return client.unsubscribe(channel);
        },
    };
}
/**
 * Creates a Publisher adapter for a Redis client.
 * @param client - The Redis client to adapt
 * @returns A Publisher interface compatible with the resumable stream
 */
function createPublisherAdapter(client) {
    const adapter = {
        connect: () => {
            // ioredis Redis instances are connected by default. Nothing to do.
            return Promise.resolve();
        },
        publish: (channel, message) => client.publish(channel, message),
        set: (key, value, options) => {
            if (options === null || options === void 0 ? void 0 : options.EX) {
                return client.set(key, value, "EX", options.EX);
            }
            return client.set(key, value);
        },
        get: (key) => client.get(key),
        incr: (key) => client.incr(key),
    };
    return adapter;
}
//# sourceMappingURL=ioredis-adapters.js.map