"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.hiddenColumnTypes = exports.customLinkSupportedTypes = exports.isDateOrDateTimeCol = exports.isReadOnlyColumn = exports.durationOptions = exports.ratingIconListMap = exports.ratingIconList = exports.checkboxIconListMap = exports.checkboxIconList = exports.isSupportedDisplayValueColumn = exports.getUITypesForFormulaDataType = exports.partialUpdateAllowedTypes = exports.readonlyMetaAllowedTypes = exports.isInUIType = exports.isSelectTypeCol = exports.getEquivalentUIType = exports.numericUITypes = exports.FieldNameFromUITypes = exports.columnTypeName = exports.UITypesSearchTerms = exports.UITypesName = void 0;
exports.isNumericCol = isNumericCol;
exports.isVirtualCol = isVirtualCol;
exports.isAIPromptCol = isAIPromptCol;
exports.isCreatedOrLastModifiedTimeCol = isCreatedOrLastModifiedTimeCol;
exports.isCreatedOrLastModifiedByCol = isCreatedOrLastModifiedByCol;
exports.isOrderCol = isOrderCol;
exports.isActionButtonCol = isActionButtonCol;
exports.isHiddenCol = isHiddenCol;
exports.isLinksOrLTAR = isLinksOrLTAR;
exports.isSelfLinkCol = isSelfLinkCol;
const Api_1 = require("./Api");
const enums_1 = require("./formula/enums");
const globals_1 = require("../lib/globals");
const helperFunctions_1 = require("./helperFunctions");
var UITypes;
(function (UITypes) {
    UITypes["ID"] = "ID";
    UITypes["LinkToAnotherRecord"] = "LinkToAnotherRecord";
    UITypes["ForeignKey"] = "ForeignKey";
    UITypes["Lookup"] = "Lookup";
    UITypes["SingleLineText"] = "SingleLineText";
    UITypes["LongText"] = "LongText";
    UITypes["Attachment"] = "Attachment";
    UITypes["Checkbox"] = "Checkbox";
    UITypes["MultiSelect"] = "MultiSelect";
    UITypes["SingleSelect"] = "SingleSelect";
    UITypes["Collaborator"] = "Collaborator";
    UITypes["Date"] = "Date";
    UITypes["Year"] = "Year";
    UITypes["Time"] = "Time";
    UITypes["PhoneNumber"] = "PhoneNumber";
    UITypes["GeoData"] = "GeoData";
    UITypes["Email"] = "Email";
    UITypes["URL"] = "URL";
    UITypes["Number"] = "Number";
    UITypes["Decimal"] = "Decimal";
    UITypes["Currency"] = "Currency";
    UITypes["Percent"] = "Percent";
    UITypes["Duration"] = "Duration";
    UITypes["Rating"] = "Rating";
    UITypes["Formula"] = "Formula";
    UITypes["Rollup"] = "Rollup";
    UITypes["Count"] = "Count";
    UITypes["DateTime"] = "DateTime";
    UITypes["CreatedTime"] = "CreatedTime";
    UITypes["LastModifiedTime"] = "LastModifiedTime";
    UITypes["AutoNumber"] = "AutoNumber";
    UITypes["Geometry"] = "Geometry";
    UITypes["JSON"] = "JSON";
    UITypes["SpecificDBType"] = "SpecificDBType";
    UITypes["Barcode"] = "Barcode";
    UITypes["QrCode"] = "QrCode";
    UITypes["Button"] = "Button";
    UITypes["Links"] = "Links";
    UITypes["User"] = "User";
    UITypes["CreatedBy"] = "CreatedBy";
    UITypes["LastModifiedBy"] = "LastModifiedBy";
    UITypes["Order"] = "Order";
    UITypes["Meta"] = "Meta";
})(UITypes || (UITypes = {}));
exports.UITypesName = {
    [UITypes.ID]: 'ID',
    [UITypes.LinkToAnotherRecord]: 'Link to another record',
    [UITypes.ForeignKey]: 'Foreign key',
    [UITypes.Order]: 'Order',
    [UITypes.Lookup]: 'Lookup',
    [UITypes.SingleLineText]: 'Single line text',
    [UITypes.LongText]: 'Long text',
    RichText: 'Rich text',
    [UITypes.Attachment]: 'Attachment',
    [UITypes.Checkbox]: 'Checkbox',
    [UITypes.MultiSelect]: 'Multi select',
    [UITypes.SingleSelect]: 'Single select',
    [UITypes.Collaborator]: 'Collaborator',
    [UITypes.Date]: 'Date',
    [UITypes.Year]: 'Year',
    [UITypes.Time]: 'Time',
    [UITypes.PhoneNumber]: 'Phone number',
    [UITypes.GeoData]: 'Geo data',
    [UITypes.Email]: 'Email',
    [UITypes.URL]: 'URL',
    [UITypes.Number]: 'Number',
    [UITypes.Decimal]: 'Decimal',
    [UITypes.Currency]: 'Currency',
    [UITypes.Percent]: 'Percent',
    [UITypes.Duration]: 'Duration',
    [UITypes.Rating]: 'Rating',
    [UITypes.Formula]: 'Formula',
    [UITypes.Rollup]: 'Rollup',
    [UITypes.Count]: 'Count',
    [UITypes.DateTime]: 'Date time',
    [UITypes.CreatedTime]: 'Created time',
    [UITypes.LastModifiedTime]: 'Last modified time',
    [UITypes.AutoNumber]: 'Auto number',
    [UITypes.Geometry]: 'Geometry',
    [UITypes.JSON]: 'JSON',
    [UITypes.SpecificDBType]: 'Specific DB type',
    [UITypes.Barcode]: 'Barcode',
    [UITypes.QrCode]: 'Qr code',
    [UITypes.Button]: 'Button',
    [UITypes.Links]: 'Links',
    [UITypes.User]: 'User',
    [UITypes.CreatedBy]: 'Created by',
    [UITypes.LastModifiedBy]: 'Last modified by',
    [UITypes.Meta]: 'Row Meta',
    AIButton: 'AI Button',
    AIPrompt: 'AI Text',
};
exports.UITypesSearchTerms = {
    [UITypes.ID]: ['ID', 'record number', 'unique number'],
    [UITypes.ForeignKey]: [
        'Foreign key',
        'connect records',
        'related record',
        'reference',
        'relationship',
    ],
    [UITypes.LinkToAnotherRecord]: [
        'Link to another record',
        'connect records',
        'related record',
        'reference',
        'relationship',
    ],
    [UITypes.Lookup]: ['Lookup', 'pull data', 'get value from', 'reference data'],
    [UITypes.SingleLineText]: [
        'Single line text',
        'text',
        'short text',
        'name',
        'title',
    ],
    [UITypes.LongText]: [
        'Long text',
        'paragraph',
        'notes',
        'description',
        'comments',
        'memo',
        'Rich text',
        'formatted text',
        'styled text',
        'html text',
    ],
    [UITypes.Attachment]: ['Attachment', 'file', 'document', 'image', 'upload'],
    [UITypes.Checkbox]: ['Checkbox', 'yes/no', 'true/false', 'completed', 'done'],
    [UITypes.MultiSelect]: [
        'Multi select',
        'multiple options',
        'tags',
        'categories',
        'list',
        'options',
        'choice',
    ],
    [UITypes.SingleSelect]: [
        'Single select',
        'dropdown',
        'options',
        'choice',
        'pick one',
        'status',
        'priority',
    ],
    [UITypes.Collaborator]: [
        'Collaborator',
        'team member',
        'person',
        'assignee',
        'owner',
    ],
    [UITypes.Date]: ['Date', 'calendar', 'due date', 'start date'],
    [UITypes.Year]: ['Year'],
    [UITypes.Time]: ['Time', 'time of day', 'hour'],
    [UITypes.PhoneNumber]: ['Phone number', 'phone', 'contact number', 'mobile'],
    [UITypes.GeoData]: ['Geo data', 'location', 'address', 'map', 'coordinates'],
    [UITypes.Email]: ['Email', 'email address', 'contact email'],
    [UITypes.URL]: ['URL', 'website', 'link', 'web address'],
    [UITypes.Number]: ['Number', 'quantity', 'amount', 'count'],
    [UITypes.Decimal]: ['Decimal', 'decimal number', 'precise number'],
    [UITypes.Currency]: [
        'Currency',
        'money',
        'price',
        'cost',
        'budget',
        'revenue',
    ],
    [UITypes.Percent]: ['Percent', 'percentage', 'completion', 'progress'],
    [UITypes.Duration]: [
        'Duration',
        'time spent',
        'elapsed time',
        'hours worked',
    ],
    [UITypes.Rating]: ['Rating', 'stars', 'score', 'review', 'feedback'],
    [UITypes.Formula]: [
        'Formula',
        'calculation',
        'computed field',
        'auto calculate',
    ],
    [UITypes.Rollup]: [
        'Rollup',
        'summary',
        'total from related records',
        'aggregate',
    ],
    [UITypes.Count]: ['Count', 'number of records', 'total count'],
    [UITypes.DateTime]: ['Date time', 'date and time', 'timestamp'],
    [UITypes.CreatedTime]: ['Created time', 'date created', 'created on'],
    [UITypes.LastModifiedTime]: [
        'Last modified time',
        'last updated',
        'modified on',
    ],
    [UITypes.AutoNumber]: ['Auto number', 'auto increment', 'sequence number'],
    [UITypes.Geometry]: ['Geometry', 'shape', 'geographic shape'],
    [UITypes.JSON]: ['JSON', 'structured data', 'complex data'],
    [UITypes.SpecificDBType]: [
        'Specific DB type',
        'database type',
        'custom type',
    ],
    [UITypes.Barcode]: ['Barcode', 'product code', 'scan code'],
    [UITypes.QrCode]: ['Qr code', 'QR code', 'scan code'],
    [UITypes.Button]: ['Button', 'action button', 'click button'],
    [UITypes.Links]: ['Links', 'related links', 'connections'],
    [UITypes.User]: ['User', 'person', 'team member', 'assignee', 'owner'],
    [UITypes.CreatedBy]: ['Created by', 'author', 'who created'],
    [UITypes.LastModifiedBy]: [
        'Last modified by',
        'last updated by',
        'who changed',
    ],
    AIButton: ['AI Button', 'AI action', 'smart button'],
    AIPrompt: ['AI Text', 'AI Prompt', 'AI field', 'smart field'],
};
const columnTypeName = (column) => {
    var _a, _b;
    if (!column)
        return '';
    switch (column.uidt) {
        case UITypes.LongText: {
            if ((_a = (0, helperFunctions_1.parseProp)(column.meta)) === null || _a === void 0 ? void 0 : _a.richMode) {
                return exports.UITypesName.RichText;
            }
            if ((0, helperFunctions_1.parseProp)(column.meta)[globals_1.LongTextAiMetaProp]) {
                return exports.UITypesName.AIPrompt;
            }
            return exports.UITypesName[column.uidt];
        }
        case UITypes.Button: {
            if (column.uidt === UITypes.Button &&
                ((_b = column === null || column === void 0 ? void 0 : column.colOptions) === null || _b === void 0 ? void 0 : _b.type) === 'ai') {
                return exports.UITypesName.AIButton;
            }
            return exports.UITypesName[column.uidt];
        }
        default: {
            return column.uidt ? exports.UITypesName[column.uidt] : '';
        }
    }
};
exports.columnTypeName = columnTypeName;
exports.FieldNameFromUITypes = {
    [UITypes.ID]: 'ID',
    [UITypes.LinkToAnotherRecord]: '{TableName}',
    [UITypes.ForeignKey]: 'Foreign key',
    [UITypes.Lookup]: '{FieldName} (from {TableName})',
    [UITypes.SingleLineText]: 'Text',
    [UITypes.LongText]: 'Notes',
    [UITypes.Attachment]: 'Attachment',
    [UITypes.Checkbox]: 'Done',
    [UITypes.MultiSelect]: 'Tags',
    [UITypes.SingleSelect]: 'Status',
    [UITypes.Collaborator]: 'User',
    [UITypes.Date]: 'Date',
    [UITypes.Year]: 'Year',
    [UITypes.Time]: 'Time',
    [UITypes.PhoneNumber]: 'Phone',
    [UITypes.GeoData]: 'Geo data',
    [UITypes.Email]: 'Email',
    [UITypes.URL]: 'URL',
    [UITypes.Number]: 'Number',
    [UITypes.Decimal]: 'Decimal',
    [UITypes.Currency]: 'Currency',
    [UITypes.Percent]: 'Percent',
    [UITypes.Duration]: 'Duration',
    [UITypes.Rating]: 'Rating',
    [UITypes.Formula]: 'Formula',
    [UITypes.Rollup]: '{RollupFunction}({FieldName}) from {TableName}',
    [UITypes.Count]: 'Count',
    [UITypes.DateTime]: 'Date time',
    [UITypes.CreatedTime]: 'Created time',
    [UITypes.LastModifiedTime]: 'Last modified time',
    [UITypes.AutoNumber]: 'Auto number',
    [UITypes.Geometry]: 'Geometry',
    [UITypes.JSON]: 'JSON',
    [UITypes.SpecificDBType]: 'Specific DB type',
    [UITypes.Barcode]: 'Barcode',
    [UITypes.QrCode]: 'Qr code',
    [UITypes.Button]: 'Button',
    [UITypes.Links]: '{TableName}',
    [UITypes.User]: 'User',
    [UITypes.CreatedBy]: 'Created by',
    [UITypes.LastModifiedBy]: 'Last modified by',
    [UITypes.Order]: 'Order',
    [UITypes.Meta]: 'Row Meta',
};
exports.numericUITypes = [
    UITypes.Duration,
    UITypes.Currency,
    UITypes.Percent,
    UITypes.Number,
    UITypes.Decimal,
    UITypes.Rating,
    UITypes.Rollup,
    UITypes.Year,
    UITypes.Links,
    UITypes.ID,
];
function isNumericCol(col) {
    return exports.numericUITypes.includes((typeof col === 'object' ? col === null || col === void 0 ? void 0 : col.uidt : col));
}
function isVirtualCol(col) {
    return [
        // Shouldn't be treated as virtual column (Issue with SQL View column data display)
        // UITypes.SpecificDBType,
        UITypes.LinkToAnotherRecord,
        UITypes.Formula,
        UITypes.QrCode,
        UITypes.Barcode,
        UITypes.Rollup,
        UITypes.Lookup,
        UITypes.Links,
        UITypes.CreatedTime,
        UITypes.LastModifiedTime,
        UITypes.CreatedBy,
        UITypes.LastModifiedBy,
        UITypes.Button,
        // UITypes.Count,
    ].includes((typeof col === 'object' ? col === null || col === void 0 ? void 0 : col.uidt : col));
}
function isAIPromptCol(col) {
    var _a;
    return (col.uidt === UITypes.LongText &&
        ((_a = (0, helperFunctions_1.parseProp)(col === null || col === void 0 ? void 0 : col.meta)) === null || _a === void 0 ? void 0 : _a[globals_1.LongTextAiMetaProp]));
}
function isCreatedOrLastModifiedTimeCol(col) {
    return [UITypes.CreatedTime, UITypes.LastModifiedTime].includes((typeof col === 'object' ? col === null || col === void 0 ? void 0 : col.uidt : col));
}
function isCreatedOrLastModifiedByCol(col) {
    return [UITypes.CreatedBy, UITypes.LastModifiedBy].includes((typeof col === 'object' ? col === null || col === void 0 ? void 0 : col.uidt : col));
}
function isOrderCol(col) {
    return [UITypes.Order].includes((typeof col === 'object' ? col === null || col === void 0 ? void 0 : col.uidt : col));
}
function isActionButtonCol(col) {
    var _a;
    return (col.uidt === UITypes.Button &&
        [
            Api_1.ButtonActionsType.Script,
            Api_1.ButtonActionsType.Webhook,
            Api_1.ButtonActionsType.Ai,
        ].includes((_a = col === null || col === void 0 ? void 0 : col.colOptions) === null || _a === void 0 ? void 0 : _a.type));
}
function isHiddenCol(col, tableMeta) {
    var _a, _b;
    if (!col.system)
        return false;
    // hide belongs to column in mm tables only
    if (col.uidt === UITypes.LinkToAnotherRecord) {
        if (((_a = col.colOptions) === null || _a === void 0 ? void 0 : _a.type) === globals_1.RelationTypes.BELONGS_TO && (tableMeta === null || tableMeta === void 0 ? void 0 : tableMeta.mm)) {
            return true;
        }
        // hide system columns in other tables which are has-many used for mm
        return ((_b = col.colOptions) === null || _b === void 0 ? void 0 : _b.type) === globals_1.RelationTypes.HAS_MANY;
    }
    if (col.uidt === UITypes.Order || col.uidt === UITypes.Meta) {
        return true;
    }
    return [UITypes.CreatedBy, UITypes.LastModifiedBy].includes(col.uidt);
}
function isLinksOrLTAR(colOrUidt) {
    return [UITypes.LinkToAnotherRecord, UITypes.Links].includes((typeof colOrUidt === 'object' ? colOrUidt === null || colOrUidt === void 0 ? void 0 : colOrUidt.uidt : colOrUidt));
}
function isSelfLinkCol(col) {
    var _a;
    return (isLinksOrLTAR(col) &&
        col.system &&
        // except has-many all other relation types are self link
        // has-many system column get created to mm table only
        ((_a = col.colOptions) === null || _a === void 0 ? void 0 : _a.type) !== globals_1.RelationTypes.HAS_MANY);
}
const getEquivalentUIType = ({ formulaColumn, }) => {
    var _a, _b;
    switch ((_b = (_a = formulaColumn === null || formulaColumn === void 0 ? void 0 : formulaColumn.colOptions) === null || _a === void 0 ? void 0 : _a.parsed_tree) === null || _b === void 0 ? void 0 : _b.dataType) {
        case enums_1.FormulaDataTypes.NUMERIC:
            return UITypes.Number;
        case enums_1.FormulaDataTypes.DATE:
            return UITypes.DateTime;
        case enums_1.FormulaDataTypes.LOGICAL:
        case enums_1.FormulaDataTypes.COND_EXP:
        case enums_1.FormulaDataTypes.BOOLEAN:
            return UITypes.Checkbox;
    }
};
exports.getEquivalentUIType = getEquivalentUIType;
const isSelectTypeCol = (colOrUidt) => {
    return [UITypes.SingleSelect, UITypes.MultiSelect, UITypes.User].includes((typeof colOrUidt === 'object' ? colOrUidt === null || colOrUidt === void 0 ? void 0 : colOrUidt.uidt : colOrUidt));
};
exports.isSelectTypeCol = isSelectTypeCol;
const isInUIType = (colOrUidt, types) => {
    const uidt = ((typeof colOrUidt === 'object' ? colOrUidt === null || colOrUidt === void 0 ? void 0 : colOrUidt.uidt : colOrUidt));
    return Array.isArray(types) ? types.includes(uidt) : uidt === types;
};
exports.isInUIType = isInUIType;
exports.default = UITypes;
exports.readonlyMetaAllowedTypes = [
    UITypes.Lookup,
    UITypes.Rollup,
    UITypes.Formula,
    UITypes.Button,
    UITypes.Barcode,
    UITypes.QrCode,
];
exports.partialUpdateAllowedTypes = [
    // Single/Multi select is disabled for now since it involves updating type in some cases
    // UITypes.SingleSelect,
    // UITypes.MultiSelect,
    UITypes.Checkbox,
    UITypes.Number,
    UITypes.Decimal,
    UITypes.Currency,
    UITypes.Percent,
    UITypes.Duration,
    UITypes.Rating,
    UITypes.DateTime,
    UITypes.Date,
    UITypes.Time,
    UITypes.CreatedTime,
    UITypes.LastModifiedTime,
    UITypes.LinkToAnotherRecord,
    UITypes.Links,
    UITypes.PhoneNumber,
    UITypes.Email,
    UITypes.URL,
];
const getUITypesForFormulaDataType = (dataType) => {
    switch (dataType) {
        case enums_1.FormulaDataTypes.NUMERIC:
            return [
                UITypes.Decimal,
                UITypes.Currency,
                UITypes.Percent,
                UITypes.Rating,
                UITypes.Time,
            ];
        case enums_1.FormulaDataTypes.DATE:
            return [UITypes.DateTime, UITypes.Date, UITypes.Time];
        case enums_1.FormulaDataTypes.BOOLEAN:
        case enums_1.FormulaDataTypes.COND_EXP:
            return [UITypes.Checkbox];
        case enums_1.FormulaDataTypes.STRING:
            return [UITypes.Email, UITypes.URL, UITypes.PhoneNumber];
        default:
            return [];
    }
};
exports.getUITypesForFormulaDataType = getUITypesForFormulaDataType;
const isSupportedDisplayValueColumn = (column) => {
    var _a;
    if (!(column === null || column === void 0 ? void 0 : column.uidt))
        return false;
    switch (column.uidt) {
        case UITypes.SingleLineText:
        case UITypes.Date:
        case UITypes.DateTime:
        case UITypes.Time:
        case UITypes.Year:
        case UITypes.PhoneNumber:
        case UITypes.Email:
        case UITypes.URL:
        case UITypes.Number:
        case UITypes.Currency:
        case UITypes.Percent:
        case UITypes.Duration:
        case UITypes.Decimal:
        case UITypes.Formula: {
            return true;
        }
        case UITypes.LongText: {
            if (((_a = (0, helperFunctions_1.parseProp)(column.meta)) === null || _a === void 0 ? void 0 : _a.richMode) ||
                (0, helperFunctions_1.parseProp)(column.meta)[globals_1.LongTextAiMetaProp]) {
                return false;
            }
            return true;
        }
        default: {
            return false;
        }
    }
};
exports.isSupportedDisplayValueColumn = isSupportedDisplayValueColumn;
exports.checkboxIconList = [
    {
        checked: 'mdi-check-bold',
        unchecked: 'mdi-crop-square',
        label: 'square',
    },
    {
        checked: 'mdi-check-circle-outline',
        unchecked: 'mdi-checkbox-blank-circle-outline',
        label: 'circle-check',
    },
    {
        checked: 'mdi-star',
        unchecked: 'mdi-star-outline',
        label: 'star',
    },
    {
        checked: 'mdi-heart',
        unchecked: 'mdi-heart-outline',
        label: 'heart',
    },
    {
        checked: 'mdi-moon-full',
        unchecked: 'mdi-moon-new',
        label: 'circle-filled',
    },
    {
        checked: 'mdi-thumb-up',
        unchecked: 'mdi-thumb-up-outline',
        label: 'thumbs-up',
    },
    {
        checked: 'mdi-flag',
        unchecked: 'mdi-flag-outline',
        label: 'flag',
    },
];
exports.checkboxIconListMap = exports.checkboxIconList.reduce((acc, curr) => {
    acc[curr.label] = curr;
    return acc;
}, {});
exports.ratingIconList = [
    {
        full: 'mdi-star',
        empty: 'mdi-star-outline',
        label: 'star',
    },
    {
        full: 'mdi-heart',
        empty: 'mdi-heart-outline',
        label: 'heart',
    },
    {
        full: 'mdi-moon-full',
        empty: 'mdi-moon-new',
        label: 'circle-filled',
    },
    {
        full: 'mdi-thumb-up',
        empty: 'mdi-thumb-up-outline',
        label: 'thumbs-up',
    },
    {
        full: 'mdi-flag',
        empty: 'mdi-flag-outline',
        label: 'flag',
    },
];
exports.ratingIconListMap = exports.ratingIconList.reduce((acc, curr) => {
    acc[curr.label] = curr;
    return acc;
}, {});
exports.durationOptions = [
    {
        id: 0,
        title: 'h:mm',
        example: '(e.g. 1:23)',
        regex: /(\d+)(?::(\d+))?/,
    },
    {
        id: 1,
        title: 'h:mm:ss',
        example: '(e.g. 3:45, 1:23:40)',
        regex: /(?=\d)(\d+)?(?::(\d+))?(?::(\d+))?/,
    },
    {
        id: 2,
        title: 'h:mm:ss.s',
        example: '(e.g. 3:34.6, 1:23:40.0)',
        regex: /(\d+)?(?::(\d+))?(?::(\d+))?(?:.(\d{0,4})?)?/,
    },
    {
        id: 3,
        title: 'h:mm:ss.ss',
        example: '(e.g. 3.45.67, 1:23:40.00)',
        regex: /(\d+)?(?::(\d+))?(?::(\d+))?(?:.(\d{0,4})?)?/,
    },
    {
        id: 4,
        title: 'h:mm:ss.sss',
        example: '(e.g. 3.45.678, 1:23:40.000)',
        regex: /(\d+)?(?::(\d+))?(?::(\d+))?(?:.(\d{0,4})?)?/,
    },
];
/**
 * Checks if a given column is read-only.
 * A column is considered read-only if it belongs to specific UI types
 * (e.g., Lookup, Rollup, Formula, etc.) or if it represents system-generated
 * metadata such as created/modified timestamps or ordering information.
 *
 * @param {ColumnType} column - The column to check.
 * @returns {boolean} - Returns `true` if the column is read-only, otherwise `false`.
 */
const isReadOnlyColumn = (column) => {
    var _a;
    return (
    // Check if the column belongs to a predefined set of read-only UI types
    [
        UITypes.Lookup,
        UITypes.Rollup,
        UITypes.Formula,
        UITypes.Button,
        UITypes.Barcode,
        UITypes.QrCode,
        UITypes.ForeignKey,
    ].includes(column.uidt) ||
        // Check if the column is a system-generated user tracking field (CreatedBy, LastModifiedBy)
        isCreatedOrLastModifiedByCol(column) ||
        // Check if the column is a system-generated timestamp field (CreatedTime, LastModifiedTime)
        isCreatedOrLastModifiedTimeCol(column) ||
        // Check if the column is used for row ordering
        isOrderCol(column) ||
        // if primary key and auto generated then treat as readonly
        (column.pk && (column.ai || ((_a = (0, helperFunctions_1.parseProp)(column.meta)) === null || _a === void 0 ? void 0 : _a.ag))));
};
exports.isReadOnlyColumn = isReadOnlyColumn;
/**
 * Determines whether a given column type represents a Date or DateTime field.
 *
 * @param column - The column type to check.
 * @returns `true` if the column is a Date, DateTime, CreatedTime, or LastModifiedTime field;
 *          `true` if it is a Formula column that evaluates to DateTime;
 *          otherwise, `false`.
 */
const isDateOrDateTimeCol = (column) => {
    // Check if the column's UI type is one of the predefined date-related types
    if ([
        UITypes.Date,
        UITypes.DateTime,
        UITypes.CreatedTime,
        UITypes.LastModifiedTime,
    ].includes(column.uidt)) {
        return true;
    }
    // If the column is a Formula, determine if its evaluated type is DateTime
    if (column.uidt === UITypes.Formula) {
        return (0, exports.getEquivalentUIType)({ formulaColumn: column }) === UITypes.DateTime;
    }
    return false;
};
exports.isDateOrDateTimeCol = isDateOrDateTimeCol;
exports.customLinkSupportedTypes = [
    UITypes.SingleSelect,
    UITypes.SingleLineText,
    UITypes.Number,
    UITypes.Decimal,
    UITypes.Email,
    UITypes.PhoneNumber,
    UITypes.URL,
    UITypes.ID,
    UITypes.ForeignKey,
];
// column types that are not shown in the GUI
exports.hiddenColumnTypes = [UITypes.Meta];
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiVUlUeXBlcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9saWIvVUlUeXBlcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUF1VUEsb0NBVUM7QUFFRCxvQ0F3QkM7QUFFRCxzQ0FLQztBQUVELHdFQVVDO0FBRUQsb0VBVUM7QUFFRCxnQ0FVQztBQUVELDhDQWFDO0FBRUQsa0NBeUJDO0FBRUQsc0NBTUM7QUFFRCxzQ0FVQztBQXBkRCwrQkFNZTtBQUNmLDJDQUFtRDtBQUNuRCwyQ0FBa0U7QUFDbEUsdURBQThDO0FBRTlDLElBQUssT0E0Q0o7QUE1Q0QsV0FBSyxPQUFPO0lBQ1Ysb0JBQVMsQ0FBQTtJQUNULHNEQUEyQyxDQUFBO0lBQzNDLG9DQUF5QixDQUFBO0lBQ3pCLDRCQUFpQixDQUFBO0lBQ2pCLDRDQUFpQyxDQUFBO0lBQ2pDLGdDQUFxQixDQUFBO0lBQ3JCLG9DQUF5QixDQUFBO0lBQ3pCLGdDQUFxQixDQUFBO0lBQ3JCLHNDQUEyQixDQUFBO0lBQzNCLHdDQUE2QixDQUFBO0lBQzdCLHdDQUE2QixDQUFBO0lBQzdCLHdCQUFhLENBQUE7SUFDYix3QkFBYSxDQUFBO0lBQ2Isd0JBQWEsQ0FBQTtJQUNiLHNDQUEyQixDQUFBO0lBQzNCLDhCQUFtQixDQUFBO0lBQ25CLDBCQUFlLENBQUE7SUFDZixzQkFBVyxDQUFBO0lBQ1gsNEJBQWlCLENBQUE7SUFDakIsOEJBQW1CLENBQUE7SUFDbkIsZ0NBQXFCLENBQUE7SUFDckIsOEJBQW1CLENBQUE7SUFDbkIsZ0NBQXFCLENBQUE7SUFDckIsNEJBQWlCLENBQUE7SUFDakIsOEJBQW1CLENBQUE7SUFDbkIsNEJBQWlCLENBQUE7SUFDakIsMEJBQWUsQ0FBQTtJQUNmLGdDQUFxQixDQUFBO0lBQ3JCLHNDQUEyQixDQUFBO0lBQzNCLGdEQUFxQyxDQUFBO0lBQ3JDLG9DQUF5QixDQUFBO0lBQ3pCLGdDQUFxQixDQUFBO0lBQ3JCLHdCQUFhLENBQUE7SUFDYiw0Q0FBaUMsQ0FBQTtJQUNqQyw4QkFBbUIsQ0FBQTtJQUNuQiw0QkFBaUIsQ0FBQTtJQUNqQiw0QkFBaUIsQ0FBQTtJQUNqQiwwQkFBZSxDQUFBO0lBQ2Ysd0JBQWEsQ0FBQTtJQUNiLGtDQUF1QixDQUFBO0lBQ3ZCLDRDQUFpQyxDQUFBO0lBQ2pDLDBCQUFlLENBQUE7SUFDZix3QkFBYSxDQUFBO0FBQ2YsQ0FBQyxFQTVDSSxPQUFPLEtBQVAsT0FBTyxRQTRDWDtBQUVZLFFBQUEsV0FBVyxHQUFHO0lBQ3pCLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxFQUFFLElBQUk7SUFDbEIsQ0FBQyxPQUFPLENBQUMsbUJBQW1CLENBQUMsRUFBRSx3QkFBd0I7SUFDdkQsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLEVBQUUsYUFBYTtJQUNuQyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsRUFBRSxPQUFPO0lBQ3hCLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFLFFBQVE7SUFDMUIsQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLEVBQUUsa0JBQWtCO0lBQzVDLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFLFdBQVc7SUFDL0IsUUFBUSxFQUFFLFdBQVc7SUFDckIsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLEVBQUUsWUFBWTtJQUNsQyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsRUFBRSxVQUFVO0lBQzlCLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxFQUFFLGNBQWM7SUFDckMsQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLEVBQUUsZUFBZTtJQUN2QyxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsRUFBRSxjQUFjO0lBQ3RDLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxFQUFFLE1BQU07SUFDdEIsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTTtJQUN0QixDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRSxNQUFNO0lBQ3RCLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxFQUFFLGNBQWM7SUFDckMsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLEVBQUUsVUFBVTtJQUM3QixDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsRUFBRSxPQUFPO0lBQ3hCLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxFQUFFLEtBQUs7SUFDcEIsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUUsUUFBUTtJQUMxQixDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRSxTQUFTO0lBQzVCLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFLFVBQVU7SUFDOUIsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLEVBQUUsU0FBUztJQUM1QixDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsRUFBRSxVQUFVO0lBQzlCLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFLFFBQVE7SUFDMUIsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLEVBQUUsU0FBUztJQUM1QixDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRSxRQUFRO0lBQzFCLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFLE9BQU87SUFDeEIsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLEVBQUUsV0FBVztJQUMvQixDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsRUFBRSxjQUFjO0lBQ3JDLENBQUMsT0FBTyxDQUFDLGdCQUFnQixDQUFDLEVBQUUsb0JBQW9CO0lBQ2hELENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxFQUFFLGFBQWE7SUFDbkMsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLEVBQUUsVUFBVTtJQUM5QixDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRSxNQUFNO0lBQ3RCLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxFQUFFLGtCQUFrQjtJQUM1QyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRSxTQUFTO0lBQzVCLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFLFNBQVM7SUFDM0IsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUUsUUFBUTtJQUMxQixDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsRUFBRSxPQUFPO0lBQ3hCLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxFQUFFLE1BQU07SUFDdEIsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLEVBQUUsWUFBWTtJQUNqQyxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsRUFBRSxrQkFBa0I7SUFDNUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUUsVUFBVTtJQUMxQixRQUFRLEVBQUUsV0FBVztJQUNyQixRQUFRLEVBQUUsU0FBUztDQUNwQixDQUFDO0FBRVcsUUFBQSxrQkFBa0IsR0FBRztJQUNoQyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLElBQUksRUFBRSxlQUFlLEVBQUUsZUFBZSxDQUFDO0lBQ3RELENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxFQUFFO1FBQ3BCLGFBQWE7UUFDYixpQkFBaUI7UUFDakIsZ0JBQWdCO1FBQ2hCLFdBQVc7UUFDWCxjQUFjO0tBQ2Y7SUFDRCxDQUFDLE9BQU8sQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFO1FBQzdCLHdCQUF3QjtRQUN4QixpQkFBaUI7UUFDakIsZ0JBQWdCO1FBQ2hCLFdBQVc7UUFDWCxjQUFjO0tBQ2Y7SUFDRCxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLFFBQVEsRUFBRSxXQUFXLEVBQUUsZ0JBQWdCLEVBQUUsZ0JBQWdCLENBQUM7SUFDN0UsQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLEVBQUU7UUFDeEIsa0JBQWtCO1FBQ2xCLE1BQU07UUFDTixZQUFZO1FBQ1osTUFBTTtRQUNOLE9BQU87S0FDUjtJQUNELENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFO1FBQ2xCLFdBQVc7UUFDWCxXQUFXO1FBQ1gsT0FBTztRQUNQLGFBQWE7UUFDYixVQUFVO1FBQ1YsTUFBTTtRQUNOLFdBQVc7UUFDWCxnQkFBZ0I7UUFDaEIsYUFBYTtRQUNiLFdBQVc7S0FDWjtJQUNELENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUMsWUFBWSxFQUFFLE1BQU0sRUFBRSxVQUFVLEVBQUUsT0FBTyxFQUFFLFFBQVEsQ0FBQztJQUMzRSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDLFVBQVUsRUFBRSxRQUFRLEVBQUUsWUFBWSxFQUFFLFdBQVcsRUFBRSxNQUFNLENBQUM7SUFDN0UsQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLEVBQUU7UUFDckIsY0FBYztRQUNkLGtCQUFrQjtRQUNsQixNQUFNO1FBQ04sWUFBWTtRQUNaLE1BQU07UUFDTixTQUFTO1FBQ1QsUUFBUTtLQUNUO0lBQ0QsQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLEVBQUU7UUFDdEIsZUFBZTtRQUNmLFVBQVU7UUFDVixTQUFTO1FBQ1QsUUFBUTtRQUNSLFVBQVU7UUFDVixRQUFRO1FBQ1IsVUFBVTtLQUNYO0lBQ0QsQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLEVBQUU7UUFDdEIsY0FBYztRQUNkLGFBQWE7UUFDYixRQUFRO1FBQ1IsVUFBVTtRQUNWLE9BQU87S0FDUjtJQUNELENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsTUFBTSxFQUFFLFVBQVUsRUFBRSxVQUFVLEVBQUUsWUFBWSxDQUFDO0lBQzlELENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDO0lBQ3hCLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsTUFBTSxFQUFFLGFBQWEsRUFBRSxNQUFNLENBQUM7SUFDL0MsQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLEVBQUUsQ0FBQyxjQUFjLEVBQUUsT0FBTyxFQUFFLGdCQUFnQixFQUFFLFFBQVEsQ0FBQztJQUM1RSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLFVBQVUsRUFBRSxVQUFVLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxhQUFhLENBQUM7SUFDNUUsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsZUFBZSxFQUFFLGVBQWUsQ0FBQztJQUM1RCxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUUsTUFBTSxFQUFFLGFBQWEsQ0FBQztJQUN4RCxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLFFBQVEsRUFBRSxVQUFVLEVBQUUsUUFBUSxFQUFFLE9BQU8sQ0FBQztJQUMzRCxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxnQkFBZ0IsRUFBRSxnQkFBZ0IsQ0FBQztJQUNsRSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsRUFBRTtRQUNsQixVQUFVO1FBQ1YsT0FBTztRQUNQLE9BQU87UUFDUCxNQUFNO1FBQ04sUUFBUTtRQUNSLFNBQVM7S0FDVjtJQUNELENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsU0FBUyxFQUFFLFlBQVksRUFBRSxZQUFZLEVBQUUsVUFBVSxDQUFDO0lBQ3RFLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFO1FBQ2xCLFVBQVU7UUFDVixZQUFZO1FBQ1osY0FBYztRQUNkLGNBQWM7S0FDZjtJQUNELENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLE9BQU8sRUFBRSxPQUFPLEVBQUUsUUFBUSxFQUFFLFVBQVUsQ0FBQztJQUNwRSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRTtRQUNqQixTQUFTO1FBQ1QsYUFBYTtRQUNiLGdCQUFnQjtRQUNoQixnQkFBZ0I7S0FDakI7SUFDRCxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRTtRQUNoQixRQUFRO1FBQ1IsU0FBUztRQUNULDRCQUE0QjtRQUM1QixXQUFXO0tBQ1o7SUFDRCxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsRUFBRSxhQUFhLENBQUM7SUFDOUQsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxXQUFXLEVBQUUsZUFBZSxFQUFFLFdBQVcsQ0FBQztJQUMvRCxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsRUFBRSxDQUFDLGNBQWMsRUFBRSxjQUFjLEVBQUUsWUFBWSxDQUFDO0lBQ3JFLENBQUMsT0FBTyxDQUFDLGdCQUFnQixDQUFDLEVBQUU7UUFDMUIsb0JBQW9CO1FBQ3BCLGNBQWM7UUFDZCxhQUFhO0tBQ2Q7SUFDRCxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxnQkFBZ0IsRUFBRSxpQkFBaUIsQ0FBQztJQUMxRSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDLFVBQVUsRUFBRSxPQUFPLEVBQUUsa0JBQWtCLENBQUM7SUFDN0QsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsaUJBQWlCLEVBQUUsY0FBYyxDQUFDO0lBQzNELENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxFQUFFO1FBQ3hCLGtCQUFrQjtRQUNsQixlQUFlO1FBQ2YsYUFBYTtLQUNkO0lBQ0QsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxTQUFTLEVBQUUsY0FBYyxFQUFFLFdBQVcsQ0FBQztJQUMzRCxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUUsV0FBVyxDQUFDO0lBQ3JELENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLGVBQWUsRUFBRSxjQUFjLENBQUM7SUFDN0QsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsZUFBZSxFQUFFLGFBQWEsQ0FBQztJQUMxRCxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxRQUFRLEVBQUUsYUFBYSxFQUFFLFVBQVUsRUFBRSxPQUFPLENBQUM7SUFDdEUsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQyxZQUFZLEVBQUUsUUFBUSxFQUFFLGFBQWEsQ0FBQztJQUM1RCxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsRUFBRTtRQUN4QixrQkFBa0I7UUFDbEIsaUJBQWlCO1FBQ2pCLGFBQWE7S0FDZDtJQUNELFFBQVEsRUFBRSxDQUFDLFdBQVcsRUFBRSxXQUFXLEVBQUUsY0FBYyxDQUFDO0lBQ3BELFFBQVEsRUFBRSxDQUFDLFNBQVMsRUFBRSxXQUFXLEVBQUUsVUFBVSxFQUFFLGFBQWEsQ0FBQztDQUM5RCxDQUFDO0FBRUssTUFBTSxjQUFjLEdBQUcsQ0FBQyxNQUFtQixFQUFFLEVBQUU7O0lBQ3BELElBQUksQ0FBQyxNQUFNO1FBQUUsT0FBTyxFQUFFLENBQUM7SUFFdkIsUUFBUSxNQUFNLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDcEIsS0FBSyxPQUFPLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztZQUN0QixJQUFJLE1BQUEsSUFBQSwyQkFBUyxFQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsMENBQUUsUUFBUSxFQUFFLENBQUM7Z0JBQ3JDLE9BQU8sbUJBQVcsQ0FBQyxRQUFRLENBQUM7WUFDOUIsQ0FBQztZQUVELElBQUksSUFBQSwyQkFBUyxFQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyw0QkFBa0IsQ0FBQyxFQUFFLENBQUM7Z0JBQy9DLE9BQU8sbUJBQVcsQ0FBQyxRQUFRLENBQUM7WUFDOUIsQ0FBQztZQUVELE9BQU8sbUJBQVcsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDbEMsQ0FBQztRQUNELEtBQUssT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7WUFDcEIsSUFDRSxNQUFNLENBQUMsSUFBSSxLQUFLLE9BQU8sQ0FBQyxNQUFNO2dCQUM5QixDQUFBLE1BQUMsTUFBTSxhQUFOLE1BQU0sdUJBQU4sTUFBTSxDQUFFLFVBQWtCLDBDQUFFLElBQUksTUFBSyxJQUFJLEVBQzFDLENBQUM7Z0JBQ0QsT0FBTyxtQkFBVyxDQUFDLFFBQVEsQ0FBQztZQUM5QixDQUFDO1lBRUQsT0FBTyxtQkFBVyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNsQyxDQUFDO1FBQ0QsT0FBTyxDQUFDLENBQUMsQ0FBQztZQUNSLE9BQU8sTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsbUJBQVcsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUNyRCxDQUFDO0lBQ0gsQ0FBQztBQUNILENBQUMsQ0FBQztBQTdCVyxRQUFBLGNBQWMsa0JBNkJ6QjtBQUVXLFFBQUEsb0JBQW9CLEdBQTRCO0lBQzNELENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxFQUFFLElBQUk7SUFDbEIsQ0FBQyxPQUFPLENBQUMsbUJBQW1CLENBQUMsRUFBRSxhQUFhO0lBQzVDLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxFQUFFLGFBQWE7SUFDbkMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUUsZ0NBQWdDO0lBQ2xELENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxFQUFFLE1BQU07SUFDaEMsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTztJQUMzQixDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsRUFBRSxZQUFZO0lBQ2xDLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFLE1BQU07SUFDMUIsQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLEVBQUUsTUFBTTtJQUM3QixDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsRUFBRSxRQUFRO0lBQ2hDLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxFQUFFLE1BQU07SUFDOUIsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTTtJQUN0QixDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRSxNQUFNO0lBQ3RCLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxFQUFFLE1BQU07SUFDdEIsQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLEVBQUUsT0FBTztJQUM5QixDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRSxVQUFVO0lBQzdCLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFLE9BQU87SUFDeEIsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUUsS0FBSztJQUNwQixDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRSxRQUFRO0lBQzFCLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxFQUFFLFNBQVM7SUFDNUIsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLEVBQUUsVUFBVTtJQUM5QixDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRSxTQUFTO0lBQzVCLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFLFVBQVU7SUFDOUIsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUUsUUFBUTtJQUMxQixDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRSxTQUFTO0lBQzVCLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFLGdEQUFnRDtJQUNsRSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsRUFBRSxPQUFPO0lBQ3hCLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFLFdBQVc7SUFDL0IsQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLEVBQUUsY0FBYztJQUNyQyxDQUFDLE9BQU8sQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLG9CQUFvQjtJQUNoRCxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsRUFBRSxhQUFhO0lBQ25DLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFLFVBQVU7SUFDOUIsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTTtJQUN0QixDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsRUFBRSxrQkFBa0I7SUFDNUMsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLEVBQUUsU0FBUztJQUM1QixDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRSxTQUFTO0lBQzNCLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFLFFBQVE7SUFDMUIsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUUsYUFBYTtJQUM5QixDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRSxNQUFNO0lBQ3RCLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxFQUFFLFlBQVk7SUFDakMsQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLEVBQUUsa0JBQWtCO0lBQzVDLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFLE9BQU87SUFDeEIsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUUsVUFBVTtDQUMzQixDQUFDO0FBRVcsUUFBQSxjQUFjLEdBQUc7SUFDNUIsT0FBTyxDQUFDLFFBQVE7SUFDaEIsT0FBTyxDQUFDLFFBQVE7SUFDaEIsT0FBTyxDQUFDLE9BQU87SUFDZixPQUFPLENBQUMsTUFBTTtJQUNkLE9BQU8sQ0FBQyxPQUFPO0lBQ2YsT0FBTyxDQUFDLE1BQU07SUFDZCxPQUFPLENBQUMsTUFBTTtJQUNkLE9BQU8sQ0FBQyxJQUFJO0lBQ1osT0FBTyxDQUFDLEtBQUs7SUFDYixPQUFPLENBQUMsRUFBRTtDQUNYLENBQUM7QUFFRixTQUFnQixZQUFZLENBQzFCLEdBSWM7SUFFZCxPQUFPLHNCQUFjLENBQUMsUUFBUSxDQUNuQixDQUFDLE9BQU8sR0FBRyxLQUFLLFFBQVEsQ0FBQyxDQUFDLENBQUMsR0FBRyxhQUFILEdBQUcsdUJBQUgsR0FBRyxDQUFFLElBQUksQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQ3JELENBQUM7QUFDSixDQUFDO0FBRUQsU0FBZ0IsWUFBWSxDQUMxQixHQUljO0lBRWQsT0FBTztRQUNMLG1GQUFtRjtRQUNuRiwwQkFBMEI7UUFDMUIsT0FBTyxDQUFDLG1CQUFtQjtRQUMzQixPQUFPLENBQUMsT0FBTztRQUNmLE9BQU8sQ0FBQyxNQUFNO1FBQ2QsT0FBTyxDQUFDLE9BQU87UUFDZixPQUFPLENBQUMsTUFBTTtRQUNkLE9BQU8sQ0FBQyxNQUFNO1FBQ2QsT0FBTyxDQUFDLEtBQUs7UUFDYixPQUFPLENBQUMsV0FBVztRQUNuQixPQUFPLENBQUMsZ0JBQWdCO1FBQ3hCLE9BQU8sQ0FBQyxTQUFTO1FBQ2pCLE9BQU8sQ0FBQyxjQUFjO1FBQ3RCLE9BQU8sQ0FBQyxNQUFNO1FBQ2QsaUJBQWlCO0tBQ2xCLENBQUMsUUFBUSxDQUFVLENBQUMsT0FBTyxHQUFHLEtBQUssUUFBUSxDQUFDLENBQUMsQ0FBQyxHQUFHLGFBQUgsR0FBRyx1QkFBSCxHQUFHLENBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO0FBQ25FLENBQUM7QUFFRCxTQUFnQixhQUFhLENBQUMsR0FBK0I7O0lBQzNELE9BQU8sQ0FDTCxHQUFHLENBQUMsSUFBSSxLQUFLLE9BQU8sQ0FBQyxRQUFRO1NBQzdCLE1BQUEsSUFBQSwyQkFBUyxFQUFFLEdBQVcsYUFBWCxHQUFHLHVCQUFILEdBQUcsQ0FBVSxJQUFJLENBQUMsMENBQUcsNEJBQWtCLENBQUMsQ0FBQSxDQUNwRCxDQUFDO0FBQ0osQ0FBQztBQUVELFNBQWdCLDhCQUE4QixDQUM1QyxHQUljO0lBRWQsT0FBTyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsT0FBTyxDQUFDLGdCQUFnQixDQUFDLENBQUMsUUFBUSxDQUNwRCxDQUFDLE9BQU8sR0FBRyxLQUFLLFFBQVEsQ0FBQyxDQUFDLENBQUMsR0FBRyxhQUFILEdBQUcsdUJBQUgsR0FBRyxDQUFFLElBQUksQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQ3JELENBQUM7QUFDSixDQUFDO0FBRUQsU0FBZ0IsNEJBQTRCLENBQzFDLEdBSWM7SUFFZCxPQUFPLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsY0FBYyxDQUFDLENBQUMsUUFBUSxDQUNoRCxDQUFDLE9BQU8sR0FBRyxLQUFLLFFBQVEsQ0FBQyxDQUFDLENBQUMsR0FBRyxhQUFILEdBQUcsdUJBQUgsR0FBRyxDQUFFLElBQUksQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQ3JELENBQUM7QUFDSixDQUFDO0FBRUQsU0FBZ0IsVUFBVSxDQUN4QixHQUljO0lBRWQsT0FBTyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLENBQ3BCLENBQUMsT0FBTyxHQUFHLEtBQUssUUFBUSxDQUFDLENBQUMsQ0FBQyxHQUFHLGFBQUgsR0FBRyx1QkFBSCxHQUFHLENBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FDckQsQ0FBQztBQUNKLENBQUM7QUFFRCxTQUFnQixpQkFBaUIsQ0FDL0IsR0FFQzs7SUFFRCxPQUFPLENBQ0wsR0FBRyxDQUFDLElBQUksS0FBSyxPQUFPLENBQUMsTUFBTTtRQUMzQjtZQUNFLHVCQUFpQixDQUFDLE1BQU07WUFDeEIsdUJBQWlCLENBQUMsT0FBTztZQUN6Qix1QkFBaUIsQ0FBQyxFQUFFO1NBQ3JCLENBQUMsUUFBUSxDQUFDLE1BQUMsR0FBRyxhQUFILEdBQUcsdUJBQUgsR0FBRyxDQUFFLFVBQWtCLDBDQUFFLElBQUksQ0FBQyxDQUMzQyxDQUFDO0FBQ0osQ0FBQztBQUVELFNBQWdCLFdBQVcsQ0FDekIsR0FHQyxFQUNELFNBQTZCOztJQUU3QixJQUFJLENBQUMsR0FBRyxDQUFDLE1BQU07UUFBRSxPQUFPLEtBQUssQ0FBQztJQUU5QiwyQ0FBMkM7SUFDM0MsSUFBSSxHQUFHLENBQUMsSUFBSSxLQUFLLE9BQU8sQ0FBQyxtQkFBbUIsRUFBRSxDQUFDO1FBQzdDLElBQUksQ0FBQSxNQUFBLEdBQUcsQ0FBQyxVQUFVLDBDQUFFLElBQUksTUFBSyx1QkFBYSxDQUFDLFVBQVUsS0FBSSxTQUFTLGFBQVQsU0FBUyx1QkFBVCxTQUFTLENBQUUsRUFBRSxDQUFBLEVBQUUsQ0FBQztZQUN2RSxPQUFPLElBQUksQ0FBQztRQUNkLENBQUM7UUFDRCxxRUFBcUU7UUFDckUsT0FBTyxDQUFBLE1BQUEsR0FBRyxDQUFDLFVBQVUsMENBQUUsSUFBSSxNQUFLLHVCQUFhLENBQUMsUUFBUSxDQUFDO0lBQ3pELENBQUM7SUFFRCxJQUFJLEdBQUcsQ0FBQyxJQUFJLEtBQUssT0FBTyxDQUFDLEtBQUssSUFBSSxHQUFHLENBQUMsSUFBSSxLQUFLLE9BQU8sQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUM1RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRCxPQUFRLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsY0FBYyxDQUFjLENBQUMsUUFBUSxDQUN2RSxHQUFHLENBQUMsSUFBSSxDQUNULENBQUM7QUFDSixDQUFDO0FBRUQsU0FBZ0IsYUFBYSxDQUMzQixTQUFxRTtJQUVyRSxPQUFPLENBQUMsT0FBTyxDQUFDLG1CQUFtQixFQUFFLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLENBQ2pELENBQUMsT0FBTyxTQUFTLEtBQUssUUFBUSxDQUFDLENBQUMsQ0FBQyxTQUFTLGFBQVQsU0FBUyx1QkFBVCxTQUFTLENBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FDdkUsQ0FBQztBQUNKLENBQUM7QUFFRCxTQUFnQixhQUFhLENBQzNCLEdBQXlDOztJQUV6QyxPQUFPLENBQ0wsYUFBYSxDQUFDLEdBQUcsQ0FBQztRQUNsQixHQUFHLENBQUMsTUFBTTtRQUNWLHlEQUF5RDtRQUN6RCxzREFBc0Q7UUFDdEQsQ0FBQSxNQUFDLEdBQUcsQ0FBQyxVQUFzQywwQ0FBRSxJQUFJLE1BQUssdUJBQWEsQ0FBQyxRQUFRLENBQzdFLENBQUM7QUFDSixDQUFDO0FBRU0sTUFBTSxtQkFBbUIsR0FBRyxDQUFDLEVBQ2xDLGFBQWEsR0FHZCxFQUFrQixFQUFFOztJQUNuQixRQUFRLE1BQUEsTUFBQyxhQUFhLGFBQWIsYUFBYSx1QkFBYixhQUFhLENBQUUsVUFBa0IsMENBQUUsV0FBVywwQ0FBRSxRQUFRLEVBQUUsQ0FBQztRQUNsRSxLQUFLLHdCQUFnQixDQUFDLE9BQU87WUFDM0IsT0FBTyxPQUFPLENBQUMsTUFBTSxDQUFDO1FBQ3hCLEtBQUssd0JBQWdCLENBQUMsSUFBSTtZQUN4QixPQUFPLE9BQU8sQ0FBQyxRQUFRLENBQUM7UUFDMUIsS0FBSyx3QkFBZ0IsQ0FBQyxPQUFPLENBQUM7UUFDOUIsS0FBSyx3QkFBZ0IsQ0FBQyxRQUFRLENBQUM7UUFDL0IsS0FBSyx3QkFBZ0IsQ0FBQyxPQUFPO1lBQzNCLE9BQU8sT0FBTyxDQUFDLFFBQVEsQ0FBQztJQUM1QixDQUFDO0FBQ0gsQ0FBQyxDQUFDO0FBZlcsUUFBQSxtQkFBbUIsdUJBZTlCO0FBRUssTUFBTSxlQUFlLEdBQUcsQ0FDN0IsU0FBcUUsRUFDckUsRUFBRTtJQUNGLE9BQU8sQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sQ0FBQyxXQUFXLEVBQUUsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDOUQsQ0FBQyxPQUFPLFNBQVMsS0FBSyxRQUFRLENBQUMsQ0FBQyxDQUFDLFNBQVMsYUFBVCxTQUFTLHVCQUFULFNBQVMsQ0FBRSxJQUFJLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUN2RSxDQUFDO0FBQ0osQ0FBQyxDQUFDO0FBTlcsUUFBQSxlQUFlLG1CQU0xQjtBQUVLLE1BQU0sVUFBVSxHQUFHLENBQ3hCLFNBQXFFLEVBQ3JFLEtBQXVCLEVBQ29DLEVBQUU7SUFDN0QsTUFBTSxJQUFJLEdBQVksQ0FDcEIsQ0FBQyxPQUFPLFNBQVMsS0FBSyxRQUFRLENBQUMsQ0FBQyxDQUFDLFNBQVMsYUFBVCxTQUFTLHVCQUFULFNBQVMsQ0FBRSxJQUFJLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUM5RCxDQUFDO0lBQ0YsT0FBTyxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLElBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLEtBQUssS0FBSyxDQUFDO0FBQzNFLENBQUMsQ0FBQztBQVJXLFFBQUEsVUFBVSxjQVFyQjtBQUVGLGtCQUFlLE9BQU8sQ0FBQztBQUVWLFFBQUEsd0JBQXdCLEdBQUc7SUFDdEMsT0FBTyxDQUFDLE1BQU07SUFDZCxPQUFPLENBQUMsTUFBTTtJQUNkLE9BQU8sQ0FBQyxPQUFPO0lBQ2YsT0FBTyxDQUFDLE1BQU07SUFDZCxPQUFPLENBQUMsT0FBTztJQUNmLE9BQU8sQ0FBQyxNQUFNO0NBQ2YsQ0FBQztBQUVXLFFBQUEseUJBQXlCLEdBQUc7SUFDdkMsd0ZBQXdGO0lBQ3hGLHdCQUF3QjtJQUN4Qix1QkFBdUI7SUFDdkIsT0FBTyxDQUFDLFFBQVE7SUFDaEIsT0FBTyxDQUFDLE1BQU07SUFDZCxPQUFPLENBQUMsT0FBTztJQUNmLE9BQU8sQ0FBQyxRQUFRO0lBQ2hCLE9BQU8sQ0FBQyxPQUFPO0lBQ2YsT0FBTyxDQUFDLFFBQVE7SUFDaEIsT0FBTyxDQUFDLE1BQU07SUFDZCxPQUFPLENBQUMsUUFBUTtJQUNoQixPQUFPLENBQUMsSUFBSTtJQUNaLE9BQU8sQ0FBQyxJQUFJO0lBQ1osT0FBTyxDQUFDLFdBQVc7SUFDbkIsT0FBTyxDQUFDLGdCQUFnQjtJQUN4QixPQUFPLENBQUMsbUJBQW1CO0lBQzNCLE9BQU8sQ0FBQyxLQUFLO0lBQ2IsT0FBTyxDQUFDLFdBQVc7SUFDbkIsT0FBTyxDQUFDLEtBQUs7SUFDYixPQUFPLENBQUMsR0FBRztDQUNaLENBQUM7QUFFSyxNQUFNLDRCQUE0QixHQUFHLENBQzFDLFFBQTBCLEVBQ2YsRUFBRTtJQUNiLFFBQVEsUUFBUSxFQUFFLENBQUM7UUFDakIsS0FBSyx3QkFBZ0IsQ0FBQyxPQUFPO1lBQzNCLE9BQU87Z0JBQ0wsT0FBTyxDQUFDLE9BQU87Z0JBQ2YsT0FBTyxDQUFDLFFBQVE7Z0JBQ2hCLE9BQU8sQ0FBQyxPQUFPO2dCQUNmLE9BQU8sQ0FBQyxNQUFNO2dCQUNkLE9BQU8sQ0FBQyxJQUFJO2FBQ2IsQ0FBQztRQUNKLEtBQUssd0JBQWdCLENBQUMsSUFBSTtZQUN4QixPQUFPLENBQUMsT0FBTyxDQUFDLFFBQVEsRUFBRSxPQUFPLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN4RCxLQUFLLHdCQUFnQixDQUFDLE9BQU8sQ0FBQztRQUM5QixLQUFLLHdCQUFnQixDQUFDLFFBQVE7WUFDNUIsT0FBTyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUM1QixLQUFLLHdCQUFnQixDQUFDLE1BQU07WUFDMUIsT0FBTyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLEdBQUcsRUFBRSxPQUFPLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDM0Q7WUFDRSxPQUFPLEVBQUUsQ0FBQztJQUNkLENBQUM7QUFDSCxDQUFDLENBQUM7QUF0QlcsUUFBQSw0QkFBNEIsZ0NBc0J2QztBQUVLLE1BQU0sNkJBQTZCLEdBQUcsQ0FBQyxNQUEyQixFQUFFLEVBQUU7O0lBQzNFLElBQUksQ0FBQyxDQUFBLE1BQU0sYUFBTixNQUFNLHVCQUFOLE1BQU0sQ0FBRSxJQUFJLENBQUE7UUFBRSxPQUFPLEtBQUssQ0FBQztJQUVoQyxRQUFRLE1BQU0sQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNwQixLQUFLLE9BQU8sQ0FBQyxjQUFjLENBQUM7UUFDNUIsS0FBSyxPQUFPLENBQUMsSUFBSSxDQUFDO1FBQ2xCLEtBQUssT0FBTyxDQUFDLFFBQVEsQ0FBQztRQUN0QixLQUFLLE9BQU8sQ0FBQyxJQUFJLENBQUM7UUFDbEIsS0FBSyxPQUFPLENBQUMsSUFBSSxDQUFDO1FBQ2xCLEtBQUssT0FBTyxDQUFDLFdBQVcsQ0FBQztRQUN6QixLQUFLLE9BQU8sQ0FBQyxLQUFLLENBQUM7UUFDbkIsS0FBSyxPQUFPLENBQUMsR0FBRyxDQUFDO1FBQ2pCLEtBQUssT0FBTyxDQUFDLE1BQU0sQ0FBQztRQUNwQixLQUFLLE9BQU8sQ0FBQyxRQUFRLENBQUM7UUFDdEIsS0FBSyxPQUFPLENBQUMsT0FBTyxDQUFDO1FBQ3JCLEtBQUssT0FBTyxDQUFDLFFBQVEsQ0FBQztRQUN0QixLQUFLLE9BQU8sQ0FBQyxPQUFPLENBQUM7UUFDckIsS0FBSyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztZQUNyQixPQUFPLElBQUksQ0FBQztRQUNkLENBQUM7UUFDRCxLQUFLLE9BQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO1lBQ3RCLElBQ0UsQ0FBQSxNQUFBLElBQUEsMkJBQVMsRUFBQyxNQUFNLENBQUMsSUFBSSxDQUFDLDBDQUFFLFFBQVE7Z0JBQ2hDLElBQUEsMkJBQVMsRUFBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsNEJBQWtCLENBQUMsRUFDMUMsQ0FBQztnQkFDRCxPQUFPLEtBQUssQ0FBQztZQUNmLENBQUM7WUFDRCxPQUFPLElBQUksQ0FBQztRQUNkLENBQUM7UUFFRCxPQUFPLENBQUMsQ0FBQyxDQUFDO1lBQ1IsT0FBTyxLQUFLLENBQUM7UUFDZixDQUFDO0lBQ0gsQ0FBQztBQUNILENBQUMsQ0FBQztBQWxDVyxRQUFBLDZCQUE2QixpQ0FrQ3hDO0FBRVcsUUFBQSxnQkFBZ0IsR0FBRztJQUM5QjtRQUNFLE9BQU8sRUFBRSxnQkFBZ0I7UUFDekIsU0FBUyxFQUFFLGlCQUFpQjtRQUM1QixLQUFLLEVBQUUsUUFBUTtLQUNoQjtJQUNEO1FBQ0UsT0FBTyxFQUFFLDBCQUEwQjtRQUNuQyxTQUFTLEVBQUUsbUNBQW1DO1FBQzlDLEtBQUssRUFBRSxjQUFjO0tBQ3RCO0lBQ0Q7UUFDRSxPQUFPLEVBQUUsVUFBVTtRQUNuQixTQUFTLEVBQUUsa0JBQWtCO1FBQzdCLEtBQUssRUFBRSxNQUFNO0tBQ2Q7SUFDRDtRQUNFLE9BQU8sRUFBRSxXQUFXO1FBQ3BCLFNBQVMsRUFBRSxtQkFBbUI7UUFDOUIsS0FBSyxFQUFFLE9BQU87S0FDZjtJQUNEO1FBQ0UsT0FBTyxFQUFFLGVBQWU7UUFDeEIsU0FBUyxFQUFFLGNBQWM7UUFDekIsS0FBSyxFQUFFLGVBQWU7S0FDdkI7SUFDRDtRQUNFLE9BQU8sRUFBRSxjQUFjO1FBQ3ZCLFNBQVMsRUFBRSxzQkFBc0I7UUFDakMsS0FBSyxFQUFFLFdBQVc7S0FDbkI7SUFDRDtRQUNFLE9BQU8sRUFBRSxVQUFVO1FBQ25CLFNBQVMsRUFBRSxrQkFBa0I7UUFDN0IsS0FBSyxFQUFFLE1BQU07S0FDZDtDQUNGLENBQUM7QUFFVyxRQUFBLG1CQUFtQixHQUFHLHdCQUFnQixDQUFDLE1BQU0sQ0FBQyxDQUFDLEdBQUcsRUFBRSxJQUFJLEVBQUUsRUFBRTtJQUN2RSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLElBQUksQ0FBQztJQUV2QixPQUFPLEdBQUcsQ0FBQztBQUNiLENBQUMsRUFBRSxFQUF1RCxDQUFDLENBQUM7QUFFL0MsUUFBQSxjQUFjLEdBQUc7SUFDNUI7UUFDRSxJQUFJLEVBQUUsVUFBVTtRQUNoQixLQUFLLEVBQUUsa0JBQWtCO1FBQ3pCLEtBQUssRUFBRSxNQUFNO0tBQ2Q7SUFDRDtRQUNFLElBQUksRUFBRSxXQUFXO1FBQ2pCLEtBQUssRUFBRSxtQkFBbUI7UUFDMUIsS0FBSyxFQUFFLE9BQU87S0FDZjtJQUNEO1FBQ0UsSUFBSSxFQUFFLGVBQWU7UUFDckIsS0FBSyxFQUFFLGNBQWM7UUFDckIsS0FBSyxFQUFFLGVBQWU7S0FDdkI7SUFDRDtRQUNFLElBQUksRUFBRSxjQUFjO1FBQ3BCLEtBQUssRUFBRSxzQkFBc0I7UUFDN0IsS0FBSyxFQUFFLFdBQVc7S0FDbkI7SUFDRDtRQUNFLElBQUksRUFBRSxVQUFVO1FBQ2hCLEtBQUssRUFBRSxrQkFBa0I7UUFDekIsS0FBSyxFQUFFLE1BQU07S0FDZDtDQUNGLENBQUM7QUFFVyxRQUFBLGlCQUFpQixHQUFHLHNCQUFjLENBQUMsTUFBTSxDQUFDLENBQUMsR0FBRyxFQUFFLElBQUksRUFBRSxFQUFFO0lBQ25FLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsSUFBSSxDQUFDO0lBRXZCLE9BQU8sR0FBRyxDQUFDO0FBQ2IsQ0FBQyxFQUFFLEVBQXFELENBQUMsQ0FBQztBQUU3QyxRQUFBLGVBQWUsR0FBRztJQUM3QjtRQUNFLEVBQUUsRUFBRSxDQUFDO1FBQ0wsS0FBSyxFQUFFLE1BQU07UUFDYixPQUFPLEVBQUUsYUFBYTtRQUN0QixLQUFLLEVBQUUsa0JBQWtCO0tBQzFCO0lBQ0Q7UUFDRSxFQUFFLEVBQUUsQ0FBQztRQUNMLEtBQUssRUFBRSxTQUFTO1FBQ2hCLE9BQU8sRUFBRSxzQkFBc0I7UUFDL0IsS0FBSyxFQUFFLG9DQUFvQztLQUM1QztJQUNEO1FBQ0UsRUFBRSxFQUFFLENBQUM7UUFDTCxLQUFLLEVBQUUsV0FBVztRQUNsQixPQUFPLEVBQUUsMEJBQTBCO1FBQ25DLEtBQUssRUFBRSw4Q0FBOEM7S0FDdEQ7SUFDRDtRQUNFLEVBQUUsRUFBRSxDQUFDO1FBQ0wsS0FBSyxFQUFFLFlBQVk7UUFDbkIsT0FBTyxFQUFFLDRCQUE0QjtRQUNyQyxLQUFLLEVBQUUsOENBQThDO0tBQ3REO0lBQ0Q7UUFDRSxFQUFFLEVBQUUsQ0FBQztRQUNMLEtBQUssRUFBRSxhQUFhO1FBQ3BCLE9BQU8sRUFBRSw4QkFBOEI7UUFDdkMsS0FBSyxFQUFFLDhDQUE4QztLQUN0RDtDQUNGLENBQUM7QUFFRjs7Ozs7Ozs7R0FRRztBQUNJLE1BQU0sZ0JBQWdCLEdBQUcsQ0FBQyxNQUFrQixFQUFXLEVBQUU7O0lBQzlELE9BQU87SUFDTCx3RUFBd0U7SUFDeEU7UUFDRSxPQUFPLENBQUMsTUFBTTtRQUNkLE9BQU8sQ0FBQyxNQUFNO1FBQ2QsT0FBTyxDQUFDLE9BQU87UUFDZixPQUFPLENBQUMsTUFBTTtRQUNkLE9BQU8sQ0FBQyxPQUFPO1FBQ2YsT0FBTyxDQUFDLE1BQU07UUFDZCxPQUFPLENBQUMsVUFBVTtLQUNuQixDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsSUFBZSxDQUFDO1FBQ2xDLDRGQUE0RjtRQUM1Riw0QkFBNEIsQ0FBQyxNQUFNLENBQUM7UUFDcEMsNEZBQTRGO1FBQzVGLDhCQUE4QixDQUFDLE1BQU0sQ0FBQztRQUN0QywrQ0FBK0M7UUFDL0MsVUFBVSxDQUFDLE1BQU0sQ0FBQztRQUNsQiwyREFBMkQ7UUFDM0QsQ0FBQyxNQUFNLENBQUMsRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLEVBQUUsS0FBSSxNQUFBLElBQUEsMkJBQVMsRUFBQyxNQUFNLENBQUMsSUFBSSxDQUFDLDBDQUFFLEVBQUUsQ0FBQSxDQUFDLENBQUMsQ0FDekQsQ0FBQztBQUNKLENBQUMsQ0FBQztBQXJCVyxRQUFBLGdCQUFnQixvQkFxQjNCO0FBRUY7Ozs7Ozs7R0FPRztBQUNJLE1BQU0sbUJBQW1CLEdBQUcsQ0FBQyxNQUFrQixFQUFFLEVBQUU7SUFDeEQsNEVBQTRFO0lBQzVFLElBQ0U7UUFDRSxPQUFPLENBQUMsSUFBSTtRQUNaLE9BQU8sQ0FBQyxRQUFRO1FBQ2hCLE9BQU8sQ0FBQyxXQUFXO1FBQ25CLE9BQU8sQ0FBQyxnQkFBZ0I7S0FDekIsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLElBQWUsQ0FBQyxFQUNsQyxDQUFDO1FBQ0QsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQsMEVBQTBFO0lBQzFFLElBQUksTUFBTSxDQUFDLElBQUksS0FBSyxPQUFPLENBQUMsT0FBTyxFQUFFLENBQUM7UUFDcEMsT0FBTyxJQUFBLDJCQUFtQixFQUFDLEVBQUUsYUFBYSxFQUFFLE1BQU0sRUFBRSxDQUFDLEtBQUssT0FBTyxDQUFDLFFBQVEsQ0FBQztJQUM3RSxDQUFDO0lBRUQsT0FBTyxLQUFLLENBQUM7QUFDZixDQUFDLENBQUM7QUFuQlcsUUFBQSxtQkFBbUIsdUJBbUI5QjtBQUVXLFFBQUEsd0JBQXdCLEdBQWM7SUFDakQsT0FBTyxDQUFDLFlBQVk7SUFDcEIsT0FBTyxDQUFDLGNBQWM7SUFDdEIsT0FBTyxDQUFDLE1BQU07SUFDZCxPQUFPLENBQUMsT0FBTztJQUNmLE9BQU8sQ0FBQyxLQUFLO0lBQ2IsT0FBTyxDQUFDLFdBQVc7SUFDbkIsT0FBTyxDQUFDLEdBQUc7SUFDWCxPQUFPLENBQUMsRUFBRTtJQUNWLE9BQU8sQ0FBQyxVQUFVO0NBQ25CLENBQUM7QUFFRiw2Q0FBNkM7QUFDaEMsUUFBQSxpQkFBaUIsR0FBYyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyJ9