"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.formatAggregation = exports.AllAggregations = exports.AttachmentAggregations = exports.DateAggregations = exports.BooleanAggregations = exports.CommonAggregations = exports.NumericalAggregations = exports.getAvailableAggregations = void 0;
const UITypes_1 = __importDefault(require("./UITypes"));
const enums_1 = require("../lib/formula/enums");
const dateTimeHelper_1 = require("../lib/dateTimeHelper");
const helperFunctions_1 = require("../lib/helperFunctions");
const durationUtils_1 = require("../lib/durationUtils");
const numberUtils_1 = require("../lib/numberUtils");
const columnHelper_1 = require("./columnHelper");
var NumericalAggregations;
(function (NumericalAggregations) {
    NumericalAggregations["Sum"] = "sum";
    NumericalAggregations["Min"] = "min";
    NumericalAggregations["Max"] = "max";
    NumericalAggregations["Avg"] = "avg";
    NumericalAggregations["Median"] = "median";
    NumericalAggregations["StandardDeviation"] = "std_dev";
    //   Histogram = 'histogram',
    NumericalAggregations["Range"] = "range";
})(NumericalAggregations || (exports.NumericalAggregations = NumericalAggregations = {}));
var CommonAggregations;
(function (CommonAggregations) {
    CommonAggregations["Count"] = "count";
    CommonAggregations["CountEmpty"] = "count_empty";
    CommonAggregations["CountFilled"] = "count_filled";
    CommonAggregations["CountUnique"] = "count_unique";
    CommonAggregations["PercentEmpty"] = "percent_empty";
    CommonAggregations["PercentFilled"] = "percent_filled";
    CommonAggregations["PercentUnique"] = "percent_unique";
    CommonAggregations["None"] = "none";
})(CommonAggregations || (exports.CommonAggregations = CommonAggregations = {}));
var AttachmentAggregations;
(function (AttachmentAggregations) {
    AttachmentAggregations["AttachmentSize"] = "attachment_size";
})(AttachmentAggregations || (exports.AttachmentAggregations = AttachmentAggregations = {}));
var BooleanAggregations;
(function (BooleanAggregations) {
    BooleanAggregations["Checked"] = "checked";
    BooleanAggregations["Unchecked"] = "unchecked";
    BooleanAggregations["PercentChecked"] = "percent_checked";
    BooleanAggregations["PercentUnchecked"] = "percent_unchecked";
})(BooleanAggregations || (exports.BooleanAggregations = BooleanAggregations = {}));
var DateAggregations;
(function (DateAggregations) {
    DateAggregations["EarliestDate"] = "earliest_date";
    DateAggregations["LatestDate"] = "latest_date";
    DateAggregations["DateRange"] = "date_range";
    DateAggregations["MonthRange"] = "month_range";
})(DateAggregations || (exports.DateAggregations = DateAggregations = {}));
const AllAggregations = Object.assign(Object.assign(Object.assign(Object.assign(Object.assign({}, CommonAggregations), NumericalAggregations), AttachmentAggregations), BooleanAggregations), DateAggregations);
exports.AllAggregations = AllAggregations;
const getAvailableAggregations = (type, parsed_tree) => {
    let returnAggregations = [];
    if (type === UITypes_1.default.Formula && (parsed_tree === null || parsed_tree === void 0 ? void 0 : parsed_tree.dataType)) {
        switch (parsed_tree.dataType) {
            case enums_1.FormulaDataTypes.BOOLEAN:
                returnAggregations = [
                    ...Object.values(BooleanAggregations),
                    CommonAggregations.None,
                ];
                break;
            case enums_1.FormulaDataTypes.DATE:
                returnAggregations = [
                    ...Object.values(DateAggregations),
                    ...Object.values(CommonAggregations),
                ];
                break;
            case enums_1.FormulaDataTypes.NUMERIC:
                returnAggregations = [
                    ...Object.values(NumericalAggregations),
                    ...Object.values(CommonAggregations),
                ];
                break;
            default:
                returnAggregations = [...Object.values(CommonAggregations)];
                break;
        }
    }
    switch (type) {
        case UITypes_1.default.Number:
        case UITypes_1.default.Decimal:
        case UITypes_1.default.Currency:
        case UITypes_1.default.Percent:
        case UITypes_1.default.Duration:
        case UITypes_1.default.Rating:
        case UITypes_1.default.Rollup:
        case UITypes_1.default.Links:
            returnAggregations = [
                ...Object.values(NumericalAggregations),
                ...Object.values(CommonAggregations),
            ];
            break;
        case UITypes_1.default.Checkbox:
            returnAggregations = [
                ...Object.values(BooleanAggregations),
                CommonAggregations.None,
            ];
            break;
        case UITypes_1.default.Date:
        case UITypes_1.default.DateTime:
        case UITypes_1.default.LastModifiedTime:
        case UITypes_1.default.CreatedTime:
            returnAggregations = [
                ...Object.values(DateAggregations),
                ...Object.values(CommonAggregations),
            ];
            break;
        case UITypes_1.default.SpecificDBType:
        case UITypes_1.default.ForeignKey:
            returnAggregations = [CommonAggregations.None];
            break;
        case UITypes_1.default.Button:
        case UITypes_1.default.Attachment:
            return [CommonAggregations.None, AttachmentAggregations.AttachmentSize];
    }
    if (!returnAggregations.length) {
        returnAggregations = [...Object.values(CommonAggregations)];
    }
    return returnAggregations;
};
exports.getAvailableAggregations = getAvailableAggregations;
const formatAggregation = (aggregation, value, column, 
/**
 * If columnHelperParams is provided then it will be used to format the aggregation value
 */
columnHelperParams) => {
    var _a, _b, _c, _d, _e, _f;
    if (column.uidt === UITypes_1.default.Formula) {
        if ((_a = column === null || column === void 0 ? void 0 : column.meta) === null || _a === void 0 ? void 0 : _a.display_type) {
            const childColumn = Object.assign({ uidt: (_b = column === null || column === void 0 ? void 0 : column.meta) === null || _b === void 0 ? void 0 : _b.display_type }, (_c = column === null || column === void 0 ? void 0 : column.meta) === null || _c === void 0 ? void 0 : _c.display_column_meta);
            return formatAggregation(aggregation, value, childColumn, Object.assign(Object.assign({}, columnHelperParams), { col: childColumn }));
        }
    }
    if ([DateAggregations.EarliestDate, DateAggregations.LatestDate].includes(aggregation)) {
        if (column.uidt === UITypes_1.default.DateTime) {
            return (0, dateTimeHelper_1.getDateTimeValue)(value, column);
        }
        else if (column.uidt === UITypes_1.default.Date) {
            return (0, dateTimeHelper_1.getDateValue)(value, column);
        }
        return (0, dateTimeHelper_1.getDateTimeValue)(value, column);
    }
    if ([
        CommonAggregations.PercentEmpty,
        CommonAggregations.PercentFilled,
        CommonAggregations.PercentUnique,
        BooleanAggregations.PercentChecked,
        BooleanAggregations.PercentUnchecked,
    ].includes(aggregation)) {
        return `${(_d = (0, numberUtils_1.roundTo)(value, 1)) !== null && _d !== void 0 ? _d : 0}%`;
    }
    if ([DateAggregations.MonthRange, DateAggregations.DateRange].includes(aggregation)) {
        return aggregation === DateAggregations.DateRange
            ? `${value !== null && value !== void 0 ? value : 0} days`
            : `${value !== null && value !== void 0 ? value : 0} months`;
    }
    if ([
        CommonAggregations.Count,
        CommonAggregations.CountEmpty,
        CommonAggregations.CountFilled,
        CommonAggregations.CountUnique,
    ].includes(aggregation)) {
        return value;
    }
    if ([AttachmentAggregations.AttachmentSize].includes(aggregation)) {
        return (0, helperFunctions_1.formatBytes)(value !== null && value !== void 0 ? value : 0);
    }
    if (columnHelperParams &&
        (columnHelperParams === null || columnHelperParams === void 0 ? void 0 : columnHelperParams.col) &&
        Object.values(NumericalAggregations).includes(aggregation) &&
        [
            UITypes_1.default.Number,
            UITypes_1.default.Decimal,
            UITypes_1.default.Currency,
            UITypes_1.default.Percent,
            UITypes_1.default.Duration,
            UITypes_1.default.Rollup,
            UITypes_1.default.Time,
        ].includes(column.uidt)) {
        return columnHelper_1.ColumnHelper.parsePlainCellValue(value, Object.assign(Object.assign({}, columnHelperParams), { isAggregation: true }));
    }
    if (column.uidt === UITypes_1.default.Currency) {
        return (0, numberUtils_1.getCurrencyValue)(value, column);
    }
    if (column.uidt === UITypes_1.default.Percent) {
        return `${(0, numberUtils_1.roundTo)(value, 1)}%`;
    }
    if (column.uidt === UITypes_1.default.Duration) {
        return (0, durationUtils_1.convertMS2Duration)(value, ((_e = (0, helperFunctions_1.parseProp)(column.meta)) === null || _e === void 0 ? void 0 : _e.duration) || 0);
    }
    if (typeof value === 'number') {
        return (_f = (0, numberUtils_1.roundTo)(value, 1)) !== null && _f !== void 0 ? _f : '∞';
    }
    return value;
};
exports.formatAggregation = formatAggregation;
//# sourceMappingURL=data:application/json;base64,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