"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DateTimeHelper = void 0;
const dateTimeHelper_1 = require("../../../lib/dateTimeHelper");
const column_interface_1 = __importDefault(require("../column.interface"));
const utils_1 = require("../utils");
const error_1 = require("../../../lib/error");
const fill_handler_1 = require("../utils/fill-handler");
const helperFunctions_1 = require("../../../lib/helperFunctions");
const dayjs_1 = __importDefault(require("dayjs"));
const is_1 = require("../../../lib/is");
class DateTimeHelper extends column_interface_1.default {
    constructor() {
        super(...arguments);
        this.columnDefaultMeta = {
            date_format: dateTimeHelper_1.dateFormats[0],
            time_format: dateTimeHelper_1.timeFormats[0],
            is12hrFormat: false,
        };
    }
    serializeValue(value, params) {
        if (params.serializeSearchQuery)
            return null;
        value = (0, utils_1.serializeDateOrDateTimeValue)(value, params);
        if (value === null) {
            if (params.isMultipleCellPaste) {
                return null;
            }
            else {
                throw new error_1.SilentTypeConversionError();
            }
        }
        return value;
    }
    parseValue(value, params) {
        return (0, utils_1.parseDateTimeValue)(value, params);
    }
    parsePlainCellValue(value, params) {
        var _a;
        return (_a = (0, utils_1.parseDateTimeValue)(value, params)) !== null && _a !== void 0 ? _a : '';
    }
    populateFillHandle(params) {
        var _a;
        const { column, highlightedData, numberOfRows } = params;
        if (highlightedData.length < 2) {
            return (0, fill_handler_1.populateFillHandleStrictCopy)(params);
        }
        // data is in form like 'YYYY-MM-DD HH:mm' depends on meta dateformat
        const meta = (0, helperFunctions_1.parseProp)(column.meta);
        const metaDateFormat = (_a = meta.date_format) !== null && _a !== void 0 ? _a : 'YYYY-MM-DD';
        const metaTimeFormat = (0, dateTimeHelper_1.constructTimeFormat)(params.column);
        const dateTimeFormat = `${metaDateFormat} ${metaTimeFormat}`;
        let lastData;
        let modifier = undefined;
        let scale = 's';
        let canUseSecondScale = true;
        if (!dateTimeFormat.includes('ss')) {
            canUseSecondScale = false;
        }
        const setModifierForScaleIfRound = (currentData, lastData, diffScale) => {
            const currentModifier = currentData.diff(lastData, diffScale, true);
            if ((0, utils_1.isNumberRound)(currentModifier)) {
                scale = utils_1.DATE_DIFF_TO_SCALE_LABEL_MAP[diffScale];
                modifier = currentModifier;
            }
        };
        // map to dayjs
        const dayJsHighlightedData = [];
        for (const date of highlightedData) {
            if ((0, is_1.ncIsNullOrUndefined)(date) ||
                date === '' ||
                typeof date !== 'string') {
                return (0, fill_handler_1.populateFillHandleStrictCopy)(params);
            }
            const currentData = (0, dayjs_1.default)(date, dateTimeFormat);
            // unlikely on normal case
            if (!currentData.isValid()) {
                return (0, fill_handler_1.populateFillHandleStrictCopy)(params);
            }
            if (!lastData) {
                lastData = currentData;
            }
            else {
                if ((0, is_1.ncIsUndefined)(modifier)) {
                    const modifierInSeconds = currentData.diff(lastData, 'second');
                    if (modifierInSeconds > 60 * 60 * 24 * 354) {
                        setModifierForScaleIfRound(currentData, lastData, 'year');
                    }
                    else if (modifierInSeconds > 60 * 60 * 24 * 28) {
                        setModifierForScaleIfRound(currentData, lastData, 'month');
                    }
                    else if (modifierInSeconds > 60 * 60 * 24) {
                        setModifierForScaleIfRound(currentData, lastData, 'day');
                    }
                    else if (modifierInSeconds > 60 * 60) {
                        setModifierForScaleIfRound(currentData, lastData, 'hour');
                    }
                    else if (modifierInSeconds > 60) {
                        setModifierForScaleIfRound(currentData, lastData, 'minute');
                    }
                    else if (!canUseSecondScale) {
                        modifier = currentData.diff(lastData, 'minute');
                        scale = 'm';
                    }
                    else {
                        modifier = modifierInSeconds;
                        scale = 's';
                    }
                }
                else {
                    const currentModifier = currentData.diff(lastData, utils_1.DATE_SCALE_LABEL_TO_DIFF_MAP[scale]);
                    if (currentModifier !== modifier) {
                        return (0, fill_handler_1.populateFillHandleStrictCopy)(params);
                    }
                }
                lastData = currentData;
            }
            dayJsHighlightedData.push(lastData);
        }
        if (modifier === 0) {
            return (0, fill_handler_1.populateFillHandleStrictCopy)(params);
        }
        const numberToGenerate = numberOfRows - highlightedData.length;
        return Array.from({ length: numberToGenerate }).map(() => {
            const currentData = lastData.add(modifier, utils_1.DATE_SCALE_LABEL_TO_DIFF_MAP[scale]);
            lastData = currentData;
            return currentData.format(dateTimeFormat);
        });
    }
}
exports.DateTimeHelper = DateTimeHelper;
//# sourceMappingURL=data:application/json;base64,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