"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const date_time_1 = require("./date-time");
const dayjs_1 = __importDefault(require("dayjs"));
const UITypes_1 = __importDefault(require("../../../lib/UITypes"));
describe('parse date', () => {
    it('should parse a valid date string with default format', () => {
        const col = { meta: null, uidt: UITypes_1.default.Date };
        expect((0, date_time_1.parseDateValue)('2023-10-27', col)).toBe('2023-10-27');
    });
    it('should parse a valid date string with custom format', () => {
        const col = { meta: { date_format: 'MM/DD/YYYY' }, uidt: UITypes_1.default.Date };
        expect((0, date_time_1.parseDateValue)('10/27/2023', col)).toBe('10/27/2023');
    });
    it('should parse a valid numeric timestamp', () => {
        const col = { meta: null, uidt: UITypes_1.default.Date };
        // Timestamp for 2023-10-27T00:00:00.000Z
        expect((0, date_time_1.parseDateValue)('1698364800000', col)).toBe('2023-10-27');
    });
    it('should parse a valid date string for a system column', () => {
        const col = { meta: null, uidt: UITypes_1.default.Date };
        expect((0, date_time_1.parseDateValue)('2023-10-27 10:30:00', col, true)).toBe('2023-10-27 10:30:00');
    });
    it('should serialize a valid date value for Date type', () => {
        const col = { uidt: UITypes_1.default.Date, meta: null };
        expect((0, date_time_1.serializeDateOrDateTimeValue)('2023-10-27', { col: col })).toBe('2023-10-27');
    });
});
describe('parse date-time', () => {
    it('should parse a valid ISO datetime string with quotes', () => {
        const col = { uidt: UITypes_1.default.DateTime, meta: {} };
        const params = { col, isMysql: () => false };
        const value = '"2023-10-27T10:30:00.000Z"';
        const expected = (0, dayjs_1.default)(value.replace(/["']/g, '')).format('YYYY-MM-DD HH:mm');
        expect((0, date_time_1.parseDateTimeValue)(value, params)).toBe(expected);
    });
    it('should parse a valid ISO datetime string without quotes', () => {
        const col = { uidt: UITypes_1.default.DateTime, meta: {} };
        const params = { col, isMysql: () => false };
        const value = '2023-10-27T10:30:00.000Z';
        const expected = (0, dayjs_1.default)(value).format('YYYY-MM-DD HH:mm');
        expect((0, date_time_1.parseDateTimeValue)(value, params)).toBe(expected);
    });
    it('should parse a valid datetime string in YYYY-MM-DD HH:mm:ss format with isMysql true', () => {
        const col = { uidt: UITypes_1.default.DateTime, meta: {} };
        const params = { col, isMysql: () => true };
        const value = '2023-10-27 10:30:00';
        const expected = (0, dayjs_1.default)(value, 'YYYY-MM-DD HH:mm:ss').format('YYYY-MM-DD HH:mm');
        expect((0, date_time_1.parseDateTimeValue)(value, params)).toBe(expected);
    });
    it('should parse a valid datetime string in YYYY-MM-DD HH:mm:ssZ format', () => {
        const col = { uidt: UITypes_1.default.DateTime, meta: {} };
        const params = { col, isMysql: () => false };
        const value = '2023-10-27 10:30:00Z';
        const expected = (0, dayjs_1.default)(value, 'YYYY-MM-DD HH:mm:ssZ').format('YYYY-MM-DD HH:mm');
        expect((0, date_time_1.parseDateTimeValue)(value, params)).toBe(expected);
    });
    it('should return null for invalid datetime string', () => {
        const col = { uidt: UITypes_1.default.DateTime, meta: {} };
        const params = { col, isMysql: () => false };
        const value = 'invalid-date-string';
        expect((0, date_time_1.parseDateTimeValue)(value, params)).toBeNull();
    });
    it('should return null for null input', () => {
        const col = { uidt: UITypes_1.default.DateTime, meta: {} };
        const params = { col, isMysql: () => false };
        const value = null;
        expect((0, date_time_1.parseDateTimeValue)(value, params)).toBeFalsy();
    });
});
describe('serialize', () => {
    it('should serialize a valid datetime value for non-Date type', () => {
        const col = {
            uidt: UITypes_1.default.DateTime,
            meta: {},
        };
        const value = '2023-10-27 10:00:00';
        const expected = (0, dayjs_1.default)(value).utc().format('YYYY-MM-DD HH:mm:ssZ');
        expect((0, date_time_1.serializeDateOrDateTimeValue)(value, { col: col })).toBe(expected);
    });
    it('should serialize a date string to datetime format for DateTime type', () => {
        const col = { uidt: UITypes_1.default.DateTime, meta: {} };
        const value = '2023-10-27';
        const expected = (0, dayjs_1.default)(value).utc().format('YYYY-MM-DD HH:mm:ssZ');
        expect((0, date_time_1.serializeDateOrDateTimeValue)(value, { col: col })).toBe(expected);
    });
    it('should serialize a date string to datetime format for custom date format (1)', () => {
        const col = {
            uidt: UITypes_1.default.Date,
            meta: { date_format: 'MM/DD/YYYY' },
        };
        const value = '09/05/1980';
        const expected = (0, dayjs_1.default)(value, 'MM/DD/YYYY').format('YYYY-MM-DD');
        expect((0, date_time_1.serializeDateOrDateTimeValue)(value, { col: col })).toBe(expected);
    });
    it('should serialize a date string to datetime format for custom date format (2)', () => {
        const col = {
            uidt: UITypes_1.default.Date,
            meta: { date_format: 'DD/MM/YYYY' },
        };
        const value = '09/05/1980';
        const expected = (0, dayjs_1.default)(value, 'DD/MM/YYYY').format('YYYY-MM-DD');
        expect((0, date_time_1.serializeDateOrDateTimeValue)(value, { col: col })).toBe(expected);
    });
});
//# sourceMappingURL=data:application/json;base64,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