"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseLinksValue = exports.parseUserValue = exports.parseYearValue = exports.parseTimeValue = exports.parseCurrencyValue = exports.parseJsonValue = exports.parseCheckboxValue = exports.parseDurationValue = exports.parsePercentValue = exports.parseDecimalValue = exports.parseIntValue = exports.parseDefault = void 0;
const dayjs_1 = __importDefault(require("dayjs"));
const durationUtils_1 = require("../../../lib/durationUtils");
const helperFunctions_1 = require("../../../lib/helperFunctions");
const is_1 = require("../../../lib/is");
const dateTimeHelper_1 = require("../../../lib/dateTimeHelper");
const common_1 = require("../../../lib/columnHelper/utils/common");
const parseDefault = (value) => {
    var _a;
    try {
        if (typeof value === 'object') {
            return JSON.stringify(value);
        }
        else {
            return (_a = value === null || value === void 0 ? void 0 : value.toString()) !== null && _a !== void 0 ? _a : null;
        }
    }
    catch (_b) {
        return null;
    }
};
exports.parseDefault = parseDefault;
const parseIntValue = (value, col) => {
    if ((0, is_1.ncIsNaN)(value)) {
        return null;
    }
    value = parseInt(value.toString(), 10);
    const columnMeta = (0, helperFunctions_1.parseProp)(col === null || col === void 0 ? void 0 : col.meta);
    if (columnMeta.isLocaleString) {
        return Number(value).toLocaleString();
    }
    return Number(value);
};
exports.parseIntValue = parseIntValue;
const parseDecimalValue = (value, col) => {
    var _a, _b, _c, _d;
    if ((0, is_1.ncIsNaN)(value)) {
        return null;
    }
    const columnMeta = (0, helperFunctions_1.parseProp)(col.meta);
    if (columnMeta.isLocaleString) {
        return Number((0, helperFunctions_1.roundUpToPrecision)(Number(value), (_a = columnMeta.precision) !== null && _a !== void 0 ? _a : 1)).toLocaleString(undefined, {
            minimumFractionDigits: (_b = columnMeta.precision) !== null && _b !== void 0 ? _b : 1,
            maximumFractionDigits: (_c = columnMeta.precision) !== null && _c !== void 0 ? _c : 1,
        });
    }
    return (0, helperFunctions_1.roundUpToPrecision)(Number(value), (_d = columnMeta.precision) !== null && _d !== void 0 ? _d : 1);
};
exports.parseDecimalValue = parseDecimalValue;
const parsePercentValue = (value, col) => {
    var _a;
    if ((0, is_1.ncIsNaN)(value)) {
        return null;
    }
    /**
     * We have to keep cell display and parse value (copy) consistent
     * ref: check `formatPercentage` function in `~/utils/cell.ts`
     */
    if (Number(value) % 1 === 0) {
        return `${Number(value)}%`;
    }
    const columnMeta = (0, helperFunctions_1.parseProp)(col.meta);
    const percentValue = (0, helperFunctions_1.roundUpToPrecision)(Number(value), (_a = columnMeta.precision) !== null && _a !== void 0 ? _a : 2);
    return `${percentValue}%`;
};
exports.parsePercentValue = parsePercentValue;
const parseDurationValue = (value, col) => {
    var _a;
    const durationType = ((_a = (0, helperFunctions_1.parseProp)(col.meta)) === null || _a === void 0 ? void 0 : _a.duration) || 0;
    return (0, durationUtils_1.convertMS2Duration)(value, durationType);
};
exports.parseDurationValue = parseDurationValue;
const parseCheckboxValue = (value) => {
    if ((0, is_1.ncIsBoolean)(value))
        return value;
    if ((0, is_1.ncIsString)(value)) {
        const strval = value.trim().toLowerCase();
        const parsedValue = common_1.checkboxTypeMap[strval];
        if (!(0, is_1.ncIsNull)(parsedValue) && !(0, is_1.ncIsUndefined)(parsedValue)) {
            return parsedValue;
        }
    }
    if ((0, is_1.ncIsNumber)(value)) {
        if (value === 1)
            return true;
        if (value === 0)
            return false;
    }
    return null;
};
exports.parseCheckboxValue = parseCheckboxValue;
const parseJsonValue = (value) => {
    try {
        return (0, is_1.ncIsString)(value)
            ? JSON.stringify(JSON.parse(value)) // Ensure it's a valid JSON string
            : JSON.stringify(value); // Convert object/array to JSON
    }
    catch (_a) {
        return null;
    }
};
exports.parseJsonValue = parseJsonValue;
const parseCurrencyValue = (value, col) => {
    var _a, _b, _c;
    if ((0, is_1.ncIsNaN)(value)) {
        return null;
    }
    const columnMeta = (0, helperFunctions_1.parseProp)(col.meta);
    try {
        // Round the value to the specified precision
        const roundedValue = (0, helperFunctions_1.roundUpToPrecision)(Number(value), (_a = columnMeta.precision) !== null && _a !== void 0 ? _a : 2);
        return new Intl.NumberFormat(columnMeta.currency_locale || 'en-US', {
            style: 'currency',
            currency: columnMeta.currency_code || 'USD',
            minimumFractionDigits: (_b = columnMeta.precision) !== null && _b !== void 0 ? _b : 2,
            maximumFractionDigits: (_c = columnMeta.precision) !== null && _c !== void 0 ? _c : 2,
        }).format(+roundedValue);
    }
    catch (_d) {
        return value;
    }
};
exports.parseCurrencyValue = parseCurrencyValue;
const parseTimeValue = (value, params) => {
    var _a, _b;
    value = value === null || value === void 0 ? void 0 : value.toString().trim();
    if (!value)
        return null;
    // remove `"`
    // e.g. "2023-05-12T08:03:53.000Z" -> 2023-05-12T08:03:53.000Z
    value = value.replace(/["']/g, '');
    const isMySQL = (_a = params.isMysql) === null || _a === void 0 ? void 0 : _a.call(params, params.col.source_id);
    const isPostgres = (_b = params.isPg) === null || _b === void 0 ? void 0 : _b.call(params, params.col.source_id);
    let d = (0, dayjs_1.default)(value);
    if (!d.isValid()) {
        // insert a datetime value, copy the value without refreshing
        // e.g. value = 2023-05-12T03:49:25.000Z
        // feed custom parse format
        d = (0, dayjs_1.default)(value, isMySQL ? 'YYYY-MM-DD HH:mm:ss' : 'YYYY-MM-DD HH:mm:ssZ');
    }
    if (!d.isValid()) {
        // MySQL and Postgres store time in HH:mm:ss format so we need to feed custom parse format
        d = isMySQL || isPostgres ? (0, dayjs_1.default)(value, 'HH:mm:ss') : (0, dayjs_1.default)(value);
    }
    if (!d.isValid()) {
        d = (0, dayjs_1.default)(`1999-01-01 ${value}`);
    }
    if (!d.isValid()) {
        // return empty string for invalid time
        return null;
    }
    return d.format((0, dateTimeHelper_1.constructTimeFormat)(params.col));
};
exports.parseTimeValue = parseTimeValue;
const parseYearValue = (value) => {
    value = value === null || value === void 0 ? void 0 : value.toString().trim();
    if (!value || !(0, dayjs_1.default)(value).isValid()) {
        return null;
    }
    value = (0, dayjs_1.default)(value, 'YYYY').format('YYYY');
    return value ? +value : value;
};
exports.parseYearValue = parseYearValue;
const parseUserValue = (value, withDisplayName = false, useUserId = false) => {
    let data = value;
    try {
        if (typeof value === 'string') {
            data = JSON.parse(value);
        }
    }
    catch (_a) { }
    return ((0, is_1.ncIsArray)(data) ? data : (0, is_1.ncIsObject)(data) ? [data] : [])
        .map((user) => withDisplayName && user.display_name
        ? `${user.display_name}<${user.email}>`
        : useUserId
            ? `${user.id}`
            : `${user.email}`)
        .join(', ');
};
exports.parseUserValue = parseUserValue;
const parseLinksValue = (value, params) => {
    var _a, _b, _c;
    const { col, t } = params;
    const columnMeta = (0, helperFunctions_1.parseProp)(col.meta);
    const parsedValue = +value || 0;
    if (!parsedValue) {
        return `0 ${columnMeta.plural || ((_a = t === null || t === void 0 ? void 0 : t('general.links')) !== null && _a !== void 0 ? _a : 'Links')}`;
    }
    else if (parsedValue === 1) {
        return `1 ${columnMeta.singular || ((_b = t === null || t === void 0 ? void 0 : t('general.link')) !== null && _b !== void 0 ? _b : 'Link')}`;
    }
    else {
        return `${parsedValue} ${columnMeta.plural || ((_c = t('general.links')) !== null && _c !== void 0 ? _c : 'Links')}`;
    }
};
exports.parseLinksValue = parseLinksValue;
//# sourceMappingURL=data:application/json;base64,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