"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.serializeEmail = exports.serializeSelectValue = exports.serializeYearValue = exports.serializeTimeValue = exports.serializeCurrencyValue = exports.serializeJsonValue = exports.serializeCheckboxValue = exports.serializeDurationValue = exports.serializeIntValue = exports.serializeDecimalValue = exports.serializeStringValue = void 0;
const dayjs_1 = __importDefault(require("dayjs"));
const durationUtils_1 = require("../../../lib/durationUtils");
const helperFunctions_1 = require("../../../lib/helperFunctions");
const is_1 = require("../../../lib/is");
const UITypes_1 = __importDefault(require("../../../lib/UITypes"));
const error_1 = require("../../../lib/error");
const common_1 = require("../../../lib/columnHelper/utils/common");
const currencyHelpers_1 = require("../../../lib/currencyHelpers");
/**
 * Remove outer quotes & unescape
 */
const serializeStringValue = (value) => {
    var _a;
    value = (_a = value === null || value === void 0 ? void 0 : value.toString()) !== null && _a !== void 0 ? _a : null;
    if (!value)
        return null;
    if (value.match(/^"[\s\S]*"$/)) {
        value = value.slice(1, -1).replace(/\\"/g, '"');
    }
    return value;
};
exports.serializeStringValue = serializeStringValue;
const serializeDecimalValue = (value, callback, params) => {
    var _a;
    // If we have clipboard data, use it
    if (((_a = params === null || params === void 0 ? void 0 : params.clipboardItem) === null || _a === void 0 ? void 0 : _a.dbCellValue) &&
        (0, is_1.ncIsNumber)(params.clipboardItem.dbCellValue)) {
        return params.clipboardItem.dbCellValue;
    }
    if ((0, is_1.ncIsNumber)(value)) {
        return Number(value);
    }
    // If it's a string, remove commas and check if it's a valid number
    if ((0, is_1.ncIsString)(value)) {
        const cleanedValue = (0, is_1.ncIsFunction)(callback)
            ? callback(value)
            : value
                .replace(/[\s\u00A0]/g, '') // remove spaces/non-breaking spaces
                .replace(/(?!^-)[^\d.-]/g, ''); // keep only digits, dot, one leading minus
        if (!cleanedValue)
            return null;
        // Try converting the cleaned value to a number
        const numberValue = Number(cleanedValue);
        // If it's a valid number, return it
        if (!isNaN(numberValue)) {
            return numberValue;
        }
    }
    return null;
};
exports.serializeDecimalValue = serializeDecimalValue;
const serializeIntValue = (value) => {
    value = (0, exports.serializeDecimalValue)(value);
    if ((0, is_1.ncIsNumber)(value)) {
        return parseInt(value.toString(), 10);
    }
    return null; // Return null if it's not a valid number
};
exports.serializeIntValue = serializeIntValue;
const serializeDurationValue = (value, col) => {
    var _a;
    // Check if the value is a pure number (interpreted as seconds)
    if (!(0, is_1.ncIsNaN)(value)) {
        return parseInt(value, 10); // Directly return seconds
    }
    const columnMeta = (0, helperFunctions_1.parseProp)(col.meta);
    const res = (0, durationUtils_1.convertDurationToSeconds)(value, (_a = columnMeta.duration) !== null && _a !== void 0 ? _a : 0);
    return res._isValid ? res._sec : null;
};
exports.serializeDurationValue = serializeDurationValue;
const serializeCheckboxValue = (value) => {
    if ((0, is_1.ncIsBoolean)(value))
        return value;
    if ((0, is_1.ncIsString)(value)) {
        const strval = value.trim().toLowerCase();
        const parsedValue = common_1.checkboxTypeMap[strval];
        if (!(0, is_1.ncIsNull)(parsedValue) && !(0, is_1.ncIsUndefined)(parsedValue)) {
            return parsedValue;
        }
    }
    return null;
};
exports.serializeCheckboxValue = serializeCheckboxValue;
const serializeJsonValue = (value) => {
    try {
        return (0, is_1.ncIsString)(value)
            ? JSON.stringify(JSON.parse(value)) // Ensure it's a valid JSON string
            : JSON.stringify(value); // Convert object/array to JSON
    }
    catch (_a) {
        return null;
    }
};
exports.serializeJsonValue = serializeJsonValue;
const serializeCurrencyValue = (value, params) => {
    var _a;
    // If we have clipboard data, use it
    if (((_a = params === null || params === void 0 ? void 0 : params.clipboardItem) === null || _a === void 0 ? void 0 : _a.dbCellValue) &&
        (0, is_1.ncIsNumber)(params.clipboardItem.dbCellValue)) {
        return params.clipboardItem.dbCellValue;
    }
    return (0, exports.serializeDecimalValue)(value, (value) => {
        var _a;
        const columnMeta = (0, helperFunctions_1.parseProp)(params.col.meta);
        // Create a number formatter for the target locale (e.g., 'de-DE', 'en-US')
        const formatter = new Intl.NumberFormat((columnMeta === null || columnMeta === void 0 ? void 0 : columnMeta.currency_locale) || 'en-US');
        // If the locale is not set or is 'en-US', or the formatter does not support formatToParts, use the default behavior
        if (!(columnMeta === null || columnMeta === void 0 ? void 0 : columnMeta.currency_locale) ||
            columnMeta.currency_locale === 'en-US' ||
            typeof formatter.formatToParts !== 'function') {
            return (_a = value === null || value === void 0 ? void 0 : value.replace(/[^0-9.]/g, '')) === null || _a === void 0 ? void 0 : _a.trim();
        }
        const { group, decimal } = (0, currencyHelpers_1.getGroupDecimalSymbolFromLocale)(columnMeta === null || columnMeta === void 0 ? void 0 : columnMeta.currency_locale);
        return value
            .replace(new RegExp('\\' + group, 'g'), '') // 1. Remove all group (thousands) separators
            .replace(new RegExp('\\' + decimal), '.') // 2. Replace the locale-specific decimal separator with a dot (.)
            .replace(/[^\d.-]/g, '') // 3. Remove any non-digit, non-dot, non-minus characters (e.g., currency symbols, spaces)
            .trim(); // 4. Trim whitespace from both ends of the string
    });
};
exports.serializeCurrencyValue = serializeCurrencyValue;
const serializeTimeValue = (value, params) => {
    var _a;
    value = value === null || value === void 0 ? void 0 : value.toString().trim();
    if (!value)
        return null;
    let parsedTime = (0, dayjs_1.default)(value);
    if (!parsedTime.isValid()) {
        parsedTime = (0, dayjs_1.default)(value, 'HH:mm:ss');
    }
    if (!parsedTime.isValid()) {
        parsedTime = (0, dayjs_1.default)(`1999-01-01 ${value}`);
    }
    if (!parsedTime.isValid()) {
        return null;
    }
    const dateFormat = ((_a = params.isMysql) === null || _a === void 0 ? void 0 : _a.call(params, params.col.source_id))
        ? 'YYYY-MM-DD HH:mm:ss'
        : 'YYYY-MM-DD HH:mm:ssZ';
    return parsedTime.format(dateFormat);
};
exports.serializeTimeValue = serializeTimeValue;
const serializeYearValue = (value) => {
    value = value === null || value === void 0 ? void 0 : value.toString().trim();
    if (!value)
        return null;
    value = (0, exports.serializeIntValue)(value);
    const parsedDate = (0, dayjs_1.default)(value === null || value === void 0 ? void 0 : value.toString());
    return parsedDate.isValid() ? +parsedDate.format('YYYY') : null;
};
exports.serializeYearValue = serializeYearValue;
const serializeSelectValue = (value, col) => {
    var _a;
    // Always trim values for select fields to prevent creating options with leading/trailing spaces
    value = value === null || value === void 0 ? void 0 : value.toString().trim();
    // return null if value is empty
    if (!value)
        return null;
    // Todo: discuss new value creation
    const availableOptions = (((_a = col.colOptions) === null || _a === void 0 ? void 0 : _a.options) || []).map((o) => o.title);
    const optionsSet = new Set(availableOptions);
    let vals = value.split(',').map((val) => val.trim());
    const invalidVals = vals.filter((v) => !optionsSet.has(v));
    if (vals.length && col.uidt === UITypes_1.default.SingleSelect) {
        vals = [vals[0]];
    }
    // return null if no valid values
    if (invalidVals.length > 0) {
        throw new error_1.SelectTypeConversionError(vals, invalidVals);
    }
    return vals.join(',');
};
exports.serializeSelectValue = serializeSelectValue;
const serializeEmail = (v) => {
    const matches = v.match(/(([^<>()[\].,;:\s@"]+(\.[^<>()[\].,;:\s@"]+)*)|(".+"))@(([^<>()[\].,;:\s@"]+\.)+[^<>()[\].,;:\s@"]{2,})/i);
    return matches ? matches[0] : null;
};
exports.serializeEmail = serializeEmail;
//# sourceMappingURL=data:application/json;base64,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