"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isValidImageURL = exports.isValidURL = void 0;
const validator_1 = require("validator");
const is_1 = require("../../../lib/is");
const isValidURL = (str, extraProps) => {
    var _a;
    // Check if the string is empty or null
    if (!str || !(0, is_1.ncIsString)(str))
        return false;
    if (str.startsWith('mailto'))
        return (0, validator_1.isMailtoURI)(str);
    let require_host = (_a = extraProps === null || extraProps === void 0 ? void 0 : extraProps.require_host) !== null && _a !== void 0 ? _a : true;
    if (str.startsWith('file://') || str.startsWith('tel:')) {
        require_host = false;
    }
    return (0, validator_1.isURL)(`${str}`, Object.assign(Object.assign({}, (extraProps || {})), { require_host, protocols: ['http', 'https', 'ftp', 'file', 'tel'] }));
};
exports.isValidURL = isValidURL;
const isValidImageURL = async (url, options = {
    allowDataUrl: false,
    timeout: 5000,
}) => {
    if (!url || typeof url !== 'string')
        return false;
    const trimmed = url.trim();
    // 🚫 Always block dangerous / local schemes
    if (trimmed.startsWith('javascript:') || trimmed.startsWith('blob:')) {
        return false;
    }
    // 🚫 Block data URLs unless explicitly allowed
    if (!options.allowDataUrl && trimmed.startsWith('data:')) {
        return false;
    }
    // If data URLs are allowed, validate image data URL explicitly
    if (options.allowDataUrl && trimmed.startsWith('data:')) {
        return /^data:image\/[a-zA-Z0-9.+-]+;base64,/.test(trimmed);
    }
    // Normal URL validation
    if (!(0, validator_1.isURL)(trimmed, {
        protocols: ['http', 'https'],
        require_protocol: true,
    })) {
        return false;
    }
    return new Promise((resolve) => {
        const img = new Image();
        const timeout = setTimeout(() => {
            img.src = '';
            resolve(false);
        }, options.timeout);
        img.onload = () => {
            clearTimeout(timeout);
            resolve(true);
        };
        img.onerror = () => {
            clearTimeout(timeout);
            resolve(false);
        };
        img.src = trimmed;
    });
};
exports.isValidImageURL = isValidImageURL;
//# sourceMappingURL=data:application/json;base64,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