"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.removeUndefinedFromObj = void 0;
exports.extractProps = extractProps;
const is_1 = require("./is");
/**
 * Extracts properties from an object.
 * @param body - The object to extract properties from.
 * @param props - The properties to extract.
 * @param _throwError - Whether to throw an error if no properties are found.
 * @returns - The extracted properties.
 */
function extractProps(body, props, _throwError) {
    if (!(0, is_1.ncIsObject)(body))
        return {};
    // todo: throw error if no props found
    return props.reduce((o, key) => {
        if (key in body && body[key] !== undefined)
            o[key] = body[key];
        return o;
    }, {});
}
/**
 * Removes all `undefined` values and empty objects (`{}`) from an object.
 * Can optionally run recursively with the `deep` flag.
 *
 * - Works only on plain objects (`ncIsObject`).
 * - Arrays are preserved as-is (including `undefined` entries).
 * - Empty objects are always removed if they occur as object values.
 *
 * @typeParam T - Type of the input value.
 * @param obj - The object or value to clean.
 * @param deep - If `true`, cleans recursively. If `false`, cleans only top-level. Defaults to `true`.
 * @returns A cleaned copy of `obj`.
 *
 * @example
 * ```ts
 * const data = {
 *   a: 1,
 *   b: undefined,
 *   c: { d: 2, e: undefined, f: { g: undefined } },
 *   arr: [1, undefined, { x: undefined, y: 5 }]
 * }
 *
 * removeUndefinedFromObj(data)
 * // → { a: 1, c: { d: 2 }, arr: [1, undefined, { y: 5 }] }
 *
 * removeUndefinedFromObj(data, false)
 * // → { a: 1, c: { d: 2, f: { g: undefined } }, arr: [1, undefined, { x: undefined, y: 5 }] }
 * ```
 */
const removeUndefinedFromObj = (obj, deep = true) => {
    if ((0, is_1.ncIsObject)(obj)) {
        const cleanedEntries = Object.entries(obj)
            .map(([k, v]) => {
            const cleanedValue = deep && ((0, is_1.ncIsObject)(v) || Array.isArray(v))
                ? (0, exports.removeUndefinedFromObj)(v, deep)
                : v;
            return [k, cleanedValue];
        })
            .filter(([_, v]) => {
            if (v === undefined)
                return false;
            if ((0, is_1.ncIsObject)(v) && !Array.isArray(v) && (0, is_1.ncIsEmptyObject)(v))
                return false;
            return true;
        });
        return Object.keys(cleanedEntries).reduce((obj, key) => {
            obj[key] = cleanedEntries[key];
            return obj;
        }, {});
    }
    return obj;
};
exports.removeUndefinedFromObj = removeUndefinedFromObj;
//# sourceMappingURL=data:application/json;base64,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