"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.defaultGaugeWidgetRange = exports.getDefaultConfig = exports.calculateNextPosition = void 0;
const _1 = require(".");
const lib_1 = require("../../lib");
const calculateNextPosition = (existingWidgets, newWidgetDimensions, gridColumns = 4) => {
    if (existingWidgets.length === 0) {
        return { x: 0, y: 0 };
    }
    const { w: newW, h: newH } = newWidgetDimensions;
    // Early return if widget is too wide
    if (newW > gridColumns) {
        return { x: 0, y: 0 };
    }
    // Create a 2D grid to track occupied cells more efficiently
    const grid = new Map();
    let maxY = 0;
    // Build the occupation map
    for (const widget of existingWidgets) {
        const { x, y, w, h } = widget.position;
        const endY = y + h;
        maxY = Math.max(maxY, endY);
        for (let row = y; row < endY; row++) {
            if (!grid.has(row)) {
                grid.set(row, new Set());
            }
            const rowSet = grid.get(row);
            for (let col = x; col < x + w; col++) {
                rowSet.add(col);
            }
        }
    }
    // position checking
    const isPositionAvailable = (x, y) => {
        for (let row = y; row < y + newH; row++) {
            const rowSet = grid.get(row);
            if (rowSet) {
                for (let col = x; col < x + newW; col++) {
                    if (rowSet.has(col))
                        return false;
                }
            }
        }
        return true;
    };
    // Find the first available position, scanning row by row
    const maxX = gridColumns - newW;
    for (let y = 0; y <= maxY + 1; y++) {
        for (let x = 0; x <= maxX; x++) {
            if (isPositionAvailable(x, y)) {
                return { x, y };
            }
        }
    }
    // Fallback: place at bottom-left
    return { x: 0, y: maxY };
};
exports.calculateNextPosition = calculateNextPosition;
const defaultGaugeWidgetRange = [
    { color: '#FF6E76', min: 0, max: 33, label: 'Low' },
    { color: '#FDDD60', min: 33, max: 67, label: 'Medium' },
    { color: '#7CFFB2', min: 67, max: 100, label: 'High' },
];
exports.defaultGaugeWidgetRange = defaultGaugeWidgetRange;
const getDefaultConfig = (widgetType, chartType, columns) => {
    switch (widgetType) {
        case _1.WidgetTypes.METRIC:
            return {
                dataSource: _1.WidgetDataSourceTypes.MODEL,
                metric: {
                    type: 'count',
                    aggregation: 'count',
                },
                appearance: {
                    type: 'filled',
                    theme: 'purple',
                },
            };
        case _1.WidgetTypes.GAUGE:
            return {
                dataSource: _1.WidgetDataSourceTypes.MODEL,
                metric: {
                    type: 'count',
                    aggregation: 'count',
                },
                appearance: {
                    showValue: true,
                    ranges: defaultGaugeWidgetRange,
                },
            };
        case _1.WidgetTypes.CHART:
            return getDefaultChartConfig(chartType, columns);
        case _1.WidgetTypes.IFRAME:
            return {
                url: 'https://nocodb.com',
                allowFullScreen: false,
            };
        case _1.WidgetTypes.TEXT:
            return {
                content: 'Enter text here',
                type: _1.TextWidgetTypes.Markdown,
                formatting: {
                    horizontalAlign: 'flex-start',
                    verticalAlign: 'flex-start',
                    bold: false,
                    italic: false,
                    underline: false,
                    strikethrough: false,
                },
                appearance: {
                    font: {
                        family: 'Arial',
                        weight: 400,
                        size: 16,
                        lineHeight: 1,
                    },
                    color: '#000000',
                },
            };
        default:
            return {};
    }
};
exports.getDefaultConfig = getDefaultConfig;
const getDefaultChartConfig = (chartType, columns) => {
    const baseConfig = {
        dataSource: _1.WidgetDataSourceTypes.MODEL,
    };
    // Helper function to get the best column for category field
    const getDefaultCategoryColumn = (columns) => {
        var _a;
        if (!columns || (columns === null || columns === void 0 ? void 0 : columns.length) === 0)
            return '';
        // Priority 1: SingleSelect
        const singleSelectColumn = columns.find((col) => col.uidt === lib_1.UITypes.SingleSelect);
        if (singleSelectColumn)
            return singleSelectColumn.id;
        // Priority 2: SingleLineText
        const singleLineTextColumn = columns.find((col) => col.uidt === lib_1.UITypes.SingleLineText);
        if (singleLineTextColumn)
            return singleLineTextColumn.id;
        // Fallback: first column
        return ((_a = columns[0]) === null || _a === void 0 ? void 0 : _a.id) || '';
    };
    const getDefaultYAxisColumn = (columns) => {
        var _a;
        if (!columns || (columns === null || columns === void 0 ? void 0 : columns.length) === 0)
            return '';
        // Priority 1: SingleSelect
        const singleSelectColumn = columns.find((col) => col.uidt === lib_1.UITypes.SingleSelect);
        if (singleSelectColumn)
            return singleSelectColumn.id;
        // Priority 2: SingleLineText
        const singleLineTextColumn = columns.find((col) => col.uidt === lib_1.UITypes.SingleLineText);
        if (singleLineTextColumn)
            return singleLineTextColumn.id;
        // Fallback: first column
        return (((_a = columns.find((col) => !(0, lib_1.isSystemColumn)(col) &&
            !(0, lib_1.isButton)(col) &&
            !(0, lib_1.isLookup)(col) &&
            !(0, lib_1.isQrCode)(col) &&
            !(0, lib_1.isBarcode)(col))) === null || _a === void 0 ? void 0 : _a.id) || null);
    };
    switch (chartType) {
        case _1.ChartTypes.PIE:
            return Object.assign(Object.assign({}, baseConfig), { chartType: _1.ChartTypes.PIE, data: {
                    category: {
                        column_id: getDefaultCategoryColumn(columns),
                        orderBy: 'default',
                        includeEmptyRecords: false,
                        includeOthers: false,
                        categoryLimit: 10,
                    },
                    value: {
                        type: 'count',
                    },
                }, appearance: {
                    size: 'medium',
                    showCountInLegend: true,
                    showPercentageOnChart: true,
                    legendPosition: 'right',
                    colorSchema: 'default',
                    customColorSchema: [],
                }, permissions: {
                    allowUserToPrint: true,
                    allowUsersToViewRecords: false,
                } });
        case _1.ChartTypes.DONUT:
            return Object.assign(Object.assign({}, baseConfig), { chartType: _1.ChartTypes.DONUT, data: {
                    category: {
                        column_id: getDefaultCategoryColumn(columns),
                        orderBy: 'default',
                        includeEmptyRecords: false,
                        includeOthers: false,
                        categoryLimit: 10,
                    },
                    value: {
                        type: 'count',
                    },
                }, appearance: {
                    size: 'medium',
                    showCountInLegend: true,
                    showPercentageOnChart: true,
                    legendPosition: 'right',
                    colorSchema: 'default',
                    customColorSchema: [],
                }, permissions: {
                    allowUserToPrint: true,
                    allowUsersToViewRecords: false,
                } });
        case _1.ChartTypes.BAR:
        case _1.ChartTypes.LINE:
            return Object.assign(Object.assign({}, baseConfig), { chartType: chartType, appearance: {
                    showCountInLegend: true,
                    legendPosition: 'top',
                    showPercentageOnChart: true,
                }, data: {
                    xAxis: {
                        column_id: getDefaultCategoryColumn(columns),
                        sortBy: 'xAxis',
                        orderBy: 'default',
                        includeEmptyRecords: false,
                        includeOthers: false,
                        categoryLimit: 10,
                    },
                    yAxis: {
                        startAtZero: true,
                        fields: [
                            {
                                column_id: getDefaultYAxisColumn(columns),
                                aggregation: lib_1.AllAggregations.CountUnique,
                            },
                        ],
                        groupBy: null,
                    },
                }, permissions: {
                    allowUserToPrint: false,
                    allowUsersToViewRecords: false,
                } });
        default:
            return Object.assign(Object.assign({}, baseConfig), { chartType });
    }
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoid2lkZ2V0VXRpbHMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi9zcmMvbGliL2Rhc2hib2FyZC93aWRnZXRVdGlscy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx3QkFNVztBQUNYLCtCQVNlO0FBRVIsTUFBTSxxQkFBcUIsR0FBRyxDQUNuQyxlQUE2QixFQUM3QixtQkFBNkMsRUFDN0MsY0FBc0IsQ0FBQyxFQUN2QixFQUFFO0lBQ0YsSUFBSSxlQUFlLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ2pDLE9BQU8sRUFBRSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQztJQUN4QixDQUFDO0lBRUQsTUFBTSxFQUFFLENBQUMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxFQUFFLElBQUksRUFBRSxHQUFHLG1CQUFtQixDQUFDO0lBRWpELHFDQUFxQztJQUNyQyxJQUFJLElBQUksR0FBRyxXQUFXLEVBQUUsQ0FBQztRQUN2QixPQUFPLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUM7SUFDeEIsQ0FBQztJQUVELDREQUE0RDtJQUM1RCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsRUFBdUIsQ0FBQztJQUM1QyxJQUFJLElBQUksR0FBRyxDQUFDLENBQUM7SUFFYiwyQkFBMkI7SUFDM0IsS0FBSyxNQUFNLE1BQU0sSUFBSSxlQUFlLEVBQUUsQ0FBQztRQUNyQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxFQUFFLEdBQUcsTUFBTSxDQUFDLFFBQVEsQ0FBQztRQUN2QyxNQUFNLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ25CLElBQUksR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztRQUU1QixLQUFLLElBQUksR0FBRyxHQUFHLENBQUMsRUFBRSxHQUFHLEdBQUcsSUFBSSxFQUFFLEdBQUcsRUFBRSxFQUFFLENBQUM7WUFDcEMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQztnQkFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLEVBQUUsSUFBSSxHQUFHLEVBQUUsQ0FBQyxDQUFDO1lBQzNCLENBQUM7WUFDRCxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBRSxDQUFDO1lBQzlCLEtBQUssSUFBSSxHQUFHLEdBQUcsQ0FBQyxFQUFFLEdBQUcsR0FBRyxDQUFDLEdBQUcsQ0FBQyxFQUFFLEdBQUcsRUFBRSxFQUFFLENBQUM7Z0JBQ3JDLE1BQU0sQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDbEIsQ0FBQztRQUNILENBQUM7SUFDSCxDQUFDO0lBRUQsb0JBQW9CO0lBQ3BCLE1BQU0sbUJBQW1CLEdBQUcsQ0FBQyxDQUFTLEVBQUUsQ0FBUyxFQUFXLEVBQUU7UUFDNUQsS0FBSyxJQUFJLEdBQUcsR0FBRyxDQUFDLEVBQUUsR0FBRyxHQUFHLENBQUMsR0FBRyxJQUFJLEVBQUUsR0FBRyxFQUFFLEVBQUUsQ0FBQztZQUN4QyxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQzdCLElBQUksTUFBTSxFQUFFLENBQUM7Z0JBQ1gsS0FBSyxJQUFJLEdBQUcsR0FBRyxDQUFDLEVBQUUsR0FBRyxHQUFHLENBQUMsR0FBRyxJQUFJLEVBQUUsR0FBRyxFQUFFLEVBQUUsQ0FBQztvQkFDeEMsSUFBSSxNQUFNLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQzt3QkFBRSxPQUFPLEtBQUssQ0FBQztnQkFDcEMsQ0FBQztZQUNILENBQUM7UUFDSCxDQUFDO1FBQ0QsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDLENBQUM7SUFFRix5REFBeUQ7SUFDekQsTUFBTSxJQUFJLEdBQUcsV0FBVyxHQUFHLElBQUksQ0FBQztJQUNoQyxLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLElBQUksSUFBSSxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDO1FBQ25DLEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsSUFBSSxJQUFJLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQztZQUMvQixJQUFJLG1CQUFtQixDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDO2dCQUM5QixPQUFPLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDO1lBQ2xCLENBQUM7UUFDSCxDQUFDO0lBQ0gsQ0FBQztJQUVELGlDQUFpQztJQUNqQyxPQUFPLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsSUFBSSxFQUFFLENBQUM7QUFDM0IsQ0FBQyxDQUFDO0FBOURXLFFBQUEscUJBQXFCLHlCQThEaEM7QUFFRixNQUFNLHVCQUF1QixHQUFHO0lBQzlCLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFFLEdBQUcsRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBRTtJQUNuRCxFQUFFLEtBQUssRUFBRSxTQUFTLEVBQUUsR0FBRyxFQUFFLEVBQUUsRUFBRSxHQUFHLEVBQUUsRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLEVBQUU7SUFDdkQsRUFBRSxLQUFLLEVBQUUsU0FBUyxFQUFFLEdBQUcsRUFBRSxFQUFFLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFO0NBQ3ZELENBQUM7QUFzT3lCLDBEQUF1QjtBQXBPbEQsTUFBTSxnQkFBZ0IsR0FBRyxDQUN2QixVQUF1QixFQUN2QixTQUFzQixFQUN0QixPQUEyQixFQUMzQixFQUFFO0lBQ0YsUUFBUSxVQUFVLEVBQUUsQ0FBQztRQUNuQixLQUFLLGNBQVcsQ0FBQyxNQUFNO1lBQ3JCLE9BQU87Z0JBQ0wsVUFBVSxFQUFFLHdCQUFxQixDQUFDLEtBQUs7Z0JBQ3ZDLE1BQU0sRUFBRTtvQkFDTixJQUFJLEVBQUUsT0FBZ0I7b0JBQ3RCLFdBQVcsRUFBRSxPQUFnQjtpQkFDOUI7Z0JBQ0QsVUFBVSxFQUFFO29CQUNWLElBQUksRUFBRSxRQUFpQjtvQkFDdkIsS0FBSyxFQUFFLFFBQWlCO2lCQUN6QjthQUNGLENBQUM7UUFFSixLQUFLLGNBQVcsQ0FBQyxLQUFLO1lBQ3BCLE9BQU87Z0JBQ0wsVUFBVSxFQUFFLHdCQUFxQixDQUFDLEtBQUs7Z0JBQ3ZDLE1BQU0sRUFBRTtvQkFDTixJQUFJLEVBQUUsT0FBZ0I7b0JBQ3RCLFdBQVcsRUFBRSxPQUFnQjtpQkFDOUI7Z0JBQ0QsVUFBVSxFQUFFO29CQUNWLFNBQVMsRUFBRSxJQUFJO29CQUNmLE1BQU0sRUFBRSx1QkFBdUI7aUJBQ2hDO2FBQ0YsQ0FBQztRQUVKLEtBQUssY0FBVyxDQUFDLEtBQUs7WUFDcEIsT0FBTyxxQkFBcUIsQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDbkQsS0FBSyxjQUFXLENBQUMsTUFBTTtZQUNyQixPQUFPO2dCQUNMLEdBQUcsRUFBRSxvQkFBb0I7Z0JBQ3pCLGVBQWUsRUFBRSxLQUFLO2FBQ3ZCLENBQUM7UUFDSixLQUFLLGNBQVcsQ0FBQyxJQUFJO1lBQ25CLE9BQU87Z0JBQ0wsT0FBTyxFQUFFLGlCQUFpQjtnQkFDMUIsSUFBSSxFQUFFLGtCQUFlLENBQUMsUUFBUTtnQkFDOUIsVUFBVSxFQUFFO29CQUNWLGVBQWUsRUFBRSxZQUFZO29CQUM3QixhQUFhLEVBQUUsWUFBWTtvQkFDM0IsSUFBSSxFQUFFLEtBQUs7b0JBQ1gsTUFBTSxFQUFFLEtBQUs7b0JBQ2IsU0FBUyxFQUFFLEtBQUs7b0JBQ2hCLGFBQWEsRUFBRSxLQUFLO2lCQUNyQjtnQkFDRCxVQUFVLEVBQUU7b0JBQ1YsSUFBSSxFQUFFO3dCQUNKLE1BQU0sRUFBRSxPQUFPO3dCQUNmLE1BQU0sRUFBRSxHQUFHO3dCQUNYLElBQUksRUFBRSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxDQUFDO3FCQUNkO29CQUNELEtBQUssRUFBRSxTQUFTO2lCQUNqQjthQUNGLENBQUM7UUFFSjtZQUNFLE9BQU8sRUFBRSxDQUFDO0lBQ2QsQ0FBQztBQUNILENBQUMsQ0FBQztBQW1LTyw0Q0FBZ0I7QUFqS3pCLE1BQU0scUJBQXFCLEdBQUcsQ0FDNUIsU0FBc0IsRUFDdEIsT0FBMkIsRUFDM0IsRUFBRTtJQUNGLE1BQU0sVUFBVSxHQUFHO1FBQ2pCLFVBQVUsRUFBRSx3QkFBcUIsQ0FBQyxLQUFLO0tBQ3hDLENBQUM7SUFFRiw0REFBNEQ7SUFDNUQsTUFBTSx3QkFBd0IsR0FBRyxDQUFDLE9BQTJCLEVBQVUsRUFBRTs7UUFDdkUsSUFBSSxDQUFDLE9BQU8sSUFBSSxDQUFBLE9BQU8sYUFBUCxPQUFPLHVCQUFQLE9BQU8sQ0FBRSxNQUFNLE1BQUssQ0FBQztZQUFFLE9BQU8sRUFBRSxDQUFDO1FBRWpELDJCQUEyQjtRQUMzQixNQUFNLGtCQUFrQixHQUFHLE9BQU8sQ0FBQyxJQUFJLENBQ3JDLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxHQUFHLENBQUMsSUFBSSxLQUFLLGFBQU8sQ0FBQyxZQUFZLENBQzNDLENBQUM7UUFDRixJQUFJLGtCQUFrQjtZQUFFLE9BQU8sa0JBQWtCLENBQUMsRUFBRSxDQUFDO1FBRXJELDZCQUE2QjtRQUM3QixNQUFNLG9CQUFvQixHQUFHLE9BQU8sQ0FBQyxJQUFJLENBQ3ZDLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxHQUFHLENBQUMsSUFBSSxLQUFLLGFBQU8sQ0FBQyxjQUFjLENBQzdDLENBQUM7UUFDRixJQUFJLG9CQUFvQjtZQUFFLE9BQU8sb0JBQW9CLENBQUMsRUFBRSxDQUFDO1FBRXpELHlCQUF5QjtRQUN6QixPQUFPLENBQUEsTUFBQSxPQUFPLENBQUMsQ0FBQyxDQUFDLDBDQUFFLEVBQUUsS0FBSSxFQUFFLENBQUM7SUFDOUIsQ0FBQyxDQUFDO0lBRUYsTUFBTSxxQkFBcUIsR0FBRyxDQUFDLE9BQTJCLEVBQVUsRUFBRTs7UUFDcEUsSUFBSSxDQUFDLE9BQU8sSUFBSSxDQUFBLE9BQU8sYUFBUCxPQUFPLHVCQUFQLE9BQU8sQ0FBRSxNQUFNLE1BQUssQ0FBQztZQUFFLE9BQU8sRUFBRSxDQUFDO1FBRWpELDJCQUEyQjtRQUMzQixNQUFNLGtCQUFrQixHQUFHLE9BQU8sQ0FBQyxJQUFJLENBQ3JDLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxHQUFHLENBQUMsSUFBSSxLQUFLLGFBQU8sQ0FBQyxZQUFZLENBQzNDLENBQUM7UUFDRixJQUFJLGtCQUFrQjtZQUFFLE9BQU8sa0JBQWtCLENBQUMsRUFBRSxDQUFDO1FBRXJELDZCQUE2QjtRQUM3QixNQUFNLG9CQUFvQixHQUFHLE9BQU8sQ0FBQyxJQUFJLENBQ3ZDLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxHQUFHLENBQUMsSUFBSSxLQUFLLGFBQU8sQ0FBQyxjQUFjLENBQzdDLENBQUM7UUFDRixJQUFJLG9CQUFvQjtZQUFFLE9BQU8sb0JBQW9CLENBQUMsRUFBRSxDQUFDO1FBRXpELHlCQUF5QjtRQUN6QixPQUFPLENBQ0wsQ0FBQSxNQUFBLE9BQU8sQ0FBQyxJQUFJLENBQ1YsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUNOLENBQUMsSUFBQSxvQkFBYyxFQUFDLEdBQUcsQ0FBQztZQUNwQixDQUFDLElBQUEsY0FBUSxFQUFDLEdBQUcsQ0FBQztZQUNkLENBQUMsSUFBQSxjQUFRLEVBQUMsR0FBRyxDQUFDO1lBQ2QsQ0FBQyxJQUFBLGNBQVEsRUFBQyxHQUFHLENBQUM7WUFDZCxDQUFDLElBQUEsZUFBUyxFQUFDLEdBQUcsQ0FBQyxDQUNsQiwwQ0FBRSxFQUFFLEtBQUksSUFBSSxDQUNkLENBQUM7SUFDSixDQUFDLENBQUM7SUFFRixRQUFRLFNBQVMsRUFBRSxDQUFDO1FBQ2xCLEtBQUssYUFBVSxDQUFDLEdBQUc7WUFDakIsdUNBQ0ssVUFBVSxLQUNiLFNBQVMsRUFBRSxhQUFVLENBQUMsR0FBRyxFQUN6QixJQUFJLEVBQUU7b0JBQ0osUUFBUSxFQUFFO3dCQUNSLFNBQVMsRUFBRSx3QkFBd0IsQ0FBQyxPQUFPLENBQUM7d0JBQzVDLE9BQU8sRUFBRSxTQUFrQjt3QkFDM0IsbUJBQW1CLEVBQUUsS0FBSzt3QkFDMUIsYUFBYSxFQUFFLEtBQUs7d0JBQ3BCLGFBQWEsRUFBRSxFQUFFO3FCQUNsQjtvQkFDRCxLQUFLLEVBQUU7d0JBQ0wsSUFBSSxFQUFFLE9BQWdCO3FCQUN2QjtpQkFDRixFQUNELFVBQVUsRUFBRTtvQkFDVixJQUFJLEVBQUUsUUFBaUI7b0JBQ3ZCLGlCQUFpQixFQUFFLElBQUk7b0JBQ3ZCLHFCQUFxQixFQUFFLElBQUk7b0JBQzNCLGNBQWMsRUFBRSxPQUFnQjtvQkFDaEMsV0FBVyxFQUFFLFNBQWtCO29CQUMvQixpQkFBaUIsRUFBRSxFQUFFO2lCQUN0QixFQUNELFdBQVcsRUFBRTtvQkFDWCxnQkFBZ0IsRUFBRSxJQUFJO29CQUN0Qix1QkFBdUIsRUFBRSxLQUFLO2lCQUMvQixJQUNEO1FBRUosS0FBSyxhQUFVLENBQUMsS0FBSztZQUNuQix1Q0FDSyxVQUFVLEtBQ2IsU0FBUyxFQUFFLGFBQVUsQ0FBQyxLQUFLLEVBQzNCLElBQUksRUFBRTtvQkFDSixRQUFRLEVBQUU7d0JBQ1IsU0FBUyxFQUFFLHdCQUF3QixDQUFDLE9BQU8sQ0FBQzt3QkFDNUMsT0FBTyxFQUFFLFNBQWtCO3dCQUMzQixtQkFBbUIsRUFBRSxLQUFLO3dCQUMxQixhQUFhLEVBQUUsS0FBSzt3QkFDcEIsYUFBYSxFQUFFLEVBQUU7cUJBQ2xCO29CQUNELEtBQUssRUFBRTt3QkFDTCxJQUFJLEVBQUUsT0FBZ0I7cUJBQ3ZCO2lCQUNGLEVBQ0QsVUFBVSxFQUFFO29CQUNWLElBQUksRUFBRSxRQUFpQjtvQkFDdkIsaUJBQWlCLEVBQUUsSUFBSTtvQkFDdkIscUJBQXFCLEVBQUUsSUFBSTtvQkFDM0IsY0FBYyxFQUFFLE9BQWdCO29CQUNoQyxXQUFXLEVBQUUsU0FBa0I7b0JBQy9CLGlCQUFpQixFQUFFLEVBQUU7aUJBQ3RCLEVBQ0QsV0FBVyxFQUFFO29CQUNYLGdCQUFnQixFQUFFLElBQUk7b0JBQ3RCLHVCQUF1QixFQUFFLEtBQUs7aUJBQy9CLElBQ0Q7UUFFSixLQUFLLGFBQVUsQ0FBQyxHQUFHLENBQUM7UUFDcEIsS0FBSyxhQUFVLENBQUMsSUFBSTtZQUNsQix1Q0FDSyxVQUFVLEtBQ2IsU0FBUyxFQUFFLFNBQVMsRUFDcEIsVUFBVSxFQUFFO29CQUNWLGlCQUFpQixFQUFFLElBQUk7b0JBQ3ZCLGNBQWMsRUFBRSxLQUFLO29CQUNyQixxQkFBcUIsRUFBRSxJQUFJO2lCQUM1QixFQUNELElBQUksRUFBRTtvQkFDSixLQUFLLEVBQUU7d0JBQ0wsU0FBUyxFQUFFLHdCQUF3QixDQUFDLE9BQU8sQ0FBQzt3QkFDNUMsTUFBTSxFQUFFLE9BQWdCO3dCQUN4QixPQUFPLEVBQUUsU0FBa0I7d0JBQzNCLG1CQUFtQixFQUFFLEtBQUs7d0JBQzFCLGFBQWEsRUFBRSxLQUFLO3dCQUNwQixhQUFhLEVBQUUsRUFBRTtxQkFDbEI7b0JBQ0QsS0FBSyxFQUFFO3dCQUNMLFdBQVcsRUFBRSxJQUFJO3dCQUNqQixNQUFNLEVBQUU7NEJBQ047Z0NBQ0UsU0FBUyxFQUFFLHFCQUFxQixDQUFDLE9BQU8sQ0FBQztnQ0FDekMsV0FBVyxFQUFFLHFCQUFlLENBQUMsV0FBVzs2QkFDekM7eUJBQ0Y7d0JBQ0QsT0FBTyxFQUFFLElBQUk7cUJBQ2Q7aUJBQ0YsRUFDRCxXQUFXLEVBQUU7b0JBQ1gsZ0JBQWdCLEVBQUUsS0FBSztvQkFDdkIsdUJBQXVCLEVBQUUsS0FBSztpQkFDL0IsSUFDRDtRQUVKO1lBQ0UsdUNBQ0ssVUFBVSxLQUNiLFNBQVMsSUFDVDtJQUNOLENBQUM7QUFDSCxDQUFDLENBQUMifQ==