"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getDateValue = exports.getDateTimeValue = exports.isValidTimeFormat = exports.timeAgo = exports.handleTZ = exports.isDateMonthFormat = exports.dateFormats = exports.timeFormats = exports.dateMonthFormats = void 0;
exports.validateDateWithUnknownFormat = validateDateWithUnknownFormat;
exports.getDateFormat = getDateFormat;
exports.getDateTimeFormat = getDateTimeFormat;
exports.parseStringDate = parseStringDate;
exports.parseStringDateTime = parseStringDateTime;
exports.convertToTargetFormat = convertToTargetFormat;
exports.validateDateFormat = validateDateFormat;
exports.constructDateTimeFormat = constructDateTimeFormat;
exports.constructDateFormat = constructDateFormat;
exports.constructTimeFormat = constructTimeFormat;
exports.workerWithTimezone = workerWithTimezone;
const dayjs_1 = __importDefault(require("dayjs"));
const relativeTime_js_1 = __importDefault(require("dayjs/plugin/relativeTime.js"));
const customParseFormat_js_1 = __importDefault(require("dayjs/plugin/customParseFormat.js"));
const duration_js_1 = __importDefault(require("dayjs/plugin/duration.js"));
const utc_js_1 = __importDefault(require("dayjs/plugin/utc.js"));
const weekday_js_1 = __importDefault(require("dayjs/plugin/weekday.js"));
const timezone_js_1 = __importDefault(require("dayjs/plugin/timezone.js"));
const helperFunctions_1 = require("./helperFunctions");
const is_1 = require("./is");
dayjs_1.default.extend(utc_js_1.default);
dayjs_1.default.extend(relativeTime_js_1.default);
dayjs_1.default.extend(customParseFormat_js_1.default);
dayjs_1.default.extend(duration_js_1.default);
dayjs_1.default.extend(weekday_js_1.default);
dayjs_1.default.extend(timezone_js_1.default);
exports.dateMonthFormats = ['YYYY-MM', 'YYYY MM'];
exports.timeFormats = ['HH:mm', 'HH:mm:ss', 'HH:mm:ss.SSS'];
exports.dateFormats = [
    'YYYY-MM-DD',
    'YYYY/MM/DD',
    'DD-MM-YYYY',
    'MM-DD-YYYY',
    'DD/MM/YYYY',
    'MM/DD/YYYY',
    'DD MM YYYY',
    'MM DD YYYY',
    'YYYY MM DD',
    ///added 2 new format#9652
    'DD MMM YYYY',
    'DD MMM YY',
    // German date notations
    'DD.MM.YYYY',
    'DD.MM.YY',
];
const isDateMonthFormat = (format) => exports.dateMonthFormats.includes(format);
exports.isDateMonthFormat = isDateMonthFormat;
function validateDateWithUnknownFormat(v) {
    for (const format of exports.dateFormats) {
        if ((0, dayjs_1.default)(v, format, true).isValid()) {
            return true;
        }
        for (const timeFormat of exports.timeFormats) {
            if ((0, dayjs_1.default)(v, `${format} ${timeFormat}`, true).isValid()) {
                return true;
            }
        }
    }
    return false;
}
function getDateFormat(v) {
    for (const format of exports.dateFormats.concat(exports.dateMonthFormats)) {
        if ((0, dayjs_1.default)(v, format, true).isValid()) {
            return format;
        }
    }
    return 'YYYY/MM/DD';
}
function getDateTimeFormat(v) {
    for (const format of exports.dateFormats) {
        for (const timeFormat of exports.timeFormats) {
            const dateTimeFormat = `${format} ${timeFormat}`;
            if ((0, dayjs_1.default)(v, dateTimeFormat, true).isValid()) {
                return dateTimeFormat;
            }
        }
    }
    return 'YYYY/MM/DD HH:mm';
}
function parseStringDate(v, dateFormat) {
    const dayjsObj = (0, dayjs_1.default)(v);
    if (dayjsObj.isValid()) {
        v = dayjsObj.format('YYYY-MM-DD');
    }
    else {
        v = (0, dayjs_1.default)(v, dateFormat).format('YYYY-MM-DD');
    }
    return v;
}
function parseStringDateTime(v, dateTimeFormat = `${exports.dateFormats[0]} ${exports.timeFormats[0]}`, toLocal = true) {
    const dayjsObj = toLocal ? (0, dayjs_1.default)(v).local() : (0, dayjs_1.default)(v);
    if (dayjsObj.isValid()) {
        v = dayjsObj.format(dateTimeFormat);
    }
    else {
        v = toLocal
            ? (0, dayjs_1.default)(v, dateTimeFormat).local().format(dateTimeFormat)
            : (0, dayjs_1.default)(v, dateTimeFormat).format(dateTimeFormat);
    }
    return v;
}
function convertToTargetFormat(v, oldDataFormat, newDateFormat) {
    if (!exports.dateFormats.includes(oldDataFormat) ||
        !exports.dateFormats.includes(newDateFormat))
        return v;
    return (0, dayjs_1.default)(v, oldDataFormat).format(newDateFormat);
}
const handleTZ = (val) => {
    if (val === undefined || val === null) {
        return;
    }
    if (typeof val !== 'string') {
        return val;
    }
    return val.replace(
    // match and extract dates and times in the ISO 8601 format
    /((?:-?(?:[1-9][0-9]*)?[0-9]{4})-(?:1[0-2]|0[1-9])-(?:3[01]|0[1-9]|[12][0-9])T(?:2[0-3]|[01][0-9]):(?:[0-5][0-9]):(?:[0-5][0-9])(?:\.[0-9]+)?(?:Z|[+-](?:2[0-3]|[01][0-9]):[0-5][0-9]))/g, (_, v) => {
        return (0, dayjs_1.default)(v).format('YYYY-MM-DD HH:mm');
    });
};
exports.handleTZ = handleTZ;
function validateDateFormat(v) {
    return exports.dateFormats.includes(v);
}
const timeAgo = (date) => {
    if (/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}$/.test(date)) {
        // if there is no timezone info, consider as UTC
        // e.g. 2023-01-01 08:00:00 (MySQL)
        date += '+00:00';
    }
    // show in local time
    return (0, dayjs_1.default)(date).fromNow();
};
exports.timeAgo = timeAgo;
const isValidTimeFormat = (value, format) => {
    const regexValidator = {
        [exports.timeFormats[0]]: /^([01]\d|2[0-3]):[0-5]\d$/,
        [exports.timeFormats[1]]: /^([01]\d|2[0-3]):[0-5]\d:[0-5]\d$/,
        [exports.timeFormats[2]]: /^([01]\d|2[0-3]):[0-5]\d:[0-5]\d\.\d{3}$/,
    };
    if (regexValidator[format]) {
        return regexValidator[format].test(value);
    }
    return false;
};
exports.isValidTimeFormat = isValidTimeFormat;
function constructDateTimeFormat(column) {
    const dateFormat = constructDateFormat(column);
    const timeFormat = constructTimeFormat(column);
    return `${dateFormat} ${timeFormat}`;
}
function constructDateFormat(column) {
    var _a, _b;
    return (_b = (_a = (0, helperFunctions_1.parseProp)(column === null || column === void 0 ? void 0 : column.meta)) === null || _a === void 0 ? void 0 : _a.date_format) !== null && _b !== void 0 ? _b : exports.dateFormats[0];
}
function constructTimeFormat(column) {
    var _a;
    const columnMeta = (0, helperFunctions_1.parseProp)(column === null || column === void 0 ? void 0 : column.meta);
    const metaTimeFormat = (_a = columnMeta.time_format) !== null && _a !== void 0 ? _a : exports.timeFormats[0];
    return (columnMeta === null || columnMeta === void 0 ? void 0 : columnMeta.is12hrFormat)
        ? metaTimeFormat.replace('HH', 'hh') + ' A' // if 12h, replace HH and add AM/PM at the end
        : metaTimeFormat;
}
function workerWithTimezone(isEeUI, timezone) {
    // Check if the timezone is UTC or GMT (case insensitive)
    const isUtcOrGmt = timezone && /^(utc|gmt)$/i.test(timezone);
    return {
        dayjsTz(value, format) {
            if (!isEeUI) {
                return (0, dayjs_1.default)(value, format);
            }
            if ((0, is_1.ncIsNull)(value) || (0, is_1.ncIsUndefined)(value)) {
                if (timezone) {
                    return dayjs_1.default.tz(undefined, timezone);
                }
                else {
                    return (0, dayjs_1.default)();
                }
            }
            else if (typeof value === 'object' && value.isValid()) {
                return value;
            }
            if (timezone) {
                if (isUtcOrGmt) {
                    const strValue = typeof value === 'object' &&
                        typeof value.isValid === 'function' &&
                        value.isValid()
                        ? value.toISOString()
                        : value;
                    return format
                        ? dayjs_1.default.tz(strValue, format, timezone)
                        : dayjs_1.default.tz(strValue, timezone);
                }
                else {
                    if (!format) {
                        return dayjs_1.default.tz(value, timezone);
                    }
                    else {
                        return dayjs_1.default.tz(value, format, timezone);
                    }
                }
            }
            else {
                return (0, dayjs_1.default)(value, format);
            }
        },
        timezonize(value) {
            if (!isEeUI || !timezone) {
                return (0, dayjs_1.default)(value);
            }
            if (!value) {
                return this.dayjsTz();
            }
            let dayjsObject;
            if (typeof value === 'object' &&
                typeof value.isValid === 'function' &&
                value.isValid()) {
                dayjsObject = value.isUTC() ? value : value.utc();
            }
            else {
                dayjsObject = dayjs_1.default.utc(value);
            }
            if (!isEeUI) {
                return dayjsObject.local();
            }
            if (timezone) {
                if (isUtcOrGmt) {
                    return (0, dayjs_1.default)(dayjsObject.toISOString()).tz(timezone);
                }
                else {
                    return dayjsObject.tz(timezone);
                }
            }
            return dayjsObject.local();
        },
    };
}
const getDateTimeValue = (modelValue, col, isXcdbBase) => {
    var _a, _b, _c, _d;
    if (!modelValue || !(0, dayjs_1.default)(modelValue).isValid()) {
        return '';
    }
    const dateFormat = (_b = (_a = (0, helperFunctions_1.parseProp)(col === null || col === void 0 ? void 0 : col.meta)) === null || _a === void 0 ? void 0 : _a.date_format) !== null && _b !== void 0 ? _b : exports.dateFormats[0];
    const timeFormat = (_d = (_c = (0, helperFunctions_1.parseProp)(col === null || col === void 0 ? void 0 : col.meta)) === null || _c === void 0 ? void 0 : _c.time_format) !== null && _d !== void 0 ? _d : exports.timeFormats[0];
    const dateTimeFormat = `${dateFormat} ${timeFormat}`;
    if (!isXcdbBase) {
        return (0, dayjs_1.default)(/^\d+$/.test(modelValue) ? +modelValue : modelValue, dateTimeFormat).format(dateTimeFormat);
    }
    return (0, dayjs_1.default)(modelValue).utc().local().format(dateTimeFormat);
};
exports.getDateTimeValue = getDateTimeValue;
const getDateValue = (modelValue, col, isSystemCol) => {
    var _a, _b;
    const dateFormat = !isSystemCol
        ? (_b = (_a = (0, helperFunctions_1.parseProp)(col.meta)) === null || _a === void 0 ? void 0 : _a.date_format) !== null && _b !== void 0 ? _b : 'YYYY-MM-DD'
        : 'YYYY-MM-DD HH:mm:ss';
    if (!modelValue || !(0, dayjs_1.default)(modelValue).isValid()) {
        return '';
    }
    return (0, dayjs_1.default)(/^\d+$/.test(String(modelValue)) ? +modelValue : modelValue).format(dateFormat);
};
exports.getDateValue = getDateValue;
//# sourceMappingURL=data:application/json;base64,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