"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NcErrorBase = void 0;
const nc_base_error_1 = require("../error/nc-base.error");
const globals_1 = require("../globals");
const payment_1 = require("../payment");
const nc_error_codex_manager_1 = require("./nc-error-codex-manager");
class NcErrorBase {
    constructor() {
        this._errorCodex = new nc_error_codex_manager_1.NcErrorCodexManager();
    }
    get errorCodex() {
        return this._errorCodex;
    }
    authenticationRequired(args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_AUTHENTICATION_REQUIRED, args);
    }
    apiTokenNotAllowed(args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_API_TOKEN_NOT_ALLOWED, args);
    }
    workspaceNotFound(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_WORKSPACE_NOT_FOUND, Object.assign({ params: id }, args));
    }
    columnAssociatedWithLink(_id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_COLUMN_ASSOCIATED_WITH_LINK, args);
    }
    tableAssociatedWithLink(_id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_TABLE_ASSOCIATED_WITH_LINK, args);
    }
    viewColumnNotFound(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_VIEW_COLUMN_NOT_FOUND, Object.assign({ params: id }, args));
    }
    baseNotFound(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_BASE_NOT_FOUND, Object.assign({ params: id }, args));
    }
    dashboardNotFound(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_DASHBOARD_NOT_FOUND, Object.assign({ params: id }, args));
    }
    workflowNotFound(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_WORKFLOW_NOT_FOUND, Object.assign({ params: id }, args));
    }
    widgetNotFound(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_WIDGET_NOT_FOUND, Object.assign({ params: id }, args));
    }
    apiClientNotFound(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_API_CLIENT_NOT_FOUND, Object.assign({ params: id }, args));
    }
    sourceNotFound(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_SOURCE_NOT_FOUND, Object.assign({ params: id }, args));
    }
    noSourcesFound(args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_DATA_SOURCES_NOT_FOUND, args);
    }
    tableNotFound(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_TABLE_NOT_FOUND, Object.assign({ params: id }, args));
    }
    userNotFound(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_USER_NOT_FOUND, Object.assign({ params: id }, args));
    }
    teamNotFound(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_TEAM_NOT_FOUND, Object.assign({ params: id }, args));
    }
    viewNotFound(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_VIEW_NOT_FOUND, Object.assign({ params: id }, args));
    }
    hookNotFound(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_HOOK_NOT_FOUND, Object.assign({ params: id }, args));
    }
    genericNotFound(resource, id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_GENERIC_NOT_FOUND, Object.assign({ params: [resource, id] }, args));
    }
    requiredFieldMissing(field, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_REQUIRED_FIELD_MISSING, Object.assign({ params: field }, args));
    }
    duplicateRecord(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_DUPLICATE_RECORD, Object.assign({ params: id }, args));
    }
    fieldNotFound(param, args) {
        let message = '';
        if (typeof param === 'string') {
            message = `'${param}'`;
        }
        else {
            const onSection = param.onSection ? ` on ${param.onSection}` : '';
            message = `'${param.field}'${onSection}`;
        }
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_FIELD_NOT_FOUND, Object.assign({ params: message }, args));
    }
    extensionNotFound(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_EXTENSION_NOT_FOUND, Object.assign({ params: id }, args));
    }
    invalidOffsetValue(offset, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_INVALID_OFFSET_VALUE, Object.assign({ params: `${offset}` }, args));
    }
    invalidPageValue(page, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_INVALID_PAGE_VALUE, Object.assign({ params: `${page}` }, args));
    }
    invalidPrimaryKey(value, pkColumn, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_INVALID_PK_VALUE, Object.assign({ params: [value, pkColumn] }, args));
    }
    invalidLimitValue(args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_INVALID_LIMIT_VALUE, Object.assign({}, args));
    }
    invalidFilter(filter, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_INVALID_FILTER, Object.assign({ params: filter }, args));
    }
    invalidValueForField(payload, args) {
        const withReason = typeof payload === 'object' && payload.reason
            ? `, reason: ${payload.reason}`
            : ``;
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_INVALID_VALUE_FOR_FIELD, Object.assign({ params: typeof payload === 'string'
                ? payload
                : `Invalid value '${payload.value}' for type '${payload.type}' on column '${payload.column}'${withReason}` }, args));
    }
    unsupportedFilterOperation(operation, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_UNSUPPORTED_FILTER_OPERATION, Object.assign({ params: operation }, args));
    }
    valueLengthExceedLimit(payload, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_INVALID_VALUE_FOR_FIELD, Object.assign({ params: `Value length '${payload.length}' is exceeding allowed limit '${payload.maxLength}' for type '${payload.type}' on column '${payload.column}'` }, args));
    }
    invalidSharedViewPassword(args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_INVALID_SHARED_VIEW_PASSWORD, Object.assign({}, args));
    }
    invalidSharedDashboardPassword(args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_SHARED_DASHBOARD_PASSWORD_INVALID, Object.assign({}, args));
    }
    invalidAttachmentJson(payload, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_INVALID_ATTACHMENT_JSON, Object.assign({ params: payload }, args));
    }
    notImplemented(feature = 'Feature', args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_NOT_IMPLEMENTED, Object.assign({ params: feature }, args));
    }
    internalServerError(message, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_INTERNAL_SERVER, Object.assign({ params: message }, args));
    }
    formulaError(message, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_FORMULA, Object.assign({ params: message }, args));
    }
    formulaCircularRefError(message, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_CIRCULAR_REF_IN_FORMULA, Object.assign({ params: message !== null && message !== void 0 ? message : 'Circular reference detected in formula' }, args));
    }
    unauthorized(message, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_AUTHENTICATION_REQUIRED, Object.assign({ params: message }, args));
    }
    forbidden(message, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_FORBIDDEN, Object.assign({ params: message }, args));
    }
    sourceDataReadOnly(name) {
        this.forbidden(`Source '${name}' is read-only`);
    }
    sourceMetaReadOnly(name) {
        this.forbidden(`Source '${name}' schema is read-only`);
    }
    integrationNotFound(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_INTEGRATION_NOT_FOUND, Object.assign({ params: id }, (args || {})));
    }
    cannotCalculateIntermediateOrderError() {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_CANNOT_CALCULATE_INTERMEDIATE_ORDER, {});
    }
    reorderFailed() {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_REORDER_FAILED, {});
    }
    integrationLinkedWithMultiple(bases, sources, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_INTEGRATION_LINKED_WITH_BASES, Object.assign({ params: bases.map((s) => s.title).join(', '), details: {
                bases: bases.map((b) => {
                    return {
                        id: b.id,
                        title: b.title,
                    };
                }),
                sources: sources.map((s) => {
                    return {
                        id: s.id,
                        base_id: s.base_id,
                        title: s.alias,
                    };
                }),
            } }, (args || {})));
    }
    invalidAttachmentUploadScope(args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_INVALID_ATTACHMENT_UPLOAD_SCOPE, args);
    }
    webhookError(message, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_WEBHOOK_ERROR, Object.assign({ params: message }, args));
    }
    invalidWebhookUrl(url, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_WEBHOOK_URL_INVALID, Object.assign(Object.assign({}, args), { params: `Invalid URl ${url || ''}` }));
    }
    planLimitExceeded(message, details, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_PLAN_LIMIT_EXCEEDED, Object.assign(Object.assign({ params: message }, args), { details: Object.assign(Object.assign({}, details), ((details === null || details === void 0 ? void 0 : details.plan) ? { higherPlan: payment_1.HigherPlan[details.plan] } : {})) }));
    }
    duplicateAlias(param, args) {
        var _a, _b;
        const stackTrace = [
            ...Object.keys((_a = param.additionalTrace) !== null && _a !== void 0 ? _a : {}).map((key) => `${key} '${param.additionalTrace[key]}'`),
            `base '${param.base}'`,
        ].join(', ');
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_DUPLICATE_IN_ALIAS, Object.assign({ params: `Duplicate ${param.type} ${(_b = param.label) !== null && _b !== void 0 ? _b : 'alias'} '${param.alias}' at ${stackTrace}` }, args));
    }
    allowedOnlySSOAccess(ncWorkspaceId) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_SSO_LOGIN_REQUIRED, {
            params: ncWorkspaceId,
        });
    }
    allowedOnlySSOGeneratedToken(ncWorkspaceId) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_SSO_GENERATED_TOKEN_REQUIRED, {
            params: ncWorkspaceId,
        });
    }
    maxPayloadLimitExceeded(limit, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_MAX_PAYLOAD_LIMIT_EXCEEDED, Object.assign({ params: limit.toString() }, args));
    }
    baseUserError(message, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_BASE_COLLABORATION, Object.assign({ params: message }, args));
    }
    orgUserError(message, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_ORG_USER, Object.assign({ params: message }, args));
    }
    tableError(message, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_TABLE_OP_FAILED, Object.assign({ params: message }, args));
    }
    columnError(message, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_COLUMN_OP_FAILED, Object.assign({ params: message }, args));
    }
    baseError(message, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_BASE_OP_FAILED, Object.assign({ params: message }, args));
    }
    maxWorkspaceLimitReached(args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_MAX_WORKSPACE_LIMIT_REACHED, Object.assign({}, args));
    }
    pluginTestError(message, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_TEST_PLUGIN_FAILED, Object.assign({ params: message }, args));
    }
    relationFieldNotFound(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_RELATION_FIELD_NOT_FOUND, Object.assign({ params: id }, args));
    }
    unSupportedRelation(relation, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_UNSUPPORTED_RELATION, Object.assign({ params: `Relation ${relation} is not supported` }, args));
    }
    // for nc-sql-executor, the error returned is possible to be an Error object
    // thus `error.message` is needed to access it
    externalError(error, args) {
        let message = '';
        if (['string'].includes(typeof error)) {
            message = `${error}`;
        }
        else if (typeof error === 'object') {
            if (error.message) {
                message = error.message;
            }
            else {
                // we log the error if we don't know the schema yet
                console.log(`Unknown error schema from nc-sql-executor: ${JSON.stringify(error)}`);
            }
        }
        if (!message || message === '') {
            // generic error message to prevent programmatic error to propagate to UI
            message = 'Error when executing query in external data source';
        }
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_IN_EXTERNAL_DATA_SOURCE, Object.assign({ params: message }, args));
    }
    externalTimeOut(message, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_EXTERNAL_DATA_SOURCE_TIMEOUT, Object.assign({ params: message }, args));
    }
    prohibitedSyncTableOperation(param, args) {
        let message = '';
        switch (param.operation) {
            case 'insert':
            case 'update':
            case 'delete':
                message = `Prohibited data insert / update / delete operation on synced table ${param.modelName}`;
                break;
            case 'create_form_view':
                message = `Form view creation is not supported for synced table ${param.modelName}`;
                break;
        }
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_SYNC_TABLE_OPERATION_PROHIBITED, Object.assign({ params: message }, args));
    }
    featureNotSupported(props, args) {
        var _a, _b;
        if (props.isOnPrem) {
            throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_FEATURE_NOT_SUPPORTED, Object.assign({ params: `Please upgrade your license ${(_a = payment_1.PlanFeatureUpgradeMessages[props.feature]) !== null && _a !== void 0 ? _a : 'to use this feature.'}` }, args));
        }
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_FEATURE_NOT_SUPPORTED, Object.assign({ params: `Upgrade to a higher plan ${(_b = payment_1.PlanFeatureUpgradeMessages[props.feature]) !== null && _b !== void 0 ? _b : 'to use this feature.'}` }, args));
    }
    invalidRequestBody(message) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_INVALID_REQUEST_BODY, {
            params: message,
        });
    }
    unprocessableEntity(message = 'Unprocessable entity') {
        throw new nc_base_error_1.UnprocessableEntity(message);
    }
    testConnectionError(message = 'Unprocessable entity', code) {
        throw new nc_base_error_1.TestConnectionError(message, code);
    }
    notAllowed(message = 'Not allowed') {
        throw new nc_base_error_1.NotAllowed(message);
    }
    emailDomainNotAllowed(domain) {
        throw new nc_base_error_1.SsoError(`Email domain ${domain} is not allowed for this organization`);
    }
    metaError(param) {
        throw new nc_base_error_1.MetaError(param);
    }
    notFound(message = 'Not found') {
        throw new nc_base_error_1.NotFound(message);
    }
    badRequest(message) {
        throw new nc_base_error_1.BadRequestV2(message);
    }
    optionsNotExists(props) {
        throw new nc_base_error_1.OptionsNotExistsError(props);
    }
    outOfSync(message) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_OUT_OF_SYNC, {
            params: message,
        });
    }
    filterVerificationFailed(errors) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_FILTER_VERIFICATION_FAILED, {
            params: errors.join(', '),
        });
    }
    storageFileCreateError(message, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_STORAGE_FILE_CREATE, Object.assign({ params: message }, args));
    }
    storageFileReadError(message, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_STORAGE_FILE_READ, Object.assign({ params: message }, args));
    }
    storageFileDeleteError(message, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_STORAGE_FILE_DELETE, Object.assign({ params: message }, args));
    }
    storageFileStreamError(message, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_STORAGE_FILE_STREAM, Object.assign({ params: message }, args));
    }
    subscriptionAlreadyExists(workspaceOrOrgId, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_SUBSCRIPTION_ALREADY_EXISTS, Object.assign({ params: workspaceOrOrgId }, args));
    }
    subscriptionNotFound(workspaceOrOrgId, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_SUBSCRIPTION_NOT_FOUND, Object.assign({ params: workspaceOrOrgId }, args));
    }
    planNotAvailable(args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_PLAN_NOT_AVAILABLE, args);
    }
    seatCountMismatch(args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_SEAT_COUNT_MISMATCH, Object.assign({ params: 'There was a mismatch in the seat count, please try again' }, args));
    }
    invalidPaymentPayload(args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_INVALID_PAYMENT_PAYLOAD, Object.assign({ params: 'Invalid payment payload' }, args));
    }
    stripeCustomerNotFound(customerId, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_STRIPE_CUSTOMER_NOT_FOUND, Object.assign({ params: customerId }, args));
    }
    stripeSubscriptionNotFound(subscriptionId, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_STRIPE_SUBSCRIPTION_NOT_FOUND, Object.assign({ params: subscriptionId }, args));
    }
    subscriptionOwnershipMismatch(entity, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_SUBSCRIPTION_OWNERSHIP_MISMATCH, Object.assign({ params: `Subscription does not belong to the ${entity}` }, args));
    }
    internalCustomerNotSupported(args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_INTERNAL_CUSTOMER_NOT_SUPPORTED, Object.assign({ params: 'Internal customer not supported' }, args));
    }
    subscriptionCreateFailed(message, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_SUBSCRIPTION_CREATE_FAILED, Object.assign({ params: message }, args));
    }
    stripeWebhookVerificationFailed(args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_STRIPE_WEBHOOK_VERIFICATION_FAILED, Object.assign({ params: 'Webhook signature verification failed' }, args));
    }
    planAlreadyExists(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_PLAN_ALREADY_EXISTS, Object.assign({ params: id }, args));
    }
    workflowEmptyNode(args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_WORKFLOW_EMPTY_NODE, Object.assign({}, args));
    }
    workflowTriggerNodeNotFound(args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_WORKFLOW_TRIGGER_NODE_NOT_FOUND, Object.assign({}, args));
    }
    workflowNodeExecutionFailed(nodeTitle, error, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_WORKFLOW_NODE_EXECUTION_FAILED, Object.assign({ params: `Node "${nodeTitle}" failed: ${error}` }, args));
    }
    workflowWaitingExecutions(count, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_WORKFLOW_WAITING_EXECUTIONS, Object.assign({ params: count.toString() }, args));
    }
    workflowMaxIterationsExceeded(args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_WORKFLOW_MAX_ITERATIONS_EXCEEDED, Object.assign({}, args));
    }
    workflowNodeNotFound(nodeTitle, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_WORKFLOW_NODE_NOT_FOUND, Object.assign({ params: `Node "${nodeTitle}" not found` }, args));
    }
    scriptNotFound(id, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_SCRIPT_NOT_FOUND, Object.assign({ params: id }, args));
    }
    methodNotAllowed(method, args) {
        throw this.errorCodex.generateError(globals_1.NcErrorType.ERR_METHOD_NOT_ALLOWED, Object.assign({ params: `${method} method not allowed` }, args));
    }
}
exports.NcErrorBase = NcErrorBase;
//# sourceMappingURL=data:application/json;base64,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