"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.presetErrorCodexMap = void 0;
const globals_1 = require("../../lib/globals");
const is_1 = require("../../lib/is");
exports.presetErrorCodexMap = {
    [globals_1.NcErrorType.ERR_UNKNOWN]: {
        message: 'Something went wrong',
        code: 500,
    },
    [globals_1.NcErrorType.ERR_INTERNAL_SERVER]: {
        message: (message) => message || `Internal server error`,
        code: 500,
    },
    [globals_1.NcErrorType.ERR_DATABASE_OP_FAILED]: {
        message: (message) => message || `There was an error while running the query`,
        code: 500,
    },
    [globals_1.NcErrorType.ERR_AUTHENTICATION_REQUIRED]: {
        message: (message) => message
            ? `Authentication required - ${message}`
            : 'Authentication required to access this resource',
        code: 401,
    },
    [globals_1.NcErrorType.ERR_FORBIDDEN]: {
        message: (message) => message ? `Forbidden - ${message}` : 'Forbidden to access this resource',
        code: 403,
    },
    [globals_1.NcErrorType.ERR_API_TOKEN_NOT_ALLOWED]: {
        message: 'This request is not allowed with API token',
        code: 401,
    },
    [globals_1.NcErrorType.ERR_WORKSPACE_NOT_FOUND]: {
        message: (id) => `Workspace '${id}' not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_BASE_NOT_FOUND]: {
        message: (id) => `Base '${id}' not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_VIEW_COLUMN_NOT_FOUND]: {
        message: (id) => `View column '${id}' not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_SOURCE_NOT_FOUND]: {
        message: (id) => `Source '${id}' not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_INTEGRATION_NOT_FOUND]: {
        message: (id) => `Connection '${id}' not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_INTEGRATION_LINKED_WITH_BASES]: {
        message: (bases) => `Connection linked with following bases '${bases}'`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_TABLE_NOT_FOUND]: {
        message: (id) => `Table '${id}' not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_VIEW_NOT_FOUND]: {
        message: (id) => `View '${id}' not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_FIELD_NOT_FOUND]: {
        message: (id) => `Field ${id} not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_HOOK_NOT_FOUND]: {
        message: (id) => `Hook '${id}' not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_RECORD_NOT_FOUND]: {
        message: (...ids) => {
            const isMultiple = Array.isArray(ids) && ids.length > 1;
            return `Record${isMultiple ? 's' : ''} '${ids.join(', ')}' not found`;
        },
        code: 404,
    },
    [globals_1.NcErrorType.ERR_GENERIC_NOT_FOUND]: {
        message: (resource, id) => `${resource} '${id}' not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_WIDGET_NOT_FOUND]: {
        message: (id) => `Widget '${id}' not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_DASHBOARD_NOT_FOUND]: {
        message: (id) => `Dashboard '${id}' not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_WORKFLOW_NOT_FOUND]: {
        message: (id) => `Workflow '${id}' not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_SCRIPT_NOT_FOUND]: {
        message: (id) => `Script '${id}' not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_REQUIRED_FIELD_MISSING]: {
        message: (field) => `Field '${field}' is required`,
        code: 422,
    },
    [globals_1.NcErrorType.ERR_DUPLICATE_RECORD]: {
        message: (...ids) => {
            const isMultiple = Array.isArray(ids) && ids.length > 1;
            return `Record${isMultiple ? 's' : ''} '${ids.join(', ')}' already exists`;
        },
        code: 422,
    },
    [globals_1.NcErrorType.ERR_USER_NOT_FOUND]: {
        message: (idOrEmail) => {
            const isEmail = idOrEmail.includes('@');
            return `User ${isEmail ? 'with email' : 'with id'} '${idOrEmail}' not found`;
        },
        code: 404,
    },
    [globals_1.NcErrorType.ERR_TEAM_NOT_FOUND]: {
        message: (id) => `Team '${id}' not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_INVALID_OFFSET_VALUE]: {
        message: (offset) => (0, is_1.ncIsNumber)(Number(offset)) && Number(offset) > 0
            ? `Offset value '${offset}' is invalid`
            : `Offset must be a non-negative integer`,
        code: 422,
    },
    [globals_1.NcErrorType.ERR_INVALID_PAGE_VALUE]: {
        message: (page) => `Page value '${page}' is invalid`,
        code: 422,
    },
    [globals_1.NcErrorType.ERR_INVALID_PK_VALUE]: {
        message: (value, pkColumn) => `Primary key value '${value}' is invalid for column '${pkColumn}'`,
        code: 422,
    },
    [globals_1.NcErrorType.ERR_INVALID_LIMIT_VALUE]: {
        message: (limitMin, limitMax) => `Limit value should be between ${limitMin} and ${limitMax}`,
        code: 422,
    },
    [globals_1.NcErrorType.ERR_INVALID_FILTER]: {
        message: (filter) => `Filter '${filter}' is invalid`,
        code: 422,
    },
    [globals_1.NcErrorType.ERR_UNSUPPORTED_FILTER_OPERATION]: {
        message: (operation) => `Operation '${operation}' is not supported`,
        code: 400,
    },
    [globals_1.NcErrorType.ERR_INVALID_SHARED_VIEW_PASSWORD]: {
        message: 'Invalid shared view password',
        code: 403,
    },
    [globals_1.NcErrorType.ERR_SHARED_DASHBOARD_PASSWORD_INVALID]: {
        message: 'Invalid shared dashboard password',
        code: 403,
    },
    [globals_1.NcErrorType.ERR_INVALID_ATTACHMENT_JSON]: {
        message: (payload) => `Invalid JSON for attachment field: ${payload}`,
        code: 400,
    },
    [globals_1.NcErrorType.ERR_NOT_IMPLEMENTED]: {
        message: (feature) => `${feature} is not implemented`,
        code: 501,
    },
    [globals_1.NcErrorType.ERR_INVALID_JSON]: {
        message: 'Invalid JSON in request body',
        code: 400,
    },
    [globals_1.NcErrorType.ERR_COLUMN_ASSOCIATED_WITH_LINK]: {
        message: 'Column is associated with a link, please remove the link first',
        code: 400,
    },
    [globals_1.NcErrorType.ERR_TABLE_ASSOCIATED_WITH_LINK]: {
        message: 'Table is associated with a link, please remove the link first',
        code: 400,
    },
    [globals_1.NcErrorType.ERR_FORMULA]: {
        message: (message) => {
            return message;
        },
        code: 400,
    },
    [globals_1.NcErrorType.ERR_CIRCULAR_REF_IN_FORMULA]: {
        message: (message) => {
            return message;
        },
        code: 400,
    },
    [globals_1.NcErrorType.ERR_PERMISSION_DENIED]: {
        message: 'Permission denied',
        code: 403,
    },
    [globals_1.NcErrorType.ERR_INVALID_ATTACHMENT_UPLOAD_SCOPE]: {
        message: 'Invalid attachment upload scope',
        code: 400,
    },
    [globals_1.NcErrorType.ERR_REORDER_FAILED]: {
        message: 'Reorder failed',
        code: 400,
    },
    [globals_1.NcErrorType.ERR_CANNOT_CALCULATE_INTERMEDIATE_ORDER]: {
        message: 'Cannot calculate intermediate order',
        code: 400,
    },
    [globals_1.NcErrorType.ERR_PLAN_LIMIT_EXCEEDED]: {
        message: (message) => message || 'Plan limit exceeded',
        code: 403,
    },
    [globals_1.NcErrorType.ERR_SSO_LOGIN_REQUIRED]: {
        message: (_workspaceId) => 'SSO login required for workspace',
        code: 403,
    },
    [globals_1.NcErrorType.ERR_SSO_GENERATED_TOKEN_REQUIRED]: {
        message: (_workspaceId) => 'This workspace requires SSO-authenticated tokens. Please generate a new token after signing in with SSO',
        code: 403,
    },
    [globals_1.NcErrorType.ERR_MAX_PAYLOAD_LIMIT_EXCEEDED]: {
        message: (limit) => `Maximum ${limit} entities are allowed per request`,
        code: 422,
    },
    [globals_1.NcErrorType.ERR_MAX_WORKSPACE_LIMIT_REACHED]: {
        message: () => `The maximum workspace limit has been reached. Please contact your administrator to request access to a workspace.`,
        code: 403,
    },
    [globals_1.NcErrorType.ERR_INVALID_VALUE_FOR_FIELD]: {
        message: (message) => message,
        code: 422,
    },
    [globals_1.NcErrorType.ERR_INVALID_REQUEST_BODY]: {
        message: (message) => message,
        code: 400,
    },
    [globals_1.NcErrorType.ERR_BASE_COLLABORATION]: {
        message: (message) => message || 'Something went wrong',
        code: 422,
    },
    [globals_1.NcErrorType.ERR_ORG_USER]: {
        message: (message) => message || 'Something went wrong',
        code: 422,
    },
    [globals_1.NcErrorType.ERR_TABLE_OP_FAILED]: {
        message: (message) => message || 'Something went wrong',
        code: 422,
    },
    [globals_1.NcErrorType.ERR_COLUMN_OP_FAILED]: {
        message: (message) => message || 'Something went wrong',
        code: 422,
    },
    [globals_1.NcErrorType.ERR_SYNC_TABLE_OPERATION_PROHIBITED]: {
        message: (message) => message ||
            `Prohibited data insert / update / delete operation on synced table`,
        code: 422,
    },
    [globals_1.NcErrorType.ERR_FEATURE_NOT_SUPPORTED]: {
        message: (message) => message || `Upgrade to a higher plan to use this feature.`,
        code: 403,
    },
    [globals_1.NcErrorType.ERR_DUPLICATE_IN_ALIAS]: {
        message: (message) => message,
        code: 422,
    },
    [globals_1.NcErrorType.ERR_OUT_OF_SYNC]: {
        message: (message) => message || `Please refresh the page and try again.`,
        code: 409,
    },
    [globals_1.NcErrorType.ERR_FILTER_VERIFICATION_FAILED]: {
        message: (message) => `Filter verification failed: ${message}`,
        code: 422,
    },
    [globals_1.NcErrorType.ERR_WEBHOOK_ERROR]: {
        message: (message) => message,
        code: 400,
    },
    [globals_1.NcErrorType.ERR_DATA_SOURCES_NOT_FOUND]: {
        message: (message) => message,
        code: 400,
    },
    [globals_1.NcErrorType.ERR_TEST_PLUGIN_FAILED]: {
        message: (message) => message,
        code: 400,
    },
    [globals_1.NcErrorType.ERR_UNSUPPORTED_RELATION]: {
        message: (message) => message,
        code: 400,
    },
    [globals_1.NcErrorType.ERR_IN_EXTERNAL_DATA_SOURCE]: {
        message: (message) => message ||
            'Error running query on external source. Confirm if source is accessible.',
        code: 400,
    },
    [globals_1.NcErrorType.ERR_EXTERNAL_DATA_SOURCE_TIMEOUT]: {
        message: (message) => message ||
            'External source taking long to respond. Reconsider sorts/filters for this view and confirm if source is accessible.',
        code: 504,
    },
    [globals_1.NcErrorType.ERR_RELATION_FIELD_NOT_FOUND]: {
        message: (id) => `Relation Field '${id}' not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_EXTENSION_NOT_FOUND]: {
        message: (id) => `Extension '${id}' not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_STORAGE_FILE_CREATE]: {
        message: (details) => `Failed to create file: ${details}`,
        code: 500,
    },
    [globals_1.NcErrorType.ERR_STORAGE_FILE_READ]: {
        message: (details) => `Failed to read file: ${details}`,
        code: 500,
    },
    [globals_1.NcErrorType.ERR_STORAGE_FILE_DELETE]: {
        message: (details) => `Failed to delete file: ${details}`,
        code: 500,
    },
    [globals_1.NcErrorType.ERR_STORAGE_FILE_STREAM]: {
        message: (details) => `Failed to stream file: ${details}`,
        code: 500,
    },
    [globals_1.NcErrorType.ERR_SUBSCRIPTION_ALREADY_EXISTS]: {
        message: (id) => `Subscription already exists for ${id}`,
        code: 409,
    },
    [globals_1.NcErrorType.ERR_SUBSCRIPTION_NOT_FOUND]: {
        message: (id) => `Subscription not found for ${id}`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_PLAN_NOT_AVAILABLE]: {
        message: () => 'This plan is not available',
        code: 400,
    },
    [globals_1.NcErrorType.ERR_SEAT_COUNT_MISMATCH]: {
        message: (msg) => msg,
        code: 400,
    },
    [globals_1.NcErrorType.ERR_INVALID_PAYMENT_PAYLOAD]: {
        message: (msg) => msg,
        code: 400,
    },
    [globals_1.NcErrorType.ERR_STRIPE_CUSTOMER_NOT_FOUND]: {
        message: (id) => `Stripe customer '${id}' not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_STRIPE_SUBSCRIPTION_NOT_FOUND]: {
        message: (id) => `Stripe subscription '${id}' not found`,
        code: 404,
    },
    [globals_1.NcErrorType.ERR_SUBSCRIPTION_OWNERSHIP_MISMATCH]: {
        message: (msg) => msg,
        code: 403,
    },
    [globals_1.NcErrorType.ERR_INTERNAL_CUSTOMER_NOT_SUPPORTED]: {
        message: (msg) => msg,
        code: 501,
    },
    [globals_1.NcErrorType.ERR_SUBSCRIPTION_CREATE_FAILED]: {
        message: (msg) => msg,
        code: 500,
    },
    [globals_1.NcErrorType.ERR_STRIPE_WEBHOOK_VERIFICATION_FAILED]: {
        message: (msg) => msg,
        code: 400,
    },
    [globals_1.NcErrorType.ERR_PLAN_ALREADY_EXISTS]: {
        message: (id) => `Plan already exists with id ${id}`,
        code: 409,
    },
    [globals_1.NcErrorType.ERR_WORKFLOW_WAITING_EXECUTIONS]: {
        message: (count) => `This workflow has ${count} pending execution(s) waiting to resume`,
        code: 400,
    },
    [globals_1.NcErrorType.ERR_WORKFLOW_NODE_NOT_FOUND]: {
        message: (message) => message,
        code: 400,
    },
    [globals_1.NcErrorType.ERR_METHOD_NOT_ALLOWED]: {
        message: (method) => `Method ${method} not allowed`,
        code: 405,
    },
};
//# sourceMappingURL=data:application/json;base64,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