"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const validate_row_filters_1 = require("./validate-row-filters");
const UITypes_1 = __importDefault(require("../../lib/UITypes"));
const dayjs_1 = __importDefault(require("dayjs"));
const utc_js_1 = __importDefault(require("dayjs/plugin/utc.js"));
const timezone_js_1 = __importDefault(require("dayjs/plugin/timezone.js"));
const customParseFormat_js_1 = __importDefault(require("dayjs/plugin/customParseFormat.js"));
const isBetween_1 = __importDefault(require("dayjs/plugin/isBetween"));
const isSameOrAfter_1 = __importDefault(require("dayjs/plugin/isSameOrAfter"));
const isSameOrBefore_1 = __importDefault(require("dayjs/plugin/isSameOrBefore"));
const lib_1 = require("../../lib");
dayjs_1.default.extend(utc_js_1.default);
dayjs_1.default.extend(timezone_js_1.default);
dayjs_1.default.extend(customParseFormat_js_1.default);
dayjs_1.default.extend(isSameOrBefore_1.default);
dayjs_1.default.extend(isSameOrAfter_1.default);
dayjs_1.default.extend(isBetween_1.default);
const mockColumns = [
    { id: '1', title: 'Name', uidt: UITypes_1.default.SingleLineText },
    { id: '2', title: 'Age', uidt: UITypes_1.default.Number },
    { id: '3', title: 'IsActive', uidt: UITypes_1.default.Checkbox },
    {
        id: '4',
        title: 'CreatedAt',
        uidt: UITypes_1.default.DateTime,
    },
    { id: '5', title: 'CreatedBy', uidt: UITypes_1.default.User },
    {
        id: '6',
        title: 'RelatedRecords',
        uidt: UITypes_1.default.LinkToAnotherRecord,
        colOptions: {
            fk_related_model_id: 'relatedModel',
        },
    },
    { id: '7', title: 'JsonData', uidt: UITypes_1.default.JSON },
    { id: '8', title: 'TimeData', uidt: UITypes_1.default.Time },
];
const mockMetas = {
    relatedModel: {
        columns: [
            { id: 'r1', title: 'Primary', pv: true, uidt: UITypes_1.default.SingleLineText },
        ],
    },
};
const mockClient = 'pg'; // or 'mysql2'
describe('validateRowFilters', () => {
    it('should return true if no filters are provided', () => {
        const filters = [];
        const data = { Name: 'Test' };
        const result = (0, validate_row_filters_1.validateRowFilters)({
            filters,
            data,
            columns: mockColumns,
            client: mockClient,
            metas: mockMetas,
        });
        expect(result).toBe(true);
    });
    // Test cases for basic comparisons (eq, neq, like, nlike)
    // FIXME: not reviewed
    describe.skip('Basic comparisons', () => {
        it('should correctly evaluate "eq" for string', () => {
            const filters = [
                { fk_column_id: '1', comparison_op: 'eq', value: 'Alice' },
            ];
            const data = { Name: 'Alice' };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Name: 'Bob' };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "neq" for string', () => {
            const filters = [
                { fk_column_id: '1', comparison_op: 'neq', value: 'Alice' },
            ];
            const data = { Name: 'Bob' };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Name: 'Alice' };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "like" for string', () => {
            const filters = [
                { fk_column_id: '1', comparison_op: 'like', value: 'ali' },
            ];
            const data = { Name: 'Alice' };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Name: 'Bob' };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "nlike" for string', () => {
            const filters = [
                { fk_column_id: '1', comparison_op: 'nlike', value: 'ali' },
            ];
            const data = { Name: 'Bob' };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Name: 'Alice' };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "eq" for number', () => {
            const filters = [
                { fk_column_id: '2', comparison_op: 'eq', value: 30 },
            ];
            const data = { Age: 30 };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Age: 25 };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "gt" for number', () => {
            const filters = [
                { fk_column_id: '2', comparison_op: 'gt', value: 25 },
            ];
            const data = { Age: 30 };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Age: 20 };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "lte" for number', () => {
            const filters = [
                { fk_column_id: '2', comparison_op: 'lte', value: 30 },
            ];
            const data = { Age: 30 };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Age: 35 };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "checked" for checkbox', () => {
            const filters = [
                { fk_column_id: '3', comparison_op: 'checked' },
            ];
            const data = { IsActive: true };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { IsActive: false };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "notchecked" for checkbox', () => {
            const filters = [
                { fk_column_id: '3', comparison_op: 'notchecked' },
            ];
            const data = { IsActive: false };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { IsActive: true };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
    });
    // Test cases for empty/blank/null checks
    // FIXME: not reviewed
    describe.skip('Empty/Blank/Null checks', () => {
        it('should correctly evaluate "empty" for string', () => {
            const filters = [
                { fk_column_id: '1', comparison_op: 'empty' },
            ];
            const data = { Name: '' };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Name: null };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data3 = { Name: undefined };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data3,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data4 = { Name: 'Test' };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data4,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "notempty" for string', () => {
            const filters = [
                { fk_column_id: '1', comparison_op: 'notempty' },
            ];
            const data = { Name: 'Test' };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Name: '' };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "null" for string', () => {
            const filters = [
                { fk_column_id: '1', comparison_op: 'null' },
            ];
            const data = { Name: null };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Name: 'Test' };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "notnull" for string', () => {
            const filters = [
                { fk_column_id: '1', comparison_op: 'notnull' },
            ];
            const data = { Name: 'Test' };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Name: null };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
    });
    // Test cases for Date/DateTime filters
    describe('Date/DateTime filters', () => {
        const today = (0, dayjs_1.default)()
            .tz('Etc/UTC')
            .startOf('day')
            .format('YYYY-MM-DD HH:mm:ssZ');
        const todayEnd = (0, dayjs_1.default)()
            .tz('Etc/UTC')
            .endOf('day')
            .format('YYYY-MM-DD HH:mm:ssZ');
        const yesterday = (0, dayjs_1.default)()
            .tz('Etc/UTC')
            .subtract(1, 'day')
            .format('YYYY-MM-DD HH:mm:ssZ');
        const tzAsiaKolkata = 'Asia/Kolkata';
        const todayKolkata = (0, dayjs_1.default)()
            .tz(tzAsiaKolkata)
            .startOf('day')
            .format('YYYY-MM-DD HH:mm:ssZ');
        const todayKolkataEnd = (0, dayjs_1.default)()
            .tz(tzAsiaKolkata)
            .endOf('day')
            .format('YYYY-MM-DD HH:mm:ssZ');
        it('should correctly evaluate "eq" with "today" sub-op', () => {
            const filters = [
                { fk_column_id: '4', comparison_op: 'eq', comparison_sub_op: 'today' },
            ];
            const data = { CreatedAt: today };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(true);
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: { CreatedAt: todayEnd },
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(true);
            const data2 = { CreatedAt: yesterday };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(false);
        });
        it('should correctly evaluate "eq" with "today" sub-op for other timezone', () => {
            const filters = [
                { fk_column_id: '4', comparison_op: 'eq', comparison_sub_op: 'today' },
            ];
            const data = { CreatedAt: todayKolkata };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: tzAsiaKolkata },
            })).toBe(true);
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: { CreatedAt: todayKolkataEnd },
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: tzAsiaKolkata },
            })).toBe(true);
            const data2 = { CreatedAt: yesterday };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(false);
        });
        it('should correctly evaluate "eq" with "yesterday" sub-op', () => {
            const filters = [
                {
                    fk_column_id: '4',
                    comparison_op: 'eq',
                    comparison_sub_op: 'yesterday',
                },
            ];
            const data = { CreatedAt: yesterday };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(true);
            const data2 = { CreatedAt: today };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(false);
        });
        it('should correctly evaluate "gt" with "daysAgo" sub-op', () => {
            const filters = [
                {
                    fk_column_id: '4',
                    comparison_op: 'gt',
                    comparison_sub_op: 'daysAgo',
                    value: 2,
                },
            ];
            const data = { CreatedAt: yesterday }; // 1 day ago
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(true);
            const data2 = {
                CreatedAt: (0, dayjs_1.default)()
                    .tz('Etc/UTC')
                    .subtract(3, 'day')
                    .format('YYYY-MM-DD HH:mm:ss'),
            }; // 3 days ago
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(false);
        });
        it('should correctly evaluate "isWithin" with "pastWeek" sub-op', () => {
            const filters = [
                {
                    fk_column_id: '4',
                    comparison_op: 'isWithin',
                    comparison_sub_op: 'pastWeek',
                },
            ];
            const data = {
                CreatedAt: (0, dayjs_1.default)()
                    .tz('Etc/UTC')
                    .subtract(3, 'day')
                    .format('YYYY-MM-DD HH:mm:ss'),
            }; // 3 days ago
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(true);
            const data2 = {
                CreatedAt: (0, dayjs_1.default)()
                    .tz('Etc/UTC')
                    .subtract(8, 'day')
                    .format('YYYY-MM-DD HH:mm:ss'),
            }; // 8 days ago
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(false);
        });
        it('should correctly evaluate "isWithin" with "nextNumberOfDays" sub-op', () => {
            const filters = [
                {
                    fk_column_id: '4',
                    comparison_op: 'isWithin',
                    comparison_sub_op: 'nextNumberOfDays',
                    value: 5,
                },
            ];
            const data = {
                CreatedAt: (0, dayjs_1.default)()
                    .tz('Etc/UTC')
                    .add(3, 'day')
                    .format('YYYY-MM-DD HH:mm:ss'),
            }; // 3 days from now
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(true);
            const data2 = {
                CreatedAt: (0, dayjs_1.default)()
                    .tz('Etc/UTC')
                    .add(6, 'day')
                    .format('YYYY-MM-DD HH:mm:ss'),
            }; // 6 days from now
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(false);
        });
    });
    // Test cases for User/CreatedBy/LastModifiedBy filters
    // FIXME: not reviewed
    describe.skip('User/CreatedBy/LastModifiedBy filters', () => {
        const currentUser = { id: 'user1', email: 'user1@example.com' };
        it('should correctly evaluate "anyof" for single user', () => {
            const filters = [
                { fk_column_id: '5', comparison_op: 'anyof', value: 'user1' },
            ];
            const data = { CreatedBy: { id: 'user1', email: 'user1@example.com' } };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(true);
            const data2 = { CreatedBy: { id: 'user2', email: 'user2@example.com' } };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(false);
        });
        it('should correctly evaluate "anyof" for multiple users', () => {
            const filters = [
                { fk_column_id: '5', comparison_op: 'anyof', value: 'user1, user3' },
            ];
            const data = { CreatedBy: [{ id: 'user1' }, { id: 'user2' }] };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(true);
            const data2 = { CreatedBy: [{ id: 'user4' }] };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(false);
        });
        it('should correctly evaluate "anyof" with CURRENT_USER_TOKEN', () => {
            const filters = [
                {
                    fk_column_id: '5',
                    comparison_op: 'anyof',
                    value: lib_1.CURRENT_USER_TOKEN,
                },
            ];
            const data = { CreatedBy: { id: 'user1', email: 'user1@example.com' } };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(true);
            const data2 = { CreatedBy: { id: 'user2', email: 'user2@example.com' } };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(false);
        });
        it('should correctly evaluate "allof" for multiple users', () => {
            const filters = [
                { fk_column_id: '5', comparison_op: 'allof', value: 'user1, user2' },
            ];
            const data = {
                CreatedBy: [{ id: 'user1' }, { id: 'user2' }, { id: 'user3' }],
            };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(true);
            const data2 = { CreatedBy: [{ id: 'user1' }, { id: 'user3' }] };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(false);
        });
        it('should correctly evaluate "empty" for user field', () => {
            const filters = [
                { fk_column_id: '5', comparison_op: 'empty' },
            ];
            const data = { CreatedBy: [] };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(true);
            const data2 = { CreatedBy: null };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(true);
            const data3 = { CreatedBy: { id: 'user1' } };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data3,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(false);
        });
    });
    // Test cases for LinkToAnotherRecord filters
    // FIXME: not reviewed
    describe.skip('LinkToAnotherRecord filters', () => {
        it('should correctly evaluate "eq" for linked record primary value', () => {
            const filters = [
                { fk_column_id: '6', comparison_op: 'eq', value: 'RecordA' },
            ];
            const data = { RelatedRecords: [{ Primary: 'RecordA' }] };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { RelatedRecords: [{ Primary: 'RecordB' }] };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "like" for linked record primary value', () => {
            const filters = [
                { fk_column_id: '6', comparison_op: 'like', value: 'record' },
            ];
            const data = { RelatedRecords: [{ Primary: 'RecordA' }] };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { RelatedRecords: [{ Primary: 'SomethingElse' }] };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "anyof" for linked record primary value', () => {
            const filters = [
                {
                    fk_column_id: '6',
                    comparison_op: 'anyof',
                    value: 'RecordA, RecordC',
                },
            ];
            const data = {
                RelatedRecords: [{ Primary: 'RecordA' }, { Primary: 'RecordB' }],
            };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { RelatedRecords: [{ Primary: 'RecordB' }] };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "empty" for linked record', () => {
            const filters = [
                { fk_column_id: '6', comparison_op: 'empty' },
            ];
            const data = { RelatedRecords: [] };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { RelatedRecords: [{ Primary: 'RecordA' }] };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
    });
    // Edge cases
    // FIXME: not reviewed
    describe.skip('Edge cases', () => {
        it('should return null if a date sub-op value is missing', () => {
            const filters = [
                {
                    fk_column_id: '4',
                    comparison_op: 'gt',
                    comparison_sub_op: 'daysAgo',
                    value: null,
                },
            ];
            const data = {
                CreatedAt: (0, dayjs_1.default)()
                    .tz('Etc/UTC')
                    .subtract(1, 'day')
                    .format('YYYY-MM-DD HH:mm:ss'),
            };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(null);
        });
        it('should handle missing column gracefully', () => {
            const filters = [
                { fk_column_id: '99', comparison_op: 'eq', value: 'test' },
            ]; // Non-existent column
            const data = { Name: 'Test' };
            expect((0, validate_row_filters_1.validateRowFilters)({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(null);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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