"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IS_WITHIN_COMPARISON_SUB_OPS = exports.GROUPBY_COMPARISON_OPS = exports.COMPARISON_SUB_OPS = exports.COMPARISON_OPS = void 0;
exports.extractFilterFromXwhere = extractFilterFromXwhere;
exports.validateFilterComparison = validateFilterComparison;
exports.extractCondition = extractCondition;
const errorUtils_1 = require("../lib/errorUtils");
const index_1 = require("../lib/index");
var query_filter_lexer_1 = require("../lib/parser/queryFilter/query-filter-lexer");
Object.defineProperty(exports, "COMPARISON_OPS", { enumerable: true, get: function () { return query_filter_lexer_1.COMPARISON_OPS; } });
Object.defineProperty(exports, "COMPARISON_SUB_OPS", { enumerable: true, get: function () { return query_filter_lexer_1.COMPARISON_SUB_OPS; } });
Object.defineProperty(exports, "GROUPBY_COMPARISON_OPS", { enumerable: true, get: function () { return query_filter_lexer_1.GROUPBY_COMPARISON_OPS; } });
Object.defineProperty(exports, "IS_WITHIN_COMPARISON_SUB_OPS", { enumerable: true, get: function () { return query_filter_lexer_1.IS_WITHIN_COMPARISON_SUB_OPS; } });
function extractFilterFromXwhere(str, aliasColObjMap, throwErrorIfInvalid = false, errors = []) {
    if (!str) {
        return { filters: [] };
    }
    // if array treat it as `and` group
    if (Array.isArray(str)) {
        const nestedFilters = str.map((s) => extractFilterFromXwhere(s, aliasColObjMap, throwErrorIfInvalid, errors));
        const filters = nestedFilters.reduce((acc, { filters }) => {
            if (!filters)
                return acc;
            return [...acc, ...filters];
        }, []);
        const collectedErrors = nestedFilters.reduce((acc, { errors }) => {
            if (!errors)
                return acc;
            return [...acc, ...errors];
        }, []);
        // If errors exist, return them
        if (collectedErrors.length > 0) {
            return { errors: collectedErrors };
        }
        // If there's only one filter, return it directly
        if (filters.length === 1) {
            return { filters };
        }
        // If there's only one filter, return it directly; otherwise, wrap in an AND group
        return {
            filters: [
                {
                    is_group: true,
                    logical_op: 'and',
                    children: filters,
                },
            ],
        };
    }
    // Validate input type
    if (typeof str !== 'string') {
        const error = {
            message: 'Invalid filter format. Expected string or array of strings.',
        };
        if (throwErrorIfInvalid)
            throw new Error(error.message);
        errors.push(error);
        return { errors };
    }
    let openIndex = str.indexOf('((');
    if (openIndex === -1)
        openIndex = str.indexOf('(~');
    // If it's a simple query, extract conditions directly
    if (openIndex === -1) {
        if (str !== '~not') {
            const nestedArrayConditions = str.split(/(?=~(?:or(?:not)?|and(?:not)?|not)\()/);
            return extractCondition(nestedArrayConditions, aliasColObjMap, throwErrorIfInvalid, errors);
        }
        return { filters: [] };
    }
    let closingIndex = str.indexOf('))');
    let nextOpenIndex = openIndex;
    // Iterate until the correct closing bracket is found
    while ((nextOpenIndex = str
        .substring(0, closingIndex)
        .indexOf('((', nextOpenIndex + 1)) !== -1) {
        closingIndex = str.indexOf('))', closingIndex + 1);
    }
    // If no closing bracket is found, return an error
    if (closingIndex === -1) {
        const error = {
            message: `${str
                .substring(0, openIndex + 1)
                .slice(-10)} : Closing bracket not found`,
        };
        if (throwErrorIfInvalid)
            throw new Error(error.message);
        errors.push(error);
        return { errors };
    }
    // Extract operator and left-hand side of nested query
    const operandStartIndex = str.lastIndexOf('~', openIndex);
    const operator = operandStartIndex !== -1
        ? str.substring(operandStartIndex + 1, openIndex)
        : '';
    const lhsOfNestedQuery = str.substring(0, openIndex);
    // Recursively process left-hand side, nested query, and right-hand side
    const lhsResult = extractFilterFromXwhere(lhsOfNestedQuery, aliasColObjMap, throwErrorIfInvalid, errors);
    const nestedQueryResult = extractFilterFromXwhere(str.substring(openIndex + 1, closingIndex + 1), aliasColObjMap, throwErrorIfInvalid, errors);
    const rhsResult = extractFilterFromXwhere(str.substring(closingIndex + 2), aliasColObjMap, throwErrorIfInvalid, errors);
    // If any errors occurred during recursion, return them
    if (lhsResult.errors || nestedQueryResult.errors || rhsResult.errors) {
        return {
            errors: [
                ...(lhsResult.errors || []),
                ...(nestedQueryResult.errors || []),
                ...(rhsResult.errors || []),
            ],
        };
    }
    // Return the combined filters
    return {
        filters: [
            ...(lhsResult.filters || []),
            {
                is_group: true,
                logical_op: operator,
                children: nestedQueryResult.filters || [],
            },
            ...(rhsResult.filters || []),
        ],
    };
}
/**
 * Validates a filter comparison operation and its sub-operation.
 *
 * @param {UITypes} uidt - The UI type to validate against.
 * @param {any} op - The main comparison operator.
 * @param {any} [sub_op] - The optional sub-operation.
 * @param {FilterParseError[]} [errors=[]] - An optional array to collect errors.
 * @returns {FilterParseError[]} - An array of validation errors, empty if no errors.
 *
 * This function checks if the given `op` is a valid comparison operator and, if a `sub_op` is provided,
 * ensures it is compatible with the given `uidt`. If any validation fails, errors are added to the array
 * and returned instead of throwing an exception.
 */
function validateFilterComparison(uidt, op, sub_op, errors = [], validateFilterComparison = false) {
    // Check if the main comparison operator is valid
    if (!index_1.COMPARISON_OPS.includes(op) && !index_1.GROUPBY_COMPARISON_OPS.includes(op)) {
        if (validateFilterComparison) {
            throw new errorUtils_1.BadRequest(`${op} is not supported.`);
        }
        errors.push({ message: `${op} is not supported.` });
    }
    if (sub_op) {
        // Ensure that sub-operators are only used with specific UI types
        if (![
            index_1.UITypes.Date,
            index_1.UITypes.DateTime,
            index_1.UITypes.CreatedTime,
            index_1.UITypes.LastModifiedTime,
        ].includes(uidt)) {
            if (validateFilterComparison) {
                throw new errorUtils_1.BadRequest(`'${sub_op}' is not supported for UI Type'${uidt}'.`);
            }
            errors.push({
                message: `'${sub_op}' is not supported for UI Type '${uidt}'.`,
            });
        }
        // Validate if the sub-operator exists in the allowed set
        if (!index_1.COMPARISON_SUB_OPS.includes(sub_op)) {
            if (validateFilterComparison) {
                throw new errorUtils_1.BadRequest(`'${sub_op}' is not supported.`);
            }
            errors.push({ message: `'${sub_op}' is not supported.` });
        }
        // Ensure `isWithin` has correct sub-operators, and other operators don't use `isWithin` sub-operators
        if ((op === 'isWithin' && !index_1.IS_WITHIN_COMPARISON_SUB_OPS.includes(sub_op)) ||
            (op !== 'isWithin' && index_1.IS_WITHIN_COMPARISON_SUB_OPS.includes(sub_op))) {
            if (validateFilterComparison) {
                throw new errorUtils_1.BadRequest(`'${sub_op}' is not supported for '${op}'`);
            }
            errors.push({ message: `'${sub_op}' is not supported for '${op}'.` });
        }
    }
    // Return collected errors, if any
    return errors.length > 0 ? errors : [];
}
function extractCondition(nestedArrayConditions, aliasColObjMap, throwErrorIfInvalid, errors) {
    if (!nestedArrayConditions || nestedArrayConditions.length === 0) {
        return { filters: [] };
    }
    const parsedFilters = nestedArrayConditions
        .map((str) => {
        var _a, _b, _c;
        let logicOp;
        let alias;
        let op;
        let value;
        [logicOp, alias, op, value] =
            ((_a = str.match(/(?:~(and|or|not))?\((.*?),(\w+),(.*)\)/)) === null || _a === void 0 ? void 0 : _a.slice(1)) || [];
        if (!alias && !op && !value) {
            // Attempt to match blank filter format
            [logicOp, alias, op, value] =
                ((_b = str.match(/(?:~(and|or|not))?\((.*?),(\w+)\)/)) === null || _b === void 0 ? void 0 : _b.slice(1)) || [];
        }
        // Normalize filter operations
        switch (op) {
            case 'is':
                if (value === 'blank') {
                    op = 'blank';
                    value = undefined;
                }
                else if (value === 'notblank') {
                    op = 'notblank';
                    value = undefined;
                }
                break;
            case 'isblank':
            case 'is_blank':
                op = 'blank';
                break;
            case 'isnotblank':
            case 'is_not_blank':
            case 'is_notblank':
                op = 'notblank';
                break;
        }
        let sub_op = null;
        if (aliasColObjMap[alias]) {
            const columnType = aliasColObjMap[alias].uidt;
            // Handle date and datetime values
            if ([
                index_1.UITypes.Date,
                index_1.UITypes.DateTime,
                index_1.UITypes.LastModifiedTime,
                index_1.UITypes.CreatedTime,
            ].includes(columnType)) {
                value = value === null || value === void 0 ? void 0 : value.split(',');
                sub_op = value === null || value === void 0 ? void 0 : value.shift();
                value = value === null || value === void 0 ? void 0 : value[0];
                if (sub_op === 'null') {
                    sub_op = undefined;
                    value = null;
                }
            }
            else if (op === 'in') {
                value = value.split(',');
            }
            validateFilterComparison(columnType, op, sub_op, errors, throwErrorIfInvalid);
        }
        else {
            const error = {
                message: alias
                    ? `Column alias '${alias}' not found.`
                    : 'Invalid filter format.',
            };
            if (throwErrorIfInvalid)
                throw new errorUtils_1.NcSDKError(error.message);
            errors.push(error);
            return null;
        }
        let columnId = (_c = aliasColObjMap[alias]) === null || _c === void 0 ? void 0 : _c.id;
        // If alias is not found, check if it matches a column ID directly
        if (!columnId &&
            Object.values(aliasColObjMap).some((col) => (col === null || col === void 0 ? void 0 : col.id) === alias)) {
            columnId = alias;
        }
        return Object.assign(Object.assign({ comparison_op: op }, (sub_op && {
            comparison_sub_op: sub_op,
        })), { fk_column_id: columnId, logical_op: logicOp, value });
    })
        .filter(Boolean);
    if (errors.length > 0) {
        return { errors };
    }
    return { filters: parsedFilters };
}
//# sourceMappingURL=data:application/json;base64,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