"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.IS_WITHIN_COMPARISON_SUB_OPS = exports.GROUPBY_COMPARISON_OPS = exports.COMPARISON_SUB_OPS = exports.COMPARISON_OPS = void 0;
exports.extractFilterFromXwhere = extractFilterFromXwhere;
const query_filter_parser_1 = require("../lib/parser/queryFilter/query-filter-parser");
const UITypes_1 = __importDefault(require("./UITypes"));
const invalid_filter_error_1 = require("./error/invalid-filter.error");
const filterHelpers_1 = require("./filterHelpers");
const arrayHelpers_1 = require("./arrayHelpers");
const error_message_parser_1 = require("./parser/queryFilter/error-message-parser");
const timezoneUtils_1 = require("./timezoneUtils");
const helperFunctions_1 = require("./helperFunctions");
var query_filter_lexer_1 = require("../lib/parser/queryFilter/query-filter-lexer");
Object.defineProperty(exports, "COMPARISON_OPS", { enumerable: true, get: function () { return query_filter_lexer_1.COMPARISON_OPS; } });
Object.defineProperty(exports, "COMPARISON_SUB_OPS", { enumerable: true, get: function () { return query_filter_lexer_1.COMPARISON_SUB_OPS; } });
Object.defineProperty(exports, "GROUPBY_COMPARISON_OPS", { enumerable: true, get: function () { return query_filter_lexer_1.GROUPBY_COMPARISON_OPS; } });
Object.defineProperty(exports, "IS_WITHIN_COMPARISON_SUB_OPS", { enumerable: true, get: function () { return query_filter_lexer_1.IS_WITHIN_COMPARISON_SUB_OPS; } });
function extractFilterFromXwhere(context, { str, aliasColObjMap, throwErrorIfInvalid = false, errors = [], }) {
    if (!str) {
        return { filters: [] };
    }
    for (const columnName of Object.keys(aliasColObjMap)) {
        const column = aliasColObjMap[columnName];
        aliasColObjMap[column.id] = column;
    }
    return innerExtractFilterFromXwhere(context, {
        str,
        aliasColObjMap,
        throwErrorIfInvalid,
        errors,
    });
}
function innerExtractFilterFromXwhere(context, { str, aliasColObjMap, throwErrorIfInvalid = false, errors = [], }) {
    if (!str) {
        return { filters: [] };
    } // if array treat it as `and` group
    else if (Array.isArray(str)) {
        // calling recursively for nested query
        const nestedFilters = [].concat(...str.map((s) => extractFilterFromXwhere(context, {
            str: s,
            aliasColObjMap,
            throwErrorIfInvalid,
        })));
        // extract and flatten filters
        const filters = nestedFilters.reduce((acc, { filters }) => {
            if (!filters)
                return acc;
            return [...acc, ...filters];
        }, []);
        // extract and flatten errors
        const collectedErrors = nestedFilters.reduce((acc, { errors }) => {
            if (!errors)
                return acc;
            return [...acc, ...errors];
        }, []);
        // If errors exist, return them
        if (collectedErrors.length > 0) {
            return { errors: collectedErrors };
        }
        // If there's only one filter, return it directly
        if (filters.length === 1) {
            return { filters: nestedFilters };
        }
        // Otherwise, wrap it in an AND group
        return {
            filters: [
                {
                    is_group: true,
                    logical_op: 'and',
                    children: filters,
                },
            ],
        };
    }
    else if (typeof str !== 'string') {
        const message = 'Invalid filter format. Expected string or array of strings';
        if (throwErrorIfInvalid) {
            throw new invalid_filter_error_1.InvalidFilterError({ message });
        }
        else {
            errors.push({ message });
            return { errors };
        }
    }
    const parseResult = query_filter_parser_1.QueryFilterParser.parse(str);
    if (parseResult.lexErrors.length > 0 || parseResult.parseErrors.length > 0) {
        if (throwErrorIfInvalid)
            throw new invalid_filter_error_1.InvalidFilterError({
                lexingError: parseResult.lexErrors,
                parsingError: parseResult.parseErrors,
            });
        else {
            if (parseResult.lexErrors.length > 0) {
                errors.push({
                    message: parseResult.lexErrors
                        .map((k) => (0, error_message_parser_1.parseLexingError)(k))
                        .join(', '),
                });
            }
            else if (parseResult.parseErrors.length > 0) {
                errors.push({
                    message: parseResult.parseErrors
                        .map((k) => (0, error_message_parser_1.parseParsingError)(k))
                        .join(', '),
                });
            }
            return { errors };
        }
    }
    const filterSubType = parseResult.parsedCst;
    const { filter, errors: parseErrors } = mapFilterGroupSubType(context, {
        filter: filterSubType,
        aliasColObjMap,
        throwErrorIfInvalid,
        errors,
    });
    if ((parseErrors === null || parseErrors === void 0 ? void 0 : parseErrors.length) > 0) {
        return { errors: parseErrors };
    }
    return { filters: [filter] };
}
function mapFilterGroupSubType(context, { filter, aliasColObjMap, throwErrorIfInvalid = false, errors = [], }) {
    const children = filter.children
        .map((k) => k.is_group
        ? mapFilterGroupSubType(context, {
            filter: k,
            aliasColObjMap,
            throwErrorIfInvalid,
            errors,
        })
        : mapFilterClauseSubType(context, {
            filter: k,
            aliasColObjMap,
            throwErrorIfInvalid,
            errors,
        }))
        .filter((k) => k);
    if (children.length === 1) {
        return children[0];
    }
    else {
        return {
            filter: {
                is_group: filter.is_group,
                logical_op: filter.logical_op,
                children: children.map((k) => k.filter),
            },
            errors: (0, arrayHelpers_1.arrFlatMap)(children.map((k) => k.errors || [])).filter((k) => k),
        };
    }
}
function mapFilterClauseSubType(context, { filter, aliasColObjMap, throwErrorIfInvalid = false, errors = [], }) {
    const aliasCol = aliasColObjMap[filter.field];
    if (!aliasCol) {
        if (throwErrorIfInvalid) {
            throw new invalid_filter_error_1.InvalidFilterError({
                message: `Invalid filter field '${filter.field}' not found`,
            });
        }
        else {
            errors.push({
                message: `Invalid filter field '${filter.field}' not found`,
            });
            return { errors };
        }
    }
    const result = {
        fk_column_id: aliasCol.id,
        is_group: false,
        logical_op: filter.logical_op,
        comparison_op: filter.comparison_op,
        comparison_sub_op: undefined,
        value: filter.value,
    };
    return handleDataTypes(context, {
        filterType: result,
        column: aliasCol,
        throwErrorIfInvalid,
        errors,
    });
}
function handleDataTypes(context, { filterType, column, throwErrorIfInvalid = false, errors = [], }) {
    var _a, _b;
    if (filterType.value === null) {
        return { filter: filterType };
    }
    if ([
        UITypes_1.default.Date,
        UITypes_1.default.DateTime,
        UITypes_1.default.CreatedTime,
        UITypes_1.default.LastModifiedTime,
    ].includes(column.uidt) &&
        filterType.value) {
        const [subOp, ...value] = Array.isArray(filterType.value)
            ? filterType.value
            : filterType.value.split(',').map((k) => k.trim());
        filterType.comparison_sub_op = subOp;
        filterType.value = value.join('');
        if (filterType.comparison_sub_op) {
            if (!filterHelpers_1.COMPARISON_SUB_OPS.includes(filterType.comparison_sub_op)) {
                if (throwErrorIfInvalid)
                    throw new invalid_filter_error_1.InvalidFilterError({
                        message: `Invalid filter '${filterType.comparison_sub_op}' is not supported`,
                    });
                else {
                    errors.push({
                        message: `Invalid filter '${filterType.comparison_sub_op}' is not supported`,
                    });
                    return { errors };
                }
            }
        }
        if ((filterType.comparison_op === 'isWithin' &&
            !filterHelpers_1.IS_WITHIN_COMPARISON_SUB_OPS.includes(filterType.comparison_sub_op)) ||
            (filterType.comparison_op !== 'isWithin' &&
                filterHelpers_1.IS_WITHIN_COMPARISON_SUB_OPS.includes(filterType.comparison_sub_op))) {
            if (throwErrorIfInvalid)
                throw new invalid_filter_error_1.InvalidFilterError({
                    message: `Invalid filter '${filterType.comparison_sub_op}' is not supported for '${filterType.comparison_op}'`,
                });
            else {
                errors.push({
                    message: `Invalid filter '${filterType.comparison_sub_op}' is not supported for '${filterType.comparison_op}'`,
                });
                return { errors };
            }
        }
        if (filterType.value === '') {
            filterType.value = undefined;
        }
        filterType.meta = (_a = filterType.meta) !== null && _a !== void 0 ? _a : {};
        // use || to also include falsy values
        filterType.meta.timezone = (0, timezoneUtils_1.getNodejsTimezone)((_b = (0, helperFunctions_1.parseProp)(column.meta)) === null || _b === void 0 ? void 0 : _b.timezone, context.timezone);
    }
    return { filter: filterType };
}
//# sourceMappingURL=data:application/json;base64,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