"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.formulas = exports.API_DOC_PREFIX = void 0;
const dateTimeHelper_1 = require("../dateTimeHelper");
const enums_1 = require("./enums");
const error_1 = require("./error");
const lib_1 = require("../../lib");
exports.API_DOC_PREFIX = 'https://nocodb.com/docs/product-docs/fields';
const customValidationArray = (_argTypes, parsedTree) => {
    var _a, _b, _c, _d, _e;
    if (
    // check if first parameter exists
    !((_a = parsedTree.arguments) === null || _a === void 0 ? void 0 : _a[0]) ||
        (!((_c = (_b = parsedTree.arguments) === null || _b === void 0 ? void 0 : _b[0]) === null || _c === void 0 ? void 0 : _c.isDataArray) &&
            // data type array is backward compatibility
            ((_e = (_d = parsedTree.arguments) === null || _d === void 0 ? void 0 : _d[0]) === null || _e === void 0 ? void 0 : _e.dataType) !== enums_1.FormulaDataTypes.ARRAY)) {
        throw new error_1.FormulaError(enums_1.FormulaErrorType.TYPE_MISMATCH, { key: 'msg.formula.firstParamArray' }, 'First parameter need to be array. Either it is a link with Has Many / Many to Many relation, or either a Formula or Lookup that reference the link');
    }
};
exports.formulas = {
    AVG: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#avg`,
        validation: {
            args: {
                min: 1,
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Average of input parameters',
        syntax: 'AVG(value1, [value2, ...])',
        examples: [
            'AVG(10, 5) => 7.5',
            'AVG({column1}, {column2})',
            'AVG({column1}, {column2}, {column3})',
        ],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
    },
    ADD: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#add`,
        validation: {
            args: {
                min: 1,
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Sum of input parameters',
        syntax: 'ADD(value1, [value2, ...])',
        examples: [
            'ADD(5, 5) => 10',
            'ADD({column1}, {column2})',
            'ADD({column1}, {column2}, {column3})',
        ],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
    },
    DATEADD: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/date-functions#dateadd`,
        validation: {
            args: {
                rqd: 3,
                type: enums_1.FormulaDataTypes.DATE,
            },
            custom: (_argTypes, parsedTree) => {
                if (parsedTree.arguments[0].type === enums_1.JSEPNode.LITERAL) {
                    if (!(0, dateTimeHelper_1.validateDateWithUnknownFormat)(parsedTree.arguments[0].value)) {
                        throw new error_1.FormulaError(enums_1.FormulaErrorType.TYPE_MISMATCH, { key: 'msg.formula.firstParamDateAddHaveDate' }, 'First parameter of DATEADD should be a date');
                    }
                }
                if (parsedTree.arguments[1].type === enums_1.JSEPNode.LITERAL) {
                    if (typeof parsedTree.arguments[1].value !== 'number') {
                        throw new error_1.FormulaError(enums_1.FormulaErrorType.TYPE_MISMATCH, { key: 'msg.formula.secondParamDateAddHaveNumber' }, 'Second parameter of DATEADD should be a number');
                    }
                }
                if (parsedTree.arguments[2].type === enums_1.JSEPNode.LITERAL) {
                    if (![
                        'day',
                        'week',
                        'month',
                        'year',
                        'hour',
                        'minute',
                        'second',
                    ].includes(parsedTree.arguments[2].value)) {
                        throw new error_1.FormulaError(enums_1.FormulaErrorType.TYPE_MISMATCH, { key: 'msg.formula.thirdParamDateAddHaveDate' }, "Third parameter of DATEADD should be one of 'day', 'week', 'month', 'year', 'hour', 'minute', 'second'");
                    }
                }
            },
        },
        description: 'Adds "count" units to Datetime.',
        syntax: 'DATEADD(date | datetime, count, ["day" | "week" | "month" | "year" | "hour" | "minute" | "second"])',
        examples: [
            'DATEADD({column1}, 2, "day")',
            'DATEADD({column1}, -2, "day")',
            'DATEADD({column1}, 2, "week")',
            'DATEADD({column1}, -2, "week")',
            'DATEADD({column1}, 2, "month")',
            'DATEADD({column1}, -2, "month")',
            'DATEADD({column1}, 2, "year")',
            'DATEADD({column1}, -2, "year")',
        ],
        returnType: enums_1.FormulaDataTypes.DATE,
    },
    DATESTR: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/date-functions#datestr`,
        validation: {
            args: {
                rqd: 1,
            },
        },
        syntax: 'DATESTR(date | datetime)',
        description: 'Formats input field into a string in "YYYY-MM-DD" format',
        examples: ['DATESTR({column1})'],
        returnType: enums_1.FormulaDataTypes.STRING,
    },
    DAY: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/date-functions#day`,
        validation: {
            args: {
                rqd: 1,
            },
        },
        syntax: 'DAY(date | datetime)',
        description: 'Extract day from a date field (1-31)',
        examples: ['DAY({column1})'],
        returnType: enums_1.FormulaDataTypes.STRING,
    },
    MONTH: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/date-functions#month`,
        validation: {
            args: {
                rqd: 1,
            },
        },
        syntax: 'MONTH(date | datetime)',
        description: 'Extract month from a date field (1-12)',
        examples: ['MONTH({column1})'],
        returnType: enums_1.FormulaDataTypes.STRING,
    },
    YEAR: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/date-functions#year`,
        validation: {
            args: {
                rqd: 1,
            },
        },
        syntax: 'YEAR(date | datetime)',
        description: 'Extract year from a date field',
        examples: ['YEAR({column1})'],
        returnType: enums_1.FormulaDataTypes.STRING,
    },
    HOUR: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/date-functions#hour`,
        validation: {
            args: {
                rqd: 1,
            },
        },
        syntax: 'DAY(time | datetime)',
        description: 'Extract hour from a time field (0-23)',
        examples: ['HOUR({column1})'],
        returnType: enums_1.FormulaDataTypes.STRING,
    },
    DATETIME_DIFF: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/date-functions#datetime_diff`,
        validation: {
            args: {
                min: 2,
                max: 3,
                type: enums_1.FormulaDataTypes.DATE,
            },
            custom: (_argTypes, parsedTree) => {
                if (parsedTree.arguments[0].type === enums_1.JSEPNode.LITERAL) {
                    if (!(0, dateTimeHelper_1.validateDateWithUnknownFormat)(parsedTree.arguments[0].value)) {
                        throw new error_1.FormulaError(enums_1.FormulaErrorType.TYPE_MISMATCH, { key: 'msg.formula.firstParamDateDiffHaveDate' }, 'First parameter of DATETIME_DIFF should be a date');
                    }
                }
                if (parsedTree.arguments[1].type === enums_1.JSEPNode.LITERAL) {
                    if (!(0, dateTimeHelper_1.validateDateWithUnknownFormat)(parsedTree.arguments[1].value)) {
                        throw new error_1.FormulaError(enums_1.FormulaErrorType.TYPE_MISMATCH, { key: 'msg.formula.secondParamDateDiffHaveDate' }, 'Second parameter of DATETIME_DIFF should be a date');
                    }
                }
                if (parsedTree.arguments[2] &&
                    parsedTree.arguments[2].type === enums_1.JSEPNode.LITERAL) {
                    if (![
                        'milliseconds',
                        'ms',
                        'seconds',
                        's',
                        'minutes',
                        'm',
                        'hours',
                        'h',
                        'days',
                        'd',
                        'weeks',
                        'w',
                        'months',
                        'M',
                        'quarters',
                        'Q',
                        'years',
                        'y',
                    ].includes(parsedTree.arguments[2].value)) {
                        throw new error_1.FormulaError(enums_1.FormulaErrorType.TYPE_MISMATCH, { key: 'msg.formula.thirdParamDateDiffHaveDate' }, "Third parameter of DATETIME_DIFF should be one of 'milliseconds', 'ms', 'seconds', 's', 'minutes', 'm', 'hours', 'h', 'days', 'd', 'weeks', 'w', 'months', 'M', 'quarters', 'Q', 'years', 'y'");
                    }
                }
            },
        },
        description: 'Calculate the difference of two given date / datetime fields in specified units.',
        syntax: 'DATETIME_DIFF(date | datetime, date | datetime, ["milliseconds" | "ms" | "seconds" | "s" | "minutes" | "m" | "hours" | "h" | "days" | "d" | "weeks" | "w" | "months" | "M" | "quarters" | "Q" | "years" | "y"])',
        examples: [
            'DATEDIFF({column1}, {column2})',
            'DATEDIFF({column1}, {column2}, "seconds")',
            'DATEDIFF({column1}, {column2}, "s")',
            'DATEDIFF({column1}, {column2}, "years")',
            'DATEDIFF({column1}, {column2}, "y")',
            'DATEDIFF({column1}, {column2}, "minutes")',
            'DATEDIFF({column1}, {column2}, "m")',
            'DATEDIFF({column1}, {column2}, "days")',
            'DATEDIFF({column1}, {column2}, "d")',
        ],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
    },
    AND: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/conditional-expressions#and`,
        validation: {
            args: {
                min: 1,
            },
        },
        description: 'Result is TRUE if all conditions are met',
        syntax: 'AND(expr1, [expr2, ...])',
        examples: ['AND(5 > 2, 5 < 10) => 1', 'AND({column1} > 2, {column2} < 10)'],
        returnType: enums_1.FormulaDataTypes.COND_EXP,
    },
    OR: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/conditional-expressions#or`,
        validation: {
            args: {
                min: 1,
            },
        },
        description: 'Result is TRUE if at least one condition is met',
        syntax: 'OR(expr1, [expr2, ...])',
        examples: ['OR(5 > 2, 5 < 10) => 1', 'OR({column1} > 2, {column2} < 10)'],
        returnType: enums_1.FormulaDataTypes.COND_EXP,
    },
    CONCAT: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#concat`,
        validation: {
            args: {
                min: 1,
            },
        },
        description: 'Concatenate input parameters into a single string',
        syntax: 'CONCAT(str1, [str2, ...])',
        examples: [
            'CONCAT("AA", "BB", "CC") => "AABBCC"',
            'CONCAT({column1}, {column2}, {column3})',
        ],
        returnType: enums_1.FormulaDataTypes.STRING,
    },
    TRIM: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#trim`,
        validation: {
            args: {
                rqd: 1,
                type: enums_1.FormulaDataTypes.STRING,
            },
        },
        description: 'Remove trailing and leading whitespaces from input parameter',
        syntax: 'TRIM(str)',
        examples: [
            'TRIM("         HELLO WORLD  ") => "HELLO WORLD"',
            'TRIM({column1})',
        ],
        returnType: enums_1.FormulaDataTypes.STRING,
    },
    UPPER: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#upper`,
        validation: {
            args: {
                rqd: 1,
                type: enums_1.FormulaDataTypes.STRING,
            },
        },
        description: 'Converts the input parameter to an upper-case string.',
        syntax: 'UPPER(str)',
        examples: ['UPPER("nocodb") => "NOCODB"', 'UPPER({column1})'],
        returnType: enums_1.FormulaDataTypes.STRING,
    },
    LOWER: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#lower`,
        validation: {
            args: {
                rqd: 1,
                type: enums_1.FormulaDataTypes.STRING,
            },
        },
        description: 'Converts the input parameter to an lower-case string.',
        syntax: 'LOWER(str)',
        examples: ['LOWER("NOCODB") => "nocodb"', 'LOWER({column1})'],
        returnType: enums_1.FormulaDataTypes.STRING,
    },
    LEN: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#len`,
        validation: {
            args: {
                rqd: 1,
                type: enums_1.FormulaDataTypes.STRING,
            },
        },
        description: 'Calculate the character length of the input parameter.',
        syntax: 'LEN(value)',
        examples: ['LEN("NocoDB") => 6', 'LEN({column1})'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
    },
    MIN: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#min`,
        validation: {
            args: {
                min: 1,
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Find the minimum value among the input parameters.',
        syntax: 'MIN(value1, [value2, ...])',
        examples: ['MIN(1000, 2000) => 1000', 'MIN({column1}, {column2})'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
    },
    MAX: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#max`,
        validation: {
            args: {
                min: 1,
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Find the maximum value among the input parameters.',
        syntax: 'MAX(value1, [value2, ...])',
        examples: ['MAX(1000, 2000) => 2000', 'MAX({column1}, {column2})'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
    },
    CEILING: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#ceiling`,
        validation: {
            args: {
                rqd: 1,
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Rounds the input parameter to the next largest integer value.',
        syntax: 'CEILING(value)',
        examples: ['CEILING(1.01) => 2', 'CEILING({column1})'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
    },
    FLOOR: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#floor`,
        validation: {
            args: {
                rqd: 1,
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Round down the input parameter to the nearest integer.',
        syntax: 'FLOOR(value)',
        examples: ['FLOOR(3.1415) => 3', 'FLOOR({column1})'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
    },
    ROUND: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#round`,
        validation: {
            args: {
                min: 1,
                max: 2,
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Rounds the number to a specified decimal places or the nearest integer if precision is not specified',
        syntax: 'ROUND(value, precision), ROUND(value)',
        examples: [
            'ROUND(3.1415) => 3',
            'ROUND(3.1415, 2) => 3.14',
            'ROUND({column1}, 3)',
        ],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
    },
    MOD: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#mod`,
        validation: {
            args: {
                rqd: 2,
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Calculate the remainder resulting from integer division of input parameters.',
        syntax: 'MOD(value1, value2)',
        examples: ['MOD(1024, 1000) => 24', 'MOD({column}, 2)'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
    },
    REPEAT: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#repeat`,
        validation: {
            args: {
                rqd: 2,
            },
            custom(argTypes, parsedTree) {
                var _a, _b;
                if (argTypes[1] !== enums_1.FormulaDataTypes.NUMERIC) {
                    throw new error_1.FormulaError(enums_1.FormulaErrorType.INVALID_ARG, {
                        key: 'msg.formula.typeIsExpected',
                        type: 'Numeric',
                        calleeName: (_b = (_a = parsedTree.callee) === null || _a === void 0 ? void 0 : _a.name) === null || _b === void 0 ? void 0 : _b.toUpperCase(),
                        position: 2,
                    }, 'The REPEAT function requires a numeric as the parameter at position 2');
                }
            },
        },
        description: 'Concatenate the specified number of copies of the input parameter string.',
        syntax: 'REPEAT(str, count)',
        examples: ['REPEAT("A", 5) => "AAAAA"', 'REPEAT({column}, 5)'],
        returnType: enums_1.FormulaDataTypes.STRING,
    },
    LOG: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#log`,
        validation: {
            args: {
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Compute the logarithm of the input parameter to the specified base (default = e).',
        syntax: 'LOG([base], value)',
        examples: ['LOG(2, 1024) => 10', 'LOG(2, {column1})'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
    },
    EXP: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#exp`,
        validation: {
            args: {
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Compute the exponential value of the input parameter (e raised to the power specified)',
        syntax: 'EXP(power)',
        examples: ['EXP(1) => 2.718281828459045', 'EXP({column1})'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
    },
    POWER: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#power`,
        validation: {
            args: {
                rqd: 2,
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Compute base raised to the exponent power.',
        syntax: 'POWER(base, exponent)',
        examples: ['POWER(2, 10) => 1024', 'POWER({column1}, 10)'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
    },
    SQRT: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#sqrt`,
        validation: {
            args: {
                rqd: 1,
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Calculate the square root of the input parameter.',
        syntax: 'SQRT(value)',
        examples: ['SQRT(100) => 10', 'SQRT({column1})'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
    },
    ABS: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#abs`,
        validation: {
            args: {
                rqd: 1,
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Obtain the absolute value of the input parameter.',
        syntax: 'ABS(value)',
        examples: ['ABS({column1})'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
    },
    NOW: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/date-functions#now`,
        validation: {
            args: {
                rqd: 0,
                type: enums_1.FormulaDataTypes.DATE,
            },
        },
        description: 'Retrieve the current time and day.',
        syntax: 'NOW()',
        examples: ['NOW() => 2022-05-19 17:20:43'],
        returnType: enums_1.FormulaDataTypes.DATE,
    },
    REPLACE: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#replace`,
        validation: {
            args: {
                rqd: 3,
                type: enums_1.FormulaDataTypes.STRING,
            },
        },
        description: 'Replace all occurrences of "searchStr" with "replaceStr" in the given string.',
        syntax: 'REPLACE(str, searchStr, replaceStr)',
        examples: [
            'REPLACE("AABBCC", "AA", "BB") => "BBBBCC"',
            'REPLACE({column1}, {column2}, {column3})',
        ],
        returnType: enums_1.FormulaDataTypes.STRING,
    },
    SEARCH: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#search`,
        validation: {
            args: {
                rqd: 2,
                type: enums_1.FormulaDataTypes.STRING,
            },
        },
        description: 'Retrieve the index of the specified "searchStr" if found; otherwise, returns 0.',
        syntax: 'SEARCH(str, searchStr)',
        examples: [
            'SEARCH("HELLO WORLD", "WORLD") => 7',
            'SEARCH({column1}, "abc")',
        ],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
    },
    INT: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#int`,
        validation: {
            args: {
                rqd: 1,
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Obtain the integer value of the input parameter',
        syntax: 'INT(value)',
        examples: ['INT(3.1415) => 3', 'INT({column1})'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
    },
    RIGHT: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#right`,
        validation: {
            args: {
                rqd: 2,
                type: [enums_1.FormulaDataTypes.STRING, enums_1.FormulaDataTypes.NUMERIC],
            },
        },
        description: 'Retrieve the last n characters from the input string.',
        syntax: 'RIGHT(str, n)',
        examples: ['RIGHT("HELLO WORLD", 5) => WORLD', 'RIGHT({column1}, 3)'],
        returnType: enums_1.FormulaDataTypes.STRING,
    },
    LEFT: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#left`,
        validation: {
            args: {
                rqd: 2,
                type: [enums_1.FormulaDataTypes.STRING, enums_1.FormulaDataTypes.NUMERIC],
            },
        },
        description: 'Retrieve the first n characters from the input string.',
        syntax: 'LEFT(str, n)',
        examples: ['LEFT({column1}, 2)', 'LEFT("ABCD", 2) => "AB"'],
        returnType: enums_1.FormulaDataTypes.STRING,
    },
    SUBSTR: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#substr`,
        validation: {
            args: {
                min: 2,
                max: 3,
                type: [
                    enums_1.FormulaDataTypes.STRING,
                    enums_1.FormulaDataTypes.NUMERIC,
                    enums_1.FormulaDataTypes.NUMERIC,
                ],
            },
        },
        description: 'Extracts a substring of length "n" from the input string, starting from the specified position.',
        syntax: '	SUBTR(str, position, [n])',
        examples: [
            'SUBSTR("HELLO WORLD", 7) => WORLD',
            'SUBSTR("HELLO WORLD", 7, 3) => WOR',
            'SUBSTR({column1}, 7, 5)',
        ],
        returnType: enums_1.FormulaDataTypes.STRING,
    },
    MID: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#mid`,
        validation: {
            args: {
                rqd: 3,
                type: [
                    enums_1.FormulaDataTypes.STRING,
                    enums_1.FormulaDataTypes.NUMERIC,
                    enums_1.FormulaDataTypes.NUMERIC,
                ],
            },
        },
        description: 'Extracts a substring; an alias for SUBSTR.',
        syntax: 'MID(str, position, [count])',
        examples: ['MID("NocoDB", 3, 2) => "co"', 'MID({column1}, 3, 2)'],
        returnType: enums_1.FormulaDataTypes.STRING,
    },
    ISBLANK: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#isblank`,
        validation: {
            args: {
                rqd: 1,
            },
        },
        description: 'Check if the input parameter is blank.',
        syntax: 'ISBLANK(value)',
        examples: ['ISBLANK({column1}) => false', 'ISBLANK("") => true'],
        returnType: enums_1.FormulaDataTypes.BOOLEAN,
    },
    ISNOTBLANK: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#isnotblank`,
        validation: {
            args: {
                rqd: 1,
            },
        },
        description: 'Check if the input parameter is not blank.',
        syntax: 'ISNOTBLANK(value)',
        examples: ['ISNOTBLANK({column1}) => true', 'ISNOTBLANK("") => false'],
        returnType: enums_1.FormulaDataTypes.BOOLEAN,
    },
    IF: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/conditional-expressions#if`,
        validation: {
            args: {
                min: 2,
                max: 3,
            },
        },
        description: 'Evaluate successCase if the expression is TRUE, else the failureCase.',
        syntax: 'IF(expr, successCase, failureCase)',
        examples: [
            'IF(5 > 1, "YES", "NO") => "YES"',
            'IF({column} > 1, "YES", "NO")',
        ],
        returnType: (argTypes) => {
            // extract all return types except NULL, since null can be returned by any type
            const returnValueTypes = new Set(argTypes.slice(1).filter((type) => type !== enums_1.FormulaDataTypes.NULL));
            // if there are more than one return types or if there is a string return type
            // return type as string else return the type
            if (returnValueTypes.size > 1 ||
                returnValueTypes.has(enums_1.FormulaDataTypes.STRING)) {
                return enums_1.FormulaDataTypes.STRING;
            }
            else if (returnValueTypes.has(enums_1.FormulaDataTypes.NUMERIC)) {
                return enums_1.FormulaDataTypes.NUMERIC;
            }
            else if (returnValueTypes.has(enums_1.FormulaDataTypes.BOOLEAN)) {
                return enums_1.FormulaDataTypes.BOOLEAN;
            }
            else if (returnValueTypes.has(enums_1.FormulaDataTypes.DATE)) {
                return enums_1.FormulaDataTypes.DATE;
            }
            // if none of the above conditions are met, return the first return argument type
            return argTypes[1];
        },
    },
    SWITCH: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/conditional-expressions#switch`,
        validation: {
            args: {
                min: 3,
            },
            custom: (_argTypes, _parseTree) => {
                // Todo: Add validation for switch
            },
        },
        description: 'Evaluate case value based on expression output; if no match is found, evaluate default case.',
        syntax: 'SWITCH(expr, [pattern, value, ..., default])',
        examples: [
            'SWITCH(1, 1, "One", 2, "Two", "N/A") => "One""',
            'SWITCH(2, 1, "One", 2, "Two", "N/A") => "Two"',
            'SWITCH(3, 1, "One", 2, "Two", "N/A") => "N/A"',
            'SWITCH({column1}, 1, "One", 2, "Two", "N/A")',
        ],
        returnType: (argTypes) => {
            // extract all return types except NULL, since null can be returned by any type
            const returnValueTypes = new Set(argTypes.slice(2).filter((_, i) => i % 2 === 0));
            // if there are more than one return types or if there is a string return type
            // return type as string else return the type
            if (returnValueTypes.size > 1 ||
                returnValueTypes.has(enums_1.FormulaDataTypes.STRING)) {
                return enums_1.FormulaDataTypes.STRING;
            }
            else if (returnValueTypes.has(enums_1.FormulaDataTypes.NUMERIC)) {
                return enums_1.FormulaDataTypes.NUMERIC;
            }
            else if (returnValueTypes.has(enums_1.FormulaDataTypes.BOOLEAN)) {
                return enums_1.FormulaDataTypes.BOOLEAN;
            }
            else if (returnValueTypes.has(enums_1.FormulaDataTypes.DATE)) {
                return enums_1.FormulaDataTypes.DATE;
            }
            // if none of the above conditions are met, return the first return argument type
            return argTypes[1];
        },
    },
    URL: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#url`,
        validation: {
            args: {
                min: 1,
                max: 2,
                type: [enums_1.FormulaDataTypes.STRING, enums_1.FormulaDataTypes.STRING],
            },
        },
        description: 'Verify and convert to a hyperlink if the input is a valid URL.',
        syntax: 'URL(string, [label])',
        examples: [
            'URL("https://github.com/nocodb/nocodb")',
            'URL({column1})',
            'URL("https://github.com/nocodb/nocodb", "NocoDB")',
            'URL({column1}, {column1})',
        ],
        returnType: enums_1.FormulaDataTypes.STRING,
    },
    URLENCODE: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#urlencode`,
        validation: {
            args: {
                rqd: 1,
                type: enums_1.FormulaDataTypes.STRING,
            },
        },
        description: 'Percent-encode the input parameter for use in URLs',
        syntax: 'URLENCODE(str)',
        examples: [
            'URLENCODE("Hello, world") => "Hello%2C%20world"',
            'URLENCODE({column1})',
        ],
        returnType: enums_1.FormulaDataTypes.STRING,
    },
    WEEKDAY: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/date-functions#weekday`,
        validation: {
            args: {
                min: 1,
                max: 2,
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
            custom(_argTypes, parsedTree) {
                if (parsedTree.arguments[0].type === enums_1.JSEPNode.LITERAL) {
                    if (!(0, dateTimeHelper_1.validateDateWithUnknownFormat)(parsedTree.arguments[0].value)) {
                        throw new error_1.FormulaError(enums_1.FormulaErrorType.TYPE_MISMATCH, { key: 'msg.formula.firstParamWeekDayHaveDate' }, 'First parameter of WEEKDAY should be a date');
                    }
                }
                // if second argument is present and literal then validate it
                if (parsedTree.arguments[1] &&
                    parsedTree.arguments[1].type === enums_1.JSEPNode.LITERAL) {
                    const value = parsedTree.arguments[1].value;
                    if (typeof value !== 'string' ||
                        ![
                            'sunday',
                            'monday',
                            'tuesday',
                            'wednesday',
                            'thursday',
                            'friday',
                            'saturday',
                        ].includes(value.toLowerCase())) {
                        throw new error_1.FormulaError(enums_1.FormulaErrorType.TYPE_MISMATCH, { key: 'msg.formula.secondParamWeekDayHaveDate' }, 'Second parameter of WEEKDAY should be day of week string');
                    }
                }
            },
        },
        description: 'Retrieve the day of the week as an integer (0-6), starting from Monday by default.',
        syntax: 'WEEKDAY(date, [startDayOfWeek])',
        examples: ['WEEKDAY("2021-06-09")', 'WEEKDAY(NOW(), "sunday")'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
    },
    TRUE: {
        validation: {
            args: {
                max: 0,
            },
        },
        description: 'Returns 1',
        syntax: 'TRUE()',
        examples: ['TRUE()'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
        // TODO: Add docs url
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/logical-functions#true`,
    },
    FALSE: {
        validation: {
            args: {
                max: 0,
            },
        },
        description: 'Returns 0',
        syntax: 'FALSE()',
        examples: ['FALSE()'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
        // TODO: Add docs url
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/logical-functions#false`,
    },
    ARRAYUNIQUE: {
        validation: {
            args: {
                rqd: 1,
            },
            custom: customValidationArray,
        },
        description: 'Return unique items from the given array',
        syntax: 'ARRAYUNIQUE(value)',
        examples: ['ARRAYUNIQUE({column})'],
        returnType: enums_1.FormulaDataTypes.ARRAY,
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/array-functions#arrayunique`,
    },
    ARRAYSORT: {
        validation: {
            args: {
                min: 1,
                max: 2,
            },
            custom: customValidationArray,
        },
        description: 'Sort an array result',
        syntax: 'ARRAYSORT(value, [direction])',
        examples: ['ARRAYSORT({column}, "desc")'],
        returnType: enums_1.FormulaDataTypes.ARRAY,
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/array-functions#arraysort`,
    },
    ARRAYCOMPACT: {
        validation: {
            args: {
                rqd: 1,
            },
            custom: customValidationArray,
        },
        description: 'Removes empty strings and null values from the array',
        syntax: 'ARRAYCOMPACT(value)',
        examples: ['ARRAYCOMPACT({column})'],
        returnType: enums_1.FormulaDataTypes.ARRAY,
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/array-functions#arraycompact`,
    },
    ARRAYSLICE: {
        validation: {
            args: {
                min: 2,
                max: 3,
            },
            custom: customValidationArray,
        },
        description: 'Removes empty strings and null values from the array',
        syntax: 'ARRAYSLICE(value, start, [end])',
        examples: ['ARRAYSLICE({column})'],
        returnType: enums_1.FormulaDataTypes.ARRAY,
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/array-functions#arrayslice`,
    },
    REGEX_MATCH: {
        validation: {
            args: {
                rqd: 2,
                type: enums_1.FormulaDataTypes.STRING,
            },
        },
        description: 'Verifies whether the input text matches a regular expression, returning 1 for a match and 0 otherwise.',
        syntax: 'REGEX_MATCH(string, regex)',
        examples: ['REGEX_MATCH({title}, "abc.*")'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#regex_match`,
    },
    REGEX_EXTRACT: {
        validation: {
            args: {
                rqd: 2,
                type: enums_1.FormulaDataTypes.STRING,
            },
        },
        description: 'Retrieve the first match of a regular expression in a string.',
        syntax: 'REGEX_EXTRACT(string, regex)',
        examples: ['REGEX_EXTRACT({title}, "abc.*")'],
        returnType: enums_1.FormulaDataTypes.STRING,
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#regex_extract`,
    },
    REGEX_REPLACE: {
        validation: {
            args: {
                rqd: 3,
                type: enums_1.FormulaDataTypes.STRING,
            },
        },
        description: 'Replace all occurrences of a regular expression in a string with a specified replacement string.',
        syntax: 'REGEX_MATCH(string, regex, replacement)',
        examples: ['REGEX_EXTRACT({title}, "abc.*", "abcd")'],
        returnType: enums_1.FormulaDataTypes.STRING,
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/string-functions#regex_replace`,
    },
    BLANK: {
        validation: {
            args: {
                rqd: 0,
            },
        },
        description: 'Yields a null value.',
        syntax: 'BLANK()',
        examples: ['BLANK()'],
        returnType: enums_1.FormulaDataTypes.NULL,
        // TODO: Add docs url
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/logical-functions#blank`,
    },
    XOR: {
        validation: {
            args: {
                min: 1,
            },
            // todo: validation for boolean
        },
        description: 'Verifies whether an odd number of arguments are true, returning true if so, and false otherwise.',
        syntax: 'XOR(expression, [exp2, ...])',
        examples: ['XOR(TRUE(), FALSE(), TRUE())'],
        returnType: enums_1.FormulaDataTypes.BOOLEAN,
        // TODO: Add docs url
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/logical-functions#xor`,
    },
    EVEN: {
        validation: {
            args: {
                rqd: 1,
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Rounds up the specified value to the nearest even integer that is greater than or equal to the specified value',
        syntax: 'EVEN(value)',
        examples: ['EVEN({column})'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#even`,
    },
    ODD: {
        validation: {
            args: {
                rqd: 1,
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Rounds up the specified value to the nearest odd integer that is greater than or equal to the specified value',
        syntax: 'ODD(value)',
        examples: ['ODD({column})'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#odd`,
    },
    RECORD_ID: {
        validation: {
            args: {
                rqd: 0,
            },
        },
        description: 'Retrieve the record ID of the current record.',
        syntax: 'RECORD_ID()',
        examples: ['RECORD_ID()'],
        // todo: resolve return type based on the args
        returnType: () => {
            return enums_1.FormulaDataTypes.STRING;
        },
    },
    COUNTA: {
        validation: {
            args: {
                min: 1,
            },
        },
        description: 'Counts non-empty arguments',
        syntax: 'COUNTA(value1, [value2, ...])',
        examples: ['COUNTA({field1}, {field2})'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#counta`,
    },
    COUNT: {
        validation: {
            args: {
                min: 1,
            },
        },
        description: 'Counts numerical arguments',
        syntax: 'COUNT(value1, [value2, ...])',
        examples: ['COUNT({field1}, {field2})'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#count`,
    },
    COUNTALL: {
        validation: {
            args: {
                min: 1,
            },
        },
        description: 'Counts the number of arguments',
        syntax: 'COUNTALL(value1, [value2, ...])',
        examples: ['COUNTALL({field1}, {field2})'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#countall`,
    },
    ROUNDDOWN: {
        validation: {
            args: {
                min: 1,
                max: 2,
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Rounds down the value after the decimal point to the specified number of decimal places given by "precision" (default is 0).',
        syntax: 'ROUNDDOWN(value, [precision])',
        examples: ['ROUNDDOWN({field1})', 'ROUNDDOWN({field1}, 2)'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#rounddown`,
    },
    ROUNDUP: {
        validation: {
            args: {
                min: 1,
                max: 2,
                type: enums_1.FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Rounds up the value after the decimal point to the specified number of decimal places given by "precision" (default is 0).',
        syntax: 'ROUNDUP(value, [precision])',
        examples: ['ROUNDUP({field1})', 'ROUNDUP({field1}, 2)'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#roundup`,
    },
    VALUE: {
        validation: {
            args: {
                rqd: 1,
            },
        },
        description: 'Extracts the numeric value from a string, handling % or - appropriately, and returns the resulting numeric value.',
        syntax: 'VALUE(value)',
        examples: ['VALUE({field})', 'VALUE("abc10000%")', 'VALUE("$10000")'],
        returnType: enums_1.FormulaDataTypes.NUMERIC,
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/numeric-functions#value`,
    },
    JSON_EXTRACT: {
        docsUrl: `${exports.API_DOC_PREFIX}/field-types/formula/json-functions#json_extract`,
        validation: {
            args: {
                min: 2,
                max: 2,
                type: [enums_1.FormulaDataTypes.STRING, enums_1.FormulaDataTypes.STRING],
            },
        },
        description: 'Extracts a value from a JSON string using a jq-like syntax',
        syntax: 'JSON_EXTRACT(json_string, path)',
        examples: [
            'JSON_EXTRACT(\'{"a": {"b": "c"}}\', \'.a.b\') => "c"',
            "JSON_EXTRACT({json_column}, '.key')",
        ],
        returnType: enums_1.FormulaDataTypes.STRING,
    },
    // Disabling these functions for now; these act as alias for CreatedAt & UpdatedAt fields;
    // Issue: Error noticed if CreatedAt & UpdatedAt fields are removed from the table after creating these formulas
    //
    // CREATED_TIME: {
    //   validation: {
    //     args: {
    //       rqd: 0,
    //     },
    //   },
    //   description: 'Returns the created time of the current record if it exists',
    //   syntax: 'CREATED_TIME()',
    //   examples: ['CREATED_TIME()'],
    // },
    LAST_MODIFIED_TIME: {
        validation: {
            args: {
                min: 0,
            },
            custom(_argTypes, parsedTree, columns) {
                if (parsedTree.arguments.length) {
                    // check if meta column exists or not, if not throw error when user provides args}
                    if (!columns.find((column) => column.uidt === lib_1.UITypes.Meta)) {
                        throw new error_1.FormulaError(enums_1.FormulaErrorType.INVALID_ARG, {}, 'LAST_MODIFIED_TIME with field arguments is not supported');
                    }
                }
                for (const arg of parsedTree.arguments) {
                    if (arg.type !== enums_1.JSEPNode.IDENTIFIER) {
                        throw new error_1.FormulaError(enums_1.FormulaErrorType.INVALID_ARG, {}, 'Only column references are allowed as arguments for LAST_MODIFIED_TIME');
                    }
                    // now check if the column allows LAST_MODIFIED_TIME
                    const virtualOrSystemColRef = columns.find((c) => (c.title === arg.name || c.id === arg.name) &&
                        ((0, lib_1.isSystemColumn)(c) || (0, lib_1.isVirtualCol)(c)));
                    if (virtualOrSystemColRef) {
                        throw new error_1.FormulaError(enums_1.FormulaErrorType.INVALID_ARG, {}, `LAST_MODIFIED_TIME() is not supported for ${virtualOrSystemColRef.system
                            ? 'System'
                            : virtualOrSystemColRef.uidt} field.`);
                    }
                }
            },
        },
        description: 'Returns the last modified time of the current record or selection if it exists',
        syntax: 'LAST_MODIFIED_TIME([field1, ...])',
        examples: [
            'LAST_MODIFIED_TIME()',
            'LAST_MODIFIED_TIME({Status}, {Priority})',
        ],
        returnType: enums_1.FormulaDataTypes.DATE,
    },
};
//# sourceMappingURL=data:application/json;base64,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