"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractCallExpressionReferencedInfo = exports.extractBinaryExpReferencedInfo = exports.getReferencedInfoFromArgs = void 0;
const arrayHelpers_1 = require("../arrayHelpers");
const UITypes_1 = __importDefault(require("../UITypes"));
const enums_1 = require("./enums");
const arrayHelpers_2 = require("../arrayHelpers");
const IMPURE_OPR_UIDT_MAP = new Map([
    [UITypes_1.default.SingleSelect, UITypes_1.default.SingleLineText],
    [UITypes_1.default.MultiSelect, UITypes_1.default.SingleLineText],
    [UITypes_1.default.Attachment, UITypes_1.default.SingleLineText],
    [UITypes_1.default.User, UITypes_1.default.SingleLineText],
    [UITypes_1.default.Email, UITypes_1.default.SingleLineText],
    [UITypes_1.default.URL, UITypes_1.default.SingleLineText],
    [UITypes_1.default.Number, UITypes_1.default.Decimal],
    [UITypes_1.default.Rating, UITypes_1.default.Decimal],
    [UITypes_1.default.Year, UITypes_1.default.Number],
    [UITypes_1.default.Year, UITypes_1.default.Number],
]);
/**
 * Filters and refines referenced column information based on allowed UI data types (UIDTs).
 * This function ensures that the data types associated with referenced columns and their candidates
 * are valid and consistent with a set of allowed UIDTs for a specific operation or formula.
 *
 * @param {object} params - The parameters for filtering.
 * @param {ReferencedInfo} params.referencedInfo - An object containing details about a referenced column,
 *   including the `referencedColumn` itself (with its ID and UIDT) and a list of `uidtCandidates` (potential UIDTs).
 * @param {UITypes[]} params.allowedUidts - An array of `UITypes` that are considered valid for the current context.
 * @param {UITypes} params.defaultUidt - A fallback `UIType` to use if no valid `uidtCandidates` are found after filtering.
 * @param {boolean} [params.isPureOperation=false] - A flag indicating if the operation is "pure."
 *   A pure operation (e.g., TRIM, ARRAYUNIQUE) doesn't fundamentally change the display value's
 *   underlying type, even if it modifies the value itself. Impure operations might map certain
 *   UI types (like SingleSelect) to a more generic type (like SingleLineText).
 * @returns {ReferencedInfo} A new `ReferencedInfo` object with the updated `referencedColumn`,
 *   `uidtCandidates`, and `invalidForReferenceColumn` status, ensuring consistency with formula requirements.
 */
const filterReferencedInfoByUidt = ({ referencedInfo, allowedUidts, defaultUidt, isPureOperation = false, }) => {
    var _a, _b;
    let referencedColumn = referencedInfo.referencedColumn;
    if (referencedColumn && !isPureOperation) {
        referencedColumn =
            allowedUidts.includes(referencedInfo.referencedColumn.uidt) &&
                !IMPURE_OPR_UIDT_MAP.has(referencedInfo.referencedColumn.uidt)
                ? referencedInfo.referencedColumn
                : undefined;
    }
    let uidtCandidates = (_b = (_a = referencedInfo.uidtCandidates) === null || _a === void 0 ? void 0 : _a.filter((uidt) => allowedUidts.includes(uidt))) !== null && _b !== void 0 ? _b : [];
    if (!isPureOperation) {
        uidtCandidates = uidtCandidates
            .map((c) => { var _a; return (_a = IMPURE_OPR_UIDT_MAP.get(c)) !== null && _a !== void 0 ? _a : c; })
            .filter((k) => k);
    }
    return Object.assign(Object.assign({}, referencedInfo), { referencedColumn, uidtCandidates: uidtCandidates.length > 0 ? (0, arrayHelpers_1.arrUniq)(uidtCandidates) : [defaultUidt], 
        // Determine if the referenced column is invalid. This can happen if it was already marked as invalid,
        // or if the `referencedColumn` was present in the original `referencedInfo` but got filtered out
        // during the current processing (i.e., `referencedColumn` is now undefined while `referencedInfo.referencedColumn` was not).
        invalidForReferenceColumn: referencedInfo.invalidForReferenceColumn ||
            !!referencedColumn !== !!referencedInfo.referencedColumn });
};
/**
 * Extracts and refines information about referenced columns from an array of parsed formula nodes.
 * This function is crucial for ensuring type compatibility and validity of referenced columns
 * within formula arguments based on the expected data type of the formula's output.
 *
 * @param {ParsedFormulaNode[]} nodes - An array of parsed formula nodes, representing the arguments
 *   of a formula. Each node may contain information about referenced columns and their potential
 *   UI data types (UIDTs).
 * @param {FormulaDataTypes} dataType - The expected output data type of the formula. This is used
 *   to filter and validate the UIDTs of the referenced columns.
 * @param {{ isPureOperation?: boolean }} [option] - An optional object containing additional
 *   configuration.
 * @param {boolean} [option.isPureOperation=false] - A flag indicating if the operation is "pure."
 *   A pure operation (e.g., TRIM, ARRAYUNIQUE) does not fundamentally change the display value's
 *   underlying type, even if it modifies the value itself. Impure operations might map certain
 *   UI types (like SingleSelect) to a more generic type (like SingleLineText).
 * @returns {ReferencedInfo} A `ReferencedInfo` object containing the aggregated and filtered
 *   information about referenced columns, their UIDT candidates, and a flag indicating if the
 *   reference is invalid.
 */
const getReferencedInfoFromArgs = (nodes, dataType, option = { isPureOperation: false }) => {
    let referencedColumn;
    let invalidForReferenceColumn = nodes.some((c) => c.invalidForReferenceColumn);
    const referencedColumns = invalidForReferenceColumn
        ? []
        : nodes.map((k) => k.referencedColumn).filter((k) => k);
    const uniqueLength = (0, arrayHelpers_1.arrUniq)(referencedColumns.map((k) => k.id)).length;
    if (uniqueLength === 1) {
        referencedColumn = referencedColumns[0];
    }
    else if (uniqueLength > 1) {
        invalidForReferenceColumn = true;
    }
    const uidtCandidates = (0, arrayHelpers_2.arrFlatMap)(nodes.map((k) => k.uidtCandidates));
    const referencedInfo = {
        referencedColumn,
        uidtCandidates,
        invalidForReferenceColumn,
    };
    switch (dataType) {
        case enums_1.FormulaDataTypes.BOOLEAN: {
            return filterReferencedInfoByUidt(Object.assign({ referencedInfo, allowedUidts: [UITypes_1.default.Checkbox], defaultUidt: UITypes_1.default.Checkbox }, option));
        }
        case enums_1.FormulaDataTypes.COND_EXP: {
            return filterReferencedInfoByUidt(Object.assign({ referencedInfo, allowedUidts: [UITypes_1.default.Checkbox], defaultUidt: UITypes_1.default.Checkbox }, option));
        }
        case enums_1.FormulaDataTypes.STRING: {
            return filterReferencedInfoByUidt(Object.assign({ referencedInfo, allowedUidts: [
                    UITypes_1.default.SingleLineText,
                    UITypes_1.default.LongText,
                    UITypes_1.default.Email,
                    UITypes_1.default.URL,
                ], defaultUidt: UITypes_1.default.SingleLineText }, option));
        }
        case enums_1.FormulaDataTypes.NUMERIC: {
            return filterReferencedInfoByUidt(Object.assign({ referencedInfo, allowedUidts: [
                    UITypes_1.default.Number,
                    UITypes_1.default.Decimal,
                    UITypes_1.default.Currency,
                    UITypes_1.default.Percent,
                    UITypes_1.default.Rating,
                ], defaultUidt: UITypes_1.default.Decimal }, option));
        }
        case enums_1.FormulaDataTypes.DATE: {
            return filterReferencedInfoByUidt(Object.assign({ referencedInfo, allowedUidts: [UITypes_1.default.Date, UITypes_1.default.DateTime], defaultUidt: UITypes_1.default.DateTime }, option));
        }
        case enums_1.FormulaDataTypes.INTERVAL: {
            return filterReferencedInfoByUidt(Object.assign({ referencedInfo, allowedUidts: [UITypes_1.default.Time, UITypes_1.default.Duration], defaultUidt: UITypes_1.default.Time }, option));
        }
        case enums_1.FormulaDataTypes.ARRAY: {
            return filterReferencedInfoByUidt(Object.assign({ referencedInfo, allowedUidts: [UITypes_1.default.LinkToAnotherRecord, UITypes_1.default.Lookup], defaultUidt: UITypes_1.default.SingleLineText }, option));
        }
        default: {
            return {
                uidtCandidates: [UITypes_1.default.SingleLineText],
                invalidForReferenceColumn,
            };
        }
    }
};
exports.getReferencedInfoFromArgs = getReferencedInfoFromArgs;
/**
 * get referenced column info for call expression
 */
const extractBinaryExpReferencedInfo = ({ parsedTree, left, right, }) => {
    return (0, exports.getReferencedInfoFromArgs)([left, right], parsedTree.dataType, {
        isPureOperation: false,
    });
};
exports.extractBinaryExpReferencedInfo = extractBinaryExpReferencedInfo;
/**
 * get referenced column info for call expression
 */
const extractCallExpressionReferencedInfo = ({ parsedTree, }) => {
    var _a;
    let nodeArgs = (_a = parsedTree.arguments) !== null && _a !== void 0 ? _a : [];
    let isPureOperation = false;
    switch (parsedTree.callee.name) {
        case 'TRIM':
        case 'MAX':
        case 'MIN':
            isPureOperation = true;
            break;
        case 'DATEADD':
        case 'ROUND':
        case 'MOD':
        case 'POWER':
        case 'REPEAT':
        case 'REPLACE':
        case 'REGEX_EXTRACT':
        case 'REGEX_REPLACE':
        case 'MID':
        case 'LEFT':
        case 'RIGHT':
        case 'ROUNDDOWN':
        case 'ROUNDUP':
        case 'JSON_EXTRACT':
        case 'SUBSTR': {
            nodeArgs = nodeArgs.slice(0, 1);
            break;
        }
        case 'LOG': {
            nodeArgs = nodeArgs.slice(1, 2);
            break;
        }
        // URL won't need the referenced column info
        case 'URL':
        case 'REGEX_MATCH':
        case 'ISBLANK':
        case 'ISNOTBLANK':
        case 'DAY':
        case 'MONTH':
        case 'YEAR':
        case 'HOUR':
        case 'WEEKDAY':
        case 'LEN':
        case 'DATESTR':
        case 'AND':
        case 'OR':
        case 'XOR':
        case 'COUNTA':
        case 'COUNT':
        case 'COUNTALL':
        case 'SEARCH': {
            nodeArgs = [];
            break;
        }
        case 'ARRAYUNIQUE':
        case 'ARRAYSORT':
        case 'ARRAYCOMPACT':
        case 'ARRAYSLICE': {
            nodeArgs = nodeArgs.slice(0, 1);
            isPureOperation = true;
            break;
        }
        case 'IF': {
            nodeArgs = nodeArgs.slice(1);
            // is pure operation if the 2nd and 3rd arguments only has data type NULL or single referenced column
            const uniqueReferencedColumnIds = (0, arrayHelpers_1.arrUniq)(nodeArgs.map((n) => { var _a; return (_a = n.referencedColumn) === null || _a === void 0 ? void 0 : _a.id; }).filter((k) => k));
            const notNullDataTypes = nodeArgs
                .filter((n) => !n.referencedColumn && n.dataType !== enums_1.FormulaDataTypes.NULL)
                .map((n) => n.dataType);
            isPureOperation =
                notNullDataTypes.length === 0 && uniqueReferencedColumnIds.length <= 1;
            break;
        }
        case 'SWITCH': {
            nodeArgs = nodeArgs.slice(1).filter((_val, index) => index % 2 === 1);
            // is pure operation if resulting arguments only has data type NULL or single referenced column
            // is pure operation if the 2nd and 3rd arguments only has data type NULL or single referenced column
            const uniqueReferencedColumnIds = (0, arrayHelpers_1.arrUniq)(nodeArgs.map((n) => { var _a; return (_a = n.referencedColumn) === null || _a === void 0 ? void 0 : _a.id; }).filter((k) => k));
            const notNullDataTypes = nodeArgs
                .filter((n) => !n.referencedColumn && n.dataType !== enums_1.FormulaDataTypes.NULL)
                .map((n) => n.dataType);
            isPureOperation =
                notNullDataTypes.length === 0 && uniqueReferencedColumnIds.length <= 1;
            break;
        }
    }
    return (0, exports.getReferencedInfoFromArgs)(nodeArgs, parsedTree.dataType, {
        isPureOperation,
    });
};
exports.extractCallExpressionReferencedInfo = extractCallExpressionReferencedInfo;
//# sourceMappingURL=data:application/json;base64,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