"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateFormulaAndExtractTreeWithType = validateFormulaAndExtractTreeWithType;
const UITypes_1 = __importDefault(require("../../lib/UITypes"));
const enums_1 = require("../../lib/formula/enums");
const error_1 = require("../../lib/formula/error");
const handle_formula_error_1 = require("../../lib/formula/handle-formula-error");
const formulas_1 = require("../../lib/formula/formulas");
const jsepInstances_1 = require("../../lib/formula/jsepInstances");
const sqlUi_1 = require("../../lib/sqlUi");
const referenced_info_extractor_1 = require("../../lib/formula/referenced-info-extractor");
const unifiedMeta_1 = require("../../lib/unifiedMeta");
const getColOptions_1 = require("../../lib/unifiedMeta/getColOptions");
async function extractColumnIdentifierType({ col, columns, getMeta, clientOrSqlUi, }) {
    var _a, _b, _c, _d, _e, _f;
    const res = {};
    const sqlUI = typeof clientOrSqlUi === 'string'
        ? sqlUi_1.SqlUiFactory.create({ client: clientOrSqlUi })
        : clientOrSqlUi;
    switch (col === null || col === void 0 ? void 0 : col.uidt) {
        // string
        case UITypes_1.default.SingleLineText:
        case UITypes_1.default.LongText:
        case UITypes_1.default.MultiSelect:
        case UITypes_1.default.SingleSelect:
        case UITypes_1.default.PhoneNumber:
        case UITypes_1.default.Email:
        case UITypes_1.default.URL:
        case UITypes_1.default.User:
        case UITypes_1.default.CreatedBy:
        case UITypes_1.default.LastModifiedBy:
            res.dataType = enums_1.FormulaDataTypes.STRING;
            break;
        // numeric
        case UITypes_1.default.Year:
        case UITypes_1.default.Number:
        case UITypes_1.default.Decimal:
        case UITypes_1.default.Rating:
        case UITypes_1.default.Count:
        case UITypes_1.default.AutoNumber:
            res.dataType = enums_1.FormulaDataTypes.NUMERIC;
            break;
        // date
        case UITypes_1.default.Date:
        case UITypes_1.default.DateTime:
        case UITypes_1.default.CreatedTime:
        case UITypes_1.default.LastModifiedTime:
            res.dataType = enums_1.FormulaDataTypes.DATE;
            break;
        case UITypes_1.default.Currency:
        case UITypes_1.default.Percent:
        case UITypes_1.default.Duration:
        case UITypes_1.default.Links:
            res.dataType = enums_1.FormulaDataTypes.NUMERIC;
            break;
        case UITypes_1.default.Rollup:
            {
                const rollupFunction = (await unifiedMeta_1.unifiedMeta.getColOptions(unifiedMeta_1.unifiedMeta.getContextFromObject(col), {
                    column: col,
                })).rollup_function;
                if ([
                    'count',
                    'avg',
                    'sum',
                    'countDistinct',
                    'sumDistinct',
                    'avgDistinct',
                ].includes(rollupFunction)) {
                    // these functions produce a numeric value, which can be used in numeric functions
                    res.dataType = enums_1.FormulaDataTypes.NUMERIC;
                }
                else {
                    const rollupColOptions = await (0, getColOptions_1.getColOptions)(unifiedMeta_1.unifiedMeta.getContextFromObject(col), { column: col });
                    const relationColumn = columns.find((column) => column.id === rollupColOptions.fk_relation_column_id);
                    const relationColumnOpt = await (0, getColOptions_1.getColOptions)(unifiedMeta_1.unifiedMeta.getContextFromObject(col), { column: relationColumn });
                    // the value is based on the foreign rollup column type
                    const refTableMeta = await getMeta(unifiedMeta_1.unifiedMeta.getContextFromObject(Object.assign(Object.assign({}, col), { base_id: (_a = relationColumnOpt.fk_related_base_id) !== null && _a !== void 0 ? _a : col.base_id })), {
                        id: relationColumnOpt.fk_related_model_id,
                    });
                    const refTableColumns = refTableMeta.columns;
                    const childFieldColumn = refTableColumns.find((col) => col.id === rollupColOptions.fk_rollup_column_id);
                    // extract type and add to res
                    Object.assign(res, await extractColumnIdentifierType({
                        col: childFieldColumn,
                        columns: refTableColumns,
                        getMeta,
                        clientOrSqlUi,
                    }));
                }
            }
            break;
        case UITypes_1.default.Attachment:
            res.dataType = enums_1.FormulaDataTypes.STRING;
            break;
        case UITypes_1.default.Checkbox:
            if (col.dt === 'boolean' || col.dt === 'bool') {
                res.dataType = enums_1.FormulaDataTypes.BOOLEAN;
            }
            else {
                res.dataType = enums_1.FormulaDataTypes.NUMERIC;
            }
            break;
        case UITypes_1.default.Time:
            res.dataType = enums_1.FormulaDataTypes.INTERVAL;
            break;
        case UITypes_1.default.ID:
        case UITypes_1.default.ForeignKey:
        case UITypes_1.default.SpecificDBType:
            {
                if (sqlUI) {
                    const abstractType = sqlUI.getAbstractType(col);
                    if (['integer', 'float', 'decimal'].includes(abstractType)) {
                        res.dataType = enums_1.FormulaDataTypes.NUMERIC;
                    }
                    else if (['boolean'].includes(abstractType)) {
                        res.dataType = enums_1.FormulaDataTypes.BOOLEAN;
                    }
                    else if (['date', 'datetime', 'time', 'year'].includes(abstractType)) {
                        res.dataType = enums_1.FormulaDataTypes.DATE;
                    }
                    else {
                        res.dataType = enums_1.FormulaDataTypes.STRING;
                    }
                }
                else {
                    res.dataType = enums_1.FormulaDataTypes.UNKNOWN;
                }
            }
            break;
        // not supported
        case UITypes_1.default.Lookup: {
            const colContext = unifiedMeta_1.unifiedMeta.getContextFromObject(col);
            const lookupColOption = await unifiedMeta_1.unifiedMeta.getColOptions(colContext, {
                column: col,
            });
            const lookupInfo = await unifiedMeta_1.unifiedMeta.getLookupRelatedInfo(colContext, {
                colOptions: lookupColOption,
                columns,
                getMeta,
            });
            const relationColumn = lookupInfo.relationColumn;
            const lookupColumn = lookupInfo.lookupColumn;
            const lookupColumnIdentifierType = await extractColumnIdentifierType({
                col: lookupColumn,
                clientOrSqlUi,
                columns: await unifiedMeta_1.unifiedMeta.getColumns(unifiedMeta_1.unifiedMeta.getContextFromObject(lookupInfo.relatedTable), { model: lookupInfo.relatedTable }),
                getMeta,
            });
            res.dataType = lookupColumnIdentifierType.dataType;
            res.isDataArray = lookupColumnIdentifierType.isDataArray;
            if (!res.isDataArray) {
                const relationColOptions = await unifiedMeta_1.unifiedMeta.getColOptions(colContext, {
                    column: relationColumn,
                });
                res.isDataArray = ['hm', 'mm'].includes(relationColOptions.type);
            }
            res.referencedColumn = {
                id: (_b = lookupColumnIdentifierType === null || lookupColumnIdentifierType === void 0 ? void 0 : lookupColumnIdentifierType.referencedColumn) === null || _b === void 0 ? void 0 : _b.id,
                // if array, we present it as lookup column
                uidt: (_c = lookupColumnIdentifierType === null || lookupColumnIdentifierType === void 0 ? void 0 : lookupColumnIdentifierType.referencedColumn) === null || _c === void 0 ? void 0 : _c.uidt,
                intermediaryUidt: UITypes_1.default.Lookup,
                intermediaryId: col.id,
            };
            break;
        }
        case UITypes_1.default.LinkToAnotherRecord: {
            const colOptions = await unifiedMeta_1.unifiedMeta.getColOptions(unifiedMeta_1.unifiedMeta.getContextFromObject(col), {
                column: col,
            });
            const relatedTable = await unifiedMeta_1.unifiedMeta.getLTARRelatedTable(unifiedMeta_1.unifiedMeta.getContextFromObject(col), {
                colOptions,
                getMeta,
            });
            const relatedTableColumns = await unifiedMeta_1.unifiedMeta.getColumns(unifiedMeta_1.unifiedMeta.getContextFromObject(relatedTable), {
                model: relatedTable,
            });
            const relatedTableDisplayColumn = relatedTableColumns.find((col) => col.pv);
            const relatedColumnIdentifierType = await extractColumnIdentifierType({
                col: relatedTableDisplayColumn,
                clientOrSqlUi,
                columns: relatedTableColumns,
                getMeta,
            });
            res.dataType = relatedColumnIdentifierType.dataType;
            res.isDataArray =
                relatedColumnIdentifierType.isDataArray ||
                    ['hm', 'mm'].includes(colOptions.type);
            res.referencedColumn = {
                id: (_d = relatedColumnIdentifierType === null || relatedColumnIdentifierType === void 0 ? void 0 : relatedColumnIdentifierType.referencedColumn) === null || _d === void 0 ? void 0 : _d.id,
                uidt: (_e = relatedColumnIdentifierType === null || relatedColumnIdentifierType === void 0 ? void 0 : relatedColumnIdentifierType.referencedColumn) === null || _e === void 0 ? void 0 : _e.uidt,
                intermediaryUidt: UITypes_1.default.LinkToAnotherRecord,
                intermediaryId: col.id,
            };
            break;
        }
        case UITypes_1.default.Formula: {
            const colOptions = await unifiedMeta_1.unifiedMeta.getColOptions(unifiedMeta_1.unifiedMeta.getContextFromObject(col), {
                column: col,
            });
            const parsedTree = await unifiedMeta_1.unifiedMeta.getParsedTree(unifiedMeta_1.unifiedMeta.getContextFromObject(col), { colOptions, getMeta });
            // parsedTree may not exists when formula column create / update
            if (parsedTree) {
                res.isDataArray = parsedTree.isDataArray;
                res.referencedColumn = parsedTree.referencedColumn;
            }
            break;
        }
        case UITypes_1.default.Barcode:
        case UITypes_1.default.Button:
        case UITypes_1.default.Collaborator:
        case UITypes_1.default.QrCode:
        default:
            res.dataType = enums_1.FormulaDataTypes.UNKNOWN;
            break;
    }
    res.referencedColumn = Object.assign({ id: col.id, uidt: col.uidt }, ((_f = res.referencedColumn) !== null && _f !== void 0 ? _f : {}));
    return res;
}
function handleBinaryExpressionForDateAndTime(params) {
    const { sourceBinaryNode } = params;
    let res;
    if ([enums_1.FormulaDataTypes.DATE, enums_1.FormulaDataTypes.INTERVAL].includes(sourceBinaryNode.left.dataType) &&
        [enums_1.FormulaDataTypes.DATE, enums_1.FormulaDataTypes.INTERVAL].includes(sourceBinaryNode.right.dataType) &&
        sourceBinaryNode.operator === '-') {
        // when it's interval and interval, we return diff in minute (numeric)
        if ([enums_1.FormulaDataTypes.INTERVAL].includes(sourceBinaryNode.left.dataType) &&
            [enums_1.FormulaDataTypes.INTERVAL].includes(sourceBinaryNode.right.dataType)) {
            res = {
                type: enums_1.JSEPNode.CALL_EXP,
                arguments: [
                    sourceBinaryNode.left,
                    sourceBinaryNode.right,
                    {
                        type: 'Literal',
                        value: 'minutes',
                        raw: '"minutes"',
                        dataType: 'string',
                    },
                ],
                callee: {
                    type: 'Identifier',
                    name: 'DATETIME_DIFF',
                },
                dataType: enums_1.FormulaDataTypes.NUMERIC,
            };
        }
        // when it's date - date, show the difference in minute
        else if ([enums_1.FormulaDataTypes.DATE].includes(sourceBinaryNode.left.dataType) &&
            [enums_1.FormulaDataTypes.DATE].includes(sourceBinaryNode.right.dataType)) {
            res = {
                type: enums_1.JSEPNode.CALL_EXP,
                arguments: [
                    sourceBinaryNode.left,
                    sourceBinaryNode.right,
                    {
                        type: 'Literal',
                        value: 'minutes',
                        raw: '"minutes"',
                        dataType: 'string',
                    },
                ],
                callee: {
                    type: 'Identifier',
                    name: 'DATETIME_DIFF',
                },
                dataType: enums_1.FormulaDataTypes.NUMERIC,
            };
        }
        // else interval and date can be addedd seamlessly A - B
        // with result as DATE
        // may be changed if we find other db use case
        else if ([enums_1.FormulaDataTypes.INTERVAL, enums_1.FormulaDataTypes.DATE].includes(sourceBinaryNode.left.dataType) &&
            [enums_1.FormulaDataTypes.INTERVAL, enums_1.FormulaDataTypes.DATE].includes(sourceBinaryNode.right.dataType) &&
            sourceBinaryNode.left.dataType != sourceBinaryNode.right.dataType) {
            res = {
                type: enums_1.JSEPNode.BINARY_EXP,
                left: sourceBinaryNode.left,
                right: sourceBinaryNode.right,
                operator: '-',
                dataType: enums_1.FormulaDataTypes.DATE,
            };
        }
    }
    else if ([enums_1.FormulaDataTypes.DATE, enums_1.FormulaDataTypes.INTERVAL].includes(sourceBinaryNode.left.dataType) &&
        [enums_1.FormulaDataTypes.DATE, enums_1.FormulaDataTypes.INTERVAL].includes(sourceBinaryNode.right.dataType) &&
        sourceBinaryNode.operator === '+') {
        // when it's interval and interval, we return addition in minute (numeric)
        if ([enums_1.FormulaDataTypes.INTERVAL].includes(sourceBinaryNode.left.dataType) &&
            [enums_1.FormulaDataTypes.INTERVAL].includes(sourceBinaryNode.right.dataType)) {
            const left = {
                type: enums_1.JSEPNode.CALL_EXP,
                arguments: [
                    sourceBinaryNode.left,
                    {
                        type: 'Literal',
                        value: '00:00:00',
                        raw: '"00:00:00"',
                        dataType: enums_1.FormulaDataTypes.INTERVAL,
                    },
                    {
                        type: 'Literal',
                        value: 'minutes',
                        raw: '"minutes"',
                        dataType: 'string',
                    },
                ],
                callee: {
                    type: 'Identifier',
                    name: 'DATETIME_DIFF',
                },
                dataType: enums_1.FormulaDataTypes.NUMERIC,
            };
            const right = {
                type: enums_1.JSEPNode.CALL_EXP,
                arguments: [
                    sourceBinaryNode.right,
                    {
                        type: 'Literal',
                        value: '00:00:00',
                        raw: '"00:00:00"',
                        dataType: enums_1.FormulaDataTypes.INTERVAL,
                    },
                    {
                        type: 'Literal',
                        value: 'minutes',
                        raw: '"minutes"',
                        dataType: 'string',
                    },
                ],
                callee: {
                    type: 'Identifier',
                    name: 'DATETIME_DIFF',
                },
                dataType: enums_1.FormulaDataTypes.NUMERIC,
            };
            return {
                type: enums_1.JSEPNode.BINARY_EXP,
                left,
                right,
                operator: '+',
                dataType: enums_1.FormulaDataTypes.NUMERIC,
            };
        }
        // else interval and date can be addedd seamlessly A + B
        // with result as DATE
        // may be changed if we find other db use case
        else if ([enums_1.FormulaDataTypes.INTERVAL, enums_1.FormulaDataTypes.DATE].includes(sourceBinaryNode.left.dataType) &&
            [enums_1.FormulaDataTypes.INTERVAL, enums_1.FormulaDataTypes.DATE].includes(sourceBinaryNode.right.dataType) &&
            sourceBinaryNode.left.dataType != sourceBinaryNode.right.dataType) {
            res = {
                type: enums_1.JSEPNode.BINARY_EXP,
                left: sourceBinaryNode.left,
                right: sourceBinaryNode.right,
                operator: '+',
                dataType: enums_1.FormulaDataTypes.DATE,
            };
        }
    }
    return res;
}
async function checkForCircularFormulaRef(formulaCol, parsedTree, columns, getMeta) {
    // Extract formula references
    const formulaPaths = await columns.reduce(async (promiseRes, c) => {
        const res = await promiseRes;
        if (c.id !== formulaCol.id && c.uidt === UITypes_1.default.Formula) {
            const neighbours = [
                ...new Set((c.colOptions.formula.match(/cl?_?\w{14,15}/g) || []).filter((colId) => columns.some((col) => col.id === colId && col.uidt === UITypes_1.default.Formula))),
            ];
            if (neighbours.length)
                res.push({ [c.id]: neighbours });
        }
        else if (c.uidt === UITypes_1.default.Lookup ||
            c.uidt === UITypes_1.default.LinkToAnotherRecord) {
            const neighbours = await processLookupOrLTARColumn(c);
            if (neighbours === null || neighbours === void 0 ? void 0 : neighbours.length)
                res.push({ [c.id]: neighbours });
        }
        return res;
    }, Promise.resolve([]));
    async function processLookupFormula(col, columns) {
        const neighbours = [];
        if (formulaCol.fk_model_id === col.fk_model_id) {
            return [col.id];
        }
        // Extract columns used in the formula and check for cycles
        const referencedColumns = col.colOptions.formula.match(/cl?_?\w{14,15}/g) || [];
        for (const refColId of referencedColumns) {
            const refCol = columns.find((c) => c.id === refColId);
            // if refColId is not valid column, maybe just a concidence value of CONCAT
            // we ignore
            if (refCol) {
                if (refCol.uidt === UITypes_1.default.Formula) {
                    neighbours.push(...(await processLookupFormula(refCol, columns)));
                }
                else if (refCol.uidt === UITypes_1.default.Lookup ||
                    refCol.uidt === UITypes_1.default.LinkToAnotherRecord) {
                    neighbours.push(...(await processLookupOrLTARColumn(refCol)));
                }
            }
        }
        return neighbours;
    }
    // Function to process lookup columns recursively
    async function processLookupOrLTARColumn(lookupOrLTARCol) {
        var _a;
        const neighbours = [];
        let ltarColumn;
        let lookupFilterFn;
        if (lookupOrLTARCol.uidt === UITypes_1.default.Lookup) {
            const relationColId = lookupOrLTARCol.colOptions
                .fk_relation_column_id;
            const lookupColId = lookupOrLTARCol.colOptions
                .fk_lookup_column_id;
            ltarColumn = columns.find((c) => c.id === relationColId);
            lookupFilterFn = (column) => column.id === lookupColId;
        }
        else if (lookupOrLTARCol.uidt === UITypes_1.default.LinkToAnotherRecord) {
            ltarColumn = lookupOrLTARCol;
            lookupFilterFn = (column) => !!column.pv;
        }
        if (ltarColumn) {
            const relatedTableMeta = await getMeta(unifiedMeta_1.unifiedMeta.getContextFromObject(Object.assign(Object.assign({}, ltarColumn), { base_id: (_a = ltarColumn.colOptions
                    .fk_related_base_id) !== null && _a !== void 0 ? _a : ltarColumn.base_id })), {
                id: ltarColumn.colOptions
                    .fk_related_model_id,
            });
            const lookupTarget = (await unifiedMeta_1.unifiedMeta.getColumns(unifiedMeta_1.unifiedMeta.getContextFromObject(relatedTableMeta), {
                model: relatedTableMeta,
            })).find(lookupFilterFn);
            if (lookupTarget) {
                if (lookupTarget.uidt === UITypes_1.default.Formula) {
                    neighbours.push(...(await processLookupFormula(lookupTarget, relatedTableMeta.columns)));
                }
                else if (lookupTarget.uidt === UITypes_1.default.Lookup ||
                    lookupTarget.uidt === UITypes_1.default.LinkToAnotherRecord) {
                    neighbours.push(...(await processLookupOrLTARColumn(lookupTarget)));
                }
            }
        }
        return [...new Set(neighbours)];
    }
    // include target formula column (i.e. the one to be saved if applicable)
    const targetFormulaCol = columns.find((c) => c.title === parsedTree.name &&
        [UITypes_1.default.Formula, UITypes_1.default.LinkToAnotherRecord, UITypes_1.default.Lookup].includes(c.uidt));
    if (targetFormulaCol && (formulaCol === null || formulaCol === void 0 ? void 0 : formulaCol.id)) {
        formulaPaths.push({
            [formulaCol === null || formulaCol === void 0 ? void 0 : formulaCol.id]: [targetFormulaCol.id],
        });
    }
    const vertices = formulaPaths.length;
    if (vertices > 0) {
        // perform kahn's algo for cycle detection
        const adj = new Map();
        const inDegrees = new Map();
        // init adjacency list & indegree
        for (const [_, v] of Object.entries(formulaPaths)) {
            const src = Object.keys(v)[0];
            const neighbours = v[src];
            inDegrees.set(src, inDegrees.get(src) || 0);
            for (const neighbour of neighbours) {
                adj.set(src, (adj.get(src) || new Set()).add(neighbour));
                inDegrees.set(neighbour, (inDegrees.get(neighbour) || 0) + 1);
            }
        }
        const queue = [];
        // put all vertices with in-degree = 0 (i.e. no incoming edges) to queue
        inDegrees.forEach((inDegree, col) => {
            if (inDegree === 0) {
                // in-degree = 0 means we start traversing from this node
                queue.push(col);
            }
        });
        // init count of visited vertices
        let visited = 0;
        // BFS
        while (queue.length !== 0) {
            // remove a vertex from the queue
            const src = queue.shift();
            // if this node has neighbours, increase visited by 1
            const neighbours = adj.get(src) || new Set();
            if (neighbours.size > 0) {
                visited += 1;
            }
            // iterate each neighbouring nodes
            neighbours.forEach((neighbour) => {
                // decrease in-degree of its neighbours by 1
                inDegrees.set(neighbour, inDegrees.get(neighbour) - 1);
                // if in-degree becomes 0
                if (inDegrees.get(neighbour) === 0) {
                    // then put the neighboring node to the queue
                    queue.push(neighbour);
                }
            });
        }
        // vertices not same as visited = cycle found
        if (vertices !== visited) {
            throw new error_1.FormulaError(enums_1.FormulaErrorType.CIRCULAR_REFERENCE, {
                key: 'msg.formula.cantSaveCircularReference',
            }, 'Circular reference detected');
        }
    }
}
async function validateFormulaAndExtractTreeWithType({ formula, column, columns, clientOrSqlUi, getMeta, trackPosition, }) {
    var _a, _b, _c;
    // extract column list from meta since columns array might not have all columns(system columns)
    const meta = await getMeta(unifiedMeta_1.unifiedMeta.getContextFromObject((_b = (_a = ((column === null || column === void 0 ? void 0 : column.base_id) && column)) !== null && _a !== void 0 ? _a : columns === null || columns === void 0 ? void 0 : columns[0]) !== null && _b !== void 0 ? _b : {}), { id: (column === null || column === void 0 ? void 0 : column.fk_model_id) || ((_c = columns === null || columns === void 0 ? void 0 : columns[0]) === null || _c === void 0 ? void 0 : _c.fk_model_id) || '' });
    const allColumns = (meta === null || meta === void 0 ? void 0 : meta.columns) || columns;
    const sqlUI = typeof clientOrSqlUi === 'string'
        ? sqlUi_1.SqlUiFactory.create({ client: clientOrSqlUi })
        : clientOrSqlUi;
    const colAliasToColMap = {};
    const colIdToColMap = {};
    for (const col of columns) {
        colAliasToColMap[col.title] = col;
        colIdToColMap[col.id] = col;
    }
    const validateAndExtract = async (parsedTree) => {
        var _a, _b, _c, _d, _e, _f, _g;
        const res = Object.assign({}, parsedTree);
        if (parsedTree.type === enums_1.JSEPNode.CALL_EXP) {
            const calleeName = parsedTree.callee.name.toUpperCase();
            // validate function name
            if (!formulas_1.formulas[calleeName]) {
                throw new error_1.FormulaError(enums_1.FormulaErrorType.INVALID_FUNCTION_NAME, {
                    calleeName,
                    position: parsedTree.indexStart >= 0
                        ? {
                            index: parsedTree.indexStart,
                            length: parsedTree.nodeLength,
                        }
                        : undefined,
                }, `Function ${calleeName} is not available`);
            }
            else if (sqlUI === null || sqlUI === void 0 ? void 0 : sqlUI.getUnsupportedFnList().includes(calleeName)) {
                throw new error_1.FormulaError(enums_1.FormulaErrorType.INVALID_FUNCTION_NAME, {
                    calleeName,
                    position: parsedTree.indexStart >= 0
                        ? {
                            index: parsedTree.indexStart,
                            length: parsedTree.nodeLength,
                        }
                        : undefined,
                }, `Function ${calleeName} is unavailable for your database`);
            }
            // validate arguments
            const validation = formulas_1.formulas[calleeName] && formulas_1.formulas[calleeName].validation;
            if (validation && validation.args) {
                if (validation.args.rqd !== undefined &&
                    validation.args.rqd !== parsedTree.arguments.length) {
                    throw new error_1.FormulaError(enums_1.FormulaErrorType.INVALID_ARG, {
                        key: 'msg.formula.requiredArgumentsFormula',
                        requiredArguments: validation.args.rqd,
                        calleeName,
                        position: parsedTree.indexStart >= 0
                            ? {
                                index: parsedTree.indexStart,
                                length: parsedTree.nodeLength,
                            }
                            : undefined,
                    }, 'Required arguments missing');
                }
                else if (validation.args.min !== undefined &&
                    validation.args.min > parsedTree.arguments.length) {
                    throw new error_1.FormulaError(enums_1.FormulaErrorType.MIN_ARG, {
                        key: 'msg.formula.minRequiredArgumentsFormula',
                        minRequiredArguments: validation.args.min,
                        calleeName,
                        position: parsedTree.indexStart >= 0
                            ? {
                                index: parsedTree.indexStart,
                                length: parsedTree.nodeLength,
                            }
                            : undefined,
                    }, 'Minimum arguments required');
                }
                else if (validation.args.max !== undefined &&
                    validation.args.max < parsedTree.arguments.length) {
                    throw new error_1.FormulaError(enums_1.FormulaErrorType.INVALID_ARG, {
                        key: 'msg.formula.maxRequiredArgumentsFormula',
                        maxRequiredArguments: validation.args.max,
                        calleeName,
                        position: parsedTree.indexStart >= 0
                            ? {
                                index: parsedTree.indexStart,
                                length: parsedTree.nodeLength,
                            }
                            : undefined,
                    }, 'Maximum arguments missing');
                }
            }
            // get args type and validate
            const validateResult = (res.arguments =
                await Promise.all(parsedTree.arguments.map((arg) => {
                    return validateAndExtract(arg);
                })));
            const argTypes = validateResult.map((v) => v.dataType);
            // if validation function is present, call it
            if ((_a = formulas_1.formulas[calleeName].validation) === null || _a === void 0 ? void 0 : _a.custom) {
                (_b = formulas_1.formulas[calleeName].validation) === null || _b === void 0 ? void 0 : _b.custom(argTypes, 
                // need to use res rather than parsedTree
                // because post-processing like referencedColumn is needed
                res, allColumns);
            }
            // validate against expected arg types if present
            else if ((_d = (_c = formulas_1.formulas[calleeName].validation) === null || _c === void 0 ? void 0 : _c.args) === null || _d === void 0 ? void 0 : _d.type) {
                for (let i = 0; i < validateResult.length; i++) {
                    const argPt = validateResult[i];
                    // if type
                    const expectedArgType = Array.isArray(formulas_1.formulas[calleeName].validation.args.type)
                        ? formulas_1.formulas[calleeName].validation.args.type[i]
                        : formulas_1.formulas[calleeName].validation.args.type;
                    if (argPt.dataType !== expectedArgType &&
                        argPt.dataType !== enums_1.FormulaDataTypes.NULL &&
                        argPt.dataType !== enums_1.FormulaDataTypes.UNKNOWN &&
                        expectedArgType !== enums_1.FormulaDataTypes.STRING) {
                        if (argPt.type === enums_1.JSEPNode.IDENTIFIER) {
                            const name = ((_e = columns === null || columns === void 0 ? void 0 : columns.find((c) => c.id === argPt.name || c.title === argPt.name)) === null || _e === void 0 ? void 0 : _e.title) || argPt.name;
                            throw new error_1.FormulaError(enums_1.FormulaErrorType.INVALID_ARG, {
                                key: 'msg.formula.columnWithTypeFoundButExpected',
                                columnName: name,
                                columnType: argPt.dataType,
                                expectedType: expectedArgType,
                            }, `Field ${name} with ${argPt.dataType} type is found but ${expectedArgType} type is expected`);
                        }
                        else {
                            let key = '';
                            const position = i + 1;
                            let type = '';
                            if (expectedArgType === enums_1.FormulaDataTypes.NUMERIC) {
                                key = 'msg.formula.typeIsExpected';
                                type = 'numeric';
                            }
                            else if (expectedArgType === enums_1.FormulaDataTypes.BOOLEAN) {
                                key = 'msg.formula.typeIsExpected';
                                type = 'boolean';
                            }
                            else if (expectedArgType === enums_1.FormulaDataTypes.DATE) {
                                key = 'msg.formula.typeIsExpected';
                                type = 'date';
                            }
                            throw new error_1.FormulaError(enums_1.FormulaErrorType.INVALID_ARG, {
                                type,
                                key,
                                position,
                                calleeName,
                            }, `${calleeName === null || calleeName === void 0 ? void 0 : calleeName.toUpperCase()} requires a ${type || expectedArgType} at position ${position}`);
                        }
                    }
                    // if expected type is string and arg type is not string, then cast it to string
                    if (expectedArgType === enums_1.FormulaDataTypes.STRING &&
                        expectedArgType !== argPt.dataType) {
                        argPt.cast = enums_1.FormulaDataTypes.STRING;
                    }
                }
            }
            if (typeof formulas_1.formulas[calleeName].returnType === 'function') {
                res.dataType = (_g = (_f = formulas_1.formulas[calleeName]).returnType) === null || _g === void 0 ? void 0 : _g.call(_f, argTypes);
            }
            else if (formulas_1.formulas[calleeName].returnType) {
                res.dataType = formulas_1.formulas[calleeName].returnType;
            }
            if (calleeName.toUpperCase().startsWith('ARRAY')) {
                res.inArrayFormat = true;
            }
            Object.assign(res, (0, referenced_info_extractor_1.extractCallExpressionReferencedInfo)({
                parsedTree: res,
            }));
        }
        else if (parsedTree.type === enums_1.JSEPNode.IDENTIFIER) {
            const identifierName = parsedTree.name;
            const col = (colIdToColMap[identifierName] ||
                colAliasToColMap[identifierName]);
            if (!col) {
                if (formulas_1.formulas[identifierName]) {
                    throw new error_1.FormulaError(enums_1.FormulaErrorType.INVALID_SYNTAX, {
                        key: 'msg.formula.formulaMissingParentheses',
                        calleeName: identifierName,
                        position: parsedTree.indexStart >= 0
                            ? {
                                index: parsedTree.indexEnd,
                                length: 1,
                            }
                            : undefined,
                    }, `Missing parentheses after function name "${JSON.stringify(identifierName)}"`);
                }
                throw new error_1.FormulaError(enums_1.FormulaErrorType.INVALID_COLUMN, {
                    key: 'msg.formula.columnNotAvailable',
                    columnName: identifierName,
                    position: parsedTree.indexStart >= 0
                        ? {
                            index: parsedTree.indexStart,
                            length: parsedTree.nodeLength,
                        }
                        : undefined,
                }, `Invalid column name/id ${JSON.stringify(identifierName)} in formula`);
            }
            res.name = col.id;
            if ((col === null || col === void 0 ? void 0 : col.uidt) === UITypes_1.default.Formula) {
                if (column) {
                    // check for circular reference when column is present(only available when calling root formula)
                    await checkForCircularFormulaRef(column, parsedTree, columns, getMeta);
                }
                const formulaRes = col.colOptions.parsed_tree ||
                    (await validateFormulaAndExtractTreeWithType(
                    // formula may include double curly brackets in previous version
                    // convert to single curly bracket here for compatibility
                    {
                        formula: col.colOptions.formula
                            .replaceAll('{{', '{')
                            .replaceAll('}}', '}'),
                        columns,
                        clientOrSqlUi,
                        getMeta,
                    }));
                res.dataType = formulaRes === null || formulaRes === void 0 ? void 0 : formulaRes.dataType;
                res.inArrayFormat = formulaRes.inArrayFormat;
            }
            else {
                if ((col === null || col === void 0 ? void 0 : col.uidt) === UITypes_1.default.Lookup ||
                    (col === null || col === void 0 ? void 0 : col.uidt) === UITypes_1.default.LinkToAnotherRecord) {
                    // check for circular reference when column is present(only available when calling root formula)
                    if (column) {
                        await checkForCircularFormulaRef(column, parsedTree, columns, getMeta);
                    }
                }
            }
            // extract type and add to res
            Object.assign(res, await extractColumnIdentifierType({
                col,
                columns,
                getMeta,
                clientOrSqlUi,
            }));
        }
        else if (parsedTree.type === enums_1.JSEPNode.LITERAL) {
            if (typeof parsedTree.value === 'number') {
                res.dataType = enums_1.FormulaDataTypes.NUMERIC;
            }
            else if (typeof parsedTree.value === 'string') {
                res.dataType = enums_1.FormulaDataTypes.STRING;
            }
            else if (typeof parsedTree.value === 'boolean') {
                res.dataType = enums_1.FormulaDataTypes.BOOLEAN;
            }
            else {
                res.dataType = enums_1.FormulaDataTypes.STRING;
            }
        }
        else if (parsedTree.type === enums_1.JSEPNode.UNARY_EXP) {
            // only support -ve values
            if (['-'].includes(parsedTree.operator) &&
                parsedTree.argument.type === enums_1.JSEPNode.LITERAL &&
                typeof parsedTree.argument.value === 'number') {
                res.dataType = enums_1.FormulaDataTypes.NUMERIC;
            }
            else {
                throw new error_1.FormulaError(enums_1.FormulaErrorType.NOT_SUPPORTED, {}, `Unary expression '${parsedTree.operator}' is not supported`);
            }
        }
        else if (parsedTree.type === enums_1.JSEPNode.BINARY_EXP) {
            const argsLeft = await validateAndExtract(parsedTree.left);
            const argsRight = await validateAndExtract(parsedTree.right);
            res.left = argsLeft;
            res.right = argsRight;
            const dateAndTimeParsedNode = handleBinaryExpressionForDateAndTime({
                sourceBinaryNode: res,
            });
            if (dateAndTimeParsedNode) {
                Object.assign(res, handleBinaryExpressionForDateAndTime({ sourceBinaryNode: res }));
                if (res.type !== enums_1.JSEPNode.BINARY_EXP) {
                    res.left = undefined;
                    res.right = undefined;
                    res.operator = undefined;
                }
            }
            else if (['==', '<', '>', '<=', '>=', '!='].includes(parsedTree.operator)) {
                res.dataType = enums_1.FormulaDataTypes.COND_EXP;
            }
            else if (parsedTree.operator === '+') {
                res.dataType = enums_1.FormulaDataTypes.NUMERIC;
                // if any side is string/date/other type, then the result will be concatenated string
                // e.g. 1 + '2' = '12'
                if ([
                    res.left,
                    res.right,
                ].some((r) => ![
                    enums_1.FormulaDataTypes.NUMERIC,
                    enums_1.FormulaDataTypes.BOOLEAN,
                    enums_1.FormulaDataTypes.NULL,
                    enums_1.FormulaDataTypes.UNKNOWN,
                ].includes(r.dataType))) {
                    res.dataType = enums_1.FormulaDataTypes.STRING;
                }
            }
            else if (['&'].includes(parsedTree.operator)) {
                res.dataType = enums_1.FormulaDataTypes.STRING;
            }
            else {
                res.dataType = enums_1.FormulaDataTypes.NUMERIC;
            }
            Object.assign(res, (0, referenced_info_extractor_1.extractBinaryExpReferencedInfo)({
                parsedTree: res,
                left: argsLeft,
                right: argsRight,
            }));
        }
        else if (parsedTree.type === enums_1.JSEPNode.MEMBER_EXP) {
            throw new error_1.FormulaError(enums_1.FormulaErrorType.NOT_SUPPORTED, {}, 'Bracket notation is not supported');
        }
        else if (parsedTree.type === enums_1.JSEPNode.ARRAY_EXP) {
            throw new error_1.FormulaError(enums_1.FormulaErrorType.NOT_SUPPORTED, {}, 'Array is not supported');
        }
        else if (parsedTree.type === enums_1.JSEPNode.COMPOUND) {
            throw new error_1.FormulaError(enums_1.FormulaErrorType.NOT_SUPPORTED, {}, 'Compound statement is not supported');
        }
        return res;
    };
    try {
        const jsepInstance = trackPosition ? jsepInstances_1.formulaJsepWithIndex : jsepInstances_1.formulaJsep;
        const parsedFormula = jsepInstance(formula);
        // TODO: better jsep expression handling
        const result = await validateAndExtract(parsedFormula);
        return result;
    }
    catch (ex) {
        if (trackPosition) {
            (0, handle_formula_error_1.handleFormulaError)({ formula, error: ex });
        }
        throw ex;
    }
}
//# sourceMappingURL=data:application/json;base64,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